/**
 * unidata.hpp autogenerated 2024-07-11 13:00:50.532224
 *
 * Provides access to the Unicode Character Database.
 * Properties of a single Unicode character are accessed by its code point value.
 *
 * Format conventions
 *  https://www.unicode.org/reports/tr44/
 *
 * Character presentation width rules
 *  EAW:    https://www.unicode.org/reports/tr11
 *  Emoji:  https://www.unicode.org/reports/tr51
 *
 * Grapheme clusterization
 *  https://www.unicode.org/reports/tr29
 *
 * Unicode Character Database properties
 *  https://www.unicode.org/reports/tr44/#Property_Index
 *
 * All emoji characters are treated 'East Asian Wide'.
 *  Current practice is for emoji to have a square aspect ratio, deriving from their origin in Japanese.
 *  https://www.unicode.org/reports/tr51/#Display
 *
 * VS15/16
 *  Emoji presentation sequences behave as though they were East Asian Wide,
 *  regardless of their assigned East_Asian_Width property value.
 *  https://www.unicode.org/reports/tr11/#Recommendations
 *
 * Ambiguous characters
 *  Ambiguous characters behave like wide or narrow characters depending on the context.
 *  If the context cannot be established reliably, they should be treated as narrow characters by default.
 *  https://www.unicode.org/reports/tr11/#Recommendations
 *
 * Categories of the character width
 *  0 - Non-printable
 *  1 - Halfwidth
 *  2 - Fullwidth
 *
 * C0 controls 0x00..0x1F
 *  Since C0 codes are based on bytes, they are excluded from the property list of controls.
 *  They are analyzed in static tables before decoding UTF-8.
 *
 * C1 controls 0x80..0x9F
 *  The C1 codes require two bytes to be encoded in UTF-8
 *  (for instance CSI at U+009B is encoded as the bytes 0xC2, 0x9B in UTF-8),
 *  so there is no advantage to using them rather than the equivalent
 *  two-byte ESC+letter sequence, so the C1 controls are represented
 *  as C0 escaped sequences.
 *  The C1 characters appear outdated now.
 *  https://en.wikipedia.org/wiki/C0_and_C1_control_codes#Unicode
 *
 * Soft Hyphen
 *  Two ways:
 *    1. Interpret it as a command and divide the text strings into two independent.
 *    2. Append it to the last grapheme cluster.
 *
 * Printable format characters
 *  A 'Prepend' characters always have the width 'Narrow' to be the basis of the grapheme cluster.
 *  https://www.unicode.org/reports/tr29/#Table_Combining_Char_Sequences_and_Grapheme_Clusters
 *  https://www.unicode.org/reports/tr29/#GB9b
 *
 * Names and Aliases
 *  Character name aliases are immutable.
 *  https://www.unicode.org/versions/Unicode12.0.0/ch04.pdf#page=24
 *
 * Unicode Bidirectional Algorithm
 *  https://www.unicode.org/cldr/utility/bidi.jsp
 *  https://unicode.org/reports/tr9/
 *
 * Controls (should be enumerated in ascending order)
 *    command: <NON_CONTROL (paragraph break possible)
 *            C0
 *            C1
 *            \u2029 PARAGRAPH SEPARATOR
 *    visible: =NON_CONTROL - non control chars
 *     format: >NON_CONTROL
 *             All other control chars
 **/

#pragma once

#include <cstdint>
#include <vector>
#include <iterator>

namespace netxs::unidata
{
    namespace widths
    {
        static constexpr auto zero = 0; // 0x0 non-printable
        static constexpr auto slim = 1; // 1x1 narrow
        static constexpr auto wide = 2; // 2x1 fullwidth
    }

    namespace gbreak
    {
        static constexpr auto any   =  0; // Other
        static constexpr auto cr    =  1; // CR
        static constexpr auto lf    =  2; // LF
        static constexpr auto ctrl  =  3; // Control
        static constexpr auto ext   =  4; // Extend or Emoji_Modifier_Base
        static constexpr auto l     =  5; // HANGUL CHOSEONG
        static constexpr auto v     =  6; // HANGUL JUNGSEONG
        static constexpr auto t     =  7; // HANGUL JUNGSEONG
        static constexpr auto lv    =  8; // HANGUL SYLLABLE
        static constexpr auto lvt   =  9; // HANGUL SYLLABLE
        static constexpr auto ri    = 10; // Regional_Indicator
        static constexpr auto sm    = 11; // SpacingMark
        static constexpr auto prep  = 12; // Prepend
        static constexpr auto zwj   = 13; // ZERO WIDTH JOINER
        static constexpr auto ep    = 14; // Extended_Pictographic
        static constexpr auto combo = 15; // EP + ZWJ
        static constexpr auto count = 16; // GB class count.
    }

    enum cntrls : ui32
    {
        nul                                       , //   0 0x0   NULL
        soh                                       , //   1 0x1   START OF HEADING
        stx                                       , //   2 0x2   START OF TEXT
        etx                                       , //   3 0x3   END OF TEXT
        eot                                       , //   4 0x4   END OF TRANSMISSION
        enq                                       , //   5 0x5   ENQUIRY
        ack                                       , //   6 0x6   ACKNOWLEDGE
        bel                                       , //   7 0x7   ALERT
        bs                                        , //   8 0x8   BACKSPACE
        tab                                       , //   9 0x9   CHARACTER TABULATION
        eol                                       , //  10 0xA   LINE FEED
        vt                                        , //  11 0xB   LINE TABULATION
        ff                                        , //  12 0xC   FORM FEED
        cr                                        , //  13 0xD   CARRIAGE RETURN
        so                                        , //  14 0xE   SHIFT OUT
        si                                        , //  15 0xF   SHIFT IN
        dle                                       , //  16 0x10  DATA LINK ESCAPE
        dc1                                       , //  17 0x11  DEVICE CONTROL ONE
        dc2                                       , //  18 0x12  DEVICE CONTROL TWO
        dc3                                       , //  19 0x13  DEVICE CONTROL THREE
        dc4                                       , //  20 0x14  DEVICE CONTROL FOUR
        nak                                       , //  21 0x15  NEGATIVE ACKNOWLEDGE
        syn                                       , //  22 0x16  SYNCHRONOUS IDLE
        etb                                       , //  23 0x17  END OF TRANSMISSION BLOCK
        can                                       , //  24 0x18  CANCEL
        em                                        , //  25 0x19  END OF MEDIUM
        sub                                       , //  26 0x1A  SUBSTITUTE
        esc                                       , //  27 0x1B  ESCAPE
        fs                                        , //  28 0x1C  INFORMATION SEPARATOR FOUR
        gs                                        , //  29 0x1D  INFORMATION SEPARATOR THREE
        rs                                        , //  30 0x1E  INFORMATION SEPARATOR TWO
        us                                        , //  31 0x1F  INFORMATION SEPARATOR ONE
        del                                       , //  32 0x7F  DELETE
        pad                                       , //  33 0x80  PADDING CHARACTER
        hop                                       , //  34 0x81  HIGH OCTET PRESET
        bph                                       , //  35 0x82  BREAK PERMITTED HERE
        nbh                                       , //  36 0x83  NO BREAK HERE
        ind                                       , //  37 0x84  INDEX
        nel                                       , //  38 0x85  NEXT LINE
        ssa                                       , //  39 0x86  START OF SELECTED AREA
        esa                                       , //  40 0x87  END OF SELECTED AREA
        hts                                       , //  41 0x88  CHARACTER TABULATION SET
        htj                                       , //  42 0x89  CHARACTER TABULATION WITH JUSTIFICATION
        vts                                       , //  43 0x8A  LINE TABULATION SET
        pld                                       , //  44 0x8B  PARTIAL LINE FORWARD
        plu                                       , //  45 0x8C  PARTIAL LINE BACKWARD
        ri                                        , //  46 0x8D  REVERSE LINE FEED
        ss2                                       , //  47 0x8E  SINGLE SHIFT TWO
        ss3                                       , //  48 0x8F  SINGLE SHIFT THREE
        dcs                                       , //  49 0x90  DEVICE CONTROL STRING
        pu1                                       , //  50 0x91  PRIVATE USE ONE
        pu2                                       , //  51 0x92  PRIVATE USE TWO
        sts                                       , //  52 0x93  SET TRANSMIT STATE
        cch                                       , //  53 0x94  CANCEL CHARACTER
        mw                                        , //  54 0x95  MESSAGE WAITING
        spa                                       , //  55 0x96  START OF GUARDED AREA
        epa                                       , //  56 0x97  END OF GUARDED AREA
        sos                                       , //  57 0x98  START OF STRING
        sgc                                       , //  58 0x99  SINGLE GRAPHIC CHARACTER INTRODUCER
        sci                                       , //  59 0x9A  SINGLE CHARACTER INTRODUCER
        csi                                       , //  60 0x9B  CONTROL SEQUENCE INTRODUCER
        st                                        , //  61 0x9C  STRING TERMINATOR
        osc                                       , //  62 0x9D  OPERATING SYSTEM COMMAND
        pm                                        , //  63 0x9E  PRIVACY MESSAGE
        apc                                       , //  64 0x9F  APPLICATION PROGRAM COMMAND
        paragraph_separator                       , //  65 0x2029 PARAGRAPH SEPARATOR
        non_control                               , //  66 -1    NON CONTROL
        shy                                       , //  67 0xAD  SOFT HYPHEN
        alm                                       , //  68 0x61C ARABIC LETTER MARK
        mvs                                       , //  69 0x180E MONGOLIAN VOWEL SEPARATOR
        lrm                                       , //  70 0x200E LEFT-TO-RIGHT MARK
        rlm                                       , //  71 0x200F RIGHT-TO-LEFT MARK
        line_separator                            , //  72 0x2028 LINE SEPARATOR
        lre                                       , //  73 0x202A LEFT-TO-RIGHT EMBEDDING
        rle                                       , //  74 0x202B RIGHT-TO-LEFT EMBEDDING
        pdf                                       , //  75 0x202C POP DIRECTIONAL FORMATTING
        lro                                       , //  76 0x202D LEFT-TO-RIGHT OVERRIDE
        rlo                                       , //  77 0x202E RIGHT-TO-LEFT OVERRIDE
        wj                                        , //  78 0x2060 WORD JOINER
        function_application                      , //  79 0x2061 FUNCTION APPLICATION
        invisible_times                           , //  80 0x2062 INVISIBLE TIMES
        invisible_separator                       , //  81 0x2063 INVISIBLE SEPARATOR
        invisible_plus                            , //  82 0x2064 INVISIBLE PLUS
        lri                                       , //  83 0x2066 LEFT-TO-RIGHT ISOLATE
        rli                                       , //  84 0x2067 RIGHT-TO-LEFT ISOLATE
        fsi                                       , //  85 0x2068 FIRST STRONG ISOLATE
        pdi                                       , //  86 0x2069 POP DIRECTIONAL ISOLATE
        inhibit_symmetric_swapping                , //  87 0x206A INHIBIT SYMMETRIC SWAPPING
        activate_symmetric_swapping               , //  88 0x206B ACTIVATE SYMMETRIC SWAPPING
        inhibit_arabic_form_shaping               , //  89 0x206C INHIBIT ARABIC FORM SHAPING
        activate_arabic_form_shaping              , //  90 0x206D ACTIVATE ARABIC FORM SHAPING
        national_digit_shapes                     , //  91 0x206E NATIONAL DIGIT SHAPES
        nominal_digit_shapes                      , //  92 0x206F NOMINAL DIGIT SHAPES
        zwnbsp                                    , //  93 0xFEFF ZERO WIDTH NO-BREAK SPACE
        interlinear_annotation_anchor             , //  94 0xFFF9 INTERLINEAR ANNOTATION ANCHOR
        interlinear_annotation_separator          , //  95 0xFFFA INTERLINEAR ANNOTATION SEPARATOR
        interlinear_annotation_terminator         , //  96 0xFFFB INTERLINEAR ANNOTATION TERMINATOR
        egyptian_hieroglyph_vertical_joiner       , //  97 0x13430 EGYPTIAN HIEROGLYPH VERTICAL JOINER
        egyptian_hieroglyph_horizontal_joiner     , //  98 0x13431 EGYPTIAN HIEROGLYPH HORIZONTAL JOINER
        egyptian_hieroglyph_insert_at_top_start   , //  99 0x13432 EGYPTIAN HIEROGLYPH INSERT AT TOP START
        egyptian_hieroglyph_insert_at_bottom_start, // 100 0x13433 EGYPTIAN HIEROGLYPH INSERT AT BOTTOM START
        egyptian_hieroglyph_insert_at_top_end     , // 101 0x13434 EGYPTIAN HIEROGLYPH INSERT AT TOP END
        egyptian_hieroglyph_insert_at_bottom_end  , // 102 0x13435 EGYPTIAN HIEROGLYPH INSERT AT BOTTOM END
        egyptian_hieroglyph_overlay_middle        , // 103 0x13436 EGYPTIAN HIEROGLYPH OVERLAY MIDDLE
        egyptian_hieroglyph_begin_segment         , // 104 0x13437 EGYPTIAN HIEROGLYPH BEGIN SEGMENT
        egyptian_hieroglyph_end_segment           , // 105 0x13438 EGYPTIAN HIEROGLYPH END SEGMENT
        egyptian_hieroglyph_insert_at_middle      , // 106 0x13439 EGYPTIAN HIEROGLYPH INSERT AT MIDDLE
        egyptian_hieroglyph_insert_at_top         , // 107 0x1343A EGYPTIAN HIEROGLYPH INSERT AT TOP
        egyptian_hieroglyph_insert_at_bottom      , // 108 0x1343B EGYPTIAN HIEROGLYPH INSERT AT BOTTOM
        egyptian_hieroglyph_begin_enclosure       , // 109 0x1343C EGYPTIAN HIEROGLYPH BEGIN ENCLOSURE
        egyptian_hieroglyph_end_enclosure         , // 110 0x1343D EGYPTIAN HIEROGLYPH END ENCLOSURE
        egyptian_hieroglyph_begin_walled_enclosure, // 111 0x1343E EGYPTIAN HIEROGLYPH BEGIN WALLED ENCLOSURE
        egyptian_hieroglyph_end_walled_enclosure  , // 112 0x1343F EGYPTIAN HIEROGLYPH END WALLED ENCLOSURE
        shorthand_format_letter_overlap           , // 113 0x1BCA0 SHORTHAND FORMAT LETTER OVERLAP
        shorthand_format_continuing_overlap       , // 114 0x1BCA1 SHORTHAND FORMAT CONTINUING OVERLAP
        shorthand_format_down_step                , // 115 0x1BCA2 SHORTHAND FORMAT DOWN STEP
        shorthand_format_up_step                  , // 116 0x1BCA3 SHORTHAND FORMAT UP STEP
        musical_symbol_begin_beam                 , // 117 0x1D173 MUSICAL SYMBOL BEGIN BEAM
        musical_symbol_end_beam                   , // 118 0x1D174 MUSICAL SYMBOL END BEAM
        musical_symbol_begin_tie                  , // 119 0x1D175 MUSICAL SYMBOL BEGIN TIE
        musical_symbol_end_tie                    , // 120 0x1D176 MUSICAL SYMBOL END TIE
        musical_symbol_begin_slur                 , // 121 0x1D177 MUSICAL SYMBOL BEGIN SLUR
        musical_symbol_end_slur                   , // 122 0x1D178 MUSICAL SYMBOL END SLUR
        musical_symbol_begin_phrase               , // 123 0x1D179 MUSICAL SYMBOL BEGIN PHRASE
        musical_symbol_end_phrase                 , // 124 0x1D17A MUSICAL SYMBOL END PHRASE
        count                                     , // 125
    };

    struct unidata;
    inline unidata const& select(ui32 cp);

    struct unidata
    {
        ui32 ucwidth : 2; // 0 - zero, 1 - slim, 2 - wide.
        ui32 brgroup : 4;
        ui32 control : 7;
        //ui32 reserv : 19;

        constexpr unidata(unidata const&) = default;
        constexpr unidata()
            : ucwidth{ widths::slim },
              brgroup{ gbreak::any },
              control{ cntrls::non_control }
        { }
        constexpr unidata(ui32 ucwidth, ui32 brgroup, ui32 control)
            : ucwidth{ ucwidth },
              brgroup{ brgroup },
              control{ control }
        { }
        unidata(ui32 cp)
            : unidata{ select(cp) }
        { }

        constexpr unidata& operator = (unidata const&) = default;

        auto is_cmd()
        {
            return control < cntrls::non_control;
        }

        // Unicode 15.1.0 UAX #29 https://www.unicode.org/reports/tr29/#Grapheme_Cluster_Boundary_Rules
        bool allied(unidata const& next)
        {
            static const auto lut = []
            {
                auto table = std::array<byte, gbreak::count * gbreak::count>{};
                auto check = [](auto l, auto r)
                {
                    return (  l == gbreak::cr    &&  r == gbreak::lf   )  ? true: // GB3
                           (  l >= gbreak::cr    &&  l <= gbreak::ctrl )  ? faux: // GB4
                           (  r >= gbreak::cr    &&  r <= gbreak::ctrl )  ? faux: // GB5
                           (  l == gbreak::l     && (r == gbreak::l
                                                 ||  r == gbreak::v
                                                 ||  r == gbreak::lv
                                                 ||  r == gbreak::lvt  )) ? true: // GB6
                           (( l == gbreak::lv    ||  l == gbreak::v    )
                         && ( r == gbreak::v     ||  r == gbreak::t    )) ? true: // GB7
                           (( l == gbreak::lvt   ||  l == gbreak::t    )
                                                 &&  r == gbreak::t    )  ? true: // GB8
                           (  l == gbreak::prep  ||  r == gbreak::zwj
                                                 ||  r == gbreak::sm
                                                 ||  r == gbreak::ext  )  ? true: // GB9,a,b
                           (  l == gbreak::combo &&  r == gbreak::ep   )  ? true: // GB11
                           (  l == gbreak::ri    &&  r == gbreak::ri   )  ? true: // GB12,13
                                                                            faux; // GB999
                  };
                  for (auto l = 0; l < gbreak::count; l++)
                  for (auto r = 0; r < gbreak::count; r++)
                  {
                      table[l + r * gbreak::count] = check(l, r);
                  }
                  return table;
            }();
            auto l = brgroup;
            auto r = next.brgroup;
            if (l == gbreak::ep)
            {
                brgroup = (r == gbreak::ext) ? gbreak::ep    :
                          (r == gbreak::zwj) ? gbreak::combo : r;
            }
            else
            {
                brgroup = (l == gbreak::ri && r == gbreak::ri) ? gbreak::any : r;
            }
            auto result = lut[l + r * gbreak::count];
            return result;
        }
    };

    struct base
    {
        static constexpr auto blocks_size = sz_t{ 4352 };
        static constexpr auto blocks_pack = std::to_array<si32>(
        {
              0,  -2, 256, 512, 768, 1024, 1280, 1536, 1792, 2048,
            2304, 2560, 2816, 3072, 3328, 3584, 3840, 4096, 4352, 4608,
             -2, 256, 4864, 5120, 5376, 5632, 5888, 6144, 6400, 6656,
            256, 6912, 7168, 7424, 256, 7680, 7936, 8192, 8448, 8704,
            256, 8960, 256, 9216, 9472, 9728, 9984, 10240, 10496, 10752,
            11008, -113, 11264, 11520, 256, 11776, 12032, 12288, 12544, 12800,
            13056, 13312, 13568, 13824, 14080, 14336, 14592, 14848, 13312, 13568,
            13824, 14080, 14336, 14592, 14848, 13312, 13568, 13824, 14080, 14336,
            14592, 14848, 13312, 13568, 13824, 14080, 14336, 14592, 14848, 13312,
            13568, 13824, 14080, 14336, 14592, 14848, 13312, 13568, 13824, 14080,
            14336, 14592, 14848, 13312, 15104,  -8, 15360, -25, 256, 11264,
            15616, 15872, 256, 16128, 16384, 16640, 16896, 17152, 17408, 17664,
            17920, 18176, 256, 18432, 18688, 18944, 19200, 19456, 19712, 19968,
            20224, 20480, 20736, 20992, 21248, 21504, 21760, 22016, 22272, 22528,
            22784, 23040, 23296, 23552, 23808, 24064, 24320, 24576,  -3, 256,
            24832, 25088, 25344,  -9, 15360, 25600,  -4, 256, 25856, -14,
            256, 26112,  -2, 256, 26368, -26, 15360, 26624,  -6, 15360,
             -2, 256, 26880, 27136, 15360, 27392, 27648, 27904, -23, 11264,
            28160,  -4, 11264, 28416, 28672, -33, 15360, 28928, 11264, 29184,
            29440,  -9, 15360, 29696, -15, 15360, 29952, 256, 30208, 30464,
            30720, 30976, 31232, 31488, 31744, 32000, 32256, 32512,  -2, 256,
            32768,  -4, 15360, 33024, 33280, 33536, 33792, 15360, 34048, 34304,
            15360, 34560, 34816, 35072,  -2, 15360, 35328, 35584, 35840, 15360,
            36096, 36352, 36608, 36864, 37120, 37376, 37632, 37888, 38144, 38400,
            38656, 38912,  -3, 39168, 39424, -166, 11264, 39680, -16, 11264,
            39936, 40192, -21, 11264, 40448, -28, 11264, 40704,  -2, 11264,
            40960,  -9, 15360,  -2, 11264, 41216,  -5, 15360, -19, 11264,
            41472, -15, 11264, 41728, -2524, 15360, -16, 41984, -240, 15360,
            42240, 42496, -14, 42752, -240, 15360, -255, 256, 43008, -255,
            256, 43008
        });

        static constexpr auto offset_size = sz_t{ 43264 };
        static constexpr auto offset_pack = std::to_array<si32>(
        {
              1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15, 16,
             17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32,
            -95,  0, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46,
             47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62,
             63, 64, 65, -9,  0, 66, -3,  0, 67, 66,-337,  0,-112, 68, -8,  0,
             -2, 69, -6,  0, -4, 69, -7,  0, 69,  0, 69,-20,  0, 69,-224,  0,
             -7, 68,-166,  0, 69,-38,  0, -2, 69,-50,  0, -2, 69, -3,  0, 69,
            -45, 68,  0, 68,  0, -2, 68,  0, -2, 68,  0, 68, -8, 69,-27,  0,
             -4, 69, -6,  0,-11, 69, -6, 70,-10,  0,-11, 68,  0, 71,-46,  0,
            -21, 68,-16,  0, 68,-101,  0, -7, 68, 70,  0, -6, 68, -2,  0, -2,
             68,  0, -4, 68,-32,  0, 69, 70,  0, 68,-30,  0,-27, 68, -2, 69,
            -89,  0,-11, 68,  0,-14, 69,-43,  0, -9, 68, -7,  0, -2, 69, 68,
            -24,  0, -4, 68,  0, -9, 68,  0, -3, 68,  0, -5, 68, -2, 69,-15,
              0, 69,-25,  0, -3, 68, -2, 69,  0, 69,-11,  0, -5, 69,-31,  0,
             69, -2, 70, -5, 69, -9, 68,-42,  0,-24, 68, 70,-32, 68, 72,-54,
              0, 68, 72, 68,  0, -3, 72, -8, 68, -4, 72, 68, -2, 72,  0, -7,

             68,-10,  0, -2, 68,-29,  0, 68, -2, 72, 69, -8,  0, -2, 69, -2,
              0, -2, 69,-22,  0, 69, -7,  0, 69,  0, -3, 69, -4,  0, -2, 69,
             68,  0, 68, -2, 72, -4, 68, -2, 69, -2, 72, -2, 69, -2, 72, 68,
              0, -8, 69, 68, -4, 69, -2,  0, 69, -3,  0, -2, 68, -2, 69,-24,
              0, 68, -2, 69, -2, 68, 72, 69, -6,  0, -4, 69, -2,  0, -2, 69,
            -22,  0, 69, -7,  0, 69, -2,  0, 69, -2,  0, 69, -2,  0, -2, 69,
             68, 69, -3, 72, -2, 68, -4, 69, -2, 68, -2, 69, -3, 68, -3, 69,
             68, -7, 69, -4,  0, 69,  0, -7, 69,-10,  0, -2, 68, -3,  0, 68,
              0,-10, 69, -2, 68, 72, 69, -9,  0, 69, -3,  0, 69,-22,  0, 69,
             -7,  0, 69, -2,  0, 69, -5,  0, -2, 69, 68,  0, -3, 72, -5, 68,
             69, -2, 68, 72, 69, -2, 72, 68, -2, 69,  0,-15, 69, -2,  0, -2,
             68, -2, 69,-12,  0, -7, 69,  0, -6, 68, 69, 68, -2, 72, 69, -8,
              0, -2, 69, -2,  0, -2, 69,-22,  0, 69, -7,  0, 69, -2,  0, 69,
             -5,  0, -2, 69, 68,  0, -2, 68, 72, -4, 68, -2, 69, -2, 72, -2,
             69, -2, 72, 68, -7, 69, -3, 68, -4, 69, -2,  0, 69, -3,  0, -2,
             68, -2, 69,-18,  0,-10, 69, 68,  0, 69, -6,  0, -3, 69, -3,  0,

             69, -4,  0, -3, 69, -2,  0, 69,  0, 69, -2,  0, -3, 69, -2,  0,
             -3, 69, -3,  0, -3, 69,-12,  0, -4, 69, 68, 72, 68, -2, 72, -3,
             69, -3, 72, 69, -3, 72, 68, -2, 69,  0, -6, 69, 68,-14, 69,-21,
              0, -5, 69, 68, -3, 72, 68, -8,  0, 69, -3,  0, 69,-23,  0, 69,
            -16,  0, -2, 69, 68,  0, -3, 68, -4, 72, 69, -3, 68, 69, -4, 68,
             -7, 69, -2, 68, 69, -3,  0, -2, 69,  0, -2, 69, -2,  0, -2, 68,
             -2, 69,-10,  0, -7, 69,-10,  0, 68, -2, 72, -9,  0, 69, -3,  0,
             69,-23,  0, 69,-10,  0, 69, -5,  0, -2, 69, 68,  0, 72, 68, -2,
             72, 68, -2, 72, 69, 68, -2, 72, 69, -2, 72, -2, 68, -7, 69, -2,
             68, -6, 69, -2,  0, 69, -2,  0, -2, 68, -2, 69,-10,  0, 69, -2,
              0, 72,-12, 69, -2, 68, -2, 72, -9,  0, 69, -3,  0, 69,-41,  0,
             -2, 68,  0, 68, -2, 72, -4, 68, 69, -3, 72, 69, -3, 72, 68, 70,
              0, -4, 69, -3,  0, 68,-10,  0, -2, 68, -2, 69,-26,  0, 69, 68,
             -2, 72, 69,-18,  0, -3, 69,-24,  0, 69, -9,  0, 69,  0, -2, 69,
             -7,  0, -3, 69, 68, -4, 69, 68, -2, 72, -3, 68, 69, 68, 69, -7,
             72, 68, -6, 69,-10,  0, -2, 69, -2, 72,  0,-12, 69,-48,  0, 68,

              0, 73, -7, 68, -4, 69, -8,  0, -8, 68,-13,  0,-37, 69, -2,  0,
             69,  0, 69, -5,  0, 69,-24,  0, 69,  0, 69,-10,  0, 68,  0, 73,
             -9, 68,  0, -2, 69, -5,  0, 69,  0, 69, -7, 68, 69,-10,  0, -2,
             69, -4,  0,-32, 69,-24,  0, -2, 68,-27,  0, 68,  0, 68,  0, 68,
             -4,  0, -2, 72, -8,  0, 69,-36,  0, -4, 69,-14, 68, 72, -5, 68,
              0, -2, 68, -5,  0,-11, 68, 69,-36, 68, 69, -8,  0, 68, -6,  0,
             69,-13,  0,-37, 69,-43,  0, -2, 69, -4, 68, 72, -6, 68, 69, -2,
             68, -2, 72, -2, 68,-23,  0, -2, 72, -2, 68, -4,  0, -3, 68,  0,
             -3, 69, -2,  0, -7, 69, -3,  0, -4, 68,-13,  0, 68, 69, 72, -2,
             68, -6, 69, 68,  0, 69,-10,  0, -3, 69, 68,-40,  0, 69,  0, -5,
             69,  0, -2, 69,-48,  0,-96, 74,-72, 75,-88, 76,-73,  0, 69, -4,
              0, -2, 69, -7,  0, 69,  0, 69, -4,  0, -2, 69,-41,  0, 69, -4,
              0, -2, 69,-33,  0, 69, -4,  0, -2, 69, -7,  0, 69,  0, 69, -4,
              0, -2, 69,-15,  0, 69,-57,  0, 69, -4,  0, -2, 69,-67,  0, -2,
             69, -3, 68,-29,  0, -3, 69,-26,  0, -6, 69,-86,  0, -2, 69, -6,
              0, -2, 69,-157,  0, -3, 69,-89,  0, -7, 69,-18,  0, -4, 68, -9,

             69,-19,  0, -3, 68, -2,  0, -9, 69,-18,  0, -2, 68,-12, 69,-13,
              0, 69, -3,  0, 69, -2, 68,-12, 69,-52,  0, -2, 68, 72, -7, 68,
             -8, 72, 68, -2, 72,-11, 68, -9,  0, 68, -2, 69,-10,  0, -6, 69,
            -10,  0, -6, 69,-11,  0, -3, 68, 77, 68,-10,  0, -6, 69,-89,  0,
             -7, 69, -5,  0, -2, 68,-34,  0, 68,  0, -5, 69,-70,  0,-10, 69,
            -31,  0, 69, -3, 68, -4, 72, -2, 68, -3, 72, -4, 69, -2, 72, 68,
             -6, 72, -3, 68, -4, 69,  0, -3, 69,-42,  0, -2, 69, -5,  0,-11,
             69,-44,  0, -4, 69,-26,  0, -6, 69,-11,  0, -3, 69,-57,  0, -2,
             68, -2, 72, 68, -2, 69,-55,  0, 72, 68, 72, -7, 68, 69, 68, 69,
             68, -2, 69, -8, 68, -6, 72,-10, 68, -2, 69, 68,-10,  0, -6, 69,
            -10,  0, -6, 69,-14,  0, -2, 69,-31, 68,-49, 69, -4, 68, 72,-47,
              0, -7, 68, 72, 68, -5, 72, 68, 72, 68, -8,  0, 69,-29,  0, -9,
             68,-12,  0, -2, 68, 72,-30,  0, 72, -4, 68, -2, 72, -6, 68,-56,
              0, 68, 72, -2, 68, -3, 72, 68, 72, -5, 68, -8, 69,-40,  0, -8,
             72, -8, 68, -2, 72, -2, 68, -3, 69,-15,  0, -3, 69,-62,  0, -5,
             69,-43,  0, -2, 69,-11,  0, -8, 69, -3, 68,  0,-13, 68, 72, -7,

             68, -4,  0, 68, -6,  0, 68, -2,  0, 72, -2, 68,  0, -5, 69,-192,
              0,-64, 68,-22,  0, -2, 69, -6,  0, -2, 69,-38,  0, -2, 69, -6,
              0, -2, 69, -8,  0, 69,  0, 69,  0, 69,  0, 69,-31,  0, -2, 69,
            -53,  0, 69,-15,  0, 69,-14,  0, -2, 69, -6,  0, 69,-19,  0, -2,
             69, -3,  0, 69, -9,  0, 69,-11,  0, -2, 68, 78, 79, 80,-24,  0,
             81, 82, 83, 84, 85, 86, 87,-13,  0, 66,-12,  0, 66,-22,  0, 88,
             89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99,100,101,102,103, -2,
              0, -2, 69,-27,  0, 69,-13,  0, -3, 69,-33,  0,-15, 69,-33, 68,
            -15, 69,-34,  0, 66,-22,  0, 66,-82,  0, -4, 69, -4,  0, -6, 66,
            -15,  0, -2, 66,-111,  0, -2,104,-12,  0, 66, -2,105,-93,  0, 66,
            -70,  0, 66,-25,  0, -4,104, -3, 66,104, -2, 66,104, -4,  0, -3,
             66,-47,  0,-22, 69,-11,  0,-21, 69,-98,  0, 66,-231,  0, -2, 66,
            -10,  0, 66, -9,  0, 66,-58,  0, -2, 66, -2,104,  0, -6, 66,  0,
            -12, 66,  0, -2,104,-26, 66, -8,104,-16, 66,-12,104,-43, 66,104,
             -6, 66, -4,  0, -6,105, -3, 66,104,-13, 66,104, -8, 66, -2,104,
            -17, 66, -2,104, -5, 66, -2,104, -8, 66,104, -5, 66,104,-21, 66,

            104, -7, 66, -2,104, 66,104, -4, 66,104, -2, 66,104, -7, 66,104,
             -2,  0, -2, 66, -2,104, -7, 66,  0, 66,  0, 66, -6,  0, 66, -3,
              0, 66, -6,  0,104,-10,  0, -2, 66,-15,  0, 66, -2,  0, 66, -4,
              0,104,  0,104, -4,  0, -3,104,  0,104,-11,  0, -5, 66,-45,  0,
             -3,104, -9,  0, 66,-14,  0,104,-14,  0,104,-116,  0, -2, 66,-207,
              0, -3, 66,-19,  0, -2,104,-51,  0,104, -4,  0,104,-30,  0, -2,
             69,-32,  0, 69,-344,  0, -3, 68, -2,  0, -5, 69,-45,  0, 69,  0,
             -5, 69,  0, -2, 69,-56,  0, -7, 69, -2,  0,-14, 69, 68,-23,  0,
             -9, 69, -7,  0, 69, -7,  0, 69, -7,  0, 69, -7,  0, 69, -7,  0,
             69, -7,  0, 69, -7,  0, 69, -7,  0, 69,-32, 68,-94,  0,-34, 69,
            -26,105, 69,-89,105,-12, 69,-214,105,-26, 69,-58,105, -6, 68,104,
            -12,105,104,105,  0, 69,-86,105, -2, 69, -2, 68,-101,105, -5, 69,
            -43,105, 69,-94,105, 69,-86,105, -9, 69,-48,105, 69,-40,105, -8,
              0,-71,105,104,105,104,-499,105, -3, 69,-55,105, -9, 69,-92,  0,
            -20, 69,-47,  0, -4, 68,  0,-10, 68,-32,  0, -2, 68,-80,  0, -2,
             68, -6,  0, -8, 69,-206,  0, -2, 69, -2,  0, 69,  0, 69, -8,  0,

            -21, 69,-16,  0, 68, -3,  0, 68, -4,  0, 68,-23,  0, -2, 72, -2,
             68, 72, -4,  0, 68, -3, 69,-10,  0, -6, 69,-56,  0, -8, 69, -2,
             72,-50,  0,-16, 72, -2, 68, -8, 69,-12,  0, -6, 69,-18, 68,-13,
              0, 68,-38,  0, -8, 68,-25,  0,-11, 68, 72, 68,-11, 69,  0,-29,
             74, -3, 69, -3, 68, 72,-47,  0, 68, -2, 72, -4, 68, -2, 72, -2,
             68, -2, 72, 68,-13,  0, 69,-11,  0, -4, 69, -7,  0, 68,-25,  0,
             69,-41,  0, -6, 68, -2, 72, -2, 68, -2, 72, -2, 68, -9, 69, -3,
              0, 68, -8,  0, 68, 72, -2, 69,-10,  0, -2, 69,-31,  0, 69, 68,
             69,-50,  0, 68,  0, -3, 68, -2,  0, -2, 68, -5,  0, -2, 68,  0,
             68,  0,-24, 69,-16,  0, 72, -2, 68, -2, 72, -5,  0, 72, 68,-10,
             69, -6,  0, -2, 69, -6,  0, -2, 69, -6,  0, -9, 69, -7,  0, 69,
             -7,  0, 69,-60,  0, -4, 69,-115,  0, -2, 72, 68, -2, 72, 68, -2,
             72,  0, 72, 68, -2, 69,-10,  0, -6, 69,106,-27,107,106,-27,107,
            106,-27,107,106,-27,107,106,-27,107,106,-27,107,106,-27,107,106,
            -27,107,106,-27,107,106,-27,107,106,-27,107,106,-27,107,106,-27,
            107,106,-27,107,106,-27,107,106,-27,107,106,-27,107,106,-27,107,

            106,-27,107,106,-27,107,106,-27,107,106,-27,107,106,-27,107,106,
            -27,107,106,-27,107,106,-27,107,106,-27,107,106,-27,107,106,-27,
            107,106,-27,107,106,-27,107,106,-27,107,106,-27,107,106,-27,107,
            106,-27,107,106,-27,107,106,-27,107,106,-27,107,106,-27,107,106,
            -27,107,106,-27,107,106,-27,107,106,-27,107,106,-27,107,106,-27,
            107,106,-27,107,106,-27,107,106,-27,107,106,-27,107,106,-27,107,
            106,-27,107,106,-27,107,106,-27,107,106,-27,107,106,-27,107,106,
            -27,107,106,-27,107,106,-27,107,106,-27,107,106,-27,107,106,-27,
            107,106,-27,107,106,-27,107,106,-51,107,106,-27,107,106,-27,107,
            106,-27,107,106,-27,107,106,-27,107,-12, 69,-23, 75, -4, 69,-49,
             76,-260, 69,-110,105, -2, 69,-106,105,-38, 69, -7,  0,-12, 69, -5,
              0, -5, 69,  0, 68,-24,  0, 69, -5,  0, 69,  0, 69, -2,  0, 69,
             -2,  0, 69,-125,  0,-16, 69,-189,  0, -2, 69,-54,  0, -7, 69,  0,
            -32, 69,-16,  0,-16, 68,-10,105, -6, 69,-16, 68,-35,105, 69,-19,
            105, 69, -4,105, -4, 69, -5,  0, 69,-135,  0, -2, 69,108, 69,-96,
            105,-61,  0, -2,109,-31,  0, -3, 69, -6,  0, -2, 69, -6,  0, -2,

             69, -6,  0, -2, 69, -3,  0, -3, 69, -7,105, 69, -7,  0, 69, -9,
             93,110,111,112, -2,  0, -2, 69,-12,  0, 69,-26,  0, 69,-19,  0,
             69, -2,  0, 69,-15,  0, -2, 69,-14,  0,-34, 69,-123,  0, -5, 69,
             -3,  0, -4, 69,-45,  0, -3, 69,-88,  0, 69,-13,  0, -3, 69,  0,
            -47, 69,-45,  0, 68,-130, 69,-29,  0, -3, 69,-49,  0,-15, 69, 68,
            -27,  0, -4, 69,-36,  0, -9, 69,-30,  0, -5, 69,-38,  0, -5, 68,
             -5, 69,-30,  0, 69,-37,  0, -4, 69,-14,  0,-42, 69,-158,  0, -2,
             69,-10,  0, -6, 69,-36,  0, -4, 69,-36,  0, -4, 69,-40,  0, -8,
             69,-52,  0,-11, 69,-12,  0, 69,-15,  0, 69, -7,  0, 69, -2,  0,
             69,-11,  0, 69,-15,  0, 69, -7,  0, 69, -2,  0, -3, 69,-52,  0,
            -12, 69,-55,  0, -9, 69,-22,  0,-10, 69, -8,  0,-24, 69, -6,  0,
             69,-42,  0, 69, -9,  0,-69, 69, -6,  0, -2, 69,  0, 69,-44,  0,
             69, -2,  0, -3, 69,  0, -2, 69,-23,  0, 69,-72,  0, -8, 69, -9,
              0,-48, 69,-19,  0, 69, -2,  0, -5, 69,-33,  0, -3, 69,-27,  0,
             -5, 69,  0,-64, 69,-56,  0, -4, 69,-20,  0, -2, 69,-47,  0, -3,
             68, 69, -2, 68, -5, 69, -4, 68, -4,  0, 69, -3,  0, 69,-29,  0,

             -2, 69, -3, 68, -4, 69, 68, -9,  0, -7, 69, -9,  0, -7, 69,-64,
              0,-32, 69,-37,  0, -2, 68, -4, 69,-12,  0, -9, 69,-54,  0, -3,
             69,-29,  0, -2, 69,-27,  0, -5, 69,-26,  0, -7, 69, -4,  0,-12,
             69, -7,  0,-80, 69,-73,  0,-55, 69,-51,  0,-13, 69,-51,  0, -7,
             69,-42,  0, -4, 68, -8, 69,-10,  0, -6, 69,-38,  0, -3, 69, -5,
             68,-24,  0, -8, 69, -2,  0,-208, 69,-31,  0, 69,-42,  0, 69, -2,
             68,  0, -2, 69, -2,  0,-16, 69, -3,  0,-55, 69, -4, 68,-40,  0,
             -8, 69,-22,  0,-11, 68, -9,  0,-22, 69,-18,  0, -4, 68, -4,  0,
            -38, 69,-28,  0,-20, 69,-23,  0, -9, 69, 72, 68, 72,-53,  0,-15,
             68, -7,  0, -4, 69,-30,  0, 68, -2,  0, -2, 68,  0, -9, 69, -3,
             68, 72,-45,  0, -3, 72, -4, 68, -2, 72, -2, 68, -2,  0, 70, -4,
              0, 68,-10, 69, 70, -2, 69,-25,  0, -7, 69,-10,  0, -6, 69, -3,
             68,-36,  0, -5, 68, 72, -8, 68, 69,-15,  0, -2, 72,  0, -8, 69,
            -35,  0, 68, -3,  0, -9, 69, -2, 68, 72,-48,  0, -3, 72, -9, 68,
             72, 68,  0, -2, 70, -5,  0, -4, 68,  0, 72, 68,-16,  0, 69,-20,
              0,-11, 69,-18,  0, 69,-25,  0, -3, 72, -3, 68, -2, 72, -4, 68,

             -6,  0, 68, -2,  0, 68,-62, 69, -7,  0, 69,  0, 69, -4,  0, 69,
            -15,  0, 69,-11,  0, -6, 69,-47,  0, 68, -3, 72, -8, 68, -5, 69,
            -10,  0, -6, 69, -2, 68, -2, 72, 69, -8,  0, -2, 69, -2,  0, -2,
             69,-22,  0, 69, -7,  0, 69, -2,  0, 69, -5,  0, 69, -2, 68,  0,
             68, 72, 68, -4, 72, -2, 69, -2, 72, -2, 69, -2, 72, 68, -2, 69,
              0, -6, 69, 68, -5, 69, -5,  0, -2, 72, -2, 69, -7, 68, -3, 69,
             -5, 68,-11, 69,-10,  0, 69,  0, -2, 69,  0, 69,-38,  0, 69,  0,
             68, -2, 72, -6, 68, 69, 68, -2, 69, 68, 69, -3, 68, 72, 69, -2,
             72, -3, 68, 70, 68, -3,  0, 69, -2,  0, -8, 69, -2, 68,-29, 69,
            -53,  0, -3, 72, -8, 68, -2, 72, -3, 68, 72, 68,-21,  0, 69,  0,
             68, -3,  0,-30, 69,-48,  0, 68, -2, 72, -6, 68, 72, 68, -2, 72,
             68, 72, -2, 68, 72, -2, 68, -4,  0, -8, 69,-10,  0,-166, 69,-47,
              0, 68, -2, 72, -4, 68, -2, 69, -4, 72, -2, 68, 72, -2, 68,-27,
              0, -2, 68,-34, 69,-48,  0, -3, 72, -8, 68, -2, 72, 68, 72, -2,
             68, -4,  0,-11, 69,-10,  0, -6, 69,-13,  0,-19, 69,-43,  0, 68,
             72, 68, -2, 72, -8, 68, -2,  0, -6, 69,-10,  0, -6, 69,-20,  0,

            -28, 69,-27,  0, -2, 69, 68, 72, 68, -2, 69, -4, 68, 72, -5, 68,
             -4, 69,-23,  0,-185, 69,-44,  0, -3, 72, -9, 68, 72, -2, 68,  0,
            -100, 69,-83,  0,-12, 69, -8,  0, -2, 69,  0, -2, 69, -8,  0, 69,
             -2,  0, 69,-24,  0, 68, -5, 72, 69, -2, 72, -2, 69, -4, 68, 70,
             72, 70, 72, 68, -3,  0, -9, 69,-10,  0,-70, 69, -8,  0, -2, 69,
            -39,  0, -3, 72, -4, 68, -2, 69, -2, 68, -4, 72, 68, -3,  0, 72,
            -27, 69,  0,-10, 68,-40,  0, -6, 68, 72, 70, -4, 68, -8,  0, 68,
             -8, 69,  0, -6, 68, -2, 72, -3, 68,-40,  0, -6, 70,-13, 68, 72,
             -2, 68, -9,  0,-13, 69,-73,  0, -7, 69,-10,  0,-182, 69,-34,  0,
            -14, 69,-10,  0, -6, 69, -9,  0, 69,-37,  0, 72, -7, 68, 69, -6,
             68, 72, 68, -6,  0,-10, 69,-29,  0, -3, 69,-32,  0, -2, 69,-22,
             68, 69, 72, -7, 68, 72, -2, 68, 72, -2, 68,-73, 69, -7,  0, 69,
             -2,  0, 69,-38,  0, -6, 68, -3, 69, 68, 69, -2, 68, 69, -7, 68,
             70, 68, -8, 69,-10,  0, -6, 69, -6,  0, 69, -2,  0, 69,-32,  0,
             -5, 72, 69, -2, 68, 69, -2, 72, 68, 72, 68,  0, -7, 69,-10,  0,
            -310, 69,-19,  0, -2, 68, -2, 72, -2,  0, -7, 69, -2, 68, 70, 72,

            -13,  0, 69,-34,  0, -2, 72, -5, 68, -3, 69, -2, 72, -3, 68,-23,
              0, 68,-85, 69,  0,-15, 69,-50,  0,-13, 69,-155,  0,-102, 69,-111,
              0, 69, -5,  0,-11, 69,-196,  0,-332, 69,-99,  0,-13, 69,-48,  0,
            113,114,115,116,117,118,119,120,121,122,123,124,125,126,127,128,
             68, -6,  0,-15, 68,-10, 69,-411,  0, -5, 69,-71,  0,-185, 69,-30,
              0,-12, 68, -3, 72, -3, 68,-10,  0,-198, 69,-57,  0, -7, 69,-31,
              0, 69,-10,  0, -4, 69,-81,  0, 69,-10,  0, -6, 69,-30,  0, -2,
             69, -5, 68,  0,-10, 69,-48,  0, -7, 68,-15,  0,-10, 69,-10,  0,
             69, -7,  0, 69,-21,  0, -5, 69,-19,  0,-176, 69,-35,  0, 75, -3,
              0, -4, 75,-15,  0,-198, 69,-91,  0,-101, 69,-75,  0, -4, 69, 68,
              0,-55, 72, -7, 69, -4, 68,-13,  0,-64, 69, -4,105, 68,-11, 69,
             -2, 68,-14, 69,-248,105, -8, 69,-214,105,-41, 69,-10,105,-487, 69,
             -4,105, 69, -7,105, 69, -2,105, 69,-35,105,-15, 69,105,-29, 69,
             -3,105, -2, 69,105,-14, 69, -4,105, -8, 69,-396,105, -4, 69,-107,
              0, -5, 69,-13,  0, -3, 69, -9,  0, -7, 69,-10,  0, -2, 69,  0,
             -2, 68,  0,129,130,131,132,-92, 69,-250,  0, -6, 69,-180,  0,-76,

             69,-46, 68, -2, 69,-23, 68, -9, 69,-116,  0,-60, 69,-246,  0,-10,
             69,-39,  0, -2, 69,-60,  0, -5, 68, -3,  0, -6, 68,133,134,135,
            136,137,138,139,140, -8, 68, -2,  0, -7, 68,-30,  0, -4, 68,-61,
              0,-21, 69,-66,  0, -3, 68,  0,-122, 69,-20,  0,-12, 69,-20,  0,
            -12, 69,-87,105, -9, 69,-23,105, -2,  0,-135, 69,-85,  0, 69,-71,
              0, 69, -2,  0, -2, 69,  0, -2, 69, -2,  0, -2, 69, -4,  0, 69,
            -12,  0, 69,  0, 69, -7,  0, 69,-65,  0, 69, -4,  0, -2, 69, -8,
              0, 69, -7,  0, 69,-28,  0, 69, -4,  0, 69, -5,  0, 69,  0, -3,
             69, -7,  0, 69,-340,  0, -2, 69,-292,  0, -2, 69,-50,  0,-55, 68,
             -4,  0,-50, 68, -8,  0, 68,-14,  0, 68, -7,  0,-15, 69, -5, 68,
             69,-15, 68,-80, 69,-31,  0, -6, 69, -6,  0,-213, 69, -7, 68, 69,
            -17, 68, -2, 69, -7, 68, 69, -2, 68, 69, -5, 68, -5, 69,-62,  0,
            -33, 69, 68,-112, 69,-45,  0, -3, 69, -7, 68, -7,  0, -2, 69,-10,
              0, -4, 69, -2,  0,-320, 69,-30,  0, 68,-17, 69,-44,  0, -4, 68,
            -10,  0, -5, 69,  0,-208, 69,-28,  0, -4, 68,-10,  0,-214, 69,-30,
              0, -2, 68,-11,  0, -4, 69,  0,-224, 69, -7,  0, 69, -4,  0, 69,

             -2,  0, 69,-15,  0, 69,-197,  0, -2, 69, -9,  0, -7, 68,-41, 69,
            -68,  0, -7, 68,  0, -4, 69,-10,  0, -4, 69, -2,  0,-273, 69,-68,
              0,-76, 69,-61,  0,-194, 69, -4,  0, 69,-27,  0, 69, -2,  0, 69,
              0, -2, 69,  0, 69,-10,  0, 69, -4,  0, 69,  0, 69,  0, -6, 69,
              0, -4, 69,  0, 69,  0, 69,  0, 69, -3,  0, 69, -2,  0, 69,  0,
             -2, 69,  0, 69,  0, 69,  0, 69,  0, 69,  0, 69, -2,  0, 69,  0,
             -2, 69, -4,  0, 69, -7,  0, 69, -4,  0, 69, -4,  0, 69,  0, 69,
            -10,  0, 69,-17,  0, -5, 69, -3,  0, 69, -5,  0, 69,-17,  0,-52,
             69, -2,  0,-14, 69, -4, 66,104,-39, 66, -4,141,-100, 66,-12,141,
            -15, 66, -2,141,-15, 66,141,-14, 66,104,141,-37, 66,-10,141,-13,
              0, -3, 66,-31,  0, 66,-60,  0, -6, 66,-12,  0, -2, 66,-14,  0,
            104, -2,  0,-10,104,-18,  0, 66,-56,141,-26,142,105, -2,104,-13,
            141,-10,105,104,-20,105,104, -2,105, -9,104,105, -4,141, -9,105,
             -7,141, -2,104,-14,141, -6,104,-154,141,-33,104,-12, 66, -9,104,
             66,-70,104, 66,-22,104,-12, 66,-43,104, -4, 66, -5,104,-12, 66,
            -17,104, -3, 66,104, -3, 66, -3,104, -5,143,-63,104, 66,104, 66,

            -187,104, -2, 66,-63,104, -8,  0, -5, 66, -4,104, 66,-24,104,-18,
             66,104,-26, 66, -2,104,-13, 66,104,-86, 66,-85,104,-48,  0,-70,
            104, -6, 66,104, -3, 66, -3,104, -2, 66, -3,104, -4,141, -4,104,
            -11, 66, -2,104, -3,141, -4, 66, -9,104, -3,141,-116,  0, -3, 66,
             -4,141, -5, 66,-85,  0, -5, 66, -6,141,-12,104, -4,141,104,-15,
            141,-12,  0, -4,141,-56,  0, -8,141,-10,  0, -6,141,-40,  0, -8,
            141,-30,  0, -2,141,-12, 66, -4,141, -2, 66,-62,141,-12,  0,-47,
            104,  0,-10,104,  0,-185,104,-84, 66,-12,141,-14, 66, -2,141,-13,
            104, -3,141,-10,104, -5,141,-56,104, -7,141,-15,104, -2,141,-11,
            104, -6,141, -9,104, -7,141,-147,  0, 69,-102,  0, -6, 69,-510,141,
             -2, 69,-224,105,-32, 69,-58,105, -6, 69,-222,105, -2, 69,-386,105,
            -14, 69,-305,105,-15, 69,-110,105,-162, 69,-30,105,-226, 69,-75,105,
             -5, 69,-352,105,-80, 69,-256, 68,-32, 93,-96, 68,-128, 93,-240, 68,
            -272, 93,-254,  0, -2, 69
        });

        static constexpr auto scripts = std::to_array<ui32>(
        {
            0x4103E6, 0x1A00D7, 0x603E6, 0x1A00D7, 0x2F03E6, 0x100D7, 0xF03E6, 0x100D7, 0x503E6, 0x1700D7, 0x103E6, 0x1F00D7, 0x103E6, 0x1C100D7, 0x2703E6, 0x500D7, 0x503E6, 0x2011D, 0x1403E6, 0x7003E2,
            0x400C8, 0x103E6, 0x300C8, 0x203E7, 0x400C8, 0x103E6, 0x100C8, 0x403E7, 0x100C8, 0x103E6, 0x100C8, 0x103E6, 0x300C8, 0x103E7, 0x100C8, 0x103E7, 0x1400C8, 0x103E7, 0x3F00C8, 0xE00CC,
            0x1000C8, 0x8500DC, 0x203E2, 0xA900DC, 0x103E7, 0x2600E6, 0x203E7, 0x3200E6, 0x203E7, 0x300E6, 0x103E7, 0x37007D, 0x803E7, 0x1B007D, 0x403E7, 0x6007D, 0xB03E7, 0x500A0, 0x103E6, 0x600A0,
            0x103E6, 0xE00A0, 0x103E6, 0x300A0, 0x103E6, 0x2000A0, 0x103E6, 0xA00A0, 0xB03E2, 0x1A00A0, 0x103E2, 0x6C00A0, 0x103E6, 0x2200A0, 0xE0087, 0x103E7, 0x3C0087, 0x203E7, 0x30087, 0x3000A0,
            0x3200AA, 0xE03E7, 0x3B00A5, 0x203E7, 0x300A5, 0x2E007B, 0x203E7, 0xF007B, 0x103E7, 0x1C008C, 0x203E7, 0x1008C, 0x103E7, 0xB0087, 0x503E7, 0x1F00A0, 0x103E7, 0x200A0, 0x503E7, 0x4B00A0,
            0x103E6, 0x1D00A0, 0x51013B, 0x403E2, 0xF013B, 0x203E6, 0x1A013B, 0x40145, 0x103E7, 0x80145, 0x203E7, 0x20145, 0x203E7, 0x160145, 0x103E7, 0x70145, 0x103E7, 0x10145, 0x303E7, 0x40145,
            0x203E7, 0x90145, 0x203E7, 0x20145, 0x203E7, 0x40145, 0x803E7, 0x10145, 0x403E7, 0x20145, 0x103E7, 0x50145, 0x203E7, 0x190145, 0x203E7, 0x30136, 0x103E7, 0x60136, 0x403E7, 0x20136,
            0x203E7, 0x160136, 0x103E7, 0x70136, 0x103E7, 0x20136, 0x103E7, 0x20136, 0x103E7, 0x20136, 0x203E7, 0x10136, 0x103E7, 0x50136, 0x403E7, 0x20136, 0x203E7, 0x30136, 0x303E7, 0x10136,
            0x703E7, 0x40136, 0x103E7, 0x10136, 0x703E7, 0x110136, 0xA03E7, 0x30140, 0x103E7, 0x90140, 0x103E7, 0x30140, 0x103E7, 0x160140, 0x103E7, 0x70140, 0x103E7, 0x20140, 0x103E7, 0x50140,
            0x203E7, 0xA0140, 0x103E7, 0x30140, 0x103E7, 0x30140, 0x203E7, 0x10140, 0xF03E7, 0x40140, 0x203E7, 0xC0140, 0x703E7, 0x70140, 0x103E7, 0x30147, 0x103E7, 0x80147, 0x203E7, 0x20147,
            0x203E7, 0x160147, 0x103E7, 0x70147, 0x103E7, 0x20147, 0x103E7, 0x50147, 0x203E7, 0x90147, 0x203E7, 0x20147, 0x203E7, 0x30147, 0x703E7, 0x30147, 0x403E7, 0x20147, 0x103E7, 0x50147,
            0x203E7, 0x120147, 0xA03E7, 0x2015A, 0x103E7, 0x6015A, 0x303E7, 0x3015A, 0x103E7, 0x4015A, 0x303E7, 0x2015A, 0x103E7, 0x1015A, 0x103E7, 0x2015A, 0x303E7, 0x2015A, 0x303E7, 0x3015A,
            0x303E7, 0xC015A, 0x403E7, 0x5015A, 0x303E7, 0x3015A, 0x103E7, 0x4015A, 0x203E7, 0x1015A, 0x603E7, 0x1015A, 0xE03E7, 0x15015A, 0x503E7, 0xD0154, 0x103E7, 0x30154, 0x103E7, 0x170154,
            0x103E7, 0x100154, 0x203E7, 0x90154, 0x103E7, 0x30154, 0x103E7, 0x40154, 0x703E7, 0x20154, 0x103E7, 0x30154, 0x203E7, 0x10154, 0x203E7, 0x40154, 0x203E7, 0xA0154, 0x703E7, 0x90154,
            0xD0159, 0x103E7, 0x30159, 0x103E7, 0x170159, 0x103E7, 0xA0159, 0x103E7, 0x50159, 0x203E7, 0x90159, 0x103E7, 0x30159, 0x103E7, 0x40159, 0x703E7, 0x20159, 0x603E7, 0x20159, 0x103E7,
            0x40159, 0x203E7, 0xA0159, 0x103E7, 0x30159, 0xC03E7, 0xD015B, 0x103E7, 0x3015B, 0x103E7, 0x33015B, 0x103E7, 0x3015B, 0x103E7, 0x6015B, 0x403E7, 0x10015B, 0x203E7, 0x1A015B, 0x103E7,
            0x3015C, 0x103E7, 0x12015C, 0x303E7, 0x18015C, 0x103E7, 0x9015C, 0x103E7, 0x1015C, 0x203E7, 0x7015C, 0x303E7, 0x1015C, 0x403E7, 0x6015C, 0x103E7, 0x1015C, 0x103E7, 0x8015C, 0x603E7,
            0xA015C, 0x203E7, 0x3015C, 0xC03E7, 0x3A0160, 0x403E7, 0x103E6, 0x1C0160, 0x2503E7, 0x20164, 0x103E7, 0x10164, 0x103E7, 0x50164, 0x103E7, 0x180164, 0x103E7, 0x10164, 0x103E7, 0x170164,
            0x203E7, 0x50164, 0x103E7, 0x10164, 0x103E7, 0x70164, 0x103E7, 0xA0164, 0x203E7, 0x40164, 0x2003E7, 0x48014A, 0x103E7, 0x24014A, 0x403E7, 0x27014A, 0x103E7, 0x24014A, 0x103E7, 0xF014A,
            0x103E7, 0x7014A, 0x403E6, 0x2014A, 0x2503E7, 0xA0015E, 0x2600F0, 0x103E7, 0x100F0, 0x503E7, 0x100F0, 0x203E7, 0x2B00F0, 0x103E6, 0x400F0, 0x100011E, 0x4901AE, 0x103E7, 0x401AE, 0x203E7,
            0x701AE, 0x103E7, 0x101AE, 0x103E7, 0x401AE, 0x203E7, 0x2901AE, 0x103E7, 0x401AE, 0x203E7, 0x2101AE, 0x103E7, 0x401AE, 0x203E7, 0x701AE, 0x103E7, 0x101AE, 0x103E7, 0x401AE, 0x203E7,
            0xF01AE, 0x103E7, 0x3901AE, 0x103E7, 0x401AE, 0x203E7, 0x4301AE, 0x203E7, 0x2001AE, 0x303E7, 0x1A01AE, 0x603E7, 0x5601BD, 0x203E7, 0x601BD, 0x203E7, 0x28001B8, 0x1D00D4, 0x303E7, 0x4B00D3,
            0x303E6, 0xB00D3, 0x703E7, 0x160172, 0x903E7, 0x10172, 0x150173, 0x203E6, 0x903E7, 0x140174, 0xC03E7, 0xD0175, 0x103E7, 0x30175, 0x103E7, 0x20175, 0xC03E7, 0x5E0163, 0x203E7, 0xA0163,
            0x603E7, 0xA0163, 0x603E7, 0x20091, 0x203E6, 0x10091, 0x103E6, 0x140091, 0x603E7, 0x590091, 0x703E7, 0x2B0091, 0x503E7, 0x4601B8, 0xA03E7, 0x1F0150, 0x103E7, 0xC0150, 0x403E7, 0xC0150,
            0x403E7, 0x10150, 0x303E7, 0xC0150, 0x1E0161, 0x203E7, 0x50161, 0xB03E7, 0x2C0162, 0x403E7, 0x1A0162, 0x603E7, 0xB0162, 0x303E7, 0x20162, 0x200163, 0x1C016F, 0x203E7, 0x2016F, 0x3F015F,
            0x103E7, 0x1D015F, 0x203E7, 0xB015F, 0x603E7, 0xA015F, 0x603E7, 0xE015F, 0x203E7, 0x1F03E2, 0x3103E7, 0x4D0168, 0x103E7, 0x320168, 0x40016A, 0x34016D, 0x803E7, 0x4016D, 0x38014F, 0x303E7,
            0xF014F, 0x303E7, 0x3014F, 0x300105, 0xB00DC, 0x503E7, 0x2B00F0, 0x203E7, 0x300F0, 0x8016A, 0x803E7, 0x303E2, 0x103E6, 0xD03E2, 0x103E6, 0x703E2, 0x403E6, 0x103E2, 0x603E6, 0x103E2,
            0x303E6, 0x203E2, 0x103E6, 0x503E7, 0x2600D7, 0x500C8, 0x100DC, 0x3100D7, 0x500C8, 0x400D7, 0x500C8, 0xD00D7, 0x100DC, 0x4600D7, 0x100C8, 0x4003E2, 0x10000D7, 0x1600C8, 0x203E7, 0x600C8,
            0x203E7, 0x2600C8, 0x203E7, 0x600C8, 0x203E7, 0x800C8, 0x103E7, 0x100C8, 0x103E7, 0x100C8, 0x103E7, 0x100C8, 0x103E7, 0x1F00C8, 0x203E7, 0x3500C8, 0x103E7, 0xF00C8, 0x103E7, 0xE00C8,
            0x203E7, 0x600C8, 0x103E7, 0x1300C8, 0x203E7, 0x300C8, 0x103E7, 0x900C8, 0x103E7, 0xC03E6, 0x203E2, 0x5703E6, 0x103E7, 0xB03E6, 0x100D7, 0x203E7, 0xB03E6, 0x100D7, 0xF03E6, 0x103E7,
            0xD00D7, 0x303E7, 0x2103E6, 0xF03E7, 0x2103E2, 0xF03E7, 0x2603E6, 0x100C8, 0x303E6, 0x200D7, 0x603E6, 0x100D7, 0x1B03E6, 0x100D7, 0x1103E6, 0x2900D7, 0x303E6, 0x403E7, 0x29A03E6, 0x1603E7,
            0xB03E6, 0x1503E7, 0x3A003E6, 0x100023A, 0x27403E6, 0x203E7, 0x2003E6, 0x103E7, 0x6903E6, 0x6000E1, 0x2000D7, 0x7400CC, 0x503E7, 0x700CC, 0x2600F0, 0x103E7, 0x100F0, 0x503E7, 0x100F0, 0x203E7,
            0x380078, 0x703E7, 0x20078, 0xE03E7, 0x10078, 0x1701AE, 0x903E7, 0x701AE, 0x103E7, 0x701AE, 0x103E7, 0x701AE, 0x103E7, 0x701AE, 0x103E7, 0x701AE, 0x103E7, 0x701AE, 0x103E7, 0x701AE,
            0x103E7, 0x701AE, 0x103E7, 0x2000DC, 0x5E03E6, 0x2203E7, 0x1A01F4, 0x103E7, 0x5901F4, 0xC03E7, 0xD601F4, 0x1A03E7, 0x1503E6, 0x101F4, 0x103E6, 0x101F4, 0x1903E6, 0x901F4, 0x403E2, 0x2011E,
            0x803E6, 0x401F4, 0x403E6, 0x103E7, 0x56019A, 0x203E7, 0x203E2, 0x203E6, 0x3019A, 0x103E6, 0x5A019B, 0x203E6, 0x3019B, 0x503E7, 0x2B011D, 0x103E7, 0x5E011E, 0x103E7, 0x1003E6, 0x20011D,
            0x2603E6, 0x903E7, 0x103E6, 0x10019B, 0x1F011E, 0x103E7, 0x4003E6, 0x1F011E, 0x5103E6, 0x2F019B, 0x103E6, 0x58019B, 0xA803E6, 0x19C001F4, 0x4003E6, 0x520001F4, 0x48D01CC, 0x303E7, 0x3701CC, 0x903E7,
            0x30018F, 0x12C01D6, 0x1403E7, 0x6000DC, 0x5801B3, 0x803E7, 0x2203E6, 0x6600D7, 0x303E6, 0x4300D7, 0x203E7, 0x200D7, 0x103E7, 0x100D7, 0x103E7, 0x800D7, 0x1503E7, 0xE00D7, 0x2D013C, 0x303E7,
            0xA03E6, 0x603E7, 0x38014B, 0x803E7, 0x460158, 0x803E7, 0xC0158, 0x603E7, 0x20013B, 0x2E0165, 0x103E6, 0x10165, 0x24016B, 0xB03E7, 0x1016B, 0x1D011E, 0x303E7, 0x4E0169, 0x103E7, 0x103E6,
            0xA0169, 0x403E7, 0x20169, 0x1F015E, 0x103E7, 0x370166, 0x903E7, 0xE0166, 0x203E7, 0xA0166, 0x203E7, 0x40166, 0x20015E, 0x430167, 0x1803E7, 0x50167, 0x170151, 0xA03E7, 0x601AE, 0x203E7,
            0x601AE, 0x203E7, 0x601AE, 0x903E7, 0x701AE, 0x103E7, 0x701AE, 0x103E7, 0x2B00D7, 0x103E6, 0x900D7, 0x100C8, 0x400D7, 0x203E6, 0x403E7, 0x5001BD, 0x2E0151, 0x203E7, 0xA0151, 0x603E7,
            0x2BA4011E, 0xC03E7, 0x17011E, 0x403E7, 0x31011E, 0x210403E7, 0x16E01F4, 0x203E7, 0x6A01F4, 0x2603E7, 0x700D7, 0xC03E7, 0x500E6, 0x503E7, 0x1A007D, 0x103E7, 0x5007D, 0x103E7, 0x1007D, 0x103E7,
            0x2007D, 0x103E7, 0x2007D, 0x103E7, 0xA007D, 0x7300A0, 0x1003E7, 0x16B00A0, 0x203E6, 0x5000A0, 0x203E7, 0x3600A0, 0x703E7, 0x100A0, 0x2003E7, 0x1000A0, 0x1003E2, 0xA03E6, 0x603E7, 0xE03E2,
            0x200DC, 0x2303E6, 0x103E7, 0x1303E6, 0x103E7, 0x403E6, 0x403E7, 0x500A0, 0x103E7, 0x8700A0, 0x203E7, 0x103E6, 0x103E7, 0x2003E6, 0x1A00D7, 0x603E6, 0x1A00D7, 0xB03E6, 0xA019B, 0x103E6,
            0x2D019B, 0x203E6, 0x1F011E, 0x303E7, 0x6011E, 0x203E7, 0x6011E, 0x203E7, 0x6011E, 0x203E7, 0x3011E, 0x303E7, 0x703E6, 0x103E7, 0x703E6, 0xA03E7, 0x503E6, 0x203E7, 0xC0191, 0x103E7,
            0x1A0191, 0x103E7, 0x130191, 0x103E7, 0x20191, 0x103E7, 0xF0191, 0x203E7, 0xE0191, 0x2203E7, 0x7B0191, 0x503E7, 0x303E6, 0x403E7, 0x2D03E6, 0x303E7, 0x903E6, 0x4F00C8, 0x103E7, 0xD03E6,
            0x303E7, 0x100C8, 0x2F03E7, 0x2D03E6, 0x103E2, 0x8203E7, 0x1D00CA, 0x303E7, 0x3100C9, 0xF03E7, 0x103E2, 0x1B03E6, 0x403E7, 0x2400D2, 0x903E7, 0x300D2, 0x1B00CE, 0x503E7, 0x2B00E3, 0x503E7,
            0x1E0028, 0x103E7, 0x10028, 0x24001E, 0x403E7, 0xE001E, 0x2A03E7, 0x5000FA, 0x300119, 0x1E0104, 0x203E7, 0xA0104, 0x603E7, 0x2400DB, 0x403E7, 0x2400DB, 0x403E7, 0x2800E2, 0x803E7, 0x3400EF,
            0xB03E7, 0x100EF, 0xB00E4, 0x103E7, 0xF00E4, 0x103E7, 0x700E4, 0x103E7, 0x200E4, 0x103E7, 0xB00E4, 0x103E7, 0xF00E4, 0x103E7, 0x700E4, 0x103E7, 0x200E4, 0x303E7, 0x3400E5, 0xC03E7,
            0x1370190, 0x903E7, 0x160190, 0xA03E7, 0x80190, 0x1803E7, 0x600D7, 0x103E7, 0x2A00D7, 0x103E7, 0x900D7, 0x4503E7, 0x60193, 0x203E7, 0x10193, 0x103E7, 0x2C0193, 0x103E7, 0x20193, 0x303E7,
            0x10193, 0x203E7, 0x10193, 0x16007C, 0x103E7, 0x9007C, 0x20007E, 0x1F009F, 0x803E7, 0x9009F, 0x3003E7, 0x13007F, 0x103E7, 0x2007F, 0x503E7, 0x5007F, 0x1C0073, 0x303E7, 0x10073, 0x1A0074,
            0x503E7, 0x10074, 0x4003E7, 0x200064, 0x180065, 0x403E7, 0x140065, 0x203E7, 0x2E0065, 0x40131, 0x103E7, 0x20131, 0x503E7, 0x80131, 0x103E7, 0x30131, 0x103E7, 0x1D0131, 0x203E7, 0x30131,
            0x403E7, 0xA0131, 0x703E7, 0x90131, 0x703E7, 0x200069, 0x20006A, 0x2003E7, 0x27008B, 0x403E7, 0xC008B, 0x903E7, 0x360086, 0x303E7, 0x70086, 0x160082, 0x203E7, 0x80082, 0x130083, 0x503E7,
            0x80083, 0x120084, 0x703E7, 0x40084, 0xC03E7, 0x70084, 0x5003E7, 0x4900AF, 0x3703E7, 0x3300B0, 0xD03E7, 0x3300B0, 0x703E7, 0x600B0, 0x2800A7, 0x803E7, 0xA00A7, 0x603E7, 0x2600A4, 0x303E7,
            0x1D00A4, 0x803E7, 0x200A4, 0xD003E7, 0x1F00A0, 0x103E7, 0x2A00C0, 0x103E7, 0x300C0, 0x203E7, 0x200C0, 0x1003E7, 0x300A0, 0x3703E7, 0x400A0, 0x28008E, 0x803E7, 0x2A008D, 0x1603E7, 0x1A008F,
            0x2603E7, 0x1C006D, 0x1403E7, 0x170080, 0x903E7, 0x4E012C, 0x403E7, 0x24012C, 0x903E7, 0x1012C, 0x43013D, 0xA03E7, 0x1013D, 0x203E7, 0x19018E, 0x703E7, 0xA018E, 0x603E7, 0x35015D, 0x103E7,
            0x12015D, 0x803E7, 0x27013A, 0x903E7, 0x60013F, 0x103E7, 0x14015C, 0xB03E7, 0x120142, 0x103E7, 0x2F0142, 0x3E03E7, 0x70143, 0x103E7, 0x10143, 0x103E7, 0x40143, 0x103E7, 0xF0143, 0x103E7,
            0xB0143, 0x603E7, 0x3B013E, 0x503E7, 0xA013E, 0x603E7, 0x40157, 0x103E7, 0x80157, 0x203E7, 0x20157, 0x203E7, 0x160157, 0x103E7, 0x70157, 0x103E7, 0x20157, 0x103E7, 0x50157, 0x103E7,
            0x103E2, 0x90157, 0x203E7, 0x20157, 0x203E7, 0x30157, 0x203E7, 0x10157, 0x603E7, 0x10157, 0x503E7, 0x70157, 0x203E7, 0x70157, 0x303E7, 0x50157, 0xB03E7, 0xA0155, 0x103E7, 0x10155,
            0x203E7, 0x10155, 0x103E7, 0x260155, 0x103E7, 0xA0155, 0x103E7, 0x10155, 0x203E7, 0x10155, 0x103E7, 0x40155, 0x103E7, 0xA0155, 0x103E7, 0x20155, 0x803E7, 0x20155, 0x1D03E7, 0x5C014D,
            0x103E7, 0x5014D, 0x1E03E7, 0x480146, 0x803E7, 0xA0146, 0xA603E7, 0x36012E, 0x203E7, 0x26012E, 0x2203E7, 0x450144, 0xB03E7, 0xA0144, 0x603E7, 0xD0091, 0x1303E7, 0x3A0141, 0x603E7, 0xA0141,
            0x603E7, 0x14015E, 0x1C03E7, 0x1B0152, 0x203E7, 0xF0152, 0x403E7, 0x170152, 0xB903E7, 0x3C0148, 0x6403E7, 0x530106, 0xC03E7, 0x10106, 0x70156, 0x203E7, 0x10156, 0x203E7, 0x80156, 0x103E7,
            0x20156, 0x103E7, 0x1E0156, 0x103E7, 0x20156, 0x203E7, 0xC0156, 0x903E7, 0xA0156, 0x4603E7, 0x80137, 0x203E7, 0x2E0137, 0x203E7, 0xB0137, 0x1B03E7, 0x480153, 0x803E7, 0x530149, 0xD03E7,
            0x1001B8, 0x390107, 0x703E7, 0xA013B, 0xB603E7, 0x220112, 0xE03E7, 0xA0112, 0x603E7, 0x9014E, 0x103E7, 0x2D014E, 0x103E7, 0xE014E, 0xA03E7, 0x1D014E, 0x303E7, 0x20014C, 0x203E7, 0x16014C,
            0x103E7, 0xE014C, 0x4903E7, 0x70139, 0x103E7, 0x20139, 0x103E7, 0x2C0139, 0x303E7, 0x10139, 0x103E7, 0x20139, 0x103E7, 0x90139, 0x803E7, 0xA0139, 0x603E7, 0x60138, 0x103E7, 0x20138,
            0x103E7, 0x250138, 0x103E7, 0x20138, 0x103E7, 0x60138, 0x703E7, 0xA0138, 0x13603E7, 0x19016E, 0x703E7, 0x110170, 0x103E7, 0x290170, 0x303E7, 0x1D0170, 0x5503E7, 0x1018F, 0xF03E7, 0x32015A,
            0xD03E7, 0x1015A, 0x39A0014, 0x6603E7, 0x6F0014, 0x103E7, 0x50014, 0xB03E7, 0xC40014, 0xA4C03E7, 0x630192, 0xD03E7, 0x4560032, 0xA03E7, 0xF9B0032, 0x503E7, 0x2470050, 0x1AB903E7, 0x3A018D, 0x6C603E7,
            0x23901B3, 0x703E7, 0x1F0108, 0x103E7, 0xA0108, 0x403E7, 0x20108, 0x4F0113, 0x103E7, 0xA0113, 0x603E7, 0x1E0103, 0x203E7, 0x60103, 0xA03E7, 0x4601C2, 0xA03E7, 0xA01C2, 0x103E7, 0x701C2,
            0x103E7, 0x1501C2, 0x503E7, 0x1301C2, 0x1B003E7, 0x3A018C, 0xC603E7, 0x5B0109, 0x6503E7, 0x4B011A, 0x403E7, 0x39011A, 0x703E7, 0x11011A, 0x4003E7, 0x10208, 0x101F3, 0x201F4, 0x10120, 0xB03E7,
            0x201F4, 0xE03E7, 0x17F80208, 0x803E7, 0x3000208, 0x1D60120, 0x2903E7, 0x10120, 0x90208, 0x22E703E7, 0x4019B, 0x103E7, 0x7019B, 0x103E7, 0x2019B, 0x103E7, 0x1019B, 0x11F019A, 0x3019B, 0xF03E7,
            0x1019A, 0x1D03E7, 0x3019A, 0x203E7, 0x1019B, 0xE03E7, 0x4019B, 0x803E7, 0x18C01F3, 0x90403E7, 0x6B02F3, 0x503E7, 0xD02F3, 0x303E7, 0x902F3, 0x703E7, 0xA02F3, 0x203E7, 0x402F3, 0x403E6,
            0xF5C03E7, 0xFA03E6, 0x603E7, 0x1B403E6, 0x4C03E7, 0x2E03E2, 0x203E7, 0x1703E2, 0x903E7, 0x7403E6, 0x3C03E7, 0xF603E6, 0xA03E7, 0x2703E6, 0x203E7, 0x3E03E6, 0x303E2, 0x1103E6, 0x803E2, 0x203E6,
            0x703E2, 0x1E03E6, 0x403E2, 0x3D03E6, 0x1503E7, 0x4600C8, 0x7A03E7, 0x1403E6, 0xC03E7, 0x1403E6, 0xC03E7, 0x5703E6, 0x903E7, 0x1903E6, 0x8703E7, 0x5503E6, 0x103E7, 0x4703E6, 0x103E7, 0x203E6,
            0x203E7, 0x103E6, 0x203E7, 0x203E6, 0x203E7, 0x403E6, 0x103E7, 0xC03E6, 0x103E7, 0x103E6, 0x103E7, 0x703E6, 0x103E7, 0x4103E6, 0x103E7, 0x403E6, 0x203E7, 0x803E6, 0x103E7, 0x703E6,
            0x103E7, 0x1C03E6, 0x103E7, 0x403E6, 0x103E7, 0x503E6, 0x103E7, 0x103E6, 0x303E7, 0x703E6, 0x103E7, 0x15403E6, 0x203E7, 0x12403E6, 0x203E7, 0x3203E6, 0x28C005F, 0xF03E7, 0x5005F, 0x103E7,
            0xF005F, 0x45003E7, 0x1F00D7, 0x603E7, 0x600D7, 0xD503E7, 0x700E1, 0x103E7, 0x1100E1, 0x203E7, 0x700E1, 0x103E7, 0x200E1, 0x103E7, 0x500E1, 0x503E7, 0x3E00DC, 0x2103E7, 0x100DC, 0x7003E7,
            0x2D01C3, 0x303E7, 0xE01C3, 0x203E7, 0xA01C3, 0x403E7, 0x201C3, 0x14003E7, 0x1F0126, 0x1103E7, 0x3A011B, 0x503E7, 0x1011B, 0x1D003E7, 0x2A0127, 0xD603E7, 0x2B0128, 0x403E7, 0x10128, 0x1E003E7,
            0x701AE, 0x103E7, 0x401AE, 0x103E7, 0x201AE, 0x103E7, 0xF01AE, 0x103E7, 0xC501B6, 0x203E7, 0x1001B6, 0x2903E7, 0x4C00A6, 0x403E7, 0xA00A6, 0x403E7, 0x200A6, 0x31103E7, 0x4403E6, 0x4C03E7,
            0x3D03E6, 0xC203E7, 0x400A0, 0x103E7, 0x1B00A0, 0x103E7, 0x200A0, 0x103E7, 0x100A0, 0x203E7, 0x100A0, 0x103E7, 0xA00A0, 0x103E7, 0x400A0, 0x103E7, 0x100A0, 0x103E7, 0x100A0, 0x603E7,
            0x100A0, 0x403E7, 0x100A0, 0x103E7, 0x100A0, 0x103E7, 0x100A0, 0x103E7, 0x300A0, 0x103E7, 0x200A0, 0x103E7, 0x100A0, 0x203E7, 0x100A0, 0x103E7, 0x100A0, 0x103E7, 0x100A0, 0x103E7,
            0x100A0, 0x103E7, 0x100A0, 0x103E7, 0x200A0, 0x103E7, 0x100A0, 0x203E7, 0x400A0, 0x103E7, 0x700A0, 0x103E7, 0x400A0, 0x103E7, 0x400A0, 0x103E7, 0x100A0, 0x103E7, 0xA00A0, 0x103E7,
            0x1100A0, 0x503E7, 0x300A0, 0x103E7, 0x500A0, 0x103E7, 0x1100A0, 0x3403E7, 0x200A0, 0x10E03E7, 0x2C03E6, 0x403E7, 0x6403E6, 0xC03E7, 0xF03E6, 0x203E7, 0xF03E6, 0x103E7, 0xF03E6, 0x103E7,
            0x2503E6, 0xA03E7, 0xAE03E6, 0x3803E7, 0x1A03E6, 0x1019A, 0x203E6, 0xD03E7, 0x2C03E6, 0x403E7, 0x903E6, 0x703E7, 0x203E6, 0xE03E7, 0x603E6, 0x9A03E7, 0x3D803E6, 0x403E7, 0x1103E6, 0x303E7,
            0xD03E6, 0x303E7, 0x7703E6, 0x403E7, 0x5F03E6, 0x603E7, 0xC03E6, 0x403E7, 0x103E6, 0xF03E7, 0xC03E6, 0x403E7, 0x3803E6, 0x803E7, 0xA03E6, 0x603E7, 0x2803E6, 0x803E7, 0x1E03E6, 0x203E7,
            0xC03E6, 0x403E7, 0x203E6, 0x3E03E7, 0x15403E6, 0xC03E7, 0xE03E6, 0x203E7, 0xD03E6, 0x303E7, 0xA03E6, 0x503E7, 0x3803E6, 0x703E7, 0xF03E6, 0x203E7, 0xB03E6, 0x603E7, 0x903E6, 0x703E7,
            0x9303E6, 0x103E7, 0x6603E6, 0x40603E7, 0xA6E001F4, 0x2003E7, 0x103A01F4, 0x603E7, 0xDE01F4, 0x203E7, 0x168201F4, 0xE03E7, 0x1D3101F4, 0xF03E7, 0x26E01F4, 0x9A203E7, 0x21E01F4, 0x5E203E7, 0x134B01F4, 0x503E7,
            0x106001F4, 0xFFFF03E7, 0xFFFF03E7, 0xFFFF03E7, 0xFFFF03E7, 0xFFFF03E7, 0xFFFF03E7, 0xFFFF03E7, 0xFFFF03E7, 0xFFFF03E7, 0xFFFF03E7, 0xDC5B03E7, 0x103E6, 0x1E03E7, 0x6003E6, 0x8003E7, 0xF003E2, 0xFFFF03E7, 0xFFFF03E7, 0x3E7,
            0xFE12, 
        });

        static constexpr auto ucspec = std::to_array<unidata>(
        {
            { widths::slim, gbreak::any , cntrls::non_control            },  //   0
            { widths::zero, gbreak::ctrl, cntrls::non_control            },  //   1
            { widths::zero, gbreak::ctrl, cntrls::soh                    },  //   2
            { widths::zero, gbreak::ctrl, cntrls::stx                    },  //   3
            { widths::zero, gbreak::ctrl, cntrls::etx                    },  //   4
            { widths::zero, gbreak::ctrl, cntrls::eot                    },  //   5
            { widths::zero, gbreak::ctrl, cntrls::enq                    },  //   6
            { widths::zero, gbreak::ctrl, cntrls::ack                    },  //   7
            { widths::zero, gbreak::ctrl, cntrls::bel                    },  //   8
            { widths::zero, gbreak::ctrl, cntrls::bs                     },  //   9
            { widths::zero, gbreak::ctrl, cntrls::tab                    },  //  10
            { widths::zero, gbreak::lf  , cntrls::eol                    },  //  11
            { widths::zero, gbreak::ctrl, cntrls::vt                     },  //  12
            { widths::zero, gbreak::ctrl, cntrls::ff                     },  //  13
            { widths::zero, gbreak::cr  , cntrls::cr                     },  //  14
            { widths::zero, gbreak::ctrl, cntrls::so                     },  //  15
            { widths::zero, gbreak::ctrl, cntrls::si                     },  //  16
            { widths::zero, gbreak::ctrl, cntrls::dle                    },  //  17
            { widths::zero, gbreak::ctrl, cntrls::dc1                    },  //  18
            { widths::zero, gbreak::ctrl, cntrls::dc2                    },  //  19
            { widths::zero, gbreak::ctrl, cntrls::dc3                    },  //  20
            { widths::zero, gbreak::ctrl, cntrls::dc4                    },  //  21
            { widths::zero, gbreak::ctrl, cntrls::nak                    },  //  22
            { widths::zero, gbreak::ctrl, cntrls::syn                    },  //  23
            { widths::zero, gbreak::ctrl, cntrls::etb                    },  //  24
            { widths::zero, gbreak::ctrl, cntrls::can                    },  //  25
            { widths::zero, gbreak::ctrl, cntrls::em                     },  //  26
            { widths::zero, gbreak::ctrl, cntrls::sub                    },  //  27
            { widths::zero, gbreak::ctrl, cntrls::esc                    },  //  28
            { widths::zero, gbreak::ctrl, cntrls::fs                     },  //  29
            { widths::zero, gbreak::ctrl, cntrls::gs                     },  //  30
            { widths::zero, gbreak::ctrl, cntrls::rs                     },  //  31
            { widths::zero, gbreak::ctrl, cntrls::us                     },  //  32
            { widths::zero, gbreak::ctrl, cntrls::del                    },  //  33
            { widths::zero, gbreak::ctrl, cntrls::pad                    },  //  34
            { widths::zero, gbreak::ctrl, cntrls::hop                    },  //  35
            { widths::zero, gbreak::ctrl, cntrls::bph                    },  //  36
            { widths::zero, gbreak::ctrl, cntrls::nbh                    },  //  37
            { widths::zero, gbreak::ctrl, cntrls::ind                    },  //  38
            { widths::zero, gbreak::ctrl, cntrls::nel                    },  //  39
            { widths::zero, gbreak::ctrl, cntrls::ssa                    },  //  40
            { widths::zero, gbreak::ctrl, cntrls::esa                    },  //  41
            { widths::zero, gbreak::ctrl, cntrls::hts                    },  //  42
            { widths::zero, gbreak::ctrl, cntrls::htj                    },  //  43
            { widths::zero, gbreak::ctrl, cntrls::vts                    },  //  44
            { widths::zero, gbreak::ctrl, cntrls::pld                    },  //  45
            { widths::zero, gbreak::ctrl, cntrls::plu                    },  //  46
            { widths::zero, gbreak::ctrl, cntrls::ri                     },  //  47
            { widths::zero, gbreak::ctrl, cntrls::ss2                    },  //  48
            { widths::zero, gbreak::ctrl, cntrls::ss3                    },  //  49
            { widths::zero, gbreak::ctrl, cntrls::dcs                    },  //  50
            { widths::zero, gbreak::ctrl, cntrls::pu1                    },  //  51
            { widths::zero, gbreak::ctrl, cntrls::pu2                    },  //  52
            { widths::zero, gbreak::ctrl, cntrls::sts                    },  //  53
            { widths::zero, gbreak::ctrl, cntrls::cch                    },  //  54
            { widths::zero, gbreak::ctrl, cntrls::mw                     },  //  55
            { widths::zero, gbreak::ctrl, cntrls::spa                    },  //  56
            { widths::zero, gbreak::ctrl, cntrls::epa                    },  //  57
            { widths::zero, gbreak::ctrl, cntrls::sos                    },  //  58
            { widths::zero, gbreak::ctrl, cntrls::sgc                    },  //  59
            { widths::zero, gbreak::ctrl, cntrls::sci                    },  //  60
            { widths::zero, gbreak::ctrl, cntrls::csi                    },  //  61
            { widths::zero, gbreak::ctrl, cntrls::st                     },  //  62
            { widths::zero, gbreak::ctrl, cntrls::osc                    },  //  63
            { widths::zero, gbreak::ctrl, cntrls::pm                     },  //  64
            { widths::zero, gbreak::ctrl, cntrls::apc                    },  //  65
            { widths::slim, gbreak::ep  , cntrls::non_control            },  //  66
            { widths::zero, gbreak::ctrl, cntrls::shy                    },  //  67
            { widths::zero, gbreak::ext , cntrls::non_control            },  //  68
            { widths::zero, gbreak::any , cntrls::non_control            },  //  69
            { widths::slim, gbreak::prep, cntrls::non_control            },  //  70
            { widths::zero, gbreak::ctrl, cntrls::alm                    },  //  71
            { widths::zero, gbreak::sm  , cntrls::non_control            },  //  72
            { widths::slim, gbreak::sm  , cntrls::non_control            },  //  73
            { widths::wide, gbreak::l   , cntrls::non_control            },  //  74
            { widths::slim, gbreak::v   , cntrls::non_control            },  //  75
            { widths::slim, gbreak::t   , cntrls::non_control            },  //  76
            { widths::zero, gbreak::ctrl, cntrls::mvs                    },  //  77
            { widths::zero, gbreak::zwj , cntrls::non_control            },  //  78
            { widths::zero, gbreak::ctrl, cntrls::lrm                    },  //  79
            { widths::zero, gbreak::ctrl, cntrls::rlm                    },  //  80
            { widths::zero, gbreak::ctrl, cntrls::line_separator         },  //  81
            { widths::zero, gbreak::ctrl, cntrls::paragraph_separator    },  //  82
            { widths::zero, gbreak::ctrl, cntrls::lre                    },  //  83
            { widths::zero, gbreak::ctrl, cntrls::rle                    },  //  84
            { widths::zero, gbreak::ctrl, cntrls::pdf                    },  //  85
            { widths::zero, gbreak::ctrl, cntrls::lro                    },  //  86
            { widths::zero, gbreak::ctrl, cntrls::rlo                    },  //  87
            { widths::zero, gbreak::ctrl, cntrls::wj                     },  //  88
            { widths::zero, gbreak::ctrl, cntrls::function_application   },  //  89
            { widths::zero, gbreak::ctrl, cntrls::invisible_times        },  //  90
            { widths::zero, gbreak::ctrl, cntrls::invisible_separator    },  //  91
            { widths::zero, gbreak::ctrl, cntrls::invisible_plus         },  //  92
            { widths::zero, gbreak::ctrl, cntrls::non_control            },  //  93
            { widths::zero, gbreak::ctrl, cntrls::lri                    },  //  94
            { widths::zero, gbreak::ctrl, cntrls::rli                    },  //  95
            { widths::zero, gbreak::ctrl, cntrls::fsi                    },  //  96
            { widths::zero, gbreak::ctrl, cntrls::pdi                    },  //  97
            { widths::zero, gbreak::ctrl, cntrls::inhibit_symmetric_swapping },  //  98
            { widths::zero, gbreak::ctrl, cntrls::activate_symmetric_swapping },  //  99
            { widths::zero, gbreak::ctrl, cntrls::inhibit_arabic_form_shaping },  // 100
            { widths::zero, gbreak::ctrl, cntrls::activate_arabic_form_shaping },  // 101
            { widths::zero, gbreak::ctrl, cntrls::national_digit_shapes  },  // 102
            { widths::zero, gbreak::ctrl, cntrls::nominal_digit_shapes   },  // 103
            { widths::wide, gbreak::ep  , cntrls::non_control            },  // 104
            { widths::wide, gbreak::any , cntrls::non_control            },  // 105
            { widths::wide, gbreak::lv  , cntrls::non_control            },  // 106
            { widths::wide, gbreak::lvt , cntrls::non_control            },  // 107
            { widths::zero, gbreak::ctrl, cntrls::zwnbsp                 },  // 108
            { widths::slim, gbreak::ext , cntrls::non_control            },  // 109
            { widths::zero, gbreak::ctrl, cntrls::interlinear_annotation_anchor },  // 110
            { widths::zero, gbreak::ctrl, cntrls::interlinear_annotation_separator },  // 111
            { widths::zero, gbreak::ctrl, cntrls::interlinear_annotation_terminator },  // 112
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_vertical_joiner },  // 113
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_horizontal_joiner },  // 114
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_insert_at_top_start },  // 115
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_insert_at_bottom_start },  // 116
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_insert_at_top_end },  // 117
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_insert_at_bottom_end },  // 118
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_overlay_middle },  // 119
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_begin_segment },  // 120
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_end_segment },  // 121
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_insert_at_middle },  // 122
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_insert_at_top },  // 123
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_insert_at_bottom },  // 124
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_begin_enclosure },  // 125
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_end_enclosure },  // 126
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_begin_walled_enclosure },  // 127
            { widths::zero, gbreak::ctrl, cntrls::egyptian_hieroglyph_end_walled_enclosure },  // 128
            { widths::zero, gbreak::ctrl, cntrls::shorthand_format_letter_overlap },  // 129
            { widths::zero, gbreak::ctrl, cntrls::shorthand_format_continuing_overlap },  // 130
            { widths::zero, gbreak::ctrl, cntrls::shorthand_format_down_step },  // 131
            { widths::zero, gbreak::ctrl, cntrls::shorthand_format_up_step },  // 132
            { widths::zero, gbreak::ctrl, cntrls::musical_symbol_begin_beam },  // 133
            { widths::zero, gbreak::ctrl, cntrls::musical_symbol_end_beam },  // 134
            { widths::zero, gbreak::ctrl, cntrls::musical_symbol_begin_tie },  // 135
            { widths::zero, gbreak::ctrl, cntrls::musical_symbol_end_tie },  // 136
            { widths::zero, gbreak::ctrl, cntrls::musical_symbol_begin_slur },  // 137
            { widths::zero, gbreak::ctrl, cntrls::musical_symbol_end_slur },  // 138
            { widths::zero, gbreak::ctrl, cntrls::musical_symbol_begin_phrase },  // 139
            { widths::zero, gbreak::ctrl, cntrls::musical_symbol_end_phrase },  // 140
            { widths::zero, gbreak::ep  , cntrls::non_control            },  // 141
            { widths::slim, gbreak::ri  , cntrls::non_control            },  // 142
            { widths::wide, gbreak::ext , cntrls::non_control            },  // 143
        });
    };

    template<class T, class D>
    auto unpack(D const& pack, sz_t size)
    {
        auto data = std::vector<T>{};
        data.reserve(size);
        auto iter = pack.begin();
        auto tail = pack.end();
        while (iter != tail)
        {
            auto n = *iter++;
            if (n < 0) data.insert(data.end(), -n, static_cast<T>(*iter++));
            else       data.push_back(static_cast<T>(n));
        }
        return data;
    }
    unidata const& select(ui32 cp)
    {
        static auto offset = unpack<byte>(base::offset_pack, base::offset_size);
        static auto blocks = unpack<ui16>(base::blocks_pack, base::blocks_size);
        return cp > 0x10FFFF ? base::ucspec[0]
                             : base::ucspec[offset[blocks[cp >> 8] + (cp & 0xFF)]];
    }
    auto script(ui32 cp)
    {
        static auto data = []
        {
            auto v = std::vector<ui16>(0x10FFFF); // ISO 15924 Script No: 0 - 999.
            auto j = 0;
            auto s = 0;
            auto l = 0;
            for (auto i = 0u; i < v.size(); i++)
            {
                if (l == 0)
                {
                    s = base::scripts[j] & 0xFFFF;
                    l = base::scripts[j] >> 16;
                    j++;
                }
                v[i] = (ui16)s;
                l--;
            }
            return v;
        }();
        return cp > 0x10FFFF ? data[0] : data[cp];
    }
}