
///////////////////////////////////////////////////////////
//                                                       //
//                         SAGA                          //
//                                                       //
//      System for Automated Geoscientific Analyses      //
//                                                       //
//                     Tool Library                      //
//                       pj_proj4                        //
//                                                       //
//-------------------------------------------------------//
//                                                       //
//                 crs_transform_utm.cpp                 //
//                                                       //
//                 Copyright (C) 2017 by                 //
//                      Olaf Conrad                      //
//                                                       //
//-------------------------------------------------------//
//                                                       //
// This file is part of 'SAGA - System for Automated     //
// Geoscientific Analyses'. SAGA is free software; you   //
// can redistribute it and/or modify it under the terms  //
// of the GNU General Public License as published by the //
// Free Software Foundation, either version 2 of the     //
// License, or (at your option) any later version.       //
//                                                       //
// SAGA is distributed in the hope that it will be       //
// useful, but WITHOUT ANY WARRANTY; without even the    //
// implied warranty of MERCHANTABILITY or FITNESS FOR A  //
// PARTICULAR PURPOSE. See the GNU General Public        //
// License for more details.                             //
//                                                       //
// You should have received a copy of the GNU General    //
// Public License along with this program; if not, see   //
// <http://www.gnu.org/licenses/>.                       //
//                                                       //
//-------------------------------------------------------//
//                                                       //
//    e-mail:     oconrad@saga-gis.org                   //
//                                                       //
//    contact:    Olaf Conrad                            //
//                Institute of Geography                 //
//                University of Hamburg                  //
//                Germany                                //
//                                                       //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
#include "crs_transform_utm.h"


///////////////////////////////////////////////////////////
//                                                       //
//                                                       //
//                                                       //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
class CUTM_Zones
{
public:

	enum
	{
		DATUM_FIRST = 0,
		DATUM_WGS84 = DATUM_FIRST,
		DATUM_ETRS89,
		DATUM_NAD27,
		DATUM_NAD83,
		DATUM_COUNT
	 };

	//-----------------------------------------------------
	static const char *	Get_Datum_Name		(int DatumID)
	{
		const char *Datums[DATUM_COUNT] = {
			"WGS84",
			"ETRS89",
			"NAD27",
			"NAD83"
		};

		if( DatumID >= DATUM_FIRST && DatumID < DATUM_COUNT )
		{
			return( Datums[DatumID] );
		}

		return( "" );
	}

	//-----------------------------------------------------
	static CSG_String	Get_Datum_Choices	(void)
	{
		CSG_String Datums;

		for(int i=DATUM_FIRST; i<DATUM_COUNT; i++)
		{
			if( i > 0 ) { Datums += "|"; } Datums += Get_Datum_Name(i);
		}

		return( Datums );
	}

	//-----------------------------------------------------
	static bool			Get_Zone			(const CSG_Rect &Extent, const CSG_Projection &Source, int &Zone, bool &bSouth)
	{
		CSG_CRSProjector Projector; CSG_Point Point(Extent.Get_Center());

		if( Projector.Set_Transformation(Source, CSG_Projection::Get_GCS_WGS84()) && Projector.Get_Projection(Point) )
		{
			if( Point.x < -180. )
			{
				Point.x = 360. + fmod(Point.x, 360.);
			}

			Zone   = 1 + (int)fmod(floor((Point.x + 180.) / 6.), 60.);

			bSouth = Point.y < 0.;

			return( true );
		}

		return( false );
	}

	//-----------------------------------------------------
	static CSG_Projection	Get_Projection	(int Zone, bool bSouth, int DatumID)
	{
		CSG_Projection Projection;

		if( Zone >= 1 && Zone <= 60 )
		{
			CSG_String Datum;

			switch( DatumID )
			{
			default: Projection.Set_UTM_WGS84(Zone, bSouth); break;

			//-------------------------------------------------
			case DATUM_ETRS89: Datum = "+ellps=GRS80"; if( Zone >= 28 && Zone <= 38 ) { Projection.Create(25800 + Zone); } break;
			case DATUM_NAD27 : Datum = "+datum=NAD27"; if( Zone >=  1 && Zone <= 23 ) { Projection.Create(26700 + Zone); } break;
			case DATUM_NAD83 : Datum = "+datum=NAD83"; if( Zone >=  1 && Zone <= 23 ) { Projection.Create(26900 + Zone); } break;
			}

			if( Projection.is_Okay() == false )
			{
				Projection.Create(CSG_String::Format("+proj=utm +zone=%d%s %s",
					Zone, bSouth ? SG_T(" +south") : SG_T(""), Datum.c_str()
				));
			}
		}

		return( Projection );
	}
};


///////////////////////////////////////////////////////////
//                                                       //
//                                                       //
//                                                       //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
CCRS_Transform_UTM_Grids::CCRS_Transform_UTM_Grids(bool bList)
	: CCRS_Transform_Grid(bList)
{
	Set_Name       (CSG_String::Format("%s (%s)", _TL("UTM Projection"),
		bList ? _TL("Grid List") : _TL("Grid")
	));

	Set_Author     ("O.Conrad (c) 2017");

	Set_Description(_TW(
		"Project grids to UTM coordinates."
	));

	Set_Description(Get_Description() + "\n" + CSG_CRSProjector::Get_Description());

	//-----------------------------------------------------
	Parameters.Add_Choice("",
		"DATUM"    , _TL("Datum"),
		_TL(""),
		CUTM_Zones::Get_Datum_Choices(), CUTM_Zones::DATUM_WGS84
	);

	Parameters.Add_Int("",
		"UTM_ZONE" , _TL("Zone"),
		_TL(""),
		1, 1, true, 60, true
	);

	Parameters.Add_Bool("",
		"UTM_SOUTH", _TL("South"),
		_TL(""),
		false
	);

	//-----------------------------------------------------
	Parameters.Set_Enabled("CRS_STRING", false);
}


///////////////////////////////////////////////////////////
//                                                       //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
int CCRS_Transform_UTM_Grids::On_Parameter_Changed(CSG_Parameters *pParameters, CSG_Parameter *pParameter)
{
	if( pParameter->Cmp_Identifier("SOURCE") )
	{
		int Zone; bool bSouth; CSG_Data_Object *pObject = pParameter->is_DataObject() ? pParameter->asDataObject() : pParameter->asList()->Get_Item(0);

		if( pObject )
		{
			CSG_Grid *pGrid = pObject->Get_ObjectType() == SG_DATAOBJECT_TYPE_Grid ? pObject->asGrid() : pObject->asGrids()->Get_Grid_Ptr(0);

			if( CUTM_Zones::Get_Zone(pGrid->Get_Extent(), pGrid->Get_Projection(), Zone, bSouth) )
			{
				CSG_Projection UTM = CUTM_Zones::Get_Projection(Zone, bSouth, (*pParameters)("DATUM")->asInt());

				pParameters->Set_Parameter("UTM_ZONE" , Zone  );
				pParameters->Set_Parameter("UTM_SOUTH", bSouth);
				pParameters->Set_Parameter("CRS_WKT"  , UTM.Get_WKT2());
				pParameters->Set_Parameter("CRS_PROJ" , UTM.Get_PROJ());
			}
		}
	}

	//-----------------------------------------------------
	if( pParameter->Cmp_Identifier("UTM_ZONE" )
	||  pParameter->Cmp_Identifier("UTM_SOUTH")
	||  pParameter->Cmp_Identifier("DATUM"    ) )
	{
		CSG_Projection	UTM	= CUTM_Zones::Get_Projection(
			(*pParameters)("UTM_ZONE" )->asInt (),
			(*pParameters)("UTM_SOUTH")->asBool(),
			(*pParameters)("DATUM"    )->asInt()
		);

		pParameters->Set_Parameter("CRS_WKT" , UTM.Get_WKT2());
		pParameters->Set_Parameter("CRS_PROJ", UTM.Get_PROJ());
	}

	//-----------------------------------------------------
	return( CCRS_Transform_Grid::On_Parameter_Changed(pParameters, pParameter) );
}

//---------------------------------------------------------
int CCRS_Transform_UTM_Grids::On_Parameters_Enable(CSG_Parameters *pParameters, CSG_Parameter *pParameter)
{
	if( pParameter->Cmp_Identifier("DATUM") )
	{
		pParameters->Set_Enabled("UTM_SOUTH", pParameter->asInt() != 1);
	}

	return( CCRS_Transform_Grid::On_Parameters_Enable(pParameters, pParameter) );
}


///////////////////////////////////////////////////////////
//                                                       //
//                                                       //
//                                                       //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
CCRS_Transform_UTM_Shapes::CCRS_Transform_UTM_Shapes(bool bList)
	: CCRS_Transform_Shapes(bList)
{
	Set_Name       (CSG_String::Format("%s (%s)", _TL("UTM Projection"),
		bList ? _TL("Shapes List") : _TL("Shapes")
	));

	Set_Author     ("O. Conrad (c) 2017");

	Set_Description(_TW(
		"Project shapes into UTM coordinates."
	));

	Set_Description(Get_Description() + "\n" + CSG_CRSProjector::Get_Description());

	//-----------------------------------------------------
	Parameters.Add_Choice("",
		"DATUM"    , _TL("Datum"),
		_TL(""),
		CUTM_Zones::Get_Datum_Choices(), CUTM_Zones::DATUM_WGS84
	);

	Parameters.Add_Int("",
		"UTM_ZONE" , _TL("Zone"),
		_TL(""),
		1, 1, true, 60, true
	);

	Parameters.Add_Bool("",
		"UTM_SOUTH", _TL("South"),
		_TL(""),
		false
	);

	//-----------------------------------------------------
	Parameters.Set_Enabled("CRS_STRING", false);
}


///////////////////////////////////////////////////////////
//                                                       //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
int CCRS_Transform_UTM_Shapes::On_Parameter_Changed(CSG_Parameters *pParameters, CSG_Parameter *pParameter)
{
	if( pParameter->Cmp_Identifier("SOURCE") )
	{
		int Zone; bool bSouth; CSG_Shapes *pObject = (CSG_Shapes *)(pParameter->is_DataObject() ? pParameter->asDataObject() : pParameter->asShapesList()->Get_Item(0));

		if( pObject && CUTM_Zones::Get_Zone(pObject->Get_Extent(), pObject->Get_Projection(), Zone, bSouth) )
		{
			CSG_Projection UTM = CUTM_Zones::Get_Projection(Zone, bSouth, (*pParameters)("DATUM")->asInt());

			pParameters->Set_Parameter("UTM_ZONE" , Zone  );
			pParameters->Set_Parameter("UTM_SOUTH", bSouth);
			pParameters->Set_Parameter("CRS_WKT"  , UTM.Get_WKT2());
			pParameters->Set_Parameter("CRS_PROJ" , UTM.Get_PROJ());
		}
	}

	//-----------------------------------------------------
	if( pParameter->Cmp_Identifier("UTM_ZONE" )
	||  pParameter->Cmp_Identifier("UTM_SOUTH")
	||  pParameter->Cmp_Identifier("DATUM"    ) )
	{
		CSG_Projection UTM = CUTM_Zones::Get_Projection(
			(*pParameters)("UTM_ZONE" )->asInt (),
			(*pParameters)("UTM_SOUTH")->asBool(),
			(*pParameters)("DATUM"    )->asInt ()
		);

		pParameters->Set_Parameter("CRS_WKT" , UTM.Get_WKT2());
		pParameters->Set_Parameter("CRS_PROJ", UTM.Get_PROJ());
	}

	//-----------------------------------------------------
	return( CCRS_Transform_Shapes::On_Parameter_Changed(pParameters, pParameter) );
}

//---------------------------------------------------------
int CCRS_Transform_UTM_Shapes::On_Parameters_Enable(CSG_Parameters *pParameters, CSG_Parameter *pParameter)
{
	if( pParameter->Cmp_Identifier("DATUM") )
	{
		pParameters->Set_Enabled("UTM_SOUTH", pParameter->asInt() != 1);
	}

	return( CCRS_Transform_Shapes::On_Parameters_Enable(pParameters, pParameter) );
}


///////////////////////////////////////////////////////////
//                                                       //
//                                                       //
//                                                       //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
