/**
 *    Copyright (C) 2021 Graham Leggett <minfrin@sharp.fm>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

/*
 * Common utility functions for all tools.
 */

#ifndef DEVICE_UTIL_H
#define DEVICE_UTIL_H

#include <apr_pools.h>

#include <redwax-tool.h>

int redwax_certcmp(redwax_certificate_t *cert1, redwax_certificate_t *cert2);

apr_size_t redwax_strrcspn(const char *s, const char *charset);

const char *redwax_stroff(const char *s, apr_size_t off);

const char *redwax_pstrntrim(apr_pool_t *p, const char *s, apr_size_t off);

const char *redwax_pescape_echo_quoted(apr_pool_t *p, const char *str,
        redwax_token_quoted_e quoted, int close);

apr_status_t redwax_urlescape_path(char *escaped,
        const char *str, apr_ssize_t slen, apr_size_t *len);

const char* redwax_pescape_path(apr_pool_t *p, const char *str);

apr_status_t redwax_urlescape_all(char *escaped,
        const char *str, apr_ssize_t slen, apr_size_t *len);

const char* redwax_purlescape_all(apr_pool_t *p, const char *str);

apr_status_t redwax_escape_entity(char *escaped, const char *str,
        apr_ssize_t slen, int toasc, apr_size_t *len);

const char* apr_pescape_entity(apr_pool_t *p, const char *str, int toasc);

apr_status_t redwax_escape_json(char *escaped, const char *str,
        apr_ssize_t slen, apr_size_t *len);

const char *redwax_pescape_json(apr_pool_t *p, const char *str);

apr_status_t redwax_encode_base16_binary(char *dest, const unsigned char *src,
        apr_ssize_t slen, int flags, apr_size_t *len);

const char* redwax_pencode_base16_binary(apr_pool_t *p,
        const unsigned char *src, apr_ssize_t slen, int flags, apr_size_t *len);

apr_status_t redwax_strip_whitespace(char *escaped, const char *str,
        apr_size_t slen, apr_size_t *len);

const char* apr_pstrip_whitespace(apr_pool_t *p, const char *str, apr_size_t slen);

#define REDWAX_ESCAPE_STRING -1

/**
 * Generate RFC4648 base16/base32/base64.
 */
#define REDWAX_ENCODE_NONE 0

/**
 * Generate base16 with colons between each token.
 */
#define REDWAX_ENCODE_COLON 16

/**
 * Generate base16 with lower case characters.
 */
#define REDWAX_ENCODE_LOWER 32

/**
 * Keep track of nesting level.
 */
typedef struct redwax_metadata_level_t {
    void *k;
    int klen;
    unsigned int root:1;
    unsigned int object:1;
    unsigned int array:1;
    unsigned int empty:1;
    unsigned int next:1;
} redwax_metadata_level_t;

/**
 * A structure to hold metadata context.
 */
typedef struct redwax_metadata_t {
    apr_pool_t *pool;
    apr_array_header_t *levels;
    redwax_metadata_level_t *level;
    apr_status_t (*wv)(void *ctx, const struct iovec *vec,
                    apr_size_t nvec);
    void *ctx;
    redwax_format_e format;
    char *object_prefix;
    char *array_prefix;
    int prefix_len;
} redwax_metadata_t;

apr_status_t redwax_metadata_push_root(apr_pool_t *pool, const char *k,
        apr_status_t (*wv)(void *ctx, const struct iovec *vec, apr_size_t nvec),
        void *ctx, redwax_format_e format, redwax_metadata_t **mm);

apr_status_t redwax_metadata_pop_root(redwax_metadata_t *m);

apr_status_t redwax_metadata_push_array(redwax_metadata_t *m, const char *k, int empty);

apr_status_t redwax_metadata_pop_array(redwax_metadata_t *m);

apr_status_t redwax_metadata_push_object(redwax_metadata_t *m, const char *k, int empty);

apr_status_t redwax_metadata_pop_object(redwax_metadata_t *m);

apr_status_t redwax_metadata_add_string(redwax_metadata_t *m, const char *k, const char *v);

apr_status_t redwax_metadata_add_number(redwax_metadata_t *m, const char *key,
        void *v, apr_size_t vlen);

apr_status_t redwax_metadata_add_long(redwax_metadata_t *m, const char *key, long val);

apr_status_t redwax_metadata_add_boolean(redwax_metadata_t *m, const char *key, int val);

apr_status_t redwax_metadata_add_null(redwax_metadata_t *m, const char *key);

#endif
