// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { nullTranslator } from '@jupyterlab/translation';
import { DisposableDelegate } from '@lumino/disposable';
import { Signal } from '@lumino/signaling';
/**
 * Search provider registry
 */
export class SearchProviderRegistry {
    /**
     * Constructor
     *
     * @param translator Application translator object
     */
    constructor(translator = nullTranslator) {
        this.translator = translator;
        this._changed = new Signal(this);
        this._providerMap = new Map();
    }
    /**
     * Add a provider to the registry.
     *
     * @param key - The provider key.
     * @returns A disposable delegate that, when disposed, deregisters the given search provider
     */
    add(key, provider) {
        this._providerMap.set(key, provider);
        this._changed.emit();
        return new DisposableDelegate(() => {
            this._providerMap.delete(key);
            this._changed.emit();
        });
    }
    /**
     * Returns a matching provider for the widget.
     *
     * @param widget - The widget to search over.
     * @returns the search provider, or undefined if none exists.
     */
    getProvider(widget) {
        // iterate through all providers and ask each one if it can search on the
        // widget.
        for (const P of this._providerMap.values()) {
            if (P.isApplicable(widget)) {
                return P.createNew(widget, this.translator);
            }
        }
        return undefined;
    }
    /**
     * Whether the registry as a matching provider for the widget.
     *
     * @param widget - The widget to search over.
     * @returns Provider existence
     */
    hasProvider(widget) {
        for (const P of this._providerMap.values()) {
            if (P.isApplicable(widget)) {
                return true;
            }
        }
        return false;
    }
    /**
     * Signal that emits when a new search provider has been registered
     * or removed.
     */
    get changed() {
        return this._changed;
    }
}
//# sourceMappingURL=searchproviderregistry.js.map