package test_core_crypto

import ec "core:crypto/_weierstrass"
import "core:encoding/hex"
import "core:math/big"
import "core:testing"

@(private="file")
P256_G_X : string : "6b17d1f2e12c4247f8bce6e563a440f277037d812deb33a0f4a13945d898c296"
@(private="file")
P256_G_Y : string : "4fe342e2fe1a7f9b8ee7eb4a7c0f9e162bce33576b315ececbb6406837bf51f5"
@(private="file")
P256_G_UNCOMPRESSED :: "04" + P256_G_X + P256_G_Y

@(private="file")
P384_G_X : string : "aa87ca22be8b05378eb1c71ef320ad746e1d3b628ba79b9859f741e082542a385502f25dbf55296c3a545e3872760ab7"
@(private="file")
P384_G_Y : string : "3617de4a96262c6f5d9e98bf9292dc29f8f41dbd289a147ce9da3113b5f0b8c00a60b1ce1d7e819d7a431d7c90ea0e5f"

@(test)
test_p256_a :: proc(t: ^testing.T) {
	a_str := "ffffffff00000001000000000000000000000000fffffffffffffffffffffffc"

	fe, a_fe: ec.Field_Element_p256r1
	ec.fe_a(&fe)
	ec.fe_a(&a_fe)

	b: [32]byte
	ec.fe_bytes(b[:], &fe)

	s := (string)(hex.encode(b[:], context.temp_allocator))

	testing.expect(t, s == a_str)

	b_, _ := hex.decode(transmute([]byte)(a_str), context.temp_allocator)
	ec.fe_zero(&fe)
	ec.fe_set_bytes(&fe, b_)

	testing.expectf(t, ec.fe_equal(&fe, &a_fe) == 1, "%v", &fe)
}

@(test)
test_p384_a :: proc(t: ^testing.T) {
	a_str := "fffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffeffffffff0000000000000000fffffffc"
	fe, a_fe: ec.Field_Element_p384r1
	ec.fe_a(&fe)
	ec.fe_a(&a_fe)

	b: [48]byte
	ec.fe_bytes(b[:], &fe)

	s := (string)(hex.encode(b[:], context.temp_allocator))

	testing.expect(t, s == a_str)

	b_, _ := hex.decode(transmute([]byte)(a_str), context.temp_allocator)
	ec.fe_zero(&fe)
	ec.fe_set_bytes(&fe, b_)

	testing.expectf(t, ec.fe_equal(&fe, &a_fe) == 1, "%v", &fe)
}

@(test)
test_p256_b :: proc(t: ^testing.T) {
	b_str := "5ac635d8aa3a93e7b3ebbd55769886bc651d06b0cc53b0f63bce3c3e27d2604b"

	fe, b_fe: ec.Field_Element_p256r1
	ec.fe_b(&fe)
	ec.fe_b(&b_fe)

	b: [32]byte
	ec.fe_bytes(b[:], &fe)

	s := (string)(hex.encode(b[:], context.temp_allocator))

	testing.expect(t, s == b_str)

	b_, _ := hex.decode(transmute([]byte)(b_str), context.temp_allocator)
	ec.fe_zero(&fe)
	ec.fe_set_bytes(&fe, b_)

	testing.expectf(t, ec.fe_equal(&fe, &b_fe) == 1, "%v", &fe)
}

@(test)
test_p384_b :: proc(t: ^testing.T) {
	b_str := "b3312fa7e23ee7e4988e056be3f82d19181d9c6efe8141120314088f5013875ac656398d8a2ed19d2a85c8edd3ec2aef"
	fe, b_fe: ec.Field_Element_p384r1
	ec.fe_b(&fe)
	ec.fe_b(&b_fe)

	b: [48]byte
	ec.fe_bytes(b[:], &fe)

	s := (string)(hex.encode(b[:], context.temp_allocator))

	testing.expect(t, s == b_str)

	b_, _ := hex.decode(transmute([]byte)(b_str), context.temp_allocator)
	ec.fe_zero(&fe)
	ec.fe_set_bytes(&fe, b_)

	testing.expectf(t, ec.fe_equal(&fe, &b_fe) == 1, "%v", &fe)
}

@(test)
test_p256_g_x :: proc(t: ^testing.T) {
	fe, x_fe: ec.Field_Element_p256r1
	ec.fe_gen_x(&fe)
	ec.fe_gen_x(&x_fe)

	b: [32]byte
	ec.fe_bytes(b[:], &fe)

	s := (string)(hex.encode(b[:], context.temp_allocator))
	testing.expect(t, s == P256_G_X)

	b_, _ := hex.decode(transmute([]byte)(P256_G_X), context.temp_allocator)
	ec.fe_zero(&fe)
	ec.fe_set_bytes(&fe, b_)

	testing.expectf(t, ec.fe_equal(&fe, &x_fe) == 1, "%v", &fe)
}

@(test)
test_p384_g_x :: proc(t: ^testing.T) {
	fe, x_fe: ec.Field_Element_p384r1
	ec.fe_gen_x(&fe)
	ec.fe_gen_x(&x_fe)

	b: [48]byte
	ec.fe_bytes(b[:], &fe)

	s := (string)(hex.encode(b[:], context.temp_allocator))
	testing.expect(t, s == P384_G_X)

	b_, _ := hex.decode(transmute([]byte)(P384_G_X), context.temp_allocator)
	ec.fe_zero(&fe)
	ec.fe_set_bytes(&fe, b_)

	testing.expectf(t, ec.fe_equal(&fe, &x_fe) == 1, "%v", &fe)
}

@(test)
test_p256_g_y :: proc(t: ^testing.T) {
	fe, y_fe: ec.Field_Element_p256r1
	ec.fe_gen_y(&fe)
	ec.fe_gen_y(&y_fe)

	b: [32]byte
	ec.fe_bytes(b[:], &fe)

	s := (string)(hex.encode(b[:], context.temp_allocator))
	testing.expect(t, s == P256_G_Y)

	b_, _ := hex.decode(transmute([]byte)(P256_G_Y), context.temp_allocator)
	ec.fe_zero(&fe)
	ec.fe_set_bytes(&fe, b_)

	testing.expectf(t, ec.fe_equal(&fe, &y_fe) == 1, "%v", &fe)
}

@(test)
test_p384_g_y :: proc(t: ^testing.T) {
	fe, y_fe: ec.Field_Element_p384r1
	ec.fe_gen_y(&fe)
	ec.fe_gen_y(&y_fe)

	b: [48]byte
	ec.fe_bytes(b[:], &fe)

	s := (string)(hex.encode(b[:], context.temp_allocator))
	testing.expect(t, s == P384_G_Y)

	b_, _ := hex.decode(transmute([]byte)(P384_G_Y), context.temp_allocator)
	ec.fe_zero(&fe)
	ec.fe_set_bytes(&fe, b_)

	testing.expectf(t, ec.fe_equal(&fe, &y_fe) == 1, "%v", &fe)
}

@(test)
test_p256_scalar_reduce :: proc(t: ^testing.T) {
	test_vectors := []struct {
		raw:     string,
		reduced: string,
	} {
		// n
		{
			"ffffffff00000000ffffffffffffffffbce6faada7179e84f3b9cac2fc632551",
			"0000000000000000000000000000000000000000000000000000000000000000",
		},
		// n + 1
		{
			"ffffffff00000000ffffffffffffffffbce6faada7179e84f3b9cac2fc632552",
			"0000000000000000000000000000000000000000000000000000000000000001",
		},
		// 2^384 (Sage)
		{
			"01000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000",
			"431905529c0166ce652e96b7ccca0a99679b73e19ad16947f01cf013fc632551",
		},
		// SHA384 odin-linux-amd64-dev-2026-01.tar.gz (Sage)
		{
			"87622f79b4b0e76001f9c99b0337b61a0bcd2b5a8e9a3937176825ad75ef0fe8742a348a251dd2682d711f76b33df3e6",
			"f66db86e28d903033d1e17d818c0eb13fe3d1ae095b4d2ecbcd1a1eccf9f2f8c",
		},
		// SHA512 odin-linux-amd64-dev-2026-01.tar.gz (Sage)
		{
			"6f85507cec3a35fdb3d4f40d23583681144561e77bc4ea88ab0ea219d5c17b7c9178f5f5a6296a2d18eddd4bdf19e61830fc85d7de23fd4fbde31c4cf6694719",
			"3217ecbee32c8b0dfcca0f10a884fe43658fbe91458f25d0f1bf2075759c5ebe",
		},
	}

	for v, _ in test_vectors {
		raw_bytes, _ := hex.decode(transmute([]byte)(v.raw), context.temp_allocator)

		sc: ec.Scalar_p256r1
		_ = ec.sc_set_bytes(&sc, raw_bytes)

		b: [ec.SC_SIZE_P256R1]byte
		ec.sc_bytes(b[:], &sc)
		s := (string)(hex.encode(b[:], context.temp_allocator))

		testing.expectf(t, v.reduced == s, "sc: raw %s reduced: %s, expected: %s", v.raw, s, v.reduced)
	}
}

@(test)
test_p384_scalar_reduce :: proc(t: ^testing.T) {
	test_vectors := []struct {
		raw:     string,
		reduced: string,
	} {
		// n
		{
			"ffffffffffffffffffffffffffffffffffffffffffffffffc7634d81f4372ddf581a0db248b0a77aecec196accc52973",
			"000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000",
		},
		// n + 1
		{
			"ffffffffffffffffffffffffffffffffffffffffffffffffc7634d81f4372ddf581a0db248b0a77aecec196accc52974",
			"000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001",
		},
		// SHA384 odin-linux-amd64-dev-2026-01.tar.gz
		{
			"87622f79b4b0e76001f9c99b0337b61a0bcd2b5a8e9a3937176825ad75ef0fe8742a348a251dd2682d711f76b33df3e6",
			"87622f79b4b0e76001f9c99b0337b61a0bcd2b5a8e9a3937176825ad75ef0fe8742a348a251dd2682d711f76b33df3e6",
		},
		// SHA512 odin-linux-amd64-dev-2026-01.tar.gz (Sage)
		{
			"6f85507cec3a35fdb3d4f40d23583681144561e77bc4ea88ab0ea219d5c17b7c9178f5f5a6296a2d18eddd4bdf19e61830fc85d7de23fd4fbde31c4cf6694719",
			"144561e77bc4ea88c3b80eb3d044daa78b2dd5878dca766578e9c1aad3dba0d211ec862c36d2a7412917b70237c92226",
		},
	}

	for v, _ in test_vectors {
		raw_bytes, _ := hex.decode(transmute([]byte)(v.raw), context.temp_allocator)

		sc: ec.Scalar_p384r1
		_ = ec.sc_set_bytes(&sc, raw_bytes)

		b: [ec.SC_SIZE_P384R1]byte
		ec.sc_bytes(b[:], &sc)
		s := (string)(hex.encode(b[:], context.temp_allocator))

		testing.expectf(t, v.reduced == s, "sc: raw %s reduced: %s (%v), expected: %s", v.raw, s, &sc, v.reduced)
	}
}

@(test)
test_p256_scalar_mul :: proc(t: ^testing.T) {
	test_vectors := []struct {
		scalar:  string, // NOTE: Base 10
		x, y:    string,
	} {
		// Test vectors from http://point-at-infinity.org/ecc/nisttv
		{
			"1",
			"6B17D1F2E12C4247F8BCE6E563A440F277037D812DEB33A0F4A13945D898C296",
			"4FE342E2FE1A7F9B8EE7EB4A7C0F9E162BCE33576B315ECECBB6406837BF51F5",

		},
		{
			"2",
			"7CF27B188D034F7E8A52380304B51AC3C08969E277F21B35A60B48FC47669978",
			"07775510DB8ED040293D9AC69F7430DBBA7DADE63CE982299E04B79D227873D1",
		},
		{
			"3",
			"5ECBE4D1A6330A44C8F7EF951D4BF165E6C6B721EFADA985FB41661BC6E7FD6C",
			"8734640C4998FF7E374B06CE1A64A2ECD82AB036384FB83D9A79B127A27D5032",
		},
		{
			"4",
			"E2534A3532D08FBBA02DDE659EE62BD0031FE2DB785596EF509302446B030852",
			"E0F1575A4C633CC719DFEE5FDA862D764EFC96C3F30EE0055C42C23F184ED8C6",
		},
		{
			"5",
			"51590B7A515140D2D784C85608668FDFEF8C82FD1F5BE52421554A0DC3D033ED",
			"E0C17DA8904A727D8AE1BF36BF8A79260D012F00D4D80888D1D0BB44FDA16DA4",
		},
		{
			"6",
			"B01A172A76A4602C92D3242CB897DDE3024C740DEBB215B4C6B0AAE93C2291A9",
			"E85C10743237DAD56FEC0E2DFBA703791C00F7701C7E16BDFD7C48538FC77FE2",
		},
		{
			"7",
			"8E533B6FA0BF7B4625BB30667C01FB607EF9F8B8A80FEF5B300628703187B2A3",
			"73EB1DBDE03318366D069F83A6F5900053C73633CB041B21C55E1A86C1F400B4",
		},
		{
			"8",
			"62D9779DBEE9B0534042742D3AB54CADC1D238980FCE97DBB4DD9DC1DB6FB393",
			"AD5ACCBD91E9D8244FF15D771167CEE0A2ED51F6BBE76A78DA540A6A0F09957E",
		},
		{
			"9",
			"EA68D7B6FEDF0B71878938D51D71F8729E0ACB8C2C6DF8B3D79E8A4B90949EE0",
			"2A2744C972C9FCE787014A964A8EA0C84D714FEAA4DE823FE85A224A4DD048FA",
		},
		{
			"10",
			"CEF66D6B2A3A993E591214D1EA223FB545CA6C471C48306E4C36069404C5723F",
			"878662A229AAAE906E123CDD9D3B4C10590DED29FE751EEECA34BBAA44AF0773",
		},
		{
			"11",
			"3ED113B7883B4C590638379DB0C21CDA16742ED0255048BF433391D374BC21D1",
			"9099209ACCC4C8A224C843AFA4F4C68A090D04DA5E9889DAE2F8EEFCE82A3740",
		},
		{
			"12",
			"741DD5BDA817D95E4626537320E5D55179983028B2F82C99D500C5EE8624E3C4",
			"0770B46A9C385FDC567383554887B1548EEB912C35BA5CA71995FF22CD4481D3",
		},
		{
			"13",
			"177C837AE0AC495A61805DF2D85EE2FC792E284B65EAD58A98E15D9D46072C01",
			"63BB58CD4EBEA558A24091ADB40F4E7226EE14C3A1FB4DF39C43BBE2EFC7BFD8",
		},
		{
			"14",
			"54E77A001C3862B97A76647F4336DF3CF126ACBE7A069C5E5709277324D2920B",
			"F599F1BB29F4317542121F8C05A2E7C37171EA77735090081BA7C82F60D0B375",
		},
		{
			"15",
			"F0454DC6971ABAE7ADFB378999888265AE03AF92DE3A0EF163668C63E59B9D5F",
			"B5B93EE3592E2D1F4E6594E51F9643E62A3B21CE75B5FA3F47E59CDE0D034F36",
		},
		{
			"16",
			"76A94D138A6B41858B821C629836315FCD28392EFF6CA038A5EB4787E1277C6E",
			"A985FE61341F260E6CB0A1B5E11E87208599A0040FC78BAA0E9DDD724B8C5110",
		},
		{
			"17",
			"47776904C0F1CC3A9C0984B66F75301A5FA68678F0D64AF8BA1ABCE34738A73E",
			"AA005EE6B5B957286231856577648E8381B2804428D5733F32F787FF71F1FCDC",
		},
		{
			"18",
			"1057E0AB5780F470DEFC9378D1C7C87437BB4C6F9EA55C63D936266DBD781FDA",
			"F6F1645A15CBE5DC9FA9B7DFD96EE5A7DCC11B5C5EF4F1F78D83B3393C6A45A2",
		},
		{
			"19",
			"CB6D2861102C0C25CE39B7C17108C507782C452257884895C1FC7B74AB03ED83",
			"58D7614B24D9EF515C35E7100D6D6CE4A496716E30FA3E03E39150752BCECDAA",
		},
		{
			"20",
			"83A01A9378395BAB9BCD6A0AD03CC56D56E6B19250465A94A234DC4C6B28DA9A",
			"76E49B6DE2F73234AE6A5EB9D612B75C9F2202BB6923F54FF8240AAA86F640B8",
		},
		{
			"112233445566778899",
			"339150844EC15234807FE862A86BE77977DBFB3AE3D96F4C22795513AEAAB82F",
			"B1C14DDFDC8EC1B2583F51E85A5EB3A155840F2034730E9B5ADA38B674336A21",
		},
		{
			"112233445566778899112233445566778899",
			"1B7E046A076CC25E6D7FA5003F6729F665CC3241B5ADAB12B498CD32F2803264",
			"BFEA79BE2B666B073DB69A2A241ADAB0738FE9D2DD28B5604EB8C8CF097C457B",
		},
		{
			"029852220098221261079183923314599206100666902414330245206392788703677545185283",
			"9EACE8F4B071E677C5350B02F2BB2B384AAE89D58AA72CA97A170572E0FB222F",
			"1BBDAEC2430B09B93F7CB08678636CE12EAAFD58390699B5FD2F6E1188FC2A78",
		},
		{
			"057896042899961394862005778464643882389978449576758748073725983489954366354431",
			"878F22CC6DB6048D2B767268F22FFAD8E56AB8E2DC615F7BD89F1E350500DD8D",
			"714A5D7BB901C9C5853400D12341A892EF45D87FC553786756C4F0C9391D763E",
		},
		{
			"57896042899961394862005778464643882389978449576758748073725983489954366354431",
			"878F22CC6DB6048D2B767268F22FFAD8E56AB8E2DC615F7BD89F1E350500DD8D",
			"714A5D7BB901C9C5853400D12341A892EF45D87FC553786756C4F0C9391D763E",
		},
		{
			"1766845392945710151501889105729049882997660004824848915955419660366636031",
			"659A379625AB122F2512B8DADA02C6348D53B54452DFF67AC7ACE4E8856295CA",
			"49D81AB97B648464D0B4A288BD7818FAB41A16426E943527C4FED8736C53D0F6",
		},
		{
			"28948025760307534517734791687894775804466072615242963443097661355606862201087",
			"CBCEAAA8A4DD44BBCE58E8DB7740A5510EC2CB7EA8DA8D8F036B3FB04CDA4DE4",
			"4BD7AA301A80D7F59FD983FEDBE59BB7B2863FE46494935E3745B360E32332FA",
		},
		{
			"113078210460870548944811695960290644973229224625838436424477095834645696384",
			"F0C4A0576154FF3A33A3460D42EAED806E854DFA37125221D37935124BA462A4",
			"5B392FA964434D29EEC6C9DBC261CF116796864AA2FAADB984A2DF38D1AEF7A3",
		},
		{
			"12078056106883488161242983286051341125085761470677906721917479268909056",
			"5E6C8524B6369530B12C62D31EC53E0288173BD662BDF680B53A41ECBCAD00CC",
			"447FE742C2BFEF4D0DB14B5B83A2682309B5618E0064A94804E9282179FE089F",
		},
		{
			"57782969857385448082319957860328652998540760998293976083718804450708503920639",
			"03792E541BC209076A3D7920A915021ECD396A6EB5C3960024BE5575F3223484",
			"FC774AE092403101563B712F68170312304F20C80B40C06282063DB25F268DE4",
		},
		{
			"57896017119460046759583662757090100341435943767777707906455551163257755533312",
			"2379FF85AB693CDF901D6CE6F2473F39C04A2FE3DCD842CE7AAB0E002095BCF8",
			"F8B476530A634589D5129E46F322B02FBC610A703D80875EE70D7CE1877436A1",
		},
		{
			"452312848374287284681282171017647412726433684238464212999305864837160993279",
			"C1E4072C529BF2F44DA769EFC934472848003B3AF2C0F5AA8F8DDBD53E12ED7C",
			"39A6EE77812BB37E8079CD01ED649D3830FCA46F718C1D3993E4A591824ABCDB",
		},
		{
			"904571339174065134293634407946054000774746055866917729876676367558469746684",
			"34DFBC09404C21E250A9B40FA8772897AC63A094877DB65862B61BD1507B34F3",
			"CF6F8A876C6F99CEAEC87148F18C7E1E0DA6E165FFC8ED82ABB65955215F77D3",
		},
		{
			"115792089210356248762697446949407573529996955224135760342422259061068512044349",
			"83A01A9378395BAB9BCD6A0AD03CC56D56E6B19250465A94A234DC4C6B28DA9A",
			"891B64911D08CDCC5195A14629ED48A360DDFD4596DC0AB007DBF5557909BF47",
		},
		{
			"115792089210356248762697446949407573529996955224135760342422259061068512044350",
			"CB6D2861102C0C25CE39B7C17108C507782C452257884895C1FC7B74AB03ED83",
			"A7289EB3DB2610AFA3CA18EFF292931B5B698E92CF05C1FC1C6EAF8AD4313255",
		},
		{
			"115792089210356248762697446949407573529996955224135760342422259061068512044351",
			"1057E0AB5780F470DEFC9378D1C7C87437BB4C6F9EA55C63D936266DBD781FDA",
			"090E9BA4EA341A246056482026911A58233EE4A4A10B0E08727C4CC6C395BA5D",
		},
		{
			"115792089210356248762697446949407573529996955224135760342422259061068512044352",
			"47776904C0F1CC3A9C0984B66F75301A5FA68678F0D64AF8BA1ABCE34738A73E",
			"55FFA1184A46A8D89DCE7A9A889B717C7E4D7FBCD72A8CC0CD0878008E0E0323",
		},
		{
			"115792089210356248762697446949407573529996955224135760342422259061068512044353",
			"76A94D138A6B41858B821C629836315FCD28392EFF6CA038A5EB4787E1277C6E",
			"567A019DCBE0D9F2934F5E4A1EE178DF7A665FFCF0387455F162228DB473AEEF",
		},
		{
			"115792089210356248762697446949407573529996955224135760342422259061068512044354",
			"F0454DC6971ABAE7ADFB378999888265AE03AF92DE3A0EF163668C63E59B9D5F",
			"4A46C11BA6D1D2E1B19A6B1AE069BC19D5C4DE328A4A05C0B81A6321F2FCB0C9",
		},
		{
			"115792089210356248762697446949407573529996955224135760342422259061068512044355",
			"54E77A001C3862B97A76647F4336DF3CF126ACBE7A069C5E5709277324D2920B",
			"0A660E43D60BCE8BBDEDE073FA5D183C8E8E15898CAF6FF7E45837D09F2F4C8A",
		},
		{
			"115792089210356248762697446949407573529996955224135760342422259061068512044356",
			"177C837AE0AC495A61805DF2D85EE2FC792E284B65EAD58A98E15D9D46072C01",
			"9C44A731B1415AA85DBF6E524BF0B18DD911EB3D5E04B20C63BC441D10384027",
		},
		{
			"115792089210356248762697446949407573529996955224135760342422259061068512044357",
			"741DD5BDA817D95E4626537320E5D55179983028B2F82C99D500C5EE8624E3C4",
			"F88F4B9463C7A024A98C7CAAB7784EAB71146ED4CA45A358E66A00DD32BB7E2C",
		},
		{
			"115792089210356248762697446949407573529996955224135760342422259061068512044358",
			"3ED113B7883B4C590638379DB0C21CDA16742ED0255048BF433391D374BC21D1",
			"6F66DF64333B375EDB37BC505B0B3975F6F2FB26A16776251D07110317D5C8BF",
		},
		{
			"115792089210356248762697446949407573529996955224135760342422259061068512044359",
			"CEF66D6B2A3A993E591214D1EA223FB545CA6C471C48306E4C36069404C5723F",
			"78799D5CD655517091EDC32262C4B3EFA6F212D7018AE11135CB4455BB50F88C",
		},
		{
			"115792089210356248762697446949407573529996955224135760342422259061068512044360",
			"EA68D7B6FEDF0B71878938D51D71F8729E0ACB8C2C6DF8B3D79E8A4B90949EE0",
			"D5D8BB358D36031978FEB569B5715F37B28EB0165B217DC017A5DDB5B22FB705",
		},
		{
			"115792089210356248762697446949407573529996955224135760342422259061068512044361",
			"62D9779DBEE9B0534042742D3AB54CADC1D238980FCE97DBB4DD9DC1DB6FB393",
			"52A533416E1627DCB00EA288EE98311F5D12AE0A4418958725ABF595F0F66A81",
		},
		{
			"115792089210356248762697446949407573529996955224135760342422259061068512044362",
			"8E533B6FA0BF7B4625BB30667C01FB607EF9F8B8A80FEF5B300628703187B2A3",
			"8C14E2411FCCE7CA92F9607C590A6FFFAC38C9CD34FBE4DE3AA1E5793E0BFF4B",
		},
		{
			"115792089210356248762697446949407573529996955224135760342422259061068512044363",
			"B01A172A76A4602C92D3242CB897DDE3024C740DEBB215B4C6B0AAE93C2291A9",
			"17A3EF8ACDC8252B9013F1D20458FC86E3FF0890E381E9420283B7AC7038801D",
		},
		{
			"115792089210356248762697446949407573529996955224135760342422259061068512044364",
			"51590B7A515140D2D784C85608668FDFEF8C82FD1F5BE52421554A0DC3D033ED",
			"1F3E82566FB58D83751E40C9407586D9F2FED1002B27F7772E2F44BB025E925B",
		},
		{
			"115792089210356248762697446949407573529996955224135760342422259061068512044365",
			"E2534A3532D08FBBA02DDE659EE62BD0031FE2DB785596EF509302446B030852",
			"1F0EA8A4B39CC339E62011A02579D289B103693D0CF11FFAA3BD3DC0E7B12739",
		},
		{
			"115792089210356248762697446949407573529996955224135760342422259061068512044366",
			"5ECBE4D1A6330A44C8F7EF951D4BF165E6C6B721EFADA985FB41661BC6E7FD6C",
			"78CB9BF2B6670082C8B4F931E59B5D1327D54FCAC7B047C265864ED85D82AFCD",
		},
		{
			"115792089210356248762697446949407573529996955224135760342422259061068512044367",
			"7CF27B188D034F7E8A52380304B51AC3C08969E277F21B35A60B48FC47669978",
			"F888AAEE24712FC0D6C26539608BCF244582521AC3167DD661FB4862DD878C2E",
		},
		{
			"115792089210356248762697446949407573529996955224135760342422259061068512044368",
			"6B17D1F2E12C4247F8BCE6E563A440F277037D812DEB33A0F4A13945D898C296",
			"B01CBD1C01E58065711814B583F061E9D431CCA994CEA1313449BF97C840AE0A",
		},
	}

	for v, _ in test_vectors {
		x_bytes, _ := hex.decode(transmute([]byte)(v.x), context.temp_allocator)
		y_bytes, _ := hex.decode(transmute([]byte)(v.y), context.temp_allocator)

		k_big: big.Int
		err := big.set(&k_big, v.scalar, 10, context.temp_allocator)
		testing.expectf(t, err == nil, "failed to parse k", err)

		k_sz: int
		k_sz, err = big.int_to_bytes_size(&k_big, allocator = context.temp_allocator)
		testing.expect(t, err == nil)

		k_bytes := make([]byte, k_sz, context.temp_allocator)
		err = big.int_to_bytes_big(&k_big, k_bytes, allocator = context.temp_allocator)

		p, q, expected, g: ec.Point_p256r1
		sc: ec.Scalar_p256r1

		_ = ec.sc_set_bytes(&sc, k_bytes)

		ec.pt_generator(&g)
		ok := ec.pt_set_xy_bytes(&expected, x_bytes, y_bytes)
		testing.expectf(t, ok, "failed to set point; %s, %s", v.x, v.y)

		ec.pt_scalar_mul(&p, &g, &sc)
		ec.pt_scalar_mul_generator(&q, &sc)
		ec.pt_rescale(&p, &p)
		ec.pt_rescale(&q, &q)

		testing.expect(t, ec.pt_equal(&p, &q) == 1)
		testing.expectf(t, ec.pt_equal(&p, &expected) == 1, "sc: %s actual: %v expected: %v", v.scalar, &p, &expected)
	}
}

@(test)
test_p384_scalar_mul :: proc(t: ^testing.T) {
	test_vectors := []struct {
		scalar:  string, // NOTE: Base 10
		x, y:    string,
	} {
		// Test vectors from http://point-at-infinity.org/ecc/nisttv
		{
			"1",
			"AA87CA22BE8B05378EB1C71EF320AD746E1D3B628BA79B9859F741E082542A385502F25DBF55296C3A545E3872760AB7",
			"3617DE4A96262C6F5D9E98BF9292DC29F8F41DBD289A147CE9DA3113B5F0B8C00A60B1CE1D7E819D7A431D7C90EA0E5F",
		},
		{
			"2",
			"08D999057BA3D2D969260045C55B97F089025959A6F434D651D207D19FB96E9E4FE0E86EBE0E64F85B96A9C75295DF61",
			"8E80F1FA5B1B3CEDB7BFE8DFFD6DBA74B275D875BC6CC43E904E505F256AB4255FFD43E94D39E22D61501E700A940E80",
		},
		{
			"3",
			"077A41D4606FFA1464793C7E5FDC7D98CB9D3910202DCD06BEA4F240D3566DA6B408BBAE5026580D02D7E5C70500C831",
			"C995F7CA0B0C42837D0BBE9602A9FC998520B41C85115AA5F7684C0EDC111EACC24ABD6BE4B5D298B65F28600A2F1DF1",
		},
		{
			"4",
			"138251CD52AC9298C1C8AAD977321DEB97E709BD0B4CA0ACA55DC8AD51DCFC9D1589A1597E3A5120E1EFD631C63E1835",
			"CACAE29869A62E1631E8A28181AB56616DC45D918ABC09F3AB0E63CF792AA4DCED7387BE37BBA569549F1C02B270ED67",
		},
		{
			"5",
			"11DE24A2C251C777573CAC5EA025E467F208E51DBFF98FC54F6661CBE56583B037882F4A1CA297E60ABCDBC3836D84BC",
			"8FA696C77440F92D0F5837E90A00E7C5284B447754D5DEE88C986533B6901AEB3177686D0AE8FB33184414ABE6C1713A",
		},
		{
			"6",
			"627BE1ACD064D2B2226FE0D26F2D15D3C33EBCBB7F0F5DA51CBD41F26257383021317D7202FF30E50937F0854E35C5DF",
			"09766A4CB3F8B1C21BE6DDA6C14F1575B2C95352644F774C99864F613715441604C45B8D84E165311733A408D3F0F934",
		},
		{
			"7",
			"283C1D7365CE4788F29F8EBF234EDFFEAD6FE997FBEA5FFA2D58CC9DFA7B1C508B05526F55B9EBB2040F05B48FB6D0E1",
			"9475C99061E41B88BA52EFDB8C1690471A61D867ED799729D9C92CD01DBD225630D84EDE32A78F9E64664CDAC512EF8C",
		},
		{
			"8",
			"1692778EA596E0BE75114297A6FA383445BF227FBE58190A900C3C73256F11FB5A3258D6F403D5ECE6E9B269D822C87D",
			"DCD2365700D4106A835388BA3DB8FD0E22554ADC6D521CD4BD1C30C2EC0EEC196BADE1E9CDD1708D6F6ABFA4022B0AD2",
		},
		{
			"9",
			"8F0A39A4049BCB3EF1BF29B8B025B78F2216F7291E6FD3BAC6CB1EE285FB6E21C388528BFEE2B9535C55E4461079118B",
			"62C77E1438B601D6452C4A5322C3A9799A9B3D7CA3C400C6B7678854AED9B3029E743EFEDFD51B68262DA4F9AC664AF8",
		},
		{
			"10",
			"A669C5563BD67EEC678D29D6EF4FDE864F372D90B79B9E88931D5C29291238CCED8E85AB507BF91AA9CB2D13186658FB",
			"A988B72AE7C1279F22D9083DB5F0ECDDF70119550C183C31C502DF78C3B705A8296D8195248288D997784F6AB73A21DD",
		},
		{
			"11",
			"099056E27DA7B998DA1EEEC2904816C57FE935ED5837C37456C9FD14892D3F8C4749B66E3AFB81D626356F3B55B4DDD8",
			"2E4C0C234E30AB96688505544AC5E0396FC4EED8DFC363FD43FF93F41B52A3255466D51263AAFF357D5DBA8138C5E0BB",
		},
		{
			"12",
			"952A7A349BD49289AB3AC421DCF683D08C2ED5E41F6D0E21648AF2691A481406DA4A5E22DA817CB466DA2EA77D2A7022",
			"A0320FAF84B5BC0563052DEAE6F66F2E09FB8036CE18A0EBB9028B096196B50D031AA64589743E229EF6BACCE21BD16E",
		},
		{
			"13",
			"A567BA97B67AEA5BAFDAF5002FFCC6AB9632BFF9F01F873F6267BCD1F0F11C139EE5F441ABD99F1BAAF1CA1E3B5CBCE7",
			"DE1B38B3989F3318644E4147AF164ECC5185595046932EC086329BE057857D66776BCB8272218A7D6423A12736F429CC",
		},
		{
			"14",
			"E8C8F94D44FBC2396BBEAC481B89D2B0877B1DFFD23E7DC95DE541EB651CCA2C41ABA24DBC02DE6637209ACCF0F59EA0",
			"891AE44356FC8AE0932BCBF6DE52C8A933B86191E7728D79C8319413A09D0F48FC468BA05509DE22D7EE5C9E1B67B888",
		},
		{
			"15",
			"B3D13FC8B32B01058CC15C11D813525522A94156FFF01C205B21F9F7DA7C4E9CA849557A10B6383B4B88701A9606860B",
			"152919E7DF9162A61B049B2536164B1BEEBAC4A11D749AF484D1114373DFBFD9838D24F8B284AF50985D588D33F7BD62",
		},
		{
			"16",
			"D5D89C3B5282369C5FBD88E2B231511A6B80DFF0E5152CF6A464FA9428A8583BAC8EBC773D157811A462B892401DAFCF",
			"D815229DE12906D241816D5E9A9448F1D41D4FC40E2A3BDB9CABA57E440A7ABAD1210CB8F49BF2236822B755EBAB3673",
		},
		{
			"17",
			"4099952208B4889600A5EBBCB13E1A32692BEFB0733B41E6DCC614E42E5805F817012A991AF1F486CAF3A9ADD9FFCC03",
			"5ECF94777833059839474594AF603598163AD3F8008AD0CD9B797D277F2388B304DA4D2FAA9680ECFA650EF5E23B09A0",
		},
		{
			"18",
			"DFB1FE3A40F7AC9B64C41D39360A7423828B97CB088A4903315E402A7089FA0F8B6C2355169CC9C99DFB44692A9B93DD",
			"453ACA1243B5EC6B423A68A25587E1613A634C1C42D2EE7E6C57F449A1C91DC89168B7036EC0A7F37A366185233EC522",
		},
		{
			"19",
			"8D481DAB912BC8AB16858A211D750B77E07DBECCA86CD9B012390B430467AABF59C8651060801C0E9599E68713F5D41B",
			"A1592FF0121460857BE99F2A60669050B2291B68A1039AA0594B32FD7ADC0E8C11FFBA5608004E646995B07E75E52245",
		},
		{
			"20",
			"605508EC02C534BCEEE9484C86086D2139849E2B11C1A9CA1E2808DEC2EAF161AC8A105D70D4F85C50599BE5800A623F",
			"5158EE87962AC6B81F00A103B8543A07381B7639A3A65F1353AEF11B733106DDE92E99B78DE367B48E238C38DAD8EEDD",
		},
		{
			"112233445566778899",
			"A499EFE48839BC3ABCD1C5CEDBDD51904F9514DB44F4686DB918983B0C9DC3AEE05A88B72433E9515F91A329F5F4FA60",
			"3B7CA28EF31F809C2F1BA24AAED847D0F8B406A4B8968542DE139DB5828CA410E615D1182E25B91B1131E230B727D36A",
		},
		{
			"112233445566778899112233445566778899",
			"90A0B1CAC601676B083F21E07BC7090A3390FE1B9C7F61D842D27FA315FB38D83667A11A71438773E483F2A114836B24",
			"3197D3C6123F0D6CD65D5F0DE106FEF36656CB16DC7CD1A6817EB1D51510135A8F492F72665CFD1053F75ED03A7D04C9",
		},
		{
			"10158184112867540819754776755819761756724522948540419979637868435924061464745859402573149498125806098880003248619520",
			"F2A066BD332DC59BBC3D01DA1B124C687D8BB44611186422DE94C1DA4ECF150E664D353CCDB5CB2652685F8EB4D2CD49",
			"D6ED0BF75FDD8E53D87765FA746835B673881D6D1907163A2C43990D75B454294F942EC571AD5AAE1806CAF2BB8E9A4A",
		},
		{
			"9850501551105991028245052605056992139810094908912799254115847683881357749738726091734403950439157209401153690566655",
			"5C7F9845D1C4AA44747F9137B6F9C39B36B26B8A62E8AF97290434D5F3B214F5A0131550ADB19058DC4C8780C4165C4A",
			"712F7FCCC86F647E70DB8798228CB16344AF3D00B139B6F8502939C2A965AF0EB4E39E2E16AB8F597B8D5630A50C9D85",
		},
		{
			"9850502723405747097317271194763310482462751455185699630571661657946308788426092983270628740691202018691293898608608",
			"DD5838F7EC3B8ACF1BECFD746F8B668C577107E93548ED93ED0D254C112E76B10F053109EF8428BFCD50D38C4C030C57",
			"33244F479CDAC34F160D9E4CE2D19D2FF0E3305B5BF0EEF29E91E9DE6E28F678C61B773AA7E3C03740E1A49D1AA2493C",
		},
		{
			"1146189371817832990947611400450889406070215735255370280811736587845016396640969656447803207438173695115264",
			"CB8ED893530BFBA04B4CA655923AAAD109A62BC8411D5925316C32D33602459C33057A1FBCB5F70AEB295D90F9165FBC",
			"426AEE3E91B08420F9B357B66D5AFCBCF3956590BF5564DBF9086042EB880493D19DA39AAA6436C6B5FC66CE5596B43F",
		},
		{
			"9619341438217097641865390297189708858938017986426152622639500179774624579127744608993294698873437325090751520764",
			"67F714012B6B070182122DDD435CC1C2262A1AB88939BC6A2906CB2B4137C5E82B4582160F6403CAB887ACDF5786A268",
			"90E31CF398CE2F8C5897C7380BF541075D1B4D3CB70547262B7095731252F181AC0597C66AF8311C7780DB39DEC0BD32",
		},
		{
			"1231307996623833742387400352380172566077927415136813282735641918395585376659282194317590461518639141730493780722175",
			"55A79DF7B53A99D31462C7E1A5ED5623970715BB1021098CB973A7520CBD6365E613E4B2467486FB37E86E01CEE09B8F",
			"B95AEB71693189911661B709A886A1867F056A0EFE401EE11C06030E46F7A87731DA4575863178012208707DD666727C",
		},
		{
			"587118838854683800942906722504810343086699021451906946003274128973058942197377013128840514404789143516741631",
			"9539A968CF819A0E52E10EEA3BACA1B6480D7E4DF69BC07002C568569047110EE4FE72FCA423FDD5179D6E0E19C44844",
			"A7728F37A0AE0DF2716061900D83A4DA149144129F89A214A8260464BAB609BB322E4E67DE5E4C4C6CB8D25983EC19B0",
		},
		{
			"153914077530671739663795070876894766451466019374644150541452557147890542143280855693795882295846834387672681660416",
			"933FC13276672AB360D909161CD02D830B1628935DF0D800C6ED602C59D575A86A8A97E3A2D697E3ED06BE741C0097D6",
			"F35296BD7A6B4C6C025ED6D84338CCCC7522A45C5D4FBDB1442556CAEFB598128FA188793ADA510EB5F44E90A4E4BEF1",
		},
		{
			"75148784606135150476268171850082176256856776750560539466196504390587921789283134009866871754361028131485122560",
			"0CE31E1C4A937071E6EBACA026A93D783848BCC0C1585DAF639518125FCD1F1629D63041ABFB11FFC8F03FA8B6FCF6BF",
			"A69EA55BE4BEAB2D5224050FEBFFBDFCFD614624C3B4F228909EB80012F003756D1C377E52F04FA539237F24DD080E2E",
		},
		{
			"19691383761310193665095292424754807745686799029814707849273381514021788371252213000473497648851202400395528761229312",
			"6842CFE3589AC268818291F31D44177A9168DCBC19F321ED66D81ECF59E31B54CCA0DDFD4C4136780171748D69A91C54",
			"E3A5ECD5AC725F13DBC631F358C6E817EDCF3A613B83832741A9DB591A0BAE767FC714F70C2E7EA891E4312047DECCC0",
		},
		{
			"39402006196394479212279040100143613805079739270465446667946905279627659399113263569398956308152294913554433653942623",
			"605508EC02C534BCEEE9484C86086D2139849E2B11C1A9CA1E2808DEC2EAF161AC8A105D70D4F85C50599BE5800A623F",
			"AEA7117869D53947E0FF5EFC47ABC5F8C7E489C65C59A0ECAC510EE48CCEF92116D16647721C984B71DC73C825271122",
		},
		{
			"39402006196394479212279040100143613805079739270465446667946905279627659399113263569398956308152294913554433653942624",
			"8D481DAB912BC8AB16858A211D750B77E07DBECCA86CD9B012390B430467AABF59C8651060801C0E9599E68713F5D41B",
			"5EA6D00FEDEB9F7A841660D59F996FAF4DD6E4975EFC655FA6B4CD028523F172EE0045A8F7FFB19B966A4F828A1ADDBA",
		},
		{
			"39402006196394479212279040100143613805079739270465446667946905279627659399113263569398956308152294913554433653942625",
			"DFB1FE3A40F7AC9B64C41D39360A7423828B97CB088A4903315E402A7089FA0F8B6C2355169CC9C99DFB44692A9B93DD",
			"BAC535EDBC4A1394BDC5975DAA781E9EC59CB3E3BD2D118193A80BB65E36E2366E9748FB913F580C85C99E7BDCC13ADD",
		},
		{
			"39402006196394479212279040100143613805079739270465446667946905279627659399113263569398956308152294913554433653942626",
			"4099952208B4889600A5EBBCB13E1A32692BEFB0733B41E6DCC614E42E5805F817012A991AF1F486CAF3A9ADD9FFCC03",
			"A1306B8887CCFA67C6B8BA6B509FCA67E9C52C07FF752F32648682D880DC774BFB25B2CF55697F13059AF10B1DC4F65F",
		},
		{
			"39402006196394479212279040100143613805079739270465446667946905279627659399113263569398956308152294913554433653942627",
			"D5D89C3B5282369C5FBD88E2B231511A6B80DFF0E5152CF6A464FA9428A8583BAC8EBC773D157811A462B892401DAFCF",
			"27EADD621ED6F92DBE7E92A1656BB70E2BE2B03BF1D5C42463545A81BBF585442EDEF3460B640DDC97DD48AB1454C98C",
		},
		{
			"39402006196394479212279040100143613805079739270465446667946905279627659399113263569398956308152294913554433653942628",
			"B3D13FC8B32B01058CC15C11D813525522A94156FFF01C205B21F9F7DA7C4E9CA849557A10B6383B4B88701A9606860B",
			"EAD6E618206E9D59E4FB64DAC9E9B4E411453B5EE28B650B7B2EEEBC8C2040257C72DB064D7B50AF67A2A773CC08429D",
		},
		{
			"39402006196394479212279040100143613805079739270465446667946905279627659399113263569398956308152294913554433653942629",
			"E8C8F94D44FBC2396BBEAC481B89D2B0877B1DFFD23E7DC95DE541EB651CCA2C41ABA24DBC02DE6637209ACCF0F59EA0",
			"76E51BBCA903751F6CD4340921AD3756CC479E6E188D728637CE6BEC5F62F0B603B9745EAAF621DD2811A362E4984777",
		},
		{
			"39402006196394479212279040100143613805079739270465446667946905279627659399113263569398956308152294913554433653942630",
			"A567BA97B67AEA5BAFDAF5002FFCC6AB9632BFF9F01F873F6267BCD1F0F11C139EE5F441ABD99F1BAAF1CA1E3B5CBCE7",
			"21E4C74C6760CCE79BB1BEB850E9B133AE7AA6AFB96CD13F79CD641FA87A82988894347C8DDE75829BDC5ED9C90BD633",
		},
		{
			"39402006196394479212279040100143613805079739270465446667946905279627659399113263569398956308152294913554433653942631",
			"952A7A349BD49289AB3AC421DCF683D08C2ED5E41F6D0E21648AF2691A481406DA4A5E22DA817CB466DA2EA77D2A7022",
			"5FCDF0507B4A43FA9CFAD215190990D1F6047FC931E75F1446FD74F69E694AF1FCE559B9768BC1DD610945341DE42E91",
		},
		{
			"39402006196394479212279040100143613805079739270465446667946905279627659399113263569398956308152294913554433653942632",
			"099056E27DA7B998DA1EEEC2904816C57FE935ED5837C37456C9FD14892D3F8C4749B66E3AFB81D626356F3B55B4DDD8",
			"D1B3F3DCB1CF5469977AFAABB53A1FC6903B1127203C9C02BC006C0BE4AD5CD9AB992AEC9C5500CA82A2457FC73A1F44",
		},
		{
			"39402006196394479212279040100143613805079739270465446667946905279627659399113263569398956308152294913554433653942633",
			"A669C5563BD67EEC678D29D6EF4FDE864F372D90B79B9E88931D5C29291238CCED8E85AB507BF91AA9CB2D13186658FB",
			"567748D5183ED860DD26F7C24A0F132208FEE6AAF3E7C3CE3AFD20873C48FA56D6927E69DB7D77266887B09648C5DE22",
		},
		{
			"39402006196394479212279040100143613805079739270465446667946905279627659399113263569398956308152294913554433653942634",
			"8F0A39A4049BCB3EF1BF29B8B025B78F2216F7291E6FD3BAC6CB1EE285FB6E21C388528BFEE2B9535C55E4461079118B",
			"9D3881EBC749FE29BAD3B5ACDD3C56866564C2835C3BFF39489877AB51264CFC618BC100202AE497D9D25B075399B507",
		},
		{
			"39402006196394479212279040100143613805079739270465446667946905279627659399113263569398956308152294913554433653942635",
			"1692778EA596E0BE75114297A6FA383445BF227FBE58190A900C3C73256F11FB5A3258D6F403D5ECE6E9B269D822C87D",
			"232DC9A8FF2BEF957CAC7745C24702F1DDAAB52392ADE32B42E3CF3D13F113E594521E15322E8F729095405CFDD4F52D",
		},
		{
			"39402006196394479212279040100143613805079739270465446667946905279627659399113263569398956308152294913554433653942636",
			"283C1D7365CE4788F29F8EBF234EDFFEAD6FE997FBEA5FFA2D58CC9DFA7B1C508B05526F55B9EBB2040F05B48FB6D0E1",
			"6B8A366F9E1BE47745AD102473E96FB8E59E2798128668D62636D32FE242DDA8CF27B120CD5870619B99B3263AED1073",
		},
		{
			"39402006196394479212279040100143613805079739270465446667946905279627659399113263569398956308152294913554433653942637",
			"627BE1ACD064D2B2226FE0D26F2D15D3C33EBCBB7F0F5DA51CBD41F26257383021317D7202FF30E50937F0854E35C5DF",
			"F68995B34C074E3DE41922593EB0EA8A4D36ACAD9BB088B36679B09EC8EABBE8FB3BA4717B1E9ACEE8CC5BF82C0F06CB",
		},
		{
			"39402006196394479212279040100143613805079739270465446667946905279627659399113263569398956308152294913554433653942638",
			"11DE24A2C251C777573CAC5EA025E467F208E51DBFF98FC54F6661CBE56583B037882F4A1CA297E60ABCDBC3836D84BC",
			"705969388BBF06D2F0A7C816F5FF183AD7B4BB88AB2A211773679ACC496FE513CE889791F51704CCE7BBEB55193E8EC5",
		},
		{
			"39402006196394479212279040100143613805079739270465446667946905279627659399113263569398956308152294913554433653942639",
			"138251CD52AC9298C1C8AAD977321DEB97E709BD0B4CA0ACA55DC8AD51DCFC9D1589A1597E3A5120E1EFD631C63E1835",
			"35351D679659D1E9CE175D7E7E54A99E923BA26E7543F60C54F19C3086D55B22128C7840C8445A96AB60E3FE4D8F1298",
		},
		{
			"39402006196394479212279040100143613805079739270465446667946905279627659399113263569398956308152294913554433653942640",
			"077A41D4606FFA1464793C7E5FDC7D98CB9D3910202DCD06BEA4F240D3566DA6B408BBAE5026580D02D7E5C70500C831",
			"366A0835F4F3BD7C82F44169FD5603667ADF4BE37AEEA55A0897B3F123EEE1523DB542931B4A2D6749A0D7A0F5D0E20E",
		},
		{
			"39402006196394479212279040100143613805079739270465446667946905279627659399113263569398956308152294913554433653942641",
			"08D999057BA3D2D969260045C55B97F089025959A6F434D651D207D19FB96E9E4FE0E86EBE0E64F85B96A9C75295DF61",
			"717F0E05A4E4C312484017200292458B4D8A278A43933BC16FB1AFA0DA954BD9A002BC15B2C61DD29EAFE190F56BF17F",
		},
		{
			"39402006196394479212279040100143613805079739270465446667946905279627659399113263569398956308152294913554433653942642",
			"AA87CA22BE8B05378EB1C71EF320AD746E1D3B628BA79B9859F741E082542A385502F25DBF55296C3A545E3872760AB7",
			"C9E821B569D9D390A26167406D6D23D6070BE242D765EB831625CEEC4A0F473EF59F4E30E2817E6285BCE2846F15F1A0",
		},
	}

	for v, _ in test_vectors {
		x_bytes, _ := hex.decode(transmute([]byte)(v.x), context.temp_allocator)
		y_bytes, _ := hex.decode(transmute([]byte)(v.y), context.temp_allocator)

		k_big: big.Int
		err := big.set(&k_big, v.scalar, 10, context.temp_allocator)
		testing.expectf(t, err == nil, "failed to parse k", err)

		k_sz: int
		k_sz, err = big.int_to_bytes_size(&k_big, allocator = context.temp_allocator)
		testing.expect(t, err == nil)

		k_bytes := make([]byte, k_sz, context.temp_allocator)
		err = big.int_to_bytes_big(&k_big, k_bytes, allocator = context.temp_allocator)

		p, q, expected, g: ec.Point_p384r1
		sc: ec.Scalar_p384r1

		_ = ec.sc_set_bytes(&sc, k_bytes)

		ec.pt_generator(&g)
		ok := ec.pt_set_xy_bytes(&expected, x_bytes, y_bytes)
		testing.expectf(t, ok, "failed to set point; %s, %s", v.x, v.y)

		ec.pt_scalar_mul(&p, &g, &sc)
		ec.pt_scalar_mul_generator(&q, &sc)
		ec.pt_rescale(&p, &p)
		ec.pt_rescale(&q, &q)

		testing.expect(t, ec.pt_equal(&p, &q) == 1)
		testing.expectf(t, ec.pt_equal(&p, &expected) == 1, "sc: %s actual: %v expected: %v", v.scalar, &p, &expected)
	}
}

@(test)
test_p256_s11n_sec_identity ::proc(t: ^testing.T) {
	p: ec.Point_p256r1

	ec.pt_generator(&p)
	ok := ec.pt_set_sec_bytes(&p, []byte{0x00})
	testing.expect(t, ok)
	testing.expectf(t, ec.pt_is_identity(&p) == 1, "%v", p)

	b := []byte{0xff}
	ok = ec.pt_sec_bytes(b, &p, true)
	testing.expect(t, ok)
	testing.expect(t, b[0] == 0x00)

	b = []byte{0xff}
	ok = ec.pt_sec_bytes(b, &p, false)
	testing.expect(t, ok)
	testing.expect(t, b[0] == 0x00)
}

@(test)
test_p256_s11n_sec_generator ::proc(t: ^testing.T) {
	p, g: ec.Point_p256r1

	ec.pt_generator(&g)
	ec.pt_identity(&p)

	b: [65]byte
	ok := ec.pt_sec_bytes(b[:], &g, false)
	testing.expect(t, ok)
	s := (string)(hex.encode(b[:], context.temp_allocator))
	testing.expectf(t, s == P256_G_UNCOMPRESSED, "g: %v bytes: %v, %v", g, P256_G_UNCOMPRESSED, s)

	ok = ec.pt_set_sec_bytes(&p, b[:])
	testing.expectf(t, ok, "%s", s)
	testing.expect(t, ec.pt_equal(&g, &p) == 1)
}
