///////////////////////////////////////////////////////////////////////////////
//
// File: TestFullMatrixStoragePolicy.cpp
//
// For more information, please see: http://www.nektar.info
//
// The MIT License
//
// Copyright (c) 2006 Division of Applied Mathematics, Brown University (USA),
// Department of Aeronautics, Imperial College London (UK), and Scientific
// Computing and Imaging Institute, University of Utah (USA).
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included
// in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//
// Description:
//
///////////////////////////////////////////////////////////////////////////////

#include <boost/test/unit_test.hpp>

#include <LibUtilities/LinearAlgebra/NekMatrix.hpp>
#include <UnitTests/CountedObject.h>
#include <UnitTests/util.h>
#include <boost/test/tools/floating_point_comparison.hpp>
#include <boost/test/unit_test.hpp>

namespace Nektar::FullMatrixStoragePolicyUnitTests
{
typedef FullMatrixFuncs Policy;

BOOST_AUTO_TEST_CASE(TestCalculateIndex)
{
    UnitTests::RedirectCerrIfNeeded();
    BOOST_CHECK_EQUAL(0u, Policy::CalculateIndex(5, 5, 0, 0));
    BOOST_CHECK_EQUAL(1u, Policy::CalculateIndex(5, 5, 1, 0));
    BOOST_CHECK_EQUAL(2u, Policy::CalculateIndex(5, 5, 2, 0));
    BOOST_CHECK_EQUAL(3u, Policy::CalculateIndex(5, 5, 3, 0));
    BOOST_CHECK_EQUAL(4u, Policy::CalculateIndex(5, 5, 4, 0));
    BOOST_CHECK_EQUAL(5u, Policy::CalculateIndex(5, 5, 5, 0));
    BOOST_CHECK_EQUAL(6u, Policy::CalculateIndex(5, 5, 6, 0));
}

BOOST_AUTO_TEST_CASE(TestAdvanceFull)
{
    UnitTests::RedirectCerrIfNeeded();
    typedef FullMatrixFuncs Policy;

    {
        unsigned int curRow         = 0;
        unsigned int curColumn      = 0;
        std::tie(curRow, curColumn) = Policy::Advance(4, 3, curRow, curColumn);
        BOOST_CHECK_EQUAL(1u, curRow);
        BOOST_CHECK_EQUAL(0u, curColumn);

        std::tie(curRow, curColumn) = Policy::Advance(4, 3, curRow, curColumn);
        BOOST_CHECK_EQUAL(2u, curRow);
        BOOST_CHECK_EQUAL(0u, curColumn);

        std::tie(curRow, curColumn) = Policy::Advance(4, 3, curRow, curColumn);
        BOOST_CHECK_EQUAL(3u, curRow);
        BOOST_CHECK_EQUAL(0u, curColumn);

        std::tie(curRow, curColumn) = Policy::Advance(4, 3, curRow, curColumn);
        BOOST_CHECK_EQUAL(0u, curRow);
        BOOST_CHECK_EQUAL(1u, curColumn);

        std::tie(curRow, curColumn) = Policy::Advance(4, 3, curRow, curColumn);
        BOOST_CHECK_EQUAL(1u, curRow);
        BOOST_CHECK_EQUAL(1u, curColumn);

        std::tie(curRow, curColumn) = Policy::Advance(4, 3, curRow, curColumn);
        BOOST_CHECK_EQUAL(2u, curRow);
        BOOST_CHECK_EQUAL(1u, curColumn);

        std::tie(curRow, curColumn) = Policy::Advance(4, 3, curRow, curColumn);
        BOOST_CHECK_EQUAL(3u, curRow);
        BOOST_CHECK_EQUAL(1u, curColumn);

        std::tie(curRow, curColumn) = Policy::Advance(4, 3, curRow, curColumn);
        BOOST_CHECK_EQUAL(0u, curRow);
        BOOST_CHECK_EQUAL(2u, curColumn);

        std::tie(curRow, curColumn) = Policy::Advance(4, 3, curRow, curColumn);
        BOOST_CHECK_EQUAL(1u, curRow);
        BOOST_CHECK_EQUAL(2u, curColumn);

        std::tie(curRow, curColumn) = Policy::Advance(4, 3, curRow, curColumn);
        BOOST_CHECK_EQUAL(2u, curRow);
        BOOST_CHECK_EQUAL(2u, curColumn);

        std::tie(curRow, curColumn) = Policy::Advance(4, 3, curRow, curColumn);
        BOOST_CHECK_EQUAL(3u, curRow);
        BOOST_CHECK_EQUAL(2u, curColumn);

        std::tie(curRow, curColumn) = Policy::Advance(4, 3, curRow, curColumn);
        BOOST_CHECK_EQUAL(std::numeric_limits<unsigned int>::max(), curRow);
        BOOST_CHECK_EQUAL(std::numeric_limits<unsigned int>::max(), curColumn);
    }

    {

        unsigned int curRow         = 0;
        unsigned int curColumn      = 0;
        std::tie(curRow, curColumn) = Policy::Advance(1, 1, curRow, curColumn);
        BOOST_CHECK_EQUAL(std::numeric_limits<unsigned int>::max(), curRow);
        BOOST_CHECK_EQUAL(std::numeric_limits<unsigned int>::max(), curColumn);
    }
}

} // namespace Nektar::FullMatrixStoragePolicyUnitTests
