# -*- coding: utf-8 -*- #
# Copyright 2019 Google Inc. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Common classes and functions for organization security policy rules."""

from __future__ import absolute_import
from __future__ import division
from __future__ import unicode_literals

import re

from googlecloudsdk.calliope import exceptions as calliope_exceptions
from googlecloudsdk.command_lib.compute import exceptions

ALLOWED_METAVAR = 'PROTOCOL[:PORT[-PORT]]'
LEGAL_SPECS = re.compile(
    r"""

    (?P<protocol>[a-zA-Z0-9+.-]+) # The protocol group.

    (:(?P<ports>\d+(-\d+)?))?     # The optional ports group.
                                  # May specify a range.

    $                             # End of input marker.
    """, re.VERBOSE)


def ParseDestPorts(dest_ports, message_classes):
  """Parses protocol:port mappings for --dest-ports command line."""
  dest_port_list = []
  for spec in dest_ports or []:
    match = LEGAL_SPECS.match(spec)
    if not match:
      raise exceptions.ArgumentError(
          'Organization security policy rules must be of the form {0}; '
          'received [{1}].'.format(ALLOWED_METAVAR, spec))
    if match.group('ports'):
      ports = [match.group('ports')]
    else:
      ports = []
    dest_port = message_classes.SecurityPolicyRuleMatcherConfigDestinationPort(
        ipProtocol=match.group('protocol'), ports=ports)
    dest_port_list.append(dest_port)
  return dest_port_list


def ParseLayer4Configs(layer4_conifigs, message_classes):
  """Parses protocol:port mappings for --layer4-configs command line."""
  layer4_config_list = []
  for spec in layer4_conifigs or []:
    match = LEGAL_SPECS.match(spec)
    if not match:
      raise exceptions.ArgumentError(
          'Organization security policy rules must be of the form {0}; '
          'received [{1}].'.format(ALLOWED_METAVAR, spec))
    if match.group('ports'):
      ports = [match.group('ports')]
    else:
      ports = []
    layer4_conifig = (
        message_classes.SecurityPolicyRuleMatcherConfigLayer4Config(
            ipProtocol=match.group('protocol'), ports=ports))
    layer4_config_list.append(layer4_conifig)
  return layer4_config_list


def ConvertPriorityToInt(priority):
  try:
    int_priority = int(priority)
  except ValueError:
    raise calliope_exceptions.InvalidArgumentException(
        'priority', 'priority must be a valid non-negative integer.')
  if int_priority < 0:
    raise calliope_exceptions.InvalidArgumentException(
        'priority', 'priority must be a valid non-negative integer.')
  return int_priority


def ConvertAction(action):
  return {
      'goto-next': 'goto_next',
      'deny-403': 'deny(403)',
      'deny-404': 'deny(404)',
      'deny-502': 'deny(502)'
  }.get(action, action)
