#!/bin/bash

# Copyright 2024 The Kubernetes Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

set -o errexit
set -o nounset
set -o pipefail

REPO_ROOT="$(git rev-parse --show-toplevel)"
cd "${REPO_ROOT}"


function generate_k8s_hashes() {
  prefix=$1
  max_patch=$2

  cat > "${REPO_ROOT}/pkg/assets/assetdata/k8s-${prefix}.yaml" <<EOF
# This file is generated by generate-asset-hashes.sh

filestores:
- base: https://dl.k8s.io/release/

files:
EOF

  for ((patch = 0 ; patch <= max_patch ; patch++ )); do
    version="${prefix}.${patch}"
    echo "k8s ${version}"

    # We exclude some files that we don't currently need, to keep the size down
    go run ./pkg/assets/assetdata/tools/cmd/generatefileassets \
      --base https://dl.k8s.io/release/ \
      --prefix "v${version}/" \
      --exclude "**/arm/**" \
      --exclude "**/386/**" \
      --exclude "**/ppc64le/**" \
      --exclude "**/s390x/**" \
      --exclude "**/windows/**" \
      --exclude "**/*.docker_tag" \
      --exclude "**/*.spdx" \
      --exclude "**/*.tar" \
      --exclude "**/*.tar.gz" \
      | sed "s@files:@# kubernetes ${version}@g" >> "${REPO_ROOT}/pkg/assets/assetdata/k8s-${prefix}.yaml"
  done
}

function generate_runc_hashes() {
  prefix=$1
  max_patch=$2

  cat > "${REPO_ROOT}/pkg/assets/assetdata/runc-${prefix}.yaml" <<EOF
# This file is generated by generate-asset-hashes.sh

filestores:
- base: https://github.com/opencontainers/runc/releases/download/

files:
EOF

  for ((patch = 0 ; patch <= max_patch ; patch++ )); do
    version="${prefix}.${patch}"
    echo "runc ${version}"

    # We exclude some files that we don't currently need, to keep the size down
    go run ./pkg/assets/assetdata/tools/cmd/generatefileassets \
      --base https://github.com/opencontainers/runc/releases/download/ \
      --prefix "v${version}/" \
      --exclude "**/*.386" \
      --exclude "**/*.armel" \
      --exclude "**/*.armhf" \
      --exclude "**/*.ppc64le" \
      --exclude "**/*.riscv64" \
      --exclude "**/*.s390x" \
      --exclude "**/*.tar.xz" \
      --sums "https://github.com/opencontainers/runc/releases/download/v${version}/runc.sha256sum" \
      | sed "s@files:@# runc ${version}@g" >> "${REPO_ROOT}/pkg/assets/assetdata/runc-${prefix}.yaml"
  done
}

function generate_containerd_hashes() {
  prefix=$1
  max_patch=$2

  cat > "${REPO_ROOT}/pkg/assets/assetdata/containerd-${prefix}.yaml" <<EOF
# This file is generated by generate-asset-hashes.sh

filestores:
- base: https://github.com/containerd/containerd/releases/download/

files:
EOF

  for ((patch = 0 ; patch <= max_patch ; patch++ )); do
    version="${prefix}.${patch}"
    echo "containerd ${version}"

    {
      echo "# containerd ${version}"
      go run ./pkg/assets/assetdata/tools/cmd/generatefileassets \
        --base https://github.com/containerd/containerd/releases/download/ \
        --prefix "v${version}/" \
        --sums "https://github.com/containerd/containerd/releases/download/v${version}/containerd-${version}-linux-amd64.tar.gz.sha256sum" \
        | grep -v "^files:"
      go run ./pkg/assets/assetdata/tools/cmd/generatefileassets \
        --base https://github.com/containerd/containerd/releases/download/ \
        --prefix "v${version}/" \
        --sums "https://github.com/containerd/containerd/releases/download/v${version}/containerd-${version}-linux-arm64.tar.gz.sha256sum" \
        | grep -v "^files:"
    } >> "${REPO_ROOT}/pkg/assets/assetdata/containerd-${prefix}.yaml"
  done
}

function generate_cni_hashes() {
  prefix=$1
  patch=$2

  cat > "${REPO_ROOT}/pkg/assets/assetdata/cni-${prefix}.yaml" <<EOF
# This file is generated by generate-asset-hashes.sh

filestores:
- base: https://storage.googleapis.com/k8s-artifacts-cni/release/

files:
EOF

  version="${prefix}.${patch}"
  echo "cni ${version}"

  {
    echo "# cni ${version}"
    go run ./pkg/assets/assetdata/tools/cmd/generatefileassets \
      --base https://storage.googleapis.com/k8s-artifacts-cni/release/ \
      --prefix "v${version}/" \
      --sums "https://storage.googleapis.com/k8s-artifacts-cni/release/v${version}/cni-plugins-linux-amd64-v${version}.tgz.sha256" \
      | grep -v "^files:"
    go run ./pkg/assets/assetdata/tools/cmd/generatefileassets \
      --base https://storage.googleapis.com/k8s-artifacts-cni/release/ \
      --prefix "v${version}/" \
      --sums "https://storage.googleapis.com/k8s-artifacts-cni/release/v${version}/cni-plugins-linux-arm64-v${version}.tgz.sha256" \
      | grep -v "^files:"
  } >> "${REPO_ROOT}/pkg/assets/assetdata/cni-${prefix}.yaml"
}

# Generate k8s hashes.
# The first argument is the major and minor version, the second is the maximum patch version.
generate_k8s_hashes 1.25 16
generate_k8s_hashes 1.26 15
generate_k8s_hashes 1.27 16
generate_k8s_hashes 1.28 15
generate_k8s_hashes 1.29 11
generate_k8s_hashes 1.30 7
generate_k8s_hashes 1.31 3

# Generate runc hashes.
# The first argument is the major and minor version, the second is the maximum patch version.
generate_runc_hashes 1.1 15
generate_runc_hashes 1.2 4

# Generate containerd hashes.
generate_containerd_hashes 1.7 25
generate_containerd_hashes 2.0 1

# Generate CNI network plugins hashes.
generate_cni_hashes 0.9 1
generate_cni_hashes 1.2 0
generate_cni_hashes 1.3 0
generate_cni_hashes 1.4 1
generate_cni_hashes 1.5 1
generate_cni_hashes 1.6 1
