package integration

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/http/httputil"
	"os/exec"
	"strings"
	"testing"

	"github.com/sclevine/spec"
	"github.com/stretchr/testify/require"
)

var _ = suite("gradient/agent/functionroute/delete", func(t *testing.T, when spec.G, it spec.S) {
	var (
		expect *require.Assertions
		cmd    *exec.Cmd
		server *httptest.Server
	)

	it.Before(func() {
		expect = require.New(t)

		server = httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, req *http.Request) {
			switch req.URL.Path {
			case "/v2/gen-ai/agents/0f0e928f-4649-11f0-bf8f-4e013e2ddde4/functions/e40dc785-5e69-11f0-bf8f-4e013e2ddde4":
				auth := req.Header.Get("Authorization")
				if auth != "Bearer some-magic-token" {
					w.WriteHeader(http.StatusUnauthorized)
					return
				}

				if req.Method != http.MethodDelete {
					w.WriteHeader(http.StatusMethodNotAllowed)
					return
				}

				w.Header().Set("Content-Type", "application/json")
				w.WriteHeader(http.StatusOK)
				w.Write([]byte(functionRouteDeleteResponse))
			default:
				dump, err := httputil.DumpRequest(req, true)
				if err != nil {
					t.Fatal("failed to dump request")
				}

				t.Fatalf("received unknown request: %s", dump)
			}
		}))
	})

	it.After(func() {
		server.Close()
	})

	when("all required flags are passed", func() {
		it("deletes a function route", func() {
			aliases := []string{"delete", "d", "del", "rm"}

			for _, alias := range aliases {
				cmd = exec.Command(builtBinaryPath,
					"-t", "some-magic-token",
					"-u", server.URL,
					"gradient",
					"agent",
					"functionroute",
					alias,
					"--agent-id", "0f0e928f-4649-11f0-bf8f-4e013e2ddde4",
					"--function-id", "e40dc785-5e69-11f0-bf8f-4e013e2ddde4",
				)

				output, err := cmd.CombinedOutput()
				expect.NoError(err, fmt.Sprintf("received error output: %s", output))
				expect.Equal(strings.TrimSpace(functionRouteDeleteOutput), strings.TrimSpace(string(output)))
			}
		})
	})

	when("required flags are missing", func() {
		it("returns an error when agent-id is missing", func() {
			cmd = exec.Command(builtBinaryPath,
				"-t", "some-magic-token",
				"-u", server.URL,
				"gradient",
				"agent",
				"functionroute",
				"delete",
				"--function-id", "e40dc785-5e69-11f0-bf8f-4e013e2ddde4",
			)

			output, err := cmd.CombinedOutput()
			expect.Error(err)
			expect.Contains(string(output), "missing required arguments")
		})

		it("returns an error when function-id is missing", func() {
			cmd = exec.Command(builtBinaryPath,
				"-t", "some-magic-token",
				"-u", server.URL,
				"gradient",
				"agent",
				"functionroute",
				"delete",
				"--agent-id", "0f0e928f-4649-11f0-bf8f-4e013e2ddde4",
			)

			output, err := cmd.CombinedOutput()
			expect.Error(err)
			expect.Contains(string(output), "missing required arguments")
		})
	})
})

const (
	functionRouteDeleteOutput = `
ID                                      Name                   Region    Project ID                              Model ID    Created At                       User ID
0f0e928f-4649-11f0-bf8f-4e013e2ddde4    terraform-testing-1    tor1      84e1e297-ee40-41ac-95ff-1067cf2206e9                2025-06-10 22:20:04 +0000 UTC    18697494
`
	functionRouteDeleteResponse = `
{
  "agent": {
    "uuid": "0f0e928f-4649-11f0-bf8f-4e013e2ddde4",
    "name": "terraform-testing-1",
    "region": "tor1",
    "project_id": "84e1e297-ee40-41ac-95ff-1067cf2206e9",
    "model": {
      "uuid": ""
    },
    "instruction": "You are a weather assistant",
    "description": "Creating via doctl again",
    "created_at": "2025-06-10T22:20:04Z",
    "updated_at": "2025-06-10T22:20:04Z",
    "user_id": "18697494",
    "retrieval_method": "RETRIEVAL_METHOD_UNKNOWN",
    "function_routes": []
  }
}
`
)
