
#include "gtk-local.h"

#include "hsv-control.h"

#include "gtk-graph.h"
#include "message-c.h"
#include "gcore/rgb_hsv.h"
#include "gsys/sic_util.h"
#include "gtv/xsub.h"
#include "gtv/event-stack.h"
#include <stdlib.h>
#include <math.h>
#ifndef WIN32
#include <dirent.h>
#endif

#define _GGTK_USE_DIALOG

typedef enum {
    HSV_HUE, HSV_SAT, HSV_VAL, HSV_LUT
} hsv_drawing_cell_id;
#define _HSV_DRAWING_WIDTH 128
#define _HSV_DRAWING_HEIGHT 128
#define _HSV_LUT_HEIGHT 32
#define MAX_INTENSITY 0xFFFF           /* according to X... */

typedef struct {
    int size;
    GdkPoint *hue_points;
    GdkPoint *sat_points;
    GdkPoint *val_points;
    float *hue;
    float *sat;
    float *val;
    float *red;
    float *green;
    float *blue;
    float lowbound;
    float highbound;
} _hsv_lut;

typedef struct {
    GtkWidget *main_window;
    GtkWidget *table;
    GtkWidget *hue_drawing_area;
    GtkWidget *lut_drawing_area;
    _hsv_lut *lut;
    G_env *genv;
} _hsv_context;

static int _scale(int value, int src_dim, int dest_dim)
{
    return (int)((float)value / (src_dim - 1) * (dest_dim - 1) + 0.5);
}

static void _hsv_redraw(_hsv_context *context)
{
    gtk_widget_queue_draw( context->main_window);
}

static void _hsv_lut_alloc(_hsv_lut *lut, int size)
{
    lut->size = size;
    lut->hue_points = calloc(size, sizeof(GdkPoint));
    lut->sat_points = calloc(size, sizeof(GdkPoint));
    lut->val_points = calloc(size, sizeof(GdkPoint));
    lut->hue   = calloc(size, sizeof(float));
    lut->sat   = calloc(size, sizeof(float));
    lut->val   = calloc(size, sizeof(float));
    lut->red   = calloc(size, sizeof(float));
    lut->green = calloc(size, sizeof(float));
    lut->blue  = calloc(size, sizeof(float));
    lut->lowbound = 0;
    lut->highbound = 360.;
}

static void _hsv_lut_free(_hsv_lut *lut)
{
    if (lut == NULL)
        return;
    free(lut->hue_points);
    free(lut->sat_points);
    free(lut->val_points);
    free(lut->hue);
    free(lut->sat);
    free(lut->val);
    free(lut->red);
    free(lut->green);
    free(lut->blue);
    free(lut);
}

static void _hsv_context_free(_hsv_context *context)
{
    if (context == NULL)
        return;
    if (context->lut != NULL) {
        _hsv_lut_free(context->lut);
        context->lut = NULL;
    }
    free(context);
}

static void _hsv_update_from_rgb(_hsv_lut *lut)
{
    int i;
    float l;

    l = lut->size - 1.;
    for (i = 0; i < lut->size; i++) {
        rgb_to_hsv( &lut->red[i], &lut->green[i], &lut->blue[i], &lut->hue[i],
         &lut->sat[i], &lut->val[i]);
        lut->hue_points[i].y = (l - 1.) * (1. - lut->hue[i] / 360.);
        lut->sat_points[i].y = (l - 1.) * (1. - lut->sat[i]);
        lut->val_points[i].y = (l - 1.) * (1. - lut->val[i]);
    }
}

int _load_default_colormap(_hsv_lut *lut)
{
    int i;
    GdkColor *default_colormap;
    int default_colormap_size;

    default_colormap_size = ggtk_default_colormap_size();
    if (lut == NULL) {
        return default_colormap_size;
    }
    default_colormap = ggtk_default_colormap();
    for (i = 0; i < lut->size; i++) {
        int j = _scale(i, lut->size, default_colormap_size);
        lut->red[i] = (float)default_colormap[j].red / MAX_INTENSITY;
        lut->green[i] = (float)default_colormap[j].green / MAX_INTENSITY;
        lut->blue[i] = (float)default_colormap[j].blue / MAX_INTENSITY;
    }
    return lut->size;
}

static void _hsv_lut_init(_hsv_lut *lut)
{
    int i;
#ifndef _RESET_ON_INIT
    _load_default_colormap(lut);
    _hsv_update_from_rgb(lut);
#else
    float l;
    float t;

    l = lut->size - 1.;
#endif
    for (i = 0; i < lut->size; i++) {
        lut->hue_points[i].x = i;
        lut->sat_points[i].x = i;
        lut->val_points[i].x = i;

#ifndef _RESET_ON_INIT
    }
#else
        t = i / l;

        //hp[i] = t;
        lut->hue_points[i].y = (l - 1.) * (1. - t);
        lut->sat_points[i].y = 0;
        lut->val_points[i].y = 0;

        lut->hue[i] = 360. * t;
        lut->val[i] = 1.;
        lut->sat[i] = 1.;
    }
    for (i = 0; i < lut->size; i++) {
        hsv_to_rgb( &lut->hue[i], &lut->sat[i], &lut->val[i], &lut->red[i],
         &lut->green[i], &lut->blue[i]);
    }
#endif
}

static void _hsv_draw_wedge(GtkWidget *widget, cairo_t *cr, _hsv_context *context)
{
    _hsv_lut *lut = context->lut;
    const int width = gtk_widget_get_allocated_width(widget);
    const int height = gtk_widget_get_allocated_height(widget);
    int i, j, k;

    cairo_save(cr);
    cairo_set_source_rgb(cr, 0.0, 0.0, 0.0);
    cairo_paint(cr);
    cairo_restore(cr);

    cairo_surface_t *surface = cairo_image_surface_create(CAIRO_FORMAT_RGB24,
        width, height);
    if (surface == NULL) {
        ggtk_c_message(seve_e, "GTK", "Unable to create cairo surface");
        return;
    }
    unsigned char *cbuf = cairo_image_surface_get_data(surface);
    const int stride = cairo_image_surface_get_stride(surface);
    if (stride <= 0) {
        ggtk_c_message(seve_e, "GTK", "Invalid stride");
        cairo_surface_destroy(surface);
        return;
    }
    for (j = 0; j < height; j++) {
        for (i = 0; i < width; i++) {
            k = _scale(i, width, lut->size);
            const unsigned char red = (unsigned char)((gushort)(lut->red[k] * MAX_INTENSITY) >> 8);
            const unsigned char green = (unsigned char)((gushort)(lut->green[k] * MAX_INTENSITY) >> 8);
            const unsigned char blue = (unsigned char)((gushort)(lut->blue[k] * MAX_INTENSITY) >> 8);
            cbuf[j * stride + i * 4] = blue;
            cbuf[j * stride + i * 4 + 1] = green;
            cbuf[j * stride + i * 4 + 2] = red;
            cbuf[j * stride + i * 4 + 3] = 0;
        }
    }
    cairo_surface_mark_dirty(surface);
    cairo_set_source_surface(cr, surface, 0, 0);
    cairo_paint(cr);
    cairo_surface_destroy(surface);
}

static void _draw_lines(GtkWidget *widget, cairo_t *cr, GdkPoint *points, int size)
{
    const int width = gtk_widget_get_allocated_width(widget);
    const int height = gtk_widget_get_allocated_height(widget);
    if (size <= 0 || width <= 0 || height <= 0)
        return;

    cairo_save(cr);
    cairo_set_source_rgb(cr, 0.0, 0.0, 0.0);
    cairo_paint(cr);
    cairo_restore(cr);

    cairo_set_source_rgb(cr, 1.0, 1.0, 1.0);
    cairo_set_line_width(cr, 1.0);

    double x = _scale(points[0].x, size, width);
    double y = _scale(points[0].y, size, height);
    cairo_move_to(cr, x, y);
    for (int i = 1; i < size; i++) {
        x = _scale(points[i].x, size, width);
        y = _scale(points[i].y, size, height);
        cairo_line_to(cr, x, y);
    }
    cairo_stroke(cr);
}

static gboolean _hsv_draw_callback(GtkWidget *widget, cairo_t *cr, gpointer data)
{
    _hsv_context *context = (_hsv_context *)g_object_get_data(G_OBJECT(widget),
     "CONTEXT");
    hsv_drawing_cell_id id = (hsv_drawing_cell_id)GPOINTER_TO_SIZE(data);

    switch (id) {
    case HSV_HUE:
        _draw_lines(widget, cr, context->lut->hue_points, context->lut->size);
        break;
    case HSV_SAT:
        _draw_lines(widget, cr, context->lut->sat_points, context->lut->size);
        break;
    case HSV_VAL:
        _draw_lines(widget, cr, context->lut->val_points, context->lut->size);
        break;
    case HSV_LUT:
        _hsv_draw_wedge(widget, cr, context);
        break;
    }
    return TRUE; // handled
}

static GtkWidget *_hsv_create_drawing_area(hsv_drawing_cell_id id, _hsv_context
 *context, int col_start, int col_end, int row, int width, int height)
{
    GtkWidget *drawing_area;
    int colspan;

    drawing_area = gtk_drawing_area_new();
    gtk_widget_set_size_request( drawing_area, width, height);
    gtk_widget_set_hexpand(drawing_area, TRUE);
    gtk_widget_set_vexpand(drawing_area, FALSE);
    gtk_widget_set_halign(drawing_area, GTK_ALIGN_FILL);
    gtk_widget_set_valign(drawing_area, GTK_ALIGN_START);
    colspan = col_end - col_start + 1;
    if (colspan < 1) colspan = 1;
    gtk_grid_attach( GTK_GRID(context->table), drawing_area, col_start, row,
     colspan, 1);
    gtk_widget_set_can_focus( drawing_area, FALSE);
    g_signal_connect( drawing_area, "draw",
     G_CALLBACK(_hsv_draw_callback), GINT_TO_POINTER(id));
    g_object_set_data(G_OBJECT(drawing_area), "CONTEXT", context);

    return drawing_area;
}

static GtkWidget *_hsv_create_square_drawing_area(hsv_drawing_cell_id id,
 _hsv_context *context, int col, int row)
{
    return _hsv_create_drawing_area(id, context, col, col, row,
     _HSV_DRAWING_WIDTH, _HSV_DRAWING_HEIGHT);
}

enum {
    LIST_ITEM = 0,
    PATH,
    N_COLUMNS
};

static void
_init_list (GtkWidget *list)
{

    GtkCellRenderer         *renderer;
    GtkTreeViewColumn       *column;
    GtkListStore            *store;

    /* create and append the single column */

    renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes ("Existing luts",
            renderer, "text", LIST_ITEM, NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (list), column);

    /* create the model and add to tree view */

    store = gtk_list_store_new (N_COLUMNS, G_TYPE_STRING, G_TYPE_STRING);

    gtk_tree_view_set_model (GTK_TREE_VIEW (list), GTK_TREE_MODEL (store));

    /* free reference to the store */

    g_object_unref (store);
}

static void
_add_to_list (GtkWidget *list, const gchar *str, const gchar *path)
{
    GtkListStore            *store;
    GtkTreeIter             iter;

    store = GTK_LIST_STORE (gtk_tree_view_get_model (GTK_TREE_VIEW (list)));

    gtk_list_store_append (store, &iter);
    gtk_list_store_set (store, &iter, LIST_ITEM, str, PATH, path, -1);
}

static void _hsv_get_luts(GtkWidget *list)
{
    char *path;
#ifndef WIN32
    DIR *dirp;
    struct dirent *dp;
#else
    struct _finddata_t lut_file;
    long hFile;
#endif
    char **entries = NULL;
    size_t entries_count = 0;
    size_t entries_cap = 0;
    size_t i;

    /* add predefined lut names */
    _add_to_list(list, "default", NULL);
    _add_to_list(list, "color", NULL);
    _add_to_list(list, "black", NULL);
    _add_to_list(list, "white", NULL);
    _add_to_list(list, "red", NULL);
    _add_to_list(list, "green", NULL);
    _add_to_list(list, "blue", NULL);
    _add_to_list(list, "yellow", NULL);
    _add_to_list(list, "cyan", NULL);
    _add_to_list(list, "magenta", NULL);
    _add_to_list(list, "null", NULL);

    /* add lut file name from gag_lut: */
    path = sic_s_get_logical_path( "gag_lut:");
#ifndef WIN32
    dirp = opendir( path);
    if (dirp != NULL) {
        while ((dp = readdir(dirp)) != NULL) {
            if (dp->d_type == DT_REG) {
                if (entries_count == entries_cap) {
                    size_t new_cap = entries_cap ? entries_cap * 2 : 16;
                    char **tmp = realloc(entries, new_cap * sizeof(char *));
                    if (tmp == NULL) {
                        continue;
                    }
                    entries = tmp;
                    entries_cap = new_cap;
                }
                entries[entries_count++] = strdup(dp->d_name);
            }
        }
        (void)closedir(dirp);
    }
#else
    strcat( path, "*.*");
    if( (hFile = _findfirst(path, &lut_file)) != -1L ) {
        do {
            if (entries_count == entries_cap) {
                size_t new_cap = entries_cap ? entries_cap * 2 : 16;
                char **tmp = realloc(entries, new_cap * sizeof(char *));
                if (tmp == NULL) {
                    break;
                }
                entries = tmp;
                entries_cap = new_cap;
            }
            entries[entries_count++] = strdup(lut_file.name);
        } while(_findnext(hFile, &lut_file) == 0);
        _findclose(hFile);
    }
#endif
    if (entries_count > 0) {
        for (i = 0; i < entries_count; i++) {
            size_t j = i;
            while (j > 0 && strcmp(entries[j - 1], entries[j]) > 0) {
                char *tmp = entries[j];
                entries[j] = entries[j - 1];
                entries[j - 1] = tmp;
                j--;
            }
        }
        for (i = 0; i < entries_count; i++) {
            _add_to_list(list, entries[i], "gag_lut:");
            free(entries[i]);
        }
        free(entries);
    }
}

#if 0
static void _load_from_file(char *filename)
{
    char *path;
    int i;
    FILE *lut_fp;
    char lutname[SIC_MAX_PATH_LENGTH];

    path = sic_s_get_logical_path( "gag_lut:");
    sprintf( lutname, "%s/%s", path, filename);
    lut_fp = fopen( lutname, "r");

    if (lut_fp != NULL) {
        for (i = 0; i < context->lut->size; i++) {
            if (fscanf( lut_fp, "%f %f %f", &context->lut->red[i],
             &context->lut->green[i], &context->lut->blue[i]) != 3) {
                ggtk_c_message(seve_w, "HSVCONTROL",
                 "Unable to read values in %s", lutname);
                return;
            }
        }
        fclose( lut_fp);

        _hsv_update_from_rgb(context->lut);

        _hsv_redraw(context);
    }
}
#endif

static void _post_silent_command( const char *args, ...)
{
    va_list l;
    char command[MAXBUF];

    va_start( l, args);
    vsprintf( command, args, l);
    if (sic_post_command_text_from( command, SIC_SYNCHRONOUS_MODE) == -1) {
        ggtk_c_message(seve_w, "HSVCONTROL", "Unable to post command %s",
         command);
    }
    va_end( l);
}

static void _hsv_update_after_lut(void *_context)
{
    _hsv_context *context = (_hsv_context *)_context;
    _load_default_colormap( context->lut);
    _hsv_update_from_rgb( context->lut);
    _hsv_redraw( context);
}

static void _hsv_list_cursor_changed(GtkTreeView *tree_view, gpointer user_data)
{
    _hsv_context *context = (_hsv_context *)user_data;
    GtkTreeSelection *sel;
    GtkTreeModel *model;
    GtkTreeIter iter;

    sel = gtk_tree_view_get_selection( tree_view);
    if (gtk_tree_selection_get_selected( sel, &model, &iter)) {
        char filename[SIC_MAX_PATH_LENGTH];
        char *str;

        gtk_tree_model_get( model, &iter, PATH, &str, -1);
        if (str != NULL)
            strcpy( filename, str);
        else
            filename[0] = '\0';
        gtk_tree_model_get( model, &iter, LIST_ITEM, &str, -1);
        strcat( filename, str);
        _post_silent_command( "GTVL\\LUT \"%s\"", filename);
        gtv_push_callback( _hsv_update_after_lut, context); // execute when event stack is empty
    }
}

static void _on_response(GtkDialog *dialog, gint response_id, gpointer
 user_data)
{
    if (response_id < 0 || response_id == GTK_RESPONSE_CLOSE) {
        gtk_widget_destroy(GTK_WIDGET(dialog));
    }
}

static void _hsv_destroy_callback(GtkWidget *widget, gpointer data)
{
    _hsv_context_free((_hsv_context *)data);
}

void create_hsv_control( G_env *genv)
{
    _hsv_context *context;
    int size;
    GtkWidget *main_window;
    GtkWidget *main_box;
    GtkWidget *table;
    GtkWidget *w;
    GtkWidget *sat_drawing_area;
    GtkWidget *val_drawing_area;
    GtkWidget *sw;
    GtkWidget *list;

    /* Create a context */
    context = malloc(sizeof(_hsv_context));
    context->genv = genv;
#ifdef _GGTK_USE_DEFAULT_SIZE
    size = _load_default_colormap(NULL);
#else
    size = 2048;
#endif
    context->lut = malloc(sizeof(_hsv_lut));
    _hsv_lut_alloc(context->lut, size);
    _hsv_lut_init(context->lut);

    /* Create a new window */
#ifdef _GGTK_USE_DIALOG
    main_window = gtk_dialog_new();
#else
    main_window = gtk_window_new( GTK_WINDOW_TOPLEVEL);
#endif
    context->main_window = main_window;
    g_object_set_data(G_OBJECT(context->main_window), "CONTEXT", context);
    g_signal_connect(context->main_window, "destroy",
     G_CALLBACK(_hsv_destroy_callback), context);

    /* Set the window title */
    gtk_window_set_title( GTK_WINDOW(main_window), "LUT Selector");

    /* Sets the border width of the window. */
    gtk_container_set_border_width( GTK_CONTAINER(main_window), 1);

#ifdef _GGTK_USE_DIALOG
    main_box = gtk_dialog_get_content_area(GTK_DIALOG(main_window));
#else
    main_box = gtk_box_new( GTK_ORIENTATION_VERTICAL, 0);
    gtk_container_add( GTK_CONTAINER(main_window), main_box);
#endif

    table = gtk_grid_new();
    gtk_grid_set_column_spacing(GTK_GRID(table), 2);
    gtk_grid_set_row_spacing(GTK_GRID(table), 2);
    context->table = table;
    gtk_widget_set_hexpand(table, FALSE);
    gtk_widget_set_vexpand(table, FALSE);
    gtk_box_pack_start( GTK_BOX(main_box), table, FALSE, FALSE, 0);

    w = gtk_label_new( "Hue");
    gtk_widget_set_halign(w, GTK_ALIGN_CENTER);
    gtk_grid_attach( GTK_GRID(table), w, 0, 0, 1, 1);
    w = gtk_label_new( "Saturation");
    gtk_widget_set_halign(w, GTK_ALIGN_CENTER);
    gtk_grid_attach( GTK_GRID(table), w, 1, 0, 1, 1);
    w = gtk_label_new( "Value");
    gtk_widget_set_halign(w, GTK_ALIGN_CENTER);
    gtk_grid_attach( GTK_GRID(table), w, 2, 0, 1, 1);

    context->hue_drawing_area = _hsv_create_square_drawing_area(HSV_HUE,
     context, 0, 1);
    sat_drawing_area = _hsv_create_square_drawing_area(HSV_SAT, context, 1, 1);
    val_drawing_area = _hsv_create_square_drawing_area(HSV_VAL, context, 2, 1);
    context->lut_drawing_area = _hsv_create_drawing_area(HSV_LUT, context, 0, 2,
     2, _HSV_LUT_HEIGHT, _HSV_LUT_HEIGHT);

    w = gtk_separator_new(GTK_ORIENTATION_HORIZONTAL);
    gtk_box_pack_start( GTK_BOX(main_box), w, FALSE, FALSE, 0);

    w = gtk_label_new( "Select existing luts");
    //gtk_container_add( GTK_CONTAINER(main_box), w);
    gtk_box_pack_start( GTK_BOX(main_box), w, FALSE, FALSE, 0);

    sw = gtk_scrolled_window_new(NULL, NULL);
    list = gtk_tree_view_new();

    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
     GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    list = gtk_tree_view_new();
    g_signal_connect( list, "cursor-changed",
     G_CALLBACK(_hsv_list_cursor_changed), context);

    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(list), FALSE);
    _init_list(list);
    _hsv_get_luts(list);
    gtk_container_add( GTK_CONTAINER(sw), list);
    gtk_box_pack_start( GTK_BOX(main_box), sw, TRUE, TRUE, 0);

#ifdef _GGTK_USE_DIALOG
    gtk_dialog_add_button(GTK_DIALOG(main_window), "Close", GTK_RESPONSE_CLOSE);
    g_signal_connect(main_window, "response", G_CALLBACK(_on_response),
     context);
#else
    {
    GtkWidget *hbox;

    hbox = gtk_box_new( GTK_ORIENTATION_HORIZONTAL, 5);
    w = gtk_button_new_with_label("Close");
    g_signal_connect_swapped( w, "clicked",
     G_CALLBACK(gtk_widget_destroy), main_window);
    gtk_box_pack_end( GTK_BOX(hbox), w, FALSE, FALSE, 5);
    gtk_box_pack_start( GTK_BOX(main_box), hbox, FALSE, FALSE, 5);
    }
#endif

    gtk_widget_show_all( main_window);
}

static int _on_create_hsv_control( void *data)
{
    create_hsv_control((G_env *)data);
    // do not call again
    return FALSE;
}

int run_hsv_control( G_env *genv)
{
    // g_idle_add not needed because already in gtk thread
    //g_idle_add(_on_create_hsv_control, (gpointer)genv);
    create_hsv_control( genv);
    return 0;
}
