/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package help.validator.model;

import ghidra.util.exception.AssertException;
import help.HelpBuildUtils;

import java.nio.file.Path;

/**
 * A representation of a help location, which can be a file or a file with an anchor inside of
 * that file.
 * <p>
 * This class is used to generate target information for TOC files and to generate link information
 * for the help map files.
 * <p>
 * <br>
 * <p>
 * <b>Warning: </b> The ID generated by this class is specific to the JavaHelp system.  It is of 
 * the format:
 * <p>
 * <code>TopicName_anchorName</code>
 * <p>
 * or
 * <p>
 * <code>TopicName_Filename</code>
 */
public class AnchorDefinition {

	private final Path sourceFile;
	private final Path helpRelativePath;
	private final String anchorName;
	private final int lineNum;
	private final String ID;

	public AnchorDefinition(Path file, String anchorName, int lineNum) {
		this.sourceFile = file;
		this.anchorName = anchorName;
		this.lineNum = lineNum;

		String prefix = getAnchorDefinitionPrefix(sourceFile);

		String anchor = anchorName;
		if (anchor == null) {
			anchor = getDefaultAnchor(sourceFile);
		}

		String rawID = prefix + "_" + anchor;
		ID = rawID.replace(' ', '_').replace('-', '_').replace('.', '_');

		this.helpRelativePath = HelpBuildUtils.relativizeWithHelpTopics(file);
	}

	private String getAnchorDefinitionPrefix(Path anchorSourceFile) {
		Path topicDir = HelpBuildUtils.getHelpTopicDir(anchorSourceFile);
		if (topicDir == null) {
			throw new AssertException(
				"Anchor defined in a file that does not live inside of a help topic");
		}

		return topicDir.getFileName().toString();
	}

	private String getDefaultAnchor(Path file) {
		String filename = file.getFileName().toString();
		int extension = filename.toLowerCase().indexOf(".htm");
		if (extension != -1) {
			return filename.substring(0, extension);
		}
		throw new AssertException("Cannot have HTML file without an .html extension");
	}

	public String getAnchorName() {
		return anchorName;
	}

	public Path getSrcFile() {
		return sourceFile;
	}

	public String getId() {
		return ID;
	}

	public int getLineNumber() {
		return lineNum;
	}

	private String getSource() {
		if (lineNum >= 0) {
			return "(line " + lineNum + ") in " + sourceFile.getFileName();
		}
		return "(File ID) in " + sourceFile.getFileName();
	}

	public String getHelpPath() {
		if (anchorName == null) {
			return helpRelativePath.toString();
		}
		return helpRelativePath.toString() + '#' + anchorName;
	}

	@Override
	public String toString() {
		if (lineNum < 0) {
			return "Anchor Definition: " + ID + getSource();
		}
		return "<a name=\"" + anchorName + "\"> " + getSource();
	}
}
