searchData={"items":[{"type":"module","doc":"Interface functions for TLS (Transport Layer Security)\nand DTLS (Datagram Transport Layer Security).\n\n> #### Note {: .info }\nThe application's name is still SSL because the first versions of the\nTLS protocol were named SSL (Secure Socket Layer). However, no version\nof the old SSL protocol is supported by this application.\n\nExample:\n```erlang\n1> ssl:start(), ssl:connect(\"google.com\", 443, [{verify, verify_peer},\n    {cacerts, public_key:cacerts_get()}]).\n{ok,{sslsocket, [...]}}\n```\n\nSee [Examples](using_ssl.md) for detailed usage and more examples of\nthis API.\n\nSpecial Erlang node configuration for the application can be found in\n[SSL Application](ssl_app.md).","title":"ssl","ref":"ssl.html"},{"type":"function","doc":"Make `Deferred` suites become the least preferred suites.\n\nThe `Deferred` suites will be put at the end of the cipher suite list\n`Suites` after removing them from `Suites` if present. `Deferred` can\nbe a list of cipher suites or a list of filters in which case the\nfilters are used on `Suites` to extract the deferred cipher list.","title":"ssl.append_cipher_suites/2","ref":"ssl.html#append_cipher_suites/2"},{"type":"function","doc":"Lists all available cipher suites corresponding to `Description`.\n\nThe `exclusive` and `exclusive_anonymous` option will exclusively\nlist cipher suites first supported in `Version`, whereas the other options are\ninclusive from the lowest possible version to `Version`. The `all` option\nincludes all suites except anonymous suites. No anonymous suites are supported\nby default.\n\n> #### Note {: .info }\n>\n> TLS-1.3 has no overlapping cipher suites with previous TLS versions, meaning that\n> the result of [`cipher_suites(all, 'tlsv1.3')`](`cipher_suites/2`) contains a separate\n> set of suites that can be used with TLS-1.3 and another set that can be used if a lower\n> version is negotiated. The so-called `PSK` and `SRP` suites (prior to TLS-1.3)\n> need extra configuration to work; namely the option `user_lookup_function`. No\n> anonymous suites are supported by TLS-1.3.\n>\n> Also note that the cipher suites returned by this function are the cipher\n> suites that the OTP SSL application can support provided that they are\n> supported by the crypto library linked with the OTP Crypto application. Use\n> [`ssl:filter_cipher_suites(Suites, [])`](`filter_cipher_suites/2`) to filter\n> the list for the current crypto library. Note that cipher suites may be filtered\n> out because they are too old or too new depending on the crypto library.","title":"ssl.cipher_suites/2","ref":"ssl.html#cipher_suites/2"},{"type":"function","doc":"Equivalent to `cipher_suites/2`, but lists RFC or OpenSSL string names instead of\n[`erl_cipher_suite()`](`t:erl_cipher_suite/0`).","title":"ssl.cipher_suites/3","ref":"ssl.html#cipher_suites/3"},{"type":"function","doc":"Clears the PEM cache.\n\nPEM files, used by SSL API-functions, are cached for performance\nreasons. The cache is automatically checked at regular intervals to\ndetermine whether any cache entries should be invalidated.\n\nThis function provides a way to unconditionally clear the entire cache, thereby\nforcing a reload of previously cached PEM files.","title":"ssl.clear_pem_cache/0","ref":"ssl.html#clear_pem_cache/0"},{"type":"function","doc":"Closes a TLS/DTLS connection.","title":"ssl.close/1","ref":"ssl.html#close/1"},{"type":"function","doc":"Closes or downgrades a TLS connection.\n\nIn the latter case the transport connection will be handed over to the\n`NewController` process after receiving the TLS close alert from the\npeer. The returned transport socket will have the following options\nset: `[{active, false}, {packet, 0}, {mode, binary}]`.\n\nIn case of downgrade, the close function might return some binary data that\nshould be treated by the user as the first bytes received on the downgraded\nconnection.","title":"ssl.close/2","ref":"ssl.html#close/2"},{"type":"function","doc":"","title":"ssl.connect/2","ref":"ssl.html#connect/2"},{"type":"function","doc":"Opens a TLS/DTLS connection.\n\n```erlang\nconnect(TCPSocket, TLSOptions, Timeout).\n```\n\nUpgrades a `gen_tcp` (or equivalent) connected socket to a TLS socket by\nperforming the client-side TLS handshake.\n\n\n```erlang\nconnect(Host, Port, TLSOptions).\n```\n\nOpens a TLS/DTLS connection to `Host`, `Port`. This call is equivalent to:\n\n```erlang\nconnect(Host, Port, TLSOptions, infinity).\n```","title":"ssl.connect/3","ref":"ssl.html#connect/3"},{"type":"function","doc":"Opens a TLS/DTLS connection to `Host`, `Port`.\n\nWhen the `verify` option is set to `verify_peer`, the\n`public_key:pkix_verify_hostname/2` check will be performed in addition to the usual\nX.509-path validation checks. If the check fails, the error `{bad_cert,\nhostname_check_failed}` will be propagated to the path validation fun,\nwhere it is possible to do customized checks\nby using the full possibilities of the `public_key:pkix_verify_hostname/3` API.\nWhen the `server_name_indication` option is provided, its value (the DNS name)\nwill be used as `ReferenceID` to `public_key:pkix_verify_hostname/2`. When no\n`server_name_indication` option is given, the `Host` argument will be used as\nServer Name Indication extension. The `Host` argument will also be used for the\n`public_key:pkix_verify_hostname/2` check. If the `Host` argument is an\n[`inet:ip_address()`](`t:inet:ip_address/0`) the `ReferenceID` used for the\ncheck will be `{ip, Host}`; otherwise `dns_id` will be assumed with a fallback to\n`ip` if that fails.\n\n> #### Note {: .info }\n>\n> According to good practices, certificates should not use IP addresses as\n> \"server names\", especially outside a closed network.\n\nIf the `{handshake, hello}` option is used, the handshake is paused after\nreceiving the server hello message and the success response is\n`{ok, SslSocket, Ext}` instead of `{ok, SslSocket}`. Thereafter the handshake is\ncontinued or canceled by calling `handshake_continue/3` or `handshake_cancel/1`.\n\nIf the `active` option is set to `once`, `true`, or an integer value, the process\nowning the SSL socket will receive messages of type\n[`active_msgs()`](`t:active_msgs/0`).","title":"ssl.connect/4","ref":"ssl.html#connect/4"},{"type":"function","doc":"Returns the most relevant information about the connection.\n\nSome items that are undefined will be filtered out. No values\nthat affect the security of the connection will be returned.\n\n> #### Note {: .info }\n>\n> The legacy `cipher_suite` item was removed in OTP 23. Previously it returned\n> the cipher suite in its (undocumented) legacy format. It is replaced by\n> `selected_cipher_suite`.","title":"ssl.connection_information/1","ref":"ssl.html#connection_information/1"},{"type":"function","doc":"Returns the requested information items about the connection if they are\ndefined.\n\nNote that the values for `client_random`, `server_random`, `master_secret`, and `keylog`\naffect the security of connection.\n\nIn order to retrieve `keylog` and other secret information from a TLS 1.3\nconnection, the `keep_secrets` option must be configured in advance and\nset to `true`.\n\n> #### Note {: .info }\n>\n> If only undefined options are requested the resulting list can be empty.","title":"ssl.connection_information/2","ref":"ssl.html#connection_information/2"},{"type":"function","doc":"Assigns a new controlling process to the SSL socket.\n\nA controlling process is the owner of an SSL socket and receives all\nmessages from the socket.","title":"ssl.controlling_process/2","ref":"ssl.html#controlling_process/2"},{"type":"function","doc":"Returns a list of all supported elliptic curves, including legacy\ncurves, for all TLS/DTLS versions prior to TLS-1.3.","title":"ssl.eccs/0","ref":"ssl.html#eccs/0"},{"type":"function","doc":"Returns the elliptic curves supported by default for `Version`.\n\nThis is a subset of what `eccs/0` returns.","title":"ssl.eccs/1","ref":"ssl.html#eccs/1"},{"type":"function","doc":"","title":"ssl.export_key_materials/4","ref":"ssl.html#export_key_materials/4"},{"type":"function","doc":"Uses a Pseudo-Random Function (PRF prior to TLS-1.3) or a Key\nDerivation Function (HKDF in TLS-1.3) for a TLS connection to\ngenerate and export keying materials.\n\nIn TLS-1.3, using `no_context` is equivalent to specifying an empty\ncontext (an empty binary). Prior to TLS-1.3, `no_context` and an empty\ncontext will produce different results.\n\nThe `ConsumeSecret` argument is relevant only in TLS-1.3, causing the\nTLS-1.3 `exporter_master_secret` to be consumed, thereby making it\nunavailable and increasing security. Further attempts to call this\nfunction will fail.","title":"ssl.export_key_materials/5","ref":"ssl.html#export_key_materials/5"},{"type":"function","doc":"Removes cipher suites if any of the filter functions returns `false` for any part\nof the cipher suite.\n\nIf no filter function is supplied for some part, the default behavior\ntreats it as a filter function that returns `true`. For\nexamples, see [Customizing cipher suites\n](using_ssl.md#customizing-cipher-suites). Additionally, this function\nalso filters the cipher suites to exclude cipher suites not supported\nby the crypto library used by the OTP Crypto application, meaning that\n[`ssl:filter_cipher_suites(Suites, [])`](`filter_cipher_suites/2`)\nis equivalent to applying only the filters for crypto library support.","title":"ssl.filter_cipher_suites/2","ref":"ssl.html#filter_cipher_suites/2"},{"type":"function","doc":"Presents the error returned by an SSL function as a printable string.","title":"ssl.format_error/1","ref":"ssl.html#format_error/1"},{"type":"function","doc":"Gets the values of the specified socket options.","title":"ssl.getopts/2","ref":"ssl.html#getopts/2"},{"type":"function","doc":"Get statistics for the underlying socket.","title":"ssl.getstat/1","ref":"ssl.html#getstat/1"},{"type":"function","doc":"Get one or more statistic values for the underlying socket.\n\nSee `inet:getstat/2` for further details.","title":"ssl.getstat/2","ref":"ssl.html#getstat/2"},{"type":"function","doc":"Returns all supported groups in TLS 1.3.\n\nExisted since OTP 22.0; documented as of OTP 27.","title":"ssl.groups/0","ref":"ssl.html#groups/0"},{"type":"function","doc":"Returns default supported groups in TLS 1.3.\n\nExisted since OTP 22.0; documented as of OTP 27.","title":"ssl.groups/1","ref":"ssl.html#groups/1"},{"type":"function","doc":"","title":"ssl.handshake/1","ref":"ssl.html#handshake/1"},{"type":"function","doc":"Performs the TLS/DTLS server-side handshake.\n\nIf the second argument is a timeout value:\n\n```erlang\nhandshake(HsSocket, Timeout).\n```\n\nthis call is equivalent to:\n\n```erlang\nhandshake(HsSocket, [], Timeout).\n```\n\nOtherwise, if the second argument is a list of options:\n\n\n```erlang\nhandshake(HsSocket, Options).\n```\nthis call is equivalent to:\n\n```erlang\nhandshake(HsSocket, Options, infinity).\n```","title":"ssl.handshake/2","ref":"ssl.html#handshake/2"},{"type":"function","doc":"Performs the TLS/DTLS server-side handshake.\n\nReturns a new TLS/DTLS socket if the handshake is successful.\n\nIf `Socket` is a ordinary [`socket()`](`t:socket/0`), upgrades a\n`gen_tcp` or equivalent socket to an SSL socket by performing the\nTLS server-side handshake and returning a TLS socket.\n\n> #### Note {: .info }\nThe ordinary `Socket` must be in passive mode (`{active, false}`)\nbefore calling this function and before the client tries to connect\nwith TLS; otherwise, the behavior of this function is undefined. The\nbest way to ensure this is to create the ordinary listen socket in\npassive mode.\n\nIf `Socket` is an [`sslsocket()`](`t:sslsocket/0`), provides extra\nTLS/DTLS options to those specified in `listen/2` and then performs\nthe TLS/DTLS handshake. Returns a new TLS/DTLS socket if the handshake\nis successful.\n\n> #### Warning {: .warning }\nNot setting the timeout makes the server more vulnerable to Denial of\nService (DoS) attacks.\n\nIf option `{handshake, hello}` is specified the handshake is paused after\nreceiving the client hello message and the success response is\n`{ok, SslSocket, Ext}` instead of `{ok, SslSocket}`. Thereafter the handshake is\ncontinued or canceled by calling `handshake_continue/3` or `handshake_cancel/1`.\n\nIf option `active` is set to `once`, `true`, or an integer value, the process\nowning the [`sslsocket()`](`t:sslsocket/0`) will receive messages of type\n[`active_msgs()`](`t:active_msgs/0`).","title":"ssl.handshake/3","ref":"ssl.html#handshake/3"},{"type":"function","doc":"Cancel the handshake with a fatal `USER_CANCELED` alert.","title":"ssl.handshake_cancel/1","ref":"ssl.html#handshake_cancel/1"},{"type":"function","doc":"","title":"ssl.handshake_continue/2","ref":"ssl.html#handshake_continue/2"},{"type":"function","doc":"Continue the TLS handshake, possibly with new, additional, or changed options.","title":"ssl.handshake_continue/3","ref":"ssl.html#handshake_continue/3"},{"type":"function","doc":"Creates an SSL listen socket.","title":"ssl.listen/2","ref":"ssl.html#listen/2"},{"type":"function","doc":"Returns the protocol negotiated through ALPN or NPN extensions.","title":"ssl.negotiated_protocol/1","ref":"ssl.html#negotiated_protocol/1"},{"type":"function","doc":"The peer certificate is returned as a DER-encoded binary.\n\nThe certificate can be\ndecoded with `public_key:pkix_decode_cert/2`. Suggested further reading about\ncertificates is [Public_Key User's Guide](`e:public_key:public_key_records.md`)\nand [SSL User's Guide](standards_compliance.md).","title":"ssl.peercert/1","ref":"ssl.html#peercert/1"},{"type":"function","doc":"Returns the address and port number of the peer.","title":"ssl.peername/1","ref":"ssl.html#peername/1"},{"type":"function","doc":"Make `Preferred` suites become the most preferred suites.\n\nThe `Preferred` suites will be put at the head of the cipher suite\nlist `Suites` after removing them from `Suites` if\npresent. `Preferred` can be a list of cipher suites or a list of\nfilters in which case the filters are used on `Suites` to extract the\npreferred cipher list.","title":"ssl.prepend_cipher_suites/2","ref":"ssl.html#prepend_cipher_suites/2"},{"type":"function","doc":"Uses the Pseudo-Random Function (PRF) of a TLS session to generate extra key\nmaterial.\n\nIt either takes user-generated values for `Secret` and `Seed` or atoms\ndirecting it to use a specific value from the session security parameters.\n\n> #### Note {: .info }\n\nThis function is replaced by `export_key_materials/4`, the officially\ndocumented API function since OTP 27, which is equivalent to\n[`prf(TLSSocket, master_secret, Label, [client_random, server_random,\nContext], WantedLength)`](`prf/5`). Other ways of calling this\nfunction were for testing purposes only and has no use case. When\ncalled in a TLS-1.3 context it will now behave as\n[`export_key_materials(TLSSocket, [Label], [Context],\n[WantedLength])`](`export_key_materials/4`).","title":"ssl.prf/5","ref":"ssl.html#prf/5"},{"type":"function","doc":"","title":"ssl.recv/2","ref":"ssl.html#recv/2"},{"type":"function","doc":"Receives a packet from a socket in passive mode.\n\nA closed socket is indicated by return value `{error, closed}`.\nArgument `Length` is meaningful only when the socket is in mode `raw`\nand denotes the number of bytes to read. If `Length` is zero, all\navailable bytes are returned. If `Length` is greater than zero,\nexactly `Length` bytes are returned, or an error; possibly discarding\nless than `Length` bytes of data when the socket gets closed from\nthe other side.\n\nOptional argument `Timeout` specifies a time-out in milliseconds. The default\nvalue is `infinity`.","title":"ssl.recv/3","ref":"ssl.html#recv/3"},{"type":"function","doc":"Initiates a new handshake.\n\nA notable return value is `{error, renegotiation_rejected}` indicating\nthat the peer refused to go through with the renegotiation, but the\nconnection is still active using the previously negotiated session.\n\nTLS-1.3 has removed the renegotiation feature from earlier TLS\nversions and instead adds a new feature called key update, which\nreplaces the most important part of renegotiation: the refreshing of\nsession keys. This is triggered automatically after reaching a\nplaintext limit and can be configured using the `key_update_at` option\nin `t:common_option_tls13/0`.","title":"ssl.renegotiate/1","ref":"ssl.html#renegotiate/1"},{"type":"function","doc":"Writes `Data` to `SslSocket`.\n\nA notable return value is `{error, closed}` indicating that the socket is\nclosed.","title":"ssl.send/2","ref":"ssl.html#send/2"},{"type":"function","doc":"Sets options according to `Options` for socket `SslSocket`.","title":"ssl.setopts/2","ref":"ssl.html#setopts/2"},{"type":"function","doc":"Immediately closes a socket in one or two directions.\n\n`How == write` means closing the socket for writing, but reading from\nit is still possible.\n\nTo handle siutations where the peer has performed a shutdown on the\nwrite side, option `{exit_on_close, false}` is useful.","title":"ssl.shutdown/2","ref":"ssl.html#shutdown/2"},{"type":"function","doc":"Lists all available signature algorithms corresponding to `Description`.\n\nThe `exclusive` option will exclusively list algorithms or algorithm schemes for\nthat protocol version, whereas the `default` and `all` options lists the\ncombined list to support the range of protocols from (D)TLS-1.2, the first\nversion to support configuration of the signature algorithms, to `Version`.\n\nExample:\n\n```erlang\n1> ssl:signature_algs(default, 'tlsv1.3').\n[eddsa_ed25519,eddsa_ed448,ecdsa_secp521r1_sha512,\necdsa_secp384r1_sha384,ecdsa_secp256r1_sha256,\nrsa_pss_pss_sha512,rsa_pss_pss_sha384,rsa_pss_pss_sha256,\nrsa_pss_rsae_sha512,rsa_pss_rsae_sha384,rsa_pss_rsae_sha256,\nrsa_pkcs1_sha512,rsa_pkcs1_sha384,rsa_pkcs1_sha256,\n{sha512,ecdsa},\n{sha384,ecdsa},\n{sha256,ecdsa}]\n\n2> ssl:signature_algs(all, 'tlsv1.3').\n[eddsa_ed25519,eddsa_ed448,ecdsa_secp521r1_sha512,\necdsa_secp384r1_sha384,ecdsa_secp256r1_sha256,\nrsa_pss_pss_sha512,rsa_pss_pss_sha384,rsa_pss_pss_sha256,\nrsa_pss_rsae_sha512,rsa_pss_rsae_sha384,rsa_pss_rsae_sha256,\nrsa_pkcs1_sha512,rsa_pkcs1_sha384,rsa_pkcs1_sha256,\n{sha512,ecdsa},\n{sha384,ecdsa},\n{sha256,ecdsa},\n{sha224,ecdsa},\n{sha224,rsa},\n{sha,rsa},\n{sha,dsa}]\n\n3> ssl:signature_algs(exclusive, 'tlsv1.3').\n[eddsa_ed25519,eddsa_ed448,ecdsa_secp521r1_sha512,\necdsa_secp384r1_sha384,ecdsa_secp256r1_sha256,\nrsa_pss_pss_sha512,rsa_pss_pss_sha384,rsa_pss_pss_sha256,\nrsa_pss_rsae_sha512,rsa_pss_rsae_sha384,rsa_pss_rsae_sha256]\n```\n\n> #### Note {: .info }\n>\n> Some TLS-1-3 scheme names overlap with TLS-1.2 algorithm-tuple-pair-names and\n> then TLS-1.3 names will be used, for example `rsa_pkcs1_sha256` instead of\n> `{sha256, rsa}`. These are legacy algorithms in TLS-1.3 that apply only to\n> certificate signatures in this version of the protocol.","title":"ssl.signature_algs/2","ref":"ssl.html#signature_algs/2"},{"type":"function","doc":"Returns the local address and port number of socket `SslSocket`.","title":"ssl.sockname/1","ref":"ssl.html#sockname/1"},{"type":"function","doc":"","title":"ssl.start/0","ref":"ssl.html#start/0"},{"type":"function","doc":"Starts the SSL application.","title":"ssl.start/1","ref":"ssl.html#start/1"},{"type":"function","doc":"Stops the SSL application.","title":"ssl.stop/0","ref":"ssl.html#stop/0"},{"type":"function","doc":"Converts an RFC or OpenSSL name string to an `t:erl_cipher_suite/0`\n\nReturns an error if the cipher suite is not supported or the name is\nnot a valid cipher suite name.","title":"ssl.str_to_suite/1","ref":"ssl.html#str_to_suite/1"},{"type":"function","doc":"Converts an [`erl_cipher_suite()`](`t:erl_cipher_suite/0`) value to\nan OpenSSL name string.\n\nPRE TLS-1.3 these names differ for RFC names","title":"ssl.suite_to_openssl_str/1","ref":"ssl.html#suite_to_openssl_str/1"},{"type":"function","doc":"Converts an [`erl_cipher_suite()`](`t:erl_cipher_suite/0`) value to an RFC\nname string.","title":"ssl.suite_to_str/1","ref":"ssl.html#suite_to_str/1"},{"type":"function","doc":"","title":"ssl.transport_accept/1","ref":"ssl.html#transport_accept/1"},{"type":"function","doc":"Accepts an incoming connection request on a listen socket.\n\n`ListenSocket` must be a socket returned from `listen/2`. The socket\nreturned is to be passed to [`handshake/1,2,3`](`handshake/3`) to\ncomplete the handshake and establish the TLS/DTLS connection.\n\n> #### Warning {: .warning }\n>\n> Most API functions require that the TLS/DTLS connection is established to work\n> as expected.\n\nThe accepted socket inherits the options set for `ListenSocket` in `listen/2`.\n\nThe default value for `Timeout` is `infinity`. If `Timeout` is specified and no\nconnection is accepted within the given time, `{error, timeout}` is returned.","title":"ssl.transport_accept/2","ref":"ssl.html#transport_accept/2"},{"type":"function","doc":"Create new session keys.\n\nThere are cryptographic limits on the amount of plaintext which can be safely\nencrypted under a given set of keys. If the amount of data surpasses those\nlimits, a key update is triggered and a new set of keys are installed. See also\nthe `key_update_at` option in `t:common_option_tls13/0`.\n\nThis function can be used to explicitly start a key update on a TLS-1.3\nconnection. There are two types of key updates: if `Type` is `write`,\nonly the writing key is updated; if `Type` is `read_write`, both the\nreading and writing keys are updated.","title":"ssl.update_keys/2","ref":"ssl.html#update_keys/2"},{"type":"function","doc":"Lists information, mainly concerning TLS/DTLS versions, in runtime for debugging\nand testing purposes.\n\n- **`app_vsn`** - The application version of the SSL application.\n\n- **`supported`** - TLS versions supported with current application environment\n  and crypto library configuration. Overridden by a version option on\n  [`connect/2,3,4`](`connect/2`), `listen/2`, and\n  [`handshake/2,3`](`handshake/2`). For the negotiated TLS version, see\n  [`connection_information/1`](`connection_information/1`).\n\n- **`supported_dtls`** - DTLS versions supported with current application\n  environment and crypto library configuration. Overridden by a version option\n  on [`connect/2,3,4`](`connect/2`), `listen/2`, and\n  [`handshake/2,3`](`handshake/2`). For the negotiated DTLS version, see\n  [`connection_information/1`](`connection_information/1`).\n\n- **`available`** - All TLS versions supported with the linked crypto library.\n\n- **`available_dtls`** - All DTLS versions supported with the linked crypto\n  library.\n\n- **`implemented`** - All TLS versions supported by the SSL application if\n  linked with a crypto library with the necessary support.\n\n- **`implemented_dtls`** - All DTLS versions supported by the SSL application if\n  linked with a crypto library with the necessary support.","title":"ssl.versions/0","ref":"ssl.html#versions/0"},{"type":"type","doc":"The type for the messages that are delivered to the owner of a\nTLS/DTLS socket in active mode.\n\nThe `ssl_error` reason may convey a TLS protocol alert if such an event occurs\nafter the connection has been established. The most common case when this will\nhappen is on the client side when a TLS-1.3 server requests a client certificate\nand the provided certificate is not accepted by the server, as it will be\nverified after the server has sent its last handshake message.\n\nThe `ssl_passive` message is sent only when the socket is in `{active, N}` mode\nand the counter has dropped to 0. It indicates that the socket has transitioned\nto passive (`{active, false}`) mode.","title":"ssl.active_msgs/0","ref":"ssl.html#t:active_msgs/0"},{"type":"type","doc":"Claim an intermediate CA in the chain as trusted.\n\n```erlang\nfun(Chain::[public_key:der_encoded()]) ->\n      {trusted_ca, DerCert::public_key:der_encoded()} | unknown_ca.\n```\n\nTLS then uses `public_key:pkix_path_validation/3` with the selected CA\nas the trusted anchor and verifies the rest of the chain.","title":"ssl.anchor_fun/0","ref":"ssl.html#t:anchor_fun/0"},{"type":"type","doc":"Configuration of the entity certificate and its corresponding key.\n\nA certificate (or possibly a list including the certificate and its\nchain certificates, where the entity certificate must be the first\nelement in the list or the first entry in the file) and its associated\nkey. For the PEM file format, there can also be a password associated\nwith the file containing the key.\n\nFor maximum interoperability, the certificates in the chain should be\nin the correct order, as the chain will be sent as-is to the peer. If\nchain certificates are not provided, certificates from the configured\ntrusted CA certificates will be used to construct the chain. See\n[`client_option_cert()`](`t:client_option_cert/0`) and\n[`server_option_cert()`](`t:server_option_cert/0`) for more\ninformation.","title":"ssl.cert_key_conf/0","ref":"ssl.html#t:cert_key_conf/0"},{"type":"type","doc":"Cipher algorithms that can be used for payload encryption.","title":"ssl.cipher/0","ref":"ssl.html#t:cipher/0"},{"type":"type","doc":"Filter that allows you to customize cipher suite list.","title":"ssl.cipher_filters/0","ref":"ssl.html#t:cipher_filters/0"},{"type":"type","doc":"A list of cipher suites that should be supported.\n\nFunction [ssl:cipher_suites/2 ](`cipher_suites/2`) can be used to find all\ncipher suites that are supported by default and all cipher suites that can be\nconfigured.\n\nIf you compose your own `t:cipher_suites/0` make sure they are\nfiltered for crypto library support using [ssl:filter_cipher_suites/2\n](`filter_cipher_suites/2`).\n\nThe following function can help creating customized cipher suite lists:\n\n- [ssl:append_cipher_suites/2 ](`append_cipher_suites/2`)\n- [ssl:prepend_cipher_suites/2](`prepend_cipher_suites/2`)\n- [ssl:suite_to_str/1](`suite_to_str/1`)\n- [ssl:str_to_suite/1](`str_to_suite/1`)\n- [ssl:suite_to_openssl_str/1](`suite_to_openssl_str/1`)\n\n> #### Note {: .info }\nNote that TLS-1.3 and TLS-1.2 use different sets of cipher suites. To\nsupport both versions, cipher suites from both sets need to be\nincluded. If the supplied list does not comply with the configured\nversions or crypto library, that is, resulting in an empty list, the option\nwill fall back to its appropriate default value for the configured\nversions.\n\nNon-default cipher suites, including anonymous cipher suites (prior to\nTLS 1.3), are supported for interoperability and testing\npurposes. These can be used by adding them to your cipher suite\nlist. Note that they also need to be supported and enabled by the peer\nto be actually used, and they may require additional configuration;\nsee [`srp_param_type()`](`t:srp_param_type/0`).","title":"ssl.cipher_suites/0","ref":"ssl.html#t:cipher_suites/0"},{"type":"type","doc":"Cipher suite formats.\n\nFor backwards compatibility, cipher suites can be configured as a\ncolon-separated string of cipher suite RFC names (or even old OpenSSL\nnames). However, a more flexible approach is to use utility functions\ntogether with [`cipher_filters()`](`t:cipher_filters/0`) if a customized\ncipher suite option is needed.","title":"ssl.ciphers/0","ref":"ssl.html#t:ciphers/0"},{"type":"type","doc":"The following options are specific to the client side, or have\ndifferent semantics for the client and server:\n\n- **`{alpn_advertised_protocols, AppProtocols}`** - Application layer protocol\n\n  The list of protocols supported by the client to be sent to the server to be\n  used for an Application-Layer Protocol Negotiation (ALPN). If the server\n  supports ALPN, it will choose a protocol from this list; otherwise it will\n  fail the connection with a `no_application_protocol` alert. A server that does\n  not support ALPN will ignore this value. The list of protocols must not contain\n  an empty binary.\n\n- **`{max_fragment_length, MaxLen}`** - Max fragment length extension\n\n  Specifies the maximum fragment length the client is prepared to accept from the\n  server. See [RFC 6066](http://www.ietf.org/rfc/rfc6066.txt).","title":"ssl.client_option/0","ref":"ssl.html#t:client_option/0"},{"type":"type","doc":"Certificate-related options specific to the client side, or with\ndifferent semantics for the client and server.\n\n- **`{verify, Verify}`** - Verification of certificates\n\n  This option specifies whether certificates are to be verified.\n\n  If `Verify` is `verify_peer`, which is the default, it is required\n  to also provide one of the options `cacerts` or `cacertfile` in\n  order for the certificate verification to succeed. For example, an\n  HTTPS client can use option `{cacerts, public_key:cacerts_get()}` to\n  use the trusted CA certificates provided by the operating system.\n\n  If `Verify` is `verify_none`, all X.509-certificate path\n  validation errors will be ignored.\n\n  > #### Change {: .info }\n  >\n  > The default for `Verify` was changed to `verify_peer` in\n  > Erlang/OTP 26.\n\n- **`{cacerts, CACerts}`** - Trusted certificates\n\n  The DER-encoded trusted certificates. If this option is supplied it overrides\n  option `cacertfile`.\n\n  Function `public_key:cacerts_get/0` can be used to retrieve to the\n  trusted CA certificates provided by the operating system.\n\n- **`{cacertfile, CertFile}`** - End entity certificate\n\n   Path to a file containing PEM-encoded CA certificates. The CA certificates are\n   used during server authentication and when building the client certificate\n   chain.\n\n  > #### Note {: .info }\n  >\n  > When PEM caching is enabled, files provided with this option will be checked\n  > for updates at fixed time intervals specified by the\n  > [ssl_pem_cache_clean](ssl_app.md#configuration) environment parameter.\n\n\n- **`{server_name_indication, SNI}`** - Server Name Indication extension\n\n  Specify the hostname to be used in TLS Server Name Indication extension. If not\n  specified it will default to the `Host` argument of\n  [connect/3,4](`connect/3`) unless it is of type [`inet:ip_address()`](`t:inet:ip_address/0`).\n  The hostname will also be used in the hostname verification of the peer\n  certificate using `public_key:pkix_verify_hostname/2`.\n  The special value `disable` prevents the Server Name Indication extension from\n  being sent and disables the hostname verification check.\n\n- **`{customize_hostname_check, HostNameCheckOpts}`** - Customization option\n\n  Customizes the hostname verification of the peer certificate, as various\n  protocols that use TLS, such as HTTP or LDAP, may require different approaches. For\n  example, here is how to use standard hostname checking for HTTPS implemented in\n  [Public_Key](`e:public_key:public_key_app.md`):\n\n  ```erlang\n  {customize_hostname_check, [{match_fun, public_key:pkix_verify_hostname_match_fun(https)}]}\n  ```\n\n  For futher description of the customize options, see\n  `public_key:pkix_verify_hostname/3`.\n\n- **`{client_certificate_authorities, UseCertAuth}`** - Inter-op hint option\n\n  If `UseCertAuth` is set to `true`, sends the certificate authorities\n  extension in the TLS-1.3 client hello. The default is `false`. Note\n  that setting `UseCertAuth` to `true` can result in a significant\n  overhead if there exists many trusted CA certificates. (Since\n  Erlang/OTP 24.3.)\n\n- **`{stapling, Stapling}`** - Certificate revocation check option\n\n  If `Stapling` is atom `staple` or a map, OCSP stapling will be\n  enabled, meaning that an extension of type \"status_request\" will be\n  included in the client hello to indicate the desire to receive\n  certificate status information.\n\n  If `Stapling` is set to `no_staple` (the default), OCSP stapling will be disabled.\n\n  > #### Note {: .info }\n  >\n  > Even if requested by the client, the OCSP response might not be\n  > provided by the server. In such event, SSL will proceed with\n  > the handshake and generate a `{missing, stapling_response}` logger\n  > event.\n\n  When `Stapling` is given as a map, boolean `ocsp_nonce` key can\n  indicate whether an OCSP nonce should be requested by the client\n  (default is `false`).\n\n  > #### Note {: .info }\n  >\n  > The OCSP response can be provided without a nonce value — even if it was requested\n  > by the client. In such cases SSL will proceed with the handshake and generate\n  > a `{missing, ocsp_nonce}` logger event.","title":"ssl.client_option_cert/0","ref":"ssl.html#t:client_option_cert/0"},{"type":"type","doc":"Legacy client options.\n\n- **`{client_preferred_next_protocols, NextAppProtocols}`** - Next Protocol Negotiation\n\n  ALPN (Application-Layer Protocol Negotiation)\n  deprecates NPN (Next Protocol Negotiation) and this option.\n\n  Indicates that the client wants to perform Next Protocol Negotiation.\n\n  If `Precedence` is `server`, the negotiated protocol is the first protocol to be\n  shown on the server advertised list that is also on the client preference\n  list.\n\n  If `Precedence` is `client`, the negotiated protocol is the first protocol to be\n  shown on the client preference list that is also on the server advertised\n  list.\n\n  If the client does not support any of the server advertised protocols or the\n  server does not advertise any protocols, the client falls back to the first\n  protocol in its list or to the default protocol (if a default is supplied). If\n  the server does not support Next Protocol Negotiation, the connection terminates\n  if no default protocol is supplied.","title":"ssl.client_option_legacy/0","ref":"ssl.html#t:client_option_legacy/0"},{"type":"type","doc":"Options only relevant to TLS versions prior to TLS-1.3.\n\n- **`{reuse_session, SessionRef}`** - Explicit session reuse\n\n  Reuses a specific session.\n\n  Since Erlang/OTP 21.3, if the session was saved earlier using option\n  `{reuse_sessions, save}`, the session can be referred by its session ID.\n\n  Since Erlang/OTP 22.3, the session can be explicitly specified by\n  its session ID and associated data.\n\n  See also\n  [SSL User's Guide, Session Reuse pre TLS 1.3.](using_ssl.md#session-reuse-prior-to-tls-1-3)\n\n- **`{reuse_sessions, Reuse}`** - Enables later session reuse\n\n  When `Reuse` is set to `save`, a new connection will be negotiated and saved for later\n  reuse. The session ID can be fetched with `connection_information/2` and used\n  with the client option `reuse_session`.\n\n  When `Reuse` is set to `true`, automated session reuse will be\n  performed, if possible. If a new session is created, and is unique in regard to previous\n  stored sessions, it will be saved for possible later reuse.\n\n  Since: OTP 21.3.\n\n- **`{psk_identity, PskID}`** - Option for use with PSK cipher suites\n\n  Specifies the identity the client presents to the server. The matching secret is\n  found by the fun given in the `user_lookup_fun` option.\n\n- **`{srp_identity, SrpID}`**  - Option for use SRP cipher suites\n\n  Specifies the username and password to use to authenticate to the server.\n\n- **`{fallback, LegacyFallback}`** - Inter-op legacy client option\n\n  Send special cipher suite TLS_FALLBACK_SCSV to avoid an undesired TLS version\n  downgrade. Defaults to `false`.\n\n  > #### Warning {: .warning }\n  >\n  > This option is not needed in normal TLS usage and must not be used to\n  > implement new clients. However, legacy clients that retries connections in the\n  > following manner:\n  >\n  > `ssl:connect(Host, Port, [...{versions, ['tlsv2', 'tlsv1.1', 'tlsv1']}])`\n  >\n  > `ssl:connect(Host, Port, [...{versions, [tlsv1.1', 'tlsv1']}, {fallback, true}])`\n  >\n  > `ssl:connect(Host, Port, [...{versions, ['tlsv1']}, {fallback, true}])`\n  >\n  > can use it to avoid undesired TLS version downgrade. Note that\n  > TLS_FALLBACK_SCSV must also be supported by the server for the prevention to\n  > work.","title":"ssl.client_option_pre_tls13/0","ref":"ssl.html#t:client_option_pre_tls13/0"},{"type":"type","doc":"Options only relevant for TLS-1.3.\n\n- **`{session_tickets, SessionTickets}`** - Use of session tickets\n\n  Configures the session ticket functionality. Allowed values are `disabled`,\n  `manual`, and `auto`. If it is set to `manual` the client will send the ticket\n  information to user process in a 3-tuple:\n\n  ```erlang\n  {ssl, session_ticket, {SNI, TicketData}}\n  ```\n\n  where `SNI` is the ServerNameIndication and `TicketData` is the extended ticket\n  data that can be used in subsequent session resumptions.\n\n  If it is set to `auto`, the client automatically handles received tickets and\n  tries to use them when making new TLS connections (session resumption with\n  pre-shared keys).\n\n  Ticket lifetime, the number of tickets sent by the server, and the\n  maximum number of tickets stored by the server in stateful mode are configured\n  by [application variables](ssl_app.md#configuration).\n\n  See also\n  [SSL User's Guide, Session Tickets and Session Resumption in TLS 1.3](using_ssl.md#session-tickets-and-session-resumption-in-tls-1-3).\n\n- **`{use_ticket, Tickets}`**\n\n  Configures the session tickets to be used for session resumption. It is a\n  mandatory option in `manual` mode (`{session_tickets, manual}`).\n\n  > #### Note {: .info }\n  >\n  > Session tickets are only sent to the user if option `session_tickets` is set to\n  > `manual`\n  >\n  > This option is supported by TLS-1.3. See also\n  > [SSL User's Guide, Session Tickets and Session Resumption in TLS 1.3](using_ssl.md#session-tickets-and-session-resumption-in-tls-1-3).\n\n- **`{early_data, EarlyData}`**\n\n  Configures the early data to be sent by the client.\n\n  To verify that the server has the intention to process the early\n  data, the following tuple is sent to the user process:\n\n  ```erlang\n  {ssl, SslSocket, {early_data, Result}}\n  ```\n\n  where `Result` is either `accepted` or `rejected`.\n\n  > #### Warning {: .warning }\n  >\n  > It is the responsibility of the user to handle rejected `EarlyData` and to\n  > resend when appropriate.\n\n- **`{middlebox_comp_mode, MiddleBoxMode}`**\n\n  Configures the middlebox compatibility mode for a TLS-1.3 connection.\n\n  A significant number of middleboxes misbehave when a TLS-1.3\n  connection is negotiated. Implementations can increase the chance of\n  making connections through those middleboxes by adapting the TLS-1.3\n  handshake to resemble that of a TLS-1.2 handshake.\n\n  The middlebox compatibility mode is enabled (`true`) by default.","title":"ssl.client_option_tls13/0","ref":"ssl.html#t:client_option_tls13/0"},{"type":"type","doc":"Options common to both client and server side.\n\n- **`{protocol, Protocol}`** - Choose TLS or DTLS protocol for the transport layer security.\n\n  Defaults to `tls`.\n\n- **`{handshake, Completion}`** - Possibly pause handshake at hello stage.\n\n  Defaults to `full`. If `hello` is specified the handshake will pause\n  after the hello message, allowing the user to make decisions based\n  on hello extensions before continuing or aborting the handshake by\n  calling `handshake_continue/3` or `handshake_cancel/1`.\n\n- **`{keep_secrets, KeepSecrets}`** - Configures a TLS 1.3 connection for keylogging.\n\n  In order to retrieve keylog information on a TLS 1.3 connection, it must be\n  configured in advance to keep `client_random` and various handshake secrets.\n\n  The `keep_secrets` functionality is disabled (`false`) by default.\n\n  Added in OTP 23.2.\n\n- **`{max_handshake_size, HandshakeSize}`** - Limit the acceptable handshake packet size.\n\n  Used to limit the size of valid TLS handshake packets to avoid DoS\n  attacks.\n\n  Integer (24 bits, unsigned). Defaults to `256*1024`.\n\n- **`{hibernate_after, HibernateTimeout}`** - Hibernate inactive connection processes.\n\n  When an integer-value is specified, the TLS/DTLS connection goes into hibernation\n  after the specified number of milliseconds of inactivity, thus reducing its\n  memory footprint. When not specified the process never goes into hibernation.\n\n- **`{log_level, Level}`** - Specifies the log level for a TLS/DTLS connection.\n\n  Alerts are logged on `notice`\n  level, which is the default level. The level `debug` triggers verbose logging of\n  TLS/DTLS protocol messages. See also [SSL Application](ssl_app.md)\n\n- **`{receiver|sender_spawn_opts, SpawnOpts}`** - Configure erlang spawn opts.\n\n  Configures spawn options of TLS sender and receiver processes.\n\n  Setting up garbage collection options can be helpful for trade-offs between CPU\n  usage and memory usage. See `erlang:spawn_opt/2`.\n\n  For connections using Erlang distribution, the default sender option\n  is `[...{priority, max}]`; this priority option cannot be changed. For all\n  connections, `...link` is added to receiver and cannot be changed.","title":"ssl.common_option/0","ref":"ssl.html#t:common_option/0"},{"type":"type","doc":"Common certificate related options to both client and server.\n\n- **`{certs_keys, CertsKeys}`** - At least one certificate and key pair.\n\n  A list of a certificate (or possible a certificate and its chain)\n  and the associated key of the certificate that can be used to\n  authenticate the client or the server. The certificate key pair that\n  is considered best and matches negotiated parameters for the\n  connection will be selected.\n\n  The different signature algorithms are prioritized in the following\n  order: `eddsa`, `ecdsa`, `rsa_pss_pss`, `rsa`, and `dsa`. If more\n  than one key is supplied for the same signature algorithm, they will\n  be prioritized by strength (except for _engine keys_; see the next\n  paragraph). This offers flexibility to, for instance, configure a\n  newer certificate that is expected to be used in most cases, and an\n  older but acceptable certificate that will only be used to\n  communicate with legacy systems. Note that there is a trade off\n  between the induced overhead and the flexibility; thus, alternatives\n  should be chosen for good reasons.\n\n  _Engine keys_ will be favored over other keys. As engine keys cannot\n  be inspected, supplying more than one engine key makes no sense.\n\n  When this option is specified it overrides all single certificate\n  and key options. For examples, see the [User's Guide](using_ssl.md).\n\n  > #### Note {: .info }\n  >\n  > `eddsa` certificates are only supported by TLS-1.3 implementations that do not support `dsa`\n  > certificates. `rsa_pss_pss` (RSA certificates using Probabilistic Signature\n  > Scheme) are supported in TLS-1.2 and TLS-1.3, but some TLS-1.2 implementations\n  > do not support `rsa_pss_pss`.\n\n- **`{depth, AllowedCertChainLen}`** - Limits the accepted number of certificates in the certificate chain.\n\n  Maximum number of non-self-issued intermediate certificates that can follow the\n  peer certificate in a valid certification path. So, if depth is 0 the PEER must\n  be signed by the trusted ROOT-CA directly; if 1 the path can be PEER, CA,\n  ROOT-CA; if 2 the path can be PEER, CA, CA, ROOT-CA, and so on. The default\n  value is 10. Used to mitigate DoS attack possibilities.\n\n- **`{verify_fun,  Verify}`** - Customize certificate path validation\n\n  The verification fun is to be defined as follows:\n\n  ```erlang\n  fun(OtpCert :: #'OTPCertificate'{},\n      Event, InitialUserState :: term()) ->\n\t{valid, UserState :: term()} |\n\t{fail, Reason :: term()} | {unknown, UserState :: term()}.\n\n  fun(OtpCert :: #'OTPCertificate'{}, DerCert :: public_key:der_encoded(),\n      Event, InitialUserState :: term()) ->\n\t{valid, UserState :: term()} |\n\t{fail, Reason :: term()} | {unknown, UserState :: term()}.\n\n  Types:\n        Event = {bad_cert, Reason :: atom() |\n                {revoked, atom()}} |\n\t\t{extension, #'Extension'{}} |\n                valid |\n                valid_peer\n  ```\n\n  The verification fun is called during the X.509-path validation when\n  an error occurs or an extension unknown to the SSL application is\n  encountered. It is also called when a certificate is considered\n  valid by the path validation to allow access to each certificate in\n  the path to the user application. It differentiates between the peer\n  certificate and the CA certificates by using `valid_peer` or `valid`\n  as `Event` argument to the verification fun. See the [Public_Key\n  User's Guide](`e:public_key:public_key_records.md`) for definition\n  of `#'OTPCertificate'{}` and `#'Extension'{}`.\n\n  - If the verify callback fun returns `{fail, Reason}`, the verification process\n    is immediately stopped, an alert is sent to the peer, and the TLS/DTLS\n    handshake terminates.\n  - If the verify callback fun returns `{valid, UserState}`, the verification\n    process continues.\n  - If the verify callback fun always returns `{valid, UserState}`, the TLS/DTLS\n    handshake does not terminate regardless of verification failures, and the\n    connection is established.\n  - If called with an extension unknown to the user application, the fun is to\n    return `{unknown, UserState}`.\n\n  Note that if the fun returns `unknown` for an extension marked as critical,\n  validation will fail.\n\n  Default option `verify_fun` in `verify_peer mode`:\n\n  ```erlang\n  {fun(_, _, {bad_cert, _} = Reason, _) ->\n\t {fail, Reason};\n      (_, _, {extension, _}, UserState) ->\n\t {unknown, UserState};\n      (_, _, valid, UserState) ->\n\t {valid, UserState};\n      (_, _, valid_peer, UserState) ->\n         {valid, UserState}\n   end, []}\n  ```\n\n  Default option `verify_fun` in mode `verify_none`:\n\n  ```erlang\n   {fun(_, _, {bad_cert, _}, UserState) ->\n\t {valid, UserState};\n      (_, _, {extension, #'Extension'{critical = true}}, UserState) ->\n\t {valid, UserState};\n      (_, _, {extension, _}, UserState) ->\n\t {unknown, UserState};\n      (_, _, valid, UserState) ->\n\t {valid, UserState};\n      (_, _, valid_peer, UserState) ->\n         {valid, UserState}\n   end, []}\n  ```\n\n  The possible path validation errors are given in the form `{bad_cert, Reason}`,\n  where `Reason` is:\n\n  - **`unknown_ca`**\n\n    No trusted CA was found in the trusted store. The trusted\n    CA is normally a so-called ROOT CA, which is a self-signed certificate. Trust\n    can be claimed for an intermediate CA (the trusted anchor does not have to be\n    self-signed according to X-509) by using option `partial_chain`.\n\n  - **`selfsigned_peer`**\n\n    The chain consisted only of one self-signed certificate.\n\n  - **{invalid_ext_keyusage, [public_key:oid()]} **\n\n   If the peer certificate specifies the extended keyusage extension and does\n   not include the purpose for either being a TLS server (id-kp-ServerAuth) or\n   TLS client (id-kp-ClientAuth) depending on the peers role.\n\n  - **{ca_invalid_ext_keyusage, [public_key:oid()]} **\n\n   If a CA certificate specifies the extended keyusage extension and does\n   not include the purpose for either being a TLS server\n   (id-kp-ServerAuth) or TLS client (id-kp-ClientAuth) depending\n   on the role of the peer chained with this CA, or the option allow_any_ca_purpose is set to `true`\n   but the special any-value (anyExtendedKeyUsage) is not included in the CA cert purposes.\n\n  - **`PKIX X-509-path validation error`**\n\n    For possible reasons, see `public_key:pkix_path_validation/3`.\n\n- **`{cert_policy_opts, PolicyOpts}`** - Handle certificate policies.\n\n  Configure X.509 certificate policy handling for the certificate path validation process;\n  see [public_key:pkix_path_validation/3](`public_key:pkix_path_validation/3`) for\n  more details.\n\n- **`{allow_any_ca_purpose, boolean()}`** - Handle certificate extended key usages extension\n\n  If a CA certificate has an extended key usage extension but it does not want to\n  restrict the usages of the key it can include a special `anyExtendedKeyUsage` purpose.\n  If this is option is set to `true` all key usage purposes is automatically\n  accepted for a CA that include that purpose, the options default to false.\n\n- **`{cerl_check, Check}`**  - Handle certificate revocation lists.\n\n  Perform CRL (Certificate Revocation List) verification\n  [(public_key:pkix_crls_validate/3)](`public_key:pkix_crls_validate/3`) on all\n  the certificates during the path validation\n  [(public_key:pkix_path_validation/3) ](`public_key:pkix_path_validation/3`)of\n  the certificate chain. `Check` defaults to `false`.\n\n  The meaning of `Check` is as follows:\n\n  - **`false`**\n\n    No checks are performed.\n\n  - **`peer`**\n\n    Check is only performed on the peer certificate.\n\n  - **`best_effort`**\n\n    If certificate revocation status cannot be determined it will be accepted as valid.\n\n    The CA certificates specified for the connection will be used to construct the\n    certificate chain validating the CRLs.\n\n    The CRLs will be fetched from a local or external cache. See\n    `m:ssl_crl_cache_api`.","title":"ssl.common_option_cert/0","ref":"ssl.html#t:common_option_cert/0"},{"type":"type","doc":"Common options to client and server only valid for DTLS.\n\n- **`{use_srtp, UseSrtp}`** - Configures the `use_srtp` DTLS hello extension.\n\n  In order to negotiate the use of SRTP data protection, clients include an\n  extension of type \"use_srtp\" in the DTLS extended client hello. This extension\n  MUST only be used when the data being transported is RTP or RTCP.\n\n  The value is a map with a mandatory `protection_profiles` parameter\n  and an optional `mki` parameter.\n\n  `protection_profiles` configures the list of the client's acceptable SRTP\n  Protection Profiles. Each profile is a 2-byte binary. Example:\n  `#{protection_profiles => [<<0,2>>, <<0,5>>]}`\n\n  `mki` configures the SRTP Master Key Identifier chosen by the client.\n\n  The `srtp_mki` field contains the value of the SRTP MKI which is associated with\n  the SRTP master keys derived from this handshake. Each SRTP session MUST have\n  exactly one master key that is used to protect packets at any given time. The\n  client MUST choose the MKI value so that it is distinct from the last MKI value\n  that was used, and it SHOULD make these values unique for the duration of the\n  TLS session.\n\n  > #### Note {: .info }\n  >\n  > OTP does not handle SRTP, so an external implementations of SRTP\n  > encoder/decoder and a packet demultiplexer are needed to make use of the\n  > `use_srtp` extension. See also option [transport_option](`t:transport_option/0`).\n\n  Servers that receive an extended hello containing a \"use_srtp\" extension can\n  agree to use SRTP by including an extension of type \"use_srtp\", with the chosen\n  protection profile in the extended server hello. This extension MUST only be\n  used when the data being transported is RTP or RTCP.","title":"ssl.common_option_dtls/0","ref":"ssl.html#t:common_option_dtls/0"},{"type":"type","doc":"Legacy options considered deprecated in favor of other options,\ninsecure to use, or plainly not relevant anymore.\n\n- **`{cert, Certs}`**\n\n  Use option `certs_keys` instead.\n\n- **`{certfile, CertPem}`**\n\n  Use option `certs_keys` instead.\n\n- **`{keyfile, KeyPem}`**\n\n  Use option `certs_keys` instead.\n\n- **`{password, KeyPemPasswd}`**\n\n  Use option `certs_keys` instead.\n\n- **`{log_alert, LogAlert}`**\n\n  If `LogAlert` is `false`, TLS/DTLS Alert reports are not displayed. Deprecated in OTP\n  22; use `{log_level, Level}` instead.\n\n- **`{padding_check, PaddingCheck}`** - Inter-op trade-off option\n\n  Affects TLS-1.0 connections only. If set to `false`, it disables the block\n  cipher padding check to be able to interoperate with legacy software.\n\n  > #### Warning {: .warning }\n  >\n  > Using `{padding_check, false}` makes TLS vulnerable to the Poodle attack.\n\n- **`{beast_mitigation, BeastMitigation}`**  - Inter-op trade-off option\n\n  Affects TLS-1.0 connections only. Used to change the BEAST mitigation strategy\n  to interoperate with legacy software. Defaults to `one_n_minus_one`.\n\n  `one_n_minus_one` - Perform `1/n-1` BEAST mitigation.\n\n  `zero_n` - Perform `0/n` BEAST mitigation.\n\n  `disabled` - Disable BEAST mitigation.\n\n  > #### Warning {: .warning }\n  >\n  > Using `{beast_mitigation, disabled}` makes TLS-1.0 vulnerable to the BEAST\n  > attack.\n\n- **`{ssl_imp, Imp}`**\n\n  Deprecated since OTP 17; has no effect.","title":"ssl.common_option_legacy/0","ref":"ssl.html#t:common_option_legacy/0"},{"type":"type","doc":"Options common to client and server side prior to TLS-1.3.\n\n- **`{eccs, NamedCurves}`** - Named Elliptic Curves\n\n  Elliptic curves that can be used in pre TLS-1.3 key exchange.\n\n- **`{secure_renegotiate, SecureRenegotiate}`** - Inter-operate trade-off option\n\n  Specifies whether to reject renegotiation attempt that does not live\n  up to [RFC 5746](http://www.ietf.org/rfc/rfc5746.txt). By default,\n  `SecureRenegotiate` is `true`, meaning that secure renegotiation is\n  enforced. If `SecureRenegotiate` is `false` secure renegotiation\n  will still be used if possible, but it falls back to insecure\n  renegotiation if the peer does not support if [RFC\n  5746](http://www.ietf.org/rfc/rfc5746.txt).\n\n- **`{user_lookup_fun, {LookupFun, UserState}}`** - PSK/SRP cipher suite option\n\n  The lookup fun is to be defined as follows:\n\n  ```erlang\n  fun(psk, PSKIdentity :: binary(), UserState :: term()) ->\n\t{ok, SharedSecret :: binary()} | error;\n  fun(srp, Username :: binary(), UserState :: term()) ->\n\t{ok, {SRPParams :: srp_param_type(), Salt :: binary(),\n\t      DerivedKey :: binary()}} | error.\n  ```\n\n  For Pre-Shared Key (PSK) cipher suites, the lookup fun is called by the client\n  and server to determine the shared secret. When called by the client,\n  `PSKIdentity` is the hint presented by the server or `undefined`. When\n  called by the server, `PSKIdentity` is the identity presented by the client.\n\n  For Secure Remote Password (SRP), the fun is only used by the server to obtain\n  parameters that it uses to generate its session keys. `DerivedKey` is to be\n  derived according to [RFC 2945](http://tools.ietf.org/html/rfc2945#section/3)\n  and [RFC 5054](http://tools.ietf.org/html/rfc5054#section-2.4):\n  `crypto:sha([Salt, crypto:sha([Username, <<$:>>, Password])])`","title":"ssl.common_option_pre_tls13/0","ref":"ssl.html#t:common_option_pre_tls13/0"},{"type":"type","doc":"Common options to both client and server for TLS-1.3.\n\n- **`{supported_groups, Groups}`** - Key exchange option\n\n  TLS 1.3 introduces the \"supported_groups\" extension, which is used for negotiating\n  the Diffie-Hellman parameters in a TLS 1.3 handshake. Both client and server can\n  specify a list of parameters that they are willing to use.\n\n  If not specified it will use a default list (`[x25519, x448, secp256r1,\n  secp384r1]`) that is filtered based on the installed crypto library version.\n\n- **`{key_update_at, KeyUpdateAt}`** - Session key renewal\n\n  Configures the maximum amount of bytes that can be sent on a TLS 1.3 connection\n  before an automatic key update is performed.\n\n  There are cryptographic limits on the amount of plaintext which can be safely\n  encrypted under a given set of keys. The current default ensures that data\n  integrity will not be breached with probability greater than `1/2^57`. For more\n  information see\n  [Limits on Authenticated Encryption Use in TLS](https://eprint.iacr.org/2024/051.pdf).","title":"ssl.common_option_tls13/0","ref":"ssl.html#t:common_option_tls13/0"},{"type":"type","doc":"Key value list convening some information about the established connection.","title":"ssl.connection_info/0","ref":"ssl.html#t:connection_info/0"},{"type":"type","doc":"TLS connection keys for which information can be retrieved.","title":"ssl.connection_info_keys/0","ref":"ssl.html#t:connection_info_keys/0"},{"type":"type","doc":"TLS connection information relevant prior to TLS-1.3.","title":"ssl.connection_info_pre_tls13/0","ref":"ssl.html#t:connection_info_pre_tls13/0"},{"type":"type","doc":"Options for using built-in CRL cache support.\n\nSpecify how to perform lookup and caching of certificate revocation\nlists (CRLs). `Module` defaults to `m:ssl_crl_cache` with `DbHandle`\nbeing `internal`, and `Args` being `[]`.\n\nThere are two implementations available:\n\n- **`ssl_crl_cache`** - Implementation 1\n\n  This module maintains a cache of CRLs. CRLs can be added to the\n  cache using `ssl_crl_cache:insert/1`, and can optionally be\n  automatically fetched through HTTP if the following argument is\n  specified:\n\n  - **`{http, timeout()}`**\n\n    Enables fetching of CRLs specified as http URIs in [X.509 certificate\n    extensions](`e:public_key:public_key_records.md`). Requires the\n    [Inets](`e:inets:introduction.md`) application.\n\n- **`ssl_crl_hash_dir`** - Implementation 2\n\n  This module makes use of a directory where CRLs are\n  stored in files named by the hash of the issuer name.\n\n  The file names consist of eight hexadecimal digits followed by `.rN`, where\n  `N` is an integer, for example `1a2b3c4d.r0`. For the first version of the CRL, `N`\n  starts at zero, and for each new version, `N` is incremented by one. The\n  OpenSSL utility `c_rehash` creates symlinks according to this pattern.\n\n  For a given hash value, this module finds all consecutive `.r*`\n  files starting from zero, and those files taken together make up the\n  revocation list. CRL files with `nextUpdate` fields in the past or\n  issued by a different CA that happens to have the same name hash\n  are excluded.\n\n  The following argument is required:\n\n  - **`{dir, string()}`**\n\n  Specifies the directory in which the CRLs can be found.","title":"ssl.crl_cache_opts/0","ref":"ssl.html#t:crl_cache_opts/0"},{"type":"type","doc":"A DTLS protocol version that are no longer supported by default for security reasons.","title":"ssl.dtls_legacy_version/0","ref":"ssl.html#t:dtls_legacy_version/0"},{"type":"type","doc":"DTLS protocol version.","title":"ssl.dtls_version/0","ref":"ssl.html#t:dtls_version/0"},{"type":"type","doc":"Erlang cipher suite representation\n\n> #### Warning {: .warning }\nEnabling cipher suites using RSA as a key exchange algorithm is\nstrongly discouraged (only available prior to TLS-1.3). For some\nconfigurations software preventions may exist, and can make them\nusable if they work, but relying on them to work is risky. There\nexists more reliable cipher suites that can be used instead.","title":"ssl.erl_cipher_suite/0","ref":"ssl.html#t:erl_cipher_suite/0"},{"type":"type","doc":"If a TLS connection fails a TLS protocol ALERT will be sent/received.\n\nAn atom reflecting the raised alert, according to the TLS protocol, and a description string\nwith some further details will be returned.","title":"ssl.error_alert/0","ref":"ssl.html#t:error_alert/0"},{"type":"type","doc":"TLS-1.3 key exchange configuration.","title":"ssl.group/0","ref":"ssl.html#t:group/0"},{"type":"type","doc":"Hash algorithms used together with signing and encryption functions.","title":"ssl.hash/0","ref":"ssl.html#t:hash/0"},{"type":"type","doc":"A name or address to a host.","title":"ssl.host/0","ref":"ssl.html#t:host/0"},{"type":"type","doc":"Cipher Suite Key Exchange Algorithm will be `any`\nin TLS-1.3 as key exchange is no longer part of cipher suite\nconfiguration in TLS-1.3.","title":"ssl.kex_algo/0","ref":"ssl.html#t:kex_algo/0"},{"type":"type","doc":"The user's private key.\n\nThe key can be provided either directly as a DER-encoded entity,\nindirectly using a crypto engine/provider (with key reference\ninformation), or as an Erlang fun (with possible custom options).\nThe latter two options can be used for customized signing with\nhardware security modules (HSM) or trusted platform modules (TPM).\n\n- A DER encoded key will need to specify the ASN-1 type used to create the\n  encoding.\n- An engine/provider needs to specify specific information to support this\n  concept and can optionally be password protected; see also\n  [crypto:engine_load/3 ](`crypto:engine_load/3`) and\n  [Crypto User's Guide](`e:crypto:engine_load.md`).\n- A fun option should include a fun that mimics `public_key:sign/4` and possibly\n  [public_key:private_encrypt/4](`public_key:encrypt_private/3`) if legacy\n  versions TLS-1.0 and TLS-1.1 must be supported.","title":"ssl.key/0","ref":"ssl.html#t:key/0"},{"type":"type","doc":"Cipher algorithms that are no longer supported by default for security reasons.","title":"ssl.legacy_cipher/0","ref":"ssl.html#t:legacy_cipher/0"},{"type":"type","doc":"Hash algorithms that are no longer supported by default for security reasons.","title":"ssl.legacy_hash/0","ref":"ssl.html#t:legacy_hash/0"},{"type":"type","doc":"Key exchange configuration prior to TLS-1.3.\n\nThese curves have been deprecated by RFC 8422.","title":"ssl.legacy_named_curve/0","ref":"ssl.html#t:legacy_named_curve/0"},{"type":"type","doc":"Signature algorithms that are no longer supported by default for security reasons.","title":"ssl.legacy_sign_algo/0","ref":"ssl.html#t:legacy_sign_algo/0"},{"type":"type","doc":"This is only used for certificate signatures if TLS-1.2 is negotiated,\nmeaning that the peer only supports TLS-1.2, but we also support\nTLS-1.3.","title":"ssl.legacy_sign_scheme/0","ref":"ssl.html#t:legacy_sign_scheme/0"},{"type":"type","doc":"Key exchange configuration prior to TLS-1.3.","title":"ssl.named_curve/0","ref":"ssl.html#t:named_curve/0"},{"type":"type","doc":"For backwards compatibility only; do not use.","title":"ssl.old_cipher_suite/0","ref":"ssl.html#t:old_cipher_suite/0"},{"type":"type","doc":"","title":"ssl.prf_random/0","ref":"ssl.html#t:prf_random/0"},{"type":"type","doc":"Client hello extensions.","title":"ssl.protocol_extensions/0","ref":"ssl.html#t:protocol_extensions/0"},{"type":"type","doc":"TLS or DTLS protocol version.","title":"ssl.protocol_version/0","ref":"ssl.html#t:protocol_version/0"},{"type":"type","doc":"Error reason for debug purposes.\n\nNot to be matched.","title":"ssl.reason/0","ref":"ssl.html#t:reason/0"},{"type":"type","doc":"Supported in TLS-1.3 and TLS-1.2.","title":"ssl.rsassa_pss_scheme/0","ref":"ssl.html#t:rsassa_pss_scheme/0"},{"type":"type","doc":"TLS connection information that can be used for NSS key logging.","title":"ssl.security_info/0","ref":"ssl.html#t:security_info/0"},{"type":"type","doc":"Options specific to the server side, or with different semantics for the client and server.\n\n- **`{alpn_preferred_protocols, AppProtocols}`** - Application Layer Protocol Negotiation\n\n  Indicates that the server will try to perform Application-Layer\n  Protocol Negotiation (ALPN).\n\n  The list of protocols is in order of preference. The protocol negotiated will be\n  the first in the list that matches one of the protocols advertised by the\n  client. If no protocol matches, the server will fail the connection with a\n  `no_application_protocol` alert.\n\n  The negotiated protocol can be retrieved using the\n  [`negotiated_protocol/1`](`negotiated_protocol/1`) function.\n\n- **`{sni_fun, SNIFun}`**\n\n  If the server receives a SNI (Server Name Indication) from the\n  client, the given fun `SNIFun` will be called to retrieve\n  [`server_option()`](`t:server_option/0`) for the indicated\n  hosts. These options will override previously specified options for\n  that host.\n\n  > #### Note {: .info }\n  The options `sni_fun` and `sni_hosts` are mutually exclusive.\n\n- **`{sni_hosts, SNIHosts}`**\n\n  If the server receives a SNI (Server Name Indication) from the client matching a\n  host listed in the `sni_hosts` option, the specific options for that host will\n  override previously specified options.\n\n  > #### Note {: .info }\n  The options `sni_fun` and `sni_hosts` are mutually exclusive.","title":"ssl.server_option/0","ref":"ssl.html#t:server_option/0"},{"type":"type","doc":"Certificate related options for a server.\n\n- **`{cacerts, CACerts}`** - Trusted certificates.\n\n  The DER-encoded trusted certificates. If this option is supplied, it overrides\n  the `cacertfile` option.\n\n- **`{verify, Verify}`** - Verify certificates.\n\n  Client certificates are an optional part of the TLS protocol. A server performs\n  X.509 certificate path validation only in `verify_peer` mode. By default the server\n  is in `verify_none` mode and, hence, will not send an certificate request to the\n  client. When using `verify_peer` you may also want to specify the options\n  `fail_if_no_peer_cert` and `certificate_authorities`.\n\n- **`{fail_if_no_peer_cert, FailNoPeerCert}`** - Legacy trade-off option\n\n  Used together with `{verify, verify_peer}` by an TLS/DTLS server. If set to\n  `true`, the server fails if the client does not have a certificate to send, that\n  is, sends an empty certificate. If set to `false`, it fails only if the client\n  sends an invalid certificate (an empty certificate is considered valid).\n  Defaults to `false`.\n\n- **`{certificate_authorities, ServerCertAuth}`** - Inter-operate hint option\n\n  Determines whether a TLS-1.3 server should include the authorities extension in its\n  certificate request message that is sent when the option `verify` is set to\n  `verify_peer`. Defaults to `true`.\n\n  If set to `false` for older TLS versions its corresponding certificate authorities\n  definition in its certificate request will be set to the empty list instead of\n  including the appropriate certificate authorities. This has the same affect\n  as excluding the TLS-1.3 extension.\n\n  A reason to exclude the extension would be if the server wants to communicate\n  with clients incapable of sending complete certificate chains that adhere to the\n  extension, but the server still has the capability to recreate a chain that it\n  can verify.","title":"ssl.server_option_cert/0","ref":"ssl.html#t:server_option_cert/0"},{"type":"type","doc":"Legacy server options.\n\n- **`{next_protocols_advertised, NextAppProtocols}`**\n\n  ALPN (Application-Layer Protocol Negotiation)\n  deprecates NPN (Next Protocol Negotiation) described here.\n\n  List of protocols to send to the client if the client indicates that it supports\n  the Next Protocol extension. The client can select a protocol that is not on\n  this list. The list of protocols must not contain an empty binary. If the server\n  negotiates a Next Protocol, it can be accessed using the\n  `negotiated_protocol/1` method.","title":"ssl.server_option_legacy/0","ref":"ssl.html#t:server_option_legacy/0"},{"type":"type","doc":"Options only relevant to TLS versions prior to TLS-1.3.\n\n- **`{client_renegotiation, ClientRengotiation}`** - DoS attack avoidance option\n\n  In protocols that support client-initiated renegotiation, the resource cost\n  of such an operation is higher for the server than the client. This can act as a\n  vector for denial-of-service (DoS) attacks. The SSL application already takes measures\n  to counter-act such attempts, but client-initiated renegotiation can be completely\n  disabled by setting this option to `false`. The default value is `true`. Note\n  that disabling renegotiation can result in long-lived connections becoming\n  unusable due to limits on the number of messages the underlying cipher suite can\n  encipher.\n\n- **`{reuse_sessions, ReuseSessions}`** - Enable session reuse\n\n  The boolean value `true` specifies that the server will agree to reuse sessions.\n  Setting it to `false` will result in an empty session table, meaning that no sessions\n  will be reused.\n\n- **`{reuse_session, ReuseSession}`** - Local server reuse policy\n\n  Enables the TLS/DTLS server to have a local policy for deciding whether a session\n  is to be reused. Meaningful only if `reuse_sessions` is set to `true`.\n\n  `ReuseSession` should be a fun:\n\n   `fun(SuggestedSessionId, PeerCert, Compression, CipherSuite)`\n\n  `SuggestedSessionId` is a [`binary()`](`t:binary/0`),\n  `PeerCert` is a DER-encoded certificate,\n  `Compression` is an enumeration integer, and `CipherSuite` is of type\n  [`erl_cipher_suite()`](`t:erl_cipher_suite/0`).\n\n- **`{psk_identity, PSKHint}`** - Inter-operate hint option\n\n  Specifies the server identity hint that the server presents to the client.\n\n- **`{honor_cipher_order, HonorServerCipherOrder}`** - Trade-off option alters protocol defined behavior\n\n  If `true`, use the server's preference for ECC curve selection. If `false` (the\n  default), use the client's preference.\n\n- **`{honor_ecc_order, HonorServerECCOrder}`** - Trade-off option alters protocol defined behavior\n\n  If `true`, use the server's preference for ECC curve selection. If `false` (the\n  default), use the client's preference.\n\n- **`{dh, DHder}`** - Affects DH key exchange cipher suites\n\n  The DER-encoded Diffie-Hellman parameters. If specified, it overrides option\n  `dhfile`.\n\n- **`{dh_file, DHfile}`** - Affects DH key exchange cipher suites\n\n  Path to a file containing PEM-encoded Diffie Hellman parameters to be used by\n  the server if a cipher suite using Diffie Hellman key exchange is negotiated. If\n  not specified, default parameters are used.","title":"ssl.server_option_pre_tls13/0","ref":"ssl.html#t:server_option_pre_tls13/0"},{"type":"type","doc":"Options only relevant for TLS-1.3.\n\n- **`{session_tickets, SessionTickets}`**\n\n  Configures the session ticket functionality. Allowed values for `SessionTickets` are:\n\n  * `disabled`\n  * `stateful`\n  * `stateless`\n  * `stateful_with_cert`\n  * `stateless_with_cert`\n\n  If `SessionTickets` is not set to `disabled`, session resumption with pre-shared\n  keys is enabled and the server will send stateful or stateless session tickets to the\n  client after successful connections.\n\n  > #### Note {: .info }\n  In pre-shared key session ticket resumption, there is no certificate\n  exchange involved. Therefore, `ssl:peercert/1` will not return the\n  peer certificate, as it is only communicated during the initial\n  handshake. To associate the client certificate from the original\n  handshake with the tickets it issues, the server options\n  `stateful_with_cert` or `stateless_with_cert` can be used.\n\n  A stateful session ticket is a database reference to internal state information.\n  A stateless session ticket is a self-encrypted binary that contains both\n  cryptographic keying material and state data.\n\n  > #### Warning {: .warning }\n  When `SessionTickets` is set to `stateful_with_cert`, the client\n  certificate is stored along with the internal state information,\n  leading to increased memory consumption. Conversely, when it is set\n  to `stateless_with_cert`, the client certificate is encoded in the\n  self-encrypted binary sent to the client, resulting in an increase\n  in payload size.\n\n  See also [SSL User's Guide, Session Tickets and Session Resumption in TLS 1.3](using_ssl.md#session-tickets-and-session-resumption-in-tls-1-3).\n\n- **`{stateless_tickets_seed, TicketSeed}`** - Option for statless tickets\n\n  Configures the seed used for the encryption of stateless session tickets.\n  Allowed values are any randomly generated `t:binary/0`. If this option is not\n  configured, an encryption seed will be randomly generated.\n\n  > #### Warning {: .warning }\n  >\n  > Reusing the ticket encryption seed between multiple server instances enables\n  > stateless session tickets to work across multiple server instances, but it\n  > breaks anti-replay protection across instances.\n  >\n  > Inaccurate time synchronization between server instances can also affect\n  > session ticket freshness checks, potentially causing false negatives as well\n  > as false positives.\n\n- **`{anti_replay, AntiReplay}`** - Option for statless tickets\n\n  Configures the server's built-in anti replay feature based on Bloom filters.\n\n  Allowed values for `AntiReplay` are the pre-defined `'10k'`,\n  `'100k'`, or a custom 3-tuple that defines the properties of the\n  bloom filters:\n  `{WindowSize, HashFunctions, Bits}`. `WindowSize` is the number of seconds after\n  the current Bloom filter is rotated and also the window size used for freshness\n  checks of ClientHello. `HashFunctions` is the number hash functions and `Bits`\n  is the number of bits in the bit vector. `'10k'` and `'100k'` are simple\n  defaults with the following properties:\n\n  - `'10k'`: Bloom filters can hold 10000 elements with 3% probability of false\n    positives. `WindowSize`: 10, `HashFunctions`: 5, `Bits:` 72985 (8.91 KiB).\n  - `'100k'`: Bloom filters can hold 100000 elements with 3% probability of false\n  positives. `WindowSize`: 10, `HashFunctions`: 5, `Bits`: 729845 (89.09 KiB).\n\n  See also [SSL User's Guide, Anti-Replay Protection in TLS\n  1.3](using_ssl.md#anti-replay-protection-in-tls-1-3).\n\n- **`{cookie, Cookie}`** - Option for `HelloRetryRequest` behavior\n\n  If `Cookie` is `true`, which is the default, the server sends a\n  cookie extension in its `HelloRetryRequest` messages.\n\n  The cookie extension has two main purposes. It allows the server to force the\n  client to demonstrate reachability at their apparent network address (thus\n  providing a measure of DoS protection). This is primarily useful for\n  non-connection-oriented transports. It also allows offloading the server's\n  state to the client. The cookie extension is enabled by default as it is a\n  mandatory extension in RFC8446.\n\n- **`{early_data, EarlyData}`** - Option for accepting or rejecting Early Data\n\n  Configures if the server accepts (`enabled`) or rejects (`disabled`) early data\n  sent by a client. The default value is `disabled`.","title":"ssl.server_option_tls13/0","ref":"ssl.html#t:server_option_tls13/0"},{"type":"type","doc":"Identifies a TLS session prior to TLS-1.3.","title":"ssl.session_id/0","ref":"ssl.html#t:session_id/0"},{"type":"type","doc":"SHA2 hash algorithms.","title":"ssl.sha2/0","ref":"ssl.html#t:sha2/0"},{"type":"type","doc":"Signature algorithms.","title":"ssl.sign_algo/0","ref":"ssl.html#t:sign_algo/0"},{"type":"type","doc":"Signature schemes, defined by TLS-1.3, and replacing signature algorithms from TLS-1.2.\n\nExplicitly list acceptable signature schemes in the preferred\norder.\n\nOverrides the algorithms supplied in\n[`signature_algs`](`t:signature_algs/0`) option for certificates.\nIn addition to the `signature_algorithms` extension from TLS 1.2,\n[TLS 1.3 (RFC 5246 Section 4.2.3)](http://www.ietf.org/rfc/rfc8446.txt#section-4.2.3)\nadds the `signature_algorithms_cert` extension which enables having special\nrequirements on the signatures used in the certificates that differs from the\nrequirements on digital signatures as a whole. If this is not required this\nextension is not needed.\n\nThe client will send a `signature_algorithms_cert` extension (in the\nclient hello message), if TLS version 1.2 (back-ported to TLS 1.2 in\n24.1) or later is used, and the signature_algs_cert option is\nexplicitly specified.  By default, only the\n[signature_algs](`t:signature_algs/0`) extension is sent with the\nexception of when signature_algs option is not explicitly specified,\nin which case it will append the rsa_pkcs1_sha1 algorithm to the\ndefault value of signature_algs and use it as value for\nsignature_algs_cert to allow certificates to have this signature but\nstill disallow sha1 use in the TLS protocol, since 27.0.1 and 26.2.5.2.\n\n> #### Note {: .info }\n>\n> Note that supported signature schemes for TLS-1.2 are\n[`legacy_sign_scheme()`](`t:legacy_sign_scheme/0`)\n> and [`rsassa_pss_scheme()`](`t:rsassa_pss_scheme/0`).","title":"ssl.sign_scheme/0","ref":"ssl.html#t:sign_scheme/0"},{"type":"type","doc":"Explicitly list acceptable signature algorithms for certificates and handshake\nmessages in the preferred order.\n\nThe client will send its list as the client hello\n`signature_algorithm` extension introduced in TLS-1.2; see [Section\n7.4.1.4.1 in RFC 5246](http://www.ietf.org/rfc/rfc5246.txt). Before\nTLS-1.2, these algorithms where implicitly chosen and partly derived\nfrom the cipher suite.\n\nIn TLS-1.2 a somewhat more explicit negotiation is made possible using a list of\n`{HashAlgo, SignAlgo}` tuples.\n\nIn TLS-1.3, these algorithm pairs are replaced by [signature\nschemes](`t:sign_scheme/0`) that are completely decoupled from the\ncipher suite.\n\nSignature algorithms used for certificates can be overridden by the\n[signature schemes](`t:sign_scheme/0`) supplied by the\n`signature_algs_cert` option.\n\nThe TLS-1.2 default is `Default_TLS_12_Alg_Pairs` interleaved with\n`rsa_pss_schemes` since ssl-11.0 (Erlang/OTP 25). `pss_pss` is\npreferred over `pss_rsae`, which in turn is preferred over `rsa`.\n\nThe list for `Default_TLS_12_Alg_Pairs` is defined as follows:\n\n```erlang\n[\n{sha512, ecdsa},\n{sha512, rsa},\n{sha384, ecdsa},\n{sha384, rsa},\n{sha256, ecdsa},\n{sha256, rsa}\n]\n```\n\n> #### Change {: .info }\n>\n> - Support for `{md5, rsa}` was removed from the TLS-1.2 default in\n>   ssl-8.0 (Erlang/OTP 22).\n> - Support for `{sha, _}` (SHA1) and `{sha224, _}` was removed\n>   from the TLS-1.2 default in ssl-11.0 (Erlang/OTP 26).\n\nThe list for `rsa_pss_schemes` is defined as follows:\n\n\n```erlang\n[rsa_pss_pss_sha512,\nrsa_pss_pss_sha384,\nrsa_pss_pss_sha256,\nrsa_pss_rsae_sha512,\nrsa_pss_rsae_sha384,\nrsa_pss_rsae_sha256]\n```\n\nThe list of `TLS_13_Legacy_Schemes` is defined as follows:\n\n```erlang\n[\n%% Legacy algorithms only applicable to certificate signatures\nrsa_pkcs1_sha512, %% Corresponds to {sha512, rsa}\nrsa_pkcs1_sha384, %% Corresponds to {sha384, rsa}\nrsa_pkcs1_sha256, %% Corresponds to {sha256, rsa}\n]\n```\n\nThe list of `Default_TLS_13_Schemes` is defined as follows:\n\n```text\n[\n%% EDDSA\neddsa_ed25519,\neddsa_ed448\n\n%% ECDSA\necdsa_secp521r1_sha512,\necdsa_secp384r1_sha384,\necdsa_secp256r1_sha256] ++\n\n%% RSASSA-PSS\nrsa_pss_schemes()\n```\n\n> #### Change {: .info }\n>\n> EDDSA was made highest priority in ssl-10.8 (Erlang/OTP 25).\n\nThe TLS-1.3 default is `Default_TLS_13_Schemes`.\n\nIf both TLS-1.3 and TLS-1.2 are supported the default is:\n\n```erlang\nDefault_TLS_13_Schemes ++ TLS_13_Legacy_Schemes ++\nDefault_TLS_12_Alg_Pairs %% not represented in TLS_13_Legacy_Schemes\n```\n\nto ensure that appropriate algorithms can be chosen for the negotiated\nversion.\n\n> #### Note {: .info }\nTLS-1.2 algorithms will not be negotiated for TLS-1.3, but the TLS-1.3\nRSASSA-PSS ([`rsassa_pss_scheme()`](`t:rsassa_pss_scheme/0`))\nsignature schemes can be negotiated also for TLS-1.2 from Erlang/OTP\n24.1 (fully working from Erlang/OTP 24.1.3). However, if both TLS 1.3\nand TLS 1.2 are supported using defaults, and TLS 1.3 is negotiated,\nthe corresponding TLS 1.2 algorithms for TLS 1.3 legacy signature\nschemes will be treated as legacy schemes and applied only to\ncertificate signatures.","title":"ssl.signature_algs/0","ref":"ssl.html#t:signature_algs/0"},{"type":"type","doc":"A socket that can be used to perform a so-called \"START-TLS\", which\nmeans using an already connected socket previously used for plain TCP\ntraffic and upgrading it to use TLS.\n\nBoth sides needs to agree on the upgrade.","title":"ssl.socket/0","ref":"ssl.html#t:socket/0"},{"type":"type","doc":"Options for the transport socket.\n\nThe default socket options are\n`[{mode, list}, {packet, 0}, {header, 0}, {active, true}]`.\n\nFor valid options, see `m:inet`, `m:gen_tcp`, and `m:gen_udp`\nin Kernel. Note that stream-oriented options such as `packet` are\nonly relevant for TLS and not DTLS.","title":"ssl.socket_option/0","ref":"ssl.html#t:socket_option/0"},{"type":"type","doc":"SRP cipher suite configuration prior to TLS-1.3.","title":"ssl.srp_param_type/0","ref":"ssl.html#t:srp_param_type/0"},{"type":"type","doc":"An opaque reference to the TLS/DTLS connection.\n\nNote that despite being opaque, matching `sslsocket()` instances is allowed.","title":"ssl.sslsocket/0","ref":"ssl.html#t:sslsocket/0"},{"type":"type","doc":"TLS Alert Protocol reasons.","title":"ssl.tls_alert/0","ref":"ssl.html#t:tls_alert/0"},{"type":"type","doc":"An option that can be supplied to a TLS client.","title":"ssl.tls_client_option/0","ref":"ssl.html#t:tls_client_option/0"},{"type":"type","doc":"A TLS protocol version that are no longer supported by default for security reasons.","title":"ssl.tls_legacy_version/0","ref":"ssl.html#t:tls_legacy_version/0"},{"type":"type","doc":"An option related to the TLS/DTLS protocol.","title":"ssl.tls_option/0","ref":"ssl.html#t:tls_option/0"},{"type":"type","doc":"An option that can be supplied to a TLS server.","title":"ssl.tls_server_option/0","ref":"ssl.html#t:tls_server_option/0"},{"type":"type","doc":"TLS protocol version.","title":"ssl.tls_version/0","ref":"ssl.html#t:tls_version/0"},{"type":"type","doc":"Transport option defines a callback module and message tags to handle the underlying transport socket.\n\nCan be used to customize the transport layer. The tag\nvalues should be the values used by the underlying transport in its active mode\nmessages.\n\nDefaults to `{gen_tcp, tcp, tcp_closed, tcp_error, tcp_passive}` for TLS.\n\n> #### Note {: .info }\nFor backward compatibility a tuple of size four will be converted to a\ntuple of size five, where `PassiveTag` is the `DataTag` element with\n`_passive` appended.\n\nFor TLS the callback module must implement a reliable transport\nprotocol, behave as `m:gen_tcp`, and have functions corresponding to\n`inet:setopts/2`, `inet:getopts/2`, `inet:peername/1`, `inet:sockname/1`, and\n`inet:port/1`. The callback `m:gen_tcp` is treated specially and calls `m:inet`\ndirectly. For DTLS this feature is considered experimental.","title":"ssl.transport_option/0","ref":"ssl.html#t:transport_option/0"},{"type":"module","doc":"CRL cache\n\nImplements an internal CRL (Certificate Revocation List) cache. In addition to\nimplementing the `m:ssl_crl_cache_api` behaviour the following functions are\navailable.","title":"ssl_crl_cache","ref":"ssl_crl_cache.html"},{"type":"function","doc":"Delete CRLs from the ssl applications local cache.","title":"ssl_crl_cache.delete/1","ref":"ssl_crl_cache.html#delete/1"},{"type":"function","doc":"","title":"ssl_crl_cache.insert/1","ref":"ssl_crl_cache.html#insert/1"},{"type":"function","doc":"Insert CRLs into the ssl applications local cache, with or without a\ndistribution point reference URI","title":"ssl_crl_cache.insert/2","ref":"ssl_crl_cache.html#insert/2"},{"type":"type","doc":"A source to input CRLs","title":"ssl_crl_cache.crl_src/0","ref":"ssl_crl_cache.html#t:crl_src/0"},{"type":"behaviour","doc":"API for a TLS CRL (Certificate Revocation List) cache.\n\nWhen TLS performs certificate path validation according to\n[RFC 5280 ](http://www.ietf.org/rfc/rfc5280.txt)it should also perform CRL\nvalidation checks. To enable the CRL checks the application needs access to\nCRLs. A database of CRLs can be set up in many different ways. This module\nprovides the behavior of the API needed to integrate an arbitrary CRL cache with\nthe erlang ssl application. It is also used by the application itself to provide\na simple default implementation of a CRL cache.","title":"ssl_crl_cache_api","ref":"ssl_crl_cache_api.html"},{"type":"callback","doc":"`fun fresh_crl/2` will be used as input option `update_crl` to\n`public_key:pkix_crls_validate/3`.\n\nIt is possible to return logger info, since OTP 22.2, that will be used by the TLS connection to\nproduce log events.","title":"ssl_crl_cache_api.fresh_crl/2","ref":"ssl_crl_cache_api.html#c:fresh_crl/2"},{"type":"callback","doc":"Backwards compatibility, replaced by lookup/3","title":"ssl_crl_cache_api.lookup/2","ref":"ssl_crl_cache_api.html#c:lookup/2"},{"type":"callback","doc":"Lookup the CRLs belonging to the distribution point `Distributionpoint`. This\nfunction may choose to only look in the cache or to follow distribution point\nlinks depending on how the cache is administrated.\n\nThe `Issuer` argument contains the issuer name of the certificate to\nbe checked.  Normally the returned CRL should be issued by this\nissuer, except if the `cRLIssuer` field of `DistributionPoint` has a\nvalue, in which case that value should be used instead.\n\nIn an earlier version of this API, the `lookup` function received two\narguments, omitting `Issuer`. For compatibility, this is still\nsupported: if there is no [`lookup/3`](`c:lookup/3`) function in the\ncallback module,[`lookup/2`](`c:lookup/2`) is called instead.\n\nIt is possible to return logger info, since OTP 22.2, that will be used by the TLS connection to\nproduce log events.","title":"ssl_crl_cache_api.lookup/3","ref":"ssl_crl_cache_api.html#c:lookup/3"},{"type":"callback","doc":"Select the CRLs in the cache that are issued by `Issuer` unless the value is a\nlist of so called general names, see\n[X509 certificates records](`e:public_key:public_key_records.md`), originating\nform `#'DistributionPoint'.cRLissuer` and representing different mechanism to\nobtain the CRLs. The cache callback needs to use the appropriate entry to\nretrieve the CRLs or return an empty list if it does not exist.\n\nIt is possible to return logger info, since OTP 22.2, that will be used by the TLS connection to\nproduce log events.","title":"ssl_crl_cache_api.select/2","ref":"ssl_crl_cache_api.html#c:select/2"},{"type":"type","doc":"Reference to the CRL cache.","title":"ssl_crl_cache_api.crl_cache_ref/0","ref":"ssl_crl_cache_api.html#t:crl_cache_ref/0"},{"type":"type","doc":"For description see\n[X509 certificates records](`e:public_key:public_key_records.md`)","title":"ssl_crl_cache_api.dist_point/0","ref":"ssl_crl_cache_api.html#t:dist_point/0"},{"type":"type","doc":"Information for ssl applications use of [Logger(3)](`m:logger`)","title":"ssl_crl_cache_api.logger_info/0","ref":"ssl_crl_cache_api.html#t:logger_info/0"},{"type":"behaviour","doc":"TLS session cache API\n\nDefines the API for the TLS session cache (pre TLS-1.3) so that the data storage\nscheme can be replaced by defining a new callback module implementing this API.","title":"ssl_session_cache_api","ref":"ssl_session_cache_api.html"},{"type":"callback","doc":"Deletes a cache entry.\n\nIs only called from the cache handling process.","title":"ssl_session_cache_api.delete/2","ref":"ssl_session_cache_api.html#c:delete/2"},{"type":"callback","doc":"Calls `Fun(Elem, AccIn)` on successive elements of the cache, starting with\n `AccIn == Acc0`.\n\n`Fun/2` must return a new accumulator, which is passed to the\nnext call. The function returns the final value of the accumulator. `Acc0` is\nreturned if the cache is empty.\n\n> #### Note {: .info }\n>\n> Since OTP-23.3 this functions is only used on the client side and does not\n> need to implemented for a server cache.","title":"ssl_session_cache_api.foldl/3","ref":"ssl_session_cache_api.html#c:foldl/3"},{"type":"callback","doc":"Performs possible initializations of the cache and returns a reference to it\nthat is used as parameter to the other API functions. \n\nIs called by the cache handling processes `init` function, hence\nputting the same requirements on it as a normal process `init`\nfunction. This function is called twice when starting the SSL\napplication, once with the role client and once with the role server,\nas the SSL application must be prepared to take on both roles.\n\nIncludes property `{role, client | server}` in init argument list. \nCurrently this is the only predefined property, there can also be\nuser-defined properties. See also application environment variable\n[session_cb_init_args](ssl_app.md).","title":"ssl_session_cache_api.init/1","ref":"ssl_session_cache_api.html#c:init/1"},{"type":"callback","doc":"Looks up a cache entry. Is to be callable from any process.","title":"ssl_session_cache_api.lookup/2","ref":"ssl_session_cache_api.html#c:lookup/2"},{"type":"callback","doc":"Selects sessions that can be reused, that is sessions that include `PartialKey`\nin its key. Is to be callable from any process.\n\n> #### Note {: .info }\n>\n> Since OTP-23.3 This functions is only used on the client side and does not\n> need to implemented for a server cache.","title":"ssl_session_cache_api.select_session/2","ref":"ssl_session_cache_api.html#c:select_session/2"},{"type":"callback","doc":"Returns the number of sessions in the cache.\n\nIf size exceeds the maximum number of sessions, the current cache\nentries will be invalidated regardless of their remaining lifetime. Is\nto be callable from any process.","title":"ssl_session_cache_api.size/1","ref":"ssl_session_cache_api.html#c:size/1"},{"type":"callback","doc":"Takes care of possible cleanup that is needed when the cache handling process\nterminates.","title":"ssl_session_cache_api.terminate/1","ref":"ssl_session_cache_api.html#c:terminate/1"},{"type":"callback","doc":"Caches a new session or updates an already cached one.\n\nIs only called from the cache handling process.","title":"ssl_session_cache_api.update/3","ref":"ssl_session_cache_api.html#c:update/3"},{"type":"opaque","doc":"The opaque part of the key. Does not need to be handled by the callback.","title":"ssl_session_cache_api.partial_key/0","ref":"ssl_session_cache_api.html#t:partial_key/0"},{"type":"opaque","doc":"The session data that is stored for each session.","title":"ssl_session_cache_api.session/0","ref":"ssl_session_cache_api.html#t:session/0"},{"type":"type","doc":"A key to an entry in the session cache.","title":"ssl_session_cache_api.session_cache_key/0","ref":"ssl_session_cache_api.html#t:session_cache_key/0"},{"type":"type","doc":"A term that can be used to reference the cache.","title":"ssl_session_cache_api.session_cache_ref/0","ref":"ssl_session_cache_api.html#t:session_cache_ref/0"},{"type":"extras","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# SSL Application\n\nThe ssl application provides secure communication over sockets.","title":"SSL Application","ref":"ssl_app.html"},{"type":"extras","doc":"The ssl application is an implementation of the TLS (previously known as SSL) and DTLS protocols in\nErlang.\n\nFor current statement of standards compliance see the\n[User's Guide](standards_compliance.md).","title":"Description - SSL Application","ref":"ssl_app.html#description"},{"type":"extras","doc":"The SSL application uses the `Public_Key`, `Asn1` and `Crypto` application to\nhandle public keys and encryption, hence these applications must be loaded for\nthe SSL application to work. In an embedded environment this means they must be\nstarted with `application:start/1,2` before the SSL application is started.","title":"Dependencies - SSL Application","ref":"ssl_app.html#dependencies"},{"type":"extras","doc":"The application environment configuration parameters in this section are defined\nfor the SSL application. For more information about configuration parameters,\nsee the `m:application` manual page in Kernel.\n\n> #### Note {: .info }\nAll parameters including the wording 'session_ticket' are TLS-1.3 only configuration\nand other session parameters are prior to TLS-1.3 only configuration. DTLS versions\nare based on TLS versions see [standard compliance](standards_compliance.md) for mapping.\n\nThe environment parameters can be set on the command line, for example:\n\n`erl -ssl protocol_version \"['tlsv1.2', 'tlsv1.1']\"`\n\n- **`protocol_version = ` `t:ssl:tls_version/0` | [`t:ssl:tls_version/0`]\n  ` `** - Protocol supported by started clients and servers. If this\n  option is not set, it defaults to all TLS protocols currently supported, more\n  might be configurable, by the SSL application. This option can be overridden\n  by the version option to `ssl:connect/2,3` and `ssl:listen/2`.\n\n- **`dtls_protocol_version = ` `t:ssl:dtls_version/0` | [`t:ssl:dtls_version/0`]\n  ` `** - Protocol supported by started clients and servers. If this\n  option is not set, it defaults to all DTLS protocols currently supported, more\n  might be configurable, by the SSL application. This option can be overridden\n  by the version option to `ssl:connect/2,3` and `ssl:listen/2`.\n\n- **`session_lifetime = integer()  `** - Maximum lifetime of the\n  session data in seconds. Defaults to 24 hours which is the maximum recommended\n  lifetime by [RFC 5246](http://www.ietf.org/rfc/5246rfc.txt). However sessions\n  may be invalidated earlier due to the maximum limitation of the session cache\n  table.\n\n- **`session_cb = atom()  `** - Deprecated Since OTP-23.3 replaced by\n  `client_session_cb` and `server_session_cb`\n\n- **`client_session_cb = atom()  `** - Since OTP-23.3 Name client of\n  the session cache callback module that implements the `ssl_session_cache_api`\n  behavior. Defaults to `ssl_client_session_cache_db`.\n\n- **`server_session_cb = atom()  `** - Since OTP-23.3 Name of the\n  server session cache callback module that implements the\n  `ssl_session_cache_api` behavior. Defaults to `ssl_server_session_cache_db`.\n\n- **`session_cb_init_args = proplist:proplist()  `** - Deprecated Since\n  OTP-23.3 replaced by `client_session_cb_init_args` and\n  `server_session_cb_init_args`\n\n- **`client_session_cb_init_args = proplist:proplist()  `** - List of\n  extra user-defined arguments to the `init` function in the session cache\n  callback module. Defaults to `[]`.\n\n- **`server_session_cb_init_args = proplist:proplist()  `** - List of\n  extra user-defined arguments to the `init` function in the session cache\n  callback module. Defaults to `[]`.\n\n- **`session_cache_client_max = integer()  `  \n  **  \n  Limits the growth of the clients session cache, that is how many sessions\n  towards servers that are cached to be used by new client connections. If the\n  maximum number of sessions is reached, the current cache entries will be\n  invalidated regardless of their remaining lifetime. Defaults to 1000.\n  Recommended ssl-8.2.1 or later for this option to work as intended.\n\n- **`session_cache_server_max = integer()  `** - Limits the growth of\n  the servers session cache, that is how many client sessions are cached by the\n  server. If the maximum number of sessions is reached, the current cache\n  entries will be invalidated regardless of their remaining lifetime. Defaults\n  to 1000. Recommended ssl-8.2.1 or later for this option to work as intended.\n\n- **`ssl_pem_cache_clean = integer()  `** - Number of milliseconds\n  between PEM cache validations. Defaults to 2 minutes.\n\n  Note: The cache can be reloaded by calling `ssl:clear_pem_cache/0`.\n\n- **`bypass_pem_cache = boolean()  `** - Introduced in ssl-8.0.2.\n  Disables the PEM-cache. Can be used as a workaround for the PEM-cache\n  bottleneck before ssl-8.1.1. Defaults to false.\n\n- **`alert_timeout = integer()  `** - Number of milliseconds between\n  sending of a fatal alert and closing the connection. Waiting a little while\n  improves the peers chances to properly receiving the alert so it may shutdown\n  gracefully. Defaults to 5000 milliseconds.\n\n- **`internal_active_n = integer()  `** - For TLS connections this\n  value is used to handle the internal socket. As the implementation was changed\n  from an active once to an active N behavior (N = 100), for performance\n  reasons, this option exist for possible tweaking or restoring of the old\n  behavior (internal_active_n = 1) in unforeseen scenarios. The option will not\n  affect erlang distribution over TLS that will always run in active N mode.\n  Added in ssl-9.1 (OTP-21.2).\n\n- **`server_session_tickets_amount = integer()  `** - Number of session\n  tickets sent by the server. It must be greater than 0. Defaults to 3.\n\n- **`server_session_ticket_lifetime = integer()  `** - Lifetime of\n  session tickets sent by the server. Servers must not use any value greater\n  than 604800 seconds (7 days). Expired tickets are automatically removed.\n  Defaults to 7200 seconds (2 hours).\n\n- **`server_session_ticket_store_size = integer()  `** - Sets the\n  maximum size of the server session ticket store (stateful tickets). Defaults\n  to 1000. Size limit is enforced by dropping old tickets.\n\n- **`server_session_ticket_max_early_data = integer()  `** - Sets the\n  maximum size of the early data that the server accepts and also configures its\n  NewSessionTicket messages to include this same size limit in their\n  early_data_indication extension. Defaults to 16384. Size limit is enforced by\n  both client and server.\n\n- **`client_session_ticket_lifetime = integer()  `** - Lifetime of\n  session tickets in the client ticket store. Expired tickets are automatically\n  removed. Defaults to 7200 seconds (2 hours).\n\n- **`client_session_ticket_store_size = integer()  `** - Sets the\n  maximum size of the client session ticket store. Defaults to 1000. Size limit\n  is enforced by dropping old tickets.","title":"Configuration - SSL Application","ref":"ssl_app.html#configuration"},{"type":"extras","doc":"The SSL application uses [OTP logger](`m:logger`). TLS/DTLS alerts are logged on\nnotice level. Unexpected errors are logged on error level. These log entries\nwill by default end up in the default Erlang log. The option `log_level` may be\nused to in run-time to set the log level of a specific TLS connection, which is\nhandy when you want to use level debug to inspect the TLS handshake setup.","title":"Error Logger and Event Handlers - SSL Application","ref":"ssl_app.html#error-logger-and-event-handlers"},{"type":"extras","doc":"`m:application`","title":"See Also - SSL Application","ref":"ssl_app.html#see-also"},{"type":"extras","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# SSL Release Notes\n\nThis document describes the changes made to the SSL application.","title":"SSL Release Notes","ref":"notes.html"},{"type":"extras","doc":"","title":"SSL 11.2.6 - SSL Release Notes","ref":"notes.html#ssl-11-2-6"},{"type":"extras","doc":"- Enhanced return value spec for active messages from the TLS connection socket.\n\n  Own Id: OTP-19387 Aux Id: [PR-9067]\n\n[PR-9067]: https://github.com/erlang/otp/pull/9067","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 11.2.5 - SSL Release Notes","ref":"notes.html#ssl-11-2-5"},{"type":"extras","doc":"- Avoid generating an internal alert for case that should have been an orderly shutdown by the supervisor.\n\n  Own Id: OTP-19311 Aux Id: [PR-8980]\n\n- If present, extended key-usage TLS (SSL) role check (`pk-clientAuth`, `pk-serverAuth`) should always be performed for peer-cert. An intermediate CA cert may relax the requirement if `AnyExtendedKeyUsage` purpose is present.\n  \n  In OTP-25.3.2.8, OTP-26.2 and OTP-27.0 these requirements became too relaxed. There where two problems, firstly the peer cert extension was only checked if it was marked critical, and secondly the CA cert check did not assert the relaxed `AnyExtendedKeyUsage` purpose.\n  \n  This could result in that certificates might be misused for purposes not intended by the certificate authority.\n  \n  Thanks to Bryan Paxton for reporting the issue.\n\n  Own Id: OTP-19352 Aux Id: [PR-9130], CVE-2024-53846, OTP-19240\n\n[PR-8980]: https://github.com/erlang/otp/pull/8980\n[PR-9130]: https://github.com/erlang/otp/pull/9130","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Back port certificate_authorities option for TLS-1.3 servers to pre TLS-1.3 servers to enable them to disable the sending of certificate authorities in their certificate request. This will have same affect as the the TLS-1.3 server option although it is handled by a different mechanism in these versions, where the functionality is described to be more of a guidance, although some pre TLS clients have proven to make it mandatory as in TLS-1.3 extension handling.\n\n  Own Id: OTP-19325 Aux Id: [PR-9001], ERIERL-1147\n\n[PR-9001]: https://github.com/erlang/otp/pull/9001","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 11.2.4 - SSL Release Notes","ref":"notes.html#ssl-11-2-4"},{"type":"extras","doc":"- Refactor trying to also make some optimizations introduced a bug in signature algorithms checks in OTP-26.2.1. This could manifest itself in not being able to negotiate connections using certificates needing to use some TLS-1.2 compatibility legacy signature schemes.\n\n  Own Id: OTP-19249 Aux Id: ERIERL-1137, [PR-8866]\n\n- Correct timeout handling for termination code run for own alerts, so that intended timeout is used instead of falling back to OS TCP-stack timeout that is unreasonably long on some platforms.\n\n  Own Id: OTP-19274 Aux Id: [PR-8901]\n\n- Fix assertion so that works as intended.\n  This could result in that some TLS-1.2 clients would fail to connect to the the erlang server.  Bug introduced in OTP-27.1.1\n\n  Own Id: OTP-19288 Aux Id: [GH-8908], [PR-8916]\n\n[PR-8866]: https://github.com/erlang/otp/pull/8866\n[PR-8901]: https://github.com/erlang/otp/pull/8901\n[GH-8908]: https://github.com/erlang/otp/issues/8908\n[PR-8916]: https://github.com/erlang/otp/pull/8916","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 11.2.3 - SSL Release Notes","ref":"notes.html#ssl-11-2-3"},{"type":"extras","doc":"- Starting from TLS-1.3 some server handshake alerts might arrive after ssl:connection/2,3,4 has returned. If the socket is in active mode the controlling process will get the alert message, but passive sockets would only get {error, closed} on next call to ssl:recv/2,3 or ssl/setopts/2. Passive sockets calls will now return  {error,  error_alert()}  instead.\n\n  Own Id: OTP-19236 Aux Id: [PR-8261]\n\n- Servers configured to support only version (pre TLS-1.2) should ignore hello version extension, as it is an unknown extension to them, this will result in that new clients that do not support the old server version will get an insufficient security alert from the server and not a protocol version alert, this is consistent with how old servers not able to support higher protocol versions work.\n\n  Own Id: OTP-19257\n\n[PR-8261]: https://github.com/erlang/otp/pull/8261","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 11.2.2 - SSL Release Notes","ref":"notes.html#ssl-11-2-2"},{"type":"extras","doc":"- A race in the kTLS flavour of SSL distribution has been fixed so that `inet_drv.c` doesn't read ahead too much data, which could cause the kTLS encryption to be activated too late when some encrypted data had already been read into the `inet_drv.c` buffer as unencrypted.\n\n  Own Id: OTP-19175 Aux Id: [GH-8561], [PR-8690]\n\n[GH-8561]: https://github.com/erlang/otp/issues/8561\n[PR-8690]: https://github.com/erlang/otp/pull/8690","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- All TLS-1.3 terminations are now graceful (previous TLS version terminations already were).\n\n  Own Id: OTP-17848\n\n- It is now possible to use a verification fun of arity 4, giving the user fun access to both encoded and decoded versions of the certificate. This is desirable as a workaround for encoding errors preventing re-encoding from being reliable. This also saves some work load if the encoded version is needed.\n  \n  Note that calling `public_key:pkix_path_validation/3` with only decoded certs is not recommended, due to the decoding workarounds, although it will work as long as the workarounds are not needed.\n  \n  If the decoded version is needed before thecall to `m:public_key` it is recommend to use the combined_cert- type to avoid double decoding.  Note that the path validation algorithm itself always needs both the encoded and decoded versions of the certs.\n  \n  The ssl implementation will now benefit from using this function instead of emulating the verify_fun/4.\n\n  Own Id: OTP-19169\n\n- Compiler warnings for some removed functions have been corrected to point out the correct replacement functions.\n\n  Own Id: OTP-19186 Aux Id: [PR-8709]\n\n- Include more information in logging of SNI (Server Name Indication) mismatch error.\n\n  Own Id: OTP-19187\n\n[PR-8709]: https://github.com/erlang/otp/pull/8709","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 11.2.1 - SSL Release Notes","ref":"notes.html#ssl-11-2-1"},{"type":"extras","doc":"- Check  for TLS-1.3 support should check minimum requirements.\n\n  Own Id: OTP-19094 Aux Id: [GH-8489]\n\n- If both TLS-1.3 and TLS-1.2 is supported\n  and TLS-1.2 negotiated convert TLS-1.3 ECDSA schemes to TLS-1.2 hash and signature pairs for increased interoperability.\n\n  Own Id: OTP-19107 Aux Id: [GH-8376]\n\n- TLS-1.3 negotiation now uses SNI based options correctly instead of ignoring them.\n\n  Own Id: OTP-19140\n\n[GH-8489]: https://github.com/erlang/otp/issues/8489\n[GH-8376]: https://github.com/erlang/otp/issues/8376","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Make it easier to distinguish between a invalid signature and unsupported signature.\n\n  Own Id: OTP-19091\n\n- Enhance ALERT logs to help understand what causes the alert.\n\n  Own Id: OTP-19092 Aux Id: [GH-8482]\n\n- When the default value for signature_algs is used, default the signature_algs_cert to the default value + rsa_pkcs1_sha1 to allow this algorithms for certificates but not for the TLS protocol. This is for better interoperability.  If signature_algs is set explicitly signature_algs_cert must also be set explicitly if they should be different.\n\n  Own Id: OTP-19152 Aux Id: [GH-8588]\n\n[GH-8482]: https://github.com/erlang/otp/issues/8482\n[GH-8588]: https://github.com/erlang/otp/issues/8588","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 11.2 - SSL Release Notes","ref":"notes.html#ssl-11-2"},{"type":"extras","doc":"- Starting a TLS server without sufficient credentials (certificate or anonymous cipher) would work, but it was impossible to connect to it.\n  \n  This has been corrected to return an error instead of starting the server.\n\n  Own Id: OTP-18887 Aux Id: [GH-7493], [PR-7918]\n\n- ASN.1 decoding errors are handled in more places to ensure that errors are returned instead of cause a crash.\n\n  Own Id: OTP-18969 Aux Id: [GH-8058], [PR-8256]\n\n- Improved error checking on the API functions.\n\n  Own Id: OTP-18992 Aux Id: [GH-8066], [PR-8156]\n\n[GH-7493]: https://github.com/erlang/otp/issues/7493\n[PR-7918]: https://github.com/erlang/otp/pull/7918\n[GH-8058]: https://github.com/erlang/otp/issues/8058\n[PR-8256]: https://github.com/erlang/otp/pull/8256\n[GH-8066]: https://github.com/erlang/otp/issues/8066\n[PR-8156]: https://github.com/erlang/otp/pull/8156","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- The `ssl` client can negotiate and handle certificate status request (OCSP stapling support on the client side).\n  \n  Thanks to voltone for interop testing and related discussions.\n\n  Own Id: OTP-18606 Aux Id: OTP-16875,OTP-16448\n\n- Memory consumption has been reduced and performance increased by refactoring internal data structures and their usage.\n\n  Own Id: OTP-18665 Aux Id: [PR-7447]\n\n- Added `c:ssl_crl_cache_api:lookup/2` as an optional `-callback` attribute.\n\n  Own Id: OTP-18788 Aux Id: [PR-7700]\n\n- Key customization support has been extended to allow flexibility for implementers of  for instance hardware security modules (HSM) or trusted platform modules (TPM).\n\n  Own Id: OTP-18876 Aux Id: [PR-7898], [PR-7475]\n\n- The `proc_lib:set_label/1` function is now used to increase observability of `ssl` processes.\n\n  Own Id: OTP-18879\n\n- Brainpool elliptic curves are now supported in TLS-1.3.\n\n  Own Id: OTP-18884 Aux Id: [PR-8056]\n\n- The documentation has been migrated to use Markdown and ExDoc.\n\n  Own Id: OTP-18955 Aux Id: [PR-8026]\n\n- For security reasons, the CBC ciphers are now longer included in the list of default ciphers for TLS-1.2.\n\n  *** POTENTIAL INCOMPATIBILITY ***\n\n  Own Id: OTP-19025 Aux Id: [PR-8250]\n\n- There is a new `cert_policy_opts` option to configure certificate policy options for the certificate path validation.\n\n  Own Id: OTP-19027 Aux Id: [PR-8255]\n\n[PR-7447]: https://github.com/erlang/otp/pull/7447\n[PR-7700]: https://github.com/erlang/otp/pull/7700\n[PR-7898]: https://github.com/erlang/otp/pull/7898\n[PR-7475]: https://github.com/erlang/otp/pull/7475\n[PR-8056]: https://github.com/erlang/otp/pull/8056\n[PR-8026]: https://github.com/erlang/otp/pull/8026\n[PR-8250]: https://github.com/erlang/otp/pull/8250\n[PR-8255]: https://github.com/erlang/otp/pull/8255","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 11.1.4.6 - SSL Release Notes","ref":"notes.html#ssl-11-1-4-6"},{"type":"extras","doc":"* If present, extended key-usage TLS (SSL) role check (`pk-clientAuth`, `pk-serverAuth`) should always be performed for peer-cert. An intermediate CA cert may relax the requirement if `AnyExtendedKeyUsage` purpose is present.\n\n  In OTP-25.3.2.8, OTP-26.2 and OTP-27.0 these requirements became too relaxed. There where two problems, firstly the peer cert extension was only checked if it was marked critical, and secondly the CA cert check did not assert the relaxed `AnyExtendedKeyUsage` purpose.\n\n  This could result in that certificates might be misused for purposes not intended by the certificate authority.\n\n  Thanks to Bryan Paxton for reporting the issue.\n\n  Own Id: OTP-19352 Aux Id: PR-9130, CVE-2024-53846, OTP-19240","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 11.1.4.5 - SSL Release Notes","ref":"notes.html#ssl-11-1-4-5"},{"type":"extras","doc":"* Avoid generating an internal alert for case that should have been an orderly shutdown by the supervisor.\n\n  Own Id: OTP-19311 Aux Id: PR-8980","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"* Back port certificate_authorities option for TLS-1.3 servers to pre TLS-1.3 servers to enable them to disable the sending of certificate authorities in their certificate request. This will have same affect as the the TLS-1.3 server option although it is handled by a different mechanism in these versions, where the functionality is described to be more of a guidance, although some pre TLS clients have proven to make it mandatory as in TLS-1.3 extension handling.\n\n  Own Id: OTP-19325 Aux Id: PR-9001, ERIERL-1147","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 11.1.4.4 - SSL Release Notes","ref":"notes.html#ssl-11-1-4-4"},{"type":"extras","doc":"* Starting from TLS-1.3 some server handshake alerts might arrive after ssl:connection/2,3,4 has returned. If the socket is in active mode the controlling process will get the alert message, but passive sockets would only get \\{error, closed\\} on next call to ssl:recv/2,3 or ssl/setopts/2. Passive sockets calls will now return \\{error, error_alert()\\} instead.\n\n  Own Id: OTP-19236 Aux Id: PR-8261\n* Refactor trying to also make some optimizations introduced a bug in signature algorithms checks in OTP-26.2.1. This could manifest itself in not being able to negotiate connections using certificates needing to use some TLS-1.2 compatibility legacy signature schemes.\n\n  Own Id: OTP-19249 Aux Id: ERIERL-1137, PR-8866\n* Servers configured to support only version (pre TLS-1.2) should ignore hello version extension, as it is an unknown extension to them, this will result in that new clients that do not support the old server version will get an insufficient security alert from the server and not a protocol version alert, this is consistent with how old servers not able to support higher protocol versions work.\n\n  Own Id: OTP-19257 Aux Id: ERIERL-1131\n* Correct timeout handling for termination code run for own alerts, so that intended timeout is used instead of falling back to OS TCP-stack timeout that is unreasonably long on some platforms.\n\n  Own Id: OTP-19274 Aux Id: PR-8901","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 11.1.4.3 - SSL Release Notes","ref":"notes.html#ssl-11-1-4-3"},{"type":"extras","doc":"* A race in the kTLS flavour of SSL distribution has been fixed so inet_drv.c doesn't read ahead too much data which could cause the kTLS encryption to be activated too late when some encrypted data had already been read into the inet_drv.c buffer as unencrypted.\n\n  Own Id: OTP-19175 Aux Id: GH-8561, PR-8690","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"* Make sure all TLS-1.3 terminations are graceful (previous TLS version terminations already are).\n\n  Own Id: OTP-17848\n* Include more information in logging of SNI (Server Name Indication) mismatch error.\n\n  Own Id: OTP-19187","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 11.1.4.2 - SSL Release Notes","ref":"notes.html#ssl-11-1-4-2"},{"type":"extras","doc":"* When the default value for signature_algs is used, default the signature_algs_cert to the default value + rsa_pkcs1_sha1 to allow this algorithms for certificates but not for the TLS protocol. This is for better interoperability. If signature_algs is set explicitly signature_algs_cert must also be set explicitly if they should be different.\n\n  Own Id: OTP-19152 Aux Id: GH-8588","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 11.1.4.1 - SSL Release Notes","ref":"notes.html#ssl-11-1-4-1"},{"type":"extras","doc":"* Check for TLS-1.3 support should check minimum requirements.\n\n  Own Id: OTP-19094 Aux Id: GH-8489\n* If both TLS-1.3 and TLS-1.2 is supported and TLS-1.2 negotiated convert TLS-1.3 ECDSA schemes to TLS-1.2 hash and signature pairs for increased interoperability.\n\n  Own Id: OTP-19107 Aux Id: GH-8376\n* TLS-1.3 negotiation now uses SNI based options correctly instead of ignoring them.\n\n  Own Id: OTP-19140","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"* Make it easier to distinguish between a invalid signature and unsupported signature.\n\n  Own Id: OTP-19091\n* Enhance ALERT logs to help understand what causes the alert.\n\n  Own Id: OTP-19092 Aux Id: GH-8482","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 11.1.4 - SSL Release Notes","ref":"notes.html#ssl-11-1-4"},{"type":"extras","doc":"* Fix certificate authorities check so that CA closest to peer is not lost. It could manifest itself in a failed connection as the client failed to realize it had a valid certificate chain to send to the server.\n\n  Own Id: OTP-19065 Aux Id: GH-8356, PR-8367\n* ssl:signature_algs/2 did not list some legacy algorithm schemes correctly when listing all algorithms available.\n\n  Own Id: OTP-19067 Aux Id: PR-8379","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 11.1.3 - SSL Release Notes","ref":"notes.html#ssl-11-1-3"},{"type":"extras","doc":"* Cleanup and close all connections in DTLS when the listen socket owner dies.\n\n  Improved IPv6 handling in DTLS.\n\n  Own Id: OTP-19037 Aux Id: GH-7951 GH-7955\n* Fixed a crash in dtls accept.\n\n  Own Id: OTP-19059 Aux Id: GH-8338","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 11.1.2 - SSL Release Notes","ref":"notes.html#ssl-11-1-2"},{"type":"extras","doc":"* ssl:prf/5, will start working instead of hanging in a TLS-1.3 context if called appropriately. Note that the implementation has changed and in OTP-27 a more adequate API will be documented.\n\n  Own Id: OTP-18890 Aux Id: GH-7911\n* Server name verification didn't work if a connection was made with IP-address as a string.\n\n  Own Id: OTP-18909 Aux Id: GH-7968\n* The fallback after \"dh\" ssl option was undefined was to get \"dh\" from ssl options again. This is clearly wrong and now changed to the documented fallback \"dhfile\" ssl option.\n\n  Own Id: OTP-18919 Aux Id: PR-7984\n* Correct default value selection for DTLS. Will only affect users linked with really old version of cryptolib library.\n\n  Own Id: OTP-18962 Aux Id: GH-8079\n* Adhere elliptic curves with RFC 8422 pre TLS-1.3, that is Edwards curves are added to curves that can be used for key exchange, and documentation and implementation of eccs/0,1 are aligned.\n\n  Own Id: OTP-18991","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"* Improve alert reason when ecdhe_rsa key_exchange does not have any common curves to use\n\n  Own Id: OTP-18985","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 11.1.1 - SSL Release Notes","ref":"notes.html#ssl-11-1-1"},{"type":"extras","doc":"* Legacy name handling could cause interop problems between TLS-1.3/1.2 client and TLS-1.2 server.\n\n  Own Id: OTP-18917 Aux Id: GH-7978","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 11.1 - SSL Release Notes","ref":"notes.html#ssl-11-1"},{"type":"extras","doc":"- ssl application will validate id-kp-serverAuth and id-kp-clientAuth extended\n  key usage only in end entity certificates. public_key application will\n  disallow \"anyExtendedKeyUsage\" for CA certificates that includes the extended\n  key usage extension and marks it critical.\n\n  Own Id: OTP-18739\n\n- Replaced unintentional Erlang Public License 1.1 headers in some files with\n  the intended Apache License 2.0 header.\n\n  Own Id: OTP-18815 Aux Id: PR-7780\n\n- Correct handling of TLS-1.3 legacy scheme names, could cause interop failures\n  for TLS-1.2 clients.\n\n  Own Id: OTP-18817\n\n- Add missing export for connection_info() API type.\n\n  Own Id: OTP-18886","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Fixed `server name indication` which was not handled properly.\n\n  Own Id: OTP-18836 Aux Id: GH-7795\n\n- Align documentation and implementation\n\n  Own Id: OTP-18853 Aux Id: PR-7841\n\n- Improve connection setup by optimizing certificate lookup.\n\n  Own Id: OTP-18893 Aux Id: PR-7920 PR-7921","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 11.0.3 - SSL Release Notes","ref":"notes.html#ssl-11-0-3"},{"type":"extras","doc":"- Avoid function clause error in ssl:getopts/2 by handling that inet:getopts may\n  return an empty list during some circumstances, such as the socket being in a\n  closing state.\n\n  Own Id: OTP-18697 Aux Id: GH-7506\n\n- The API function \\`ssl:recv/3\\` has been tightened to disallow negative\n  length, which has never been documented to work, but was passed through and\n  caused strange errors.\n\n  Own Id: OTP-18700 Aux Id: GH-7507\n\n- When a client initiated renegotiation was rejected and the client socket was\n  in active mode the expected error message to the controlling process was not\n  sent.\n\n  Own Id: OTP-18712 Aux Id: GH-7431","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Add some guidance for signature algorithms configuration in ssl applications\n  users guide.\n\n  Own Id: OTP-18631","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 11.0.2 - SSL Release Notes","ref":"notes.html#ssl-11-0-2"},{"type":"extras","doc":"- Added keylog information to all protocol versions in\n  `ssl:connection_information/2`.\n\n  Own Id: OTP-18643 Aux Id: ERIERL-932","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Add RFC-6083 considerations for DTLS to enable gen_sctp based callback for the\n  transport.\n\n  Own Id: OTP-18618 Aux Id: ERIERL-932","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 11.0.1 - SSL Release Notes","ref":"notes.html#ssl-11-0-1"},{"type":"extras","doc":"- Make sure that selection of client certificates handle both TLS-1.3 and\n  TLS-1.2 names correctly. Could cause valid client certificate to not be\n  selected, and an empty client certificate message to be sent to server.\n\n  Own Id: OTP-18588 Aux Id: GH-7264, PR-7277\n\n- Improved `ssl:format_error/1` to handle more error tuples.\n\n  Own Id: OTP-18596 Aux Id: GH-7247\n\n- Fixed hanging `ssl:connect` when ssl application is not started.\n\n  Own Id: OTP-18603 Aux Id: GH-7297\n\n- Correct handling of retransmission timers, current behavior could cause\n  unwanted delays.\n\n  Own Id: OTP-18632 Aux Id: PR-7300, GH-7301","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 11.0 - SSL Release Notes","ref":"notes.html#ssl-11-0"},{"type":"extras","doc":"- Remove less that 256 bit ECC from default supported ECC pre TLS-1.3\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-14771\n\n- Improved error checking and handling of ssl options.\n\n  Own Id: OTP-15903\n\n- With this change, stateless tickets generated by server with anti_replay\n  option enabled can be used for creating ClientHello throughout ticket\n  lifetime. Without this change, usability was limited to WindowSize number of\n  seconds configured for anti_replay option.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-18168 Aux Id: PR-6019, GH-6014\n\n- Support for Kernel TLS (kTLS), has been added to the SSL application, for TLS\n  distribution (`-proto_dist inet_tls`), the SSL option `{ktls, true}`. Using\n  this for general SSL sockets is uncomfortable, undocumented and not\n  recommended since it requires very platform dependent raw options.\n\n  This, for now, only works for some not too old Linux distributions. Roughly, a\n  kernel 5.2.0 or later with support for UserLand Protocols and the kernel\n  module `tls` is required.\n\n  Own Id: OTP-18235 Aux Id: PR-6104, PR-5840\n\n- With this change, TLS 1.3 server can be configured to include client\n  certificate in session ticket.\n\n  Own Id: OTP-18253\n\n- With this change, it is possible to configure encryption seed to be used with\n  TLS1.3 stateless tickets. This enables using tickets on different server\n  instances.\n\n  Own Id: OTP-18254 Aux Id: PR-5982\n\n- Debugging enhancements.\n\n  Own Id: OTP-18312\n\n- With this change, maybe keyword atom is not used as function name in ssl code.\n\n  Own Id: OTP-18335\n\n- Replace size/1 with either tuple_size/1 or byte_size/1\n\n  The [`size/1`](`size/1`) BIF is not optimized by the JIT, and its use can\n  result in worse types for Dialyzer.\n\n  When one knows that the value being tested must be a tuple,\n  [`tuple_size/1`](`tuple_size/1`) should always be preferred.\n\n  When one knows that the value being tested must be a binary,\n  [`byte_size/1`](`byte_size/1`) should be preferred. However,\n  [`byte_size/1`](`byte_size/1`) also accepts a bitstring (rounding up size to a\n  whole number of bytes), so one must make sure that the call to `byte_size/` is\n  preceded by a call to [`is_binary/1`](`is_binary/1`) to ensure that bitstrings\n  are rejected. Note that the compiler removes redundant calls to\n  [`is_binary/1`](`is_binary/1`), so if one is not sure whether previous code\n  had made sure that the argument is a binary, it does not harm to add an\n  [`is_binary/1`](`is_binary/1`) test immediately before the call to\n  [`byte_size/1`](`byte_size/1`).\n\n  Own Id: OTP-18405 Aux Id:\n  GH-6672,PR-6702,PR-6768,PR-6700,PR-6769,PR-6812,PR-6814\n\n- For security reasons remove support for SHA1 and DSA algorithms from default\n  values.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-18438 Aux Id: GH-6679\n\n- Mitigate memory usage from large certificate chains by lowering the maximum\n  handshake size. This should not effect the common cases, if needed it can be\n  configured to a higher value.\n\n  Own Id: OTP-18453\n\n- Change the client default verify option to verify_peer. Note that this makes\n  it mandatory to also supply trusted CA certificates or explicitly set verify\n  to verify_none. This also applies when using the so called anonymous test\n  cipher suites defined in TLS versions pre TLS-1.3.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-18455 Aux Id: GH-5899\n\n- Erlang distribution code in Kernel and SSL has been refactored a bit to\n  facilitate debugging and re-usability, which shouldn't have any noticeable\n  effects on behaviour or performance.\n\n  Own Id: OTP-18456\n\n- Add encoding and decoding of use_srtp hello extension to facilitate for DTLS\n  users to implement SRTP functionality.\n\n  Own Id: OTP-18459\n\n- Refactors the (`ssl` application to use macros for TLS and DTLS versions\n  instead of hard-coded tuple numbers. This change improves the maintainability\n  of `ssl`\n\n  Own Id: OTP-18465 Aux Id: GH-7065\n\n- If the function ssl:renegotiate/1 is called on connection that is running\n  TLS-1.3 return an error instead of hanging or timing out.\n\n  Own Id: OTP-18507\n\n- If a user cancel alert with level warning is received during handshake make it\n  be handled the same regardless of TLS version. If it is received in connection\n  in TLS-1.3 regard it as an error as it is inappropriate.\n\n  In TLS-1.3 all error alerts are considered FATAL regardless of legacy alert\n  type. But make sure legacy type is printed in logs to not confuse users that\n  are expecting the same legacy type as sent by peer.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-18531\n\n- Make `fail_if_no_peer_cert` default true if verify_peer is set on the server,\n  otherwise the server will accept the connection if verify_peer is set and the\n  user have forgot to set the fail_if_no_peer_cert and the client did not send a\n  certificate.\n\n  Own Id: OTP-18567\n\n- To make it easier to configure signature algorithms with algorithms that are\n  moved from the default add the API function signature_algs/2 that lists\n  possible values. Also make sha224 a non default value.\n\n  Own Id: OTP-18572","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 10.9.1.7 - SSL Release Notes","ref":"notes.html#ssl-10-9-1-7"},{"type":"extras","doc":"* Avoid generating an internal alert for case that should have been an orderly shutdown by the supervisor.\n\n  Own Id: OTP-19311 Aux Id: PR-8980\n* If present, extended key-usage TLS (SSL) role check (`pk-clientAuth`, `pk-serverAuth`) should always be performed for peer-cert. An intermediate CA cert may relax the requirement if `AnyExtendedKeyUsage` purpose is present.\n\n  In OTP-25.3.2.8, OTP-26.2 and OTP-27.0 these requirements became too relaxed. There where two problems, firstly the peer cert extension was only checked if it was marked critical, and secondly the CA cert check did not assert the relaxed `AnyExtendedKeyUsage` purpose.\n\n  This could result in that certificates might be misused for purposes not intended by the certificate authority.\n\n  Thanks to Bryan Paxton for reporting the issue.\n\n  Own Id: OTP-19352 Aux Id: PR-9130, CVE-2024-53846, OTP-19240","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.9.1.6 - SSL Release Notes","ref":"notes.html#ssl-10-9-1-6"},{"type":"extras","doc":"* Starting from TLS-1.3 some server handshake alerts might arrive after ssl:connection/2,3,4 has returned. If the socket is in active mode the controlling process will get the alert message, but passive sockets would only get \\{error, closed\\} on next call to ssl:recv/2,3 or ssl/setopts/2. Passive sockets calls will now return \\{error, error_alert()\\} instead.\n\n  Own Id: OTP-19236 Aux Id: PR-8261\n* Servers configured to support only version (pre TLS-1.2) should ignore hello version extension, as it is an unknown extension to them, this will result in that new clients that do not support the old server version will get an insufficient security alert from the server and not a protocol version alert, this is consistent with how old servers not able to support higher protocol versions work.\n\n  Own Id: OTP-19257 Aux Id: ERIERL-1131\n* Correct timeout handling for termination code run for own alerts, so that intended timeout is used instead of falling back to OS TCP-stack timeout that is unreasonably long on some platforms.\n\n  Own Id: OTP-19274 Aux Id: PR-8901","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.9.1.5 - SSL Release Notes","ref":"notes.html#ssl-10-9-1-5"},{"type":"extras","doc":"* TLS-1.3 negotiation now uses SNI based options correctly instead of ignoring them.\n\n  Own Id: OTP-19140","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.9.1.4 - SSL Release Notes","ref":"notes.html#ssl-10-9-1-4"},{"type":"extras","doc":"* Fix certificate authorities check so that CA closest to peer is not lost. It could manifest itself in a failed connection as the client failed to realize it had a valid certificate chain to send to the server.\n\n  Own Id: OTP-19065 Aux Id: GH-8356, PR-8367","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.9.1.3 - SSL Release Notes","ref":"notes.html#ssl-10-9-1-3"},{"type":"extras","doc":"- ssl application will validate id-kp-serverAuth and id-kp-clientAuth extended\n  key usage only in end entity certificates. public_key application will\n  disallow \"anyExtendedKeyUsage\" for CA certificates that includes the extended\n  key usage extension and marks it critical.\n\n  Own Id: OTP-18739\n\n- Add missing export for connection_info() API type.\n\n  Own Id: OTP-18886","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.9.1.2 - SSL Release Notes","ref":"notes.html#ssl-10-9-1-2"},{"type":"extras","doc":"- The API function \\`ssl:recv/3\\` has been tightened to disallow negative\n  length, which has never been documented to work, but was passed through and\n  caused strange errors.\n\n  Own Id: OTP-18700 Aux Id: GH-7507\n\n- When a client initiated renegotiation was rejected and the client socket was\n  in active mode the expected error message to the controlling process was not\n  sent.\n\n  Own Id: OTP-18712 Aux Id: GH-7431","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.9.1.1 - SSL Release Notes","ref":"notes.html#ssl-10-9-1-1"},{"type":"extras","doc":"- Added keylog information to all protocol versions in\n  `ssl:connection_information/2`.\n\n  Own Id: OTP-18643 Aux Id: ERIERL-932","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Add RFC-6083 considerations for DTLS to enable gen_sctp based callback for the\n  transport.\n\n  Own Id: OTP-18618 Aux Id: ERIERL-932","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 10.9.1 - SSL Release Notes","ref":"notes.html#ssl-10-9-1"},{"type":"extras","doc":"- With this change, ssl:connection_information/2 returns correct keylog data\n  after TLS1.3 key update.\n\n  Own Id: OTP-18489\n\n- Client signature algorithm list input order is now honored again , it was\n  accidently reversed by a previous fix.\n\n  Own Id: OTP-18550","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.9 - SSL Release Notes","ref":"notes.html#ssl-10-9"},{"type":"extras","doc":"- Fixed that new `dtls` connections from the same client ip port combination\n  works. If there is a process waiting for accept the new connection will\n  connect to that, otherwise it will try to re-connect to the old server\n  connection.\n\n  Own Id: OTP-18371 Aux Id: GH-6160\n\n- When shutting down a node that uses SSL distribution (`-proto_dist inet_tls`),\n  a confusing error message about an unexpected process exit was printed. This\n  particular message is no longer generated.\n\n  Own Id: OTP-18443 Aux Id: PR-6810","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- fixes the type spec for ssl:format_error/1\n\n  Own Id: OTP-18366 Aux Id: PR-6565, GH-6506\n\n- Replace size/1 with either tuple_size/1 or byte_size/1\n\n  The [`size/1`](`size/1`) BIF is not optimized by the JIT, and its use can\n  result in worse types for Dialyzer.\n\n  When one knows that the value being tested must be a tuple,\n  [`tuple_size/1`](`tuple_size/1`) should always be preferred.\n\n  When one knows that the value being tested must be a binary,\n  [`byte_size/1`](`byte_size/1`) should be preferred. However,\n  [`byte_size/1`](`byte_size/1`) also accepts a bitstring (rounding up size to a\n  whole number of bytes), so one must make sure that the call to `byte_size/` is\n  preceded by a call to [`is_binary/1`](`is_binary/1`) to ensure that bitstrings\n  are rejected. Note that the compiler removes redundant calls to\n  [`is_binary/1`](`is_binary/1`), so if one is not sure whether previous code\n  had made sure that the argument is a binary, it does not harm to add an\n  [`is_binary/1`](`is_binary/1`) test immediately before the call to\n  [`byte_size/1`](`byte_size/1`).\n\n  Own Id: OTP-18432 Aux Id:\n  GH-6672,PR-6793,PR-6784,PR-6787,PR-6785,PR-6682,PR-6800,PR-6797,PR-6798,PR-6799,PR-6796,PR-6813,PR-6671,PR-6673,PR-6684,PR-6694,GH-6677,PR-6696,PR-6670,PR-6674","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 10.8.7 - SSL Release Notes","ref":"notes.html#ssl-10-8-7"},{"type":"extras","doc":"- Maximize compatibility by ignoring change_cipher_spec during handshake even if\n  middle_box_mode is not negotiated (mandated by client)\n\n  Own Id: OTP-18433 Aux Id: GH-6772\n\n- Move assert of middlebox message after an hello_retry_request to maximize\n  interoperability. Does not changes semantics of the protocol only allows\n  unexpected message delay from server.\n\n  Own Id: OTP-18467 Aux Id: GH-6807","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 10.8.6 - SSL Release Notes","ref":"notes.html#ssl-10-8-6"},{"type":"extras","doc":"- With this change, tls_sender process is hibernated after sufficient\n  inactivity.\n\n  Own Id: OTP-18314 Aux Id: GH-6373\n\n- Correct handling of legacy schemes so that ECDSA certs using sha1 may be used\n  for some TLS-1.3 configurations.\n\n  Own Id: OTP-18332 Aux Id: GH-6435, PR-6435, ERL-6435\n\n- With this change, tls_sender does not cause logger crash upon key update.\n\n  Own Id: OTP-18349","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Enhance warning message\n\n  Own Id: OTP-18257 Aux Id: GH-6307\n\n- Provide server option to make certificate_authorities extension in the TLS-1.3\n  servers certificate request optional. This will allow clients to send\n  incomplete chains that may be reconstructable and thereby verifiable by the\n  server, but that would not adhere to the certificate_authorities extension.\n\n  Own Id: OTP-18267 Aux Id: PR-6228, GH-6106\n\n- If the `verify_fun` handles four arguments the DER cert will be supplied as\n  one of the arguments.\n\n  Own Id: OTP-18302 Aux Id: ERIERL-867","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 10.8.5 - SSL Release Notes","ref":"notes.html#ssl-10-8-5"},{"type":"extras","doc":"- Fixes handling of symlinks in cacertfile option.\n\n  Own Id: OTP-18266 Aux Id: GH-6328","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.8.4 - SSL Release Notes","ref":"notes.html#ssl-10-8-4"},{"type":"extras","doc":"- Reject unexpected application data in all relevant places for all TLS\n  versions. Also, handle TLS-1.3 middlebox compatibility with more care. This\n  will make malicious connections fail early and further, mitigate possible DoS\n  attacks, that would be caught by the handshake timeout.\n\n  Thanks to Aina Toky Rasoamanana and Olivier Levillain from Télécom SudParis\n  for alerting us of the issues in our implementation.\n\n  Own Id: OTP-18044\n\n- With this change, value of cacertfile option will be adjusted before loading\n  certs from the file. Adjustments include converting relative paths to absolute\n  and converting symlinks to actual file path.\n\n  Thanks to Marcus Johansson\n\n  Own Id: OTP-18099 Aux Id: PR-6287\n\n- In TLS-1.3, if chain certs are missing (so server auth domain adherence can\n  not be determined) send peer cert and hope the server is able to recreate a\n  chain in its auth domain.\n\n  Own Id: OTP-18191 Aux Id: GH-6105\n\n- Make sure periodical refresh of CA certificate files repopulates cache\n  properly.\n\n  Own Id: OTP-18195\n\n- Correct internal CRL cache functions to use internal format consistently.\n\n  Own Id: OTP-18203 Aux Id: PR-5996\n\n- Incorrect handling of client middlebox negotiation for TLS-1.3 could result in\n  that a TLS-1.3 server would not use middlebox mode although the client was\n  expecting it too and failing the negotiation with unexpected message.\n\n  Own Id: OTP-18219 Aux Id: GH-6241, PR-6249\n\n- If the \"User\" process, the process starting the TLS connection, gets killed in\n  the middle of spawning the dynamic connection tree make sure we do not leave\n  any processes behind.\n\n  Own Id: OTP-18233 Aux Id: GH-6244, PR-6270","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- A vulnerability has been discovered and corrected. It is registered as\n  CVE-2022-37026 \"Client Authentication Bypass\". Corrections have been released\n  on the supported tracks with patches 23.3.4.15, 24.3.4.2, and 25.0.2. The\n  vulnerability might also exist in older OTP versions. We recommend that\n  impacted users upgrade to one of these versions or later on the respective\n  tracks. OTP 25.1 would be an even better choice. Impacted are those who are\n  running an ssl/tls/dtls server using the ssl application either directly or\n  indirectly via other applications. For example via inets (httpd), cowboy, etc.\n  Note that the vulnerability only affects servers that request client\n  certification, that is sets the option \\{verify, verify_peer\\}.\n\n  Own Id: OTP-18241","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 10.8.3 - SSL Release Notes","ref":"notes.html#ssl-10-8-3"},{"type":"extras","doc":"- The link to crypto:engine_load refered the function with wrong arity.\n\n  Own Id: OTP-18173","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.8.2 - SSL Release Notes","ref":"notes.html#ssl-10-8-2"},{"type":"extras","doc":"- Improved handling of unexpected messages during the handshake, taking the\n  right action for unexpected messages.\n\n  Own Id: OTP-18145","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.8.1 - SSL Release Notes","ref":"notes.html#ssl-10-8-1"},{"type":"extras","doc":"- When a TLS-1.3 enabled client tried to talk to a TLS-1.2 server that coalesces\n  TLS-1.2 handshake message over one TLS record, the connection could fail due\n  to some message being handled in the wrong state, this has been fixed.\n\n  Own Id: OTP-18087 Aux Id: GH-5961\n\n- Correctly handles supported protocol version change from default to something\n  else by sni_fun supplied to ssl:handshake/\\[2,3] together with a TCP-socket\n  (so called upgrade).\n\n  Own Id: OTP-18100 Aux Id: GH-5985\n\n- Also, TLS-1.3 should respond with a protocol version alert if previous\n  versions, that are supported but not configured, are attempted.\n\n  Own Id: OTP-18129 Aux Id: GH-5950","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.8 - SSL Release Notes","ref":"notes.html#ssl-10-8"},{"type":"extras","doc":"- When a TLS-1.3 enabled client tried to talk to a TLS-1.2 server that coalesces\n  TLS-1.2 handshake message over one TLS record, the connection could fail due\n  to some message being handled in the wrong state, this has been fixed.\n\n  Own Id: OTP-18087 Aux Id: GH-5961\n\n- Fixed tls-1.3 session ticket lifetime which was discarded to quickly before.\n\n  Own Id: OTP-18092 Aux Id: PR-5959","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- With this change, it is possible to provide several certificates. Most\n  appropriate will be selected based on negotiated properties.\n\n  Own Id: OTP-15993 Aux Id: GH-4143\n\n- Add options for users to be able to set spawn_opts for TLS processes (sender\n  and receiver) this may be useful for tuning trade-offs between CPU and Memory\n  usage.\n\n  Own Id: OTP-17855 Aux Id: PR-5328\n\n- Allow key file passwords to be input as a single binary, that is we change the\n  data type to be the more for the purpose logical data type iodata() instead of\n  string().\n\n  Own Id: OTP-17890\n\n- Logging enhancement, add location information to the warning log message.\n\n  Own Id: OTP-18000 Aux Id: PR-5790\n\n- Now also accepts the signature_algs_cert option in TLS-1.2 configuration.\n\n  Own Id: OTP-18014\n\n- Handle certificate selection correctly for server fallback and certificate\n  authorities considerations.\n\n  Own Id: OTP-18045 Aux Id: ERIERL-792, OTP-15993\n\n- Enhance handling of handshake decoding errors, especially for certificate\n  authorities extension to ensure graceful termination.\n\n  Own Id: OTP-18085","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 10.7.3.9 - SSL Release Notes","ref":"notes.html#ssl-10-7-3-9"},{"type":"extras","doc":"- When a client initiated renegotiation was rejected and the client socket was\n  in active mode the expected error message to the controlling process was not\n  sent.\n\n  Own Id: OTP-18712 Aux Id: GH-7431","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.7.3.8 - SSL Release Notes","ref":"notes.html#ssl-10-7-3-8"},{"type":"extras","doc":"- Added keylog information to all protocol versions in\n  `ssl:connection_information/2`.\n\n  Own Id: OTP-18643 Aux Id: ERIERL-932","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Add RFC-6083 considerations for DTLS to enable gen_sctp based callback for the\n  transport.\n\n  Own Id: OTP-18618 Aux Id: ERIERL-932","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 10.7.3.7 - SSL Release Notes","ref":"notes.html#ssl-10-7-3-7"},{"type":"extras","doc":"- Client signature algorithm list input order is now honored again , it was\n  accidently reversed by a previous fix.\n\n  Own Id: OTP-18550","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.7.3.6 - SSL Release Notes","ref":"notes.html#ssl-10-7-3-6"},{"type":"extras","doc":"- Maximize compatibility by ignoring change_cipher_spec during handshake even if\n  middle_box_mode is not negotiated (mandated by client)\n\n  Own Id: OTP-18433 Aux Id: GH-6772\n\n- Move assert of middlebox message after an hello_retry_request to maximize\n  interoperability. Does not changes semantics of the protocol only allows\n  unexpected message delay from server.\n\n  Own Id: OTP-18467 Aux Id: GH-6807","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 10.7.3.5 - SSL Release Notes","ref":"notes.html#ssl-10-7-3-5"},{"type":"extras","doc":"- Fixes handling of symlinks in cacertfile option.\n\n  Own Id: OTP-18266 Aux Id: GH-6328","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.7.3.4 - SSL Release Notes","ref":"notes.html#ssl-10-7-3-4"},{"type":"extras","doc":"- With this change, value of cacertfile option will be adjusted before loading\n  certs from the file. Adjustments include converting relative paths to absolute\n  and converting symlinks to actual file path.\n\n  Thanks to Marcus Johansson\n\n  Own Id: OTP-18099 Aux Id: PR-6287\n\n- Incorrect handling of client middlebox negotiation for TLS-1.3 could result in\n  that a TLS-1.3 server would not use middlebox mode although the client was\n  expecting it too and failing the negotiation with unexpected message.\n\n  Own Id: OTP-18219 Aux Id: GH-6241, PR-6249\n\n- If the \"User\" process, the process starting the TLS connection, gets killed in\n  the middle of spawning the dynamic connection tree make sure we do not leave\n  any processes behind.\n\n  Own Id: OTP-18233 Aux Id: GH-6244, PR-6270","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.7.3.3 - SSL Release Notes","ref":"notes.html#ssl-10-7-3-3"},{"type":"extras","doc":"- Reject unexpected application data in all relevant places for all TLS\n  versions. Also, handle TLS-1.3 middlebox compatibility with more care. This\n  will make malicious connections fail early and further, mitigate possible DoS\n  attacks, that would be caught by the handshake timeout.\n\n  Thanks to Aina Toky Rasoamanana and Olivier Levillain from Télécom SudParis\n  for alerting us of the issues in our implementation.\n\n  Own Id: OTP-18044\n\n- The link to crypto:engine_load refered the function with wrong arity.\n\n  Own Id: OTP-18173\n\n- Make sure periodical refresh of CA certificate files repopulates cache\n  properly.\n\n  Own Id: OTP-18195","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.7.3.2 - SSL Release Notes","ref":"notes.html#ssl-10-7-3-2"},{"type":"extras","doc":"- Improved handling of unexpected messages during the handshake, taking the\n  right action for unexpected messages.\n\n  Own Id: OTP-18145","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.7.3.1 - SSL Release Notes","ref":"notes.html#ssl-10-7-3-1"},{"type":"extras","doc":"- When a TLS-1.3 enabled client tried to talk to a TLS-1.2 server that coalesces\n  TLS-1.2 handshake message over one TLS record, the connection could fail due\n  to some message being handled in the wrong state, this has been fixed.\n\n  Own Id: OTP-18087 Aux Id: GH-5961\n\n- Fixed tls-1.3 session ticket lifetime which was discarded to quickly before.\n\n  Own Id: OTP-18092 Aux Id: PR-5959\n\n- Correctly handles supported protocol version change from default to something\n  else by sni_fun supplied to ssl:handshake/\\[2,3] together with a TCP-socket\n  (so called upgrade).\n\n  Own Id: OTP-18100 Aux Id: GH-5985\n\n- Also, TLS-1.3 should respond with a protocol version alert if previous\n  versions, that are supported but not configured, are attempted.\n\n  Own Id: OTP-18129 Aux Id: GH-5950","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Enhance handling of handshake decoding errors, especially for certificate\n  authorities extension to ensure graceful termination.\n\n  Own Id: OTP-18085","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 10.7.3 - SSL Release Notes","ref":"notes.html#ssl-10-7-3"},{"type":"extras","doc":"- Client certification could fail if TLS-1.3 enabled client negotiated TLS-1.2\n  connection with the server, this is due to the wrong version being used when\n  decoding the certificate request message from the server.\n\n  Own Id: OTP-18028 Aux Id: GH-5835\n\n- socket option packet_size was not handled in ssl:setops/2 and ssl:getotps/2\n\n  Own Id: OTP-18062 Aux Id: GH-5898\n\n- Remove legacy code to fix interoperability with new socket inet_backend.\n\n  Own Id: OTP-18071 Aux Id: GH-5930","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.7.2 - SSL Release Notes","ref":"notes.html#ssl-10-7-2"},{"type":"extras","doc":"- With this change, potential hanging of pre TLS1.3 client receiving OSCP staple\n  message is avoided.\n\n  Own Id: OTP-17994","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.7.1 - SSL Release Notes","ref":"notes.html#ssl-10-7-1"},{"type":"extras","doc":"- Client certification could fail for TLS-1.3 servers that did not include the\n  certificate_authorities extension in its certificate request message.\n\n  Own Id: OTP-17971 Aux Id: GH-5783","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.7 - SSL Release Notes","ref":"notes.html#ssl-10-7"},{"type":"extras","doc":"- Improved error handling.\n\n  Own Id: OTP-17759 Aux Id: GH-5367\n\n- Before this change, net_kernel used with TLS distribution might be leaking\n  processes in case of connectivity issues.\n\n  Own Id: OTP-17815 Aux Id: GH-5332\n\n- Fix makefile dependency bugs.\n\n  Own Id: OTP-17847 Aux Id: PR-5574 GH-5548\n\n- Make sure the TLS sender process handles explicit calls to\n  erlang:disconnect_node properly, avoiding potential hanging problems in\n  net_kernel.\n\n  Own Id: OTP-17929 Aux Id: GH-5708","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Add support for TLS-1.3 certificate_authorities extension. And process\n  certificate_authorities field in pre-TLS-1.3 certificate requests.\n\n  Own Id: OTP-15719\n\n- Support password fun for protected keyfiles in ssl:connect function.\n\n  Own Id: OTP-17816 Aux Id: PR-5607\n\n- Add in some cases earlier detection of possible DoS attacks by malicious\n  clients sending unexpected TLS messages instead of the client hello. Note that\n  such attacks are already mitigated by providing a timeout for the TLS\n  handshake.\n\n  Own Id: OTP-17903","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 10.6.1 - SSL Release Notes","ref":"notes.html#ssl-10-6-1"},{"type":"extras","doc":"- Improve SNI (server name indication) handling so that protocol version can be\n  selected with regards to SNI. Also, make sure that\n  ssl:connection_information/1 returns the correct SNI value.\n\n  Own Id: OTP-17794 Aux Id: GH-5341, GH-4450\n\n- Fixed cipher suite listing functions so that the listing of all cipher suites\n  will be complete. Another fix for cipher suite handling in OTP-24.1\n  accidentally excludes a few cipher suites from the listing of all cipher\n  suites.\n\n  Own Id: OTP-17829 Aux Id: ERIERL-708\n\n- Reenable legacy cipher suite TLS_RSA_WITH_3DES_EDE_CBC_SHA for explicit\n  configuration in TLS-1.2, not supported by default.\n\n  Own Id: OTP-17879 Aux Id: GH-5624","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Avoid unnecessary logs by better adjusting the tls_sender process to the new\n  supervisor structure in OTP-24.2\n\n  Own Id: OTP-17831","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 10.6 - SSL Release Notes","ref":"notes.html#ssl-10-6"},{"type":"extras","doc":"- Allow re-connect on DTLS sockets\n\n  Can happen when a computer reboots and connects from the same client port\n  without the server noticing should be allowed according to RFC.\n\n  Own Id: OTP-17411 Aux Id: ERL-1203, GH-4393\n\n- Fix tls and non-tls distribution to use erl_epmd:address_please to figure out\n  if IPv4 or IPv6 addresses should be used when connecting to the remote node.\n\n  Before this fix, a dns lookup of the remote node hostname determined which IP\n  version was to be used which meant that the hostname had to resolve to a valid\n  ip address.\n\n  Own Id: OTP-17809 Aux Id: PR-5337 GH-5334","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Use supervisor significant child to manage tls connection process and tls\n  sender process dependency.\n\n  Own Id: OTP-17417\n\n- Random generation adjustment for TLS1.3\n\n  Own Id: OTP-17699\n\n- Allow any \\{03,XX\\} TLS record version in the client hello for maximum\n  interoperability\n\n  Own Id: OTP-17761 Aux Id: GH-5380","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 10.5.3 - SSL Release Notes","ref":"notes.html#ssl-10-5-3"},{"type":"extras","doc":"- Correct typo of ECC curve name in signature algorithm handling. Will make the\n  signature algorithm ecdsa_secp521r1_sha512 succeed.\n\n  Own Id: OTP-17756 Aux Id: GH-5383, PR-5397\n\n- Suppress authenticity warning when option verify_none is explicitly supplied.\n\n  Own Id: OTP-17757 Aux Id: GH-5352, PR-5395","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.5.2 - SSL Release Notes","ref":"notes.html#ssl-10-5-2"},{"type":"extras","doc":"- Fix TLS-1.2 RSA-PSS negotiation and also fix broken certificate request\n  message for pre-TLS-1.3 servers.\n\n  Own Id: OTP-17688 Aux Id: GH-5255\n\n- Fix CRL issuer verification that under some circumstances could fail with a\n  function_clause error.\n\n  Own Id: OTP-17723 Aux Id: GH-5300","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.5.1 - SSL Release Notes","ref":"notes.html#ssl-10-5-1"},{"type":"extras","doc":"- Before that change, TLS downgrade could occasionally fail when data intended\n  for downgraded socket were delivered together with CLOSE_NOTIFY alert to ssl\n  app.\n\n  Own Id: OTP-17393\n\n- Avoid re-encoding of decoded certificates. This could cause unexpected\n  failures as some subtle encoding errors can be tolerated when decoding but\n  hence creating another sequence of bytes if the decoded value is re-encoded.\n\n  Own Id: OTP-17657\n\n- Fix possible process leak when the process doing ssl:transport_accept dies\n  before initiating the TLS handshake.\n\n  Own Id: OTP-17666 Aux Id: GH-5239\n\n- Fix dtls memory leak, the replay window code was broken.\n\n  Own Id: OTP-17670 Aux Id: GH-5224","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.5 - SSL Release Notes","ref":"notes.html#ssl-10-5"},{"type":"extras","doc":"- Fix Makefile dependency generation to work no matter what the `ERL_TOP` folder\n  is called.\n\n  Own Id: OTP-17423 Aux Id: GH-4823 PR-4829\n\n- If trying to downgrade a TLS-1.3 connection to a plain TCP connection,\n  possible TLS-1.3 session ticket messages will be ignored in the \"downgrade\"\n  state while waiting for the close notify alert.\n\n  Own Id: OTP-17517 Aux Id: GH-5009\n\n- Corrected error handling to correctly generate an insufficient security alert\n  when there are no suitable groups that can be negotiated in TLS-1.3 instead of\n  crashing resulting in an internal error alert.\n\n  Own Id: OTP-17521\n\n- Properly handle default session data storage.\n\n  When a client tries to reuse an expired session the default server storage\n  handling would crash losing other session data. This would cause a error\n  report and possible loss of abbreviated handshakes.\n\n  Own Id: OTP-17635 Aux Id: GH-5192","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Add support for RSA-PSS-PSS signatures and signature_algorithms_cert in\n  TLS-1.2. This is a TLS-1.3 RFC requirement to backport this functionality.\n\n  Own Id: OTP-16590 Aux Id: ERL-625, GH-5029\n\n- Use inet:monitor/1 to monitor listen-sockets so that we are compatible with\n  the new socket backend for gen_tcp.\n\n  Own Id: OTP-17392 Aux Id: PR-5050\n\n- Enhance ssl:prf/4 handling and testing\n\n  Own Id: OTP-17464\n\n- Enhanced cipher suite filtering functionality, making sure TLS-1.3 and TLS-1.2\n  cipher suites can be supported correctly together even when TLS-1.2 anonymous\n  ciphers are included.\n\n  Own Id: OTP-17501 Aux Id: GH-4978\n\n- Enhance gracefulness especially in TLS-1.3\n\n  Own Id: OTP-17530","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 10.4.2 - SSL Release Notes","ref":"notes.html#ssl-10-4-2"},{"type":"extras","doc":"- Handle cross-signed root certificates when old root expired as reported in\n  GH-4877.\n\n  Own Id: OTP-17475 Aux Id: GH-4877\n\n- The signature selection algorithm has been changed to also verify if the\n  client supports signatures using the elliptic curve of the server's\n  public/private key pair. This change fixes #4958.\n\n  Own Id: OTP-17529 Aux Id: PR-4979, GH-4958","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Slight optimization of certificate decoding.\n\n  Own Id: OTP-17150 Aux Id: GH-4877","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 10.4.1 - SSL Release Notes","ref":"notes.html#ssl-10-4-1"},{"type":"extras","doc":"- Fix cache invalidation problem for CA certs provided by the cacertfile option.\n\n  Own Id: OTP-17435 Aux Id: ERIERL-653","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.4 - SSL Release Notes","ref":"notes.html#ssl-10-4"},{"type":"extras","doc":"- Missing runtime dependencies has been added to this application.\n\n  Own Id: OTP-17243 Aux Id: PR-4557\n\n- TLS handshake should fail if OCSP staple is requested but missing. Note that\n  OCSP support is still considered experimental and only partially implemented.\n\n  Own Id: OTP-17343","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Removed ssl:ssl_accept/1,2,3 and ssl:cipher:suites/0,1 use ssl:handshake/1,2,3\n  and ssl:cipher_suites/2,3 instead.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-16974\n\n- Make TLS handshakes in Erlang distribution concurrent.\n\n  Own Id: OTP-17044 Aux Id: PR-2654\n\n- Randomize internal `{active,n}` optimization when running Erlang distribution\n  over TLS to spread RAM/CPU spike that may occur when starting up a big\n  cluster.\n\n  Own Id: OTP-17117 Aux Id: PR-2933\n\n- TLS connections now support EdDSA certificates.\n\n  Own Id: OTP-17142 Aux Id: PR-4756, GH-4637, GH-4650\n\n- Enhance documentation and logging of certificate handling.\n\n  Own Id: OTP-17384 Aux Id: GH-4800","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 10.3.1.5 - SSL Release Notes","ref":"notes.html#ssl-10-3-1-5"},{"type":"extras","doc":"- Correct corner case of unexpected message handling for pre TLS-1.3 versions,\n  could cause \"late failure\" and make the server dependent on its handshake\n  timeout to prevent possible DoS attacks.\n\n  Own Id: OTP-18224","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.3.1.4 - SSL Release Notes","ref":"notes.html#ssl-10-3-1-4"},{"type":"extras","doc":"- The link to crypto:engine_load refered the function with wrong arity.\n\n  Own Id: OTP-18173","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.3.1.3 - SSL Release Notes","ref":"notes.html#ssl-10-3-1-3"},{"type":"extras","doc":"- Improved handling of unexpected messages during the handshake, taking the\n  right action for unexpected messages.\n\n  Own Id: OTP-18145","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.3.1.2 - SSL Release Notes","ref":"notes.html#ssl-10-3-1-2"},{"type":"extras","doc":"- Handle cross-signed root certificates when old root expired as reported in\n  GH-4877.\n\n  Own Id: OTP-17475 Aux Id: GH-4877\n\n- The signature selection algorithm has been changed to also verify if the\n  client supports signatures using the elliptic curve of the server's\n  public/private key pair. This change fixes #4958.\n\n  Own Id: OTP-17529 Aux Id: PR-4979, GH-4958","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Slight optimization of certificate decoding.\n\n  Own Id: OTP-17150 Aux Id: GH-4877","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 10.3.1.1 - SSL Release Notes","ref":"notes.html#ssl-10-3-1-1"},{"type":"extras","doc":"- Fix cache invalidation problem for CA certs provided by the cacertfile option.\n\n  Own Id: OTP-17435 Aux Id: ERIERL-653","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.3.1 - SSL Release Notes","ref":"notes.html#ssl-10-3-1"},{"type":"extras","doc":"- Retain backwards compatible behavior of verify_fun when handling incomplete\n  chains that are not verifiable.\n\n  Own Id: OTP-17296 Aux Id: GH-4682\n\n- Avoid server session handler crash, this will increase session ruse\n  opportunities.\n\n  Own Id: OTP-17348 Aux Id: ERIERL-641","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.3 - SSL Release Notes","ref":"notes.html#ssl-10-3"},{"type":"extras","doc":"- Fix CRL handling that previously could fail to find the issuer cert under some\n  circumstances.\n\n  Own Id: OTP-17261 Aux Id: GH-4589\n\n- TLS-1.3 client could, under some circumstances, select an incorrect algorithm\n  to sign the certificate verification message causing a TLS Decrypt Alert being\n  issued by the server.\n\n  Own Id: OTP-17281 Aux Id: GH-4620\n\n- Correct handling of default values for emulated socket options and retain the\n  order of the ssl options list to ensure backwards compatible behavior if\n  options should be set more than once.\n\n  Own Id: OTP-17282","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Enhance pre TLS-1.3 session handling so the client and server side handling is\n  completely separated and client disregards oldest session when reaching max\n  limit of the session table.\n\n  Own Id: OTP-16876\n\n- This change implements the early data feature for TLS 1.3 clients.\n\n  TLS 1.3 allows clients to send data in the first flight using a Pre-Shared Key\n  to authenticate the server and to encrypt the early data.\n\n  Own Id: OTP-16985\n\n- This change implements the early data feature for TLS 1.3 servers.\n\n  Own Id: OTP-17042","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 10.2.4.4 - SSL Release Notes","ref":"notes.html#ssl-10-2-4-4"},{"type":"extras","doc":"- Improved handling of unexpected messages during the handshake, taking the\n  right action for unexpected messages.\n\n  Own Id: OTP-18145","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.2.4.3 - SSL Release Notes","ref":"notes.html#ssl-10-2-4-3"},{"type":"extras","doc":"- Fix cache invalidation problem for CA certs provided by the cacertfile option.\n\n  Own Id: OTP-17435 Aux Id: ERIERL-653","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.2.4.2 - SSL Release Notes","ref":"notes.html#ssl-10-2-4-2"},{"type":"extras","doc":"- Fix handling of emulated socket options, the previous patch was incomplete,\n\n  Own Id: OTP-17305","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.2.4.1 - SSL Release Notes","ref":"notes.html#ssl-10-2-4-1"},{"type":"extras","doc":"- Backport of OTP-17282\n\n  Correct handling of default values for emulated socket options and retain the\n  order of the ssl options list to ensure backwards compatible behavior if\n  options should be set more than once.\n\n  Own Id: OTP-17289 Aux Id: GH-4585","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.2.4 - SSL Release Notes","ref":"notes.html#ssl-10-2-4"},{"type":"extras","doc":"- Enhance logging option log_level to support none and all, also restore\n  backwards compatibility for log_alert option.\n\n  Own Id: OTP-17228 Aux Id: ERIERL-614","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.2.3 - SSL Release Notes","ref":"notes.html#ssl-10-2-3"},{"type":"extras","doc":"- Avoid race when the first two upgrade server handshakes (that is servers that\n  use a gen_tcp socket as input to ssl:handshake/2,3) start close to each other.\n  Could lead to that one of the handshakes would fail.\n\n  Own Id: OTP-17190 Aux Id: ERIERL-606","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 10.2.2 - SSL Release Notes","ref":"notes.html#ssl-10-2-2"},{"type":"extras","doc":"- Avoid that upgrade (from TCP to TLS) servers starts multiple session cache\n  handlers for the same server. This applies to Erlang distribution over TLS\n  servers.\n\n  Own Id: OTP-17139 Aux Id: ERL-1458, OTP-16239\n\n- Legacy cipher suites defined before TLS-1.2 (but still supported) should be\n  possible to use in TLS-1.2. They where accidentally excluded for available\n  cipher suites for TLS-1.2 in OTP-23.2.2.\n\n  Own Id: OTP-17174 Aux Id: ERIERL-597","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Enable Erlang distribution over TLS to run TLS-1.3, although TLS-1.2 will\n  still be default.\n\n  Own Id: OTP-16239 Aux Id: ERL-1458, OTP-17139","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 10.2.1 - SSL Release Notes","ref":"notes.html#ssl-10-2-1"},{"type":"extras","doc":"- Fix CVE-2020-35733 this only affects ssl-10.2 (OTP-23.2). This vulnerability\n  could enable a man in the middle attack using a fake chain to a known trusted\n  ROOT. Also limits alternative chain handling, for handling of possibly\n  extraneous certs, to improve memory management.\n\n  Own Id: OTP-17098","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Add support for AES CCM based cipher suites defined in RFC 7251\n\n  Also Correct cipher suite name conversion to OpenSSL names. A few names where\n  corrected earlier in OTP-16267 For backwards compatible reasons we support\n  usage of openSSL names for cipher suites. Mostly anonymous suites names where\n  incorrect, but also some legacy suites.\n\n  Own Id: OTP-17100","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 10.2 - SSL Release Notes","ref":"notes.html#ssl-10-2"},{"type":"extras","doc":"- SSL's Erlang Distribution Protocol modules inet_tls_dist and inet6_tls_dist\n  lacked a callback function, so the start flag \"-dist_listen false\" did not\n  work, which has now been fixed.\n\n  Own Id: OTP-15126 Aux Id: ERL-1375\n\n- Correct OpenSSL names for newer cipher suites using DHE in their name that\n  accidentally got the wrong value when fixing other older names using EDH\n  instead.\n\n  Own Id: OTP-16267 Aux Id: ERIERL-571, ERIERL-477\n\n- This change improves the handling of DTLS listening dockets, making it\n  possible to open multiple listeners on the same port with different IP\n  addresses.\n\n  Own Id: OTP-16849 Aux Id: ERL-1339\n\n- Fix a bug that causes cross-build failure.\n\n  This change excludes the ssl.d dependency file from the source tarballs.\n\n  Own Id: OTP-16921\n\n- This change fixes ssl:peername/1 when called on a DTLS client socket.\n\n  Own Id: OTP-16923 Aux Id: ERL-1341, PR-2786\n\n- Retain emulation of active once on a closed socket to behave as before 23.1\n\n  Own Id: OTP-17018 Aux Id: ERL-1409\n\n- Corrected server session cache entry deletion pre TLS-1.3. May increase\n  session reuse.\n\n  Own Id: OTP-17019 Aux Id: ERL-1412","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Handle extraneous certs in certificate chains as well as chains that are\n  incomplete but can be reconstructed or unordered chains. The cert and certfile\n  options will now accept a list of certificates so that the user may specify\n  the chain explicitly.\n\n  Also, the default value of the depth option has been increased to allow longer\n  chains by default.\n\n  Own Id: OTP-16277\n\n- This change implements optional NSS-style keylog in\n  ssl:connection_information/2 for debugging purposes.\n\n  The keylog contains various TLS secrets that can be loaded in Wireshark to\n  decrypt TLS packets.\n\n  Own Id: OTP-16445 Aux Id: PR-2823\n\n- Use new gen_statem feature of changing callback mode to improve code\n  maintainability.\n\n  Own Id: OTP-16529\n\n- The handling of Service Name Indication has been aligned with RFC8446.\n\n  Own Id: OTP-16762\n\n- Add explicit session reuse option to TLS clients for pre TLS-1.3 sessions.\n  Also, add documentation to Users Guide for such sessions.\n\n  Own Id: OTP-16893","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 10.1 - SSL Release Notes","ref":"notes.html#ssl-10-1"},{"type":"extras","doc":"- If a passive socket is created, ssl:recv/2,3 is never called and then the peer\n  closes the socket the controlling process will no longer receive an active\n  close message.\n\n  Own Id: OTP-16697 Aux Id: ERIERL-496\n\n- Data deliver with ssl:recv/2,3 could fail for when using packet mode. This has\n  been fixed by correcting the flow control handling of passive sockets when\n  packet mode is used.\n\n  Own Id: OTP-16764\n\n- This change fixes a potential man-in-the-middle vulnerability when the ssl\n  client is configured to automatically handle session tickets\n  (\\{session_tickets, auto\\}).\n\n  Own Id: OTP-16765\n\n- Fix the internal handling of options 'verify' and 'verify_fun'.\n\n  This change fixes a vulnerability when setting the ssl option 'verify' to\n  verify_peer in a continued handshake won't take any effect resulting in the\n  acceptance of expired peer certificates.\n\n  Own Id: OTP-16767 Aux Id: ERIERL-512\n\n- This change fixes the handling of stateless session tickets when anti-replay\n  is enabled.\n\n  Own Id: OTP-16776 Aux Id: ERL-1316\n\n- Fix a crash due to the faulty handling of stateful session tickets received by\n  servers expecting stateless session tickets.\n\n  This change also improves the handling of faulty/invalid tickets.\n\n  Own Id: OTP-16777 Aux Id: ERL-1317\n\n- Correct flow ctrl checks from OTP-16764 to work as intended. Probably will not\n  have a noticeable affect but will make connections more well behaved under\n  some circumstances.\n\n  Own Id: OTP-16837 Aux Id: ERL-1319, OTP-16764\n\n- Distribution over TLS could exhibit livelock-like behaviour when there is a\n  constant stream of distribution messages. Distribution data is now chunked\n  every 16 Mb to avoid that.\n\n  Own Id: OTP-16851 Aux Id: PR-2703","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Implement the cookie extension for TLS 1.3.\n\n  Own Id: OTP-15855\n\n- Experimental OCSP client support.\n\n  Own Id: OTP-16448\n\n- TLS 1.0 -TLS-1.2 sessions tables now have a absolute max value instead of\n  using a shrinking mechanism when reaching the limit. To avoid out of memory\n  problems under heavy load situations. Note that this change infers that\n  implementations of ssl_session_cache_api needs to implement the size function\n  (introduce in OTP 19) for session reuse to be optimally utilized.\n\n  Own Id: OTP-16802 Aux Id: ERIERL-516","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 10.0 - SSL Release Notes","ref":"notes.html#ssl-10-0"},{"type":"extras","doc":"- Fix a bug that causes cross-build failure.\n\n  This change excludes the ssl.d dependency file from the source tar balls.\n\n  Own Id: OTP-16562 Aux Id: ERL-1168\n\n- Correct translation of OpenSSL legacy names for two legacy cipher suites\n\n  Own Id: OTP-16573 Aux Id: ERIERL-477\n\n- Correct documentation for PSK identity and SRP username.\n\n  Own Id: OTP-16585\n\n- Make sure client hostname check is run when client uses its own verify_fun\n\n  Own Id: OTP-16626 Aux Id: ERL-1232\n\n- Improved signature selection mechanism in TLS 1.3 for increased\n  interoperability.\n\n  Own Id: OTP-16638 Aux Id: ERL-1206","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Drop support for SSL-3.0. Support for this legacy TLS version has not been\n  enabled by default since OTP 19. Now all code to support it has been removed,\n  that is SSL-3.0 protocol version can not be used and is considered invalid.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-14790\n\n- Added support for RSA-PSS signature schemes\n\n  Own Id: OTP-15247\n\n- Improve interoperability by implementing the middlebox compatibility mode.\n\n  The middlebox compatibility mode makes the TLS 1.3 handshake look more like a\n  TLS 1.2 handshake and increases the chance of successfully establishing TLS\n  1.3 connections through legacy middleboxes.\n\n  Own Id: OTP-15589\n\n- Utilize new properties of\n  [`erlang:dist_ctrl_get_data()`](`erlang:dist_ctrl_get_data/1`) for performance\n  improvement of Erlang distribution over TLS.\n\n  Own Id: OTP-16127 Aux Id: OTP-15618\n\n- Calls of deprecated functions in the\n  [Old Crypto API](`e:crypto:new_api.md#the-old-api`) are replaced by calls of\n  their [substitutions](`e:crypto:new_api.md#the-new-api`).\n\n  Own Id: OTP-16346\n\n- Implement cipher suite TLS_AES_128_CCM_8_SHA256.\n\n  Own Id: OTP-16391\n\n- This change adds TLS-1.3 to the list of default supported versions. That is,\n  TLS-1.3 and TLS-1.2 are configured when ssl option 'versions' is not\n  explicitly set.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-16400\n\n- Refactored the internal handling of deprecated and removed functions.\n\n  Own Id: OTP-16469\n\n- Extended ssl:versions so that it lists supported, available and implemented\n  TLS/DTLS versions.\n\n  Own Id: OTP-16519\n\n- Added new option exclusive for ssl:cipher_suites/2,3\n\n  Own Id: OTP-16532\n\n- Avoid DoS attack against stateful session_tickets by making session ticket ids\n  unpredictable.\n\n  Own Id: OTP-16533\n\n- Add support for the max_fragment_length extension (RFC 6066).\n\n  Own Id: OTP-16547 Aux Id: PR-2547\n\n- Add srp_username in ssl:connection_info, update the document with types of\n  this function.\n\n  Own Id: OTP-16584","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 9.6.2.3 - SSL Release Notes","ref":"notes.html#ssl-9-6-2-3"},{"type":"extras","doc":"- Correct flow ctrl checks from OTP-16764 to work as intended. Probably will not\n  have a noticeable affect but will make connections more well behaved under\n  some circumstances.\n\n  Own Id: OTP-16837 Aux Id: ERL-1319, OTP-16764\n\n- Fix a bug that causes cross-build failure.\n\n  This change excludes the ssl.d dependency file from the source tar balls.\n\n  Own Id: OTP-16921","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 9.6.2.2 - SSL Release Notes","ref":"notes.html#ssl-9-6-2-2"},{"type":"extras","doc":"- Data deliver with ssl:recv/2,3 could fail for when using packet mode. This has\n  been fixed by correcting the flow control handling of passive sockets when\n  packet mode is used.\n\n  Own Id: OTP-16764\n\n- Fix the internal handling of options 'verify' and 'verify_fun'.\n\n  This change fixes a vulnerability when setting the ssl option 'verify' to\n  verify_peer in a continued handshake won't take any effect resulting in the\n  acceptance of expired peer certificates.\n\n  Own Id: OTP-16767 Aux Id: ERIERL-512","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 9.6.2.1 - SSL Release Notes","ref":"notes.html#ssl-9-6-2-1"},{"type":"extras","doc":"- If a passive socket is created, ssl:recv/2,3 is never called and then the peer\n  closes the socket the controlling process will no longer receive an active\n  close message.\n\n  Own Id: OTP-16697 Aux Id: ERIERL-496","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 9.6.2 - SSL Release Notes","ref":"notes.html#ssl-9-6-2"},{"type":"extras","doc":"- Fix timing bug that could cause ssl sockets to become unresponsive after an\n  ssl:recv/3 call timed out\n\n  Own Id: OTP-16619 Aux Id: ERL-1213","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 9.6.1 - SSL Release Notes","ref":"notes.html#ssl-9-6-1"},{"type":"extras","doc":"- Correct error handling when the partial_chain fun claims a certificate to be\n  the trusted cert that is not part of the chain. This bug would hide the\n  appropriate alert generating an \"INTERNAL_ERROR\" alert instead.\n\n  Own Id: OTP-16567 Aux Id: ERIERL-481","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 9.6 - SSL Release Notes","ref":"notes.html#ssl-9-6"},{"type":"extras","doc":"- Correct handling of TLS record limit in TLS-1.3. The max value differs from\n  previous versions. Also the payload data max record check was broken, that is\n  record overflow problems could occur if user sent large amounts of data.\n\n  Own Id: OTP-16258\n\n- Correct close handling for DTLS\n\n  Own Id: OTP-16348 Aux Id: ERL-1110\n\n- Fix ssl:getstat/1-2 to also work for DTLS sockets\n\n  Own Id: OTP-16352 Aux Id: ERL-1099\n\n- Correct internal handling och socket active mode to avoid reviving TCP data\n  aimed for a downgraded TLS socket.\n\n  Own Id: OTP-16425\n\n- When using the host name as fallback for SNI (server name indication) strip a\n  possible trailing dot that is allowed in a host name but not in the SNI. Also\n  if the server receives a SNI with a trailing dot send an UNRECOGNIZED_NAME\n  alert.\n\n  Own Id: OTP-16437 Aux Id: ERL-1135\n\n- Immediately remove session entries if handshake is abruptly closed at\n  transport level.\n\n  Own Id: OTP-16479","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Implementation of the key and initialization vector update feature, and\n  general hardening of TLS 1.3.\n\n  There are cryptographic limits on the amount of plaintext which can be safely\n  encrypted under a given set of keys.\n\n  This change enforces those limits by triggering automatic key updates on TLS\n  1.3 connections.\n\n  Own Id: OTP-15856\n\n- Add support for TLS 1.3 Session Tickets (stateful and stateless). This allows\n  session resumption using keying material from a previous successful handshake.\n\n  Own Id: OTP-16253\n\n- Add support for key exchange with Edward curves and PSS-RSA padding in\n  signature verification.\n\n  Own Id: OTP-16528","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 9.5.3 - SSL Release Notes","ref":"notes.html#ssl-9-5-3"},{"type":"extras","doc":"- Enhance error handling, all ALERTS shall be handled gracefully and not cause a\n  crash.\n\n  Own Id: OTP-16413 Aux Id: ERL-1136\n\n- Enhance alert logging, in some places the role indication of the alert origin\n  was missing. So the log would say undefined instead of client or server.\n\n  Own Id: OTP-16424\n\n- Two different optimizations did not work together and resulted in the possible\n  breakage of connections using stream ciphers (that is RC4). Reworked the\n  implementation to avoid this.\n\n  Own Id: OTP-16426 Aux Id: ERL-1136","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 9.5.2 - SSL Release Notes","ref":"notes.html#ssl-9-5-2"},{"type":"extras","doc":"- Fix the handling of GREASE values sent by web browsers when establishing TLS\n  1.3 connections. This change improves handling of GREASE values in various\n  protocol elements sent in a TLS 1.3 ClientHello.\n\n  Own Id: OTP-16388 Aux Id: ERL-1130\n\n- Correct DTLS listen emulation, could cause problems with opening a new DTLS\n  listen socket for a port previously used by a now closed DTLS listen socket.\n\n  Own Id: OTP-16396 Aux Id: ERL-1118","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 9.5.1 - SSL Release Notes","ref":"notes.html#ssl-9-5-1"},{"type":"extras","doc":"- Add missing alert handling clause for TLS record handling. Could sometimes\n  cause confusing error behaviors of TLS connections.\n\n  Own Id: OTP-16357 Aux Id: ERL-1166\n\n- Fix handling of ssl:recv that happens during a renegotiation. Using the\n  passive receive function ssl:recv/\\[2,3] during a renegotiation would fail the\n  connection with unexpected msg.\n\n  Own Id: OTP-16361","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 9.5 - SSL Release Notes","ref":"notes.html#ssl-9-5"},{"type":"extras","doc":"- Corrected CRL handling which could cause CRL verification to fail. This could\n  happen when the CRL distribution point explicitly specifies the CRL issuer,\n  that is not using the fallback.\n\n  Own Id: OTP-16156 Aux Id: ERL-1030\n\n- Correct handling of unordered chains so that it works as expected\n\n  Own Id: OTP-16293\n\n- Fix bug causing ssl application to crash when handshake is paused and\n  ClientHello contains extensions for session resumption\n  (psk_key_exchange_modes, pre_shared_key).\n\n  Own Id: OTP-16295 Aux Id: ERL-1095\n\n- Fix connectivity problems with legacy servers when client is configured to\n  support a range of protocol versions including TLS 1.3.\n\n  Own Id: OTP-16303","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Improve session handling for TLS-1.3 compatibility mode and cleaner internal\n  handling so that removal of old session data can be more efficient, hopefully\n  mitigating problems with big session tables during heavy load.\n\n  Own Id: OTP-15524 Aux Id: OTP-15352\n\n- Correct handling of DTLS listen socket emulation. Could cause failure to\n  create new listen socket after process that owned previous listen socket died.\n\n  Own Id: OTP-15809 Aux Id: ERL-917\n\n- Add detailed info in ALERT description when client does not send a requested\n  cert.\n\n  Own Id: OTP-16266","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 9.4 - SSL Release Notes","ref":"notes.html#ssl-9-4"},{"type":"extras","doc":"- Handling of zero size fragments in TLS could cause an infinite loop. This has\n  now been corrected.\n\n  Own Id: OTP-15328 Aux Id: ERIERL-379\n\n- DTLS record check needs to consider that a resent hello message can have a\n  different version than the negotiated.\n\n  Own Id: OTP-15807 Aux Id: ERL-920","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Basic support for TLS 1.3 Client for experimental use. For more information\n  see the Standards Compliance chapter of the User's Guide.\n\n  Own Id: OTP-15431\n\n- Correct solution for retaining tcp flow control OTP-15802 (ERL-934) as to not\n  break ssl:recv as reported in (ERL-938)\n\n  Own Id: OTP-15823 Aux Id: ERL-934, ERL-938\n\n- Enhance dialyzer specs to reflect implementation better and avoid dialyzer\n  warnings for the user that wants to use TLS with unix domain sockets.\n\n  Own Id: OTP-15851 Aux Id: PR-2235\n\n- Add support for ECDSA signature algorithms in TLS 1.3.\n\n  Own Id: OTP-15854\n\n- Correct error handling of TLS downgrade, possible return values form\n  ssl:close/2 when downgrading is \\{ok, Port\\} or \\{error, Reason\\}, it could\n  happen that only ok was returned instead of \\{error, closed\\} when downgrade\n  failed due to that the peer closed the TCP connection.\n\n  Own Id: OTP-16027","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 9.3.5 - SSL Release Notes","ref":"notes.html#ssl-9-3-5"},{"type":"extras","doc":"- Enhance error handling for erroneous alerts from the peer.\n\n  Own Id: OTP-15943","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 9.3.4 - SSL Release Notes","ref":"notes.html#ssl-9-3-4"},{"type":"extras","doc":"- Fix handling of certificate decoding problems in TLS 1.3 similarly as in TLS\n  1.2.\n\n  Own Id: OTP-15900\n\n- Hibernation now works as expected in all cases, was accidentally broken by\n  optimization efforts.\n\n  Own Id: OTP-15910\n\n- Fix interoperability problems with openssl when the TLS 1.3 server is\n  configured with the option signature_algs_cert.\n\n  Own Id: OTP-15913","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 9.3.3 - SSL Release Notes","ref":"notes.html#ssl-9-3-3"},{"type":"extras","doc":"- Correct handshake handling, might cause strange symptoms such as ASN.1\n  certificate decoding issues.\n\n  Own Id: OTP-15879 Aux Id: ERL-968\n\n- Fix handling of the signature_algorithms_cert extension in the ClientHello\n  handshake message.\n\n  Own Id: OTP-15887 Aux Id: ERL-973\n\n- Handle new ClientHello extensions when handshake is paused by the \\{handshake,\n  hello\\} ssl option.\n\n  Own Id: OTP-15888 Aux Id: ERL-975","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 9.3.2 - SSL Release Notes","ref":"notes.html#ssl-9-3-2"},{"type":"extras","doc":"- Returned \"alert error string\" is now same as logged alert string\n\n  Own Id: OTP-15844\n\n- Fix returned extension map fields to follow the documentation.\n\n  Own Id: OTP-15862 Aux Id: ERL-951\n\n- Avoid DTLS crash due to missing gen_server return value in DTLS packet demux\n  process.\n\n  Own Id: OTP-15864 Aux Id: ERL-962","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 9.3.1 - SSL Release Notes","ref":"notes.html#ssl-9-3-1"},{"type":"extras","doc":"- Missing check of size of user_data_buffer made internal socket behave as an\n  active socket instead of active N. This could cause memory problems.\n\n  Own Id: OTP-15825 Aux Id: ERL-934, OTP-15823","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 9.3 - SSL Release Notes","ref":"notes.html#ssl-9-3"},{"type":"extras","doc":"- The distribution handshake with TLS distribution (`inet_tls_dist`) does now\n  utilize the socket option `{nodelay, true}`, which decreases the distribution\n  setup time significantly.\n\n  Own Id: OTP-14792\n\n- Correct shutdown reason to avoid an incorrect crash report\n\n  Own Id: OTP-15710 Aux Id: ERL-893\n\n- Enhance documentation and type specifications.\n\n  Own Id: OTP-15746 Aux Id: ERIERL-333","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- TLS-1.0, TLS-1.1 and DTLS-1.0 are now considered legacy and not supported by\n  default\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-14865\n\n- Use new logger API in ssl. Introduce log levels and verbose debug logging for\n  SSL.\n\n  Own Id: OTP-15055\n\n- Add new API function str_to_suite/1, cipher_suites/3 (list cipher suites as\n  rfc or OpenSSL name strings) and suite_to_openssl_str/1\n\n  Own Id: OTP-15483 Aux Id: ERL-924\n\n- Basic support for TLS 1.3 Server for experimental use. The client is not yet\n  functional, for more information see the Standards Compliance chapter of the\n  User's Guide.\n\n  Own Id: OTP-15591\n\n- Add support for PSK CCM ciphers from RFC 6655\n\n  Own Id: OTP-15626","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 9.2.3.7 - SSL Release Notes","ref":"notes.html#ssl-9-2-3-7"},{"type":"extras","doc":"- Data deliver with ssl:recv/2,3 could fail for when using packet mode. This has\n  been fixed by correcting the flow control handling of passive sockets when\n  packet mode is used.\n\n  Own Id: OTP-16764","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 9.2.3.6 - SSL Release Notes","ref":"notes.html#ssl-9-2-3-6"},{"type":"extras","doc":"- Fix timing bug that could cause ssl sockets to become unresponsive after an\n  ssl:recv/3 call timed out\n\n  Own Id: OTP-16619 Aux Id: ERL-1213","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 9.2.3.5 - SSL Release Notes","ref":"notes.html#ssl-9-2-3-5"},{"type":"extras","doc":"- Handling of zero size fragments in TLS could cause an infinite loop. This has\n  now been corrected.\n\n  Own Id: OTP-15328 Aux Id: ERIERL-379","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 9.2.3.4 - SSL Release Notes","ref":"notes.html#ssl-9-2-3-4"},{"type":"extras","doc":"- Hibernation now works as expected in all cases, was accidentally broken by\n  optimization efforts.\n\n  Own Id: OTP-15910","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 9.2.3.3 - SSL Release Notes","ref":"notes.html#ssl-9-2-3-3"},{"type":"extras","doc":"- Correct handshake handling, might cause strange symptoms such as ASN.1\n  certificate decoding issues.\n\n  Own Id: OTP-15879 Aux Id: ERL-968","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 9.2.3.2 - SSL Release Notes","ref":"notes.html#ssl-9-2-3-2"},{"type":"extras","doc":"- Returned \"alert error string\" is now same as logged alert string\n\n  Own Id: OTP-15844","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 9.2.3.1 - SSL Release Notes","ref":"notes.html#ssl-9-2-3-1"},{"type":"extras","doc":"- Correct solution for retaining tcp flow control OTP-15802 (ERL-934) as to not\n  break ssl:recv as reported in (ERL-938)\n\n  Own Id: OTP-15823 Aux Id: ERL-934, ERL-938","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 9.2.3 - SSL Release Notes","ref":"notes.html#ssl-9-2-3"},{"type":"extras","doc":"- Missing check of size of user_data_buffer made internal socket behave as an\n  active socket instead of active N. This could cause memory problems.\n\n  Own Id: OTP-15802 Aux Id: ERL-934","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Back port of bug fix ERL-893 from OTP-22 and document enhancements that will\n  solve dialyzer warnings for users of the ssl application.\n\n  This change also affects public_key, eldap (and inet doc).\n\n  Own Id: OTP-15785 Aux Id: ERL-929, ERL-893, PR-2215","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 9.2.2 - SSL Release Notes","ref":"notes.html#ssl-9-2-2"},{"type":"extras","doc":"- With the default BEAST Mitigation strategy for TLS 1.0 an empty TLS fragment\n  could be sent after a one-byte fragment. This glitch has been fixed.\n\n  Own Id: OTP-15054 Aux Id: ERIERL-346","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 9.2.1 - SSL Release Notes","ref":"notes.html#ssl-9-2-1"},{"type":"extras","doc":"- The timeout for a passive receive was sometimes not cancelled and later caused\n  a server crash. This bug has now been corrected.\n\n  Own Id: OTP-14701 Aux Id: ERL-883, ERL-884\n\n- Add tag for passive message (active N) in cb_info to retain transport\n  transparency.\n\n  Own Id: OTP-15679 Aux Id: ERL-861","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 9.2 - SSL Release Notes","ref":"notes.html#ssl-9-2"},{"type":"extras","doc":"- Fix bug that an incorrect return value for gen_statem could be created when\n  alert was a result of handling renegotiation info extension\n\n  Own Id: OTP-15502\n\n- Correct check for 3des_ede_cbc, could cause ssl to claim to support\n  3des_ede_cbc when cryptolib does not.\n\n  Own Id: OTP-15539\n\n- Improved DTLS error handling, avoids unexpected connection failure in rare\n  cases.\n\n  Own Id: OTP-15561\n\n- Corrected active once emulation bug that could cause the ssl_closed meassage\n  to not be sent. Bug introduced by OTP-15449\n\n  Own Id: OTP-15666 Aux Id: ERIERL-316,","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Add client option \\{reuse_session, SessionID::binary()\\} that can be used\n  together with new option value \\{reuse_sessions, save\\}. This makes it\n  possible to reuse a session from a specific connection establishment.\n\n  Own Id: OTP-15369\n\n- The Reason part of of the error return from the functions connect and\n  handshake has a better and documented format. This will sometimes differ from\n  previous returned reasons, however those where only documented as term() and\n  should for that reason not be relied on.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-15423\n\n- Refactor of state handling to improve TLS application data throughput and\n  reduce CPU overhead\n\n  Own Id: OTP-15445\n\n- The SSL code has been optimized in many small ways to reduce CPU load for\n  encryption/decryption, especially for Erlang's distribution protocol over TLS.\n\n  Own Id: OTP-15529\n\n- Add support for active N\n\n  Own Id: OTP-15665 Aux Id: ERL-811, PR-2072","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 9.1.2 - SSL Release Notes","ref":"notes.html#ssl-9-1-2"},{"type":"extras","doc":"- Fix encoding of the SRP extension length field in ssl. The old encoding of the\n  SRP extension length could cause interoperability problems with third party\n  SSL implementations when SRP was used.\n\n  Own Id: OTP-15477 Aux Id: ERL-790\n\n- Guarantee active once data delivery, handling TCP stream properly.\n\n  Own Id: OTP-15504 Aux Id: ERL-371\n\n- Correct gen_statem returns for some error cases\n\n  Own Id: OTP-15505","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 9.1.1 - SSL Release Notes","ref":"notes.html#ssl-9-1-1"},{"type":"extras","doc":"- Fixed renegotiation bug. Client did not handle server initiated renegotiation\n  correctly after rewrite to two connection processes, due to ERL-622 commit\n  d87ac1c55188f5ba5cdf72384125d94d42118c18. This could manifest it self as a \"\n  bad_record_mac\" alert.\n\n  Also included are some optimizations\n\n  Own Id: OTP-15489 Aux Id: ERL-308","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 9.1 - SSL Release Notes","ref":"notes.html#ssl-9-1"},{"type":"extras","doc":"- PEM cache was not evicting expired entries due to due to timezone confusion.\n\n  Own Id: OTP-15368\n\n- Make sure an error is returned if a \"transport_accept socket\" is used in some\n  other call than ssl:handshake\\* or ssl:controlling_process\n\n  Own Id: OTP-15384 Aux Id: ERL-756\n\n- Fix timestamp handling in the PEM-cache could cause entries to not be\n  invalidated at the correct time.\n\n  Own Id: OTP-15402\n\n- Extend check for undelivered data at closing, could under some circumstances\n  fail to deliver all data that was actually received.\n\n  Own Id: OTP-15412 Aux Id: ERL-731\n\n- Correct signature check for TLS-1.2 that allows different algorithms for\n  signature of peer cert and peer cert key. Not all allowed combinations where\n  accepted.\n\n  Own Id: OTP-15415 Aux Id: ERL-763\n\n- Correct gen_statem return value, could cause renegotiation to fail.\n\n  Own Id: OTP-15418 Aux Id: ERL-770","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Add engine support for RSA key exchange\n\n  Own Id: OTP-15420 Aux Id: ERIERL-268\n\n- ssl now uses active n internally to boost performance. Old active once\n  behavior can be restored by setting application variable see manual page for\n  ssl application (man 6).\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-15449","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 9.0.3 - SSL Release Notes","ref":"notes.html#ssl-9-0-3"},{"type":"extras","doc":"- Correct alert handling with new TLS sender process, from ssl-9.0.2. CLOSE\n  ALERTS could under some circumstances be encoded using an incorrect cipher\n  state. This would cause the peer to regard them as unknown messages.\n\n  Own Id: OTP-15337 Aux Id: ERL-738\n\n- Correct handling of socket packet option with new TLS sender process, from\n  ssl-9.0.2. When changing the socket option \\{packet, 1|2|3|4\\} with\n  ssl:setopts/2 the option must internally be propagated to the sender process\n  as well as the reader process as this particular option also affects the data\n  to be sent.\n\n  Own Id: OTP-15348 Aux Id: ERL-747","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 9.0.2 - SSL Release Notes","ref":"notes.html#ssl-9-0-2"},{"type":"extras","doc":"- Use separate processes for sending and receiving application data for TLS\n  connections to avoid potential deadlock that was most likely to occur when\n  using TLS for Erlang distribution. Note does not change the API.\n\n  Own Id: OTP-15122\n\n- Correct handling of empty server SNI extension\n\n  Own Id: OTP-15168\n\n- Correct PSK cipher suite handling and add selected_cipher_suite to connection\n  information\n\n  Own Id: OTP-15172\n\n- Adopt to the fact that cipher suite sign restriction are relaxed in TLS-1.2\n\n  Own Id: OTP-15173\n\n- Enhance error handling of non existing PEM files\n\n  Own Id: OTP-15174\n\n- Correct close handling of transport accepted sockets in the error state\n\n  Own Id: OTP-15216\n\n- Correct PEM cache to not add references to empty entries when PEM file does\n  not exist.\n\n  Own Id: OTP-15224\n\n- Correct handling of all PSK cipher suites\n\n  Before only some PSK suites would be correctly negotiated and most PSK ciphers\n  suites would fail the connection.\n\n  Own Id: OTP-15285","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- TLS will now try to order certificate chains if they appear to be unordered.\n  That is prior to TLS 1.3, “certificate_list” ordering was required to be\n  strict, however some implementations already allowed for some flexibility. For\n  maximum compatibility, all implementations SHOULD be prepared to handle\n  potentially extraneous certificates and arbitrary orderings from any TLS\n  version.\n\n  Own Id: OTP-12983\n\n- TLS will now try to reconstructed an incomplete certificate chains from its\n  local CA-database and use that data for the certificate path validation. This\n  especially makes sense for partial chains as then the peer might not send an\n  intermediate CA as it is considered the trusted root in that case.\n\n  Own Id: OTP-15060\n\n- Option keyfile defaults to certfile and should be trumped with key. This\n  failed for engine keys.\n\n  Own Id: OTP-15193\n\n- Error message improvement when own certificate has decoding issues, see also\n  issue ERL-668.\n\n  Own Id: OTP-15234\n\n- Correct dialyzer spec for key option\n\n  Own Id: OTP-15281","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 9.0.1 - SSL Release Notes","ref":"notes.html#ssl-9-0-1"},{"type":"extras","doc":"- Correct cipher suite handling for ECDHE\\_\\*, the incorrect handling could\n  cause an incorrrect suite to be selected and most likely fail the handshake.\n\n  Own Id: OTP-15203","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 9.0 - SSL Release Notes","ref":"notes.html#ssl-9-0"},{"type":"extras","doc":"- Correct handling of ECDH suites.\n\n  Own Id: OTP-14974\n\n- Proper handling of clients that choose to send an empty answer to a\n  certificate request\n\n  Own Id: OTP-15050","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Distribution over SSL (inet_tls) has, to improve performance, been rewritten\n  to not use intermediate processes and ports.\n\n  Own Id: OTP-14465\n\n- Add support for ECDHE_PSK cipher suites\n\n  Own Id: OTP-14547\n\n- For security reasons no longer support 3-DES cipher suites by default\n\n  \\*** INCOMPATIBILITY with possibly \\***\n\n  Own Id: OTP-14768\n\n- For security reasons RSA-key exchange cipher suites are no longer supported by\n  default\n\n  \\*** INCOMPATIBILITY with possible \\***\n\n  Own Id: OTP-14769\n\n- The interoperability option to fallback to insecure renegotiation now has to\n  be explicitly turned on.\n\n  \\*** INCOMPATIBILITY with possibly \\***\n\n  Own Id: OTP-14789\n\n- Drop support for SSLv2 enabled clients. SSLv2 has been broken for decades and\n  never supported by the Erlang SSL/TLS implementation. This option was by\n  default disabled and enabling it has proved to sometimes break connections not\n  using SSLv2 enabled clients.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-14824\n\n- Remove CHACHA20_POLY1305 ciphers form default for now. We have discovered\n  interoperability problems, ERL-538, that we believe needs to be solved in\n  crypto.\n\n  \\*** INCOMPATIBILITY with possibly \\***\n\n  Own Id: OTP-14882\n\n- Generalize DTLS packet multiplexing to make it easier to add future DTLS\n  features and uses.\n\n  Own Id: OTP-14888\n\n- Use uri_string module instead of http_uri.\n\n  Own Id: OTP-14902\n\n- The SSL distribution protocol `-proto inet_tls` has stopped setting the SSL\n  option `server_name_indication`. New verify funs for client and server in\n  `inet_tls_dist` has been added, not documented yet, that checks node name if\n  present in peer certificate. Usage is still also yet to be documented.\n\n  Own Id: OTP-14969 Aux Id: OTP-14465, ERL-598\n\n- Deprecate ssl:ssl_accept/\\[1,2,3] in favour of ssl:handshake/\\[1,2,3]\n\n  Own Id: OTP-15056\n\n- Customizes the hostname verification of the peer certificate, as different\n  protocols that use TLS such as HTTP or LDAP may want to do it differently\n\n  Own Id: OTP-15102 Aux Id: ERL-542, OTP-14962\n\n- Add utility function for converting erlang cipher suites to a string\n  representation (ERL-600).\n\n  Own Id: OTP-15106\n\n- First version with support for DTLS\n\n  Own Id: OTP-15142","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 8.2.6.4 - SSL Release Notes","ref":"notes.html#ssl-8-2-6-4"},{"type":"extras","doc":"- Add engine support for RSA key exchange\n\n  Own Id: OTP-15420","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 8.2.6.3 - SSL Release Notes","ref":"notes.html#ssl-8-2-6-3"},{"type":"extras","doc":"- Extend check for undelivered data at closing, could under some circumstances\n  fail to deliverd all data that was acctualy recivied.\n\n  Own Id: OTP-15412","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 8.2.6.2 - SSL Release Notes","ref":"notes.html#ssl-8-2-6-2"},{"type":"extras","doc":"- Correct handling of empty server SNI extension\n\n  Own Id: OTP-15168\n\n- Correct cipher suite handling for ECDHE\\_\\*, the incorrect handling could\n  cause an incorrrect suite to be selected and most likely fail the handshake.\n\n  Own Id: OTP-15203","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 8.2.6.1 - SSL Release Notes","ref":"notes.html#ssl-8-2-6-1"},{"type":"extras","doc":"- Improve cipher suite handling correcting ECC and TLS-1.2 requierments.\n  Backport of solution for ERL-641\n\n  Own Id: OTP-15178","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Option keyfile defaults to certfile and should be trumped with key. This\n  failed for engine keys.\n\n  Own Id: OTP-15193","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 8.2.6 - SSL Release Notes","ref":"notes.html#ssl-8-2-6"},{"type":"extras","doc":"- Proper handling of clients that choose to send an empty answer to a\n  certificate request\n\n  Own Id: OTP-15050","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 8.2.5 - SSL Release Notes","ref":"notes.html#ssl-8-2-5"},{"type":"extras","doc":"- Fix filter function to not incorrectly exclude AEAD cipher suites\n\n  Own Id: OTP-14981","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 8.2.4 - SSL Release Notes","ref":"notes.html#ssl-8-2-4"},{"type":"extras","doc":"- Optimization of bad merge conflict resolution causing dubble decode\n\n  Own Id: OTP-14843\n\n- Restore error propagation to OTP-19.3 behaviour, in OTP-20.2 implementation\n  adjustments to gen_statem needed some further adjustments to avoid a race\n  condition. This could cause a TLS server to not always report file path errors\n  correctly.\n\n  Own Id: OTP-14852\n\n- Corrected RC4 suites listing function to regard TLS version\n\n  Own Id: OTP-14871\n\n- Fix alert handling so that unexpected messages are logged and alerted\n  correctly\n\n  Own Id: OTP-14919\n\n- Correct handling of anonymous cipher suites\n\n  Own Id: OTP-14952","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Added new API functions to facilitate cipher suite handling\n\n  Own Id: OTP-14760\n\n- Correct TLS_FALLBACK_SCSV handling so that this special flag suite is always\n  placed last in the cipher suite list in accordance with the specs. Also make\n  sure this functionality is used in DTLS.\n\n  Own Id: OTP-14828\n\n- Add TLS record version sanity check for early as possible error detection and\n  consistency in ALERT codes generated\n\n  Own Id: OTP-14892","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 8.2.3 - SSL Release Notes","ref":"notes.html#ssl-8-2-3"},{"type":"extras","doc":"- Packet options cannot be supported for unreliable transports, that is, packet\n  option for DTLS over udp will not be supported.\n\n  Own Id: OTP-14664\n\n- Ensure data delivery before close if possible. This fix is related to fix in\n  PR-1479.\n\n  Own Id: OTP-14794","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- The crypto API is extended to use private/public keys stored in an Engine for\n  sign/verify or encrypt/decrypt operations.\n\n  The ssl application provides an API to use this new engine concept in TLS.\n\n  Own Id: OTP-14448\n\n- Implemented renegotiation for DTLS\n\n  Own Id: OTP-14563\n\n- A new command line option `-ssl_dist_optfile` has been added to facilitate\n  specifying the many options needed when using SSL as the distribution\n  protocol.\n\n  Own Id: OTP-14657","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 8.2.2 - SSL Release Notes","ref":"notes.html#ssl-8-2-2"},{"type":"extras","doc":"- TLS sessions must be registered with SNI if provided, so that sessions where\n  client hostname verification would fail cannot connect reusing a session\n  created when the server name verification succeeded.\n\n  Own Id: OTP-14632\n\n- An erlang TLS server configured with cipher suites using rsa key exchange, may\n  be vulnerable to an Adaptive Chosen Ciphertext attack (AKA Bleichenbacher\n  attack) against RSA, which when exploited, may result in plaintext recovery of\n  encrypted messages and/or a Man-in-the-middle (MiTM) attack, despite the\n  attacker not having gained access to the server’s private key itself.\n  [CVE-2017-1000385](https://nvd.nist.gov/vuln/detail/CVE-2017-1000385)\n\n  Exploiting this vulnerability to perform plaintext recovery of encrypted\n  messages will, in most practical cases, allow an attacker to read the\n  plaintext only after the session has completed. Only TLS sessions established\n  using RSA key exchange are vulnerable to this attack.\n\n  Exploiting this vulnerability to conduct a MiTM attack requires the attacker\n  to complete the initial attack, which may require thousands of server\n  requests, during the handshake phase of the targeted session within the window\n  of the configured handshake timeout. This attack may be conducted against any\n  TLS session using RSA signatures, but only if cipher suites using RSA key\n  exchange are also enabled on the server. The limited window of opportunity,\n  limitations in bandwidth, and latency make this attack significantly more\n  difficult to execute.\n\n  RSA key exchange is enabled by default although least prioritized if server\n  order is honored. For such a cipher suite to be chosen it must also be\n  supported by the client and probably the only shared cipher suite.\n\n  Captured TLS sessions encrypted with ephemeral cipher suites (DHE or ECDHE)\n  are not at risk for subsequent decryption due to this vulnerability.\n\n  As a workaround if default cipher suite configuration was used you can\n  configure the server to not use vulnerable suites with the ciphers option like\n  this:\n\n  `{ciphers, [Suite || Suite <- ssl:cipher_suites(), element(1,Suite) =/= rsa]}`\n\n  that is your code will look somethingh like this:\n\n  `ssl:listen(Port, [{ciphers, [Suite || Suite <- ssl:cipher_suites(), element(1,S) =/= rsa]} | Options]).`\n\n  Thanks to Hanno Böck, Juraj Somorovsky and Craig Young for reporting this\n  vulnerability.\n\n  Own Id: OTP-14748","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- If no SNI is available and the hostname is an IP-address also check for\n  IP-address match. This check is not as good as a DNS hostname check and\n  certificates using IP-address are not recommended.\n\n  Own Id: OTP-14655","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 8.2.1 - SSL Release Notes","ref":"notes.html#ssl-8-2-1"},{"type":"extras","doc":"- Max session table works correctly again\n\n  Own Id: OTP-14556","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Customize alert handling for DTLS over UDP to mitigate DoS attacks\n\n  Own Id: OTP-14078\n\n- Improved error propagation and reports\n\n  Own Id: OTP-14236","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 8.2 - SSL Release Notes","ref":"notes.html#ssl-8-2"},{"type":"extras","doc":"- ECDH-ECDSA key exchange supported, was accidentally dismissed in earlier\n  versions.\n\n  Own Id: OTP-14421\n\n- Correct close semantics for active once connections. This was a timing\n  dependent bug the resulted in the close message not always reaching the ssl\n  user process.\n\n  Own Id: OTP-14443","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- TLS-1.2 clients will now always send hello messages on its own format, as\n  opposed to earlier versions that will send the hello on the lowest supported\n  version, this is a change supported by the latest RFC.\n\n  This will make interoperability with some newer servers smoother. Potentially,\n  but unlikely, this could cause a problem with older servers if they do not\n  adhere to the RFC and ignore unknown extensions.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-13820\n\n- Allow Erlang/OTP to use OpenSSL in FIPS-140 mode, in order to satisfy specific\n  security requirements (mostly by different parts of the US federal\n  government).\n\n  See the new crypto users guide \"FIPS mode\" chapter about building and using\n  the FIPS support which is disabled by default.\n\n  (Thanks to dszoboszlay and legoscia)\n\n  Own Id: OTP-13921 Aux Id: PR-1180\n\n- Implemented DTLS cookie generation, required by spec, instead of using a\n  hardcoded value.\n\n  Own Id: OTP-14076\n\n- Implement sliding window replay protection of DTLS records.\n\n  Own Id: OTP-14077\n\n- TLS client processes will by default call public_key:pkix_verify_hostname/2 to\n  verify the hostname of the connection with the server certificates specified\n  hostname during certificate path validation. The user may explicitly disables\n  it. Also if the hostname cannot be derived from the first argument to connect\n  or is not supplied by the server name indication option, the check will not be\n  performed.\n\n  Own Id: OTP-14197\n\n- Extend connection_information/\\[1,2] . The values session_id, master_secret,\n  client_random and server_random can no be accessed by\n  connection_information/2. Note only session_id will be added to\n  connection_information/1. The rational is that values concerning the\n  connection security should have to be explicitly requested.\n\n  Own Id: OTP-14291\n\n- Chacha cipher suites are currently not tested enough to be most preferred ones\n\n  Own Id: OTP-14382\n\n- Basic support for DTLS that been tested together with OpenSSL.\n\n  Test by providing the option \\{protocol, dtls\\} to the ssl API functions\n  connect and listen.\n\n  Own Id: OTP-14388","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 8.1.3.1.1 - SSL Release Notes","ref":"notes.html#ssl-8-1-3-1-1"},{"type":"extras","doc":"- Fix alert handling so that unexpected messages are logged and alerted\n  correctly\n\n  Own Id: OTP-14929","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 8.1.3.1 - SSL Release Notes","ref":"notes.html#ssl-8-1-3-1"},{"type":"extras","doc":"- An erlang TLS server configured with cipher suites using rsa key exchange, may\n  be vulnerable to an Adaptive Chosen Ciphertext attack (AKA Bleichenbacher\n  attack) against RSA, which when exploited, may result in plaintext recovery of\n  encrypted messages and/or a Man-in-the-middle (MiTM) attack, despite the\n  attacker not having gained access to the server’s private key itself.\n  [CVE-2017-1000385](https://nvd.nist.gov/vuln/detail/CVE-2017-1000385)\n\n  Exploiting this vulnerability to perform plaintext recovery of encrypted\n  messages will, in most practical cases, allow an attacker to read the\n  plaintext only after the session has completed. Only TLS sessions established\n  using RSA key exchange are vulnerable to this attack.\n\n  Exploiting this vulnerability to conduct a MiTM attack requires the attacker\n  to complete the initial attack, which may require thousands of server\n  requests, during the handshake phase of the targeted session within the window\n  of the configured handshake timeout. This attack may be conducted against any\n  TLS session using RSA signatures, but only if cipher suites using RSA key\n  exchange are also enabled on the server. The limited window of opportunity,\n  limitations in bandwidth, and latency make this attack significantly more\n  difficult to execute.\n\n  RSA key exchange is enabled by default although least prioritized if server\n  order is honored. For such a cipher suite to be chosen it must also be\n  supported by the client and probably the only shared cipher suite.\n\n  Captured TLS sessions encrypted with ephemeral cipher suites (DHE or ECDHE)\n  are not at risk for subsequent decryption due to this vulnerability.\n\n  As a workaround if default cipher suite configuration was used you can\n  configure the server to not use vulnerable suites with the ciphers option like\n  this:\n\n  `{ciphers, [Suite || Suite <- ssl:cipher_suites(), element(1,Suite) =/= rsa]}`\n\n  that is your code will look somethingh like this:\n\n  `ssl:listen(Port, [{ciphers, [Suite || Suite <- ssl:cipher_suites(), element(1,S) =/= rsa]} | Options]).`\n\n  Thanks to Hanno Böck, Juraj Somorovsky and Craig Young for reporting this\n  vulnerability.\n\n  Own Id: OTP-14748","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 8.1.3 - SSL Release Notes","ref":"notes.html#ssl-8-1-3"},{"type":"extras","doc":"- Remove debug printout\n\n  Own Id: OTP-14396","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 8.1.2 - SSL Release Notes","ref":"notes.html#ssl-8-1-2"},{"type":"extras","doc":"- Correct active once emulation, for TLS. Now all data received by the\n  connection process will be delivered through active once, even when the active\n  once arrives after that the gen_tcp socket is closed by the peer.\n\n  Own Id: OTP-14300","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 8.1.1 - SSL Release Notes","ref":"notes.html#ssl-8-1-1"},{"type":"extras","doc":"- Corrected termination behavior, that caused a PEM cache bug and sometimes\n  resulted in connection failures.\n\n  Own Id: OTP-14100\n\n- Fix bug that could hang ssl connection processes when failing to require more\n  data for very large handshake packages. Add option max_handshake_size to\n  mitigate DoS attacks.\n\n  Own Id: OTP-14138\n\n- Improved support for CRL handling that could fail to work as intended when an\n  id-ce-extKeyUsage was present in the certificate. Also improvements where\n  needed to distributionpoint handling so that all revocations actually are\n  found and not deemed to be not determinable.\n\n  Own Id: OTP-14141\n\n- A TLS handshake might accidentally match old sslv2 format and ssl application\n  would incorrectly aborted TLS handshake with ssl_v2_client_hello_no_supported.\n  Parsing was altered to avoid this problem.\n\n  Own Id: OTP-14222\n\n- Correct default cipher list to prefer AES 128 before 3DES\n\n  Own Id: OTP-14235","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Move PEM cache to a dedicated process, to avoid making the SSL manager process\n  a bottleneck. This improves scalability of TLS connections.\n\n  Own Id: OTP-13874","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 8.1 - SSL Release Notes","ref":"notes.html#ssl-8-1"},{"type":"extras","doc":"- List of possible anonymous suites, never supported by default, where incorrect\n  for some TLS versions.\n\n  Own Id: OTP-13926","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Experimental version of DTLS. It is runnable but not complete and cannot be\n  considered reliable for production usage.\n\n  Own Id: OTP-12982\n\n- Add API options to handle ECC curve selection.\n\n  Own Id: OTP-13959","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 8.0.3 - SSL Release Notes","ref":"notes.html#ssl-8-0-3"},{"type":"extras","doc":"- A timing related bug in event handling could cause interoperability problems\n  between an erlang TLS server and some TLS clients, especially noticed with\n  Firefox as TLS client.\n\n  Own Id: OTP-13917\n\n- Correct ECC curve selection, the error could cause the default to always be\n  selected.\n\n  Own Id: OTP-13918","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 8.0.2 - SSL Release Notes","ref":"notes.html#ssl-8-0-2"},{"type":"extras","doc":"- Correctly formed handshake messages received out of order will now correctly\n  fail the connection with unexpected message.\n\n  Own Id: OTP-13853\n\n- Correct handling of signature algorithm selection\n\n  Own Id: OTP-13711","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- ssl application now behaves gracefully also on partially incorrect input from\n  peer.\n\n  Own Id: OTP-13834\n\n- Add application environment configuration bypass_pem_cache. This can be used\n  as a workaround for the current implementation of the PEM-cache that has\n  proven to be a bottleneck.\n\n  Own Id: OTP-13883","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 8.0.1 - SSL Release Notes","ref":"notes.html#ssl-8-0-1"},{"type":"extras","doc":"- The TLS/SSL protocol version selection for the SSL server has been corrected\n  to follow RFC 5246 Appendix E.1 especially in case where the list of supported\n  versions has gaps. Now the server selects the highest protocol version it\n  supports that is not higher than what the client supports.\n\n  Own Id: OTP-13753 Aux Id: seq13150","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 8.0 - SSL Release Notes","ref":"notes.html#ssl-8-0"},{"type":"extras","doc":"- Server now rejects, a not requested client cert, as an incorrect handshake\n  message and ends the connection.\n\n  Own Id: OTP-13651","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Remove default support for DES cipher suites\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-13195\n\n- Deprecate the function `crypto:rand_bytes` and make sure that\n  `crypto:strong_rand_bytes` is used in all places that are cryptographically\n  significant.\n\n  Own Id: OTP-13214\n\n- Better error handling of user error during TLS upgrade. ERL-69 is solved by\n  gen_statem rewrite of ssl application.\n\n  Own Id: OTP-13255\n\n- Provide user friendly error message when crypto rejects a key\n\n  Own Id: OTP-13256\n\n- Add ssl:getstat/1 and ssl:getstat/2\n\n  Own Id: OTP-13415\n\n- TLS distribution connections now allow specifying the options `verify_fun`,\n  `crl_check` and `crl_cache`. See the documentation. GitHub pull req #956\n  contributed by Magnus Henoch.\n\n  Own Id: OTP-13429 Aux Id: Pull#956\n\n- Remove confusing error message when closing a distributed erlang node running\n  over TLS\n\n  Own Id: OTP-13431\n\n- Remove default support for use of md5 in TLS 1.2 signature algorithms\n\n  Own Id: OTP-13463\n\n- ssl now uses gen_statem instead of gen_fsm to implement the ssl connection\n  process, this solves some timing issues in addition to making the code more\n  intuitive as the behaviour can be used cleanly instead of having a lot of\n  workaround for shortcomings of the behaviour.\n\n  Own Id: OTP-13464\n\n- Phase out interoperability with clients that offer SSLv2. By default they are\n  no longer supported, but an option to provide interoperability is offered.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-13465\n\n- OpenSSL has functions to generate short (eight hex digits) hashes of issuers\n  of certificates and CRLs. These hashes are used by the \"c_rehash\" script to\n  populate directories of CA certificates and CRLs, e.g. in the Apache web\n  server. Add functionality to let an Erlang program find the right CRL for a\n  given certificate in such a directory.\n\n  Own Id: OTP-13530\n\n- Some legacy TLS 1.0 software does not tolerate the 1/n-1 content split BEAST\n  mitigation technique. Add a beast_mitigation SSL option (defaulting to\n  one_n_minus_one) to select or disable the BEAST mitigation technique.\n\n  Own Id: OTP-13629\n\n- Enhance error log messages to facilitate for users to understand the error\n\n  Own Id: OTP-13632\n\n- Increased default DH params to 2048-bit\n\n  Own Id: OTP-13636\n\n- Propagate CRL unknown CA error so that public_key validation process continues\n  correctly and determines what should happen.\n\n  Own Id: OTP-13656\n\n- Introduce a flight concept for handshake packages. This is a preparation for\n  enabling DTLS, however it can also have a positive effects for TLS on slow and\n  unreliable networks.\n\n  Own Id: OTP-13678","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 7.3.3.2 - SSL Release Notes","ref":"notes.html#ssl-7-3-3-2"},{"type":"extras","doc":"- An erlang TLS server configured with cipher suites using rsa key exchange, may\n  be vulnerable to an Adaptive Chosen Ciphertext attack (AKA Bleichenbacher\n  attack) against RSA, which when exploited, may result in plaintext recovery of\n  encrypted messages and/or a Man-in-the-middle (MiTM) attack, despite the\n  attacker not having gained access to the server’s private key itself.\n  [CVE-2017-1000385](https://nvd.nist.gov/vuln/detail/CVE-2017-1000385)\n\n  Exploiting this vulnerability to perform plaintext recovery of encrypted\n  messages will, in most practical cases, allow an attacker to read the\n  plaintext only after the session has completed. Only TLS sessions established\n  using RSA key exchange are vulnerable to this attack.\n\n  Exploiting this vulnerability to conduct a MiTM attack requires the attacker\n  to complete the initial attack, which may require thousands of server\n  requests, during the handshake phase of the targeted session within the window\n  of the configured handshake timeout. This attack may be conducted against any\n  TLS session using RSA signatures, but only if cipher suites using RSA key\n  exchange are also enabled on the server. The limited window of opportunity,\n  limitations in bandwidth, and latency make this attack significantly more\n  difficult to execute.\n\n  RSA key exchange is enabled by default although least prioritized if server\n  order is honored. For such a cipher suite to be chosen it must also be\n  supported by the client and probably the only shared cipher suite.\n\n  Captured TLS sessions encrypted with ephemeral cipher suites (DHE or ECDHE)\n  are not at risk for subsequent decryption due to this vulnerability.\n\n  As a workaround if default cipher suite configuration was used you can\n  configure the server to not use vulnerable suites with the ciphers option like\n  this:\n\n  `{ciphers, [Suite || Suite <- ssl:cipher_suites(), element(1,Suite) =/= rsa]}`\n\n  that is your code will look somethingh like this:\n\n  `ssl:listen(Port, [{ciphers, [Suite || Suite <- ssl:cipher_suites(), element(1,S) =/= rsa]} | Options]).`\n\n  Thanks to Hanno Böck, Juraj Somorovsky and Craig Young for reporting this\n  vulnerability.\n\n  Own Id: OTP-14748","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 7.3.3 - SSL Release Notes","ref":"notes.html#ssl-7-3-3"},{"type":"extras","doc":"- Correct ssl:prf/5 to use the negotiated cipher suite's prf function in\n  ssl:prf/5 instead of the default prf.\n\n  Own Id: OTP-13546\n\n- Timeouts may have the value 0, guards have been corrected to allow this\n\n  Own Id: OTP-13635\n\n- Change of internal handling of hash sign pairs as the used one enforced to\n  much restrictions making some valid combinations unavailable.\n\n  Own Id: OTP-13670","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 7.3.3.0.1 - SSL Release Notes","ref":"notes.html#ssl-7-3-3-0-1"},{"type":"extras","doc":"- An erlang TLS server configured with cipher suites using rsa key exchange, may\n  be vulnerable to an Adaptive Chosen Ciphertext attack (AKA Bleichenbacher\n  attack) against RSA, which when exploited, may result in plaintext recovery of\n  encrypted messages and/or a Man-in-the-middle (MiTM) attack, despite the\n  attacker not having gained access to the server’s private key itself.\n  [CVE-2017-1000385](https://nvd.nist.gov/vuln/detail/CVE-2017-1000385)\n\n  Exploiting this vulnerability to perform plaintext recovery of encrypted\n  messages will, in most practical cases, allow an attacker to read the\n  plaintext only after the session has completed. Only TLS sessions established\n  using RSA key exchange are vulnerable to this attack.\n\n  Exploiting this vulnerability to conduct a MiTM attack requires the attacker\n  to complete the initial attack, which may require thousands of server\n  requests, during the handshake phase of the targeted session within the window\n  of the configured handshake timeout. This attack may be conducted against any\n  TLS session using RSA signatures, but only if cipher suites using RSA key\n  exchange are also enabled on the server. The limited window of opportunity,\n  limitations in bandwidth, and latency make this attack significantly more\n  difficult to execute.\n\n  RSA key exchange is enabled by default although least prioritized if server\n  order is honored. For such a cipher suite to be chosen it must also be\n  supported by the client and probably the only shared cipher suite.\n\n  Captured TLS sessions encrypted with ephemeral cipher suites (DHE or ECDHE)\n  are not at risk for subsequent decryption due to this vulnerability.\n\n  As a workaround if default cipher suite configuration was used you can\n  configure the server to not use vulnerable suites with the ciphers option like\n  this:\n\n  `{ciphers, [Suite || Suite <- ssl:cipher_suites(), element(1,Suite) =/= rsa]}`\n\n  that is your code will look somethingh like this:\n\n  `ssl:listen(Port, [{ciphers, [Suite || Suite <- ssl:cipher_suites(), element(1,S) =/= rsa]} | Options]).`\n\n  Thanks to Hanno Böck, Juraj Somorovsky and Craig Young for reporting this\n  vulnerability.\n\n  Own Id: OTP-14748","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Create a little randomness in sending of session invalidation messages, to\n  mitigate load when whole table is invalidated.\n\n  Own Id: OTP-13490","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 7.3.2 - SSL Release Notes","ref":"notes.html#ssl-7-3-2"},{"type":"extras","doc":"- Correct cipher suites conversion and guard expression. Caused problems with\n  GCM cipher suites and client side option to set signature_algorithms extension\n  values.\n\n  Own Id: OTP-13525","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 7.3.1 - SSL Release Notes","ref":"notes.html#ssl-7-3-1"},{"type":"extras","doc":"- Corrections to cipher suite handling using the 3 and 4 tuple format in\n  addition to commit 89d7e21cf4ae988c57c8ef047bfe85127875c70c\n\n  Own Id: OTP-13511","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Make values for the TLS-1.2 signature_algorithms extension configurable\n\n  Own Id: OTP-13261","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 7.3 - SSL Release Notes","ref":"notes.html#ssl-7-3"},{"type":"extras","doc":"- Make sure there is only one poller validator at a time for validating the\n  session cache.\n\n  Own Id: OTP-13185\n\n- A timing related issue could cause ssl to hang, especially happened with newer\n  versions of OpenSSL in combination with ECC ciphers.\n\n  Own Id: OTP-13253\n\n- Work around a race condition in the TLS distribution start.\n\n  Own Id: OTP-13268\n\n- Big handshake messages are now correctly fragmented in the TLS record layer.\n\n  Own Id: OTP-13306\n\n- Improve portability of ECC tests in Crypto and SSL for \"exotic\" OpenSSL\n  versions.\n\n  Own Id: OTP-13311\n\n- Certificate extensions marked as critical are ignored when using verify_none\n\n  Own Id: OTP-13377\n\n- If a certificate doesn't contain a CRL Distribution Points extension, and the\n  relevant CRL is not in the cache, and the `crl_check` option is not set to\n  `best_effort` , the revocation check should fail.\n\n  Own Id: OTP-13378\n\n- Enable TLS distribution over IPv6\n\n  Own Id: OTP-13391","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Improve error reporting for TLS distribution\n\n  Own Id: OTP-13219\n\n- Include options from connect, listen and accept in\n  `connection_information/1,2`\n\n  Own Id: OTP-13232\n\n- Allow adding extra options for outgoing TLS distribution connections, as\n  supported for plain TCP connections.\n\n  Own Id: OTP-13285\n\n- Use loopback as server option in TLS-distribution module\n\n  Own Id: OTP-13300\n\n- Verify certificate signature against original certificate binary.\n\n  This avoids bugs due to encoding errors when re-encoding a decode certificate.\n  As there exists several decode step and using of different ASN.1 specification\n  this is a risk worth avoiding.\n\n  Own Id: OTP-13334\n\n- Use `application:ensure_all_started/2` instead of hard-coding dependencies\n\n  Own Id: OTP-13363","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 7.2 - SSL Release Notes","ref":"notes.html#ssl-7-2"},{"type":"extras","doc":"- Honor distribution port range options\n\n  Own Id: OTP-12838\n\n- Correct supervisor specification in TLS distribution.\n\n  Own Id: OTP-13134\n\n- Correct cache timeout\n\n  Own Id: OTP-13141\n\n- Avoid crash and restart of ssl process when key file does not exist.\n\n  Own Id: OTP-13144\n\n- Enable passing of raw socket options on the format \\{raw,_,_,\\_\\} to the\n  underlying socket.\n\n  Own Id: OTP-13166\n\n- Hibernation with small or a zero timeout will now work as expected\n\n  Own Id: OTP-13189","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Add upper limit for session cache, configurable on ssl application level.\n\n  If upper limit is reached, invalidate the current cache entries, e.i the\n  session lifetime is the max time a session will be kept, but it may be\n  invalidated earlier if the max limit for the table is reached. This will keep\n  the ssl manager process well behaved, not exhusting memory. Invalidating the\n  entries will incrementally empty the cache to make room for fresh sessions\n  entries.\n\n  Own Id: OTP-12392\n\n- Use new time functions to measure passed time.\n\n  Own Id: OTP-12457\n\n- Improved error handling in TLS distribution\n\n  Own Id: OTP-13142\n\n- Distribution over TLS now honors the nodelay distribution flag\n\n  Own Id: OTP-13143","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 7.1 - SSL Release Notes","ref":"notes.html#ssl-7-1"},{"type":"extras","doc":"- Add DER encoded ECPrivateKey as valid input format for key option.\n\n  Own Id: OTP-12974\n\n- Correct return value of default session callback module\n\n  This error had the symptom that the client check for unique session would\n  always fail, potentially making the client session table grow a lot and\n  causing long setup times.\n\n  Own Id: OTP-12980","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Add possibility to downgrade an SSL/TLS connection to a tcp connection, and\n  give back the socket control to a user process.\n\n  This also adds the possibility to specify a timeout to the ssl:close function.\n\n  Own Id: OTP-11397\n\n- Add application setting to be able to change fatal alert shutdown timeout,\n  also shorten the default timeout. The fatal alert timeout is the number of\n  milliseconds between sending of a fatal alert and closing the connection.\n  Waiting a little while improves the peers chances to properly receiving the\n  alert so it may shutdown gracefully.\n\n  Own Id: OTP-12832","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 7.0 - SSL Release Notes","ref":"notes.html#ssl-7-0"},{"type":"extras","doc":"- Ignore signature_algorithm (TLS 1.2 extension) sent to TLS 1.0 or TLS 1.1\n  server\n\n  Own Id: OTP-12670\n\n- Improve error handling in TLS distribution module to avoid lingering sockets.\n\n  Own Id: OTP-12799 Aux Id: Tom Briden\n\n- Add option \\{client_renegotiation, boolean()\\} option to the server-side of\n  the SSL application.\n\n  Own Id: OTP-12815","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Add new API functions to handle CRL-verification\n\n  Own Id: OTP-10362 Aux Id: kunagi-215 \\[126]\n\n- Remove default support for SSL-3.0, due to Poodle vunrability in protocol\n  specification.\n\n  Add padding check for TLS-1.0 to remove Poodle vunrability from TLS 1.0, also\n  add the option padding_check. This option only affects TLS-1.0 connections and\n  if set to false it disables the block cipher padding check to be able to\n  interoperate with legacy software.\n\n  Remove default support for RC4 cipher suites, as they are consider too weak.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-12390\n\n- Add support for TLS ALPN (Application-Layer Protocol Negotiation) extension.\n\n  Own Id: OTP-12580\n\n- Add SNI (Server Name Indication) support for the server side.\n\n  Own Id: OTP-12736","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 6.0.1.1 - SSL Release Notes","ref":"notes.html#ssl-6-0-1-1"},{"type":"extras","doc":"- Gracefully ignore proprietary hash_sign algorithms\n\n  Own Id: OTP-12829","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 6.0.1 - SSL Release Notes","ref":"notes.html#ssl-6-0-1"},{"type":"extras","doc":"- Terminate gracefully when receiving bad input to premaster secret calculation\n\n  Own Id: OTP-12783","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 6.0 - SSL Release Notes","ref":"notes.html#ssl-6-0"},{"type":"extras","doc":"- Exclude self-signed trusted anchor certificates from certificate prospective\n  certification path according to RFC 3280.\n\n  This will avoid some unnecessary certificate processing.\n\n  Own Id: OTP-12449","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Separate client and server session cache internally.\n\n  Avoid session table growth when client starts many connections in such a\n  manner that many connections are started before session reuse is possible.\n  Only save a new session in client if there is no equivalent session already\n  stored.\n\n  Own Id: OTP-11365\n\n- The PEM cache is now validated by a background process, instead of always\n  keeping it if it is small enough and clearing it otherwise. That strategy\n  required that small caches where cleared by API function if a file changes on\n  disk.\n\n  However export the API function to clear the cache as it may still be useful.\n\n  Own Id: OTP-12391\n\n- Add padding check for TLS-1.0 to remove Poodle vulnerability from TLS 1.0,\n  also add the option padding_check. This option only affects TLS-1.0\n  connections and if set to false it disables the block cipher padding check to\n  be able to interoperate with legacy software.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-12420\n\n- Add support for TLS_FALLBACK_SCSV used to prevent undesired TLS version\n  downgrades. If used by a client that is vulnerable to the POODLE attack, and\n  the server also supports TLS_FALLBACK_SCSV, the attack can be prevented.\n\n  Own Id: OTP-12458","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 5.3.8 - SSL Release Notes","ref":"notes.html#ssl-5-3-8"},{"type":"extras","doc":"- Make sure the clean rule for ssh, ssl, eunit and otp_mibs actually removes\n  generated files.\n\n  Own Id: OTP-12200","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Change code to reflect that state data may be secret to avoid breaking\n  dialyzer contracts.\n\n  Own Id: OTP-12341","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 5.3.7 - SSL Release Notes","ref":"notes.html#ssl-5-3-7"},{"type":"extras","doc":"- Handle the fact that servers may send an empty SNI extension to the client.\n\n  Own Id: OTP-12198","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 5.3.6 - SSL Release Notes","ref":"notes.html#ssl-5-3-6"},{"type":"extras","doc":"- Corrected handling of ECC certificates, there where several small issues with\n  the handling of such certificates in the ssl and public_key application. Now\n  ECC signed ECC certificates shall work and not only RSA signed ECC\n  certificates.\n\n  Own Id: OTP-12026\n\n- Check that the certificate chain ends with a trusted ROOT CA e.i. a\n  self-signed certificate, but provide an option partial_chain to enable the\n  application to define an intermediat CA as trusted.\n\n  Own Id: OTP-12149","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Add decode functions for SNI (Server Name Indication)\n\n  Own Id: OTP-12048","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 5.3.5 - SSL Release Notes","ref":"notes.html#ssl-5-3-5"},{"type":"extras","doc":"- ssl:recv now returns \\{error, einval\\} if applied to a non passive socket, the\n  same as gen_tcp:recv.\n\n  Thanks to Danil Zagoskin for reporting this issue\n\n  Own Id: OTP-11878\n\n- Corrected handling of default values for signature_algorithms extension in\n  TLS-1.2 and corresponding values used in previous versions that does not\n  support this extension.\n\n  Thanks to Danil Zagoskin\n\n  Own Id: OTP-11886\n\n- Handle socket option inheritance when pooling of accept sockets is used\n\n  Own Id: OTP-11897\n\n- Make sure that the list of versions, possibly supplied in the versions option,\n  is not order dependent.\n\n  Thanks to Ransom Richardson for reporting this issue\n\n  Own Id: OTP-11912\n\n- Reject connection if the next_protocol message is sent twice.\n\n  Own Id: OTP-11926\n\n- Correct options handling when ssl:ssl_accept/3 is called with new ssl options\n  after calling ssl:listen/2\n\n  Own Id: OTP-11950","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Gracefully handle unknown alerts\n\n  Thanks to Atul Atri for reporting this issue\n\n  Own Id: OTP-11874\n\n- Gracefully ignore cipher suites sent by client not supported by the SSL/TLS\n  version that the client has negotiated.\n\n  Thanks to Danil Zagoskin for reporting this issue\n\n  Own Id: OTP-11875\n\n- Gracefully handle structured garbage, i.e a client sends some garbage in a ssl\n  record instead of a valid fragment.\n\n  Thanks to Danil Zagoskin\n\n  Own Id: OTP-11880\n\n- Gracefully handle invalid alerts\n\n  Own Id: OTP-11890\n\n- Generalize handling of default ciphers\n\n  Thanks to Andreas Schultz\n\n  Own Id: OTP-11966\n\n- Make sure change cipher spec is correctly handled\n\n  Own Id: OTP-11975","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 5.3.4 - SSL Release Notes","ref":"notes.html#ssl-5-3-4"},{"type":"extras","doc":"- Fix incorrect dialyzer spec and types, also enhance documentation.\n\n  Thanks to Ayaz Tuncer.\n\n  Own Id: OTP-11627\n\n- Fix possible mismatch between SSL/TLS version and default ciphers. Could\n  happen when you specified SSL/TLS-version in optionlist to listen or accept.\n\n  Own Id: OTP-11712\n\n- Application upgrade (appup) files are corrected for the following\n  applications:\n\n  `asn1, common_test, compiler, crypto, debugger, dialyzer, edoc, eldap, erl_docgen, et, eunit, gs, hipe, inets, observer, odbc, os_mon, otp_mibs, parsetools, percept, public_key, reltool, runtime_tools, ssh, syntax_tools, test_server, tools, typer, webtool, wx, xmerl`\n\n  A new test utility for testing appup files is added to test_server. This is\n  now used by most applications in OTP.\n\n  (Thanks to Tobias Schlager)\n\n  Own Id: OTP-11744","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Moved elliptic curve definition from the crypto NIF/OpenSSL into Erlang code,\n  adds the RFC-5639 brainpool curves and makes TLS use them (RFC-7027).\n\n  Thanks to Andreas Schultz\n\n  Own Id: OTP-11578\n\n- Unicode adaptations\n\n  Own Id: OTP-11620\n\n- Added option honor_cipher_order. This instructs the server to prefer its own\n  cipher ordering rather than the client's and can help protect against things\n  like BEAST while maintaining compatibility with clients which only support\n  older ciphers.\n\n  Thanks to Andrew Thompson for the implementation, and Andreas Schultz for the\n  test cases.\n\n  Own Id: OTP-11621\n\n- Replace boolean checking in validate_option with is_boolean guard.\n\n  Thanks to Andreas Schultz.\n\n  Own Id: OTP-11634\n\n- Some function specs are corrected or moved and some edoc comments are\n  corrected in order to allow use of edoc. (Thanks to Pierre Fenoll)\n\n  Own Id: OTP-11702\n\n- Correct clean up of certificate database when certs are inputted in pure DER\n  format.The incorrect code could cause a memory leek when certs where inputted\n  in DER. Thanks to Bernard Duggan for reporting this.\n\n  Own Id: OTP-11733\n\n- Improved documentation of the cacertfile option\n\n  Own Id: OTP-11759 Aux Id: seq12535\n\n- Avoid next protocol negotiation failure due to incorrect option format.\n\n  Own Id: OTP-11760\n\n- Handle v1 CRLs, with no extensions and fixes issues with IDP (Issuing\n  Distribution Point) comparison during CRL validation.\n\n  Thanks to Andrew Thompson\n\n  Own Id: OTP-11761\n\n- Server now ignores client ECC curves that it does not support instead of\n  crashing.\n\n  Thanks to Danil Zagoskin for reporting the issue and suggesting a solution.\n\n  Own Id: OTP-11780\n\n- Handle SNI (Server Name Indication) alert unrecognized_name and gracefully\n  deal with unexpected alerts.\n\n  Thanks to Masatake Daimon for reporting this.\n\n  Own Id: OTP-11815\n\n- Add possibility to specify ssl options when calling ssl:ssl_accept\n\n  Own Id: OTP-11837","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 5.3.3 - SSL Release Notes","ref":"notes.html#ssl-5-3-3"},{"type":"extras","doc":"- Add missing validation of the server_name_indication option and test for its\n  explicit use. It was not possible to set or disable the default\n  server_name_indication as the validation of the option was missing.\n\n  Own Id: OTP-11567\n\n- Elliptic curve selection in server mode now properly selects a curve suggested\n  by the client, if possible, and the fallback alternative is changed to a more\n  widely supported curve.\n\n  Own Id: OTP-11575\n\n- Bug in the TLS hello extension handling caused the server to behave as it did\n  not understand secure renegotiation.\n\n  Own Id: OTP-11595","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 5.3.2 - SSL Release Notes","ref":"notes.html#ssl-5-3-2"},{"type":"extras","doc":"- Honors the clients advertised support of elliptic curves and no longer sends\n  incorrect elliptic curve extension in server hello.\n\n  Own Id: OTP-11370\n\n- Fix initialization of DTLS fragment reassembler, in previously contributed\n  code, for future support of DTLS . Thanks to Andreas Schultz.\n\n  Own Id: OTP-11376\n\n- Corrected type error in client_preferred_next_protocols documentation. Thanks\n  to Julien Barbot.\n\n  Own Id: OTP-11457","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- TLS code has been refactored to prepare for future DTLS support. Also some\n  DTLS code is in place but not yet runnable, some of it contributed by Andreas\n  Schultz and some of it written by the OTP team. Thanks to to Andreas for his\n  participation.\n\n  Own Id: OTP-11292\n\n- Remove extraneous dev debug code left in the close function. Thanks to Ken\n  Key.\n\n  Own Id: OTP-11447\n\n- Add SSL Server Name Indication (SNI) client support. Thanks to Julien Barbot.\n\n  Own Id: OTP-11460","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 5.3.1 - SSL Release Notes","ref":"notes.html#ssl-5-3-1"},{"type":"extras","doc":"- Setopts during renegotiation caused the renegotiation to be unsuccessful.\n\n  If calling setopts during a renegotiation the FSM state might change during\n  the handling of the setopts messages, this is now handled correctly.\n\n  Own Id: OTP-11228\n\n- Now handles signature_algorithm field in digitally_signed properly with proper\n  defaults. Prior to this change some elliptic curve cipher suites could fail\n  reporting the error \"bad certificate\".\n\n  Own Id: OTP-11229\n\n- The code emulating the inet header option was changed in the belief that it\n  made it inet compatible. However the testing is a bit hairy as the inet option\n  is actually broken, now the tests are corrected and the header option should\n  work in the same broken way as inet again, preferably use the bitsyntax\n  instead.\n\n  Own Id: OTP-11230","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Make the ssl manager name for erlang distribution over SSL/TLS relative to the\n  module name of the ssl_manager.\n\n  This can be beneficial when making tools that rename modules for internal\n  processing in the tool.\n\n  Own Id: OTP-11255\n\n- Add documentation regarding log_alert option.\n\n  Own Id: OTP-11271","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 5.3 - SSL Release Notes","ref":"notes.html#ssl-5-3"},{"type":"extras","doc":"- Honor the versions option to ssl:connect and ssl:listen.\n\n  Own Id: OTP-10905\n\n- Next protocol negotiation with reused sessions will now succeed\n\n  Own Id: OTP-10909","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Add support for PSK (Pre Shared Key) and SRP (Secure Remote Password) cipher\n  suites, thanks to Andreas Schultz.\n\n  Own Id: OTP-10450 Aux Id: kunagi-269 \\[180]\n\n- Fix SSL Next Protocol Negotiation documentation. Thanks to Julien Barbot.\n\n  Own Id: OTP-10955\n\n- Fix ssl_connection to support reading proxy/chain certificates. Thanks to\n  Valentin Kuznetsov.\n\n  Own Id: OTP-10980\n\n- Integrate elliptic curve contribution from Andreas Schultz\n\n  In order to be able to support elliptic curve cipher suites in SSL/TLS,\n  additions to handle elliptic curve infrastructure has been added to public_key\n  and crypto.\n\n  This also has resulted in a rewrite of the crypto API to gain consistency and\n  remove unnecessary overhead. All OTP applications using crypto has been\n  updated to use the new API.\n\n  Impact: Elliptic curve cryptography (ECC) offers equivalent security with\n  smaller key sizes than other public key algorithms. Smaller key sizes result\n  in savings for power, memory, bandwidth, and computational cost that make ECC\n  especially attractive for constrained environments.\n\n  Own Id: OTP-11009","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 5.2.1 - SSL Release Notes","ref":"notes.html#ssl-5-2-1"},{"type":"extras","doc":"- Transport callback handling is changed so that gen_tcp is treated as a special\n  case where inet will be called directly for functions such as setopts, as\n  gen_tcp does not have its own setopts. This will enable users to use the\n  transport callback for other customizations such as websockets.\n\n  Own Id: OTP-10847\n\n- Follow up to OTP-10451 solved in ssl-5.2 R16A. Make sure format_error return\n  good strings. Replace confusing legacy atoms with more descriptive atoms.\n\n  Own Id: OTP-10864","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 5.1.2.1 - SSL Release Notes","ref":"notes.html#ssl-5-1-2-1"},{"type":"extras","doc":"- Make log_alert configurable as option in ssl, SSLLogLevel added as option to\n  inets conf file\n\n  Own Id: OTP-11259","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 5.2 - SSL Release Notes","ref":"notes.html#ssl-5-2"},{"type":"extras","doc":"- SSL: TLS 1.2, advertise sha224 support, thanks to Andreas Schultz.\n\n  Own Id: OTP-10586\n\n- If an ssl server is restarted with new options and a client tries to reuse a\n  session the server must make sure that it complies to the new options before\n  agreeing to reuse it.\n\n  Own Id: OTP-10595\n\n- Now handles cleaning of CA-certificate database correctly so that there will\n  be no memory leek, bug was introduced in ssl- 5.1 when changing implementation\n  to increase parallel execution.\n\n  Impact: Improved memory usage, especially if you have many different\n  certificates and upgrade tcp-connections to TLS-connections.\n\n  Own Id: OTP-10710","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Support Next Protocol Negotiation in TLS, thanks to Ben Murphy for the\n  contribution.\n\n  Impact: Could give performance benefit if used as it saves a round trip.\n\n  Own Id: OTP-10361 Aux Id: kunagi-214 \\[125]\n\n- TLS 1.2 will now be the default TLS version if sufficient crypto support is\n  available otherwise TLS 1.1 will be default.\n\n  Impact: A default TLS connection will have higher security and hence it may be\n  perceived as slower then before.\n\n  Own Id: OTP-10425 Aux Id: kunagi-275 \\[186]\n\n- It is now possible to call controlling_process on a listen socket, same as in\n  gen_tcp.\n\n  Own Id: OTP-10447\n\n- Remove filter mechanisms that made error messages backwards compatible with\n  old ssl but hid information about what actually happened.\n\n  This does not break the documented API however other reason terms may be\n  returned, so code that matches on the reason part of \\{error, Reason\\} may\n  fail.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-10451 Aux Id: kunagi-270 \\[181]\n\n- Added missing dependencies to Makefile\n\n  Own Id: OTP-10594\n\n- Removed deprecated function ssl:pid/0, it has been pointless since R14 but has\n  been keep for backwards compatibility.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-10613 Aux Id: kunagi-331 \\[242]\n\n- Refactor to simplify addition of key exchange methods, thanks to Andreas\n  Schultz.\n\n  Own Id: OTP-10709","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 5.1.2 - SSL Release Notes","ref":"notes.html#ssl-5-1-2"},{"type":"extras","doc":"- ssl:ssl_accept/2 timeout is no longer ignored\n\n  Own Id: OTP-10600","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 5.1.1 - SSL Release Notes","ref":"notes.html#ssl-5-1-1"},{"type":"extras","doc":"- ssl:recv/3 could \"loose\" data when the timeout occurs. If the timeout in\n  ssl:connect or ssl:ssl_accept expired the ssl connection process was not\n  terminated as it should, this due to gen_fsm:send_all_state_event timeout is a\n  client side time out. These timouts are now handled by the gen_fsm-procss\n  instead.\n\n  Own Id: OTP-10569","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Better termination handling that avoids hanging.\n\n  Own Id: OTP-10574","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 5.1 - SSL Release Notes","ref":"notes.html#ssl-5-1"},{"type":"extras","doc":"- Sometimes the client process could receive an extra \\{error, closed\\} message\n  after ssl:recv had returned \\{error, closed\\}.\n\n  Own Id: OTP-10118\n\n- ssl v3 alert number 41 (no_certificate_RESERVED) is now recognized\n\n  Own Id: OTP-10196","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Experimental support for TLS 1.1 is now available, will be officially\n  supported from OTP-R16. Thanks to Andreas Schultz for implementing the first\n  version.\n\n  Own Id: OTP-8871\n\n- Experimental support for TLS 1.2 is now available, will be officially\n  supported from OTP-R16. Thanks to Andreas Schultz for implementing the first\n  version.\n\n  Own Id: OTP-8872\n\n- Removed some bottlenecks increasing the applications parallelism especially\n  for the client side.\n\n  Own Id: OTP-10113\n\n- Workaround for handling certificates that wrongly encode X509countryname in\n  utf-8 when the actual value is a valid ASCCI value of length 2. Such\n  certificates are accepted by many browsers such as Chrome and Fierfox so for\n  interoperability reasons we will too.\n\n  Own Id: OTP-10222","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 5.0.1 - SSL Release Notes","ref":"notes.html#ssl-5-0-1"},{"type":"extras","doc":"- Robustness and improvement to distribution over SSL\n\n  Fix a bug where ssl_tls_dist_proxy would crash at caller timeout. Fix a bug\n  where a timeout from the SSL layer would block the distribution indefinitely.\n  Run the proxy exclusively on the loopback interface. (Thanks to Paul Guyot)\n\n  Own Id: OTP-9915\n\n- Fix setup loop of SSL TLS dist proxy\n\n  Fix potential leak of processes waiting indefinitely for data from closed\n  sockets during socket setup phase. (Thanks to Paul Guyot)\n\n  Own Id: OTP-9916\n\n- Correct spelling of registered (Thanks to Richard Carlsson)\n\n  Own Id: OTP-9925\n\n- Added TLS PRF function to the SSL API for generation of additional key\n  material from a TLS session. (Thanks to Andreas Schultz)\n\n  Own Id: OTP-10024","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 5.0 - SSL Release Notes","ref":"notes.html#ssl-5-0"},{"type":"extras","doc":"- Invalidation handling of sessions could cause the time_stamp field in the\n  session record to be set to undefined crashing the session clean up process.\n  This did not affect the connections but would result in that the session table\n  would grow.\n\n  Own Id: OTP-9696 Aux Id: seq11947\n\n- Changed code to use ets:foldl and throw instead of ets:next traversal,\n  avoiding the need to explicitly call ets:safe_fixtable. It was possible to get\n  a badarg-crash under special circumstances.\n\n  Own Id: OTP-9703 Aux Id: seq11947\n\n- Send ssl_closed notification to active ssl user when a tcp error occurs.\n\n  Own Id: OTP-9734 Aux Id: seq11946\n\n- If a passive receive was ongoing during a renegotiation the process evaluating\n  ssl:recv could be left hanging for ever.\n\n  Own Id: OTP-9744","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Support for the old ssl implementation is dropped and the code is removed.\n\n  Own Id: OTP-7048\n\n- The erlang distribution can now be run over the new ssl implementation. All\n  options can currently not be set but it is enough to replace to old ssl\n  implementation.\n\n  Own Id: OTP-7053\n\n- public_key, ssl and crypto now supports PKCS-8\n\n  Own Id: OTP-9312\n\n- Implements a CBC timing attack counter measure. Thanks to Andreas Schultz for\n  providing the patch.\n\n  Own Id: OTP-9683\n\n- Mitigates an SSL/TLS Computational DoS attack by disallowing the client to\n  renegotiate many times in a row in a short time interval, thanks to Tuncer\n  Ayaz for alerting us about this.\n\n  Own Id: OTP-9739\n\n- Implements the 1/n-1 splitting countermeasure to the Rizzo Duong BEAST attack,\n  affects SSL 3.0 and TLS 1.0. Thanks to Tuncer Ayaz for alerting us about this.\n\n  Own Id: OTP-9750","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 4.1.6 - SSL Release Notes","ref":"notes.html#ssl-4-1-6"},{"type":"extras","doc":"- replace \"a ssl\" with \"an ssl\" reindent pkix_path_validation/3 Trivial\n  documentation fixes (Thanks to Christian von Roques )\n\n  Own Id: OTP-9464","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Adds function clause to avoid denial of service attack. Thanks to Vinod for\n  reporting this vulnerability.\n\n  Own Id: OTP-9364\n\n- Error handling code now takes care of inet:getopts/2 and inets:setopts/2\n  crashes. Thanks to Richard Jones for reporting this.\n\n  Own Id: OTP-9382\n\n- Support explicit use of packet option httph and httph_bin\n\n  Own Id: OTP-9461\n\n- Decoding of hello extensions could fail to come to the correct conclusion due\n  to an error in a binary match pattern. Thanks to Ben Murphy.\n\n  Own Id: OTP-9589","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 4.1.5 - SSL Release Notes","ref":"notes.html#ssl-4-1-5"},{"type":"extras","doc":"- Calling gen_tcp:connect with option \\{ip, \\{127,0,0,1\\}\\} results in an exit\n  with reason badarg. Neither SSL nor INETS This was not caught, resulting in\n  crashes with incomprehensible reasons.\n\n  Own Id: OTP-9289 Aux Id: seq11845","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 4.1.3 - SSL Release Notes","ref":"notes.html#ssl-4-1-3"},{"type":"extras","doc":"- Fixed error in cache-handling fix from ssl-4.1.2\n\n  Own Id: OTP-9018 Aux Id: seq11739\n\n- Verification of a critical extended_key_usage-extension corrected\n\n  Own Id: OTP-9029 Aux Id: seq11541","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 4.1.2 - SSL Release Notes","ref":"notes.html#ssl-4-1-2"},{"type":"extras","doc":"- The ssl application caches certificate files, it will now invalidate cache\n  entries if the diskfile is changed.\n\n  Own Id: OTP-8965 Aux Id: seq11739\n\n- Now runs the terminate function before returning from the call made by\n  ssl:close/1, as before the caller of ssl:close/1 could get problems with the\n  reuseaddr option.\n\n  Own Id: OTP-8992","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"","title":"SSL 4.1.1 - SSL Release Notes","ref":"notes.html#ssl-4-1-1"},{"type":"extras","doc":"- Correct handling of client certificate verify message When checking the client\n  certificate verify message the server used the wrong algorithm identifier to\n  determine the signing algorithm, causing a function clause error in the\n  public_key application when the key-exchange algorithm and the public key\n  algorithm of the client certificate happen to differ.\n\n  Own Id: OTP-8897","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- For testing purposes ssl now also support some anonymous cipher suites when\n  explicitly configured to do so.\n\n  Own Id: OTP-8870\n\n- Sends an error alert instead of crashing if a crypto function for the selected\n  cipher suite fails.\n\n  Own Id: OTP-8930 Aux Id: seq11720","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 4.1 - SSL Release Notes","ref":"notes.html#ssl-4-1"},{"type":"extras","doc":"- Updated ssl to ignore CA certs that violate the asn1-spec for a certificate,\n  and updated public key asn1 spec to handle inherited DSS-params.\n\n  Own Id: OTP-7884\n\n- Changed ssl implementation to retain backwards compatibility for old option\n  \\{verify, 0\\} that shall be equivalent to \\{verify, verify_none\\}, also\n  separate the cases unknown ca and selfsigned peer cert, and restored return\n  value of deprecated function public_key:pem_to_der/1.\n\n  Own Id: OTP-8858\n\n- Changed the verify fun so that it differentiate between the peer certificate\n  and CA certificates by using valid_peer or valid as the second argument to the\n  verify fun. It may not always be trivial or even possible to know when the\n  peer certificate is reached otherwise.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-8873","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 4.0.1 - SSL Release Notes","ref":"notes.html#ssl-4-0-1"},{"type":"extras","doc":"- The server now verifies the client certificate verify message correctly,\n  instead of causing a case-clause.\n\n  Own Id: OTP-8721\n\n- The client hello message now always include ALL available cipher suites (or\n  those specified by the ciphers option). Previous implementation would filter\n  them based on the client certificate key usage extension (such filtering only\n  makes sense for the server certificate).\n\n  Own Id: OTP-8772\n\n- Fixed handling of the option \\{mode, list\\} that was broken for some packet\n  types for instance line.\n\n  Own Id: OTP-8785\n\n- Empty packets were not delivered to the client.\n\n  Own Id: OTP-8790\n\n- Building in a source tree without prebuilt platform independent build results\n  failed on the SSL examples when:\n\n  - cross building. This has been solved by not building the SSL examples during\n    a cross build.\n  - building on Windows.\n\n  Own Id: OTP-8791\n\n- Fixed a handshake error which occurred on some ssl implementations.\n\n  Own Id: OTP-8793","title":"Fixed Bugs and Malfunctions - SSL Release Notes","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","doc":"- Revise the public_key API - Cleaned up and documented the public_key API to\n  make it useful for general use, also changed ssl to use the new API.\n\n  Own Id: OTP-8722\n\n- Added support for inputing certificates and keys directly in DER format these\n  options will override the pem-file options if specified.\n\n  Own Id: OTP-8723\n\n- To gain interoperability ssl will not check for padding errors when using TLS\n  1.0. It is first in TLS 1.1 that checking the padding is an requirement.\n\n  Own Id: OTP-8740\n\n- Changed the semantics of the verify_fun option in the ssl-application so that\n  it takes care of both application handling of path validation errors and\n  verification of application specific extensions. This means that it is now\n  possible for the server application in verify_peer mode to handle path\n  validation errors. This change moved some functionality earlier in ssl to the\n  public_key application.\n\n  Own Id: OTP-8770\n\n- Added the functionality so that the verification fun will be called when a\n  certificate is considered valid by the path validation to allow access to each\n  certificate in the path to the user application. Also try to verify\n  subject-AltName, if unable to verify it let the application verify it.\n\n  Own Id: OTP-8825","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"","title":"SSL 4.0 - SSL Release Notes","ref":"notes.html#ssl-4-0"},{"type":"extras","doc":"- New ssl now support client/server-certificates signed by dsa keys.\n\n  Own Id: OTP-8587\n\n- Ssl has now switched default implementation and removed deprecated certificate\n  handling. All certificate handling is done by the public_key application.\n\n  Own Id: OTP-8695","title":"Improvements and New Features - SSL Release Notes","ref":"notes.html#improvements-and-new-features"},{"type":"extras","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n\n# TLS/DTLS Protocol Overview","title":"TLS/DTLS Protocol Overview","ref":"ssl_protocol.html"},{"type":"extras","doc":"Transport Layer Security (TLS) and its predecessor, the Secure Sockets Layer\n(SSL), are cryptographic protocols designed to provide communications security\nover a computer network. The protocols use X.509 certificates and hence public\nkey (asymmetric) cryptography to authenticate the counterpart with whom they\ncommunicate, and to exchange a symmetric key for payload encryption. The\nprotocol provides data/message confidentiality (encryption), integrity (through\nmessage authentication code checks) and host verification (through certificate\npath validation). DTLS (Datagram Transport Layer Security) that is based on TLS\nbut datagram oriented instead of stream oriented.\n\n# Erlang Support\n\nThe Erlang SSL application implements the TLS/DTLS protocol for the currently\nsupported versions, see the `m:ssl` manual page.\n\nBy default TLS is run over the TCP/IP protocol even though you can plug in any\nother reliable transport protocol with the same Application Programming\nInterface (API) as the `gen_tcp` module in Kernel. DTLS is by default run over\nUDP/IP, which means that application data has no delivery guarantees. Other\ntransports, such as SCTP, may be supported in future releases.\n\nIf a client and a server wants to use an upgrade mechanism, such as defined by\nRFC 2817, to upgrade a regular TCP/IP connection to a TLS connection, this is\nsupported by the Erlang SSL application API. This can be useful for, for\nexample, supporting HTTP and HTTPS on the same port and implementing virtual\nhosting. Note this is a TLS feature only.","title":"Purpose - TLS/DTLS Protocol Overview","ref":"ssl_protocol.html#purpose"},{"type":"extras","doc":"To achieve authentication and privacy, the client and server perform a TLS/DTLS\nhandshake procedure before transmitting or receiving any data. During the\nhandshake, they agree on a protocol version and cryptographic algorithms,\ngenerate shared secrets using public key cryptographies, and optionally\nauthenticate each other with digital certificates.","title":"Security Overview - TLS/DTLS Protocol Overview","ref":"ssl_protocol.html#security-overview"},{"type":"extras","doc":"A _symmetric key_ algorithm has one key only. The key is used for both\nencryption and decryption. These algorithms are fast, compared to public key\nalgorithms (using two keys, one public and one private) and are therefore\ntypically used for encrypting bulk data.\n\nThe keys for the symmetric encryption are generated uniquely for each connection\nand are based on a secret negotiated in the TLS/DTLS handshake.\n\nThe TLS/DTLS handshake protocol and data transfer is run on top of the TLS/DTLS\nRecord Protocol, which uses a keyed-hash Message Authenticity Code (MAC), or a\nHash-based MAC (HMAC), to protect the message data integrity. From the TLS RFC:\n\"A Message Authentication Code is a one-way hash computed from a message and\nsome secret data. It is difficult to forge without knowing the secret data. Its\npurpose is to detect if the message has been altered.\"","title":"Data Privacy and Integrity - TLS/DTLS Protocol Overview","ref":"ssl_protocol.html#data-privacy-and-integrity"},{"type":"extras","doc":"A certificate is similar to a driver's license, or a passport. The holder of the\ncertificate is called the _subject_. The certificate is signed with the private\nkey of the issuer of the certificate. A chain of trust is built by having the\nissuer in its turn being certified by another certificate, and so on, until you\nreach the so called root certificate, which is self-signed, that is, issued by\nitself.\n\nCertificates are issued by Certification Authorities (CAs) only. A handful of\ntop CAs in the world issue root certificates. You can examine several of these\ncertificates by clicking through the menus of your web browser.","title":"Digital Certificates - TLS/DTLS Protocol Overview","ref":"ssl_protocol.html#digital-certificates"},{"type":"extras","doc":"Authentication of the peer is done by public key path validation as defined in\nRFC 3280. This means basically the following:\n\n- Each certificate in the certificate chain is issued by the previous one.\n- The certificates attributes are valid.\n- The root certificate is a trusted certificate that is present in the trusted\n  certificate database kept by the peer.\n\nThe server always sends a certificate chain as part of the TLS handshake, but\nthe client only sends one if requested by the server. If the client does not\nhave an appropriate certificate, it can send an \"empty\" certificate to the\nserver.\n\nThe client can choose to accept some path evaluation errors, for example, a web\nbrowser can ask the user whether to accept an unknown CA root certificate. The\nserver, if it requests a certificate, does however not accept any path\nvalidation errors. It is configurable if the server is to accept or reject an\n\"empty\" certificate as response to a certificate request.","title":"Peer Authentication - TLS/DTLS Protocol Overview","ref":"ssl_protocol.html#peer-authentication"},{"type":"extras","doc":"From the TLS RFC: \"A TLS session is an association between a client and a\nserver. Sessions are created by the handshake protocol. Sessions define a set of\ncryptographic security parameters, which can be shared among multiple\nconnections. Sessions are used to avoid the expensive negotiation of new\nsecurity parameters for each connection.\"\n\nSession data is by default kept by the SSL application in a memory storage,\nhence session data is lost at application restart or takeover. Users can define\ntheir own callback module to handle session data storage if persistent data\nstorage is required. Session data is also invalidated when session database\nexceeds its limit or 24 hours after being saved (RFC max lifetime\nrecommendation). The amount of time the session data is to be saved can be\nconfigured.\n\nBy default the TLS/DTLS clients try to reuse an available session and by default\nthe TLS/DTLS servers agree to reuse sessions when clients ask for it. See also\n[Session Reuse Prior to TLS-1.3](using_ssl.md#session-reuse-prior-to-tls-1-3)","title":"TLS Sessions - Prior to TLS-1.3 - TLS/DTLS Protocol Overview","ref":"ssl_protocol.html#tls-sessions-prior-to-tls-1-3"},{"type":"extras","doc":"In TLS 1.3 the session reuse is replaced by a new session tickets mechanism\nbased on the prior to shared key concept. This mechanism also obsoletes the session\ntickets from RFC5077, not implemented by this application. See also\n[Session Tickets and Session Resumption in TLS-1.3](using_ssl.md#session-tickets-and-session-resumption-in-tls-1-3)","title":"TLS-1.3 session tickets - TLS/DTLS Protocol Overview","ref":"ssl_protocol.html#tls-1-3-session-tickets"},{"type":"extras","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Examples\n\nTo see relevant version information for ssl, call `ssl:versions/0` .\n\nTo see all supported cipher suites, call\n[`ssl:cipher_suites(all, 'tlsv1.3')` ](`ssl:cipher_suites/2`). The available\ncipher suites for a connection depend on the TLS version and prior to TLS-1.3 also on\nthe certificate. To see the default cipher suite list change `all` to `default`.\nNote that TLS 1.3 and previous versions do not have any cipher suites in common,\nfor listing cipher suites for a specific version use\n[`ssl:cipher_suites(exclusive, 'tlsv1.3')` ](`ssl:cipher_suites/2`). Specific\ncipher suites that you want your connection to use can also be specified.\nDefault is to use the strongest available.\n\n\n> #### Warning {: .warning }\n>Enabling cipher suites using RSA as a key exchange algorithm is\n>strongly discouraged (only available prior to TLS-1.3). For some\n>configurations software preventions may exist, and can make them usable if they work,\n>but relying on them to work is risky and there are many more reliable\n>cipher suites that can be used instead.\n\nThe following sections shows small examples of how to set up client/server\nconnections using the Erlang shell. The returned value of the `sslsocket` is\nabbreviated with `[...]` as it can be fairly large and is opaque to the user\nexcept for the purpose of pattern matching.\n\n> #### Note {: .info }\n>\n> Note that client certificate verification is optional for the server and needs\n> additional conguration on both sides to work. The Certificate and keys, in the\n> examples, are provided using the `t:ssl:cert_key_conf/0` supplied in the `certs_keys`\n> introduced in OTP 25.","title":"Examples","ref":"using_ssl.html"},{"type":"extras","doc":"```erlang\n 1 > ssl:start(), ssl:connect(\"google.com\", 443, [{verify, verify_peer},\n                                                 {cacerts, public_key:cacerts_get()}]).\n   {ok,{sslsocket, [...]}}\n```","title":"Basic Client - Examples","ref":"using_ssl.html#basic-client"},{"type":"extras","doc":"_Step 1:_ Start the server side:\n\n```erlang\n1 server> ssl:start().\nok\n```\n\n_Step 2:_ with alternative certificates, in this example the EDDSA certificate\nwill be preferred if TLS-1.3 is negotiated and the RSA certificate will always\nbe used for TLS-1.2 as it does not support the EDDSA algorithm:\n\n```erlang\n2 server> {ok, ListenSocket} =\nssl:listen(9999, [{certs_keys, [#{certfile => \"eddsacert.pem\",\n                                  keyfile => \"eddsakey.pem\"},\n                                #{certfile => \"rsacert.pem\",\n                                  keyfile => \"rsakey.pem\",\n                                  password => \"foobar\"}\n                               ]},{reuseaddr, true}]).\n{ok,{sslsocket, [...]}}\n```\n\n_Step 3:_ Do a transport accept on the TLS listen socket:\n\n```erlang\n3 server> {ok, TLSTransportSocket} = ssl:transport_accept(ListenSocket).\n{ok,{sslsocket, [...]}}\n```\n\n> #### Note {: .info }\n>\n> ssl:transport_accept/1 and ssl:handshake/2 are separate functions so that the\n> handshake part can be called in a new erlang process dedicated to handling the\n> connection\n\n_Step 4:_ Start the client side:\n\n```erlang\n1 client> ssl:start().\nok\n```\n\nBe sure to configure trusted certificates to use for server certificate\nverification.\n\n```erlang\n2 client> {ok, Socket} = ssl:connect(\"localhost\", 9999,\n      [{verify, verify_peer},\n      {cacertfile, \"cacerts.pem\"}, {active, once}], infinity).\n{ok,{sslsocket, [...]}}\n```\n\n_Step 5:_ Do the TLS handshake:\n\n```erlang\n4 server> {ok, Socket} = ssl:handshake(TLSTransportSocket).\n{ok,{sslsocket, [...]}}\n```\n\n> #### Note {: .info }\n>\n> A real server should use ssl:handshake/2 that has a timeout to avoid DoS\n> attacks. In the example the timeout defaults to infinty.\n\n_Step 6:_ Send a message over TLS:\n\n```erlang\n5 server> ssl:send(Socket, \"foo\").\nok\n```\n\n_Step 7:_ Flush the shell message queue to see that the message sent on the\nserver side is recived by the client side:\n\n```erlang\n3 client> flush().\nShell got {ssl,{sslsocket,[...]},\"foo\"}\nok\n```","title":"Basic Connection - Examples","ref":"using_ssl.html#basic-connection"},{"type":"extras","doc":"Upgrading a a TCP/IP connection to a TLS connections is mostly used when there\nis a desire have unencrypted communication first and then later secure the\ncommunication channel by using TLS. Note that the client and server need to\nagree to do the upgrade in the protocol doing the communication. This is concept\nis often referenced as `STARTLS` and used in many protocols such as `SMTP`,\n`FTPS` and `HTTPS` via a proxy.\n\n> #### Warning {: .warning }\n>\n> Maximum security recommendations are however moving away from such solutions.\n\nTo upgrade to a TLS connection:\n\n_Step 1:_ Start the server side:\n\n```erlang\n1 server> ssl:start().\n  ok\n```\n\n_Step 2:_ Create a normal TCP listen socket and ensure `active` is set to\n`false` and not set to any active mode otherwise TLS handshake messages can be\ndelivered to the wrong process.\n\n```erlang\n2 server> {ok, ListenSocket} = gen_tcp:listen(9999, [{reuseaddr, true},\n  {active, false}]).\n  {ok, #Port<0.475>}\n```\n\n_Step 3:_ Accept client connection:\n\n```erlang\n3 server> {ok, Socket} = gen_tcp:accept(ListenSocket).\n  {ok, #Port<0.476>}\n```\n\n_Step 4:_ Start the client side:\n\n```erlang\n1 client> ssl:start().\n  ok\n```\n\n```erlang\n2 client> {ok, Socket} = gen_tcp:connect(\"localhost\", 9999,  [], infinity).\n```\n\n_Step 5:_ Do the TLS handshake:\n\n```erlang\n4 server> {ok, TLSSocket} = ssl:handshake(Socket, [{verify, verify_peer},\n  {fail_if_no_peer_cert, true},\n  {cacertfile, \"cacerts.pem\"},\n  {certs_keys, [#{certfile => \"cert.pem\", keyfile => \"key.pem\"}]}]).\n  {ok,{sslsocket,[...]}}\n```\n\n_Step 6:_ Upgrade to a TLS connection. The client and server must agree upon the\nupgrade. The server must be prepared to be a TLS server before the client can do\na successful connect.\n\n```erlang\n3 client>{ok, TLSSocket} = ssl:connect(Socket, [{verify, verify_peer},\n  {cacertfile, \"cacerts.pem\"},\n  {certs_keys, [#{certfile => \"cert.pem\", keyfile => \"key.pem\"}]}], infinity).\n{ok,{sslsocket,[...]}}\n```\n\n_Step 7:_ Send a message over TLS:\n\n```erlang\n4 client> ssl:send(TLSSocket, \"foo\").\n      ok\n```\n\n_Step 8:_ Set `active once` on the TLS socket:\n\n```erlang\n5 server> ssl:setopts(TLSSocket, [{active, once}]).\n      ok\n```\n\n_Step 9:_ Flush the shell message queue to see that the message sent on the\nclient side is recived by the server side:\n\n```erlang\n5 server> flush().\n      Shell got {ssl,{sslsocket,[...]},\"foo\"}\n      ok\n```","title":"Upgrade Example - TLS only - Examples","ref":"using_ssl.html#upgrade-example-tls-only"},{"type":"extras","doc":"Fetch default cipher suite list for a TLS/DTLS version. Change default to all to\nget all possible cipher suites.\n\n```erlang\n1>  Default = ssl:cipher_suites(default, 'tlsv1.2').\n    [#{cipher => aes_256_gcm,key_exchange => ecdhe_ecdsa,\n    mac => aead,prf => sha384}, ....]\n```\n\nIn OTP 20 it is desirable to remove all cipher suites that uses rsa key exchange\n(removed from default in 21)\n\n```erlang\n2> NoRSA =\n    ssl:filter_cipher_suites(Default,\n                             [{key_exchange, fun(rsa) -> false;\n                                                (_) -> true\n                                             end}]).\n    [...]\n```\n\nPick just a few suites\n\n```erlang\n 3> Suites =\n ssl:filter_cipher_suites(Default,\n                             [{key_exchange, fun(ecdh_ecdsa) -> true;\n                                                (_) -> false\n                                             end},\n                              {cipher, fun(aes_128_cbc) -> true;\n                                          (_) ->false\n                                       end}]).\n\n[#{cipher => aes_128_cbc,key_exchange => ecdh_ecdsa,\n   mac => sha256,prf => sha256},\n #{cipher => aes_128_cbc,key_exchange => ecdh_ecdsa,mac => sha,\n   prf => default_prf}]\n```\n\nMake some particular suites the most preferred, or least preferred by changing\nprepend to append.\n\n```erlang\n 4>ssl:prepend_cipher_suites(Suites, Default).\n  [#{cipher => aes_128_cbc,key_exchange => ecdh_ecdsa,\n     mac => sha256,prf => sha256},\n   #{cipher => aes_128_cbc,key_exchange => ecdh_ecdsa,mac => sha,\n     prf => default_prf},\n   #{cipher => aes_256_cbc,key_exchange => ecdhe_ecdsa,\n     mac => sha384,prf => sha384}, ...]\n```","title":"Customizing cipher suites - Examples","ref":"using_ssl.html#customizing-cipher-suites"},{"type":"extras","doc":"Starting from TLS-1.2 signature algorithms (called signature schemes in TLS-1.3)\nis something that can be negotiated and hence also configured. These\nalgorithms/schemes will be used for digital signatures in protocol messages and\nin certificates.\n\n> #### Note {: .info }\n>\n> TLS-1.3 schemes have atom names whereas TLS-1.2 configuration is two element\n> tuples composed by one hash algorithm and one signature algorithm. When both\n> versions are supported the configuration can be a mix of these as both\n> versions might be negotiated. All `rsa_pss` based schemes are back ported to\n> TLS-1.2 and can be used also in a TLS-1.2 configuration. In TLS-1.2 the\n> signature algorithms chosen by the server will also be affected by the chiper\n> suite that is chosen, which is not the case in TLS-1.3.\n\nUsing the function `ssl:signature_algs/2` will let you inspect different aspects\nof possible configurations for your system. For example if TLS-1.3 and TLS-1.2\nis supported the default signature_algorithm list in OTP-26 and cryptolib from\nOpenSSL 3.0.2 would look like:\n\n```erlang\n 1>  ssl:signature_algs(default, 'tlsv1.3').\n %% TLS-1.3 schemes\n [eddsa_ed25519,eddsa_ed448,ecdsa_secp521r1_sha512,\n  ecdsa_secp384r1_sha384,ecdsa_secp256r1_sha256,\n  rsa_pss_pss_sha512,rsa_pss_pss_sha384,rsa_pss_pss_sha256,\n  rsa_pss_rsae_sha512,rsa_pss_rsae_sha384,rsa_pss_rsae_sha256,\n  %% Legacy schemes only valid for certificate signatures in TLS-1.3\n  %% (would have a tuple name in TLS-1.2 only configuration)\n  rsa_pkcs1_sha512,rsa_pkcs1_sha384,rsa_pkcs1_sha256\n  %% TLS 1.2 algorithms\n  {sha512,ecdsa},\n  {sha384,ecdsa},\n  {sha256,ecdsa}]\n```\n\nIf you want to add support for non default supported algorithms you should\nappend them to the default list as the configuration is in prefered order,\nsomething like this:\n\n```erlang\n    MySignatureAlgs = ssl:signature_algs(default, 'tlsv1.3') ++ [{sha, rsa}, {sha, dsa}],\n    ssl:connect(Host,Port,[{signature_algs, MySignatureAlgs,...]}),\n    ...\n```\n\nSee also `ssl:signature_algs/2` and [sign_algo()](`t:ssl:signature_algs/0`)","title":"Customizing signature algorithms(TLS-1.2)/schemes(TLS-1.3) - Examples","ref":"using_ssl.html#customizing-signature-algorithms-tls-1-2-schemes-tls-1-3"},{"type":"extras","doc":"Erlang ssl application is able to use private keys provided by OpenSSL engines\nusing the following mechanism:\n\n```erlang\n1> ssl:start().\nok\n```\n\nLoad a crypto engine, should be done once per engine used. For example\ndynamically load the engine called `MyEngine`:\n\n```erlang\n2> {ok, EngineRef} =\ncrypto:engine_load(<<\"dynamic\">>,\n[{<<\"SO_PATH\">>, \"/tmp/user/engines/MyEngine\"},<<\"LOAD\">>],\n[]).\n{ok,#Ref<0.2399045421.3028942852.173962>}\n```\n\nCreate a map with the engine information and the algorithm used by the engine:\n\n```erlang\n3> PrivKey =\n #{algorithm => rsa,\n   engine => EngineRef,\n   key_id => \"id of the private key in Engine\"}.\n```\n\nUse the map in the ssl key option:\n\n```erlang\n4> {ok, SSLSocket} =\n ssl:connect(\"localhost\", 9999,\n                [{cacertfile, \"cacerts.pem\"},\n                 {certs_keys, [#{certfile => \"cert.pem\", key => PrivKey}]}\n                ], infinity).\n\n```\n\nSee also [crypto documentation](`e:crypto:engine_load.md#engine_load`)","title":"Using an Engine Stored Key - Examples","ref":"using_ssl.html#using-an-engine-stored-key"},{"type":"extras","doc":"The NSS keylog debug feature can be used by authorized users to for instance\nenable wireshark to decrypt TLS packets.\n\n_Server (with NSS key logging)_\n\n```erlang\n    server() ->\n        application:load(ssl),\n        {ok, _} = application:ensure_all_started(ssl),\n        Port = 11029,\n        LOpts = [{certs_keys, [#{certfile => \"cert.pem\", keyfile => \"key.pem\"}]},\n        {reuseaddr, true},\n        {versions, ['tlsv1.2','tlsv1.3']},\n        {keep_secrets, true} %% Enable NSS key log (debug option)\n        ],\n        {ok, LSock} = ssl:listen(Port, LOpts),\n        {ok, ASock} = ssl:transport_accept(LSock),\n        {ok, CSock} = ssl:handshake(ASock).\n```\n\n_Exporting the secrets_\n\n```erlang\n      {ok, [{keylog, KeylogItems}]} = ssl:connection_information(CSock, [keylog]).\n      file:write_file(\"key.log\", [[KeylogItem,$\\n] || KeylogItem <- KeylogItems]).\n```","title":"NSS keylog - Examples","ref":"using_ssl.html#nss-keylog"},{"type":"extras","doc":"Clients can request to reuse a session established by a previous full handshake\nbetween that client and server by sending the id of the session in the initial\nhandshake message. The server may or may not agree to reuse it. If agreed the\nserver will send back the id and if not it will send a new id. The ssl\napplication has several options for handling session reuse.\n\nOn the client side the ssl application will save session data to try to automate\nsession reuse on behalf of the client processes on the Erlang node. Note that\nonly verified sessions will be saved for security reasons, that is session\nresumption relies on the certificate validation to have been run in the original\nhandshake. To minimize memory consumption only unique sessions will be saved\nunless the special `save` value is specified for the following option\n`{reuse_sessions, boolean() | save}` in which case a full handshake will be\nperformed and that specific session will have been saved before the handshake\nreturns. The session id and even an opaque binary containing the session data\ncan be retrieved using `ssl:connection_information/1` function. A saved session\n(guaranteed by the save option) can be explicitly reused using\n`{reuse_session, SessionId}`. Also it is possible for the client to reuse a\nsession that is not saved by the ssl application using\n`{reuse_session, {SessionId, SessionData}}`.\n\n> #### Note {: .info }\n>\n> When using explicit session reuse, it is up to the client to make sure that\n> the session being reused is for the correct server and has been verified.\n\nHere follows a client side example, divide into several steps for readability.\n\nStep 1 - Automated Session Reuse\n\n```erlang\n1> ssl:start().\nok\n\n2>{ok, C1} = ssl:connect(\"localhost\", 9999, [{verify, verify_peer},\n                                           {versions, ['tlsv1.2']},\n                                           {cacertfile, \"cacerts.pem\"}]).\n{ok,{sslsocket,{gen_tcp,#Port<0.7>,tls_connection,undefined}, ...}}\n\n3> ssl:connection_information(C1, [session_id]).\n{ok,[{session_id,<<95,32,43,22,35,63,249,22,26,36,106,\n                   152,49,52,124,56,130,192,137,161,\n                   146,145,164,232,...>>}]}\n\n%% Reuse session if possible, note that if C2 is really fast the session\n%% data might not be available for reuse.\n4>{ok, C2} = ssl:connect(\"localhost\", 9999, [{verify, verify_peer},\n                                           {versions, ['tlsv1.2']},\n                                           {cacertfile, \"cacerts.pem\"},\n                                           {reuse_sessions, true}]).\n{ok,{sslsocket,{gen_tcp,#Port<0.8>,tls_connection,undefined}, ...]}}\n\n%% C2 got same session ID as client one, session was automatically reused.\n5> ssl:connection_information(C2, [session_id]).\n{ok,[{session_id,<<95,32,43,22,35,63,249,22,26,36,106,\n                   152,49,52,124,56,130,192,137,161,\n                   146,145,164,232,...>>}]}\n```\n\nStep 2- Using `save` Option\n\n```erlang\n%% We want save this particular session for\n%% reuse although it has the same basis as C1\n6> {ok, C3} = ssl:connect(\"localhost\", 9999, [{verify, verify_peer},\n                                           {versions, ['tlsv1.2']},\n                                           {cacertfile, \"cacerts.pem\"},\n                                           {reuse_sessions, save}]).\n\n{ok,{sslsocket,{gen_tcp,#Port<0.9>,tls_connection,undefined}, ...]}}\n\n%% A full handshake is performed and we get a new session ID\n7> {ok, [{session_id, ID}]} = ssl:connection_information(C3, [session_id]).\n{ok,[{session_id,<<91,84,27,151,183,39,84,90,143,141,\n                   121,190,66,192,10,1,27,192,33,95,78,\n                   8,34,180,...>>}]}\n\n%% Use automatic session reuse\n8> {ok, C4} = ssl:connect(\"localhost\", 9999, [{verify, verify_peer},\n                                           {versions, ['tlsv1.2']},\n                                           {cacertfile, \"cacerts.pem\"},\n                                           {reuse_sessions, true}]).\n\n{ok,{sslsocket,{gen_tcp,#Port<0.10>,tls_connection,\n                        undefined}, ...]}}\n\n%% The \"saved\" one happened to be selected, but this is not a guarantee\n9> ssl:connection_information(C4, [session_id]).\n{ok,[{session_id,<<91,84,27,151,183,39,84,90,143,141,\n                   121,190,66,192,10,1,27,192,33,95,78,\n                   8,34,180,...>>}]}\n\n%% Make sure to reuse the \"saved\" session\n10> {ok, C5} = ssl:connect(\"localhost\", 9999, [{verify, verify_peer},\n                                           {versions, ['tlsv1.2']},\n                                           {cacertfile, \"cacerts.pem\"},\n                                           {reuse_session, ID}]).\n{ok,{sslsocket,{gen_tcp,#Port<0.11>,tls_connection,\n                        undefined}, ...]}}\n\n11> ssl:connection_information(C5, [session_id]).\n{ok,[{session_id,<<91,84,27,151,183,39,84,90,143,141,\n                   121,190,66,192,10,1,27,192,33,95,78,\n                   8,34,180,...>>}]}\n```\n\nStep 3 - Explicit Session Reuse\n\n```erlang\n%% Perform a full handshake and the session will not be saved for reuse\n12> {ok, C9} =\nssl:connect(\"localhost\", 9999, [{verify, verify_peer},\n                                    {versions, ['tlsv1.2']},\n                                    {cacertfile, \"cacerts.pem\"},\n                                    {reuse_sessions, false},\n                                    {server_name_indication, disable}]).\n{ok,{sslsocket,{gen_tcp,#Port<0.14>,tls_connection, ...}}\n\n%% Fetch session ID and data for C9 connection\n12> {ok, [{session_id, ID1}, {session_data, SessData}]} =\n        ssl:connection_information(C9, [session_id, session_data]).\n{ok,[{session_id,<<9,233,4,54,170,88,170,180,17,96,202,\n                   85,85,99,119,47,9,68,195,50,120,52,\n                   130,239,...>>},\n     {session_data,<<131,104,13,100,0,7,115,101,115,115,105,\n                     111,110,109,0,0,0,32,9,233,4,54,170,...>>}]}\n\n%% Explicitly reuse the session from C9\n13> {ok, C10} = ssl:connect(\"localhost\", 9999, [{verify, verify_peer},\n                                    {versions, ['tlsv1.2']},\n                                    {cacertfile, \"cacerts.pem\"},\n                                    {reuse_session, {ID1, SessData}}]).\n{ok,{sslsocket,{gen_tcp,#Port<0.15>,tls_connection,\n                        undefined}, ...}}\n\n14> ssl:connection_information(C10, [session_id]).\n{ok,[{session_id,<<9,233,4,54,170,88,170,180,17,96,202,\n                   85,85,99,119,47,9,68,195,50,120,52,\n                   130,239,...>>}]}\n```\n\nStep 4 - Not Possible to Reuse Explicit Session by ID Only\n\n```erlang\n%% Try to reuse the session from C9 using only the id\n15> {ok, E} =  ssl:connect(\"localhost\", 9999, [{verify, verify_peer},\n                                    {versions, ['tlsv1.2']},\n                                    {cacertfile, \"cacerts.pem\"},\n                                    {reuse_session, ID1}]).\n{ok,{sslsocket,{gen_tcp,#Port<0.18>,tls_connection,\n                        undefined}, ...}}\n\n%% This will fail (as it is not saved for reuse)\n%% and a full handshake will be performed, we get a new id.\n16>  ssl:connection_information(E, [session_id]).\n{ok,[{session_id,<<87,46,43,126,175,68,160,153,37,29,\n                   196,240,65,160,254,88,65,224,18,63,\n                   18,17,174,39,...>>}]}\n```\n\nOn the server side the the `{reuse_sessions, boolean()}` option determines if\nthe server will save session data and allow session reuse or not. This can be\nfurther customized by the option `{reuse_session, fun()}` that may introduce a\nlocal policy for session reuse.","title":"Session Reuse Prior to TLS 1.3 - Examples","ref":"using_ssl.html#session-reuse-prior-to-tls-1-3"},{"type":"extras","doc":"TLS 1.3 introduces a new secure way of resuming sessions by using session\ntickets. A session ticket is an opaque data structure that is sent in the\npre_shared_key extension of a ClientHello, when a client attempts to resume a\nsession with keying material from a previous successful handshake.\n\nSession tickets can be stateful or stateless. A stateful session ticket is a\ndatabase reference (session ticket store) and used with stateful servers, while\na stateless ticket is a self-encrypted and self-authenticated data structure\nwith cryptographic keying material and state data, enabling session resumption\nwith stateless servers.\n\nThe choice between stateful or stateless depends on the server requirements as\nthe session tickets are opaque for the clients. Generally, stateful tickets are\nsmaller and the server can guarantee that tickets are only used once. Stateless\ntickets contain additional data, require less storage on the server side, but\nthey offer different guarantees against anti-replay. See also\n[Anti-Replay Protection in TLS 1.3](using_ssl.md#anti-replay-protection-in-tls-1-3)\n\nSession tickets are sent by servers on newly established TLS connections. The\nnumber of tickets sent and their lifetime are configurable by application\nvariables. See also [SSL's configuration](ssl_app.md#configuration).\n\nSession tickets are protected by application traffic keys, and in stateless\ntickets, the opaque data structure itself is self-encrypted.\n\nAn example with automatic and manual session resumption:\n\n```erlang\n      {ok, _} = application:ensure_all_started(ssl).\n      LOpts = [{certs_keys, [#{certfile => \"cert.pem\",\n                               keyfile => \"key.pem\"}]},\n               {versions, ['tlsv1.2','tlsv1.3']},\n               {session_tickets, stateless}].\n      {ok, LSock} = ssl:listen(8001, LOpts).\n      {ok, ASock} = ssl:transport_accept(LSock).\n```\n\n_Step 2 (client):_ Start the client and connect to server:\n\n```erlang\n      {ok, _} = application:ensure_all_started(ssl).\n      COpts = [{cacertfile, \"cert.pem\"},\n               {versions, ['tlsv1.2','tlsv1.3']},\n               {log_level, debug},\n               {session_tickets, auto}].\n      ssl:connect(\"localhost\", 8001, COpts).\n```\n\n_Step 3 (server):_ Start the TLS handshake:\n\n```erlang\n      {ok, CSocket} = ssl:handshake(ASock).\n```\n\nA connection is established using a full handshake. Below is a summary of the\nexchanged messages:\n\n```erlang\n      >>> TLS 1.3 Handshake, ClientHello ...\n      << >> Handshake, Finished ...\n      << >> TLS 1.3 Handshake, ClientHello ...\n      << >> Handshake, Finished ...\n      <<  TicketData end.\n```\n\n_Step 11 (server):_ Accept a new connection on the server:\n\n```erlang\n      {ok, ASock4} = ssl:transport_accept(LSock).\n```\n\n_Step 12 (client):_ Initiate a new connection to the server with the session\nticket received in Step 10:\n\n```erlang\n      {ok, _} = application:ensure_all_started(ssl).\n      COpts2 = [{cacertfile, \"cert.pem\"},\n                {versions, ['tlsv1.2','tlsv1.3']},\n                {log_level, debug},\n                {session_tickets, manual},\n                {use_ticket, [Ticket]}].\n      ssl:connect(\"localhost\", 8001, COpts).\n```\n\n_Step 13 (server):_ Start the handshake:\n\n```erlang\n      {ok, CSock4} = ssl:handshake(ASock4).\n```","title":"Session Tickets and Session Resumption in TLS 1.3 - Examples","ref":"using_ssl.html#session-tickets-and-session-resumption-in-tls-1-3"},{"type":"extras","doc":"TLS 1.3 allows clients to send data on the first flight if the endpoints have a\nshared crypographic secret (pre-shared key). This means that clients can send\nearly data if they have a valid session ticket received in a previous successful\nhandshake. For more information about session resumption see\n[Session Tickets and Session Resumption in TLS 1.3](using_ssl.md#session-tickets-and-session-resumption-in-tls-1-3).\n\nThe security properties of Early Data are weaker than other kinds of TLS data.\nThis data is not forward secret, and it is vulnerable to replay attacks. For\navailable mitigation strategies see\n[Anti-Replay Protection in TLS 1.3](using_ssl.md#anti-replay-protection-in-tls-1-3).\n\nIn normal operation, clients will not know which, if any, of the available\nmitigation strategies servers actually implement, and hence must only send early\ndata which they deem safe to be replayed. For example, idempotent HTTP\noperations, such as HEAD and GET, can usually be regarded as safe but even they\ncan be exploited by a large number of replays causing resource limit exhaustion\nand other similar problems.\n\nAn example of sending early data with automatic and manual session ticket\nhandling:\n\n_Server_\n\n```erlang\nearly_data_server() ->\n    application:load(ssl),\n    {ok, _} = application:ensure_all_started(ssl),\n    Port = 11029,\n    LOpts = [{certs_keys, [#{certfile => \"cert.pem\", keyfile => \"key.pem\"}]},\n             {reuseaddr, true},\n             {versions, ['tlsv1.2','tlsv1.3']},\n             {session_tickets, stateless},\n             {early_data, enabled},\n            ],\n    {ok, LSock} = ssl:listen(Port, LOpts),\n    %% Accept first connection\n    {ok, ASock0} = ssl:transport_accept(LSock),\n    {ok, CSock0} = ssl:handshake(ASock0),\n    %% Accept second connection\n    {ok, ASock1} = ssl:transport_accept(LSock),\n    {ok, CSock1} = ssl:handshake(ASock1),\n    Sock.\n```\n\n_Client (automatic ticket handling):_\n\n```erlang\nearly_data_auto() ->\n    %% First handshake 1-RTT - get session tickets\n    application:load(ssl),\n    {ok, _} = application:ensure_all_started(ssl),\n    Port = 11029,\n    Data = <<\"HEAD / HTTP/1.1\\r\\nHost: \\r\\nConnection: close\\r\\n\">>,\n    COpts0 = [{cacertfile, \"cacerts.pem\"},\n              {versions, ['tlsv1.2', 'tlsv1.3']},\n              {session_tickets, auto}],\n    {ok, Sock0} = ssl:connect(\"localhost\", Port, COpts0),\n\n    %% Wait for session tickets\n    timer:sleep(500),\n    %% Close socket if server cannot handle multiple\n    %% connections e.g. openssl s_server\n    ssl:close(Sock0),\n\n    %% Second handshake 0-RTT\n    COpts1 = [{cacertfile,  \"cacerts.pem\"},\n              {versions, ['tlsv1.2', 'tlsv1.3']},\n              {session_tickets, auto},\n              {early_data, Data}],\n    {ok, Sock} = ssl:connect(\"localhost\", Port, COpts1),\n    Sock.\n\n```\n\n_Client (manual ticket handling):_\n\n```erlang\nearly_data_manual() ->\n    %% First handshake 1-RTT - get session tickets\n    application:load(ssl),\n    {ok, _} = application:ensure_all_started(ssl),\n    Port = 11029,\n    Data = <<\"HEAD / HTTP/1.1\\r\\nHost: \\r\\nConnection: close\\r\\n\">>,\n    COpts0 = [{cacertfile, \"cacerts.pem\"},\n              {versions, ['tlsv1.2', 'tlsv1.3']},\n              {session_tickets, manual}],\n    {ok, Sock0} = ssl:connect(\"localhost\", Port, COpts0),\n\n    %% Wait for session tickets\n    Ticket =\n        receive\n            {ssl, session_ticket, Ticket0} ->\n                Ticket0\n        end,\n\n    %% Close socket if server cannot handle multiple connections\n    %% e.g. openssl s_server\n    ssl:close(Sock0),\n\n    %% Second handshake 0-RTT\n    COpts1 = [{cacertfile, \"cacerts.pem\"},\n              {versions, ['tlsv1.2', 'tlsv1.3']},\n              {session_tickets, manual},\n              {use_ticket, [Ticket]},\n              {early_data, Data}],\n    {ok, Sock} = ssl:connect(\"localhost\", Port, COpts1),\n    Sock.\n```","title":"Early Data in TLS-1.3 - Examples","ref":"using_ssl.html#early-data-in-tls-1-3"},{"type":"extras","doc":"The TLS 1.3 protocol does not provide inherent protection for replay of 0-RTT\ndata but describes mechanisms that SHOULD be implemented by compliant server\nimplementations. The implementation of TLS 1.3 in the SSL application employs\nall standard methods to prevent potential threats.\n\n_Single-use tickets_\n\nThis mechanism is available with stateful session tickets. Session tickets can\nonly be used once, subsequent use of the same ticket results in a full\nhandshake. Stateful servers enforce this rule by maintaining a database of\noutstanding valid tickets.\n\n_Client Hello Recording_\n\nThis mechanism is available with stateless session tickets. The server records a\nunique value derived from the ClientHello (PSK binder) in a given time window.\nThe ticket's age is verified by using both the \"obsfuscated_ticket_age\" and an\nadditional timestamp encrypted in the ticket data. As the used datastore allows\nfalse positives, apparent replays will be answered by doing a full 1-RTT\nhandshake.\n\n_Freshness Checks_\n\nThis mechanism is available with the stateless session tickets. As the ticket\ndata has an embedded timestamp, the server can determine if a ClientHello was\nsent reasonably recently and accept the 0-RTT handshake, otherwise if falls back\nto a full 1-RTT handshake. This mechanism is tightly coupled with the previous\none, it prevents storing an unlimited number of ClientHellos.\n\nThe current implementation uses a pair of Bloom filters to implement the last\ntwo mechanisms. Bloom filters are fast, memory-efficient, probabilistic data\nstructures that can tell if an element may be in a set or if it is definitely\nnot in the set.\n\nIf the option `anti_replay` is defined in the server, a\npair of Bloom filters (_current_ and _old_) are used to record incoming\nClientHello messages (it is the unique binder value that is actually stored).\nThe _current_ Bloom filter is used for `WindowSize` seconds to store new\nelements. At the end of the time window the Bloom filters are rotated (the\n_current_ Bloom filter becomes the _old_ and an empty Bloom filter is set as\n_current_.\n\nThe Anti-Replay protection feature in stateless servers executes in the\nfollowing steps when a new ClientHello is received:\n\n- Reported ticket age (obfuscated ticket age) shall be less than ticket\n  lifetime.\n- Actual ticket age shall be less than the ticket lifetime (stateless session\n  tickets contain the servers timestamp when the ticket was issued).\n- ClientHello created with the ticket shall be sent relatively recently\n  (freshness checks).\n- If all above checks passed both _current_ and _old_ Bloom filters are checked\n  to detect if binder was already seen. Being a probabilistic data structure,\n  false positives can occur and they trigger a full handshake.\n- If the binder is not seen, the binder is validated. If the binder is valid,\n  the server proceeds with the 0-RTT handshake.","title":"Anti-Replay Protection in TLS 1.3 - Examples","ref":"using_ssl.html#anti-replay-protection-in-tls-1-3"},{"type":"extras","doc":"Using DTLS has basically the same API as TLS. You need to add the option\n\\{protocol, dtls\\} to the connect and listen functions. For example\n\n```erlang\n client>{ok, Socket} = ssl:connect(\"localhost\", 9999, [{protocol, dtls},\n                                               {verify, verify_peer},\n                                               {cacertfile, \"cacerts.pem\"}],\n                           infinity).\n{ok,{sslsocket, [...]}}\n\n```","title":"Using DTLS - Examples","ref":"using_ssl.html#using-dtls"},{"type":"extras","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Erlang Distribution over TLS\n\nThis section describes how the Erlang distribution can use TLS to get extra\nverification and security.\n\nThe Erlang distribution can in theory use almost any connection-based protocol\nas bearer. However, a module that implements the protocol-specific parts of the\nconnection setup is needed. The default distribution module is `inet_tcp_dist`\nin the Kernel application. When starting an Erlang node distributed,\n`net_kernel` uses this module to set up listen ports and connections.\n\nIn the SSL application, an extra distribution module, `inet_tls_dist`, can be\nused as an alternative. All distribution connections will use TLS and all\nparticipating Erlang nodes in a distributed system must use this distribution\nmodule.\n\nThe security level depends on the parameters provided to the TLS connection\nsetup. Erlang node cookies are however always used, as they can be used to\ndifferentiate between two different Erlang networks.\n\nTo set up Erlang distribution over TLS:\n\n- _Step 1:_ Build boot scripts including the SSL application.\n- _Step 2:_ Specify the distribution module for `net_kernel`.\n- _Step 3:_ Specify the security options and other SSL options.\n- _Step 4:_ Set up the environment to always use TLS.\n\nThe following sections describe these steps.","title":"Erlang Distribution over TLS","ref":"ssl_distribution.html"},{"type":"extras","doc":"Boot scripts are built using the `systools` utility in the SASL application. For\nmore information on `systools`, see the SASL documentation. This is only an\nexample of what can be done.\n\nThe simplest boot script possible includes only the Kernel and STDLIB\napplications. Such a script is located in the `bin` directory of the Erlang\ndistribution. The source for the script is found under the Erlang installation\ntop directory under `releases/ /start_clean.rel`.\n\nDo the following:\n\n- Copy that script to another location (and preferably another name).\n- Add the applications Crypto, Public Key, and SSL with their current version\n  numbers after the STDLIB application.\n\nThe following shows an example `.rel` file with TLS added:\n\n```erlang\n      {release, {\"OTP  APN 181 01\",\"R15A\"}, {erts, \"5.9\"},\n      [{kernel,\"2.15\"},\n      {stdlib,\"1.18\"},\n      {crypto, \"2.0.3\"},\n      {public_key, \"0.12\"},\n      {asn1, \"4.0\"},\n      {ssl, \"5.0\"}\n      ]}.\n```\n\nThe version numbers differ in your system. Whenever one of the applications\nincluded in the script is upgraded, change the script.\n\nDo the following:\n\n- Build the boot script.\n\n  Assuming the `.rel file` is stored in a file `start_ssl.rel` in the current\n  directory, a boot script can be built as follows:\n\n```text\n   1> systools:make_script(\"start_ssl\",[]).\n```\n\nThere is now a `start_ssl.boot` file in the current directory.\n\nDo the following:\n\n- Test the boot script. To do this, start Erlang with the `-boot` command-line\n  parameter specifying this boot script (with its full path, but without the\n  `.boot` suffix). In UNIX it can look as follows:\n\n```text\n$ erl -boot /home/me/ssl/start_ssl\nErlang (BEAM) emulator version 5.0\n\nEshell V5.0  (abort with ^G)\n1> whereis(ssl_manager).\n<0.41.0>\n```\n\nThe `whereis` function-call verifies that the SSL application is started.\n\nAs an alternative to building a bootscript, you can explicitly add the path to\nthe SSL `ebin` directory on the command line. This is done with command-line\noption `-pa`. This works as the SSL application does not need to be started for\nthe distribution to come up, as a clone of the SSL application is hooked into\nthe Kernel application. So, as long as the SSL application code can be reached,\nthe distribution starts. The `-pa` method is only recommended for testing\npurposes.\n\n> #### Note {: .info }\n>\n> The clone of the SSL application must enable the use of the SSL code in such\n> an early bootstage as needed to set up the distribution. However, this makes\n> it impossible to soft upgrade the SSL application.","title":"Building Boot Scripts Including the SSL Application - Erlang Distribution over TLS","ref":"ssl_distribution.html#building-boot-scripts-including-the-ssl-application"},{"type":"extras","doc":"The distribution module for TLS is named `inet_tls_dist` and is specified on the\ncommand line with option `-proto_dist`. The argument to `-proto_dist` is to be\nthe module name without suffix `_dist`. So, this distribution module is\nspecified with `-proto_dist inet_tls` on the command line.\n\nExtending the command line gives the following:\n\n```text\n$ erl -boot /home/me/ssl/start_ssl -proto_dist inet_tls\n```\n\nFor the distribution to be started, give the emulator a name as well:\n\n```text\n$ erl -boot /home/me/ssl/start_ssl -proto_dist inet_tls -sname ssl_test\nErlang (BEAM) emulator version 5.0 [source]\n\nEshell V5.0  (abort with ^G)\n(ssl_test@myhost)1>\n```\n\nHowever, a node started in this way refuses to talk to other nodes, as no TLS\nparameters are supplied (see the next section).","title":"Specifying Distribution Module for net_kernel - Erlang Distribution over TLS","ref":"ssl_distribution.html#specifying-distribution-module-for-net_kernel"},{"type":"extras","doc":"The TLS distribution options can be written into a file that is consulted when\nthe node is started. This file name is then specified with the command line\nargument `-ssl_dist_optfile`.\n\nAny available TLS option can be specified in an options file.\n\n> #### Note {: .info }\nOptions that take a `fun()` has to use the syntax `fun Mod:Func/Arity` since a\nfunction body cannot be compiled when consulting a file. Also the encoding\nof the file can be specified as defined by module `m:epp`.\n\n> #### Warning {: .warning }\nDo not tamper with the socket options `list`, `binary`, `active`, `packet`,\n`nodelay` and `deliver` since they are used by the distribution protocol handler\nitself. Other raw socket options such as `packet_size` may interfere severely,\nso beware\\!\n\nFor TLS to work, at least a public key and a certificate must be specified for\nthe server side and the client needs to specify CAs that it trusts (client certification\nis optional and requires more configuration).\n\nIn the following example (to keep it simple), the PEM file `\"/home/me/ssl/erlserver.pem\"`\ncontains both the server certificate and its private key .\n\nCreate a file named for example `\"/home/me/ssl/ssl_test@myhost.conf\"`:\n\n```erlang\n[{server,\n  [{certfile, \"/home/me/ssl/erlserver.pem\"}]},\n {client,\n  [{cacertfile, \"/home/me/ssl/client_trusted.pem\"}]}].\n```\n\nAnd then start the node like this (line breaks in the command are for\nreadability, and shall not be there when typed):\n\n```text\n$ erl -boot /home/me/ssl/start_ssl -proto_dist inet_tls\n  -ssl_dist_optfile \"/home/me/ssl/ssl_test@myhost.conf\"\n  -sname ssl_test\n```\n\nThe options in the `{server, Opts}` tuple are used when calling\n`ssl:handshake/3`, and the options in the `{client, Opts}` tuple are used when\ncalling `ssl:connect/4`.\n\nFor the client, the option `{server_name_indication, atom_to_list(TargetNode)}`\nis added when connecting. This makes it possible to use the client option\n`{verify, verify_peer}`, and the client will verify that the certificate matches\nthe node name you are connecting to. This only works if the the server\ncertificate is issued to the name\n[`atom_to_list(TargetNode)`](`atom_to_list/1`).\n\nFor the server it is also possible to use the option `{verify, verify_peer}` and\nthe server will only accept client connections with certificates that are\ntrusted by a root certificate that the server knows. A client that presents an\nuntrusted certificate will be rejected. This option is preferably combined with\n`{fail_if_no_peer_cert, true}` or a client will still be accepted if it does not\npresent any certificate.\n\nA node started in this way is fully functional, using TLS as the distribution\nprotocol.","title":"Specifying TLS Options - Erlang Distribution over TLS","ref":"ssl_distribution.html#specifying-tls-options"},{"type":"extras","doc":"It is possible to use TLS distribution over IPv6 instead of IPv4. To do this,\npass the option `-proto_dist inet6_tls` instead of `-proto_dist inet_tls` when\nstarting Erlang, either on the command line or in the `ERL_FLAGS` environment\nvariable.\n\nAn example command line with this option would look like this:\n\n```text\n$ erl -boot /home/me/ssl/start_ssl -proto_dist inet6_tls\n  -ssl_dist_optfile \"/home/me/ssl/ssl_test@myhost.conf\"\n  -sname ssl_test\n```\n\nA node started in this way will only be able to communicate with other nodes\nusing TLS distribution over IPv6.","title":"Using TLS distribution over IPv6 - Erlang Distribution over TLS","ref":"ssl_distribution.html#using-tls-distribution-over-ipv6"},{"type":"extras","doc":"> #### Note {: .info }\n> The following section describes TLS Option handling prior to OTP 20.2\n> and can only handle a small subset of the actual available options.\n> It is here only for the sake of backwards compatibility .\n\nAs in the previous section the PEM file `\"/home/me/ssl/erlserver.pem\"` contains\nboth the server certificate and its private key.\n\nOn the `erl` command line you can specify options that the TLS distribution adds\nwhen creating a socket.\n\nThe simplest TLS options in the following list can be specified by adding the\nprefix `server_` or `client_` to the option name:\n\n- `certfile`\n- `keyfile`\n- `password`\n- `cacertfile`\n- `verify`\n- `verify_fun` (write as `{Module, Function, InitialUserState}`)\n- `crl_check`\n- `crl_cache` (write as Erlang term)\n- `reuse_sessions`\n- `secure_renegotiate`\n- `depth`\n- `hibernate_after`\n- `ciphers` (use old string format)\n\nNote that `verify_fun` needs to be written in a different form than the\ncorresponding TLS option, since funs are not accepted on the command line.\n\nThe server can also take the options `dhfile` and `fail_if_no_peer_cert` (also\nprefixed).\n\n`client_`\\-prefixed options are used when the distribution initiates a\nconnection to another node. `server_`\\-prefixed options are used when accepting\na connection from a remote node.\n\nRaw socket options, such as `packet` and `size` must not be specified on the\ncommand line.\n\nThe command-line argument for specifying the TLS options is named\n`-ssl_dist_opt` and is to be followed by pairs of SSL options and their values.\nArgument `-ssl_dist_opt` can be repeated any number of times.\n\nAn example command line doing the same as the example in the previous section\ncan now look as follows (line breaks in the command are for readability, and\nshall not be there when typed):\n\n```text\n$ erl -boot /home/me/ssl/start_ssl -proto_dist inet_tls\n  -ssl_dist_opt server_certfile \"/home/me/ssl/erlserver.pem\"\n  -ssl_dist_opt server_secure_renegotiate true client_secure_renegotiate true\n  -sname ssl_test\nErlang (BEAM) emulator version 5.0 [source]\n\nEshell V5.0  (abort with ^G)\n(ssl_test@myhost)1>\n```","title":"Specifying TLS Options (Legacy) - Erlang Distribution over TLS","ref":"ssl_distribution.html#specifying-tls-options-legacy"},{"type":"extras","doc":"A convenient way to specify arguments to Erlang is to use environment variable\n`ERL_FLAGS`. All the flags needed to use the TLS distribution can be specified\nin that variable and are then interpreted as command-line arguments for all\nsubsequent invocations of Erlang.\n\nIn a Unix (Bourne) shell, it can look as follows (line breaks are for\nreadability, they are not to be there when typed):\n\n```erlang\n$ ERL_FLAGS=\"-boot /home/me/ssl/start_ssl -proto_dist inet_tls\n  -ssl_dist_opt server_certfile /home/me/ssl/erlserver.pem\n  -ssl_dist_opt server_secure_renegotiate true client_secure_renegotiate true\"\n$ export ERL_FLAGS\n$ erl -sname ssl_test\nErlang (BEAM) emulator version 5.0 [source]\n\nEshell V5.0  (abort with ^G)\n(ssl_test@myhost)1> init:get_arguments().\n[{root,[\"/usr/local/erlang\"]},\n {progname,[\"erl \"]},\n {sname,[\"ssl_test\"]},\n {boot,[\"/home/me/ssl/start_ssl\"]},\n {proto_dist,[\"inet_tls\"]},\n {ssl_dist_opt,[\"server_certfile\",\"/home/me/ssl/erlserver.pem\"]},\n {ssl_dist_opt,[\"server_secure_renegotiate\",\"true\",\n                \"client_secure_renegotiate\",\"true\"]\n {home,[\"/home/me\"]}]\n```\n\nThe `init:get_arguments()` call verifies that the correct arguments are supplied\nto the emulator.","title":"Setting up Environment to Always Use TLS - Erlang Distribution over TLS","ref":"ssl_distribution.html#setting-up-environment-to-always-use-tls"},{"type":"extras","doc":"<!--\n%CopyrightBegin%\n\nCopyright Ericsson AB 2023-2024. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Standards Compliance","title":"Standards Compliance","ref":"standards_compliance.html"},{"type":"extras","doc":"This section describes the current state of standards compliance of the ssl\napplication.","title":"Purpose - Standards Compliance","ref":"standards_compliance.html#purpose"},{"type":"extras","doc":"- For security reasons RSA key exchange cipher suites are no longer supported by\n  default, but can be configured. (OTP 21)\n- For security reasons DES cipher suites are no longer supported by default, but\n  can be configured. (OTP 20)\n- For security reasons 3DES cipher suites are no longer supported by default,\n  but can be configured. (OTP 21)\n- Renegotiation Indication Extension\n  [RFC 5746](http://www.ietf.org/rfc/rfc5746.txt) is supported\n- Ephemeral Diffie-Hellman cipher suites are supported, but not Diffie Hellman\n  Certificates cipher suites.\n- Elliptic Curve cipher suites are supported if the Crypto application supports\n  it and named curves are used.\n- Export cipher suites are not supported as the U.S. lifted its export\n  restrictions in early 2000.\n- IDEA cipher suites are not supported as they have become deprecated by the TLS\n  1.2 specification so it is not motivated to implement them.\n- Compression is not supported.\n- It is possible to use Pre-Shared Key (PSK) and Secure Remote Password (SRP)\n  cipher suites, but they are not enabled by default and need addition configuration.","title":"Common (prior to TLS 1.3) - Standards Compliance","ref":"standards_compliance.html#common-prior-to-tls-1-3"},{"type":"extras","doc":"- CRL validation is supported.\n- Policy certificate extensions are supported. (OTP 27)\n- 'Server Name Indication' extension\n  ([RFC 6066](http://www.ietf.org/rfc/rfc6066.txt)) is supported.\n- Application Layer Protocol Negotiation (ALPN) and its successor Next Protocol\n  Negotiation (NPN) are supported.","title":"Common - Standards Compliance","ref":"standards_compliance.html#common"},{"type":"extras","doc":"For security reasons SSL-2.0 is not supported. Interoperability with SSL-2.0\nenabled clients dropped. (OTP 21)","title":"SSL 2.0 - Standards Compliance","ref":"standards_compliance.html#ssl-2-0"},{"type":"extras","doc":"For security reasons SSL-3.0 is no longer supported at all. (OTP 23)\n\nFor security reasons SSL-3.0 is no longer supported by default, but can be\nconfigured. (OTP 19)","title":"SSL 3.0 - Standards Compliance","ref":"standards_compliance.html#ssl-3-0"},{"type":"extras","doc":"For security reasons TLS-1.0 is no longer supported by default, but can be\nconfigured. (OTP 22)","title":"TLS 1.0 - Standards Compliance","ref":"standards_compliance.html#tls-1-0"},{"type":"extras","doc":"For security reasons TLS-1.1 is no longer supported by default, but can be\nconfigured. (OTP 22)","title":"TLS 1.1 - Standards Compliance","ref":"standards_compliance.html#tls-1-1"},{"type":"extras","doc":"Supported","title":"TLS 1.2 - Standards Compliance","ref":"standards_compliance.html#tls-1-2"},{"type":"extras","doc":"For security reasons DTLS-1.0 (based on TLS 1.1) is no longer supported by\ndefault, but can be configured. (OTP 22)","title":"DTLS 1.0 - Standards Compliance","ref":"standards_compliance.html#dtls-1-0"},{"type":"extras","doc":"Supported (based on TLS 1.2)","title":"DTLS 1.2 - Standards Compliance","ref":"standards_compliance.html#dtls-1-2"},{"type":"extras","doc":"Not yet supported","title":"DTLS 1.3 - Standards Compliance","ref":"standards_compliance.html#dtls-1-3"},{"type":"extras","doc":"OTP-22 introduces support for TLS 1.3. The current implementation supports a\nselective set of cryptographic algorithms:\n\n- Key Exchange: ECDHE groups supported by default\n- Groups: all standard groups supported for the Diffie-Hellman key exchange\n- Groups: Support brainpool groups from RFC 8734\n- Ciphers: all mandatory cipher suites are supported\n- Signature Algorithms: All algorithms form RFC 8446\n- Certificates: RSA, ECDSA and EDDSA keys\n\nOther notable features:\n\n- PSK and session resumption is supported (stateful and stateless tickets)\n- Anti-replay protection using Bloom-filters with stateless tickets\n- Early data and 0-RTT is supported\n- Key and Initialization Vector Update is supported\n\nFor more detailed information see the\n[Standards Compliance](standards_compliance.md#soc_table) below.\n\nThe following table describes the current state of standards compliance for TLS\n1.3.\n\n(_C_ = Compliant, _NC_ = Non-Compliant, _PC_ = Partially-Compliant, _NA_ = Not\nApplicable)\n\n[](){: #soc_table }\n\n| _Section_                                                                                                       | _Feature_                                                                                                                                                                                                                                                                                                 | _State_ | _Since_ |\n| --------------------------------------------------------------------------------------------------------------- | --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- | ------- | ------- |\n| [1\\.3. Updates Affecting TLS 1.2](https://tools.ietf.org/html/rfc8446#section-1.2)                              |                                                                                                                                                                                                                                                                                                           | _C_     | _24\\.1_ |\n|                                                                                                                 | Version downgrade protection mechanism                                                                                                                                                                                                                                                                    | _C_     | _22_    |\n|                                                                                                                 | RSASSA-PSS signature schemes                                                                                                                                                                                                                                                                              | _C_     | _24\\.1_ |\n|                                                                                                                 | supported_versions (ClientHello) extension                                                                                                                                                                                                                                                                | _C_     | _22_    |\n|                                                                                                                 | signature_algorithms_cert extension                                                                                                                                                                                                                                                                       | _C_     | _24\\.1_ |\n| [2\\. Protocol Overview](https://tools.ietf.org/html/rfc8446#section/2)                                          |                                                                                                                                                                                                                                                                                                           | _PC_    | _22_    |\n|                                                                                                                 | (EC)DHE                                                                                                                                                                                                                                                                                                   | _C_     | _22_    |\n|                                                                                                                 | PSK-only                                                                                                                                                                                                                                                                                                  | _NC_    |         |\n|                                                                                                                 | PSK with (EC)DHE                                                                                                                                                                                                                                                                                          | _C_     | _22\\.2_ |\n| [2\\.1. Incorrect DHE share](https://tools.ietf.org/html/rfc8446#section-2.1)                                    | HelloRetryRequest                                                                                                                                                                                                                                                                                         | _C_     | _22_    |\n| [2\\.2. Resumption and Pre-Shared Key (PSK)](https://tools.ietf.org/html/rfc8446#section-2.2)                    |                                                                                                                                                                                                                                                                                                           | _C_     | _22\\.2_ |\n| [2\\.3. 0-RTT Data](https://tools.ietf.org/html/rfc8446#section-2.3)                                             |                                                                                                                                                                                                                                                                                                           | _PC_    | _23\\.3_ |\n| [4\\.1.1. Cryptographic Negotiation](https://tools.ietf.org/html/rfc8446#section-4.1.1)                          |                                                                                                                                                                                                                                                                                                           | _C_     | _22\\.2_ |\n|                                                                                                                 | supported_groups extension                                                                                                                                                                                                                                                                                | _C_     | _22_    |\n|                                                                                                                 | signature_algorithms extension                                                                                                                                                                                                                                                                            | _C_     | _22_    |\n|                                                                                                                 | pre_shared_key extension                                                                                                                                                                                                                                                                                  | _C_     | _22\\.2_ |\n| [4\\.1.2. Client Hello](https://tools.ietf.org/html/rfc8446#section-4.1.2)                                       | _Client_                                                                                                                                                                                                                                                                                                  | _PC_    | _22\\.1_ |\n|                                                                                                                 | server_name (RFC6066)                                                                                                                                                                                                                                                                                     | _C_     | _23\\.2_ |\n|                                                                                                                 | max_fragment_length (RFC6066)                                                                                                                                                                                                                                                                             | _C_     | _23\\.0_ |\n|                                                                                                                 | status_request (RFC6066)                                                                                                                                                                                                                                                                                  | _C_     | _27\\.0_ |\n|                                                                                                                 | supported_groups (RFC7919)                                                                                                                                                                                                                                                                                | _C_     | _22\\.1_ |\n|                                                                                                                 | signature_algorithms (RFC8446)                                                                                                                                                                                                                                                                            | _C_     | _22\\.1_ |\n|                                                                                                                 | use_srtp (RFC5764)                                                                                                                                                                                                                                                                                        | _C_     | 26\\.0   |\n|                                                                                                                 | heartbeat (RFC6520)                                                                                                                                                                                                                                                                                       | _NC_    |         |\n|                                                                                                                 | application_layer_protocol_negotiation (RFC7301)                                                                                                                                                                                                                                                          | _C_     | _22\\.1_ |\n|                                                                                                                 | signed_certificate_timestamp (RFC6962)                                                                                                                                                                                                                                                                    | _NC_    |         |\n|                                                                                                                 | client_certificate_type (RFC7250)                                                                                                                                                                                                                                                                         | _NC_    |         |\n|                                                                                                                 | server_certificate_type (RFC7250)                                                                                                                                                                                                                                                                         | _NC_    |         |\n|                                                                                                                 | padding (RFC7685)                                                                                                                                                                                                                                                                                         | _NC_    |         |\n|                                                                                                                 | key_share (RFC8446)                                                                                                                                                                                                                                                                                       | _C_     | _22\\.1_ |\n|                                                                                                                 | pre_shared_key (RFC8446)                                                                                                                                                                                                                                                                                  | _C_     | _22\\.2_ |\n|                                                                                                                 | psk_key_exchange_modes (RFC8446)                                                                                                                                                                                                                                                                          | _C_     | _22\\.2_ |\n|                                                                                                                 | early_data (RFC8446)                                                                                                                                                                                                                                                                                      | _C_     | _23\\.3_ |\n|                                                                                                                 | cookie (RFC8446)                                                                                                                                                                                                                                                                                          | _C_     | _23\\.1_ |\n|                                                                                                                 | supported_versions (RFC8446)                                                                                                                                                                                                                                                                              | _C_     | _22\\.1_ |\n|                                                                                                                 | certificate_authorities (RFC8446)                                                                                                                                                                                                                                                                         | _C_     | 24\\.3   |\n|                                                                                                                 | oid_filters (RFC8446)                                                                                                                                                                                                                                                                                     | _NC_    |         |\n|                                                                                                                 | post_handshake_auth (RFC8446)                                                                                                                                                                                                                                                                             | _NC_    |         |\n|                                                                                                                 | signature_algorithms_cert (RFC8446)                                                                                                                                                                                                                                                                       | _C_     | _22\\.1_ |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _PC_    | _22_    |\n|                                                                                                                 | server_name (RFC6066)                                                                                                                                                                                                                                                                                     | _C_     | _23\\.2_ |\n|                                                                                                                 | max_fragment_length (RFC6066)                                                                                                                                                                                                                                                                             | _C_     | _23\\.0_ |\n|                                                                                                                 | status_request (RFC6066)                                                                                                                                                                                                                                                                                  | _NC_    |         |\n|                                                                                                                 | supported_groups (RFC7919)                                                                                                                                                                                                                                                                                | _C_     | _22_    |\n|                                                                                                                 | signature_algorithms (RFC8446)                                                                                                                                                                                                                                                                            | _C_     | _22_    |\n|                                                                                                                 | use_srtp (RFC5764)                                                                                                                                                                                                                                                                                        | _C_     | 26\\.0   |\n|                                                                                                                 | heartbeat (RFC6520)                                                                                                                                                                                                                                                                                       | _NC_    |         |\n|                                                                                                                 | application_layer_protocol_negotiation (RFC7301)                                                                                                                                                                                                                                                          | _C_     | _22\\.1_ |\n|                                                                                                                 | signed_certificate_timestamp (RFC6962)                                                                                                                                                                                                                                                                    | _NC_    |         |\n|                                                                                                                 | client_certificate_type (RFC7250)                                                                                                                                                                                                                                                                         | _NC_    |         |\n|                                                                                                                 | server_certificate_type (RFC7250)                                                                                                                                                                                                                                                                         | _NC_    |         |\n|                                                                                                                 | padding (RFC7685)                                                                                                                                                                                                                                                                                         | _NC_    |         |\n|                                                                                                                 | key_share (RFC8446)                                                                                                                                                                                                                                                                                       | _C_     | _22_    |\n|                                                                                                                 | pre_shared_key (RFC8446)                                                                                                                                                                                                                                                                                  | _C_     | _22\\.2_ |\n|                                                                                                                 | psk_key_exchange_modes (RFC8446)                                                                                                                                                                                                                                                                          | _C_     | _22\\.2_ |\n|                                                                                                                 | early_data (RFC8446)                                                                                                                                                                                                                                                                                      | _C_     | _23\\.3_ |\n|                                                                                                                 | cookie (RFC8446)                                                                                                                                                                                                                                                                                          | _C_     | _23\\.1_ |\n|                                                                                                                 | supported_versions (RFC8446)                                                                                                                                                                                                                                                                              | _C_     | _22_    |\n|                                                                                                                 | oid_filters (RFC8446)                                                                                                                                                                                                                                                                                     | _NC_    |         |\n|                                                                                                                 | post_handshake_auth (RFC8446)                                                                                                                                                                                                                                                                             | _NC_    |         |\n|                                                                                                                 | signature_algorithms_cert (RFC8446)                                                                                                                                                                                                                                                                       | _C_     | _22_    |\n| [4\\.1.3. Server Hello](https://tools.ietf.org/html/rfc8446#section-4.1.3)                                       | _Client_                                                                                                                                                                                                                                                                                                  | _C_     | _22\\.2_ |\n|                                                                                                                 | Version downgrade protection                                                                                                                                                                                                                                                                              | _C_     | _22\\.1_ |\n|                                                                                                                 | key_share (RFC8446)                                                                                                                                                                                                                                                                                       | _C_     | _22\\.1_ |\n|                                                                                                                 | pre_shared_key (RFC8446)                                                                                                                                                                                                                                                                                  | _C_     | _22\\.2_ |\n|                                                                                                                 | supported_versions (RFC8446)                                                                                                                                                                                                                                                                              | _C_     | _22\\.1_ |\n|                                                                                                                 | use_srtp (RFC5764)                                                                                                                                                                                                                                                                                        | _C_     | 26\\.0   |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _C_     | _22\\.2_ |\n|                                                                                                                 | Version downgrade protection                                                                                                                                                                                                                                                                              | _C_     | _22_    |\n|                                                                                                                 | key_share (RFC8446)                                                                                                                                                                                                                                                                                       | _C_     | _22_    |\n|                                                                                                                 | pre_shared_key (RFC8446)                                                                                                                                                                                                                                                                                  | _C_     | _22\\.2_ |\n|                                                                                                                 | supported_versions (RFC8446)                                                                                                                                                                                                                                                                              | _C_     | _22_    |\n|                                                                                                                 | use_srtp (RFC5764)                                                                                                                                                                                                                                                                                        | _C_     | 26\\.0   |\n| [4\\.1.4. Hello Retry Request](https://tools.ietf.org/html/rfc8446#section-4.1.4)                                | _Server_                                                                                                                                                                                                                                                                                                  | _C_     | _22_    |\n|                                                                                                                 | key_share (RFC8446)                                                                                                                                                                                                                                                                                       | _C_     | _22_    |\n|                                                                                                                 | cookie (RFC8446)                                                                                                                                                                                                                                                                                          | _C_     | _23\\.1_ |\n|                                                                                                                 | supported_versions (RFC8446)                                                                                                                                                                                                                                                                              | _C_     | _22_    |\n| [4\\.2.1. Supported Versions](https://tools.ietf.org/html/rfc8446#section-4.2.1)                                 | _Client_                                                                                                                                                                                                                                                                                                  | _C_     | _22\\.1_ |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _C_     | _22_    |\n| [4\\.2.2. Cookie](https://tools.ietf.org/html/rfc8446#section-4.2.2)                                             | _Client_                                                                                                                                                                                                                                                                                                  | _C_     | _23\\.1_ |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _C_     | _23\\.1_ |\n| [4\\.2.3. Signature Algorithms](https://tools.ietf.org/html/rfc8446#section-4.2.3)                               | _Client_                                                                                                                                                                                                                                                                                                  | _C_     | _24_    |\n|                                                                                                                 | rsa_pkcs1_sha256                                                                                                                                                                                                                                                                                          | _C_     | _22\\.1_ |\n|                                                                                                                 | rsa_pkcs1_sha384                                                                                                                                                                                                                                                                                          | _C_     | _22\\.1_ |\n|                                                                                                                 | rsa_pkcs1_sha512                                                                                                                                                                                                                                                                                          | _C_     | _22\\.1_ |\n|                                                                                                                 | ecdsa_secp256r1_sha256                                                                                                                                                                                                                                                                                    | _C_     | _22\\.1_ |\n|                                                                                                                 | ecdsa_secp384r1_sha384                                                                                                                                                                                                                                                                                    | _C_     | _22\\.1_ |\n|                                                                                                                 | ecdsa_secp521r1_sha512                                                                                                                                                                                                                                                                                    | _C_     | _22\\.1_ |\n|                                                                                                                 | rsa_pss_rsae_sha256                                                                                                                                                                                                                                                                                       | _C_     | _22\\.1_ |\n|                                                                                                                 | rsa_pss_rsae_sha384                                                                                                                                                                                                                                                                                       | _C_     | _22\\.1_ |\n|                                                                                                                 | rsa_pss_rsae_sha512                                                                                                                                                                                                                                                                                       | _C_     | _22\\.1_ |\n|                                                                                                                 | ed25519                                                                                                                                                                                                                                                                                                   | _C_     | _24_    |\n|                                                                                                                 | ed448                                                                                                                                                                                                                                                                                                     | _C_     | _24_    |\n|                                                                                                                 | rsa_pss_pss_sha256                                                                                                                                                                                                                                                                                        | _C_     | _23_    |\n|                                                                                                                 | rsa_pss_pss_sha384                                                                                                                                                                                                                                                                                        | _C_     | _23_    |\n|                                                                                                                 | rsa_pss_pss_sha512                                                                                                                                                                                                                                                                                        | _C_     | _23_    |\n|                                                                                                                 | rsa_pkcs1_sha1                                                                                                                                                                                                                                                                                            | _C_     | _22\\.1_ |\n|                                                                                                                 | ecdsa_sha1                                                                                                                                                                                                                                                                                                | _C_     | _22\\.1_ |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _C_     | _24_    |\n|                                                                                                                 | rsa_pkcs1_sha256                                                                                                                                                                                                                                                                                          | _C_     | _22_    |\n|                                                                                                                 | rsa_pkcs1_sha384                                                                                                                                                                                                                                                                                          | _C_     | _22_    |\n|                                                                                                                 | rsa_pkcs1_sha512                                                                                                                                                                                                                                                                                          | _C_     | _22_    |\n|                                                                                                                 | ecdsa_secp256r1_sha256                                                                                                                                                                                                                                                                                    | _C_     | _22\\.1_ |\n|                                                                                                                 | ecdsa_secp384r1_sha384                                                                                                                                                                                                                                                                                    | _C_     | _22\\.1_ |\n|                                                                                                                 | ecdsa_secp521r1_sha512                                                                                                                                                                                                                                                                                    | _C_     | _22\\.1_ |\n|                                                                                                                 | rsa_pss_rsae_sha256                                                                                                                                                                                                                                                                                       | _C_     | _22_    |\n|                                                                                                                 | rsa_pss_rsae_sha384                                                                                                                                                                                                                                                                                       | _C_     | _22_    |\n|                                                                                                                 | rsa_pss_rsae_sha512                                                                                                                                                                                                                                                                                       | _C_     | _22_    |\n|                                                                                                                 | ed25519                                                                                                                                                                                                                                                                                                   | _C_     | _24_    |\n|                                                                                                                 | ed448                                                                                                                                                                                                                                                                                                     | _C_     | _24_    |\n|                                                                                                                 | rsa_pss_pss_sha256                                                                                                                                                                                                                                                                                        | _C_     | _23_    |\n|                                                                                                                 | rsa_pss_pss_sha384                                                                                                                                                                                                                                                                                        | _C_     | _23_    |\n|                                                                                                                 | rsa_pss_pss_sha512                                                                                                                                                                                                                                                                                        | _C_     | _23_    |\n|                                                                                                                 | rsa_pkcs1_sha1                                                                                                                                                                                                                                                                                            | _C_     | _22_    |\n|                                                                                                                 | ecdsa_sha1                                                                                                                                                                                                                                                                                                | _C_     | _22_    |\n| [4\\.2.4. Certificate Authorities](https://tools.ietf.org/html/rfc8446#section-4.2.4)                            | _Client_                                                                                                                                                                                                                                                                                                  | _C_     | 24\\.3   |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _C_     | _24\\.3_ |\n| [4\\.2.5. OID Filters](https://tools.ietf.org/html/rfc8446#section-4.2.5)                                        | _Client_                                                                                                                                                                                                                                                                                                  | _NC_    |         |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _NC_    |         |\n| [4\\.2.6. Post-Handshake Client Authentication](https://tools.ietf.org/html/rfc8446#section-4.2.6)               | _Client_                                                                                                                                                                                                                                                                                                  | _NC_    |         |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _NC_    |         |\n| [4\\.2.7. Supported Groups](https://tools.ietf.org/html/rfc8446#section-4.2.7)                                   | _Client_                                                                                                                                                                                                                                                                                                  | _C_     | _22\\.1_ |\n|                                                                                                                 | secp256r1                                                                                                                                                                                                                                                                                                 | _C_     | _22\\.1_ |\n|                                                                                                                 | secp384r1                                                                                                                                                                                                                                                                                                 | _C_     | _22\\.1_ |\n|                                                                                                                 | secp521r1                                                                                                                                                                                                                                                                                                 | _C_     | _22\\.1_ |\n|                                                                                                                 | x25519                                                                                                                                                                                                                                                                                                    | _C_     | _22\\.1_ |\n|                                                                                                                 | x448                                                                                                                                                                                                                                                                                                      | _C_     | _22\\.1_ |\n|                                                                                                                 | ffdhe2048                                                                                                                                                                                                                                                                                                 | _C_     | _22\\.1_ |\n|                                                                                                                 | ffdhe3072                                                                                                                                                                                                                                                                                                 | _C_     | _22\\.1_ |\n|                                                                                                                 | ffdhe4096                                                                                                                                                                                                                                                                                                 | _C_     | _22\\.1_ |\n|                                                                                                                 | ffdhe6144                                                                                                                                                                                                                                                                                                 | _C_     | _22\\.1_ |\n|                                                                                                                 | ffdhe8192                                                                                                                                                                                                                                                                                                 | _C_     | _22\\.1_ |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _C_     | _22_    |\n|                                                                                                                 | secp256r1                                                                                                                                                                                                                                                                                                 | _C_     | _22_    |\n|                                                                                                                 | secp384r1                                                                                                                                                                                                                                                                                                 | _C_     | _22_    |\n|                                                                                                                 | secp521r1                                                                                                                                                                                                                                                                                                 | _C_     | _22_    |\n|                                                                                                                 | x25519                                                                                                                                                                                                                                                                                                    | _C_     | _22_    |\n|                                                                                                                 | x448                                                                                                                                                                                                                                                                                                      | _C_     | _22_    |\n|                                                                                                                 | ffdhe2048                                                                                                                                                                                                                                                                                                 | _C_     | _22_    |\n|                                                                                                                 | ffdhe3072                                                                                                                                                                                                                                                                                                 | _C_     | _22_    |\n|                                                                                                                 | ffdhe4096                                                                                                                                                                                                                                                                                                 | _C_     | _22_    |\n|                                                                                                                 | ffdhe6144                                                                                                                                                                                                                                                                                                 | _C_     | _22_    |\n|                                                                                                                 | ffdhe8192                                                                                                                                                                                                                                                                                                 | _C_     | _22_    |\n| [4\\.2.8. Key Share](https://tools.ietf.org/html/rfc8446#section-4.2.8)                                          | _Client_                                                                                                                                                                                                                                                                                                  | _C_     | _22\\.1_ |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _C_     | _22_    |\n| [4\\.2.9. Pre-Shared Key Exchange Modes](https://tools.ietf.org/html/rfc8446#section-4.2.9)                      | _Client_                                                                                                                                                                                                                                                                                                  | _C_     | _22\\.2_ |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _C_     | _22\\.2_ |\n| [4\\.2.10. Early Data Indication](https://tools.ietf.org/html/rfc8446#section-4.2.10)                            | _Client_                                                                                                                                                                                                                                                                                                  | _C_     | _23\\.3_ |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _C_     | _23\\.3_ |\n| [4\\.2.11. Pre-Shared Key Extension](https://tools.ietf.org/html/rfc8446#section-4.2.11)                         | _Client_                                                                                                                                                                                                                                                                                                  | _C_     | _22\\.2_ |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _C_     | _22\\.2_ |\n| [4\\.2.11.1. Ticket Age](https://tools.ietf.org/html/rfc8446#section-4.2.11.1)                                   | _Client_                                                                                                                                                                                                                                                                                                  | _C_     | _22\\.2_ |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _C_     | _22\\.2_ |\n| [4\\.2.11.2. PSK Binder](https://tools.ietf.org/html/rfc8446#section-4.2.11.2)                                   | _Client_                                                                                                                                                                                                                                                                                                  | _C_     | _22\\.2_ |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _C_     | _22\\.2_ |\n| [4\\.2.11.3. Processing Order](https://tools.ietf.org/html/rfc8446#section-4.2.11.3)                             | _Client_                                                                                                                                                                                                                                                                                                  | _NC_    |         |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _NC_    |         |\n| [4\\.3.1. Encrypted Extensions](https://tools.ietf.org/html/rfc8446#section-4.3.1)                               | _Client_                                                                                                                                                                                                                                                                                                  | _PC_    | _22\\.1_ |\n|                                                                                                                 | server_name (RFC6066)                                                                                                                                                                                                                                                                                     | _C_     | _23\\.2_ |\n|                                                                                                                 | max_fragment_length (RFC6066)                                                                                                                                                                                                                                                                             | _C_     | _23\\.0_ |\n|                                                                                                                 | supported_groups (RFC7919)                                                                                                                                                                                                                                                                                | _NC_    |         |\n|                                                                                                                 | use_srtp (RFC5764)                                                                                                                                                                                                                                                                                        | _NC_    |         |\n|                                                                                                                 | heartbeat (RFC6520)                                                                                                                                                                                                                                                                                       | _NC_    |         |\n|                                                                                                                 | application_layer_protocol_negotiation (RFC7301)                                                                                                                                                                                                                                                          | _C_     | 23\\.0   |\n|                                                                                                                 | client_certificate_type (RFC7250)                                                                                                                                                                                                                                                                         | _NC_    |         |\n|                                                                                                                 | server_certificate_type (RFC7250)                                                                                                                                                                                                                                                                         | _NC_    |         |\n|                                                                                                                 | early_data (RFC8446)                                                                                                                                                                                                                                                                                      | _C_     | _23\\.3_ |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _PC_    | _22_    |\n|                                                                                                                 | server_name (RFC6066)                                                                                                                                                                                                                                                                                     | _C_     | _23\\.2_ |\n|                                                                                                                 | max_fragment_length (RFC6066)                                                                                                                                                                                                                                                                             | _C_     | _23\\.0_ |\n|                                                                                                                 | supported_groups (RFC7919)                                                                                                                                                                                                                                                                                | _NC_    |         |\n|                                                                                                                 | use_srtp (RFC5764)                                                                                                                                                                                                                                                                                        | _NC_    |         |\n|                                                                                                                 | heartbeat (RFC6520)                                                                                                                                                                                                                                                                                       | _NC_    |         |\n|                                                                                                                 | application_layer_protocol_negotiation (RFC7301)                                                                                                                                                                                                                                                          | _C_     | 23\\.0   |\n|                                                                                                                 | client_certificate_type (RFC7250)                                                                                                                                                                                                                                                                         | _NC_    |         |\n|                                                                                                                 | server_certificate_type (RFC7250)                                                                                                                                                                                                                                                                         | _NC_    |         |\n|                                                                                                                 | early_data (RFC8446)                                                                                                                                                                                                                                                                                      | _C_     | _23\\.3_ |\n| [4\\.3.2. Certificate Request](https://tools.ietf.org/html/rfc8446#section-4.3.2)                                | _Client_                                                                                                                                                                                                                                                                                                  | _PC_    | _22\\.1_ |\n|                                                                                                                 | status_request (RFC6066)                                                                                                                                                                                                                                                                                  | _NC_    |         |\n|                                                                                                                 | signature_algorithms (RFC8446)                                                                                                                                                                                                                                                                            | _C_     | _22\\.1_ |\n|                                                                                                                 | signed_certificate_timestamp (RFC6962)                                                                                                                                                                                                                                                                    | _NC_    |         |\n|                                                                                                                 | certificate_authorities (RFC8446)                                                                                                                                                                                                                                                                         | _C_     | 24\\.3   |\n|                                                                                                                 | oid_filters (RFC8446)                                                                                                                                                                                                                                                                                     | _NC_    |         |\n|                                                                                                                 | signature_algorithms_cert (RFC8446)                                                                                                                                                                                                                                                                       | _C_     | _22\\.1_ |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _PC_    | _22_    |\n|                                                                                                                 | status_request (RFC6066)                                                                                                                                                                                                                                                                                  | _NC_    |         |\n|                                                                                                                 | signature_algorithms (RFC8446)                                                                                                                                                                                                                                                                            | _C_     | _22_    |\n|                                                                                                                 | signed_certificate_timestamp (RFC6962)                                                                                                                                                                                                                                                                    | _NC_    |         |\n|                                                                                                                 | certificate_authorities (RFC8446)                                                                                                                                                                                                                                                                         | _C_     | 24\\.3   |\n|                                                                                                                 | oid_filters (RFC8446)                                                                                                                                                                                                                                                                                     | _NC_    |         |\n|                                                                                                                 | signature_algorithms_cert (RFC8446)                                                                                                                                                                                                                                                                       | _C_     | _22_    |\n| [4\\.4.1. The Transcript Hash](https://tools.ietf.org/html/rfc8446#section-4.4.1)                                |                                                                                                                                                                                                                                                                                                           | _C_     | _22_    |\n| [4\\.4.2. Certificate](https://tools.ietf.org/html/rfc8446#section-4.4.2)                                        | _Client_                                                                                                                                                                                                                                                                                                  | _PC_    | _22\\.1_ |\n|                                                                                                                 | Arbitrary certificate chain orderings                                                                                                                                                                                                                                                                     | _C_     | _22\\.2_ |\n|                                                                                                                 | Extraneous certificates in chain                                                                                                                                                                                                                                                                          | _C_     | _23\\.2_ |\n|                                                                                                                 | status_request (RFC6066)                                                                                                                                                                                                                                                                                  | _C_     | 27\\.0   |\n|                                                                                                                 | signed_certificate_timestamp (RFC6962)                                                                                                                                                                                                                                                                    | _NC_    |         |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _PC_    | _22_    |\n|                                                                                                                 | status_request (RFC6066)                                                                                                                                                                                                                                                                                  | _NC_    |         |\n|                                                                                                                 | signed_certificate_timestamp (RFC6962)                                                                                                                                                                                                                                                                    | _NC_    |         |\n| [4\\.4.2.1. OCSP Status and SCT Extensions](https://tools.ietf.org/html/rfc8446#section-4.4.2.1)                 | _Client_                                                                                                                                                                                                                                                                                                  | _PC_    | _27\\.0_ |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _NC_    |         |\n| [4\\.4.2.2. Server Certificate Selection](https://tools.ietf.org/html/rfc8446#section-4.4.2.2)                   |                                                                                                                                                                                                                                                                                                           | _C_     | _24\\.3_ |\n|                                                                                                                 | The certificate type MUST be X.509v3, unless explicitly negotiated otherwise                                                                                                                                                                                                                              | _C_     | _22_    |\n|                                                                                                                 | The server's end-entity certificate's public key (and associated restrictions) MUST be compatible with the selected authentication algorithm from the client's \"signature_algorithms\" extension (currently RSA, ECDSA, or EdDSA).                                                                         | _C_     | _22_    |\n|                                                                                                                 | The certificate MUST allow the key to be used for signing with a signature scheme indicated in the client's \"signature_algorithms\"/\"signature_algorithms_cert\" extensions                                                                                                                                 | _C_     | _22_    |\n|                                                                                                                 | The \"server_name\" and \"certificate_authorities\" extensions are used to guide certificate selection. As servers MAY require the presence of the \"server_name\" extension, clients SHOULD send this extension, when applicable.                                                                              | _C_     | _24\\.3_ |\n| [4\\.4.2.3. Client Certificate Selection](https://tools.ietf.org/html/rfc8446#section-4.4.2.3)                   |                                                                                                                                                                                                                                                                                                           | _PC_    | _22\\.1_ |\n|                                                                                                                 | The certificate type MUST be X.509v3, unless explicitly negotiated otherwise                                                                                                                                                                                                                              | _C_     | _22\\.1_ |\n|                                                                                                                 | If the \"certificate_authorities\" extension in the CertificateRequest message was present, at least one of the certificates in the certificate chain SHOULD be issued by one of the listed CAs.                                                                                                            | _C_     | _24\\.3_ |\n|                                                                                                                 | The certificates MUST be signed using an acceptable signature algorithm                                                                                                                                                                                                                                   | _C_     | _22\\.1_ |\n|                                                                                                                 | If the CertificateRequest message contained a non-empty \"oid_filters\" extension, the end-entity certificate MUST match the extension OIDs that are recognized by the client                                                                                                                               | _NC_    |         |\n| [4\\.4.2.4. Receiving a Certificate Message](https://tools.ietf.org/html/rfc8446#section-4.4.2.4)                | _Client_                                                                                                                                                                                                                                                                                                  | _C_     | _22\\.1_ |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _C_     | _22_    |\n| [4\\.4.3. Certificate Verify](https://tools.ietf.org/html/rfc8446#section-4.4.3)                                 | _Client_                                                                                                                                                                                                                                                                                                  | _C_     | _22\\.1_ |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _C_     | _22_    |\n| [4\\.4.4. Finished](https://tools.ietf.org/html/rfc8446#section-4.4.4)                                           | _Client_                                                                                                                                                                                                                                                                                                  | _C_     | _22\\.1_ |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _C_     | _22_    |\n| [4\\.5. End of Early Data](https://tools.ietf.org/html/rfc8446#section-4.5)                                      | _Client_                                                                                                                                                                                                                                                                                                  | _C_     | _23\\.3_ |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _C_     | _23\\.3_ |\n| [4\\.6.1. New Session Ticket Message](https://tools.ietf.org/html/rfc8446#section-4.6.1)                         | _Client_                                                                                                                                                                                                                                                                                                  | _C_     | _23\\.3_ |\n|                                                                                                                 | early_data (RFC8446)                                                                                                                                                                                                                                                                                      | _C_     | _23\\.3_ |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _C_     | _23\\.3_ |\n|                                                                                                                 | early_data (RFC8446)                                                                                                                                                                                                                                                                                      | _C_     | _23\\.3_ |\n| [4\\.6.2. Post-Handshake Authentication](https://tools.ietf.org/html/rfc8446#section-4.6.2)                      | _Client_                                                                                                                                                                                                                                                                                                  | _NC_    |         |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _NC_    |         |\n| [4\\.6.3. Key and Initialization Vector Update](https://tools.ietf.org/html/rfc8446#section-4.6.3)               | _Client_                                                                                                                                                                                                                                                                                                  | _C_     | _22\\.3_ |\n|                                                                                                                 | _Server_                                                                                                                                                                                                                                                                                                  | _C_     | _22\\.3_ |\n| [5\\.1. Record Layer](https://tools.ietf.org/html/rfc8446#section-5.1)                                           |                                                                                                                                                                                                                                                                                                           | _C_     | _22_    |\n|                                                                                                                 | MUST NOT be interleaved with other record types                                                                                                                                                                                                                                                           | _C_     | _22_    |\n|                                                                                                                 | MUST NOT span key changes                                                                                                                                                                                                                                                                                 | _C_     | _22_    |\n|                                                                                                                 | MUST NOT send zero-length fragments                                                                                                                                                                                                                                                                       | _C_     | _22_    |\n|                                                                                                                 | Alert messages MUST NOT be fragmented                                                                                                                                                                                                                                                                     | _C_     | _22_    |\n| [5\\.2. Record Payload Protection](https://tools.ietf.org/html/rfc8446#section-5.2)                              |                                                                                                                                                                                                                                                                                                           | _C_     | _22_    |\n| [5\\.3. Per-Record Nonce](https://tools.ietf.org/html/rfc8446#section-5.3)                                       |                                                                                                                                                                                                                                                                                                           | _C_     | _22_    |\n| [5\\.4. Record Padding](https://tools.ietf.org/html/rfc8446#section-5.4)                                         |                                                                                                                                                                                                                                                                                                           | _PC_    | _22_    |\n|                                                                                                                 | MAY choose to pad                                                                                                                                                                                                                                                                                         | _NC_    |         |\n|                                                                                                                 | MUST NOT send Handshake and Alert records that have a zero-length TLSInnerPlaintext.content                                                                                                                                                                                                               | _NC_    |         |\n|                                                                                                                 | The padding sent is automatically verified                                                                                                                                                                                                                                                                | _C_     | _22_    |\n| [5\\.5. Limits on Key Usage](https://tools.ietf.org/html/rfc8446#section-5.5)                                    |                                                                                                                                                                                                                                                                                                           | _C_     | _22\\.3_ |\n| [6\\.1. Closure Alerts](https://tools.ietf.org/html/rfc8446#section-6.1)                                         |                                                                                                                                                                                                                                                                                                           | _22_    |         |\n|                                                                                                                 | close_notify                                                                                                                                                                                                                                                                                              | _C_     | _22_    |     |\n|                                                                                                                 | user_cancelled                                                                                                                                                                                                                                                                                            | _C_     | _22_    |     |\n| [6\\.2. Error Alerts](https://tools.ietf.org/html/rfc8446#section-6.2)                                           |                                                                                                                                                                                                                                                                                                           | _PC_    | _22_    |\n| [7\\.1. Key Schedule](https://tools.ietf.org/html/rfc8446#section-7.1)                                           |                                                                                                                                                                                                                                                                                                           | _C_     | _22_    |\n| [7\\.2. Updating Traffic Secrets](https://tools.ietf.org/html/rfc8446#section-7.2)                               |                                                                                                                                                                                                                                                                                                           | _C_     | _22_    |\n| [7\\.3. Traffic Key Calculation](https://tools.ietf.org/html/rfc8446#section-7.3)                                |                                                                                                                                                                                                                                                                                                           | _C_     | _22_    |\n| [7\\.5. Exporters](https://tools.ietf.org/html/rfc8446#section-7.5)                                              |                                                                                                                                                                                                                                                                                                           | _PC_    | _26\\.3_ |\n| [8\\. 0-RTT and Anti-Replay](https://tools.ietf.org/html/rfc8446#section/8)                                      |                                                                                                                                                                                                                                                                                                           | _C_     | _22\\.2_ |\n| [8\\.1. Single-Use Tickets](https://tools.ietf.org/html/rfc8446#section-8.1)                                     |                                                                                                                                                                                                                                                                                                           | _C_     | _22\\.2_ |\n| [8\\.2. Client Hello Recording](https://tools.ietf.org/html/rfc8446#section-8.2)                                 |                                                                                                                                                                                                                                                                                                           | _C_     | _22\\.2_ |\n| [8\\.3. Freshness Checks](https://tools.ietf.org/html/rfc8446#section-8.3)                                       |                                                                                                                                                                                                                                                                                                           | _C_     | _22\\.2_ |\n| [9\\.1. Mandatory-to-Implement Cipher Suites](https://tools.ietf.org/html/rfc8446#section-9.1)                   |                                                                                                                                                                                                                                                                                                           | _C_     | _22\\.1_ |\n|                                                                                                                 | MUST implement the TLS_AES_128_GCM_SHA256                                                                                                                                                                                                                                                                 | _C_     | _22_    |\n|                                                                                                                 | SHOULD implement the TLS_AES_256_GCM_SHA384                                                                                                                                                                                                                                                               | _C_     | _22_    |\n|                                                                                                                 | SHOULD implement the TLS_CHACHA20_POLY1305_SHA256                                                                                                                                                                                                                                                         | _C_     | _22_    |\n|                                                                                                                 | _Digital signatures_                                                                                                                                                                                                                                                                                      | _C_     | _22\\.1_ |\n|                                                                                                                 | MUST support rsa_pkcs1_sha256 (for certificates)                                                                                                                                                                                                                                                          | _C_     | _22_    |\n|                                                                                                                 | MUST support rsa_pss_rsae_sha256 (for CertificateVerify and certificates)                                                                                                                                                                                                                                 | _C_     | _22_    |\n|                                                                                                                 | MUST support ecdsa_secp256r1_sha256                                                                                                                                                                                                                                                                       | _C_     | _22\\.1_ |\n|                                                                                                                 | _Key Exchange_                                                                                                                                                                                                                                                                                            | _C_     | _22_    |\n|                                                                                                                 | MUST support key exchange with secp256r1                                                                                                                                                                                                                                                                  | _C_     | _22_    |\n|                                                                                                                 | SHOULD support key exchange with X25519                                                                                                                                                                                                                                                                   | _C_     | _22_    |\n| [9\\.2. Mandatory-to-Implement Extensions](https://tools.ietf.org/html/rfc8446#section-9.2)                      |                                                                                                                                                                                                                                                                                                           | _C_     | _23\\.2_ |\n|                                                                                                                 | Supported Versions                                                                                                                                                                                                                                                                                        | _C_     | _22_    |\n|                                                                                                                 | Cookie                                                                                                                                                                                                                                                                                                    | _C_     | _23\\.1_ |\n|                                                                                                                 | Signature Algorithms                                                                                                                                                                                                                                                                                      | _C_     | _22_    |\n|                                                                                                                 | Signature Algorithms Certificate                                                                                                                                                                                                                                                                          | _C_     | _22_    |\n|                                                                                                                 | Negotiated Groups                                                                                                                                                                                                                                                                                         | _C_     | _22_    |\n|                                                                                                                 | Key Share                                                                                                                                                                                                                                                                                                 | _C_     | _22_    |\n|                                                                                                                 | Server Name Indication                                                                                                                                                                                                                                                                                    | _C_     | _23\\.2_ |\n|                                                                                                                 | _MUST send and use these extensions_                                                                                                                                                                                                                                                                      | _C_     | _22\\.2_ |\n|                                                                                                                 | \"supported_versions\" is REQUIRED for ClientHello, ServerHello and HelloRetryRequest                                                                                                                                                                                                                       | _C_     | _22\\.1_ |\n|                                                                                                                 | \"signature_algorithms\" is REQUIRED for certificate authentication                                                                                                                                                                                                                                         | _C_     | _22_    |\n|                                                                                                                 | \"supported_groups\" is REQUIRED for ClientHello messages using (EC)DHE key exchange                                                                                                                                                                                                                        | _C_     | _22_    |\n|                                                                                                                 | \"key_share\" is REQUIRED for (EC)DHE key exchange                                                                                                                                                                                                                                                          | _C_     | _22_    |\n|                                                                                                                 | \"pre_shared_key\" is REQUIRED for PSK key agreement                                                                                                                                                                                                                                                        | _C_     | _22\\.2_ |\n|                                                                                                                 | \"psk_key_exchange_modes\" is REQUIRED for PSK key agreement                                                                                                                                                                                                                                                | _C_     | _22\\.2_ |\n|                                                                                                                 | _TLS 1.3 ClientHello_                                                                                                                                                                                                                                                                                     | _C_     | _22\\.1_ |\n|                                                                                                                 | If not containing a \"pre_shared_key\" extension, it MUST contain both a \"signature_algorithms\" extension and a \"supported_groups\" extension.                                                                                                                                                               | _C_     | _22\\.1_ |\n|                                                                                                                 | If containing a \"supported_groups\" extension, it MUST also contain a \"key_share\" extension, and vice versa. An empty KeyShare.client_shares vector is permitted.                                                                                                                                          | _C_     | _22\\.1_ |\n|                                                                                                                 | _TLS 1.3 ServerHello_                                                                                                                                                                                                                                                                                     | _C_     | _23\\.2_ |\n|                                                                                                                 | MUST support the use of the \"server_name\" extension                                                                                                                                                                                                                                                       | _C_     | _23\\.2_ |\n| [9\\.3. Protocol Invariants](https://tools.ietf.org/html/rfc8446#section-9.3)                                    |                                                                                                                                                                                                                                                                                                           | _C_     | _22\\.1_ |\n|                                                                                                                 | _MUST correctly handle extensible fields_                                                                                                                                                                                                                                                                 | _C_     | _22\\.1_ |\n|                                                                                                                 | A client sending a ClientHello MUST support all parameters advertised in it. Otherwise, the server may fail to interoperate by selecting one of those parameters.                                                                                                                                         | _C_     | _22\\.1_ |\n|                                                                                                                 | A server receiving a ClientHello MUST correctly ignore all unrecognized cipher suites, extensions, and other parameters. Otherwise, it may fail to interoperate with newer clients. In TLS 1.3, a client receiving a CertificateRequest or NewSessionTicket MUST also ignore all unrecognized extensions. | _C_     | _22\\.1_ |\n|                                                                                                                 | A middlebox which terminates a TLS connection MUST behave as a compliant TLS server                                                                                                                                                                                                                       | _NA_    |         |\n|                                                                                                                 | A middlebox which forwards ClientHello parameters it does not understand MUST NOT process any messages beyond that ClientHello. It MUST forward all subsequent traffic unmodified. Otherwise, it may fail to interoperate with newer clients and servers.                                                 | _NA_    |         |\n| [B.4. Cipher Suites](https://tools.ietf.org/html/rfc8446#section-B.4)                                           |                                                                                                                                                                                                                                                                                                           | _C_     | _23_    |\n|                                                                                                                 | TLS_AES_128_GCM_SHA256                                                                                                                                                                                                                                                                                    | _C_     | _22_    |\n|                                                                                                                 | TLS_AES_256_GCM_SHA384                                                                                                                                                                                                                                                                                    | _C_     | _22_    |\n|                                                                                                                 | TLS_CHACHA20_POLY1305_SHA256                                                                                                                                                                                                                                                                              | _C_     | _22_    |\n|                                                                                                                 | TLS_AES_128_CCM_SHA256                                                                                                                                                                                                                                                                                    | _C_     | _22_    |\n|                                                                                                                 | TLS_AES_128_CCM_8_SHA256                                                                                                                                                                                                                                                                                  | _C_     | _23_    |\n| [C.1. Random Number Generation and Seeding](https://tools.ietf.org/html/rfc8446#section-C.1)                    |                                                                                                                                                                                                                                                                                                           | _C_     | _22_    |\n| [C.2. Certificates and Authentication](https://tools.ietf.org/html/rfc8446#section-C.2)                         |                                                                                                                                                                                                                                                                                                           | _C_     | _22_    |\n| [C.3. Implementation Pitfalls](https://tools.ietf.org/html/rfc8446#section-C.3)                                 |                                                                                                                                                                                                                                                                                                           | _PC_    | _22_    |\n| [C.4. Client Tracking Prevention](https://tools.ietf.org/html/rfc8446#section-C.4)                              |                                                                                                                                                                                                                                                                                                           | _C_     | _22\\.2_ |\n| [C.5. Unauthenticated Operation](https://tools.ietf.org/html/rfc8446#section-C.5)                               |                                                                                                                                                                                                                                                                                                           | _C_     | _22_    |\n| [D.1. Negotiating with an Older Server](https://tools.ietf.org/html/rfc8446#section-D.1)                        |                                                                                                                                                                                                                                                                                                           | _C_     | _22\\.2_ |\n| [D.2. Negotiating with an Older Client](https://tools.ietf.org/html/rfc8446#section-D.2)                        |                                                                                                                                                                                                                                                                                                           | _C_     | _22_    |\n| [D.3. 0-RTT Backward Compatibility](https://tools.ietf.org/html/rfc8446#section-D.3)                            |                                                                                                                                                                                                                                                                                                           | _NC_    |         |\n| [D.4. Middlebox Compatibility Mode](https://tools.ietf.org/html/rfc8446#section-D.4)                            |                                                                                                                                                                                                                                                                                                           | _C_     | _23_    |\n| [D.5. Security Restrictions Related to Backward Compatibility](https://tools.ietf.org/html/rfc8446#section-D.5) |                                                                                                                                                                                                                                                                                                           | _C_     | _22_    |\n\n_Table: Standards Compliance_","title":"TLS 1.3 - Standards Compliance","ref":"standards_compliance.html#tls-1-3"}],"content_type":"text/plain","producer":{"name":"ex_doc","version":[48,46,51,52,46,49]}}