/*
	description: "[
			Constants used for generic conformance. Update SHARED_GEN_CONF_LEVEL in the
			compiler code to match those values.
			]"
	date:		"$Date$"
	revision:	"$Revision$"
	copyright:	"Copyright (c) 1985-2013, Eiffel Software."
	license:	"GPL version 2 see http://www.eiffel.com/licensing/gpl.txt)"
	licensing_options:	"Commercial license is available at http://www.eiffel.com/licensing"
	copying: "[
			This file is part of Eiffel Software's Runtime.
			
			Eiffel Software's Runtime is free software; you can
			redistribute it and/or modify it under the terms of the
			GNU General Public License as published by the Free
			Software Foundation, version 2 of the License
			(available at the URL listed under "license" above).
			
			Eiffel Software's Runtime is distributed in the hope
			that it will be useful,	but WITHOUT ANY WARRANTY;
			without even the implied warranty of MERCHANTABILITY
			or FITNESS FOR A PARTICULAR PURPOSE.
			See the	GNU General Public License for more details.
			
			You should have received a copy of the GNU General Public
			License along with Eiffel Software's Runtime; if not,
			write to the Free Software Foundation, Inc.,
			51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA
		]"
	source: "[
			 Eiffel Software
			 356 Storke Road, Goleta, CA 93117 USA
			 Telephone 805-685-1006, Fax 805-685-6869
			 Website http://www.eiffel.com
			 Customer support http://support.eiffel.com
		]"
*/

#ifndef _rt_gen_types_h_
#define _rt_gen_types_h_
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#include "eif_eiffel.h"
#include "rt_assert.h"
#include "rt_portable.h"

#ifdef __cplusplus
extern "C" {
#endif

/*------------------------------------------------------------------*/
/* Constant values for special type.                                */
/* You must also update SHARED_GEN_CONF_LEVEL after adding          */
/* new codes!!                                                      */
/* Note that INVALID_DTYPE is publicly exported in `eif_gen_conf.h' */
/* and has the same value as TERMINATOR.                            */
/*------------------------------------------------------------------*/

#define TERMINATOR				0xFFFF
#define NONE_TYPE				0xFFFE
#define LIKE_ARG_TYPE			0xFFFD
#define LIKE_CURRENT_TYPE		0xFFFC
/* #define NOT_USED_TYPE		0xFFFB */
#define LIKE_FEATURE_TYPE		0xFFFA	/* Only for melted code generation */
#define TUPLE_TYPE				0xFFF9
#define FORMAL_TYPE				0xFFF8
#define PARENT_TYPE_SEPARATOR	0xFFF7
/* #define NOT_USED_TYPE		0xFFF6 */
#define QUALIFIED_FEATURE_TYPE	0xFFF5
#define UNUSED_SPECIAL_TYPE_1	0xFFF4
#define UNUSED_SPECIAL_TYPE_2	0xFFF3
#define UNUSED_SPECIAL_TYPE_3	0xFFF2
#define UNUSED_SPECIAL_TYPE_4	0xFFF1
#define UNUSED_SPECIAL_TYPE_5	0xFFF0

/* Maximum valid type value one can have. */
#define MAX_DTYPE				0xFF00

#define RT_CONF_IS_NONE_TYPE(g)      ((g) == NONE_TYPE)

/* Annotations:
 * ============
 *
 * By default annotations are encoded using the lower 7 bits of the type ID where the upper 16 bits are
 * all ones, and the 8-th one is zero. That is to say, they range from value 0xFF01 to 0xFF7F. Since they
 * are bit flags, it also means that we only have 7 possible choices, but due to a poor initial choice
 * of values when introducing void-safety, we really only have 6 (we have reserved 0xFF10 to preserve
 * backward compatibility with the C storable mechanism).
 *
 * Some rules: When a type has no annotation, it means a detachable, non-separate, variant type.
 * In all other cases, there will be an annotation. In the case of actual generic parameters, we
 * can have: X [A], X [frozen A] or X [variant A] and the annotations for A are respectively POLY_TYPE,
 * FROZEN_TYPE or VARIANT_TYPE (note that for optimization purpose VARIANT_TYPE will only appear
 * if needed since its absence means variant). 
 *
 * Some annotations are mutually exclusive, it enables us to perform some optimization. For example,
 * attached/detachable is mutually exclusive of frozen/variant/poly.
 */

#define ANNOTATION_MASK			0x007F	/* All possible annotations. */
#define ATTACHED_FLAG			0x0001
#define DETACHABLE_FLAG			0x0002	/* Only present when overriding an attached type. */
#define SEPARATE_FLAG			0x0004
#define VARIANT_FLAG			0x0008	/* Only present when overriding a frozen/poly type. */
#define UNUSABLE_FLAG			0x0010	/* See below why we cannot used it. */
#define FROZEN_FLAG				0x0020
#define POLY_FLAG				0x0040

/* Due to backward compatibility for storable we cannot use the 0x0010 flag. In storable
 * whenever we encounter ATTACHED_FLAG or DETACHABLE_FLAG we will OR it with 0x0010 so
 * that older version of EiffelStudio can retrieve storable created by this new version
 * of the runtime.
 */

/* Those constants are not actually used in runtime or macros. They are mostly there
 * as this is what is generated by the compiler to describe annotations in type arrays.
 */
#define ATTACHED_TYPE			(0xFF00 | ATTACHED_FLAG)
#define DETACHABLE_TYPE			(0xFF00 | DETACHABLE_FLAG)
#define SEPARATE_TYPE			(0xFF00 | SEPARATE_FLAG)
#define VARIANT_TYPE			(0xFF00 | VARIANT_FLAG)
#define FROZEN_TYPE				(0xFF00 | FROZEN_FLAG)
#define POLY_TYPE				(0xFF00 | POLY_FLAG)

/* Does `g' represent an annotation? */
#define RT_CONF_HAS_ANNOTATION_TYPE_IN_ARRAY(g)			(((g) & ~ANNOTATION_MASK) == 0xFF00)

/* Is `g' a valid annotation? */
#define RT_CONF_IS_VALID_ANNOTATION(g)	(!(g & ~ANNOTATION_MASK))
#define RT_CONF_IS_VALID_ANNOTATION_IN_ARRAY(g)	(!g || RT_CONF_HAS_ANNOTATION_TYPE_IN_ARRAY(g))

/* Conveniences. */

/* Non-contracted versions. */
#define RT_CONF_HAS_ATTACHMENT_MARK_FLAG(g)	((g) & (ATTACHED_FLAG | DETACHABLE_FLAG))
#define RT_CONF_HAS_VARIANT_MARK_FLAG(g)		((g) & (VARIANT_FLAG | FROZEN_FLAG | POLY_FLAG))
#define RT_CONF_IS_ATTACHED_FLAG(g)			((g) & ATTACHED_FLAG)
#define RT_CONF_IS_DETACHABLE_FLAG(g)		((g) & DETACHABLE_FLAG)
#define RT_CONF_IS_SEPARATE_FLAG(g)			((g) & SEPARATE_FLAG)
#define RT_CONF_IS_FROZEN_FLAG(g)			((g) & FROZEN_FLAG)
#define RT_CONF_IS_VARIANT_FLAG(g)			((g) & VARIANT_FLAG)
#define RT_CONF_IS_POLY_FLAG(g)				((g) & POLY_FLAG)


#ifdef EIF_ASSERTIONS
	/* For each macro, we first check that `g' is indeed an annotation
	 * before checking the proper flag. */
#define RT_CONF_HAS_ATTACHMENT_MARK_TYPE_IN_ARRAY(g)	(CHECK_IN_EXPR("Has annotation", RT_CONF_HAS_ANNOTATION_TYPE_IN_ARRAY(g)), RT_CONF_HAS_ATTACHMENT_MARK_FLAG(g))
#define RT_CONF_HAS_VARIANT_MARK_TYPE_IN_ARRAY(g)		(CHECK_IN_EXPR("Has annotation", RT_CONF_HAS_ANNOTATION_TYPE_IN_ARRAY(g)), RT_CONF_HAS_VARIANT_MARK_FLAG(g))
#define RT_CONF_IS_ATTACHED_TYPE_IN_ARRAY(g)			(CHECK_IN_EXPR("Has annotation", RT_CONF_HAS_ANNOTATION_TYPE_IN_ARRAY(g)), RT_CONF_IS_ATTACHED_FLAG(g))
#define RT_CONF_IS_DETACHABLE_TYPE_IN_ARRAY(g)		(CHECK_IN_EXPR("Has annotation", RT_CONF_HAS_ANNOTATION_TYPE_IN_ARRAY(g)), RT_CONF_IS_DETACHABLE_FLAG(g))
#define RT_CONF_IS_SEPARATE_TYPE_IN_ARRAY(g)			(CHECK_IN_EXPR("Has annotation", RT_CONF_HAS_ANNOTATION_TYPE_IN_ARRAY(g)), RT_CONF_IS_SEPARATE_FLAG(g))
#define RT_CONF_IS_FROZEN_TYPE_IN_ARRAY(g)			(CHECK_IN_EXPR("Has annotation", RT_CONF_HAS_ANNOTATION_TYPE_IN_ARRAY(g)), RT_CONF_IS_FROZEN_FLAG(g))
#define RT_CONF_IS_VARIANT_TYPE_IN_ARRAY(g)			(CHECK_IN_EXPR("Has annotation", RT_CONF_HAS_ANNOTATION_TYPE_IN_ARRAY(g)), RT_CONF_IS_VARIANT_FLAG(g))
#define RT_CONF_IS_POLY_TYPE_IN_ARRAY(g)				(CHECK_IN_EXPR("Has annotation", RT_CONF_HAS_ANNOTATION_TYPE_IN_ARRAY(g)), RT_CONF_IS_POLY_FLAG(g))

#else
	/* Non-contracted versions. */
#define RT_CONF_HAS_ATTACHMENT_MARK_TYPE_IN_ARRAY(g)	RT_CONF_HAS_ATTACHMENT_MARK_FLAG(g)
#define RT_CONF_HAS_VARIANT_MARK_TYPE_IN_ARRAY(g)		RT_CONF_HAS_VARIANT_MARK_FLAG(g)
#define RT_CONF_IS_ATTACHED_TYPE_IN_ARRAY(g)			RT_CONF_IS_ATTACHED_FLAG(g)
#define RT_CONF_IS_DETACHABLE_TYPE_IN_ARRAY(g)		RT_CONF_IS_DETACHABLE_FLAG(g)
#define RT_CONF_IS_SEPARATE_TYPE_IN_ARRAY(g)			RT_CONF_IS_SEPARATE_FLAG(g)
#define RT_CONF_IS_FROZEN_TYPE_IN_ARRAY(g)			RT_CONF_IS_FROZEN_FLAG(g)
#define RT_CONF_IS_VARIANT_TYPE_IN_ARRAY(g)			RT_CONF_IS_VARIANT_FLAG(g)
#define RT_CONF_IS_POLY_TYPE_IN_ARRAY(g)				RT_CONF_IS_POLY_FLAG(g)

#endif

/*
doc:	<routine name="rt_is_conforming_annotation" return_type="int" export="private">
doc:		<summary>Check if the annotations of 2 types are conforming. If annotations are present, it is assumed that for each category with more than one choice a flag is set (i.e. that either ATTACHED_FLAG or DETACHABLE_FLAG is set, or that VARIANT_FLAG, FROZEN_FLAG or POLY_FLAG is set.</summary>
doc:		<param name="source" type="EIF_TYPE">Actual type of source.</param>
doc:		<param name="target" type="EIF_TYPE">Actual type of target.</param>
doc:		<return>1 when source annotation conforms to target, 0 otherwise.</return>
doc:		<thread_safety>Safe</thread_safety>
doc:		<synchronization>none</synchronization>
doc:	</routine>
*/
rt_private rt_inline int rt_is_conforming_annotation (EIF_TYPE source, EIF_TYPE target)
{
	int result = 0;

	REQUIRE("Is source annotation?", RT_CONF_IS_VALID_ANNOTATION(source.annotations));
	REQUIRE("Is target annotation?", RT_CONF_IS_VALID_ANNOTATION(target.annotations));

	if (source.annotations == target.annotations) {
			/* Trivally conforming. */
		result = 1;
	} else {
			/* If annotations are missing we need to create them. */
		if (!RT_CONF_HAS_ATTACHMENT_MARK_FLAG(source.annotations)) {
			source.annotations |= DETACHABLE_FLAG;
		}
		if (!RT_CONF_HAS_VARIANT_MARK_FLAG(source.annotations)) {
			source.annotations |= VARIANT_FLAG;
		}
		if (!RT_CONF_HAS_ATTACHMENT_MARK_FLAG(target.annotations)) {
			target.annotations |= DETACHABLE_FLAG;
		}
		if (!RT_CONF_HAS_VARIANT_MARK_FLAG(target.annotations)) {
			target.annotations |= VARIANT_FLAG;
		}

		REQUIRE("Attachment mark set on source", source.annotations & (ATTACHED_FLAG | DETACHABLE_FLAG));
		REQUIRE("Attachment mark set on target", target.annotations & (ATTACHED_FLAG | DETACHABLE_FLAG));
		REQUIRE("Variant mark set on source", source.annotations & (VARIANT_FLAG | POLY_FLAG | FROZEN_FLAG));
		REQUIRE("Variant mark set on target", target.annotations & (VARIANT_FLAG | POLY_FLAG | FROZEN_FLAG));

			/* 1. Check for void-safety conformance.
			 * We use a trick that the attached flag has a value greater than
			 * the detachable flag, which means that if `source_attachment_mark <= target_attachment_mark'
			 * then it does conform. */
		CHECK("Attachment flags are in proper order", ATTACHED_FLAG < DETACHABLE_FLAG);
		CHECK("No flag is lower than detachable flag", DETACHABLE_FLAG > 0);
		result = (source.annotations & (ATTACHED_FLAG | DETACHABLE_FLAG)) <= (target.annotations & (ATTACHED_FLAG | DETACHABLE_FLAG));

			/* 2. Check for SCOOP conformance. */
		if (result) {
				/* Same trick as above. */
			result = (source.annotations & SEPARATE_FLAG) <= (target.annotations & SEPARATE_FLAG);

				/* 3. Check for variance conformance. */
			if (result && !(target.annotations & VARIANT_FLAG)) {
					/* Target is not variant, then it only works if `source' and `target'
					 * are the same and that source and target have the same annotation
					 * (i.e. source is not variant too).
					 */
				result = (source.id == target.id) &&
					((source.annotations & (POLY_FLAG | FROZEN_FLAG)) == (target.annotations & (POLY_FLAG | FROZEN_FLAG)));
			}
		}
	}

	return result;
}

/* Offset that needs to be skipped when finding TUPLE_TYPE. It corresponds
 * to TUPLE_TYPE and nb generic parameters in current tuple type definition. */
#define TUPLE_OFFSET	2

/*------------------------------------------------------------------*/
/* One character codes for the basic types and one for all the      */
/* others. Make sure to assign different letters to new basic types.*/
/* You must update 'rout_obj.c' after adding new codes!!!           */
/* You must also update ROUTINE class after adding new codes!!      */
/*------------------------------------------------------------------*/

#define EIF_TUPLE_CODE_MASK 0x0F
#define EIF_REFERENCE_CODE	0x00
#define EIF_BOOLEAN_CODE	0x01
#define EIF_CHARACTER_8_CODE	0x02
#define EIF_REAL_64_CODE	0x03
#define EIF_REAL_32_CODE	0x04
#define EIF_POINTER_CODE	0x05
#define EIF_INTEGER_8_CODE	0x06
#define EIF_INTEGER_16_CODE	0x07
#define EIF_INTEGER_32_CODE	0x08
#define EIF_INTEGER_64_CODE	0x09
#define EIF_NATURAL_8_CODE	0x0A
#define EIF_NATURAL_16_CODE	0x0B
#define EIF_NATURAL_32_CODE 0x0C
#define EIF_NATURAL_64_CODE 0x0D
#define EIF_CHARACTER_32_CODE	0x0E

#define EIF_EXPANDED_CODE_EXTENSION 0x10

/*------------------------------------------------------------------*/

extern char eif_gen_typecode_with_dftype (EIF_TYPE_INDEX dftype, uint32 pos);

#ifdef EIF_ASSERTIONS
rt_unused rt_private int rt_valid_type_index(int dftype) {
	if (dftype < (int) MAX_DTYPE) {
		return 1;
	} else {
		return 0;
	}
}
#endif

#ifdef __cplusplus
}
#endif

#endif

