# libtuberia

## Copyright

Copyright (C) 2025 Nicolas Dato

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.

## About

libtuberia is a library in C to implement a pipeline.

A pipeline would be:

*[Source] -> [queue_1] -> [Stage_1] -> [Queue_2] -> [Stage_2] -> [...] -> [Sink]*

Each source, stage, and sink runs in a thread, reads from its input queue,
processes the element, and writes the result to the output queue.

This project is available:

- At my personal website: <https://ndato.com/projects/libtuberia/>
- At Savannah: <https://savannah.nongnu.org/projects/libtuberia/>

You can submit bugs at Savannah.

## Compiling and installing

If you don't have a ./configure file, run:

- `autoreconf -fi`

When you have the ./configure file, run:

- `./configure`
- `make`
- `make install`

You can see some configuration options with:

- `./configure --help`

## Using this library

Include tuberia.h, and link libtuberia.a and pthread: `-ltuberia -lpthread`

You can also use: `pkg-config --cflags --libs --static tuberia`

## Documentation

The `src/tuberia.h` file has every structure and function documented.

And if you had [Doxygen](https://www.doxygen.nl) when installing libtuberia, a
man page `tuberia.h` is also installed (you can read it with `man tuberia.h`)
and a .html is installed in `/usr/doc/libtuberia/doxygen_html/index.html`.

### Quick Guide

See the files `example/simple.c` and `example/decode_resize_encode.c`
(function `do_tuberia()` in particular)

Create a source with `tube_source_alloc()`.
If you set fetch to something, there will be a thread running fetch to inject
elements into the pipeline.
If you set fetch to NULL, you have to inject the elements with `tube_inject()`.

Create as many stages as you need with `tube_stage_alloc()`, and append them
to the source with `tube_stage_append()`.

Build the pipeline with `tube_alloc()`. If you set sink to something,
there will be a thread running sink to get the elements from the pipeline.
If you set sink to NULL, you have to retrieve the elements with
`tube_retrieve()`.

The `tube_alloc()` function copies the source and all the appended stages,
so you can free the source and stages with `tube_source_and_stages_free()`.
Or, you can reutilize the same `tube_source` to build more pipelines.

Start the pipeline with `tube_start()`.

Free the pipeline with `tube_free()`.


### Quick example

See `examples/simple.c`

Compile with: `gcc -o simple simple.c -ltuberia -lpthread`

Or with: `gcc -o simple simple.c $(pkg-config --cflags --libs --static tuberia)`

```C
#include <stdio.h>
#include <stdlib.h>
#include <tuberia.h>

void *stage_one(void *element, void *opaque)
{
    int *i = element;
    *i += 5;
    return i;
}

void *stage_two(void *element, void *opaque)
{
    int *i = element;
    *i *= 10;
    return i;
}

void sink(void *element, void *opaque)
{
    int *i = element;
    printf("sink: %d\n", *i);
    free(i);
}

int main(void)
{
    tube *ctx;
    tube_source *source;
    int *element, n;

    source = tube_source_alloc(2, NULL, NULL, free);
    tube_stage_append(source, tube_stage_alloc(2, stage_one, NULL, free));
    tube_stage_append(source, tube_stage_alloc(2, stage_two, NULL, free));
    ctx = tube_alloc(source, sink, NULL);
    tube_source_and_stages_free(&source);
    tube_start(ctx);
    while (scanf("%d", &n) == 1) {
        element = malloc(sizeof(*element));
        *element = n;
        tube_inject(ctx, 1000, element);
    }
    tube_free(&ctx);

    return 0;
}
```

