/* GSequencer - Advanced GTK Sequencer
 * Copyright (C) 2005-2025 Joël Krähemann
 *
 * This file is part of GSequencer.
 *
 * GSequencer is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GSequencer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GSequencer.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ags/audio/fx/ags_fx_buffer_audio_processor.h>

#include <ags/audio/fx/ags_fx_buffer_audio.h>

#include <ags/i18n.h>

void ags_fx_buffer_audio_processor_class_init(AgsFxBufferAudioProcessorClass *fx_buffer_audio_processor);
void ags_fx_buffer_audio_processor_init(AgsFxBufferAudioProcessor *fx_buffer_audio_processor);
void ags_fx_buffer_audio_processor_dispose(GObject *gobject);
void ags_fx_buffer_audio_processor_finalize(GObject *gobject);

void ags_fx_buffer_audio_processor_run_init_pre(AgsRecall *recall);
void ags_fx_buffer_audio_processor_cancel(AgsRecall *recall);

/**
 * SECTION:ags_fx_buffer_audio_processor
 * @short_description: fx buffer audio processor
 * @title: AgsFxBufferAudioProcessor
 * @section_id:
 * @include: ags/audio/fx/ags_fx_buffer_audio_processor.h
 *
 * The #AgsFxBufferAudioProcessor class provides ports to the effect processor.
 */

static gpointer ags_fx_buffer_audio_processor_parent_class = NULL;

GType
ags_fx_buffer_audio_processor_get_type()
{
  static gsize g_define_type_id__static = 0;

  if(g_once_init_enter(&g_define_type_id__static)){
    GType ags_type_fx_buffer_audio_processor = 0;

    static const GTypeInfo ags_fx_buffer_audio_processor_info = {
      sizeof (AgsFxBufferAudioProcessorClass),
      NULL, /* base_init */
      NULL, /* base_finalize */
      (GClassInitFunc) ags_fx_buffer_audio_processor_class_init,
      NULL, /* class_finalize */
      NULL, /* class_audio_processor */
      sizeof (AgsFxBufferAudioProcessor),
      0,    /* n_preallocs */
      (GInstanceInitFunc) ags_fx_buffer_audio_processor_init,
    };

    ags_type_fx_buffer_audio_processor = g_type_register_static(AGS_TYPE_RECALL_AUDIO_RUN,
								"AgsFxBufferAudioProcessor",
								&ags_fx_buffer_audio_processor_info,
								0);

    g_once_init_leave(&g_define_type_id__static, ags_type_fx_buffer_audio_processor);
  }

  return(g_define_type_id__static);
}

void
ags_fx_buffer_audio_processor_class_init(AgsFxBufferAudioProcessorClass *fx_buffer_audio_processor)
{
  GObjectClass *gobject;
  AgsRecallClass *recall;
  
  ags_fx_buffer_audio_processor_parent_class = g_type_class_peek_parent(fx_buffer_audio_processor);

  /* GObjectClass */
  gobject = (GObjectClass *) fx_buffer_audio_processor;

  gobject->dispose = ags_fx_buffer_audio_processor_dispose;
  gobject->finalize = ags_fx_buffer_audio_processor_finalize;

  /* AgsRecallClass */
  recall = (AgsRecallClass *) fx_buffer_audio_processor;
  
  recall->run_init_pre = ags_fx_buffer_audio_processor_run_init_pre;
  recall->cancel = ags_fx_buffer_audio_processor_cancel;
}

void
ags_fx_buffer_audio_processor_init(AgsFxBufferAudioProcessor *fx_buffer_audio_processor)
{
  AGS_RECALL(fx_buffer_audio_processor)->name = "ags-fx-buffer";
  AGS_RECALL(fx_buffer_audio_processor)->version = AGS_RECALL_DEFAULT_VERSION;
  AGS_RECALL(fx_buffer_audio_processor)->build_id = AGS_RECALL_DEFAULT_BUILD_ID;
  AGS_RECALL(fx_buffer_audio_processor)->xml_type = "ags-fx-buffer-audio-processor";
}

void
ags_fx_buffer_audio_processor_dispose(GObject *gobject)
{
  AgsFxBufferAudioProcessor *fx_buffer_audio_processor;
  
  fx_buffer_audio_processor = AGS_FX_BUFFER_AUDIO_PROCESSOR(gobject);
  
  /* call parent */
  G_OBJECT_CLASS(ags_fx_buffer_audio_processor_parent_class)->dispose(gobject);
}

void
ags_fx_buffer_audio_processor_finalize(GObject *gobject)
{
  AgsFxBufferAudioProcessor *fx_buffer_audio_processor;
  
  fx_buffer_audio_processor = AGS_FX_BUFFER_AUDIO_PROCESSOR(gobject);  
  
  /* call parent */
  G_OBJECT_CLASS(ags_fx_buffer_audio_processor_parent_class)->finalize(gobject);
}

void
ags_fx_buffer_audio_processor_run_init_pre(AgsRecall *recall)
{
  AgsFxBufferAudio *fx_buffer_audio;

  AgsFxBufferAudioScopeData *scope_data;

  gint sound_scope;

  GRecMutex *fx_buffer_audio_mutex;
  
  fx_buffer_audio = NULL;

  g_object_get(recall,
	       "recall-audio", &fx_buffer_audio,
	       NULL);
  
  sound_scope = ags_recall_get_sound_scope(recall);

  fx_buffer_audio_mutex = AGS_RECALL_GET_OBJ_MUTEX(fx_buffer_audio);

  g_rec_mutex_lock(fx_buffer_audio_mutex);
  
  scope_data = fx_buffer_audio->scope_data[sound_scope];

  if(sound_scope >= 0 &&
     scope_data != NULL){
    if(!fx_buffer_audio->scope_data_ready){
      fx_buffer_audio->scope_data_ready = TRUE;
      
      g_hash_table_remove_all(scope_data->destination);
    }
    
  }
  
  g_rec_mutex_unlock(fx_buffer_audio_mutex);
  
  /* call parent */
  AGS_RECALL_CLASS(ags_fx_buffer_audio_processor_parent_class)->run_init_pre(recall);
}

void
ags_fx_buffer_audio_processor_cancel(AgsRecall *recall)
{
  AgsFxBufferAudio *fx_buffer_audio;

  AgsFxBufferAudioScopeData *scope_data;

  gint sound_scope;

  GRecMutex *fx_buffer_audio_mutex;
  
  fx_buffer_audio = NULL;

  g_object_get(recall,
	       "recall-audio", &fx_buffer_audio,
	       NULL);
  
  sound_scope = ags_recall_get_sound_scope(recall);

  fx_buffer_audio_mutex = AGS_RECALL_GET_OBJ_MUTEX(fx_buffer_audio);

  g_rec_mutex_lock(fx_buffer_audio_mutex);
  
  scope_data = fx_buffer_audio->scope_data[sound_scope];

  if(sound_scope >= 0 &&
     scope_data != NULL){
    if(!fx_buffer_audio->scope_data_ready){
      fx_buffer_audio->scope_data_ready = FALSE;
    }
  }
  
  g_rec_mutex_unlock(fx_buffer_audio_mutex);
  
  /* call parent */
  AGS_RECALL_CLASS(ags_fx_buffer_audio_processor_parent_class)->cancel(recall);
}

/**
 * ags_fx_buffer_audio_processor_new:
 * @audio: the #AgsAudio
 *
 * Create a new instance of #AgsFxBufferAudioProcessor
 *
 * Returns: the new #AgsFxBufferAudioProcessor
 *
 * Since: 3.3.0
 */
AgsFxBufferAudioProcessor*
ags_fx_buffer_audio_processor_new(AgsAudio *audio)
{
  AgsFxBufferAudioProcessor *fx_buffer_audio_processor;

  fx_buffer_audio_processor = (AgsFxBufferAudioProcessor *) g_object_new(AGS_TYPE_FX_BUFFER_AUDIO_PROCESSOR,
									 "audio", audio,
									 NULL);

  return(fx_buffer_audio_processor);
}
