;;; Dezyne --- Dezyne command line tools
;;;
;;; Copyright © 2017, 2018, 2019, 2020, 2021, 2022, 2023, 2024, 2025 Janneke Nieuwenhuizen <janneke@gnu.org>
;;; Copyright © 2018, 2020, 2021 Paul Hoogendijk <paul@dezyne.org>
;;; Copyright © 2018, 2021, 2022, 2023, 2024 Rutger (regtur) van Beusekom <rutger@dezyne.org>
;;; Copyright © 2017, 2018 Johri van Eerd <vaneerd.johri@gmail.com>
;;; Copyright © 2017, 2018, 2019 Rob Wieringa <rma.wieringa@gmail.com>
;;; Copyright © 2017 Henk Katerberg <hank@mudball.nl>
;;;
;;; This file is part of Dezyne.
;;;
;;; Dezyne is free software: you can redistribute it and/or modify it
;;; under the terms of the GNU Affero General Public License as
;;; published by the Free Software Foundation, either version 3 of the
;;; License, or (at your option) any later version.
;;;
;;; Dezyne is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; Affero General Public License for more details.
;;;
;;; You should have received a copy of the GNU Affero General Public
;;; License along with Dezyne.  If not, see <http://www.gnu.org/licenses/>.
;;;
;;; Commentary:
;;;
;;; Code:

(define-module (dzn commands verify)
  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-26)

  #:use-module (ice-9 getopt-long)

  #:use-module (dzn ast)
  #:use-module (dzn ast ast)
  #:use-module (dzn code)
  #:use-module (dzn code language makreel)
  #:use-module (dzn command-line)
  #:use-module (dzn commands parse)
  #:use-module (dzn config)
  #:use-module (dzn misc)
  #:use-module (dzn parse)
  #:use-module (dzn shell-util)
  #:use-module (dzn verify pipeline)
  #:export (parse-opts
            main))

(define (parse-opts args)
  (let* ((option-spec
          '((all (single-char #\a))
            (fall-back (single-char #\f))
            (help (single-char #\h))
            (import (single-char #\I) (value #t))
            (keep-going (single-char #\k))
            (model (single-char #\m) (value #t))
            (no-constraint (single-char #\C))
            (no-interfaces)
            (no-unreachable (single-char #\U))
            (out (value #t))
            (queue-size (single-char #\q) (value #t))
            (queue-size-defer (value #t))
            (queue-size-external (value #t))))
         (options (getopt-long args option-spec))
         (help? (option-ref options 'help #f))
         (files (option-ref options '() '()))
         (out (option-ref options 'out #f))
         (usage? (and (not help?) (null? files))))
    (when (equal? out "help")
      (format #t "formats:~a\n" (string-join (verification:formats) "\n  " 'prefix))
      (exit EXIT_SUCCESS))
    (when (or help? usage?)
      (let ((port (if usage? (current-error-port) (current-output-port))))
        (format port "\
Usage: dzn verify [OPTION]... {DZN-FILE|DIRECTORY}...
Check FILEs and DIRECTORYs for verification errors in Dezyne models

  -a, --all                keep going after first error DEPRECATED
  -C, --no-constraint      do not use a constraining process
  -h, --help               display this help and exit
  -I, --import=DIR+        add DIR to import path
  -k, --keep-going         keep going after finding an error
  -m, --model=MODEL        restrict verification to model MODEL
      --no-interfaces      skip interface verification
      --out=FORMAT         produce output FORMAT (use \"help\" for a list)
  -U, --no-unreachable     skip the unreachable code check
  -q, --queue-size=SIZE    use queue size=SIZE for verification [~a]
      --queue-size-defer=SIZE
                           use defer queue size=SIZE for verification [~a]
      --queue-size-external=SIZE
                           use external queue size=SIZE for verification [~a]
" (%queue-size) (%queue-size-defer) (%queue-size-external))
        (exit (or (and usage? EXIT_OTHER_FAILURE) EXIT_SUCCESS))))
    options))

(define* (file->verify file-name
                       #:key
                       backtrace?
                       keep-going?
                       model-name
                       multiple?
                       no-constraint?
                       no-interfaces?
                       no-unreachable?
                       (options '())
                       out
                       queue-size
                       queue-size-defer
                       queue-size-external
                       verbose?)
  (let* ((ast (parse options file-name))
         (model (and model-name
                     (parse:call-with-handle-exceptions
                      (lambda _ (ast:get-model ast model-name))
                      #:backtrace? backtrace?
                      #:file-name file-name))))
    (when (and (and=> model ast:imported?)
               (is-a? model <component>))
      (let ((name (ast:dotted-name model)))
        (format (current-error-port)
                "~a:error: cannot verify imported model: ~a\n"
                (ast:source-file ast)
                name)
        (format (current-error-port)
                "~a:info: ~a imported from here\n"
                (ast:source-file model)
                name))
      (exit EXIT_OTHER_FAILURE))
    (let ((root (makreel:normalize ast)))
      (cond
       (out
        (let ((formats (verification:formats)))
          (unless (member out formats)
            (format #t "formats:~a\n"
                    (string-join (verification:formats)
                                 "\n  " 'prefix))
            (exit EXIT_OTHER_FAILURE))
          (verification:partial root model-name #:out out)))
       (else
        (when verbose?
          (format (current-error-port) "  VERIFY   ~a\n" file-name))
        (let ((result (verification:verify
                       options root
                       #:keep-going? keep-going?
                       #:model-name model-name
                       #:multiple? multiple?
                       #:no-interfaces? no-interfaces?)))
          (unless (zero? result)
            (exit result))))))))

(define (main args)
  (setvbuf (current-output-port) 'line)
  (setvbuf (current-error-port) 'line)
  (let* ((options (parse-opts args))
         (files (option-ref options '() '()))
         (files (append-map file-name->dzn-files files))
         (multiple? (> (length files) 1))
         (all? (option-ref options 'all #f))
         (debug? (dzn:command-line:get 'debug #f))
         (keep-going? (option-ref options 'keep-going #f))
         (out (option-ref options 'out #f))
         (model-name (option-ref options 'model #f))
         (no-constraint? (command-line:get 'no-constraint))
         (no-interfaces? (command-line:get 'no-interfaces))
         (no-unreachable? (command-line:get 'no-unreachable))
         (queue-size (option-ref options 'queue-size (%queue-size)))
         (queue-size-defer (option-ref options 'queue-size-defer
                                       (%queue-size-defer)))
         (queue-size-external (option-ref options 'queue-size-external
                                          (%queue-size-external))))
    (when all?
      (format (current-error-port)
              "warning: -a,--all is deprecated, use -k,--keep-going.\n"))
    (parameterize ((%context (%context))
                   (%language "makreel")
                   (%no-constraint? no-constraint?)
                   (%no-unreachable? no-unreachable?)
                   (%queue-size queue-size)
                   (%queue-size-defer queue-size-defer)
                   (%queue-size-external queue-size-external))
      (for-each (cut file->verify <>
                     #:backtrace? debug?
                     #:keep-going? keep-going?
                     #:model-name model-name
                     #:multiple? multiple?
                     #:no-constraint? no-constraint?
                     #:no-interfaces? no-interfaces?
                     #:no-unreachable? no-unreachable?
                     #:options options
                     #:out out
                     #:queue-size queue-size
                     #:queue-size-defer queue-size-defer
                     #:queue-size-external queue-size-external
                     #:verbose? multiple?)
                files))))
