/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import { canonicalizeBaseUrl } from "@gnu-taler/taler-util";
import { StateUpdater, useState } from "preact/hooks";
import { ValueOrFunction } from "../utils/types.js";

export function useBackendURL(url?: string): [string, StateUpdater<string>] {
  const canonUrl = canonicalizeBaseUrl(url ?? calculateRootPath());

  const [value, setter] = useSimpleLocalStorage("auditor-base-url", canonUrl);

  const checkedSetter = (v: ValueOrFunction<string>) => {
    // FIXME: Explain?!
    return setter((p) =>
      (v instanceof Function ? v(p ?? "") : v).replace(/\/$/, ""),
    );
  };

  return [value!, checkedSetter];
}

function calculateRootPath() {
  const rootPath =
    typeof window !== undefined
      ? window.location.origin + window.location.pathname
      : "/";

  /*
   * By default, auditor backend serves the html content
   * from the /webui root. This should cover most of the
   * cases and the rootPath will be the auditor backend
   * URL where the instances are
   */
  return rootPath.replace("/spa/", "");
}

function useSimpleLocalStorage(
  key: string,
  initialValue?: string,
): [string | undefined, StateUpdater<string | undefined>] {
  const [storedValue, setStoredValue] = useState<string | undefined>(
    (): string | undefined => {
      return typeof window !== "undefined"
        ? window.localStorage.getItem(key) || initialValue
        : initialValue;
    },
  );

  const setValue = (
    value?: string | ((val?: string) => string | undefined),
  ) => {
    setStoredValue((p) => {
      const toStore = value instanceof Function ? value(p) : value;
      if (typeof window !== "undefined") {
        if (!toStore) {
          window.localStorage.removeItem(key);
        } else {
          window.localStorage.setItem(key, toStore);
        }
      }
      return toStore;
    });
  };

  return [storedValue, setValue];
}
