/*
  This file is part of TALER
  Copyright (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Lesser General Public License as published by the Free Software
  Foundation; either version 2.1, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public License along with
  TALER; see the file COPYING.LGPL.  If not, see <http://www.gnu.org/licenses/>
*/
/**
 * @file merchant_api_delete_unit.c
 * @brief Implementation of DELETE /private/units/$ID
 * @author Bohdan Potuzhnyi
 */
#include "platform.h"
#include <curl/curl.h>
#include <jansson.h>
#include <microhttpd.h>
#include <gnunet/gnunet_util_lib.h>
#include <gnunet/gnunet_curl_lib.h>
#include "taler_merchant_service.h"
#include "merchant_api_curl_defaults.h"
#include "merchant_api_common.h"
#include <taler/taler_json_lib.h>


/**
 * Handle for a DELETE /private/units/$ID operation.
 */
struct TALER_MERCHANT_UnitDeleteHandle
{
  /**
   * Fully qualified request URL.
   */
  char *url;

  /**
   * In-flight CURL job.
   */
  struct GNUNET_CURL_Job *job;

  /**
   * Completion callback.
   */
  TALER_MERCHANT_UnitDeleteCallback cb;

  /**
   * Closure for @a cb.
   */
  void *cb_cls;

  /**
   * Execution context.
   */
  struct GNUNET_CURL_Context *ctx;
};


/**
 * Called when the HTTP request finishes.
 *
 * @param cls operation handle
 * @param response_code HTTP status (0 on failure)
 * @param response parsed JSON reply (NULL if unavailable)
 */
static void
handle_delete_unit_finished (void *cls,
                             long response_code,
                             const void *response)
{
  struct TALER_MERCHANT_UnitDeleteHandle *udh = cls;
  const json_t *json = response;
  struct TALER_MERCHANT_HttpResponse hr = {
    .http_status = (unsigned int) response_code,
    .reply = json
  };

  udh->job = NULL;
  GNUNET_log (GNUNET_ERROR_TYPE_INFO,
              "DELETE /private/units finished with status %u\n",
              (unsigned int) response_code);
  switch (response_code)
  {
  case MHD_HTTP_NO_CONTENT:
    break;
  case MHD_HTTP_BAD_REQUEST:
  case MHD_HTTP_UNAUTHORIZED:
  case MHD_HTTP_FORBIDDEN:
  case MHD_HTTP_NOT_FOUND:
  case MHD_HTTP_CONFLICT:
  case MHD_HTTP_INTERNAL_SERVER_ERROR:
    hr.ec = TALER_JSON_get_error_code (json);
    hr.hint = TALER_JSON_get_error_hint (json);
    break;
  case 0:
    hr.ec = TALER_EC_GENERIC_INVALID_RESPONSE;
    break;
  default:
    TALER_MERCHANT_parse_error_details_ (json,
                                         response_code,
                                         &hr);
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Unexpected response %u/%d for DELETE /private/units\n",
                (unsigned int) response_code,
                (int) hr.ec);
    GNUNET_break_op (0);
    break;
  }
  udh->cb (udh->cb_cls,
           &hr);
  TALER_MERCHANT_unit_delete_cancel (udh);
}


struct TALER_MERCHANT_UnitDeleteHandle *
TALER_MERCHANT_unit_delete (struct GNUNET_CURL_Context *ctx,
                            const char *backend_url,
                            const char *unit_id,
                            TALER_MERCHANT_UnitDeleteCallback cb,
                            void *cb_cls)
{
  struct TALER_MERCHANT_UnitDeleteHandle *udh;
  CURL *eh;
  char *path;

  GNUNET_asprintf (&path,
                   "private/units/%s",
                   unit_id);
  udh = GNUNET_new (struct TALER_MERCHANT_UnitDeleteHandle);
  udh->ctx = ctx;
  udh->cb = cb;
  udh->cb_cls = cb_cls;
  udh->url = TALER_url_join (backend_url,
                             path,
                             NULL);
  GNUNET_free (path);
  if (NULL == udh->url)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Failed to build /private/units/%s URL\n",
                unit_id);
    GNUNET_free (udh);
    return NULL;
  }
  GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
              "Requesting DELETE on '%s'\n",
              udh->url);
  eh = TALER_MERCHANT_curl_easy_get_ (udh->url);
  GNUNET_assert (CURLE_OK ==
                 curl_easy_setopt (eh,
                                   CURLOPT_CUSTOMREQUEST,
                                   MHD_HTTP_METHOD_DELETE));
  udh->job = GNUNET_CURL_job_add (ctx,
                                  eh,
                                  &handle_delete_unit_finished,
                                  udh);
  return udh;
}


void
TALER_MERCHANT_unit_delete_cancel (struct TALER_MERCHANT_UnitDeleteHandle *udh)
{
  if (NULL != udh->job)
    GNUNET_CURL_job_cancel (udh->job);
  GNUNET_free (udh->url);
  GNUNET_free (udh);
}


/* end of merchant_api_delete_unit.c */
