/*
  This file is part of TALER
  Copyright (C) 2015-2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  TALER; see the file COPYING.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file lib/exchange_api_refresh_common.c
 * @brief Serialization logic shared between melt and reveal steps during refreshing
 * @author Christian Grothoff
 * @author Özgür Kesim
 */
#include "taler/platform.h"
#include "exchange_api_refresh_common.h"


void
TALER_EXCHANGE_free_melt_data (struct MeltData *md)
{
  for (unsigned int k = 0; k < TALER_CNC_KAPPA; k++)
  {
    for (unsigned int i = 0; i < md->num_fresh_coins; i++)
      TALER_blinded_planchet_free (&md->kappa_blinded_planchets[k][i]);
    GNUNET_free (md->kappa_blinded_planchets[k]);
    GNUNET_free (md->kappa_transfer_pubs[k]);
  }
  GNUNET_free (md->denoms_h);
  GNUNET_free (md->denom_pubs);
  TALER_denom_pub_free (&md->melted_coin.pub_key);
  TALER_denom_sig_free (&md->melted_coin.sig);
  if (NULL != md->fcds)
  {
    for (unsigned int j = 0; j<md->num_fresh_coins; j++)
    {
      struct FreshCoinData *fcd = &md->fcds[j];

      TALER_denom_pub_free (&fcd->fresh_pk);
      for (size_t i = 0; i < TALER_CNC_KAPPA; i++)
      {
        TALER_age_commitment_proof_free (fcd->age_commitment_proofs[i]);
        GNUNET_free (fcd->age_commitment_proofs[i]);
      }
    }
    GNUNET_free (md->fcds);
  }
  /* Finally, clean up a bit... */
  GNUNET_CRYPTO_zero_keys (md,
                           sizeof (struct MeltData));
}


enum GNUNET_GenericReturnValue
TALER_EXCHANGE_get_melt_data (
  const struct TALER_PublicRefreshMasterSeedP *rms,
  const struct TALER_EXCHANGE_MeltInput *rd,
  const struct TALER_BlindingMasterSeedP *blinding_seed,
  const struct TALER_ExchangeBlindingValues *blinding_values,
  struct MeltData *md)
{
  struct TALER_Amount total;
  struct TALER_CoinSpendPublicKeyP coin_pub;
  struct TALER_KappaHashBlindedPlanchetsP k_h_bps;
  union GNUNET_CRYPTO_BlindSessionNonce nonces[rd->num_fresh_denom_pubs];
  bool is_cs[rd->num_fresh_denom_pubs];
  bool uses_cs = false;

  GNUNET_CRYPTO_eddsa_key_get_public (&rd->melt_priv.eddsa_priv,
                                      &coin_pub.eddsa_pub);
  memset (md,
          0,
          sizeof (*md));
  /* build up melt data structure */
  md->refresh_seed = *rms;
  md->num_fresh_coins = rd->num_fresh_denom_pubs;
  md->melted_coin.coin_priv = rd->melt_priv;
  md->melted_coin.melt_amount_with_fee = rd->melt_amount;
  md->melted_coin.fee_melt = rd->melt_pk.fees.refresh;
  md->melted_coin.original_value = rd->melt_pk.value;
  md->melted_coin.expire_deposit = rd->melt_pk.expire_deposit;
  md->melted_coin.age_commitment_proof = rd->melt_age_commitment_proof;
  md->melted_coin.h_age_commitment = rd->melt_h_age_commitment;
  md->no_blinding_seed = (NULL == blinding_seed);
  if (NULL != blinding_seed)
    md->blinding_seed = *blinding_seed;

  GNUNET_assert (GNUNET_OK ==
                 TALER_amount_set_zero (rd->melt_amount.currency,
                                        &total));
  TALER_denom_pub_copy (&md->melted_coin.pub_key,
                        &rd->melt_pk.key);
  TALER_denom_sig_copy (&md->melted_coin.sig,
                        &rd->melt_sig);
  md->fcds = GNUNET_new_array (md->num_fresh_coins,
                               struct FreshCoinData);
  md->denoms_h =
    GNUNET_new_array (md->num_fresh_coins,
                      struct  TALER_DenominationHashP);
  md->denom_pubs =
    GNUNET_new_array (md->num_fresh_coins,
                      const struct  TALER_DenominationPublicKey*);

  for (unsigned int j = 0; j<md->num_fresh_coins; j++)
  {
    struct FreshCoinData *fcd = &md->fcds[j];

    md->denoms_h[j] = rd->fresh_denom_pubs[j].h_key;
    md->denom_pubs[j] = &rd->fresh_denom_pubs[j].key;

    TALER_denom_pub_copy (&fcd->fresh_pk,
                          &rd->fresh_denom_pubs[j].key);
    GNUNET_assert (NULL != fcd->fresh_pk.bsign_pub_key);
    if (blinding_values[j].blinding_inputs->cipher !=
        fcd->fresh_pk.bsign_pub_key->cipher)
    {
      GNUNET_break (0);
      TALER_EXCHANGE_free_melt_data (md);
      return GNUNET_SYSERR;
    }
    switch (fcd->fresh_pk.bsign_pub_key->cipher)
    {
    case GNUNET_CRYPTO_BSA_INVALID:
      GNUNET_break (0);
      TALER_EXCHANGE_free_melt_data (md);
      return GNUNET_SYSERR;
    case GNUNET_CRYPTO_BSA_RSA:
      is_cs[j] = false;
      break;
    case GNUNET_CRYPTO_BSA_CS:
      uses_cs = true;
      is_cs[j] = true;
      break;
    }
    if ( (0 >
          TALER_amount_add (&total,
                            &total,
                            &rd->fresh_denom_pubs[j].value)) ||
         (0 >
          TALER_amount_add (&total,
                            &total,
                            &rd->fresh_denom_pubs[j].fees.withdraw)) )
    {
      GNUNET_break (0);
      TALER_EXCHANGE_free_melt_data (md);
      return GNUNET_SYSERR;
    }
  }

  /* verify that melt_amount is above total cost */
  if (1 ==
      TALER_amount_cmp (&total,
                        &rd->melt_amount) )
  {
    /* Eh, this operation is more expensive than the
       @a melt_amount. This is not OK. */
    GNUNET_break (0);
    TALER_EXCHANGE_free_melt_data (md);
    return GNUNET_SYSERR;
  }
  /**
   * Generate the blinding seeds and nonces for CS.
   * Note that blinding_seed was prepared upstream,
   * in TALER_EXCHANGE_melt(), as preparation for
   * the request to `/blinding-prepare`.
   */
  memset (nonces,
          0,
          sizeof(*nonces));
  if (uses_cs)
  {
    GNUNET_assert (! md->no_blinding_seed);
    TALER_cs_derive_blind_nonces_from_seed (
      blinding_seed,
      true, /* for melt */
      md->num_fresh_coins,
      is_cs,
      nonces);
  }
  /**
   * Generate the kappa private seeds for the batches.
   */
  TALER_refresh_expand_seed_to_kappa_batch_seeds (
    &md->refresh_seed,
    &rd->melt_priv,
    &md->kappa_batch_seeds);
  /**
   * Build up all candidates for coin planchets
   */
  for (unsigned int k = 0; k<TALER_CNC_KAPPA; k++)
  {
    struct TALER_PlanchetMasterSecretP
      planchet_secrets[md->num_fresh_coins];
    struct TALER_PlanchetDetail planchet_details[md->num_fresh_coins];

    md->kappa_blinded_planchets[k] =
      GNUNET_new_array (md->num_fresh_coins,
                        struct TALER_BlindedPlanchet);
    md->kappa_transfer_pubs[k] =
      GNUNET_new_array (md->num_fresh_coins,
                        struct TALER_TransferPublicKeyP);

    TALER_refresh_expand_batch_seed_to_transfer_data (
      &md->kappa_batch_seeds.tuple[k],
      &coin_pub,
      md->num_fresh_coins,
      planchet_secrets,
      md->kappa_transfer_pubs[k]);

    for (unsigned int j = 0; j<md->num_fresh_coins; j++)
    {
      struct FreshCoinData *fcd = &md->fcds[j];
      struct TALER_CoinSpendPrivateKeyP *coin_priv = &fcd->coin_priv;
      union GNUNET_CRYPTO_BlindingSecretP *bks = &fcd->bks[k];
      struct TALER_CoinPubHashP c_hash;
      struct TALER_AgeCommitmentHashP ach;
      struct TALER_AgeCommitmentHashP *pah;

      fcd->ps[k] = planchet_secrets[j];
      TALER_planchet_setup_coin_priv (&planchet_secrets[j],
                                      &blinding_values[j],
                                      coin_priv);
      TALER_planchet_blinding_secret_create (&planchet_secrets[j],
                                             &blinding_values[j],
                                             bks);
      if (NULL != rd->melt_age_commitment_proof)
      {
        fcd->age_commitment_proofs[k] =
          GNUNET_new (struct TALER_AgeCommitmentProof);
        GNUNET_assert (GNUNET_OK ==
                       TALER_age_commitment_proof_derive_from_secret (
                         md->melted_coin.age_commitment_proof,
                         &planchet_secrets[j],
                         fcd->age_commitment_proofs[k]));
        TALER_age_commitment_hash (
          &fcd->age_commitment_proofs[k]->commitment,
          &ach);
        pah = &ach;
      }
      else
      {
        pah = NULL;
      }

      if (GNUNET_OK !=
          TALER_planchet_prepare (&fcd->fresh_pk,
                                  &blinding_values[j],
                                  bks,
                                  is_cs[j] ? &nonces[j] : NULL,
                                  coin_priv,
                                  pah,
                                  &c_hash,
                                  &planchet_details[j]))
      {
        GNUNET_break_op (0);
        TALER_EXCHANGE_free_melt_data (md);
        return GNUNET_SYSERR;
      }
      md->kappa_blinded_planchets[k][j] =
        planchet_details[j].blinded_planchet;
    }
    /**
     * Compute the hash of this batch of blinded planchets
     */
    TALER_wallet_blinded_planchet_details_hash (
      rd->num_fresh_denom_pubs,
      planchet_details,
      &k_h_bps.tuple[k]);
  }

  /* Finally, compute refresh commitment */
  {
    struct TALER_KappaTransferPublicKeys k_tr_pubs = {
      .num_transfer_pubs = md->num_fresh_coins,
    };

    for (uint8_t k=0; k<TALER_CNC_KAPPA; k++)
      k_tr_pubs.batch[k] = md->kappa_transfer_pubs[k];

    TALER_refresh_get_commitment (&md->rc,
                                  &md->refresh_seed,
                                  uses_cs
                                         ? &md->blinding_seed
                                         : NULL,
                                  &k_tr_pubs,
                                  &k_h_bps,
                                  &coin_pub,
                                  &rd->melt_amount);
  }
  return GNUNET_OK;
}
