/* $Id: gdict-defbox.c,v 1.55 2005/06/16 15:03:15 vnoel Exp $ */

/*
 *  Mike Hughes <mfh@psilord.com>
 *  Papadimitriou Spiros <spapadim+@cs.cmu.edu>
 *  Bradford Hovinen <hovinen@udel.edu>
 *
 *  This code released under the GNU GPL.
 *  Read the file COPYING for more information.
 *
 *  GDict main window
 *
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <math.h>
#include <glib/gi18n.h>
#include "gdict-defbox.h"
#include "gdict-app.h"
#include <libgnomeprint/gnome-print-pango.h>
#include <libgnomeprintui/gnome-print-job-preview.h>

enum {
    WORD_LOOKUP_START_SIGNAL,
    WORD_LOOKUP_DONE_SIGNAL,
    WORD_NOT_FOUND_SIGNAL,
    SUBSTR_NOT_FOUND_SIGNAL,
    SOCKET_ERROR_SIGNAL,
    LAST_SIGNAL
};

typedef enum {
	BOLD = 1,
	ITALIC = 2,
	UNDERLINE = 4,
	EMPHASIS = 8,
	GREY = 16,
	BIG = 32
}GDictFaces;

#define LEFT_MARGIN 72.
#define RIGHT_MARGIN 72.
#define TOP_MARGIN 72.

static gint gdict_defbox_signals[LAST_SIGNAL] = { 0, 0, 0, 0 };

static void gdict_defbox_init (GDictDefbox *defbox);
static void gdict_defbox_class_init (GDictDefboxClass *class);

static void gdict_defbox_error_cb  (dict_command_t *command, DictStatusCode code, 
                                    gchar *message, gpointer data);
static void gdict_defbox_status_cb (dict_command_t *command, DictStatusCode code, 
                                    int num_found, gpointer data);
static void gdict_defbox_data_cb   (dict_command_t *command, dict_res_t *res,
                                    gpointer data);
static gboolean is_xref   (gchar *str, int len);
static gboolean is_number (gchar *str, int len);
static gboolean is_part   (gchar *str, int len);

/* Setup textview tags
 */
void
gdict_defbox_setup_tags (GDictDefbox *defbox)
{
    GtkTextBuffer *buffer;

    buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (defbox));

    gtk_text_buffer_create_tag (buffer, "italic",
			        "style", PANGO_STYLE_ITALIC, NULL);
    gtk_text_buffer_create_tag (buffer, "bold",
			        "weight", PANGO_WEIGHT_BOLD, NULL);
    gtk_text_buffer_create_tag (buffer, "underline",
			        "underline", PANGO_UNDERLINE_SINGLE, NULL);
		gtk_text_buffer_create_tag (buffer, "big",
																"scale", 1.6,
																"pixels-below-lines", 5,
																"pixels-above-lines", 5, NULL);
		gtk_text_buffer_create_tag (buffer, "grey",
																"foreground", "dark gray", NULL);
		gtk_text_buffer_create_tag (buffer, "emphasis",
																"foreground", "dark green", NULL);
}

static void
gdict_defbox_lookup_start_cb (GtkWidget *widget, gpointer data) 
{
  GDictWindow *gdict = data;
  static GdkCursor *busy_cursor = NULL;
  GtkWidget *window_widget = gtk_widget_get_toplevel(GTK_WIDGET(gdict));

  if (busy_cursor == NULL)
    busy_cursor = gdk_cursor_new (GDK_WATCH);
  gdk_window_set_cursor (GTK_WIDGET(gdict)->window, busy_cursor);
  gdict_menus_set_sensitive (gdict, FALSE);

  gtk_window_set_title (GTK_WINDOW (gdict), _("Dictionary"));
	gtk_widget_show (gdict->statusbar);
  gtk_statusbar_push (GTK_STATUSBAR (gdict->statusbar), 0, _("Looking up word..."));
}

static void
gdict_defbox_lookup_done_cb (GtkWidget *widget, gpointer data) 
{
  gchar *word, *title;
  GDictWindow *gdict = data;
  GtkAction *action;

  word = gtk_editable_get_chars (GTK_EDITABLE (gdict->word_entry), 0, -1);
  title = g_strconcat (word, " - ", _("Dictionary"), NULL);
  gtk_window_set_title (GTK_WINDOW (gdict), title);
  g_free (word);
  g_free (title);

	gtk_widget_hide (gdict->statusbar);
  gdict_menus_set_sensitive (gdict, TRUE);

  gdk_window_set_cursor (GTK_WIDGET(gdict)->window, NULL);
}

static void
gdict_defbox_not_found_cb (GtkWidget *widget, gpointer data) 
{
  GDictWindow *gdict = data;
  gchar *word;
	gtk_widget_show (gdict->statusbar);
  gtk_statusbar_pop (GTK_STATUSBAR (gdict->statusbar), 0);
  gtk_statusbar_push (GTK_STATUSBAR (gdict->statusbar), 0, _("No matches found"));
  word = gdict_defbox_get_word (gdict->defbox);
  gdict_lookup_spelling (gdict, word, FALSE);
}

static void
gdict_defbox_substr_not_found_cb (GtkWidget *widget, gpointer data) 
{
  GDictWindow *gdict = data;
	gtk_widget_show (gdict->statusbar);
  gtk_statusbar_pop (GTK_STATUSBAR (gdict->statusbar), 0);
  gtk_statusbar_push (GTK_STATUSBAR (gdict->statusbar), 0, _("String not found"));
}

static void
gdict_defbox_follow_if_link (GtkWidget *defbox, GtkTextIter *iter, GDictWindow *gdict, int mousebutton)
{
  GSList *tags = NULL, *tagp = NULL;
  gchar *str;

  tags = gtk_text_iter_get_tags (iter);
  for (tagp = tags;  tagp != NULL;  tagp = tagp->next) {
    GtkTextTag *tag = tagp->data;
    str = g_object_get_data (G_OBJECT (tag), "page");

		if (str) {

			if (mousebutton == 2)
				gdict = GDICT_WINDOW (gdict_window_new());
			else
				gdict_defbox_clear (GDICT_DEFBOX (defbox));
			
			gdict_lookup_definition (gdict, str);
			gtk_widget_grab_focus (gdict->word_entry);
			gtk_editable_set_position (GTK_EDITABLE (gdict->word_entry), -1);
			g_free (str);
		}
  }

  if (tags) 
    g_slist_free (tags);
}

static void
gdict_defbox_set_cursor_if_appropriate (GDictWindow *gdict, GtkWidget *defbox, gint x, gint y)
{
  GSList *tags = NULL, *tagp = NULL;
  GtkTextBuffer *buffer;
  GtkTextIter iter;
  static gboolean hovering_over_link = FALSE;
  static GdkCursor *hand_cursor = NULL;
  static GdkCursor *regular_cursor = NULL;
  gboolean hovering = FALSE;

  if (hand_cursor == NULL)
    hand_cursor = gdk_cursor_new (GDK_HAND2);
  if (regular_cursor == NULL)
    regular_cursor = gdk_cursor_new (GDK_XTERM);

  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (defbox));

  gtk_text_view_get_iter_at_location (GTK_TEXT_VIEW (defbox), &iter, x, y);

  tags = gtk_text_iter_get_tags (&iter);
  for (tagp = tags;  tagp != NULL;  tagp = tagp->next) {
    GtkTextTag *tag = tagp->data;
    gint page = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (tag), "page"));

    if (page != 0) {
      hovering = TRUE;
      break;
    }
  }

  if (hovering != hovering_over_link) {
    hovering_over_link = hovering;

    if (hovering_over_link)
      gdk_window_set_cursor (gtk_text_view_get_window (GTK_TEXT_VIEW (defbox),
                 GTK_TEXT_WINDOW_TEXT), hand_cursor);
    else
      gdk_window_set_cursor (gtk_text_view_get_window (GTK_TEXT_VIEW (defbox),
                 GTK_TEXT_WINDOW_TEXT), regular_cursor);
  }

  if (tags)
    g_slist_free (tags);
}

static gboolean
gdict_defbox_event_after (GtkWidget *defbox, GdkEvent *ev, gpointer data)
{
  GDictWindow *gdict = data;
  GtkTextIter iter;
  GtkTextBuffer *buffer;
  GdkEventButton *event;
  gint x, y;

  if (ev->type != GDK_BUTTON_RELEASE)
    return FALSE;

  event = (GdkEventButton *)ev;
  if (event->button != 1 && event->button !=2)
    return FALSE;

  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (defbox));

  if (gtk_text_buffer_get_selection_bounds (buffer, NULL, NULL))
    return FALSE;

  gtk_text_view_window_to_buffer_coords (GTK_TEXT_VIEW (defbox), 
                 GTK_TEXT_WINDOW_WIDGET,
                 event->x, event->y, &x, &y);

  gtk_text_view_get_iter_at_location (GTK_TEXT_VIEW (defbox), &iter, x, y);

  gdict_defbox_follow_if_link (defbox, &iter, gdict, event->button);

  return FALSE;
}

static gboolean
gdict_defbox_motion_notify_event (GtkWidget *defbox,
                     GdkEventMotion *event, gpointer data)
{
  GDictWindow *gdict = data;
  gint x, y;

  gtk_text_view_window_to_buffer_coords (GTK_TEXT_VIEW (defbox), 
                 GTK_TEXT_WINDOW_WIDGET,
                 event->x, event->y, &x, &y);
  
  gdict_defbox_set_cursor_if_appropriate (gdict, defbox, x, y);
  gdk_window_get_pointer (GTK_WIDGET(defbox)->window, NULL, NULL, NULL);

  return FALSE;
}

/* update the cursor image if the window becomes visible */
static gboolean
gdict_defbox_visibility_notify_event (GtkWidget *defbox,
                         GdkEventVisibility *event, gpointer data)
{
  GDictWindow *gdict = data;
  gint wx, wy, bx, by;

  gdk_window_get_pointer (GTK_WIDGET(defbox)->window, &wx, &wy, NULL);

  gtk_text_view_window_to_buffer_coords (GTK_TEXT_VIEW (defbox), 
                 GTK_TEXT_WINDOW_WIDGET,
                 wx, wy, &bx, &by);

  gdict_defbox_set_cursor_if_appropriate (gdict, defbox, bx, by);

  return FALSE;
}

GType 
gdict_defbox_get_type (void)
{
    static GType gdict_defbox_type = 0;
    
    if (!gdict_defbox_type) {
        static const GTypeInfo gdict_defbox_info = {
            sizeof (GDictDefboxClass),
            NULL,
            NULL,
            (GClassInitFunc) gdict_defbox_class_init,
            NULL,
            NULL,
	    sizeof (GDictDefbox),
	    0,
            (GInstanceInitFunc) gdict_defbox_init
        };
        
        gdict_defbox_type = 
            g_type_register_static (GTK_TYPE_TEXT_VIEW, "GDictDefboxClass", &gdict_defbox_info, 0);

    }
    
    return gdict_defbox_type;
}

static void 
gdict_defbox_init (GDictDefbox *defbox)
{
    defbox->context = NULL;
    defbox->def_cmd = NULL;
}

static void 
gdict_defbox_class_init (GDictDefboxClass *class)
{
    GObjectClass *object_class;

    object_class = G_OBJECT_CLASS (class);
    
    gdict_defbox_signals[WORD_LOOKUP_START_SIGNAL] =
        g_signal_new ("word_lookup_start",
                      G_TYPE_FROM_CLASS (object_class),
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (GDictDefboxClass, word_lookup_start),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE, 0);

    gdict_defbox_signals[WORD_LOOKUP_DONE_SIGNAL] =
        g_signal_new ("word_lookup_done",
                      G_TYPE_FROM_CLASS (object_class),
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (GDictDefboxClass, word_lookup_done),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE, 0);

    gdict_defbox_signals[WORD_NOT_FOUND_SIGNAL] =
        g_signal_new ("word_not_found",
                      G_TYPE_FROM_CLASS (object_class),
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (GDictDefboxClass, word_not_found),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE, 0);

    gdict_defbox_signals[SUBSTR_NOT_FOUND_SIGNAL] =
        g_signal_new ("substr_not_found",
                      G_TYPE_FROM_CLASS (object_class),
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (GDictDefboxClass, substr_not_found),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__VOID,
                      G_TYPE_NONE, 0);

    gdict_defbox_signals[SOCKET_ERROR_SIGNAL] =
        g_signal_new ("socket_error",
                      G_TYPE_FROM_CLASS (object_class),
                      G_SIGNAL_RUN_FIRST,
                      G_STRUCT_OFFSET (GDictDefboxClass, socket_error),
                      NULL, NULL,
                      g_cclosure_marshal_VOID__STRING,
                      G_TYPE_NONE,
                      1, G_TYPE_STRING);

    class->word_lookup_start = NULL;
    class->word_lookup_done = NULL;
    class->word_not_found = NULL;
    class->substr_not_found = NULL;
}

GtkWidget *
gdict_defbox_new (gpointer gdict)
{
    GDictDefbox *defbox;
    PangoFontDescription *fontdesc;
		PangoContext *context;

    defbox = GDICT_DEFBOX (g_object_new (GDICT_TYPE_DEFBOX, NULL));
    gtk_text_view_set_left_margin (GTK_TEXT_VIEW (defbox), 4);
    gtk_text_view_set_editable (GTK_TEXT_VIEW (defbox), FALSE);

		context = gtk_widget_get_pango_context (GTK_WIDGET (defbox));
		fontdesc = pango_context_get_font_description (context);

		if (pango_font_description_get_family (fontdesc)) {
			defbox->fontname = g_strdup(pango_font_description_get_family (fontdesc));
			defbox->fontsize = pango_font_description_get_size (fontdesc);
		}
		
    defbox->gdict = gdict;
    defbox->context = GDICT_WINDOW(gdict)->context;
		defbox->fontsize = 10;
    gdict_defbox_setup_tags (defbox);  
    g_signal_connect (G_OBJECT(defbox), "event-after", 
                      G_CALLBACK (gdict_defbox_event_after), gdict);
    g_signal_connect (G_OBJECT(defbox), "motion-notify-event", 
                      G_CALLBACK (gdict_defbox_motion_notify_event), gdict);
    g_signal_connect (G_OBJECT(defbox), "visibility-notify-event", 
                      G_CALLBACK (gdict_defbox_visibility_notify_event), gdict);
    g_signal_connect (G_OBJECT (defbox), "word_lookup_start",
                      G_CALLBACK (gdict_defbox_lookup_start_cb), gdict);
    g_signal_connect (G_OBJECT (defbox), "word_lookup_done",
                      G_CALLBACK (gdict_defbox_lookup_done_cb), gdict);
    g_signal_connect (G_OBJECT (defbox), "word_not_found",
                      G_CALLBACK (gdict_defbox_not_found_cb), gdict);
    g_signal_connect (G_OBJECT (defbox), "substr_not_found",
                      G_CALLBACK (gdict_defbox_substr_not_found_cb), gdict);
    g_signal_connect (G_OBJECT (defbox), "socket_error",
                      G_CALLBACK (gdict_socket_error_cb), gdict);

    return GTK_WIDGET (defbox);
}

/* gdict_defbox_lookup
 *
 * Sends the command to the server to commence looking up the definition
 * of a word and sets the callbacks so that the definition will be displayed
 * in this defbox
 *
 * Returns 0 on success and -1 on command invocation error
 */

gint
gdict_defbox_lookup (GDictDefbox *defbox, gchar *text)
{
    GDictWindow *gdict = defbox->gdict;

    g_return_val_if_fail (defbox != NULL, -1);
    g_return_val_if_fail (GDICT_IS_DEFBOX (defbox), -1);
    g_return_val_if_fail (text != NULL, -1);
    
    while (isspace (*text)) text++;
    
    if (*text == '\0')
        return 0;

    g_signal_emit (G_OBJECT (defbox), gdict_defbox_signals[WORD_LOOKUP_START_SIGNAL], 0);

    gdict_defbox_clear (defbox);
    
    if (defbox->database)
	    g_free (defbox->database);
    
    defbox->database = g_strdup (gdict->pref->database);
    
    defbox->def_cmd = dict_define_command_new (defbox->database, text);
    defbox->def_cmd->error_notify_cb = gdict_defbox_error_cb;
    defbox->def_cmd->status_notify_cb = gdict_defbox_status_cb;
    defbox->def_cmd->data_notify_cb = gdict_defbox_data_cb;
    defbox->def_cmd->user_data = defbox;

    if (dict_command_invoke (defbox->def_cmd, defbox->context) < 0)
			return -1;

    return 0;
}

void 
gdict_defbox_clear (GDictDefbox *defbox)
{
    GtkTextBuffer *buffer;
    GtkTextIter start, end;

    g_return_if_fail (defbox != NULL);
    g_return_if_fail (GDICT_IS_DEFBOX (defbox));

    buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (defbox));
    gtk_text_buffer_get_bounds (buffer, &start, &end);
    gtk_text_buffer_delete (buffer, &start, &end);

    if (defbox->def_cmd) {
        dict_command_destroy (defbox->def_cmd);
        defbox->def_cmd = NULL;
    }
}

void
gdict_defbox_set_font (GDictDefbox *defbox, gchar *fontname)
{
	PangoFontDescription *fontdesc;

	fontdesc = pango_font_description_from_string (fontname);
	if (pango_font_description_get_family (fontdesc) != NULL)
		gtk_widget_modify_font (GTK_WIDGET(defbox), fontdesc);

	g_free (defbox->fontname);
	defbox->fontname = g_strdup (fontname);
	gdict_window_store_font (defbox->gdict);
}

void
gdict_defbox_set_fontsize (GDictDefbox *defbox, int fontsize)
{
	PangoFontDescription *fontdesc;
	PangoContext *context;
	
	context = gtk_widget_get_pango_context (GTK_WIDGET(defbox));
	fontdesc = pango_context_get_font_description (context);
	pango_font_description_set_size (fontdesc, fontsize*PANGO_SCALE);
	gtk_widget_modify_font (GTK_WIDGET(defbox), fontdesc);
	defbox->fontsize = fontsize;
}	

void
gdict_defbox_zoom_in (GDictDefbox *defbox)
{
	defbox->fontsize = MIN (defbox->fontsize + 1, 24);
	gdict_defbox_set_fontsize (defbox, defbox->fontsize);
}	

void
gdict_defbox_zoom_out (GDictDefbox *defbox)
{
	defbox->fontsize = MAX (defbox->fontsize - 1, 6);
	gdict_defbox_set_fontsize (defbox, defbox->fontsize);
}	

/* gdict_defbox_find
 * Finds a string of text in the current definition
 */

gboolean 
gdict_defbox_find (GDictDefbox *defbox, const gchar *text, gboolean start)
{
    GtkTextBuffer *buffer;
    GtkTextIter start_iter, end_iter, iter;
    GtkTextIter match_start, match_end;
    GtkTextMark *mark = NULL;

    g_return_val_if_fail (defbox != NULL, FALSE);
    g_return_val_if_fail (GDICT_IS_DEFBOX (defbox), FALSE);
    
    buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (defbox));

    gtk_text_buffer_get_bounds (buffer, &start_iter, &end_iter);

    if (start) 
    	iter = start_iter;
    else {
        mark = gtk_text_buffer_get_mark (buffer, "last_search");
    	
    	if (mark)
    	    gtk_text_buffer_get_iter_at_mark (buffer, &iter, mark);
    	else {
		GDictWindow *gdict = defbox->gdict;
		gtk_statusbar_pop (GTK_STATUSBAR (gdict->statusbar), 0);
		iter = start_iter;
	}
    }
    
    if (gtk_text_iter_forward_search (&iter, text,
                                      GTK_TEXT_SEARCH_VISIBLE_ONLY |
                                      GTK_TEXT_SEARCH_TEXT_ONLY,
                                      &match_start, &match_end,
                                      NULL))
    {
        gtk_text_view_scroll_to_iter (GTK_TEXT_VIEW (defbox), &match_start,
        			      0.0, TRUE, 0.5, 0.5);
        gtk_text_buffer_place_cursor (buffer, &match_end);
        gtk_text_buffer_move_mark (buffer,
                               gtk_text_buffer_get_mark (buffer, "selection_bound"),
                               &match_start);     
        gtk_text_buffer_create_mark (buffer, "last_search", &match_end, FALSE);

	return TRUE;
    }
    else
    {
        g_signal_emit (G_OBJECT (defbox), gdict_defbox_signals[SUBSTR_NOT_FOUND_SIGNAL], 0);

	return FALSE;
    }
}

/* gdict_defbox_reset
 *
 * Resets the defbox by re-invoking the define command on a new database
 * and/or server
 */

void
gdict_defbox_reset (GDictDefbox *defbox, dict_context_t *context)
{
    gchar *word;
    GDictWindow *gdict = defbox->gdict;
    
    if (context != defbox->context || 
        defbox->database == NULL ||
        strcmp (defbox->database, gdict->pref->database))
    {
        defbox->context = context;
        
        if (defbox->def_cmd) {
            word = g_strdup (defbox->def_cmd->search_term);
            dict_command_destroy (defbox->def_cmd);
            defbox->def_cmd = NULL;
            gdict_defbox_lookup (defbox, word);
            g_free (word);
        }
    }
}

void
gdict_defbox_print_preview (GDictDefbox *defbox)
{
	GtkWidget *preview;
	GDictWindow *gdict = defbox->gdict;
	preview = gnome_print_job_preview_new (gdict->print_job, _("Definition preview"));
	gtk_window_present (GTK_WINDOW(preview));
}

/* gdict_defbox_print
 * 
 * prepare the printing job for the definition printing
 */

void 
gdict_defbox_prepare_print (GDictDefbox *defbox)
{
    GDictWindow *gdict = defbox->gdict;
    GnomePrintContext *pc;
    GList *node;
    dict_res_t *res;
    int nb_node = 0, i=0, width, height, row_size;
    int nb_lines, nb_pages, nb_lines_per_page, line_to_print, x, y;
    double page_width, page_height;
    gchar **node_definition;
    gchar *text, **textlines;
    PangoLayout *layout;
    PangoFontDescription *desc;

    g_return_if_fail (defbox != NULL);
    g_return_if_fail (GDICT_IS_DEFBOX (defbox));

    if (defbox->def_cmd == NULL)
	    return;

    nb_node = g_list_length (defbox->def_cmd->res_list);
    node_definition = (gchar **) g_new0 (gpointer, nb_node+1);
    for (node = defbox->def_cmd->res_list; node; node = g_list_next (node)) {
	    res = (dict_res_t *) node->data;
	    node_definition[i++] = res->desc;
    }
    node_definition[i] = NULL;
    text = g_strjoinv ("\n", node_definition);
    textlines = g_strsplit (text, "\n", -1);    
    for (i=0; textlines[i] != NULL; i++);
    nb_lines = i;
    
    row_size = 10;

    pc = gnome_print_job_get_context (gdict->print_job);
    g_return_if_fail(pc != NULL);
    layout = gnome_print_pango_create_layout (pc);
    desc = pango_font_description_from_string ("Serif 10");
    pango_layout_set_font_description (layout, desc);
    pango_font_description_free (desc);
    pango_layout_set_width (layout, (page_width - LEFT_MARGIN - RIGHT_MARGIN) * PANGO_SCALE);

    gnome_print_job_get_page_size (gdict->print_job, &page_width, &page_height);
    nb_lines_per_page = ((page_height - TOP_MARGIN*2.)/row_size);
    nb_pages = ceil ((float) nb_lines / (float) nb_lines_per_page);

    line_to_print = 0;
    x = 100;
    for (i=0; i<nb_pages; i++) {
      int j;
      gchar *page_name;

      page_name = g_strdup_printf("%d", i+1);
      gnome_print_beginpage (pc, page_name);
      g_free (page_name);
      
      y = page_height - TOP_MARGIN;
      for (j=0; j < nb_lines_per_page; j++) {
        if (line_to_print >= nb_lines)
          break;
        gnome_print_moveto (pc, x, y);      
        pango_layout_set_text (layout, textlines[line_to_print++], -1);
        gnome_print_pango_layout (pc, layout);
        y -= row_size;
      }
      gnome_print_showpage (pc);
      if (line_to_print >= nb_lines)
        break;
    }
    g_object_unref (G_OBJECT (pc));
    gnome_print_job_close (gdict->print_job);
}

/* gdict_defbox_get_word
 *
 * Returns the word defined in the defbox, if any
 */

gchar *
gdict_defbox_get_word (GDictDefbox *defbox)
{
    g_return_val_if_fail (defbox != NULL, NULL);
    g_return_val_if_fail (GDICT_IS_DEFBOX (defbox), NULL);
    
    return defbox->def_cmd ? defbox->def_cmd->search_term : NULL;
}

/* is_xref
 *
 * Returns true if the substring str[0..len-1] is a cross reference
 */

static gboolean 
is_xref (gchar *str, int len)
{
    gint i;
    for (i = 0;  i < len;  i++)
        if (!isupper(str[i]) && !ispunct(str[i]))
            return FALSE;
    return TRUE;
}

/* is_number
 * Returns true if the substring given by str[0..len-1] is a heading number
 */

static gboolean 
is_number (gchar *str, int len)
{
    gint i;

    if ((str[len-1] != '.') && (str[len-1] != ':'))
        return FALSE;
        
    for (i = 0;  i < len - 1;  i++)
        if (!isdigit(str[i]))
            return FALSE;

    return TRUE;
}

/* is_part
 * Returns true if the substring given by str[0..len-1] is a part of speech
 */

static gboolean 
is_part (gchar *str, int len)
{
    gchar buf[4];

    if ((len < 1) || (len > 3))
        return FALSE;

    strncpy(buf, str, 3);
    buf[len] = 0;

    return (strcmp(buf, "n") == 0) || \
			(strcmp(buf, "n.") == 0) || \
			(strcmp(buf, "vt") == 0) || \
			(strcmp(buf, "vi") == 0) || \
			(strcmp(buf, "vb") == 0) || \
			(strcmp(buf, "aj") == 0) || \
			(strcmp(buf, "v") == 0) || \
			(strcmp(buf, "adj") == 0) || \
			(strcmp(buf, "adv") == 0) || \
			(strcmp(buf, "av") == 0);
}

static void
insert_text_with_tags (GtkTextBuffer *buffer, GtkTextIter *iter, gchar *p, gint len,
											 int faces)
{
	GtkTextMark *mark;
	GtkTextIter start;
	
	mark = gtk_text_buffer_create_mark (buffer, "start_insert", iter, TRUE);
	gtk_text_buffer_insert (buffer, iter, p, len);
	gtk_text_buffer_get_iter_at_mark (buffer, &start, mark);
	gtk_text_buffer_delete_mark (buffer, mark);

	if (faces & BIG)
		gtk_text_buffer_apply_tag_by_name (buffer, "big", &start, iter);
	if (faces & BOLD)
		gtk_text_buffer_apply_tag_by_name (buffer, "bold", &start, iter);
	if (faces & ITALIC)
		gtk_text_buffer_apply_tag_by_name (buffer, "italic", &start, iter);
	if (faces & UNDERLINE)
		gtk_text_buffer_apply_tag_by_name (buffer, "underline", &start, iter);
	if (faces & GREY)
		gtk_text_buffer_apply_tag_by_name (buffer, "grey", &start, iter);
	if (faces & EMPHASIS)
		gtk_text_buffer_apply_tag_by_name (buffer, "emphasis", &start, iter);
}

static void
insert_link (GtkTextBuffer *buffer, GtkTextIter *iter, gchar *text, gint len)
{
	GtkTextTag *tag;

	tag = gtk_text_buffer_create_tag (buffer, NULL,
					  "foreground", "blue",
					  "underline", PANGO_UNDERLINE_SINGLE,
					  NULL);
	g_object_set_data (G_OBJECT (tag), "page", text);

	gtk_text_buffer_insert_with_tags (buffer, iter, text, len , tag, NULL);
}

/* defbox_add
 *
 * Adds a definition to the defbox, performing all necessary formatting
 */

static void 
gdict_defbox_add (GDictDefbox *defbox, gchar *def)
{
    GtkTextBuffer *buffer;
    GtkTextIter iter;
    gchar *p, *q, *text;
    gint len;
		int faces;

    buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (defbox));
    gtk_text_buffer_get_end_iter (buffer, &iter);
    
    p = def;
    len = strcspn(def, " \t\n");
    
		faces = BIG;

    while (*p) {
        /* Handle word token */
        insert_text_with_tags (buffer, &iter, p, len, faces);        
				faces = faces & ~BOLD;
				faces = faces & ~EMPHASIS;

        p += len;

				/* ... then handle spaces ... */
        len = strspn(p, " \t");
        
        if (len > 0)
					insert_text_with_tags (buffer, &iter, p, len, faces);
        p += len;

				/* If we just added the header word and no pronunciation, add a line break */
				if ((faces & BIG) && (*p!='\n') && (*p!='\\'))
					insert_text_with_tags (buffer, &iter, "\n", 1, faces);
				faces = faces & ~BIG;

        /* ... handle special characters ... */
        if (*p == '\\') {
					/* Pronunciation */
					if (!(faces & GREY)) {
						/* beginning the pronunciation */
						faces |= GREY;
						insert_text_with_tags (buffer, &iter, "| ", 2, faces);
			    } else {
						/* end of the pronunciation */
						insert_text_with_tags (buffer, &iter, " |\n", 3, faces);
						if (*(p+1) == ',')
							p++;
						faces = faces & ~GREY;
					}
					++p;
        } else if (*p == '[') {
					/* Etymology or synonyms */
					faces |= ITALIC;
					++p;
				} else if (*p == ']') {
					faces = faces & ~ITALIC;
					p++;
        } else if (*p == '{') {
					int length = 0;

					/* we must take care in this function
					 * not to loop after the \0 of the string */
					/* search the last '{' of a sequence of '{' */
					while (*(p+1) == '{')
						p++;
					
					q = p;
					
					/* search the first '}' after the link text */
					while (*q != '\0' && *q != '}') {
						length++;
						q++;
					}
					
					len = length;
					++p; /* go to the char behind the last '{' */
					text = g_strndup (p, len - 1);
					insert_link (buffer, &iter, text, len - 1);
					p += length; /* on first '}' */
					
					/* step through the remaining '}' */
					while (*p == '}')
						p++;
					length = 0;
				} else if (*p == '}' ) {
					/* unbalanced {} pairs, just go behind the closing... */
					while (*p == '}')
						p++;
        } else if (*p == ']' ) {
					++p;
        }
        
        len = strcspn (p, " \\[]{}");

				if (is_number(p, len)) {
					faces = faces | BOLD;
					if (*p=='1')
						insert_text_with_tags (buffer, &iter, "\n    ", 5, faces);
				}
				else if (is_part(p, len))
					faces = faces | (BOLD | EMPHASIS);
    }    
}

/* gdict_defbox_error_cb
 *
 * Callback invoked when there was an error in the last query
 */

static void
gdict_defbox_error_cb (dict_command_t *command, DictStatusCode code,
              gchar *message, gpointer data)
{
    GObject *defbox;
    dict_command_t *sec_cmd;
    
    defbox = G_OBJECT (data);
    
    if (code != DICT_SOCKET_ERROR) {
        GtkWidget *dialog;
        
        dialog = gtk_message_dialog_new_with_markup (NULL, GTK_DIALOG_DESTROY_WITH_PARENT,
                                                     GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
                                                     "<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
                                                     _("Error invoking query"), message);
        gtk_dialog_run (GTK_DIALOG (dialog));
        gtk_widget_destroy (dialog);
        
        sec_cmd = dict_disconnect_command_new ();
        dict_command_invoke (sec_cmd, command->context);
        
        g_signal_emit (defbox, gdict_defbox_signals[WORD_LOOKUP_DONE_SIGNAL], 0);
    }
    else {
        g_signal_emit (defbox, gdict_defbox_signals[SOCKET_ERROR_SIGNAL], 0, message);
    }
}

/* gdict_defbox_data_cb
 *
 * Callback used when a new definition has arrived over the link
 */

static void 
gdict_defbox_data_cb (dict_command_t *command, dict_res_t *res, gpointer data)
{
    GDictDefbox *defbox;
    
    defbox = GDICT_DEFBOX (data);
    gdict_defbox_add (defbox, res->desc);
}

/* gdict_defbox_status_cb
 *
 * Callback used when a status code has arrived over the link
 */

static void 
gdict_defbox_status_cb (dict_command_t *command, DictStatusCode code, 
               int num_found, gpointer data)
{
    GObject *defbox;

    defbox = G_OBJECT (data);

    if (code == DICT_STATUS_OK)
        g_signal_emit (defbox, gdict_defbox_signals[WORD_LOOKUP_DONE_SIGNAL], 0);
    else if (code == DICT_STATUS_NO_MATCH)
        g_signal_emit (defbox, gdict_defbox_signals[WORD_NOT_FOUND_SIGNAL], 0);
}
