#******************************************************************************
#
#       Copyright:      2009-2022 Paul Obermeier (obermeier@tcl3d.org)
#
#                       See the file "Tcl3D_License.txt" for information on
#                       usage and redistribution of this file, and for a
#                       DISCLAIMER OF ALL WARRANTIES.
#
#       Module:         Tcl3D -> tcl3dOgl
#       Filename:       tcl3dOglHelp.tcl
#
#       Author:         Paul Obermeier
#
#       Description:    Tcl module with help and information procedures
#                       related to the OpenGL module.
#
#******************************************************************************

###############################################################################
#[@e
#       Name:           tcl3dOglGetFuncList - Get list of OpenGL and GLU 
#                       functions.
#
#       Synopsis:       tcl3dOglGetFuncList { what }
#
#                       what : string (Default "gl")
#                       Allowed values for what ar: "gl", "glu" or "all"
#
#       Description:    Return a list of all wrapped OpenGL and/or GLU 
#                       function names.
#
#       See also:       tcl3dOglGetVersionList
#                       tcl3dOglIsFuncWrapped
#                       tcl3dOglGetVersionFuncs
#                       tcl3dOglGetVersionEnums
#
###############################################################################

proc tcl3dOglGetFuncList { { what "gl" } } {
    global __tcl3dOglFuncList __tcl3dGluFuncList

    if { $what eq "gl" } {
        return $__tcl3dOglFuncList
    }
    if { $what eq "glu" } {
        return $__tcl3dGluFuncList
    }
    return [concat $__tcl3dOglFuncList $__tcl3dGluFuncList]
}

# OBSOLETE tcl3dOglGetFuncSignatureList 0.4.2 None
proc tcl3dOglGetFuncSignatureList {} {
    global __tcl3dOglFuncSignatureList

    return $__tcl3dOglFuncSignatureList
}

# OBSOLETE tcl3dOglGetFuncVersionList 0.4.2 None
proc tcl3dOglGetFuncVersionList {} {
    global __tcl3dOglFuncVersionList

    return $__tcl3dOglFuncVersionList
}

###############################################################################
#[@e
#       Name:           tcl3dOglGetVersionList - Get list of OpenGL versions 
#                       and extensions.
#
#       Synopsis:       tcl3dOglGetVersionList {}
#
#       Description:    Return a list of all wrapped OpenGL versions and
#                       extension names.
#                       The names are strings identical to their corresponding
#                       C defines.
#                       Examples:
#                       GL versions  : GL_VERSION_1_5, GL_VERSION_3_2
#                       GL extensions: GL_ARB_vertex_program, GL_EXT_texture3D
#
#       See also:       tcl3dOglGetFuncList
#                       tcl3dOglGetFuncVersion
#                       tcl3dOglGetVersionFuncs
#                       tcl3dOglGetVersionEnums
#
###############################################################################

proc tcl3dOglGetVersionList {} {
    global __tcl3dOglFuncVersionList __tcl3dOglEnumVersion

    set tmp $__tcl3dOglFuncVersionList

    foreach {enum ext} [array get __tcl3dOglEnumVersion "*"] {
        lappend tmp $ext
    }
    return [lsort -unique $tmp]
}

###############################################################################
#[@e
#       Name:           tcl3dOglGetExtensionList - Get list of OpenGL extensions.
#
#       Synopsis:       tcl3dOglGetExtensionList { {what "all"} }
#
#       Description:    Return a list of OpenGL extension names.
#                       The names are strings identical to their corresponding
#                       C defines.
#                       Examples:GL_ARB_vertex_program, GL_EXT_texture3D
#
#                       If "what" is equal to "all", all OpenGL extension names
#                       are returned.
#                       If "what" is equal to "glew", only the OpenGL
#                       extension names wrapped by GLEW are returned.
#                       If "what" is equal to "driver", only the OpenGL
#                       extension names supported by the driver and hardware of
#                       the actual machine are returned.
#
#       See also:       tcl3dOglGetFuncList
#                       tcl3dOglGetFuncVersion
#                       tcl3dOglGetVersionList
#                       tcl3dOglGetVersionFuncs
#                       tcl3dOglGetVersionEnums
#
###############################################################################

proc tcl3dOglGetExtensionList { {what "all"} } {
    global __tcl3dOglExtensionList

    if { $what eq "driver" } {
        return [lindex [tcl3dOglGetExtensions] 0 1]
    } elseif { $what eq "glew" } {
        return [tcl3dOglGetVersionList]
    } else {
        return [lsort [array names __tcl3dOglExtensionList]]
    }
}

###############################################################################
#[@e
#       Name:           tcl3dOglIsFuncWrapped - Check if OpenGL or GLU function
#                       is wrapped.
#
#       Synopsis:       tcl3dOglIsFuncWrapped { func }
#
#       Description:    Return true, if OpenGL or GLU function "func" is wrapped
#                       in Tcl3D. Otherwise return false.
#
#                       Note: To check, if a function is supported by the used
#                             OpenGL driver, use procedure "tcl3dOglHaveFunc".
#
#       See also:       tcl3dOglGetFuncList
#                       tcl3dOglGetFuncSignature
#                       tcl3dOglGetFuncVersion
#                       tcl3dOglGetFuncDeprecated
#                       tcl3dOglGetUrl
#
###############################################################################

proc tcl3dOglIsFuncWrapped { func } {
    global __tcl3dOglFuncList __tcl3dGluFuncList

    set ind [lsearch -exact $__tcl3dOglFuncList $func]
    if { $ind >= 0 } {
        return true
    } else {
        set ind [lsearch -exact $__tcl3dGluFuncList $func]
        if { $ind >= 0 } {
            return true
        }
    }
    return false
}

proc __tcl3dMapReturnValue { cval } {
    set cval [string map {"*" " *"} $cval]
    switch -glob -- $cval {
        {void}                { return "" }
        {GLboolean}           { return "bool" }
        {GLenum}              { return "enum" }
        {GL*int}              { return "int" }
        {const GLubyte[ ]*\*} { return "string" }
        {GLhandleARB}         -
        {GLsync}              -
        {GLintptr}            -
        {void[ ]*\*}          -
        {GLvoid[ ]*\*}        -
        {GLU*\*}              { return "SwigPointer" }
        default               { return "NOT_SUPPORTED" }
    }
}

proc __tcl3dMapParam { cval } {
    set cval [string map {"*" " * "} $cval]
    set paramName [string range $cval [string wordstart $cval end] end]
    switch -glob -- $cval {
        {const void[ ]*\**}   -
        {const GLvoid[ ]*\**} - 
        {void[ ]*\**}         -
        {GLvoid[ ]*\**}       { return "\[tcl3dVector TYPE\] $paramName" }
        {const *[ ]*\[*\]*}   -
        {const *[ ]*\**}      { set type [string range $cval [string wordstart $cval 8] \
                                                             [string wordend $cval 8]]
                                set type [string trim $type]
                                return "\[list of $type\] $paramName"
                              }
        {*[ ]*\**}            { set type [string range $cval 0 [string wordend $cval 0]]
                                set type [string trim $type]
                                return "\[tcl3dVector $type\] $paramName"
                              }
        {GLintptr*}           { return "\[tcl3dVector GLint\] $paramName" }
        {GLsizeiptr*}         { return "\[tcl3dVector GLsizei\] $paramName" }
        {GLenum*}             { return "enum $paramName" }
        {GLboolean*}          { return "bool $paramName" }
        {GLbitfield*}         { return "bitfield $paramName" }
        {GL*byte*}            -
        {GL*short*}           -
        {const GLint*}        -
        {GL*int*}             -
        {GLsizei*}            { return "int $paramName" }
        {GLhalf*}             -
        {GLclamp?*}           -
        {const GLfloat*}      -
        {GLfloat*}            -
        {GLdouble*}           { return "double $paramName" }
        {GLhandleARB*}        -
        {GLsync*}             { return "SwigPointer $paramName" }
        {void}                { return "" }
        default               { return "NOT_SUPPORTED" }
    }
}

proc tcl3dOglGetTclSignature { cSig } {
    set funcStart  [string first "gl" $cSig]
    set paramStart [string first "("  $cSig]
    set paramEnd   [string last  ")"  $cSig]
    set cRet   [string trim [string range $cSig 0 [expr {$funcStart-1}]]]
    set cFunc  [string trim [string range $cSig $funcStart [expr {$paramStart-1}]]]
    set tclRet [__tcl3dMapReturnValue $cRet]
    set tclSig ""
    if { $tclRet ne "" } {
        append tclSig "$tclRet "
    }
    append tclSig "$cFunc " 
    set cParams [string range $cSig [expr {$paramStart +1}] [expr {$paramEnd -1}]]
    foreach param [split $cParams ","] {
        set param [__tcl3dMapParam [string trim $param]]
        append tclSig "{$param} "
    }
    return $tclSig
}

###############################################################################
#[@e
#       Name:           tcl3dOglGetFuncSignature - Get the signature of an 
#                       OpenGL or GLU function.
#
#       Synopsis:       tcl3dOglGetFuncSignature { func {what "c"} }
#
#       Description:    Return the signature of OpenGL or GLU function "func"
#                       as a string.
#                       If "func" is not contained in the list of wrapped 
#                       OpenGL functions (see tcl3dOglGetFuncList), an empty
#                       string is returned.
#
#                       If "what" is equal to "c", the signature string is returned
#                       in C style notation. This is the default case.
#                       If "what" is equal to "tcl", the signature string is returned
#                       in Tcl style notation.
#
#                       Note: This procedure replaces the obsolete
#                             tcl3dOglGetFuncSignatureList procedure.
#
#       See also:       tcl3dOglGetFuncList
#                       tcl3dOglGetFuncVersion
#                       tcl3dOglGetFuncDeprecated
#                       tcl3dOglGetUrl
#
###############################################################################

proc tcl3dOglGetFuncSignature { func { what "c" } } {
    global __tcl3dOglFuncList __tcl3dOglFuncSignatureList
    global __tcl3dGluFuncList __tcl3dGluFuncSignatureList

    set sigStr ""
    set ind [lsearch -exact $__tcl3dOglFuncList $func]
    if { $ind >= 0 } {
        set sigStr [lindex $__tcl3dOglFuncSignatureList $ind]
    } else {
        set ind [lsearch -exact $__tcl3dGluFuncList $func]
        if { $ind >= 0 } {
            set sigStr [lindex $__tcl3dGluFuncSignatureList $ind]
        }
    }
    if { $what eq "c" } {
        return $sigStr
    } else {
        return [tcl3dOglGetTclSignature $sigStr]
    }
}

###############################################################################
#[@e
#       Name:           tcl3dOglGetFuncVersion - Get the version or extension
#                       name of an OpenGL function.
#
#       Synopsis:       tcl3dOglGetFuncVersion { func }
#
#       Description:    Return the version or extension name of OpenGL function
#                       "func" as a string.
#                       If "func" is not contained in the list of wrapped 
#                       OpenGL functions (see tcl3dOglGetFuncList), an empty
#                       string is returned.
#
#                       Note: This procedure replaces the obsolete
#                             tcl3dOglGetFuncVersionList procedure.
#
#       See also:       tcl3dOglGetFuncList
#                       tcl3dOglGetFuncSignature
#                       tcl3dOglGetFuncDeprecated
#                       tcl3dOglGetUrl
#                       tcl3dOglGetEnumVersion
#
###############################################################################

proc tcl3dOglGetFuncVersion { func } {
    global __tcl3dOglFuncList __tcl3dOglFuncVersionList

    set ind [lsearch -exact $__tcl3dOglFuncList $func]
    if { $ind >= 0 } {
        return [lindex $__tcl3dOglFuncVersionList $ind]
    } else {
        return ""
    }
}

###############################################################################
#[@e
#       Name:           tcl3dOglGetEnumVersion - Get the version or extension
#                       name of an OpenGL enumeration.
#
#       Synopsis:       tcl3dOglGetEnumVersion { enum }
#
#       Description:    Return the version or extension name of OpenGL
#                       enumeration "enum" as a string.
#                       If "enum" is not a wrapped OpenGL enumeration,
#                       an empty string is returned.
#
#       See also:       tcl3dOglGetVersionList
#                       tcl3dOglGetVersionFuncs
#                       tcl3dOglGetVersionEnums
#                       tcl3dOglGetFuncVersion
#
###############################################################################

proc tcl3dOglGetEnumVersion { enum } {
    global __tcl3dOglEnumVersion

    if { [info exists __tcl3dOglEnumVersion($enum)] } {
        return $__tcl3dOglEnumVersion($enum)
    } else {
        return ""
    }
}

###############################################################################
#[@e
#       Name:           tcl3dOglGetFuncDeprecated - Get the OpenGL version, an
#                       OpenGL function has been declared deprecated.
#
#       Synopsis:       tcl3dOglGetFuncDeprecated { func }
#
#       Description:    Return the version when OpenGL function "func" has been
#                       declared deprecated.
#                       The result is in the format "3.1", "3.2".
#                       For non-deprecated functions "0.0" is returned.
#
#                       If "func" is not contained in the list of wrapped 
#                       OpenGL functions (see tcl3dOglGetFuncList), an empty
#                       string is returned.
#
#       See also:       tcl3dOglGetFuncList
#                       tcl3dOglGetFuncSignature
#                       tcl3dOglGetFuncVersion
#                       tcl3dOglGetUrl
#
###############################################################################

proc tcl3dOglGetFuncDeprecated { func } {
    global __tcl3dOglFuncList __tcl3dOglFuncDeprecatedList

    set ind [lsearch -exact $__tcl3dOglFuncList $func]
    if { $ind >= 0 } {
        return [lindex $__tcl3dOglFuncDeprecatedList $ind]
    } else {
        return ""
    }
}

###############################################################################
#[@e
#       Name:           tcl3dOglGetUrl - Get the URL of the official
#                       documentation of an OpenGL item.
#
#       Synopsis:       tcl3dOglGetUrl { item }
#
#       Description:    Return the URL of the official documentation of OpenGL
#                       item "item" as a string.
#                       Item can be the name of a function, extension or 
#                       enumeration. 
#                       If no documentation page exists, a Google search URL
#                       for that item is returned.
#
#                       Note: The documentation pages on www.opengl.org
#                             currently only include OpenGL up to version 2.1.
#
#       See also:       tcl3dOglGetFuncList
#                       tcl3dOglGetVersionFuncs
#                       tcl3dOglGetVersionEnums
#
###############################################################################

proc tcl3dOglGetUrl { item } {
    global __tcl3dOglFuncList __tcl3dOglFuncUrlList
    global __tcl3dGluFuncList __tcl3dGluFuncUrlList
    global __tcl3dOglFuncVersionList __tcl3dOglEnumVersion
    global __tcl3dOglExtensionList

    # First check, if item is an OpenGL command.
    set ind [lsearch -exact $__tcl3dOglFuncList $item]
    if { $ind >= 0 } {
        return [lindex $__tcl3dOglFuncUrlList $ind]
    }
    # Then check, if item is an OpenGL GLU command.
    set ind [lsearch -exact $__tcl3dGluFuncList $item]
    if { $ind >= 0 } {
        return [lindex $__tcl3dGluFuncUrlList $ind]
    }
    # Then check, if item is an OpenGL extension name.
    if { [info exists __tcl3dOglExtensionList($item)] } {
        return $__tcl3dOglExtensionList($item)
    }
    # Then check, if item is an OpenGL enumeration name.
    foreach {enum ext} [array get ::__tcl3dOglEnumVersion "*"] {
        if { $enum eq $item } {
            if { ! [string match "GL_VERSION_*" $ext] } {
                # It is an enumeration specified in an OpenGL extension.
                # Link to the corresponding extension page.
                if { [info exists __tcl3dOglExtensionList($ext)] } {
                    return $__tcl3dOglExtensionList($ext)
                }
                set ind [lsearch -exact $__tcl3dOglFuncVersionList $ext]
                if { $ind >= 0 } {
                    return [lindex $__tcl3dOglFuncUrlList $ind]
                }
            }
        }
    }

    # We have not found a detailled link. Issue a Google search.
    return "http://www.google.com/search?q=$item"
}

###############################################################################
#[@e
#       Name:           tcl3dOglGetVersionFuncs - Get the function names of an
#                       OpenGL version or extension.
#
#       Synopsis:       tcl3dOglGetVersionFuncs { version }
#
#       Description:    Return the function names of OpenGL version or extension
#                       "version" as a list.
#                       If "version" is not a supported version or extension,
#                       an empty list is returned.
#
#       See also:       tcl3dOglGetFuncList
#                       tcl3dOglGetVersionList
#                       tcl3dOglGetFuncVersion
#                       tcl3dOglGetVersionEnums
#
###############################################################################

proc tcl3dOglGetVersionFuncs { version } {
    global __tcl3dOglFuncList __tcl3dOglFuncVersionList

    set funcList [list]
    set indList [lsearch -exact -all $__tcl3dOglFuncVersionList $version]
    foreach ind $indList {
        lappend funcList [lindex $__tcl3dOglFuncList $ind]
    }
    return $funcList
}

###############################################################################
#[@e
#       Name:           tcl3dOglGetVersionEnums - Get the enumeration names of
#                       an OpenGL version or extension.
#
#       Synopsis:       tcl3dOglGetVersionEnums { version }
#
#       Description:    Return the enumeration names of OpenGL version or
#                       extension "version" as a list.
#                       If "version" is not a supported version or extension,
#                       an empty list is returned.
#
#       See also:       tcl3dOglGetFuncList
#                       tcl3dOglGetVersionList
#                       tcl3dOglGetEnumVersion
#                       tcl3dOglGetVersionFuncs
#
###############################################################################

proc tcl3dOglGetVersionEnums { version } {
    global __tcl3dOglEnumVersion

    set enumList [list]
    foreach {enum ext} [array get __tcl3dOglEnumVersion "*"] {
        if { $ext eq $version } {
            lappend enumList $enum
        }
    }
    return $enumList
}

# List of the names of all wrapped OpenGL functions.
set ::__tcl3dOglFuncList [list \
  glAccum \
  glAlphaFunc \
  glAreTexturesResident \
  glArrayElement \
  glBegin \
  glBindTexture \
  glBitmap \
  glBlendFunc \
  glCallList \
  glCallLists \
  glClear \
  glClearAccum \
  glClearColor \
  glClearDepth \
  glClearIndex \
  glClearStencil \
  glClipPlane \
  glColor3b \
  glColor3bv \
  glColor3d \
  glColor3dv \
  glColor3f \
  glColor3fv \
  glColor3i \
  glColor3iv \
  glColor3s \
  glColor3sv \
  glColor3ub \
  glColor3ubv \
  glColor3ui \
  glColor3uiv \
  glColor3us \
  glColor3usv \
  glColor4b \
  glColor4bv \
  glColor4d \
  glColor4dv \
  glColor4f \
  glColor4fv \
  glColor4i \
  glColor4iv \
  glColor4s \
  glColor4sv \
  glColor4ub \
  glColor4ubv \
  glColor4ui \
  glColor4uiv \
  glColor4us \
  glColor4usv \
  glColorMask \
  glColorMaterial \
  glColorPointer \
  glCopyPixels \
  glCopyTexImage1D \
  glCopyTexImage2D \
  glCopyTexSubImage1D \
  glCopyTexSubImage2D \
  glCullFace \
  glDeleteLists \
  glDeleteTextures \
  glDepthFunc \
  glDepthMask \
  glDepthRange \
  glDisable \
  glDisableClientState \
  glDrawArrays \
  glDrawBuffer \
  glDrawElements \
  glDrawPixels \
  glEdgeFlag \
  glEdgeFlagPointer \
  glEdgeFlagv \
  glEnable \
  glEnableClientState \
  glEnd \
  glEndList \
  glEvalCoord1d \
  glEvalCoord1dv \
  glEvalCoord1f \
  glEvalCoord1fv \
  glEvalCoord2d \
  glEvalCoord2dv \
  glEvalCoord2f \
  glEvalCoord2fv \
  glEvalMesh1 \
  glEvalMesh2 \
  glEvalPoint1 \
  glEvalPoint2 \
  glFeedbackBuffer \
  glFinish \
  glFlush \
  glFogf \
  glFogfv \
  glFogi \
  glFogiv \
  glFrontFace \
  glFrustum \
  glGenLists \
  glGenTextures \
  glGetBooleanv \
  glGetClipPlane \
  glGetDoublev \
  glGetError \
  glGetFloatv \
  glGetIntegerv \
  glGetLightfv \
  glGetLightiv \
  glGetMapdv \
  glGetMapfv \
  glGetMapiv \
  glGetMaterialfv \
  glGetMaterialiv \
  glGetPixelMapfv \
  glGetPixelMapuiv \
  glGetPixelMapusv \
  glGetPointerv \
  glGetPolygonStipple \
  glGetString \
  glGetTexEnvfv \
  glGetTexEnviv \
  glGetTexGendv \
  glGetTexGenfv \
  glGetTexGeniv \
  glGetTexImage \
  glGetTexLevelParameterfv \
  glGetTexLevelParameteriv \
  glGetTexParameterfv \
  glGetTexParameteriv \
  glHint \
  glIndexMask \
  glIndexPointer \
  glIndexd \
  glIndexdv \
  glIndexf \
  glIndexfv \
  glIndexi \
  glIndexiv \
  glIndexs \
  glIndexsv \
  glIndexub \
  glIndexubv \
  glInitNames \
  glInterleavedArrays \
  glIsEnabled \
  glIsList \
  glIsTexture \
  glLightModelf \
  glLightModelfv \
  glLightModeli \
  glLightModeliv \
  glLightf \
  glLightfv \
  glLighti \
  glLightiv \
  glLineStipple \
  glLineWidth \
  glListBase \
  glLoadIdentity \
  glLoadMatrixd \
  glLoadMatrixf \
  glLoadName \
  glLogicOp \
  glMap1d \
  glMap1f \
  glMap2d \
  glMap2f \
  glMapGrid1d \
  glMapGrid1f \
  glMapGrid2d \
  glMapGrid2f \
  glMaterialf \
  glMaterialfv \
  glMateriali \
  glMaterialiv \
  glMatrixMode \
  glMultMatrixd \
  glMultMatrixf \
  glNewList \
  glNormal3b \
  glNormal3bv \
  glNormal3d \
  glNormal3dv \
  glNormal3f \
  glNormal3fv \
  glNormal3i \
  glNormal3iv \
  glNormal3s \
  glNormal3sv \
  glNormalPointer \
  glOrtho \
  glPassThrough \
  glPixelMapfv \
  glPixelMapuiv \
  glPixelMapusv \
  glPixelStoref \
  glPixelStorei \
  glPixelTransferf \
  glPixelTransferi \
  glPixelZoom \
  glPointSize \
  glPolygonMode \
  glPolygonOffset \
  glPolygonStipple \
  glPopAttrib \
  glPopClientAttrib \
  glPopMatrix \
  glPopName \
  glPrioritizeTextures \
  glPushAttrib \
  glPushClientAttrib \
  glPushMatrix \
  glPushName \
  glRasterPos2d \
  glRasterPos2dv \
  glRasterPos2f \
  glRasterPos2fv \
  glRasterPos2i \
  glRasterPos2iv \
  glRasterPos2s \
  glRasterPos2sv \
  glRasterPos3d \
  glRasterPos3dv \
  glRasterPos3f \
  glRasterPos3fv \
  glRasterPos3i \
  glRasterPos3iv \
  glRasterPos3s \
  glRasterPos3sv \
  glRasterPos4d \
  glRasterPos4dv \
  glRasterPos4f \
  glRasterPos4fv \
  glRasterPos4i \
  glRasterPos4iv \
  glRasterPos4s \
  glRasterPos4sv \
  glReadBuffer \
  glReadPixels \
  glRectd \
  glRectdv \
  glRectf \
  glRectfv \
  glRecti \
  glRectiv \
  glRects \
  glRectsv \
  glRenderMode \
  glRotated \
  glRotatef \
  glScaled \
  glScalef \
  glScissor \
  glSelectBuffer \
  glShadeModel \
  glStencilFunc \
  glStencilMask \
  glStencilOp \
  glTexCoord1d \
  glTexCoord1dv \
  glTexCoord1f \
  glTexCoord1fv \
  glTexCoord1i \
  glTexCoord1iv \
  glTexCoord1s \
  glTexCoord1sv \
  glTexCoord2d \
  glTexCoord2dv \
  glTexCoord2f \
  glTexCoord2fv \
  glTexCoord2i \
  glTexCoord2iv \
  glTexCoord2s \
  glTexCoord2sv \
  glTexCoord3d \
  glTexCoord3dv \
  glTexCoord3f \
  glTexCoord3fv \
  glTexCoord3i \
  glTexCoord3iv \
  glTexCoord3s \
  glTexCoord3sv \
  glTexCoord4d \
  glTexCoord4dv \
  glTexCoord4f \
  glTexCoord4fv \
  glTexCoord4i \
  glTexCoord4iv \
  glTexCoord4s \
  glTexCoord4sv \
  glTexCoordPointer \
  glTexEnvf \
  glTexEnvfv \
  glTexEnvi \
  glTexEnviv \
  glTexGend \
  glTexGendv \
  glTexGenf \
  glTexGenfv \
  glTexGeni \
  glTexGeniv \
  glTexImage1D \
  glTexImage2D \
  glTexParameterf \
  glTexParameterfv \
  glTexParameteri \
  glTexParameteriv \
  glTexSubImage1D \
  glTexSubImage2D \
  glTranslated \
  glTranslatef \
  glVertex2d \
  glVertex2dv \
  glVertex2f \
  glVertex2fv \
  glVertex2i \
  glVertex2iv \
  glVertex2s \
  glVertex2sv \
  glVertex3d \
  glVertex3dv \
  glVertex3f \
  glVertex3fv \
  glVertex3i \
  glVertex3iv \
  glVertex3s \
  glVertex3sv \
  glVertex4d \
  glVertex4dv \
  glVertex4f \
  glVertex4fv \
  glVertex4i \
  glVertex4iv \
  glVertex4s \
  glVertex4sv \
  glVertexPointer \
  glViewport \
  glCopyTexSubImage3D \
  glDrawRangeElements \
  glTexImage3D \
  glTexSubImage3D \
  glActiveTexture \
  glClientActiveTexture \
  glCompressedTexImage1D \
  glCompressedTexImage2D \
  glCompressedTexImage3D \
  glCompressedTexSubImage1D \
  glCompressedTexSubImage2D \
  glCompressedTexSubImage3D \
  glGetCompressedTexImage \
  glLoadTransposeMatrixd \
  glLoadTransposeMatrixf \
  glMultTransposeMatrixd \
  glMultTransposeMatrixf \
  glMultiTexCoord1d \
  glMultiTexCoord1dv \
  glMultiTexCoord1f \
  glMultiTexCoord1fv \
  glMultiTexCoord1i \
  glMultiTexCoord1iv \
  glMultiTexCoord1s \
  glMultiTexCoord1sv \
  glMultiTexCoord2d \
  glMultiTexCoord2dv \
  glMultiTexCoord2f \
  glMultiTexCoord2fv \
  glMultiTexCoord2i \
  glMultiTexCoord2iv \
  glMultiTexCoord2s \
  glMultiTexCoord2sv \
  glMultiTexCoord3d \
  glMultiTexCoord3dv \
  glMultiTexCoord3f \
  glMultiTexCoord3fv \
  glMultiTexCoord3i \
  glMultiTexCoord3iv \
  glMultiTexCoord3s \
  glMultiTexCoord3sv \
  glMultiTexCoord4d \
  glMultiTexCoord4dv \
  glMultiTexCoord4f \
  glMultiTexCoord4fv \
  glMultiTexCoord4i \
  glMultiTexCoord4iv \
  glMultiTexCoord4s \
  glMultiTexCoord4sv \
  glSampleCoverage \
  glBlendColor \
  glBlendEquation \
  glBlendFuncSeparate \
  glFogCoordPointer \
  glFogCoordd \
  glFogCoorddv \
  glFogCoordf \
  glFogCoordfv \
  glMultiDrawArrays \
  glMultiDrawElements \
  glPointParameterf \
  glPointParameterfv \
  glPointParameteri \
  glPointParameteriv \
  glSecondaryColor3b \
  glSecondaryColor3bv \
  glSecondaryColor3d \
  glSecondaryColor3dv \
  glSecondaryColor3f \
  glSecondaryColor3fv \
  glSecondaryColor3i \
  glSecondaryColor3iv \
  glSecondaryColor3s \
  glSecondaryColor3sv \
  glSecondaryColor3ub \
  glSecondaryColor3ubv \
  glSecondaryColor3ui \
  glSecondaryColor3uiv \
  glSecondaryColor3us \
  glSecondaryColor3usv \
  glSecondaryColorPointer \
  glWindowPos2d \
  glWindowPos2dv \
  glWindowPos2f \
  glWindowPos2fv \
  glWindowPos2i \
  glWindowPos2iv \
  glWindowPos2s \
  glWindowPos2sv \
  glWindowPos3d \
  glWindowPos3dv \
  glWindowPos3f \
  glWindowPos3fv \
  glWindowPos3i \
  glWindowPos3iv \
  glWindowPos3s \
  glWindowPos3sv \
  glBeginQuery \
  glBindBuffer \
  glBufferData \
  glBufferSubData \
  glDeleteBuffers \
  glDeleteQueries \
  glEndQuery \
  glGenBuffers \
  glGenQueries \
  glGetBufferParameteriv \
  glGetBufferPointerv \
  glGetBufferSubData \
  glGetQueryObjectiv \
  glGetQueryObjectuiv \
  glGetQueryiv \
  glIsBuffer \
  glIsQuery \
  glMapBuffer \
  glUnmapBuffer \
  glAttachShader \
  glBindAttribLocation \
  glBlendEquationSeparate \
  glCompileShader \
  glCreateProgram \
  glCreateShader \
  glDeleteProgram \
  glDeleteShader \
  glDetachShader \
  glDisableVertexAttribArray \
  glDrawBuffers \
  glEnableVertexAttribArray \
  glGetActiveAttrib \
  glGetActiveUniform \
  glGetAttachedShaders \
  glGetAttribLocation \
  glGetProgramInfoLog \
  glGetProgramiv \
  glGetShaderInfoLog \
  glGetShaderSource \
  glGetShaderiv \
  glGetUniformLocation \
  glGetUniformfv \
  glGetUniformiv \
  glGetVertexAttribPointerv \
  glGetVertexAttribdv \
  glGetVertexAttribfv \
  glGetVertexAttribiv \
  glIsProgram \
  glIsShader \
  glLinkProgram \
  glShaderSource \
  glStencilFuncSeparate \
  glStencilMaskSeparate \
  glStencilOpSeparate \
  glUniform1f \
  glUniform1fv \
  glUniform1i \
  glUniform1iv \
  glUniform2f \
  glUniform2fv \
  glUniform2i \
  glUniform2iv \
  glUniform3f \
  glUniform3fv \
  glUniform3i \
  glUniform3iv \
  glUniform4f \
  glUniform4fv \
  glUniform4i \
  glUniform4iv \
  glUniformMatrix2fv \
  glUniformMatrix3fv \
  glUniformMatrix4fv \
  glUseProgram \
  glValidateProgram \
  glVertexAttrib1d \
  glVertexAttrib1dv \
  glVertexAttrib1f \
  glVertexAttrib1fv \
  glVertexAttrib1s \
  glVertexAttrib1sv \
  glVertexAttrib2d \
  glVertexAttrib2dv \
  glVertexAttrib2f \
  glVertexAttrib2fv \
  glVertexAttrib2s \
  glVertexAttrib2sv \
  glVertexAttrib3d \
  glVertexAttrib3dv \
  glVertexAttrib3f \
  glVertexAttrib3fv \
  glVertexAttrib3s \
  glVertexAttrib3sv \
  glVertexAttrib4Nbv \
  glVertexAttrib4Niv \
  glVertexAttrib4Nsv \
  glVertexAttrib4Nub \
  glVertexAttrib4Nubv \
  glVertexAttrib4Nuiv \
  glVertexAttrib4Nusv \
  glVertexAttrib4bv \
  glVertexAttrib4d \
  glVertexAttrib4dv \
  glVertexAttrib4f \
  glVertexAttrib4fv \
  glVertexAttrib4iv \
  glVertexAttrib4s \
  glVertexAttrib4sv \
  glVertexAttrib4ubv \
  glVertexAttrib4uiv \
  glVertexAttrib4usv \
  glVertexAttribPointer \
  glUniformMatrix2x3fv \
  glUniformMatrix2x4fv \
  glUniformMatrix3x2fv \
  glUniformMatrix3x4fv \
  glUniformMatrix4x2fv \
  glUniformMatrix4x3fv \
  glBeginConditionalRender \
  glBeginTransformFeedback \
  glBindFragDataLocation \
  glClampColor \
  glClearBufferfi \
  glClearBufferfv \
  glClearBufferiv \
  glClearBufferuiv \
  glColorMaski \
  glDisablei \
  glEnablei \
  glEndConditionalRender \
  glEndTransformFeedback \
  glGetBooleani_v \
  glGetFragDataLocation \
  glGetStringi \
  glGetTexParameterIiv \
  glGetTexParameterIuiv \
  glGetTransformFeedbackVarying \
  glGetUniformuiv \
  glGetVertexAttribIiv \
  glGetVertexAttribIuiv \
  glIsEnabledi \
  glTexParameterIiv \
  glTexParameterIuiv \
  glTransformFeedbackVaryings \
  glUniform1ui \
  glUniform1uiv \
  glUniform2ui \
  glUniform2uiv \
  glUniform3ui \
  glUniform3uiv \
  glUniform4ui \
  glUniform4uiv \
  glVertexAttribI1i \
  glVertexAttribI1iv \
  glVertexAttribI1ui \
  glVertexAttribI1uiv \
  glVertexAttribI2i \
  glVertexAttribI2iv \
  glVertexAttribI2ui \
  glVertexAttribI2uiv \
  glVertexAttribI3i \
  glVertexAttribI3iv \
  glVertexAttribI3ui \
  glVertexAttribI3uiv \
  glVertexAttribI4bv \
  glVertexAttribI4i \
  glVertexAttribI4iv \
  glVertexAttribI4sv \
  glVertexAttribI4ubv \
  glVertexAttribI4ui \
  glVertexAttribI4uiv \
  glVertexAttribI4usv \
  glVertexAttribIPointer \
  glDrawArraysInstanced \
  glDrawElementsInstanced \
  glPrimitiveRestartIndex \
  glTexBuffer \
  glFramebufferTexture \
  glGetBufferParameteri64v \
  glGetInteger64i_v \
  glVertexAttribDivisor \
  glBlendEquationSeparatei \
  glBlendEquationi \
  glBlendFuncSeparatei \
  glBlendFunci \
  glMinSampleShading \
  glGetGraphicsResetStatus \
  glGetnCompressedTexImage \
  glGetnTexImage \
  glGetnUniformdv \
  glMultiDrawArraysIndirectCount \
  glMultiDrawElementsIndirectCount \
  glSpecializeShader \
  glTbufferMask3DFX \
  glDebugMessageCallbackAMD \
  glDebugMessageEnableAMD \
  glDebugMessageInsertAMD \
  glGetDebugMessageLogAMD \
  glBlendEquationIndexedAMD \
  glBlendEquationSeparateIndexedAMD \
  glBlendFuncIndexedAMD \
  glBlendFuncSeparateIndexedAMD \
  glNamedRenderbufferStorageMultisampleAdvancedAMD \
  glRenderbufferStorageMultisampleAdvancedAMD \
  glFramebufferSamplePositionsfvAMD \
  glGetFramebufferParameterfvAMD \
  glGetNamedFramebufferParameterfvAMD \
  glNamedFramebufferSamplePositionsfvAMD \
  glVertexAttribParameteriAMD \
  glMultiDrawArraysIndirectAMD \
  glMultiDrawElementsIndirectAMD \
  glDeleteNamesAMD \
  glGenNamesAMD \
  glIsNameAMD \
  glQueryObjectParameteruiAMD \
  glBeginPerfMonitorAMD \
  glDeletePerfMonitorsAMD \
  glEndPerfMonitorAMD \
  glGenPerfMonitorsAMD \
  glGetPerfMonitorCounterDataAMD \
  glGetPerfMonitorCounterInfoAMD \
  glGetPerfMonitorCounterStringAMD \
  glGetPerfMonitorCountersAMD \
  glGetPerfMonitorGroupStringAMD \
  glGetPerfMonitorGroupsAMD \
  glSelectPerfMonitorCountersAMD \
  glSetMultisamplefvAMD \
  glTexStorageSparseAMD \
  glTextureStorageSparseAMD \
  glStencilOpValueAMD \
  glTessellationFactorAMD \
  glTessellationModeAMD \
  glBlitFramebufferANGLE \
  glRenderbufferStorageMultisampleANGLE \
  glDrawArraysInstancedANGLE \
  glDrawElementsInstancedANGLE \
  glVertexAttribDivisorANGLE \
  glBeginQueryANGLE \
  glDeleteQueriesANGLE \
  glEndQueryANGLE \
  glGenQueriesANGLE \
  glGetQueryObjecti64vANGLE \
  glGetQueryObjectivANGLE \
  glGetQueryObjectui64vANGLE \
  glGetQueryObjectuivANGLE \
  glGetQueryivANGLE \
  glIsQueryANGLE \
  glQueryCounterANGLE \
  glGetTranslatedShaderSourceANGLE \
  glCopyTextureLevelsAPPLE \
  glDrawElementArrayAPPLE \
  glDrawRangeElementArrayAPPLE \
  glElementPointerAPPLE \
  glMultiDrawElementArrayAPPLE \
  glMultiDrawRangeElementArrayAPPLE \
  glDeleteFencesAPPLE \
  glFinishFenceAPPLE \
  glFinishObjectAPPLE \
  glGenFencesAPPLE \
  glIsFenceAPPLE \
  glSetFenceAPPLE \
  glTestFenceAPPLE \
  glTestObjectAPPLE \
  glBufferParameteriAPPLE \
  glFlushMappedBufferRangeAPPLE \
  glRenderbufferStorageMultisampleAPPLE \
  glResolveMultisampleFramebufferAPPLE \
  glGetObjectParameterivAPPLE \
  glObjectPurgeableAPPLE \
  glObjectUnpurgeableAPPLE \
  glClientWaitSyncAPPLE \
  glDeleteSyncAPPLE \
  glFenceSyncAPPLE \
  glGetInteger64vAPPLE \
  glGetSyncivAPPLE \
  glIsSyncAPPLE \
  glWaitSyncAPPLE \
  glGetTexParameterPointervAPPLE \
  glTextureRangeAPPLE \
  glBindVertexArrayAPPLE \
  glDeleteVertexArraysAPPLE \
  glGenVertexArraysAPPLE \
  glIsVertexArrayAPPLE \
  glFlushVertexArrayRangeAPPLE \
  glVertexArrayParameteriAPPLE \
  glVertexArrayRangeAPPLE \
  glDisableVertexAttribAPPLE \
  glEnableVertexAttribAPPLE \
  glIsVertexAttribEnabledAPPLE \
  glMapVertexAttrib1dAPPLE \
  glMapVertexAttrib1fAPPLE \
  glMapVertexAttrib2dAPPLE \
  glMapVertexAttrib2fAPPLE \
  glClearDepthf \
  glDepthRangef \
  glGetShaderPrecisionFormat \
  glReleaseShaderCompiler \
  glShaderBinary \
  glMemoryBarrierByRegion \
  glPrimitiveBoundingBoxARB \
  glDrawArraysInstancedBaseInstance \
  glDrawElementsInstancedBaseInstance \
  glDrawElementsInstancedBaseVertexBaseInstance \
  glGetImageHandleARB \
  glGetTextureHandleARB \
  glGetTextureSamplerHandleARB \
  glGetVertexAttribLui64vARB \
  glIsImageHandleResidentARB \
  glIsTextureHandleResidentARB \
  glMakeImageHandleNonResidentARB \
  glMakeImageHandleResidentARB \
  glMakeTextureHandleNonResidentARB \
  glMakeTextureHandleResidentARB \
  glProgramUniformHandleui64ARB \
  glProgramUniformHandleui64vARB \
  glUniformHandleui64ARB \
  glUniformHandleui64vARB \
  glVertexAttribL1ui64ARB \
  glVertexAttribL1ui64vARB \
  glBindFragDataLocationIndexed \
  glGetFragDataIndex \
  glBufferStorage \
  glCreateSyncFromCLeventARB \
  glClearBufferData \
  glClearBufferSubData \
  glClearNamedBufferDataEXT \
  glClearNamedBufferSubDataEXT \
  glClearTexImage \
  glClearTexSubImage \
  glClipControl \
  glClampColorARB \
  glDispatchCompute \
  glDispatchComputeIndirect \
  glDispatchComputeGroupSizeARB \
  glCopyBufferSubData \
  glCopyImageSubData \
  glDebugMessageCallbackARB \
  glDebugMessageControlARB \
  glDebugMessageInsertARB \
  glGetDebugMessageLogARB \
  glBindTextureUnit \
  glBlitNamedFramebuffer \
  glCheckNamedFramebufferStatus \
  glClearNamedBufferData \
  glClearNamedBufferSubData \
  glClearNamedFramebufferfi \
  glClearNamedFramebufferfv \
  glClearNamedFramebufferiv \
  glClearNamedFramebufferuiv \
  glCompressedTextureSubImage1D \
  glCompressedTextureSubImage2D \
  glCompressedTextureSubImage3D \
  glCopyNamedBufferSubData \
  glCopyTextureSubImage1D \
  glCopyTextureSubImage2D \
  glCopyTextureSubImage3D \
  glCreateBuffers \
  glCreateFramebuffers \
  glCreateProgramPipelines \
  glCreateQueries \
  glCreateRenderbuffers \
  glCreateSamplers \
  glCreateTextures \
  glCreateTransformFeedbacks \
  glCreateVertexArrays \
  glDisableVertexArrayAttrib \
  glEnableVertexArrayAttrib \
  glFlushMappedNamedBufferRange \
  glGenerateTextureMipmap \
  glGetCompressedTextureImage \
  glGetNamedBufferParameteri64v \
  glGetNamedBufferParameteriv \
  glGetNamedBufferPointerv \
  glGetNamedBufferSubData \
  glGetNamedFramebufferAttachmentParameteriv \
  glGetNamedFramebufferParameteriv \
  glGetNamedRenderbufferParameteriv \
  glGetQueryBufferObjecti64v \
  glGetQueryBufferObjectiv \
  glGetQueryBufferObjectui64v \
  glGetQueryBufferObjectuiv \
  glGetTextureImage \
  glGetTextureLevelParameterfv \
  glGetTextureLevelParameteriv \
  glGetTextureParameterIiv \
  glGetTextureParameterIuiv \
  glGetTextureParameterfv \
  glGetTextureParameteriv \
  glGetTransformFeedbacki64_v \
  glGetTransformFeedbacki_v \
  glGetTransformFeedbackiv \
  glGetVertexArrayIndexed64iv \
  glGetVertexArrayIndexediv \
  glGetVertexArrayiv \
  glInvalidateNamedFramebufferData \
  glInvalidateNamedFramebufferSubData \
  glMapNamedBuffer \
  glMapNamedBufferRange \
  glNamedBufferData \
  glNamedBufferStorage \
  glNamedBufferSubData \
  glNamedFramebufferDrawBuffer \
  glNamedFramebufferDrawBuffers \
  glNamedFramebufferParameteri \
  glNamedFramebufferReadBuffer \
  glNamedFramebufferRenderbuffer \
  glNamedFramebufferTexture \
  glNamedFramebufferTextureLayer \
  glNamedRenderbufferStorage \
  glNamedRenderbufferStorageMultisample \
  glTextureBuffer \
  glTextureBufferRange \
  glTextureParameterIiv \
  glTextureParameterIuiv \
  glTextureParameterf \
  glTextureParameterfv \
  glTextureParameteri \
  glTextureParameteriv \
  glTextureStorage1D \
  glTextureStorage2D \
  glTextureStorage2DMultisample \
  glTextureStorage3D \
  glTextureStorage3DMultisample \
  glTextureSubImage1D \
  glTextureSubImage2D \
  glTextureSubImage3D \
  glTransformFeedbackBufferBase \
  glTransformFeedbackBufferRange \
  glUnmapNamedBuffer \
  glVertexArrayAttribBinding \
  glVertexArrayAttribFormat \
  glVertexArrayAttribIFormat \
  glVertexArrayAttribLFormat \
  glVertexArrayBindingDivisor \
  glVertexArrayElementBuffer \
  glVertexArrayVertexBuffer \
  glVertexArrayVertexBuffers \
  glDrawBuffersARB \
  glBlendEquationSeparateiARB \
  glBlendEquationiARB \
  glBlendFuncSeparateiARB \
  glBlendFunciARB \
  glDrawElementsBaseVertex \
  glDrawElementsInstancedBaseVertex \
  glDrawRangeElementsBaseVertex \
  glMultiDrawElementsBaseVertex \
  glDrawArraysIndirect \
  glDrawElementsIndirect \
  glFramebufferParameteri \
  glGetFramebufferParameteriv \
  glGetNamedFramebufferParameterivEXT \
  glNamedFramebufferParameteriEXT \
  glBindFramebuffer \
  glBindRenderbuffer \
  glBlitFramebuffer \
  glCheckFramebufferStatus \
  glDeleteFramebuffers \
  glDeleteRenderbuffers \
  glFramebufferRenderbuffer \
  glFramebufferTexture1D \
  glFramebufferTexture2D \
  glFramebufferTexture3D \
  glFramebufferTextureLayer \
  glGenFramebuffers \
  glGenRenderbuffers \
  glGenerateMipmap \
  glGetFramebufferAttachmentParameteriv \
  glGetRenderbufferParameteriv \
  glIsFramebuffer \
  glIsRenderbuffer \
  glRenderbufferStorage \
  glRenderbufferStorageMultisample \
  glFramebufferTextureARB \
  glFramebufferTextureFaceARB \
  glFramebufferTextureLayerARB \
  glProgramParameteriARB \
  glGetProgramBinary \
  glProgramBinary \
  glProgramParameteri \
  glGetCompressedTextureSubImage \
  glGetTextureSubImage \
  glSpecializeShaderARB \
  glGetUniformdv \
  glUniform1d \
  glUniform1dv \
  glUniform2d \
  glUniform2dv \
  glUniform3d \
  glUniform3dv \
  glUniform4d \
  glUniform4dv \
  glUniformMatrix2dv \
  glUniformMatrix2x3dv \
  glUniformMatrix2x4dv \
  glUniformMatrix3dv \
  glUniformMatrix3x2dv \
  glUniformMatrix3x4dv \
  glUniformMatrix4dv \
  glUniformMatrix4x2dv \
  glUniformMatrix4x3dv \
  glGetUniformi64vARB \
  glGetUniformui64vARB \
  glGetnUniformi64vARB \
  glGetnUniformui64vARB \
  glProgramUniform1i64ARB \
  glProgramUniform1i64vARB \
  glProgramUniform1ui64ARB \
  glProgramUniform1ui64vARB \
  glProgramUniform2i64ARB \
  glProgramUniform2i64vARB \
  glProgramUniform2ui64ARB \
  glProgramUniform2ui64vARB \
  glProgramUniform3i64ARB \
  glProgramUniform3i64vARB \
  glProgramUniform3ui64ARB \
  glProgramUniform3ui64vARB \
  glProgramUniform4i64ARB \
  glProgramUniform4i64vARB \
  glProgramUniform4ui64ARB \
  glProgramUniform4ui64vARB \
  glUniform1i64ARB \
  glUniform1i64vARB \
  glUniform1ui64ARB \
  glUniform1ui64vARB \
  glUniform2i64ARB \
  glUniform2i64vARB \
  glUniform2ui64ARB \
  glUniform2ui64vARB \
  glUniform3i64ARB \
  glUniform3i64vARB \
  glUniform3ui64ARB \
  glUniform3ui64vARB \
  glUniform4i64ARB \
  glUniform4i64vARB \
  glUniform4ui64ARB \
  glUniform4ui64vARB \
  glColorSubTable \
  glColorTable \
  glColorTableParameterfv \
  glColorTableParameteriv \
  glConvolutionFilter1D \
  glConvolutionFilter2D \
  glConvolutionParameterf \
  glConvolutionParameterfv \
  glConvolutionParameteri \
  glConvolutionParameteriv \
  glCopyColorSubTable \
  glCopyColorTable \
  glCopyConvolutionFilter1D \
  glCopyConvolutionFilter2D \
  glGetColorTable \
  glGetColorTableParameterfv \
  glGetColorTableParameteriv \
  glGetConvolutionFilter \
  glGetConvolutionParameterfv \
  glGetConvolutionParameteriv \
  glGetHistogram \
  glGetHistogramParameterfv \
  glGetHistogramParameteriv \
  glGetMinmax \
  glGetMinmaxParameterfv \
  glGetMinmaxParameteriv \
  glGetSeparableFilter \
  glHistogram \
  glMinmax \
  glResetHistogram \
  glResetMinmax \
  glSeparableFilter2D \
  glMultiDrawArraysIndirectCountARB \
  glMultiDrawElementsIndirectCountARB \
  glDrawArraysInstancedARB \
  glDrawElementsInstancedARB \
  glVertexAttribDivisorARB \
  glGetInternalformativ \
  glGetInternalformati64v \
  glInvalidateBufferData \
  glInvalidateBufferSubData \
  glInvalidateFramebuffer \
  glInvalidateSubFramebuffer \
  glInvalidateTexImage \
  glInvalidateTexSubImage \
  glFlushMappedBufferRange \
  glMapBufferRange \
  glCurrentPaletteMatrixARB \
  glMatrixIndexPointerARB \
  glMatrixIndexubvARB \
  glMatrixIndexuivARB \
  glMatrixIndexusvARB \
  glBindBuffersBase \
  glBindBuffersRange \
  glBindImageTextures \
  glBindSamplers \
  glBindTextures \
  glBindVertexBuffers \
  glMultiDrawArraysIndirect \
  glMultiDrawElementsIndirect \
  glSampleCoverageARB \
  glActiveTextureARB \
  glClientActiveTextureARB \
  glMultiTexCoord1dARB \
  glMultiTexCoord1dvARB \
  glMultiTexCoord1fARB \
  glMultiTexCoord1fvARB \
  glMultiTexCoord1iARB \
  glMultiTexCoord1ivARB \
  glMultiTexCoord1sARB \
  glMultiTexCoord1svARB \
  glMultiTexCoord2dARB \
  glMultiTexCoord2dvARB \
  glMultiTexCoord2fARB \
  glMultiTexCoord2fvARB \
  glMultiTexCoord2iARB \
  glMultiTexCoord2ivARB \
  glMultiTexCoord2sARB \
  glMultiTexCoord2svARB \
  glMultiTexCoord3dARB \
  glMultiTexCoord3dvARB \
  glMultiTexCoord3fARB \
  glMultiTexCoord3fvARB \
  glMultiTexCoord3iARB \
  glMultiTexCoord3ivARB \
  glMultiTexCoord3sARB \
  glMultiTexCoord3svARB \
  glMultiTexCoord4dARB \
  glMultiTexCoord4dvARB \
  glMultiTexCoord4fARB \
  glMultiTexCoord4fvARB \
  glMultiTexCoord4iARB \
  glMultiTexCoord4ivARB \
  glMultiTexCoord4sARB \
  glMultiTexCoord4svARB \
  glBeginQueryARB \
  glDeleteQueriesARB \
  glEndQueryARB \
  glGenQueriesARB \
  glGetQueryObjectivARB \
  glGetQueryObjectuivARB \
  glGetQueryivARB \
  glIsQueryARB \
  glMaxShaderCompilerThreadsARB \
  glPointParameterfARB \
  glPointParameterfvARB \
  glPolygonOffsetClamp \
  glGetProgramInterfaceiv \
  glGetProgramResourceIndex \
  glGetProgramResourceLocation \
  glGetProgramResourceLocationIndex \
  glGetProgramResourceName \
  glGetProgramResourceiv \
  glProvokingVertex \
  glGetGraphicsResetStatusARB \
  glGetnColorTableARB \
  glGetnCompressedTexImageARB \
  glGetnConvolutionFilterARB \
  glGetnHistogramARB \
  glGetnMapdvARB \
  glGetnMapfvARB \
  glGetnMapivARB \
  glGetnMinmaxARB \
  glGetnPixelMapfvARB \
  glGetnPixelMapuivARB \
  glGetnPixelMapusvARB \
  glGetnPolygonStippleARB \
  glGetnSeparableFilterARB \
  glGetnTexImageARB \
  glGetnUniformdvARB \
  glGetnUniformfvARB \
  glGetnUniformivARB \
  glGetnUniformuivARB \
  glReadnPixelsARB \
  glFramebufferSampleLocationsfvARB \
  glNamedFramebufferSampleLocationsfvARB \
  glMinSampleShadingARB \
  glBindSampler \
  glDeleteSamplers \
  glGenSamplers \
  glGetSamplerParameterIiv \
  glGetSamplerParameterIuiv \
  glGetSamplerParameterfv \
  glGetSamplerParameteriv \
  glIsSampler \
  glSamplerParameterIiv \
  glSamplerParameterIuiv \
  glSamplerParameterf \
  glSamplerParameterfv \
  glSamplerParameteri \
  glSamplerParameteriv \
  glActiveShaderProgram \
  glBindProgramPipeline \
  glCreateShaderProgramv \
  glDeleteProgramPipelines \
  glGenProgramPipelines \
  glGetProgramPipelineInfoLog \
  glGetProgramPipelineiv \
  glIsProgramPipeline \
  glProgramUniform1d \
  glProgramUniform1dv \
  glProgramUniform1f \
  glProgramUniform1fv \
  glProgramUniform1i \
  glProgramUniform1iv \
  glProgramUniform1ui \
  glProgramUniform1uiv \
  glProgramUniform2d \
  glProgramUniform2dv \
  glProgramUniform2f \
  glProgramUniform2fv \
  glProgramUniform2i \
  glProgramUniform2iv \
  glProgramUniform2ui \
  glProgramUniform2uiv \
  glProgramUniform3d \
  glProgramUniform3dv \
  glProgramUniform3f \
  glProgramUniform3fv \
  glProgramUniform3i \
  glProgramUniform3iv \
  glProgramUniform3ui \
  glProgramUniform3uiv \
  glProgramUniform4d \
  glProgramUniform4dv \
  glProgramUniform4f \
  glProgramUniform4fv \
  glProgramUniform4i \
  glProgramUniform4iv \
  glProgramUniform4ui \
  glProgramUniform4uiv \
  glProgramUniformMatrix2dv \
  glProgramUniformMatrix2fv \
  glProgramUniformMatrix2x3dv \
  glProgramUniformMatrix2x3fv \
  glProgramUniformMatrix2x4dv \
  glProgramUniformMatrix2x4fv \
  glProgramUniformMatrix3dv \
  glProgramUniformMatrix3fv \
  glProgramUniformMatrix3x2dv \
  glProgramUniformMatrix3x2fv \
  glProgramUniformMatrix3x4dv \
  glProgramUniformMatrix3x4fv \
  glProgramUniformMatrix4dv \
  glProgramUniformMatrix4fv \
  glProgramUniformMatrix4x2dv \
  glProgramUniformMatrix4x2fv \
  glProgramUniformMatrix4x3dv \
  glProgramUniformMatrix4x3fv \
  glUseProgramStages \
  glValidateProgramPipeline \
  glGetActiveAtomicCounterBufferiv \
  glBindImageTexture \
  glMemoryBarrier \
  glAttachObjectARB \
  glCompileShaderARB \
  glCreateProgramObjectARB \
  glCreateShaderObjectARB \
  glDeleteObjectARB \
  glDetachObjectARB \
  glGetActiveUniformARB \
  glGetAttachedObjectsARB \
  glGetHandleARB \
  glGetInfoLogARB \
  glGetObjectParameterfvARB \
  glGetObjectParameterivARB \
  glGetShaderSourceARB \
  glGetUniformLocationARB \
  glGetUniformfvARB \
  glGetUniformivARB \
  glLinkProgramARB \
  glShaderSourceARB \
  glUniform1fARB \
  glUniform1fvARB \
  glUniform1iARB \
  glUniform1ivARB \
  glUniform2fARB \
  glUniform2fvARB \
  glUniform2iARB \
  glUniform2ivARB \
  glUniform3fARB \
  glUniform3fvARB \
  glUniform3iARB \
  glUniform3ivARB \
  glUniform4fARB \
  glUniform4fvARB \
  glUniform4iARB \
  glUniform4ivARB \
  glUniformMatrix2fvARB \
  glUniformMatrix3fvARB \
  glUniformMatrix4fvARB \
  glUseProgramObjectARB \
  glValidateProgramARB \
  glShaderStorageBlockBinding \
  glGetActiveSubroutineName \
  glGetActiveSubroutineUniformName \
  glGetActiveSubroutineUniformiv \
  glGetProgramStageiv \
  glGetSubroutineIndex \
  glGetSubroutineUniformLocation \
  glGetUniformSubroutineuiv \
  glUniformSubroutinesuiv \
  glCompileShaderIncludeARB \
  glDeleteNamedStringARB \
  glGetNamedStringARB \
  glGetNamedStringivARB \
  glIsNamedStringARB \
  glNamedStringARB \
  glBufferPageCommitmentARB \
  glTexPageCommitmentARB \
  glClientWaitSync \
  glDeleteSync \
  glFenceSync \
  glGetInteger64v \
  glGetSynciv \
  glIsSync \
  glWaitSync \
  glPatchParameterfv \
  glPatchParameteri \
  glTextureBarrier \
  glTexBufferARB \
  glTexBufferRange \
  glTextureBufferRangeEXT \
  glCompressedTexImage1DARB \
  glCompressedTexImage2DARB \
  glCompressedTexImage3DARB \
  glCompressedTexSubImage1DARB \
  glCompressedTexSubImage2DARB \
  glCompressedTexSubImage3DARB \
  glGetCompressedTexImageARB \
  glGetMultisamplefv \
  glSampleMaski \
  glTexImage2DMultisample \
  glTexImage3DMultisample \
  glTexStorage1D \
  glTexStorage2D \
  glTexStorage3D \
  glTexStorage2DMultisample \
  glTexStorage3DMultisample \
  glTextureStorage2DMultisampleEXT \
  glTextureStorage3DMultisampleEXT \
  glTextureView \
  glGetQueryObjecti64v \
  glGetQueryObjectui64v \
  glQueryCounter \
  glBindTransformFeedback \
  glDeleteTransformFeedbacks \
  glDrawTransformFeedback \
  glGenTransformFeedbacks \
  glIsTransformFeedback \
  glPauseTransformFeedback \
  glResumeTransformFeedback \
  glBeginQueryIndexed \
  glDrawTransformFeedbackStream \
  glEndQueryIndexed \
  glGetQueryIndexediv \
  glDrawTransformFeedbackInstanced \
  glDrawTransformFeedbackStreamInstanced \
  glLoadTransposeMatrixdARB \
  glLoadTransposeMatrixfARB \
  glMultTransposeMatrixdARB \
  glMultTransposeMatrixfARB \
  glBindBufferBase \
  glBindBufferRange \
  glGetActiveUniformBlockName \
  glGetActiveUniformBlockiv \
  glGetActiveUniformName \
  glGetActiveUniformsiv \
  glGetIntegeri_v \
  glGetUniformBlockIndex \
  glGetUniformIndices \
  glUniformBlockBinding \
  glBindVertexArray \
  glDeleteVertexArrays \
  glGenVertexArrays \
  glIsVertexArray \
  glGetVertexAttribLdv \
  glVertexAttribL1d \
  glVertexAttribL1dv \
  glVertexAttribL2d \
  glVertexAttribL2dv \
  glVertexAttribL3d \
  glVertexAttribL3dv \
  glVertexAttribL4d \
  glVertexAttribL4dv \
  glVertexAttribLPointer \
  glBindVertexBuffer \
  glVertexArrayBindVertexBufferEXT \
  glVertexArrayVertexAttribBindingEXT \
  glVertexArrayVertexAttribFormatEXT \
  glVertexArrayVertexAttribIFormatEXT \
  glVertexArrayVertexAttribLFormatEXT \
  glVertexArrayVertexBindingDivisorEXT \
  glVertexAttribBinding \
  glVertexAttribFormat \
  glVertexAttribIFormat \
  glVertexAttribLFormat \
  glVertexBindingDivisor \
  glVertexBlendARB \
  glWeightPointerARB \
  glWeightbvARB \
  glWeightdvARB \
  glWeightfvARB \
  glWeightivARB \
  glWeightsvARB \
  glWeightubvARB \
  glWeightuivARB \
  glWeightusvARB \
  glBindBufferARB \
  glBufferDataARB \
  glBufferSubDataARB \
  glDeleteBuffersARB \
  glGenBuffersARB \
  glGetBufferParameterivARB \
  glGetBufferPointervARB \
  glGetBufferSubDataARB \
  glIsBufferARB \
  glMapBufferARB \
  glUnmapBufferARB \
  glBindProgramARB \
  glDeleteProgramsARB \
  glDisableVertexAttribArrayARB \
  glEnableVertexAttribArrayARB \
  glGenProgramsARB \
  glGetProgramEnvParameterdvARB \
  glGetProgramEnvParameterfvARB \
  glGetProgramLocalParameterdvARB \
  glGetProgramLocalParameterfvARB \
  glGetProgramStringARB \
  glGetProgramivARB \
  glGetVertexAttribPointervARB \
  glGetVertexAttribdvARB \
  glGetVertexAttribfvARB \
  glGetVertexAttribivARB \
  glIsProgramARB \
  glProgramEnvParameter4dARB \
  glProgramEnvParameter4dvARB \
  glProgramEnvParameter4fARB \
  glProgramEnvParameter4fvARB \
  glProgramLocalParameter4dARB \
  glProgramLocalParameter4dvARB \
  glProgramLocalParameter4fARB \
  glProgramLocalParameter4fvARB \
  glProgramStringARB \
  glVertexAttrib1dARB \
  glVertexAttrib1dvARB \
  glVertexAttrib1fARB \
  glVertexAttrib1fvARB \
  glVertexAttrib1sARB \
  glVertexAttrib1svARB \
  glVertexAttrib2dARB \
  glVertexAttrib2dvARB \
  glVertexAttrib2fARB \
  glVertexAttrib2fvARB \
  glVertexAttrib2sARB \
  glVertexAttrib2svARB \
  glVertexAttrib3dARB \
  glVertexAttrib3dvARB \
  glVertexAttrib3fARB \
  glVertexAttrib3fvARB \
  glVertexAttrib3sARB \
  glVertexAttrib3svARB \
  glVertexAttrib4NbvARB \
  glVertexAttrib4NivARB \
  glVertexAttrib4NsvARB \
  glVertexAttrib4NubARB \
  glVertexAttrib4NubvARB \
  glVertexAttrib4NuivARB \
  glVertexAttrib4NusvARB \
  glVertexAttrib4bvARB \
  glVertexAttrib4dARB \
  glVertexAttrib4dvARB \
  glVertexAttrib4fARB \
  glVertexAttrib4fvARB \
  glVertexAttrib4ivARB \
  glVertexAttrib4sARB \
  glVertexAttrib4svARB \
  glVertexAttrib4ubvARB \
  glVertexAttrib4uivARB \
  glVertexAttrib4usvARB \
  glVertexAttribPointerARB \
  glBindAttribLocationARB \
  glGetActiveAttribARB \
  glGetAttribLocationARB \
  glColorP3ui \
  glColorP3uiv \
  glColorP4ui \
  glColorP4uiv \
  glMultiTexCoordP1ui \
  glMultiTexCoordP1uiv \
  glMultiTexCoordP2ui \
  glMultiTexCoordP2uiv \
  glMultiTexCoordP3ui \
  glMultiTexCoordP3uiv \
  glMultiTexCoordP4ui \
  glMultiTexCoordP4uiv \
  glNormalP3ui \
  glNormalP3uiv \
  glSecondaryColorP3ui \
  glSecondaryColorP3uiv \
  glTexCoordP1ui \
  glTexCoordP1uiv \
  glTexCoordP2ui \
  glTexCoordP2uiv \
  glTexCoordP3ui \
  glTexCoordP3uiv \
  glTexCoordP4ui \
  glTexCoordP4uiv \
  glVertexAttribP1ui \
  glVertexAttribP1uiv \
  glVertexAttribP2ui \
  glVertexAttribP2uiv \
  glVertexAttribP3ui \
  glVertexAttribP3uiv \
  glVertexAttribP4ui \
  glVertexAttribP4uiv \
  glVertexP2ui \
  glVertexP2uiv \
  glVertexP3ui \
  glVertexP3uiv \
  glVertexP4ui \
  glVertexP4uiv \
  glDepthRangeArrayv \
  glDepthRangeIndexed \
  glGetDoublei_v \
  glGetFloati_v \
  glScissorArrayv \
  glScissorIndexed \
  glScissorIndexedv \
  glViewportArrayv \
  glViewportIndexedf \
  glViewportIndexedfv \
  glWindowPos2dARB \
  glWindowPos2dvARB \
  glWindowPos2fARB \
  glWindowPos2fvARB \
  glWindowPos2iARB \
  glWindowPos2ivARB \
  glWindowPos2sARB \
  glWindowPos2svARB \
  glWindowPos3dARB \
  glWindowPos3dvARB \
  glWindowPos3fARB \
  glWindowPos3fvARB \
  glWindowPos3iARB \
  glWindowPos3ivARB \
  glWindowPos3sARB \
  glWindowPos3svARB \
  glDrawBuffersATI \
  glDrawElementArrayATI \
  glDrawRangeElementArrayATI \
  glElementPointerATI \
  glGetTexBumpParameterfvATI \
  glGetTexBumpParameterivATI \
  glTexBumpParameterfvATI \
  glTexBumpParameterivATI \
  glAlphaFragmentOp1ATI \
  glAlphaFragmentOp2ATI \
  glAlphaFragmentOp3ATI \
  glBeginFragmentShaderATI \
  glBindFragmentShaderATI \
  glColorFragmentOp1ATI \
  glColorFragmentOp2ATI \
  glColorFragmentOp3ATI \
  glDeleteFragmentShaderATI \
  glEndFragmentShaderATI \
  glGenFragmentShadersATI \
  glPassTexCoordATI \
  glSampleMapATI \
  glSetFragmentShaderConstantATI \
  glMapObjectBufferATI \
  glUnmapObjectBufferATI \
  glPNTrianglesfATI \
  glPNTrianglesiATI \
  glStencilFuncSeparateATI \
  glStencilOpSeparateATI \
  glArrayObjectATI \
  glFreeObjectBufferATI \
  glGetArrayObjectfvATI \
  glGetArrayObjectivATI \
  glGetObjectBufferfvATI \
  glGetObjectBufferivATI \
  glGetVariantArrayObjectfvATI \
  glGetVariantArrayObjectivATI \
  glIsObjectBufferATI \
  glNewObjectBufferATI \
  glUpdateObjectBufferATI \
  glVariantArrayObjectATI \
  glGetVertexAttribArrayObjectfvATI \
  glGetVertexAttribArrayObjectivATI \
  glVertexAttribArrayObjectATI \
  glClientActiveVertexStreamATI \
  glNormalStream3bATI \
  glNormalStream3bvATI \
  glNormalStream3dATI \
  glNormalStream3dvATI \
  glNormalStream3fATI \
  glNormalStream3fvATI \
  glNormalStream3iATI \
  glNormalStream3ivATI \
  glNormalStream3sATI \
  glNormalStream3svATI \
  glVertexBlendEnvfATI \
  glVertexBlendEnviATI \
  glVertexStream1dATI \
  glVertexStream1dvATI \
  glVertexStream1fATI \
  glVertexStream1fvATI \
  glVertexStream1iATI \
  glVertexStream1ivATI \
  glVertexStream1sATI \
  glVertexStream1svATI \
  glVertexStream2dATI \
  glVertexStream2dvATI \
  glVertexStream2fATI \
  glVertexStream2fvATI \
  glVertexStream2iATI \
  glVertexStream2ivATI \
  glVertexStream2sATI \
  glVertexStream2svATI \
  glVertexStream3dATI \
  glVertexStream3dvATI \
  glVertexStream3fATI \
  glVertexStream3fvATI \
  glVertexStream3iATI \
  glVertexStream3ivATI \
  glVertexStream3sATI \
  glVertexStream3svATI \
  glVertexStream4dATI \
  glVertexStream4dvATI \
  glVertexStream4fATI \
  glVertexStream4fvATI \
  glVertexStream4iATI \
  glVertexStream4ivATI \
  glVertexStream4sATI \
  glVertexStream4svATI \
  glEGLImageTargetTexStorageEXT \
  glEGLImageTargetTextureStorageEXT \
  glDrawArraysInstancedBaseInstanceEXT \
  glDrawElementsInstancedBaseInstanceEXT \
  glDrawElementsInstancedBaseVertexBaseInstanceEXT \
  glGetUniformBufferSizeEXT \
  glGetUniformOffsetEXT \
  glUniformBufferEXT \
  glBlendColorEXT \
  glBlendEquationSeparateEXT \
  glBindFragDataLocationIndexedEXT \
  glGetFragDataIndexEXT \
  glGetProgramResourceLocationIndexEXT \
  glBlendFuncSeparateEXT \
  glBlendEquationEXT \
  glBufferStorageEXT \
  glNamedBufferStorageEXT \
  glClearTexImageEXT \
  glClearTexSubImageEXT \
  glClipControlEXT \
  glColorSubTableEXT \
  glCopyColorSubTableEXT \
  glLockArraysEXT \
  glUnlockArraysEXT \
  glConvolutionFilter1DEXT \
  glConvolutionFilter2DEXT \
  glConvolutionParameterfEXT \
  glConvolutionParameterfvEXT \
  glConvolutionParameteriEXT \
  glConvolutionParameterivEXT \
  glCopyConvolutionFilter1DEXT \
  glCopyConvolutionFilter2DEXT \
  glGetConvolutionFilterEXT \
  glGetConvolutionParameterfvEXT \
  glGetConvolutionParameterivEXT \
  glGetSeparableFilterEXT \
  glSeparableFilter2DEXT \
  glBinormalPointerEXT \
  glTangentPointerEXT \
  glCopyImageSubDataEXT \
  glCopyTexImage1DEXT \
  glCopyTexImage2DEXT \
  glCopyTexSubImage1DEXT \
  glCopyTexSubImage2DEXT \
  glCopyTexSubImage3DEXT \
  glCullParameterdvEXT \
  glCullParameterfvEXT \
  glGetObjectLabelEXT \
  glLabelObjectEXT \
  glInsertEventMarkerEXT \
  glPopGroupMarkerEXT \
  glPushGroupMarkerEXT \
  glDepthBoundsEXT \
  glBindMultiTextureEXT \
  glCheckNamedFramebufferStatusEXT \
  glClientAttribDefaultEXT \
  glCompressedMultiTexImage1DEXT \
  glCompressedMultiTexImage2DEXT \
  glCompressedMultiTexImage3DEXT \
  glCompressedMultiTexSubImage1DEXT \
  glCompressedMultiTexSubImage2DEXT \
  glCompressedMultiTexSubImage3DEXT \
  glCompressedTextureImage1DEXT \
  glCompressedTextureImage2DEXT \
  glCompressedTextureImage3DEXT \
  glCompressedTextureSubImage1DEXT \
  glCompressedTextureSubImage2DEXT \
  glCompressedTextureSubImage3DEXT \
  glCopyMultiTexImage1DEXT \
  glCopyMultiTexImage2DEXT \
  glCopyMultiTexSubImage1DEXT \
  glCopyMultiTexSubImage2DEXT \
  glCopyMultiTexSubImage3DEXT \
  glCopyTextureImage1DEXT \
  glCopyTextureImage2DEXT \
  glCopyTextureSubImage1DEXT \
  glCopyTextureSubImage2DEXT \
  glCopyTextureSubImage3DEXT \
  glDisableClientStateIndexedEXT \
  glDisableClientStateiEXT \
  glDisableVertexArrayAttribEXT \
  glDisableVertexArrayEXT \
  glEnableClientStateIndexedEXT \
  glEnableClientStateiEXT \
  glEnableVertexArrayAttribEXT \
  glEnableVertexArrayEXT \
  glFlushMappedNamedBufferRangeEXT \
  glFramebufferDrawBufferEXT \
  glFramebufferDrawBuffersEXT \
  glFramebufferReadBufferEXT \
  glGenerateMultiTexMipmapEXT \
  glGenerateTextureMipmapEXT \
  glGetCompressedMultiTexImageEXT \
  glGetCompressedTextureImageEXT \
  glGetDoubleIndexedvEXT \
  glGetDoublei_vEXT \
  glGetFloatIndexedvEXT \
  glGetFloati_vEXT \
  glGetFramebufferParameterivEXT \
  glGetMultiTexEnvfvEXT \
  glGetMultiTexEnvivEXT \
  glGetMultiTexGendvEXT \
  glGetMultiTexGenfvEXT \
  glGetMultiTexGenivEXT \
  glGetMultiTexImageEXT \
  glGetMultiTexLevelParameterfvEXT \
  glGetMultiTexLevelParameterivEXT \
  glGetMultiTexParameterIivEXT \
  glGetMultiTexParameterIuivEXT \
  glGetMultiTexParameterfvEXT \
  glGetMultiTexParameterivEXT \
  glGetNamedBufferParameterivEXT \
  glGetNamedBufferPointervEXT \
  glGetNamedBufferSubDataEXT \
  glGetNamedFramebufferAttachmentParameterivEXT \
  glGetNamedProgramLocalParameterIivEXT \
  glGetNamedProgramLocalParameterIuivEXT \
  glGetNamedProgramLocalParameterdvEXT \
  glGetNamedProgramLocalParameterfvEXT \
  glGetNamedProgramStringEXT \
  glGetNamedProgramivEXT \
  glGetNamedRenderbufferParameterivEXT \
  glGetPointerIndexedvEXT \
  glGetPointeri_vEXT \
  glGetTextureImageEXT \
  glGetTextureLevelParameterfvEXT \
  glGetTextureLevelParameterivEXT \
  glGetTextureParameterIivEXT \
  glGetTextureParameterIuivEXT \
  glGetTextureParameterfvEXT \
  glGetTextureParameterivEXT \
  glGetVertexArrayIntegeri_vEXT \
  glGetVertexArrayIntegervEXT \
  glGetVertexArrayPointeri_vEXT \
  glGetVertexArrayPointervEXT \
  glMapNamedBufferEXT \
  glMapNamedBufferRangeEXT \
  glMatrixFrustumEXT \
  glMatrixLoadIdentityEXT \
  glMatrixLoadTransposedEXT \
  glMatrixLoadTransposefEXT \
  glMatrixLoaddEXT \
  glMatrixLoadfEXT \
  glMatrixMultTransposedEXT \
  glMatrixMultTransposefEXT \
  glMatrixMultdEXT \
  glMatrixMultfEXT \
  glMatrixOrthoEXT \
  glMatrixPopEXT \
  glMatrixPushEXT \
  glMatrixRotatedEXT \
  glMatrixRotatefEXT \
  glMatrixScaledEXT \
  glMatrixScalefEXT \
  glMatrixTranslatedEXT \
  glMatrixTranslatefEXT \
  glMultiTexBufferEXT \
  glMultiTexCoordPointerEXT \
  glMultiTexEnvfEXT \
  glMultiTexEnvfvEXT \
  glMultiTexEnviEXT \
  glMultiTexEnvivEXT \
  glMultiTexGendEXT \
  glMultiTexGendvEXT \
  glMultiTexGenfEXT \
  glMultiTexGenfvEXT \
  glMultiTexGeniEXT \
  glMultiTexGenivEXT \
  glMultiTexImage1DEXT \
  glMultiTexImage2DEXT \
  glMultiTexImage3DEXT \
  glMultiTexParameterIivEXT \
  glMultiTexParameterIuivEXT \
  glMultiTexParameterfEXT \
  glMultiTexParameterfvEXT \
  glMultiTexParameteriEXT \
  glMultiTexParameterivEXT \
  glMultiTexRenderbufferEXT \
  glMultiTexSubImage1DEXT \
  glMultiTexSubImage2DEXT \
  glMultiTexSubImage3DEXT \
  glNamedBufferDataEXT \
  glNamedBufferSubDataEXT \
  glNamedCopyBufferSubDataEXT \
  glNamedFramebufferRenderbufferEXT \
  glNamedFramebufferTexture1DEXT \
  glNamedFramebufferTexture2DEXT \
  glNamedFramebufferTexture3DEXT \
  glNamedFramebufferTextureEXT \
  glNamedFramebufferTextureFaceEXT \
  glNamedFramebufferTextureLayerEXT \
  glNamedProgramLocalParameter4dEXT \
  glNamedProgramLocalParameter4dvEXT \
  glNamedProgramLocalParameter4fEXT \
  glNamedProgramLocalParameter4fvEXT \
  glNamedProgramLocalParameterI4iEXT \
  glNamedProgramLocalParameterI4ivEXT \
  glNamedProgramLocalParameterI4uiEXT \
  glNamedProgramLocalParameterI4uivEXT \
  glNamedProgramLocalParameters4fvEXT \
  glNamedProgramLocalParametersI4ivEXT \
  glNamedProgramLocalParametersI4uivEXT \
  glNamedProgramStringEXT \
  glNamedRenderbufferStorageEXT \
  glNamedRenderbufferStorageMultisampleCoverageEXT \
  glNamedRenderbufferStorageMultisampleEXT \
  glProgramUniform1fEXT \
  glProgramUniform1fvEXT \
  glProgramUniform1iEXT \
  glProgramUniform1ivEXT \
  glProgramUniform1uiEXT \
  glProgramUniform1uivEXT \
  glProgramUniform2fEXT \
  glProgramUniform2fvEXT \
  glProgramUniform2iEXT \
  glProgramUniform2ivEXT \
  glProgramUniform2uiEXT \
  glProgramUniform2uivEXT \
  glProgramUniform3fEXT \
  glProgramUniform3fvEXT \
  glProgramUniform3iEXT \
  glProgramUniform3ivEXT \
  glProgramUniform3uiEXT \
  glProgramUniform3uivEXT \
  glProgramUniform4fEXT \
  glProgramUniform4fvEXT \
  glProgramUniform4iEXT \
  glProgramUniform4ivEXT \
  glProgramUniform4uiEXT \
  glProgramUniform4uivEXT \
  glProgramUniformMatrix2fvEXT \
  glProgramUniformMatrix2x3fvEXT \
  glProgramUniformMatrix2x4fvEXT \
  glProgramUniformMatrix3fvEXT \
  glProgramUniformMatrix3x2fvEXT \
  glProgramUniformMatrix3x4fvEXT \
  glProgramUniformMatrix4fvEXT \
  glProgramUniformMatrix4x2fvEXT \
  glProgramUniformMatrix4x3fvEXT \
  glPushClientAttribDefaultEXT \
  glTextureBufferEXT \
  glTextureImage1DEXT \
  glTextureImage2DEXT \
  glTextureImage3DEXT \
  glTextureParameterIivEXT \
  glTextureParameterIuivEXT \
  glTextureParameterfEXT \
  glTextureParameterfvEXT \
  glTextureParameteriEXT \
  glTextureParameterivEXT \
  glTextureRenderbufferEXT \
  glTextureSubImage1DEXT \
  glTextureSubImage2DEXT \
  glTextureSubImage3DEXT \
  glUnmapNamedBufferEXT \
  glVertexArrayColorOffsetEXT \
  glVertexArrayEdgeFlagOffsetEXT \
  glVertexArrayFogCoordOffsetEXT \
  glVertexArrayIndexOffsetEXT \
  glVertexArrayMultiTexCoordOffsetEXT \
  glVertexArrayNormalOffsetEXT \
  glVertexArraySecondaryColorOffsetEXT \
  glVertexArrayTexCoordOffsetEXT \
  glVertexArrayVertexAttribDivisorEXT \
  glVertexArrayVertexAttribIOffsetEXT \
  glVertexArrayVertexAttribOffsetEXT \
  glVertexArrayVertexOffsetEXT \
  glDiscardFramebufferEXT \
  glBeginQueryEXT \
  glDeleteQueriesEXT \
  glEndQueryEXT \
  glGenQueriesEXT \
  glGetInteger64vEXT \
  glGetQueryObjectivEXT \
  glGetQueryObjectuivEXT \
  glGetQueryivEXT \
  glIsQueryEXT \
  glQueryCounterEXT \
  glDrawBuffersEXT \
  glColorMaskIndexedEXT \
  glDisableIndexedEXT \
  glEnableIndexedEXT \
  glGetBooleanIndexedvEXT \
  glGetIntegerIndexedvEXT \
  glIsEnabledIndexedEXT \
  glBlendEquationSeparateiEXT \
  glBlendEquationiEXT \
  glBlendFuncSeparateiEXT \
  glBlendFunciEXT \
  glColorMaskiEXT \
  glDisableiEXT \
  glEnableiEXT \
  glIsEnablediEXT \
  glDrawElementsBaseVertexEXT \
  glDrawElementsInstancedBaseVertexEXT \
  glDrawRangeElementsBaseVertexEXT \
  glMultiDrawElementsBaseVertexEXT \
  glDrawArraysInstancedEXT \
  glDrawElementsInstancedEXT \
  glDrawRangeElementsEXT \
  glDrawTransformFeedbackEXT \
  glDrawTransformFeedbackInstancedEXT \
  glBufferStorageExternalEXT \
  glNamedBufferStorageExternalEXT \
  glFogCoordPointerEXT \
  glFogCoorddEXT \
  glFogCoorddvEXT \
  glFogCoordfEXT \
  glFogCoordfvEXT \
  glFragmentColorMaterialEXT \
  glFragmentLightModelfEXT \
  glFragmentLightModelfvEXT \
  glFragmentLightModeliEXT \
  glFragmentLightModelivEXT \
  glFragmentLightfEXT \
  glFragmentLightfvEXT \
  glFragmentLightiEXT \
  glFragmentLightivEXT \
  glFragmentMaterialfEXT \
  glFragmentMaterialfvEXT \
  glFragmentMaterialiEXT \
  glFragmentMaterialivEXT \
  glGetFragmentLightfvEXT \
  glGetFragmentLightivEXT \
  glGetFragmentMaterialfvEXT \
  glGetFragmentMaterialivEXT \
  glLightEnviEXT \
  glBlitFramebufferEXT \
  glRenderbufferStorageMultisampleEXT \
  glBindFramebufferEXT \
  glBindRenderbufferEXT \
  glCheckFramebufferStatusEXT \
  glDeleteFramebuffersEXT \
  glDeleteRenderbuffersEXT \
  glFramebufferRenderbufferEXT \
  glFramebufferTexture1DEXT \
  glFramebufferTexture2DEXT \
  glFramebufferTexture3DEXT \
  glGenFramebuffersEXT \
  glGenRenderbuffersEXT \
  glGenerateMipmapEXT \
  glGetFramebufferAttachmentParameterivEXT \
  glGetRenderbufferParameterivEXT \
  glIsFramebufferEXT \
  glIsRenderbufferEXT \
  glRenderbufferStorageEXT \
  glFramebufferTextureEXT \
  glFramebufferTextureFaceEXT \
  glProgramParameteriEXT \
  glProgramEnvParameters4fvEXT \
  glProgramLocalParameters4fvEXT \
  glBindFragDataLocationEXT \
  glGetFragDataLocationEXT \
  glGetUniformuivEXT \
  glGetVertexAttribIivEXT \
  glGetVertexAttribIuivEXT \
  glUniform1uiEXT \
  glUniform1uivEXT \
  glUniform2uiEXT \
  glUniform2uivEXT \
  glUniform3uiEXT \
  glUniform3uivEXT \
  glUniform4uiEXT \
  glUniform4uivEXT \
  glVertexAttribI1iEXT \
  glVertexAttribI1ivEXT \
  glVertexAttribI1uiEXT \
  glVertexAttribI1uivEXT \
  glVertexAttribI2iEXT \
  glVertexAttribI2ivEXT \
  glVertexAttribI2uiEXT \
  glVertexAttribI2uivEXT \
  glVertexAttribI3iEXT \
  glVertexAttribI3ivEXT \
  glVertexAttribI3uiEXT \
  glVertexAttribI3uivEXT \
  glVertexAttribI4bvEXT \
  glVertexAttribI4iEXT \
  glVertexAttribI4ivEXT \
  glVertexAttribI4svEXT \
  glVertexAttribI4ubvEXT \
  glVertexAttribI4uiEXT \
  glVertexAttribI4uivEXT \
  glVertexAttribI4usvEXT \
  glVertexAttribIPointerEXT \
  glGetHistogramEXT \
  glGetHistogramParameterfvEXT \
  glGetHistogramParameterivEXT \
  glGetMinmaxEXT \
  glGetMinmaxParameterfvEXT \
  glGetMinmaxParameterivEXT \
  glHistogramEXT \
  glMinmaxEXT \
  glResetHistogramEXT \
  glResetMinmaxEXT \
  glIndexFuncEXT \
  glIndexMaterialEXT \
  glVertexAttribDivisorEXT \
  glApplyTextureEXT \
  glTextureLightEXT \
  glTextureMaterialEXT \
  glFlushMappedBufferRangeEXT \
  glMapBufferRangeEXT \
  glBufferStorageMemEXT \
  glCreateMemoryObjectsEXT \
  glDeleteMemoryObjectsEXT \
  glGetMemoryObjectParameterivEXT \
  glGetUnsignedBytei_vEXT \
  glGetUnsignedBytevEXT \
  glIsMemoryObjectEXT \
  glMemoryObjectParameterivEXT \
  glNamedBufferStorageMemEXT \
  glTexStorageMem1DEXT \
  glTexStorageMem2DEXT \
  glTexStorageMem2DMultisampleEXT \
  glTexStorageMem3DEXT \
  glTexStorageMem3DMultisampleEXT \
  glTextureStorageMem1DEXT \
  glTextureStorageMem2DEXT \
  glTextureStorageMem2DMultisampleEXT \
  glTextureStorageMem3DEXT \
  glTextureStorageMem3DMultisampleEXT \
  glImportMemoryFdEXT \
  glImportMemoryWin32HandleEXT \
  glImportMemoryWin32NameEXT \
  glMultiDrawArraysEXT \
  glMultiDrawElementsEXT \
  glMultiDrawArraysIndirectEXT \
  glMultiDrawElementsIndirectEXT \
  glSampleMaskEXT \
  glSamplePatternEXT \
  glFramebufferTexture2DMultisampleEXT \
  glDrawBuffersIndexedEXT \
  glGetIntegeri_vEXT \
  glReadBufferIndexedEXT \
  glColorTableEXT \
  glGetColorTableEXT \
  glGetColorTableParameterfvEXT \
  glGetColorTableParameterivEXT \
  glGetPixelTransformParameterfvEXT \
  glGetPixelTransformParameterivEXT \
  glPixelTransformParameterfEXT \
  glPixelTransformParameterfvEXT \
  glPixelTransformParameteriEXT \
  glPixelTransformParameterivEXT \
  glPointParameterfEXT \
  glPointParameterfvEXT \
  glPolygonOffsetEXT \
  glPolygonOffsetClampEXT \
  glPrimitiveBoundingBoxEXT \
  glProvokingVertexEXT \
  glCoverageModulationNV \
  glCoverageModulationTableNV \
  glGetCoverageModulationTableNV \
  glRasterSamplesEXT \
  glGetnUniformfvEXT \
  glGetnUniformivEXT \
  glReadnPixelsEXT \
  glBeginSceneEXT \
  glEndSceneEXT \
  glSecondaryColor3bEXT \
  glSecondaryColor3bvEXT \
  glSecondaryColor3dEXT \
  glSecondaryColor3dvEXT \
  glSecondaryColor3fEXT \
  glSecondaryColor3fvEXT \
  glSecondaryColor3iEXT \
  glSecondaryColor3ivEXT \
  glSecondaryColor3sEXT \
  glSecondaryColor3svEXT \
  glSecondaryColor3ubEXT \
  glSecondaryColor3ubvEXT \
  glSecondaryColor3uiEXT \
  glSecondaryColor3uivEXT \
  glSecondaryColor3usEXT \
  glSecondaryColor3usvEXT \
  glSecondaryColorPointerEXT \
  glDeleteSemaphoresEXT \
  glGenSemaphoresEXT \
  glGetSemaphoreParameterui64vEXT \
  glIsSemaphoreEXT \
  glSemaphoreParameterui64vEXT \
  glSignalSemaphoreEXT \
  glWaitSemaphoreEXT \
  glImportSemaphoreFdEXT \
  glImportSemaphoreWin32HandleEXT \
  glImportSemaphoreWin32NameEXT \
  glActiveProgramEXT \
  glCreateShaderProgramEXT \
  glUseShaderProgramEXT \
  glFramebufferFetchBarrierEXT \
  glBindImageTextureEXT \
  glMemoryBarrierEXT \
  glClearPixelLocalStorageuiEXT \
  glFramebufferPixelLocalStorageSizeEXT \
  glGetFramebufferPixelLocalStorageSizeEXT \
  glTexPageCommitmentEXT \
  glTexturePageCommitmentEXT \
  glActiveStencilFaceEXT \
  glTexSubImage1DEXT \
  glTexSubImage2DEXT \
  glTexSubImage3DEXT \
  glPatchParameteriEXT \
  glTexImage3DEXT \
  glFramebufferTextureLayerEXT \
  glGetSamplerParameterIivEXT \
  glGetSamplerParameterIuivEXT \
  glSamplerParameterIivEXT \
  glSamplerParameterIuivEXT \
  glTexBufferEXT \
  glClearColorIiEXT \
  glClearColorIuiEXT \
  glGetTexParameterIivEXT \
  glGetTexParameterIuivEXT \
  glTexParameterIivEXT \
  glTexParameterIuivEXT \
  glAreTexturesResidentEXT \
  glBindTextureEXT \
  glDeleteTexturesEXT \
  glGenTexturesEXT \
  glIsTextureEXT \
  glPrioritizeTexturesEXT \
  glTextureNormalEXT \
  glTexStorage1DEXT \
  glTexStorage2DEXT \
  glTexStorage3DEXT \
  glTextureStorage1DEXT \
  glTextureStorage2DEXT \
  glTextureStorage3DEXT \
  glTextureViewEXT \
  glGetQueryObjecti64vEXT \
  glGetQueryObjectui64vEXT \
  glBeginTransformFeedbackEXT \
  glBindBufferBaseEXT \
  glBindBufferOffsetEXT \
  glBindBufferRangeEXT \
  glEndTransformFeedbackEXT \
  glGetTransformFeedbackVaryingEXT \
  glTransformFeedbackVaryingsEXT \
  glArrayElementEXT \
  glColorPointerEXT \
  glDrawArraysEXT \
  glEdgeFlagPointerEXT \
  glIndexPointerEXT \
  glNormalPointerEXT \
  glTexCoordPointerEXT \
  glVertexPointerEXT \
  glBindArraySetEXT \
  glCreateArraySetExt \
  glDeleteArraySetsEXT \
  glGetVertexAttribLdvEXT \
  glVertexArrayVertexAttribLOffsetEXT \
  glVertexAttribL1dEXT \
  glVertexAttribL1dvEXT \
  glVertexAttribL2dEXT \
  glVertexAttribL2dvEXT \
  glVertexAttribL3dEXT \
  glVertexAttribL3dvEXT \
  glVertexAttribL4dEXT \
  glVertexAttribL4dvEXT \
  glVertexAttribLPointerEXT \
  glBeginVertexShaderEXT \
  glBindLightParameterEXT \
  glBindMaterialParameterEXT \
  glBindParameterEXT \
  glBindTexGenParameterEXT \
  glBindTextureUnitParameterEXT \
  glBindVertexShaderEXT \
  glDeleteVertexShaderEXT \
  glDisableVariantClientStateEXT \
  glEnableVariantClientStateEXT \
  glEndVertexShaderEXT \
  glExtractComponentEXT \
  glGenSymbolsEXT \
  glGenVertexShadersEXT \
  glGetInvariantBooleanvEXT \
  glGetInvariantFloatvEXT \
  glGetInvariantIntegervEXT \
  glGetLocalConstantBooleanvEXT \
  glGetLocalConstantFloatvEXT \
  glGetLocalConstantIntegervEXT \
  glGetVariantBooleanvEXT \
  glGetVariantFloatvEXT \
  glGetVariantIntegervEXT \
  glGetVariantPointervEXT \
  glInsertComponentEXT \
  glIsVariantEnabledEXT \
  glSetInvariantEXT \
  glSetLocalConstantEXT \
  glShaderOp1EXT \
  glShaderOp2EXT \
  glShaderOp3EXT \
  glSwizzleEXT \
  glVariantPointerEXT \
  glVariantbvEXT \
  glVariantdvEXT \
  glVariantfvEXT \
  glVariantivEXT \
  glVariantsvEXT \
  glVariantubvEXT \
  glVariantuivEXT \
  glVariantusvEXT \
  glWriteMaskEXT \
  glVertexWeightPointerEXT \
  glVertexWeightfEXT \
  glVertexWeightfvEXT \
  glAcquireKeyedMutexWin32EXT \
  glReleaseKeyedMutexWin32EXT \
  glWindowRectanglesEXT \
  glImportSyncEXT \
  glFrameTerminatorGREMEDY \
  glStringMarkerGREMEDY \
  glGetImageTransformParameterfvHP \
  glGetImageTransformParameterivHP \
  glImageTransformParameterfHP \
  glImageTransformParameterfvHP \
  glImageTransformParameteriHP \
  glImageTransformParameterivHP \
  glMultiModeDrawArraysIBM \
  glMultiModeDrawElementsIBM \
  glColorPointerListIBM \
  glEdgeFlagPointerListIBM \
  glFogCoordPointerListIBM \
  glIndexPointerListIBM \
  glNormalPointerListIBM \
  glSecondaryColorPointerListIBM \
  glTexCoordPointerListIBM \
  glVertexPointerListIBM \
  glGetTextureHandleIMG \
  glGetTextureSamplerHandleIMG \
  glProgramUniformHandleui64IMG \
  glProgramUniformHandleui64vIMG \
  glUniformHandleui64IMG \
  glUniformHandleui64vIMG \
  glFramebufferTexture2DDownsampleIMG \
  glFramebufferTextureLayerDownsampleIMG \
  glFramebufferTexture2DMultisampleIMG \
  glRenderbufferStorageMultisampleIMG \
  glMapTexture2DINTEL \
  glSyncTextureINTEL \
  glUnmapTexture2DINTEL \
  glColorPointervINTEL \
  glNormalPointervINTEL \
  glTexCoordPointervINTEL \
  glVertexPointervINTEL \
  glBeginPerfQueryINTEL \
  glCreatePerfQueryINTEL \
  glDeletePerfQueryINTEL \
  glEndPerfQueryINTEL \
  glGetFirstPerfQueryIdINTEL \
  glGetNextPerfQueryIdINTEL \
  glGetPerfCounterInfoINTEL \
  glGetPerfQueryDataINTEL \
  glGetPerfQueryIdByNameINTEL \
  glGetPerfQueryInfoINTEL \
  glTexScissorFuncINTEL \
  glTexScissorINTEL \
  glBlendBarrierKHR \
  glDebugMessageCallback \
  glDebugMessageControl \
  glDebugMessageInsert \
  glGetDebugMessageLog \
  glGetObjectLabel \
  glGetObjectPtrLabel \
  glObjectLabel \
  glObjectPtrLabel \
  glPopDebugGroup \
  glPushDebugGroup \
  glMaxShaderCompilerThreadsKHR \
  glGetnUniformfv \
  glGetnUniformiv \
  glGetnUniformuiv \
  glReadnPixels \
  glBufferRegionEnabled \
  glDeleteBufferRegion \
  glDrawBufferRegion \
  glNewBufferRegion \
  glReadBufferRegion \
  glFramebufferParameteriMESA \
  glGetFramebufferParameterivMESA \
  glResizeBuffersMESA \
  glWindowPos2dMESA \
  glWindowPos2dvMESA \
  glWindowPos2fMESA \
  glWindowPos2fvMESA \
  glWindowPos2iMESA \
  glWindowPos2ivMESA \
  glWindowPos2sMESA \
  glWindowPos2svMESA \
  glWindowPos3dMESA \
  glWindowPos3dvMESA \
  glWindowPos3fMESA \
  glWindowPos3fvMESA \
  glWindowPos3iMESA \
  glWindowPos3ivMESA \
  glWindowPos3sMESA \
  glWindowPos3svMESA \
  glWindowPos4dMESA \
  glWindowPos4dvMESA \
  glWindowPos4fMESA \
  glWindowPos4fvMESA \
  glWindowPos4iMESA \
  glWindowPos4ivMESA \
  glWindowPos4sMESA \
  glWindowPos4svMESA \
  glBeginConditionalRenderNVX \
  glEndConditionalRenderNVX \
  glAsyncCopyBufferSubDataNVX \
  glAsyncCopyImageSubDataNVX \
  glMulticastScissorArrayvNVX \
  glMulticastViewportArrayvNVX \
  glMulticastViewportPositionWScaleNVX \
  glUploadGpuMaskNVX \
  glLGPUCopyImageSubDataNVX \
  glLGPUInterlockNVX \
  glLGPUNamedBufferSubDataNVX \
  glClientWaitSemaphoreui64NVX \
  glSignalSemaphoreui64NVX \
  glWaitSemaphoreui64NVX \
  glStereoParameterfNV \
  glStereoParameteriNV \
  glAlphaToCoverageDitherControlNV \
  glMultiDrawArraysIndirectBindlessNV \
  glMultiDrawElementsIndirectBindlessNV \
  glMultiDrawArraysIndirectBindlessCountNV \
  glMultiDrawElementsIndirectBindlessCountNV \
  glGetImageHandleNV \
  glGetTextureHandleNV \
  glGetTextureSamplerHandleNV \
  glIsImageHandleResidentNV \
  glIsTextureHandleResidentNV \
  glMakeImageHandleNonResidentNV \
  glMakeImageHandleResidentNV \
  glMakeTextureHandleNonResidentNV \
  glMakeTextureHandleResidentNV \
  glProgramUniformHandleui64NV \
  glProgramUniformHandleui64vNV \
  glUniformHandleui64NV \
  glUniformHandleui64vNV \
  glBlendBarrierNV \
  glBlendParameteriNV \
  glViewportPositionWScaleNV \
  glCallCommandListNV \
  glCommandListSegmentsNV \
  glCompileCommandListNV \
  glCreateCommandListsNV \
  glCreateStatesNV \
  glDeleteCommandListsNV \
  glDeleteStatesNV \
  glDrawCommandsAddressNV \
  glDrawCommandsNV \
  glDrawCommandsStatesAddressNV \
  glDrawCommandsStatesNV \
  glGetCommandHeaderNV \
  glGetStageIndexNV \
  glIsCommandListNV \
  glIsStateNV \
  glListDrawCommandsStatesClientNV \
  glStateCaptureNV \
  glBeginConditionalRenderNV \
  glEndConditionalRenderNV \
  glSubpixelPrecisionBiasNV \
  glConservativeRasterParameterfNV \
  glConservativeRasterParameteriNV \
  glCopyBufferSubDataNV \
  glCopyImageSubDataNV \
  glClearDepthdNV \
  glDepthBoundsdNV \
  glDepthRangedNV \
  glDrawBuffersNV \
  glDrawArraysInstancedNV \
  glDrawElementsInstancedNV \
  glDrawTextureNV \
  glDrawVkImageNV \
  glSignalVkFenceNV \
  glSignalVkSemaphoreNV \
  glWaitVkSemaphoreNV \
  glEvalMapsNV \
  glGetMapAttribParameterfvNV \
  glGetMapAttribParameterivNV \
  glGetMapControlPointsNV \
  glGetMapParameterfvNV \
  glGetMapParameterivNV \
  glMapControlPointsNV \
  glMapParameterfvNV \
  glMapParameterivNV \
  glGetMultisamplefvNV \
  glSampleMaskIndexedNV \
  glTexRenderbufferNV \
  glDeleteFencesNV \
  glFinishFenceNV \
  glGenFencesNV \
  glGetFenceivNV \
  glIsFenceNV \
  glSetFenceNV \
  glTestFenceNV \
  glFragmentCoverageColorNV \
  glGetProgramNamedParameterdvNV \
  glGetProgramNamedParameterfvNV \
  glProgramNamedParameter4dNV \
  glProgramNamedParameter4dvNV \
  glProgramNamedParameter4fNV \
  glProgramNamedParameter4fvNV \
  glBlitFramebufferNV \
  glRenderbufferStorageMultisampleNV \
  glRenderbufferStorageMultisampleCoverageNV \
  glProgramVertexLimitNV \
  glMulticastBarrierNV \
  glMulticastBlitFramebufferNV \
  glMulticastBufferSubDataNV \
  glMulticastCopyBufferSubDataNV \
  glMulticastCopyImageSubDataNV \
  glMulticastFramebufferSampleLocationsfvNV \
  glMulticastGetQueryObjecti64vNV \
  glMulticastGetQueryObjectivNV \
  glMulticastGetQueryObjectui64vNV \
  glMulticastGetQueryObjectuivNV \
  glMulticastWaitSyncNV \
  glRenderGpuMaskNV \
  glProgramEnvParameterI4iNV \
  glProgramEnvParameterI4ivNV \
  glProgramEnvParameterI4uiNV \
  glProgramEnvParameterI4uivNV \
  glProgramEnvParametersI4ivNV \
  glProgramEnvParametersI4uivNV \
  glProgramLocalParameterI4iNV \
  glProgramLocalParameterI4ivNV \
  glProgramLocalParameterI4uiNV \
  glProgramLocalParameterI4uivNV \
  glProgramLocalParametersI4ivNV \
  glProgramLocalParametersI4uivNV \
  glGetUniformi64vNV \
  glGetUniformui64vNV \
  glProgramUniform1i64NV \
  glProgramUniform1i64vNV \
  glProgramUniform1ui64NV \
  glProgramUniform1ui64vNV \
  glProgramUniform2i64NV \
  glProgramUniform2i64vNV \
  glProgramUniform2ui64NV \
  glProgramUniform2ui64vNV \
  glProgramUniform3i64NV \
  glProgramUniform3i64vNV \
  glProgramUniform3ui64NV \
  glProgramUniform3ui64vNV \
  glProgramUniform4i64NV \
  glProgramUniform4i64vNV \
  glProgramUniform4ui64NV \
  glProgramUniform4ui64vNV \
  glUniform1i64NV \
  glUniform1i64vNV \
  glUniform1ui64NV \
  glUniform1ui64vNV \
  glUniform2i64NV \
  glUniform2i64vNV \
  glUniform2ui64NV \
  glUniform2ui64vNV \
  glUniform3i64NV \
  glUniform3i64vNV \
  glUniform3ui64NV \
  glUniform3ui64vNV \
  glUniform4i64NV \
  glUniform4i64vNV \
  glUniform4ui64NV \
  glUniform4ui64vNV \
  glColor3hNV \
  glColor3hvNV \
  glColor4hNV \
  glColor4hvNV \
  glFogCoordhNV \
  glFogCoordhvNV \
  glMultiTexCoord1hNV \
  glMultiTexCoord1hvNV \
  glMultiTexCoord2hNV \
  glMultiTexCoord2hvNV \
  glMultiTexCoord3hNV \
  glMultiTexCoord3hvNV \
  glMultiTexCoord4hNV \
  glMultiTexCoord4hvNV \
  glNormal3hNV \
  glNormal3hvNV \
  glSecondaryColor3hNV \
  glSecondaryColor3hvNV \
  glTexCoord1hNV \
  glTexCoord1hvNV \
  glTexCoord2hNV \
  glTexCoord2hvNV \
  glTexCoord3hNV \
  glTexCoord3hvNV \
  glTexCoord4hNV \
  glTexCoord4hvNV \
  glVertex2hNV \
  glVertex2hvNV \
  glVertex3hNV \
  glVertex3hvNV \
  glVertex4hNV \
  glVertex4hvNV \
  glVertexAttrib1hNV \
  glVertexAttrib1hvNV \
  glVertexAttrib2hNV \
  glVertexAttrib2hvNV \
  glVertexAttrib3hNV \
  glVertexAttrib3hvNV \
  glVertexAttrib4hNV \
  glVertexAttrib4hvNV \
  glVertexAttribs1hvNV \
  glVertexAttribs2hvNV \
  glVertexAttribs3hvNV \
  glVertexAttribs4hvNV \
  glVertexWeighthNV \
  glVertexWeighthvNV \
  glVertexAttribDivisorNV \
  glGetInternalformatSampleivNV \
  glBufferAttachMemoryNV \
  glGetMemoryObjectDetachedResourcesuivNV \
  glNamedBufferAttachMemoryNV \
  glResetMemoryObjectParameterNV \
  glTexAttachMemoryNV \
  glTextureAttachMemoryNV \
  glDrawMeshTasksIndirectNV \
  glDrawMeshTasksNV \
  glMultiDrawMeshTasksIndirectCountNV \
  glMultiDrawMeshTasksIndirectNV \
  glUniformMatrix2x3fvNV \
  glUniformMatrix2x4fvNV \
  glUniformMatrix3x2fvNV \
  glUniformMatrix3x4fvNV \
  glUniformMatrix4x2fvNV \
  glUniformMatrix4x3fvNV \
  glBeginOcclusionQueryNV \
  glDeleteOcclusionQueriesNV \
  glEndOcclusionQueryNV \
  glGenOcclusionQueriesNV \
  glGetOcclusionQueryivNV \
  glGetOcclusionQueryuivNV \
  glIsOcclusionQueryNV \
  glProgramBufferParametersIivNV \
  glProgramBufferParametersIuivNV \
  glProgramBufferParametersfvNV \
  glCopyPathNV \
  glCoverFillPathInstancedNV \
  glCoverFillPathNV \
  glCoverStrokePathInstancedNV \
  glCoverStrokePathNV \
  glDeletePathsNV \
  glGenPathsNV \
  glGetPathColorGenfvNV \
  glGetPathColorGenivNV \
  glGetPathCommandsNV \
  glGetPathCoordsNV \
  glGetPathDashArrayNV \
  glGetPathLengthNV \
  glGetPathMetricRangeNV \
  glGetPathMetricsNV \
  glGetPathParameterfvNV \
  glGetPathParameterivNV \
  glGetPathSpacingNV \
  glGetPathTexGenfvNV \
  glGetPathTexGenivNV \
  glGetProgramResourcefvNV \
  glInterpolatePathsNV \
  glIsPathNV \
  glIsPointInFillPathNV \
  glIsPointInStrokePathNV \
  glMatrixLoad3x2fNV \
  glMatrixLoad3x3fNV \
  glMatrixLoadTranspose3x3fNV \
  glMatrixMult3x2fNV \
  glMatrixMult3x3fNV \
  glMatrixMultTranspose3x3fNV \
  glPathColorGenNV \
  glPathCommandsNV \
  glPathCoordsNV \
  glPathCoverDepthFuncNV \
  glPathDashArrayNV \
  glPathFogGenNV \
  glPathGlyphIndexArrayNV \
  glPathGlyphIndexRangeNV \
  glPathGlyphRangeNV \
  glPathGlyphsNV \
  glPathMemoryGlyphIndexArrayNV \
  glPathParameterfNV \
  glPathParameterfvNV \
  glPathParameteriNV \
  glPathParameterivNV \
  glPathStencilDepthOffsetNV \
  glPathStencilFuncNV \
  glPathStringNV \
  glPathSubCommandsNV \
  glPathSubCoordsNV \
  glPathTexGenNV \
  glPointAlongPathNV \
  glProgramPathFragmentInputGenNV \
  glStencilFillPathInstancedNV \
  glStencilFillPathNV \
  glStencilStrokePathInstancedNV \
  glStencilStrokePathNV \
  glStencilThenCoverFillPathInstancedNV \
  glStencilThenCoverFillPathNV \
  glStencilThenCoverStrokePathInstancedNV \
  glStencilThenCoverStrokePathNV \
  glTransformPathNV \
  glWeightPathsNV \
  glFlushPixelDataRangeNV \
  glPixelDataRangeNV \
  glPointParameteriNV \
  glPointParameterivNV \
  glPolygonModeNV \
  glGetVideoi64vNV \
  glGetVideoivNV \
  glGetVideoui64vNV \
  glGetVideouivNV \
  glPresentFrameDualFillNV \
  glPresentFrameKeyedNV \
  glPrimitiveRestartIndexNV \
  glPrimitiveRestartNV \
  glReadBufferNV \
  glCombinerInputNV \
  glCombinerOutputNV \
  glCombinerParameterfNV \
  glCombinerParameterfvNV \
  glCombinerParameteriNV \
  glCombinerParameterivNV \
  glFinalCombinerInputNV \
  glGetCombinerInputParameterfvNV \
  glGetCombinerInputParameterivNV \
  glGetCombinerOutputParameterfvNV \
  glGetCombinerOutputParameterivNV \
  glGetFinalCombinerInputParameterfvNV \
  glGetFinalCombinerInputParameterivNV \
  glCombinerStageParameterfvNV \
  glGetCombinerStageParameterfvNV \
  glFramebufferSampleLocationsfvNV \
  glNamedFramebufferSampleLocationsfvNV \
  glResolveDepthValuesNV \
  glScissorExclusiveArrayvNV \
  glScissorExclusiveNV \
  glGetBufferParameterui64vNV \
  glGetIntegerui64vNV \
  glGetNamedBufferParameterui64vNV \
  glIsBufferResidentNV \
  glIsNamedBufferResidentNV \
  glMakeBufferNonResidentNV \
  glMakeBufferResidentNV \
  glMakeNamedBufferNonResidentNV \
  glMakeNamedBufferResidentNV \
  glProgramUniformui64NV \
  glProgramUniformui64vNV \
  glUniformui64NV \
  glUniformui64vNV \
  glBindShadingRateImageNV \
  glGetShadingRateImagePaletteNV \
  glGetShadingRateSampleLocationivNV \
  glShadingRateImageBarrierNV \
  glShadingRateImagePaletteNV \
  glShadingRateSampleOrderCustomNV \
  glCompressedTexImage3DNV \
  glCompressedTexSubImage3DNV \
  glCopyTexSubImage3DNV \
  glFramebufferTextureLayerNV \
  glTexImage3DNV \
  glTexSubImage3DNV \
  glTextureBarrierNV \
  glTexImage2DMultisampleCoverageNV \
  glTexImage3DMultisampleCoverageNV \
  glTextureImage2DMultisampleCoverageNV \
  glTextureImage2DMultisampleNV \
  glTextureImage3DMultisampleCoverageNV \
  glTextureImage3DMultisampleNV \
  glActiveVaryingNV \
  glBeginTransformFeedbackNV \
  glBindBufferBaseNV \
  glBindBufferOffsetNV \
  glBindBufferRangeNV \
  glEndTransformFeedbackNV \
  glGetActiveVaryingNV \
  glGetTransformFeedbackVaryingNV \
  glGetVaryingLocationNV \
  glTransformFeedbackAttribsNV \
  glTransformFeedbackVaryingsNV \
  glBindTransformFeedbackNV \
  glDeleteTransformFeedbacksNV \
  glDrawTransformFeedbackNV \
  glGenTransformFeedbacksNV \
  glIsTransformFeedbackNV \
  glPauseTransformFeedbackNV \
  glResumeTransformFeedbackNV \
  glVDPAUFiniNV \
  glVDPAUGetSurfaceivNV \
  glVDPAUInitNV \
  glVDPAUIsSurfaceNV \
  glVDPAUMapSurfacesNV \
  glVDPAURegisterOutputSurfaceNV \
  glVDPAURegisterVideoSurfaceNV \
  glVDPAUSurfaceAccessNV \
  glVDPAUUnmapSurfacesNV \
  glVDPAUUnregisterSurfaceNV \
  glVDPAURegisterVideoSurfaceWithPictureStructureNV \
  glFlushVertexArrayRangeNV \
  glVertexArrayRangeNV \
  glGetVertexAttribLi64vNV \
  glGetVertexAttribLui64vNV \
  glVertexAttribL1i64NV \
  glVertexAttribL1i64vNV \
  glVertexAttribL1ui64NV \
  glVertexAttribL1ui64vNV \
  glVertexAttribL2i64NV \
  glVertexAttribL2i64vNV \
  glVertexAttribL2ui64NV \
  glVertexAttribL2ui64vNV \
  glVertexAttribL3i64NV \
  glVertexAttribL3i64vNV \
  glVertexAttribL3ui64NV \
  glVertexAttribL3ui64vNV \
  glVertexAttribL4i64NV \
  glVertexAttribL4i64vNV \
  glVertexAttribL4ui64NV \
  glVertexAttribL4ui64vNV \
  glVertexAttribLFormatNV \
  glBufferAddressRangeNV \
  glColorFormatNV \
  glEdgeFlagFormatNV \
  glFogCoordFormatNV \
  glGetIntegerui64i_vNV \
  glIndexFormatNV \
  glNormalFormatNV \
  glSecondaryColorFormatNV \
  glTexCoordFormatNV \
  glVertexAttribFormatNV \
  glVertexAttribIFormatNV \
  glVertexFormatNV \
  glAreProgramsResidentNV \
  glBindProgramNV \
  glDeleteProgramsNV \
  glExecuteProgramNV \
  glGenProgramsNV \
  glGetProgramParameterdvNV \
  glGetProgramParameterfvNV \
  glGetProgramStringNV \
  glGetProgramivNV \
  glGetTrackMatrixivNV \
  glGetVertexAttribPointervNV \
  glGetVertexAttribdvNV \
  glGetVertexAttribfvNV \
  glGetVertexAttribivNV \
  glIsProgramNV \
  glLoadProgramNV \
  glProgramParameter4dNV \
  glProgramParameter4dvNV \
  glProgramParameter4fNV \
  glProgramParameter4fvNV \
  glProgramParameters4dvNV \
  glProgramParameters4fvNV \
  glRequestResidentProgramsNV \
  glTrackMatrixNV \
  glVertexAttrib1dNV \
  glVertexAttrib1dvNV \
  glVertexAttrib1fNV \
  glVertexAttrib1fvNV \
  glVertexAttrib1sNV \
  glVertexAttrib1svNV \
  glVertexAttrib2dNV \
  glVertexAttrib2dvNV \
  glVertexAttrib2fNV \
  glVertexAttrib2fvNV \
  glVertexAttrib2sNV \
  glVertexAttrib2svNV \
  glVertexAttrib3dNV \
  glVertexAttrib3dvNV \
  glVertexAttrib3fNV \
  glVertexAttrib3fvNV \
  glVertexAttrib3sNV \
  glVertexAttrib3svNV \
  glVertexAttrib4dNV \
  glVertexAttrib4dvNV \
  glVertexAttrib4fNV \
  glVertexAttrib4fvNV \
  glVertexAttrib4sNV \
  glVertexAttrib4svNV \
  glVertexAttrib4ubNV \
  glVertexAttrib4ubvNV \
  glVertexAttribPointerNV \
  glVertexAttribs1dvNV \
  glVertexAttribs1fvNV \
  glVertexAttribs1svNV \
  glVertexAttribs2dvNV \
  glVertexAttribs2fvNV \
  glVertexAttribs2svNV \
  glVertexAttribs3dvNV \
  glVertexAttribs3fvNV \
  glVertexAttribs3svNV \
  glVertexAttribs4dvNV \
  glVertexAttribs4fvNV \
  glVertexAttribs4svNV \
  glVertexAttribs4ubvNV \
  glBeginVideoCaptureNV \
  glBindVideoCaptureStreamBufferNV \
  glBindVideoCaptureStreamTextureNV \
  glEndVideoCaptureNV \
  glGetVideoCaptureStreamdvNV \
  glGetVideoCaptureStreamfvNV \
  glGetVideoCaptureStreamivNV \
  glGetVideoCaptureivNV \
  glVideoCaptureNV \
  glVideoCaptureStreamParameterdvNV \
  glVideoCaptureStreamParameterfvNV \
  glVideoCaptureStreamParameterivNV \
  glDepthRangeArrayfvNV \
  glDepthRangeIndexedfNV \
  glDisableiNV \
  glEnableiNV \
  glGetFloati_vNV \
  glIsEnablediNV \
  glScissorArrayvNV \
  glScissorIndexedNV \
  glScissorIndexedvNV \
  glViewportArrayvNV \
  glViewportIndexedfNV \
  glViewportIndexedfvNV \
  glViewportSwizzleNV \
  glEGLImageTargetRenderbufferStorageOES \
  glEGLImageTargetTexture2DOES \
  glBlendEquationSeparateOES \
  glBlendFuncSeparateOES \
  glBlendEquationOES \
  glCopyImageSubDataOES \
  glBlendEquationSeparateiOES \
  glBlendEquationiOES \
  glBlendFuncSeparateiOES \
  glBlendFunciOES \
  glColorMaskiOES \
  glDisableiOES \
  glEnableiOES \
  glIsEnablediOES \
  glBindFramebufferOES \
  glBindRenderbufferOES \
  glCheckFramebufferStatusOES \
  glDeleteFramebuffersOES \
  glDeleteRenderbuffersOES \
  glFramebufferRenderbufferOES \
  glFramebufferTexture2DOES \
  glGenFramebuffersOES \
  glGenRenderbuffersOES \
  glGenerateMipmapOES \
  glGetFramebufferAttachmentParameterivOES \
  glGetRenderbufferParameterivOES \
  glIsFramebufferOES \
  glIsRenderbufferOES \
  glRenderbufferStorageOES \
  glGetProgramBinaryOES \
  glProgramBinaryOES \
  glGetBufferPointervOES \
  glMapBufferOES \
  glUnmapBufferOES \
  glCurrentPaletteMatrixOES \
  glMatrixIndexPointerOES \
  glWeightPointerOES \
  glMinSampleShadingOES \
  glClearDepthfOES \
  glClipPlanefOES \
  glDepthRangefOES \
  glFrustumfOES \
  glGetClipPlanefOES \
  glOrthofOES \
  glCompressedTexImage3DOES \
  glCompressedTexSubImage3DOES \
  glCopyTexSubImage3DOES \
  glFramebufferTexture3DOES \
  glTexImage3DOES \
  glTexSubImage3DOES \
  glGetSamplerParameterIivOES \
  glGetSamplerParameterIuivOES \
  glGetTexParameterIivOES \
  glGetTexParameterIuivOES \
  glSamplerParameterIivOES \
  glSamplerParameterIuivOES \
  glTexParameterIivOES \
  glTexParameterIuivOES \
  glTexBufferOES \
  glTexBufferRangeOES \
  glGetTexGenfvOES \
  glGetTexGenivOES \
  glGetTexGenxvOES \
  glTexGenfOES \
  glTexGenfvOES \
  glTexGeniOES \
  glTexGenivOES \
  glTexGenxOES \
  glTexGenxvOES \
  glTexStorage3DMultisampleOES \
  glTextureViewOES \
  glBindVertexArrayOES \
  glDeleteVertexArraysOES \
  glGenVertexArraysOES \
  glIsVertexArrayOES \
  glFramebufferTextureMultiviewOVR \
  glNamedFramebufferTextureMultiviewOVR \
  glFramebufferTextureMultisampleMultiviewOVR \
  glAlphaFuncQCOM \
  glDisableDriverControlQCOM \
  glEnableDriverControlQCOM \
  glGetDriverControlStringQCOM \
  glGetDriverControlsQCOM \
  glExtGetBufferPointervQCOM \
  glExtGetBuffersQCOM \
  glExtGetFramebuffersQCOM \
  glExtGetRenderbuffersQCOM \
  glExtGetTexLevelParameterivQCOM \
  glExtGetTexSubImageQCOM \
  glExtGetTexturesQCOM \
  glExtTexObjectStateOverrideiQCOM \
  glExtGetProgramBinarySourceQCOM \
  glExtGetProgramsQCOM \
  glExtGetShadersQCOM \
  glExtIsProgramBinaryQCOM \
  glFramebufferFoveationConfigQCOM \
  glFramebufferFoveationParametersQCOM \
  glFramebufferFetchBarrierQCOM \
  glTextureFoveationParametersQCOM \
  glEndTilingQCOM \
  glStartTilingQCOM \
  glAlphaFuncx \
  glClearColorx \
  glClearDepthx \
  glColor4x \
  glDepthRangex \
  glFogx \
  glFogxv \
  glFrustumf \
  glFrustumx \
  glLightModelx \
  glLightModelxv \
  glLightx \
  glLightxv \
  glLineWidthx \
  glLoadMatrixx \
  glMaterialx \
  glMaterialxv \
  glMultMatrixx \
  glMultiTexCoord4x \
  glNormal3x \
  glOrthof \
  glOrthox \
  glPointSizex \
  glPolygonOffsetx \
  glRotatex \
  glSampleCoveragex \
  glScalex \
  glTexEnvx \
  glTexEnvxv \
  glTexParameterx \
  glTranslatex \
  glClipPlanef \
  glClipPlanex \
  glGetClipPlanef \
  glGetClipPlanex \
  glGetFixedv \
  glGetLightxv \
  glGetMaterialxv \
  glGetTexEnvxv \
  glGetTexParameterxv \
  glPointParameterx \
  glPointParameterxv \
  glPointSizePointerOES \
  glTexParameterxv \
  glErrorStringREGAL \
  glGetExtensionREGAL \
  glIsSupportedREGAL \
  glLogMessageCallbackREGAL \
  glDetailTexFuncSGIS \
  glGetDetailTexFuncSGIS \
  glFogFuncSGIS \
  glGetFogFuncSGIS \
  glSampleMaskSGIS \
  glSamplePatternSGIS \
  glInterleavedTextureCoordSetsSGIS \
  glSelectTextureCoordSetSGIS \
  glSelectTextureSGIS \
  glSelectTextureTransformSGIS \
  glMultisampleSubRectPosSGIS \
  glGetSharpenTexFuncSGIS \
  glSharpenTexFuncSGIS \
  glTexImage4DSGIS \
  glTexSubImage4DSGIS \
  glGetTexFilterFuncSGIS \
  glTexFilterFuncSGIS \
  glAsyncMarkerSGIX \
  glDeleteAsyncMarkersSGIX \
  glFinishAsyncSGIX \
  glGenAsyncMarkersSGIX \
  glIsAsyncMarkerSGIX \
  glPollAsyncSGIX \
  glAddressSpace \
  glDataPipe \
  glFlushRasterSGIX \
  glFogLayersSGIX \
  glGetFogLayersSGIX \
  glTextureFogSGIX \
  glFragmentColorMaterialSGIX \
  glFragmentLightModelfSGIX \
  glFragmentLightModelfvSGIX \
  glFragmentLightModeliSGIX \
  glFragmentLightModelivSGIX \
  glFragmentLightfSGIX \
  glFragmentLightfvSGIX \
  glFragmentLightiSGIX \
  glFragmentLightivSGIX \
  glFragmentMaterialfSGIX \
  glFragmentMaterialfvSGIX \
  glFragmentMaterialiSGIX \
  glFragmentMaterialivSGIX \
  glGetFragmentLightfvSGIX \
  glGetFragmentLightivSGIX \
  glGetFragmentMaterialfvSGIX \
  glGetFragmentMaterialivSGIX \
  glFrameZoomSGIX \
  glIglooInterfaceSGIX \
  glAllocMPEGPredictorsSGIX \
  glDeleteMPEGPredictorsSGIX \
  glGenMPEGPredictorsSGIX \
  glGetMPEGParameterfvSGIX \
  glGetMPEGParameterivSGIX \
  glGetMPEGPredictorSGIX \
  glGetMPEGQuantTableubv \
  glIsMPEGPredictorSGIX \
  glMPEGPredictorSGIX \
  glMPEGQuantTableubv \
  glSwapMPEGPredictorsSGIX \
  glGetNonlinLightfvSGIX \
  glGetNonlinMaterialfvSGIX \
  glNonlinLightfvSGIX \
  glNonlinMaterialfvSGIX \
  glPixelTexGenSGIX \
  glDeformSGIX \
  glLoadIdentityDeformationMapSGIX \
  glMeshBreadthSGIX \
  glMeshStrideSGIX \
  glReferencePlaneSGIX \
  glSpriteParameterfSGIX \
  glSpriteParameterfvSGIX \
  glSpriteParameteriSGIX \
  glSpriteParameterivSGIX \
  glTagSampleBufferSGIX \
  glGetVectorOperationSGIX \
  glVectorOperationSGIX \
  glAreVertexArraysResidentSGIX \
  glBindVertexArraySGIX \
  glDeleteVertexArraysSGIX \
  glGenVertexArraysSGIX \
  glIsVertexArraySGIX \
  glPrioritizeVertexArraysSGIX \
  glColorTableParameterfvSGI \
  glColorTableParameterivSGI \
  glColorTableSGI \
  glCopyColorTableSGI \
  glGetColorTableParameterfvSGI \
  glGetColorTableParameterivSGI \
  glGetColorTableSGI \
  glGetPixelTransformParameterfvSGI \
  glGetPixelTransformParameterivSGI \
  glPixelTransformParameterfSGI \
  glPixelTransformParameterfvSGI \
  glPixelTransformParameteriSGI \
  glPixelTransformParameterivSGI \
  glPixelTransformSGI \
  glFinishTextureSUNX \
  glGlobalAlphaFactorbSUN \
  glGlobalAlphaFactordSUN \
  glGlobalAlphaFactorfSUN \
  glGlobalAlphaFactoriSUN \
  glGlobalAlphaFactorsSUN \
  glGlobalAlphaFactorubSUN \
  glGlobalAlphaFactoruiSUN \
  glGlobalAlphaFactorusSUN \
  glReadVideoPixelsSUN \
  glReplacementCodePointerSUN \
  glReplacementCodeubSUN \
  glReplacementCodeubvSUN \
  glReplacementCodeuiSUN \
  glReplacementCodeuivSUN \
  glReplacementCodeusSUN \
  glReplacementCodeusvSUN \
  glColor3fVertex3fSUN \
  glColor3fVertex3fvSUN \
  glColor4fNormal3fVertex3fSUN \
  glColor4fNormal3fVertex3fvSUN \
  glColor4ubVertex2fSUN \
  glColor4ubVertex2fvSUN \
  glColor4ubVertex3fSUN \
  glColor4ubVertex3fvSUN \
  glNormal3fVertex3fSUN \
  glNormal3fVertex3fvSUN \
  glReplacementCodeuiColor3fVertex3fSUN \
  glReplacementCodeuiColor3fVertex3fvSUN \
  glReplacementCodeuiColor4fNormal3fVertex3fSUN \
  glReplacementCodeuiColor4fNormal3fVertex3fvSUN \
  glReplacementCodeuiColor4ubVertex3fSUN \
  glReplacementCodeuiColor4ubVertex3fvSUN \
  glReplacementCodeuiNormal3fVertex3fSUN \
  glReplacementCodeuiNormal3fVertex3fvSUN \
  glReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fSUN \
  glReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fvSUN \
  glReplacementCodeuiTexCoord2fNormal3fVertex3fSUN \
  glReplacementCodeuiTexCoord2fNormal3fVertex3fvSUN \
  glReplacementCodeuiTexCoord2fVertex3fSUN \
  glReplacementCodeuiTexCoord2fVertex3fvSUN \
  glReplacementCodeuiVertex3fSUN \
  glReplacementCodeuiVertex3fvSUN \
  glTexCoord2fColor3fVertex3fSUN \
  glTexCoord2fColor3fVertex3fvSUN \
  glTexCoord2fColor4fNormal3fVertex3fSUN \
  glTexCoord2fColor4fNormal3fVertex3fvSUN \
  glTexCoord2fColor4ubVertex3fSUN \
  glTexCoord2fColor4ubVertex3fvSUN \
  glTexCoord2fNormal3fVertex3fSUN \
  glTexCoord2fNormal3fVertex3fvSUN \
  glTexCoord2fVertex3fSUN \
  glTexCoord2fVertex3fvSUN \
  glTexCoord4fColor4fNormal3fVertex4fSUN \
  glTexCoord4fColor4fNormal3fVertex4fvSUN \
  glTexCoord4fVertex4fSUN \
  glTexCoord4fVertex4fvSUN \
  glAddSwapHintRectWIN \
]

# List of the C-signatures of all wrapped OpenGL functions.
set ::__tcl3dOglFuncSignatureList [list \
  "void  glAccum (GLenum op, GLfloat value)" \
  "void  glAlphaFunc (GLenum func, GLclampf ref)" \
  "GLboolean  glAreTexturesResident (GLsizei n, const GLuint *textures, GLboolean *residences)" \
  "void  glArrayElement (GLint i)" \
  "void  glBegin (GLenum mode)" \
  "void  glBindTexture (GLenum target, GLuint texture)" \
  "void  glBitmap (GLsizei width, GLsizei height, GLfloat xorig, GLfloat yorig, GLfloat xmove, GLfloat ymove, const GLubyte *bitmap)" \
  "void  glBlendFunc (GLenum sfactor, GLenum dfactor)" \
  "void  glCallList (GLuint list)" \
  "void  glCallLists (GLsizei n, GLenum type, const void *lists)" \
  "void  glClear (GLbitfield mask)" \
  "void  glClearAccum (GLfloat red, GLfloat green, GLfloat blue, GLfloat alpha)" \
  "void  glClearColor (GLclampf red, GLclampf green, GLclampf blue, GLclampf alpha)" \
  "void  glClearDepth (GLclampd depth)" \
  "void  glClearIndex (GLfloat c)" \
  "void  glClearStencil (GLint s)" \
  "void  glClipPlane (GLenum plane, const GLdouble *equation)" \
  "void  glColor3b (GLbyte red, GLbyte green, GLbyte blue)" \
  "void  glColor3bv (const GLbyte *v)" \
  "void  glColor3d (GLdouble red, GLdouble green, GLdouble blue)" \
  "void  glColor3dv (const GLdouble *v)" \
  "void  glColor3f (GLfloat red, GLfloat green, GLfloat blue)" \
  "void  glColor3fv (const GLfloat *v)" \
  "void  glColor3i (GLint red, GLint green, GLint blue)" \
  "void  glColor3iv (const GLint *v)" \
  "void  glColor3s (GLshort red, GLshort green, GLshort blue)" \
  "void  glColor3sv (const GLshort *v)" \
  "void  glColor3ub (GLubyte red, GLubyte green, GLubyte blue)" \
  "void  glColor3ubv (const GLubyte *v)" \
  "void  glColor3ui (GLuint red, GLuint green, GLuint blue)" \
  "void  glColor3uiv (const GLuint *v)" \
  "void  glColor3us (GLushort red, GLushort green, GLushort blue)" \
  "void  glColor3usv (const GLushort *v)" \
  "void  glColor4b (GLbyte red, GLbyte green, GLbyte blue, GLbyte alpha)" \
  "void  glColor4bv (const GLbyte *v)" \
  "void  glColor4d (GLdouble red, GLdouble green, GLdouble blue, GLdouble alpha)" \
  "void  glColor4dv (const GLdouble *v)" \
  "void  glColor4f (GLfloat red, GLfloat green, GLfloat blue, GLfloat alpha)" \
  "void  glColor4fv (const GLfloat *v)" \
  "void  glColor4i (GLint red, GLint green, GLint blue, GLint alpha)" \
  "void  glColor4iv (const GLint *v)" \
  "void  glColor4s (GLshort red, GLshort green, GLshort blue, GLshort alpha)" \
  "void  glColor4sv (const GLshort *v)" \
  "void  glColor4ub (GLubyte red, GLubyte green, GLubyte blue, GLubyte alpha)" \
  "void  glColor4ubv (const GLubyte *v)" \
  "void  glColor4ui (GLuint red, GLuint green, GLuint blue, GLuint alpha)" \
  "void  glColor4uiv (const GLuint *v)" \
  "void  glColor4us (GLushort red, GLushort green, GLushort blue, GLushort alpha)" \
  "void  glColor4usv (const GLushort *v)" \
  "void  glColorMask (GLboolean red, GLboolean green, GLboolean blue, GLboolean alpha)" \
  "void  glColorMaterial (GLenum face, GLenum mode)" \
  "void  glColorPointer (GLint size, GLenum type, GLsizei stride, const void *pointer)" \
  "void  glCopyPixels (GLint x, GLint y, GLsizei width, GLsizei height, GLenum type)" \
  "void  glCopyTexImage1D (GLenum target, GLint level, GLenum internalFormat, GLint x, GLint y, GLsizei width, GLint border)" \
  "void  glCopyTexImage2D (GLenum target, GLint level, GLenum internalFormat, GLint x, GLint y, GLsizei width, GLsizei height, GLint border)" \
  "void  glCopyTexSubImage1D (GLenum target, GLint level, GLint xoffset, GLint x, GLint y, GLsizei width)" \
  "void  glCopyTexSubImage2D (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint x, GLint y, GLsizei width, GLsizei height)" \
  "void  glCullFace (GLenum mode)" \
  "void  glDeleteLists (GLuint list, GLsizei range)" \
  "void  glDeleteTextures (GLsizei n, const GLuint *textures)" \
  "void  glDepthFunc (GLenum func)" \
  "void  glDepthMask (GLboolean flag)" \
  "void  glDepthRange (GLclampd zNear, GLclampd zFar)" \
  "void  glDisable (GLenum cap)" \
  "void  glDisableClientState (GLenum array)" \
  "void  glDrawArrays (GLenum mode, GLint first, GLsizei count)" \
  "void  glDrawBuffer (GLenum mode)" \
  "void  glDrawElements (GLenum mode, GLsizei count, GLenum type, const void *indices)" \
  "void  glDrawPixels (GLsizei width, GLsizei height, GLenum format, GLenum type, const void *pixels)" \
  "void  glEdgeFlag (GLboolean flag)" \
  "void  glEdgeFlagPointer (GLsizei stride, const void *pointer)" \
  "void  glEdgeFlagv (const GLboolean *flag)" \
  "void  glEnable (GLenum cap)" \
  "void  glEnableClientState (GLenum array)" \
  "void  glEnd (void)" \
  "void  glEndList (void)" \
  "void  glEvalCoord1d (GLdouble u)" \
  "void  glEvalCoord1dv (const GLdouble *u)" \
  "void  glEvalCoord1f (GLfloat u)" \
  "void  glEvalCoord1fv (const GLfloat *u)" \
  "void  glEvalCoord2d (GLdouble u, GLdouble v)" \
  "void  glEvalCoord2dv (const GLdouble *u)" \
  "void  glEvalCoord2f (GLfloat u, GLfloat v)" \
  "void  glEvalCoord2fv (const GLfloat *u)" \
  "void  glEvalMesh1 (GLenum mode, GLint i1, GLint i2)" \
  "void  glEvalMesh2 (GLenum mode, GLint i1, GLint i2, GLint j1, GLint j2)" \
  "void  glEvalPoint1 (GLint i)" \
  "void  glEvalPoint2 (GLint i, GLint j)" \
  "void  glFeedbackBuffer (GLsizei size, GLenum type, GLfloat *buffer)" \
  "void  glFinish (void)" \
  "void  glFlush (void)" \
  "void  glFogf (GLenum pname, GLfloat param)" \
  "void  glFogfv (GLenum pname, const GLfloat *params)" \
  "void  glFogi (GLenum pname, GLint param)" \
  "void  glFogiv (GLenum pname, const GLint *params)" \
  "void  glFrontFace (GLenum mode)" \
  "void  glFrustum (GLdouble left, GLdouble right, GLdouble bottom, GLdouble top, GLdouble zNear, GLdouble zFar)" \
  "GLuint  glGenLists (GLsizei range)" \
  "void  glGenTextures (GLsizei n, GLuint *textures)" \
  "void  glGetBooleanv (GLenum pname, GLboolean *params)" \
  "void  glGetClipPlane (GLenum plane, GLdouble *equation)" \
  "void  glGetDoublev (GLenum pname, GLdouble *params)" \
  "GLenum  glGetError (void)" \
  "void  glGetFloatv (GLenum pname, GLfloat *params)" \
  "void  glGetIntegerv (GLenum pname, GLint *params)" \
  "void  glGetLightfv (GLenum light, GLenum pname, GLfloat *params)" \
  "void  glGetLightiv (GLenum light, GLenum pname, GLint *params)" \
  "void  glGetMapdv (GLenum target, GLenum query, GLdouble *v)" \
  "void  glGetMapfv (GLenum target, GLenum query, GLfloat *v)" \
  "void  glGetMapiv (GLenum target, GLenum query, GLint *v)" \
  "void  glGetMaterialfv (GLenum face, GLenum pname, GLfloat *params)" \
  "void  glGetMaterialiv (GLenum face, GLenum pname, GLint *params)" \
  "void  glGetPixelMapfv (GLenum map, GLfloat *values)" \
  "void  glGetPixelMapuiv (GLenum map, GLuint *values)" \
  "void  glGetPixelMapusv (GLenum map, GLushort *values)" \
  "void  glGetPointerv (GLenum pname, void* *params)" \
  "void  glGetPolygonStipple (GLubyte *mask)" \
  "const GLubyte *  glGetString (GLenum name)" \
  "void  glGetTexEnvfv (GLenum target, GLenum pname, GLfloat *params)" \
  "void  glGetTexEnviv (GLenum target, GLenum pname, GLint *params)" \
  "void  glGetTexGendv (GLenum coord, GLenum pname, GLdouble *params)" \
  "void  glGetTexGenfv (GLenum coord, GLenum pname, GLfloat *params)" \
  "void  glGetTexGeniv (GLenum coord, GLenum pname, GLint *params)" \
  "void  glGetTexImage (GLenum target, GLint level, GLenum format, GLenum type, void *pixels)" \
  "void  glGetTexLevelParameterfv (GLenum target, GLint level, GLenum pname, GLfloat *params)" \
  "void  glGetTexLevelParameteriv (GLenum target, GLint level, GLenum pname, GLint *params)" \
  "void  glGetTexParameterfv (GLenum target, GLenum pname, GLfloat *params)" \
  "void  glGetTexParameteriv (GLenum target, GLenum pname, GLint *params)" \
  "void  glHint (GLenum target, GLenum mode)" \
  "void  glIndexMask (GLuint mask)" \
  "void  glIndexPointer (GLenum type, GLsizei stride, const void *pointer)" \
  "void  glIndexd (GLdouble c)" \
  "void  glIndexdv (const GLdouble *c)" \
  "void  glIndexf (GLfloat c)" \
  "void  glIndexfv (const GLfloat *c)" \
  "void  glIndexi (GLint c)" \
  "void  glIndexiv (const GLint *c)" \
  "void  glIndexs (GLshort c)" \
  "void  glIndexsv (const GLshort *c)" \
  "void  glIndexub (GLubyte c)" \
  "void  glIndexubv (const GLubyte *c)" \
  "void  glInitNames (void)" \
  "void  glInterleavedArrays (GLenum format, GLsizei stride, const void *pointer)" \
  "GLboolean  glIsEnabled (GLenum cap)" \
  "GLboolean  glIsList (GLuint list)" \
  "GLboolean  glIsTexture (GLuint texture)" \
  "void  glLightModelf (GLenum pname, GLfloat param)" \
  "void  glLightModelfv (GLenum pname, const GLfloat *params)" \
  "void  glLightModeli (GLenum pname, GLint param)" \
  "void  glLightModeliv (GLenum pname, const GLint *params)" \
  "void  glLightf (GLenum light, GLenum pname, GLfloat param)" \
  "void  glLightfv (GLenum light, GLenum pname, const GLfloat *params)" \
  "void  glLighti (GLenum light, GLenum pname, GLint param)" \
  "void  glLightiv (GLenum light, GLenum pname, const GLint *params)" \
  "void  glLineStipple (GLint factor, GLushort pattern)" \
  "void  glLineWidth (GLfloat width)" \
  "void  glListBase (GLuint base)" \
  "void  glLoadIdentity (void)" \
  "void  glLoadMatrixd (const GLdouble *m)" \
  "void  glLoadMatrixf (const GLfloat *m)" \
  "void  glLoadName (GLuint name)" \
  "void  glLogicOp (GLenum opcode)" \
  "void  glMap1d (GLenum target, GLdouble u1, GLdouble u2, GLint stride, GLint order, const GLdouble *points)" \
  "void  glMap1f (GLenum target, GLfloat u1, GLfloat u2, GLint stride, GLint order, const GLfloat *points)" \
  "void  glMap2d (GLenum target, GLdouble u1, GLdouble u2, GLint ustride, GLint uorder, GLdouble v1, GLdouble v2, GLint vstride, GLint vorder, const GLdouble *points)" \
  "void  glMap2f (GLenum target, GLfloat u1, GLfloat u2, GLint ustride, GLint uorder, GLfloat v1, GLfloat v2, GLint vstride, GLint vorder, const GLfloat *points)" \
  "void  glMapGrid1d (GLint un, GLdouble u1, GLdouble u2)" \
  "void  glMapGrid1f (GLint un, GLfloat u1, GLfloat u2)" \
  "void  glMapGrid2d (GLint un, GLdouble u1, GLdouble u2, GLint vn, GLdouble v1, GLdouble v2)" \
  "void  glMapGrid2f (GLint un, GLfloat u1, GLfloat u2, GLint vn, GLfloat v1, GLfloat v2)" \
  "void  glMaterialf (GLenum face, GLenum pname, GLfloat param)" \
  "void  glMaterialfv (GLenum face, GLenum pname, const GLfloat *params)" \
  "void  glMateriali (GLenum face, GLenum pname, GLint param)" \
  "void  glMaterialiv (GLenum face, GLenum pname, const GLint *params)" \
  "void  glMatrixMode (GLenum mode)" \
  "void  glMultMatrixd (const GLdouble *m)" \
  "void  glMultMatrixf (const GLfloat *m)" \
  "void  glNewList (GLuint list, GLenum mode)" \
  "void  glNormal3b (GLbyte nx, GLbyte ny, GLbyte nz)" \
  "void  glNormal3bv (const GLbyte *v)" \
  "void  glNormal3d (GLdouble nx, GLdouble ny, GLdouble nz)" \
  "void  glNormal3dv (const GLdouble *v)" \
  "void  glNormal3f (GLfloat nx, GLfloat ny, GLfloat nz)" \
  "void  glNormal3fv (const GLfloat *v)" \
  "void  glNormal3i (GLint nx, GLint ny, GLint nz)" \
  "void  glNormal3iv (const GLint *v)" \
  "void  glNormal3s (GLshort nx, GLshort ny, GLshort nz)" \
  "void  glNormal3sv (const GLshort *v)" \
  "void  glNormalPointer (GLenum type, GLsizei stride, const void *pointer)" \
  "void  glOrtho (GLdouble left, GLdouble right, GLdouble bottom, GLdouble top, GLdouble zNear, GLdouble zFar)" \
  "void  glPassThrough (GLfloat token)" \
  "void  glPixelMapfv (GLenum map, GLsizei mapsize, const GLfloat *values)" \
  "void  glPixelMapuiv (GLenum map, GLsizei mapsize, const GLuint *values)" \
  "void  glPixelMapusv (GLenum map, GLsizei mapsize, const GLushort *values)" \
  "void  glPixelStoref (GLenum pname, GLfloat param)" \
  "void  glPixelStorei (GLenum pname, GLint param)" \
  "void  glPixelTransferf (GLenum pname, GLfloat param)" \
  "void  glPixelTransferi (GLenum pname, GLint param)" \
  "void  glPixelZoom (GLfloat xfactor, GLfloat yfactor)" \
  "void  glPointSize (GLfloat size)" \
  "void  glPolygonMode (GLenum face, GLenum mode)" \
  "void  glPolygonOffset (GLfloat factor, GLfloat units)" \
  "void  glPolygonStipple (const GLubyte *mask)" \
  "void  glPopAttrib (void)" \
  "void  glPopClientAttrib (void)" \
  "void  glPopMatrix (void)" \
  "void  glPopName (void)" \
  "void  glPrioritizeTextures (GLsizei n, const GLuint *textures, const GLclampf *priorities)" \
  "void  glPushAttrib (GLbitfield mask)" \
  "void  glPushClientAttrib (GLbitfield mask)" \
  "void  glPushMatrix (void)" \
  "void  glPushName (GLuint name)" \
  "void  glRasterPos2d (GLdouble x, GLdouble y)" \
  "void  glRasterPos2dv (const GLdouble *v)" \
  "void  glRasterPos2f (GLfloat x, GLfloat y)" \
  "void  glRasterPos2fv (const GLfloat *v)" \
  "void  glRasterPos2i (GLint x, GLint y)" \
  "void  glRasterPos2iv (const GLint *v)" \
  "void  glRasterPos2s (GLshort x, GLshort y)" \
  "void  glRasterPos2sv (const GLshort *v)" \
  "void  glRasterPos3d (GLdouble x, GLdouble y, GLdouble z)" \
  "void  glRasterPos3dv (const GLdouble *v)" \
  "void  glRasterPos3f (GLfloat x, GLfloat y, GLfloat z)" \
  "void  glRasterPos3fv (const GLfloat *v)" \
  "void  glRasterPos3i (GLint x, GLint y, GLint z)" \
  "void  glRasterPos3iv (const GLint *v)" \
  "void  glRasterPos3s (GLshort x, GLshort y, GLshort z)" \
  "void  glRasterPos3sv (const GLshort *v)" \
  "void  glRasterPos4d (GLdouble x, GLdouble y, GLdouble z, GLdouble w)" \
  "void  glRasterPos4dv (const GLdouble *v)" \
  "void  glRasterPos4f (GLfloat x, GLfloat y, GLfloat z, GLfloat w)" \
  "void  glRasterPos4fv (const GLfloat *v)" \
  "void  glRasterPos4i (GLint x, GLint y, GLint z, GLint w)" \
  "void  glRasterPos4iv (const GLint *v)" \
  "void  glRasterPos4s (GLshort x, GLshort y, GLshort z, GLshort w)" \
  "void  glRasterPos4sv (const GLshort *v)" \
  "void  glReadBuffer (GLenum mode)" \
  "void  glReadPixels (GLint x, GLint y, GLsizei width, GLsizei height, GLenum format, GLenum type, void *pixels)" \
  "void  glRectd (GLdouble x1, GLdouble y1, GLdouble x2, GLdouble y2)" \
  "void  glRectdv (const GLdouble *v1, const GLdouble *v2)" \
  "void  glRectf (GLfloat x1, GLfloat y1, GLfloat x2, GLfloat y2)" \
  "void  glRectfv (const GLfloat *v1, const GLfloat *v2)" \
  "void  glRecti (GLint x1, GLint y1, GLint x2, GLint y2)" \
  "void  glRectiv (const GLint *v1, const GLint *v2)" \
  "void  glRects (GLshort x1, GLshort y1, GLshort x2, GLshort y2)" \
  "void  glRectsv (const GLshort *v1, const GLshort *v2)" \
  "GLint  glRenderMode (GLenum mode)" \
  "void  glRotated (GLdouble angle, GLdouble x, GLdouble y, GLdouble z)" \
  "void  glRotatef (GLfloat angle, GLfloat x, GLfloat y, GLfloat z)" \
  "void  glScaled (GLdouble x, GLdouble y, GLdouble z)" \
  "void  glScalef (GLfloat x, GLfloat y, GLfloat z)" \
  "void  glScissor (GLint x, GLint y, GLsizei width, GLsizei height)" \
  "void  glSelectBuffer (GLsizei size, GLuint *buffer)" \
  "void  glShadeModel (GLenum mode)" \
  "void  glStencilFunc (GLenum func, GLint ref, GLuint mask)" \
  "void  glStencilMask (GLuint mask)" \
  "void  glStencilOp (GLenum fail, GLenum zfail, GLenum zpass)" \
  "void  glTexCoord1d (GLdouble s)" \
  "void  glTexCoord1dv (const GLdouble *v)" \
  "void  glTexCoord1f (GLfloat s)" \
  "void  glTexCoord1fv (const GLfloat *v)" \
  "void  glTexCoord1i (GLint s)" \
  "void  glTexCoord1iv (const GLint *v)" \
  "void  glTexCoord1s (GLshort s)" \
  "void  glTexCoord1sv (const GLshort *v)" \
  "void  glTexCoord2d (GLdouble s, GLdouble t)" \
  "void  glTexCoord2dv (const GLdouble *v)" \
  "void  glTexCoord2f (GLfloat s, GLfloat t)" \
  "void  glTexCoord2fv (const GLfloat *v)" \
  "void  glTexCoord2i (GLint s, GLint t)" \
  "void  glTexCoord2iv (const GLint *v)" \
  "void  glTexCoord2s (GLshort s, GLshort t)" \
  "void  glTexCoord2sv (const GLshort *v)" \
  "void  glTexCoord3d (GLdouble s, GLdouble t, GLdouble r)" \
  "void  glTexCoord3dv (const GLdouble *v)" \
  "void  glTexCoord3f (GLfloat s, GLfloat t, GLfloat r)" \
  "void  glTexCoord3fv (const GLfloat *v)" \
  "void  glTexCoord3i (GLint s, GLint t, GLint r)" \
  "void  glTexCoord3iv (const GLint *v)" \
  "void  glTexCoord3s (GLshort s, GLshort t, GLshort r)" \
  "void  glTexCoord3sv (const GLshort *v)" \
  "void  glTexCoord4d (GLdouble s, GLdouble t, GLdouble r, GLdouble q)" \
  "void  glTexCoord4dv (const GLdouble *v)" \
  "void  glTexCoord4f (GLfloat s, GLfloat t, GLfloat r, GLfloat q)" \
  "void  glTexCoord4fv (const GLfloat *v)" \
  "void  glTexCoord4i (GLint s, GLint t, GLint r, GLint q)" \
  "void  glTexCoord4iv (const GLint *v)" \
  "void  glTexCoord4s (GLshort s, GLshort t, GLshort r, GLshort q)" \
  "void  glTexCoord4sv (const GLshort *v)" \
  "void  glTexCoordPointer (GLint size, GLenum type, GLsizei stride, const void *pointer)" \
  "void  glTexEnvf (GLenum target, GLenum pname, GLfloat param)" \
  "void  glTexEnvfv (GLenum target, GLenum pname, const GLfloat *params)" \
  "void  glTexEnvi (GLenum target, GLenum pname, GLint param)" \
  "void  glTexEnviv (GLenum target, GLenum pname, const GLint *params)" \
  "void  glTexGend (GLenum coord, GLenum pname, GLdouble param)" \
  "void  glTexGendv (GLenum coord, GLenum pname, const GLdouble *params)" \
  "void  glTexGenf (GLenum coord, GLenum pname, GLfloat param)" \
  "void  glTexGenfv (GLenum coord, GLenum pname, const GLfloat *params)" \
  "void  glTexGeni (GLenum coord, GLenum pname, GLint param)" \
  "void  glTexGeniv (GLenum coord, GLenum pname, const GLint *params)" \
  "void  glTexImage1D (GLenum target, GLint level, GLint internalformat, GLsizei width, GLint border, GLenum format, GLenum type, const void *pixels)" \
  "void  glTexImage2D (GLenum target, GLint level, GLint internalformat, GLsizei width, GLsizei height, GLint border, GLenum format, GLenum type, const void *pixels)" \
  "void  glTexParameterf (GLenum target, GLenum pname, GLfloat param)" \
  "void  glTexParameterfv (GLenum target, GLenum pname, const GLfloat *params)" \
  "void  glTexParameteri (GLenum target, GLenum pname, GLint param)" \
  "void  glTexParameteriv (GLenum target, GLenum pname, const GLint *params)" \
  "void  glTexSubImage1D (GLenum target, GLint level, GLint xoffset, GLsizei width, GLenum format, GLenum type, const void *pixels)" \
  "void  glTexSubImage2D (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLsizei width, GLsizei height, GLenum format, GLenum type, const void *pixels)" \
  "void  glTranslated (GLdouble x, GLdouble y, GLdouble z)" \
  "void  glTranslatef (GLfloat x, GLfloat y, GLfloat z)" \
  "void  glVertex2d (GLdouble x, GLdouble y)" \
  "void  glVertex2dv (const GLdouble *v)" \
  "void  glVertex2f (GLfloat x, GLfloat y)" \
  "void  glVertex2fv (const GLfloat *v)" \
  "void  glVertex2i (GLint x, GLint y)" \
  "void  glVertex2iv (const GLint *v)" \
  "void  glVertex2s (GLshort x, GLshort y)" \
  "void  glVertex2sv (const GLshort *v)" \
  "void  glVertex3d (GLdouble x, GLdouble y, GLdouble z)" \
  "void  glVertex3dv (const GLdouble *v)" \
  "void  glVertex3f (GLfloat x, GLfloat y, GLfloat z)" \
  "void  glVertex3fv (const GLfloat *v)" \
  "void  glVertex3i (GLint x, GLint y, GLint z)" \
  "void  glVertex3iv (const GLint *v)" \
  "void  glVertex3s (GLshort x, GLshort y, GLshort z)" \
  "void  glVertex3sv (const GLshort *v)" \
  "void  glVertex4d (GLdouble x, GLdouble y, GLdouble z, GLdouble w)" \
  "void  glVertex4dv (const GLdouble *v)" \
  "void  glVertex4f (GLfloat x, GLfloat y, GLfloat z, GLfloat w)" \
  "void  glVertex4fv (const GLfloat *v)" \
  "void  glVertex4i (GLint x, GLint y, GLint z, GLint w)" \
  "void  glVertex4iv (const GLint *v)" \
  "void  glVertex4s (GLshort x, GLshort y, GLshort z, GLshort w)" \
  "void  glVertex4sv (const GLshort *v)" \
  "void  glVertexPointer (GLint size, GLenum type, GLsizei stride, const void *pointer)" \
  "void  glViewport (GLint x, GLint y, GLsizei width, GLsizei height)" \
  "void glCopyTexSubImage3D (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLint x, GLint y, GLsizei width, GLsizei height)" \
  "void glDrawRangeElements (GLenum mode, GLuint start, GLuint end, GLsizei count, GLenum type, const void *indices)" \
  "void glTexImage3D (GLenum target, GLint level, GLint internalFormat, GLsizei width, GLsizei height, GLsizei depth, GLint border, GLenum format, GLenum type, const void *pixels)" \
  "void glTexSubImage3D (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, GLenum format, GLenum type, const void *pixels)" \
  "void glActiveTexture (GLenum texture)" \
  "void glClientActiveTexture (GLenum texture)" \
  "void glCompressedTexImage1D (GLenum target, GLint level, GLenum internalformat, GLsizei width, GLint border, GLsizei imageSize, const void *data)" \
  "void glCompressedTexImage2D (GLenum target, GLint level, GLenum internalformat, GLsizei width, GLsizei height, GLint border, GLsizei imageSize, const void *data)" \
  "void glCompressedTexImage3D (GLenum target, GLint level, GLenum internalformat, GLsizei width, GLsizei height, GLsizei depth, GLint border, GLsizei imageSize, const void *data)" \
  "void glCompressedTexSubImage1D (GLenum target, GLint level, GLint xoffset, GLsizei width, GLenum format, GLsizei imageSize, const void *data)" \
  "void glCompressedTexSubImage2D (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLsizei width, GLsizei height, GLenum format, GLsizei imageSize, const void *data)" \
  "void glCompressedTexSubImage3D (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, GLenum format, GLsizei imageSize, const void *data)" \
  "void glGetCompressedTexImage (GLenum target, GLint lod, void *img)" \
  "void glLoadTransposeMatrixd (const GLdouble m\[16\])" \
  "void glLoadTransposeMatrixf (const GLfloat m\[16\])" \
  "void glMultTransposeMatrixd (const GLdouble m\[16\])" \
  "void glMultTransposeMatrixf (const GLfloat m\[16\])" \
  "void glMultiTexCoord1d (GLenum target, GLdouble s)" \
  "void glMultiTexCoord1dv (GLenum target, const GLdouble *v)" \
  "void glMultiTexCoord1f (GLenum target, GLfloat s)" \
  "void glMultiTexCoord1fv (GLenum target, const GLfloat *v)" \
  "void glMultiTexCoord1i (GLenum target, GLint s)" \
  "void glMultiTexCoord1iv (GLenum target, const GLint *v)" \
  "void glMultiTexCoord1s (GLenum target, GLshort s)" \
  "void glMultiTexCoord1sv (GLenum target, const GLshort *v)" \
  "void glMultiTexCoord2d (GLenum target, GLdouble s, GLdouble t)" \
  "void glMultiTexCoord2dv (GLenum target, const GLdouble *v)" \
  "void glMultiTexCoord2f (GLenum target, GLfloat s, GLfloat t)" \
  "void glMultiTexCoord2fv (GLenum target, const GLfloat *v)" \
  "void glMultiTexCoord2i (GLenum target, GLint s, GLint t)" \
  "void glMultiTexCoord2iv (GLenum target, const GLint *v)" \
  "void glMultiTexCoord2s (GLenum target, GLshort s, GLshort t)" \
  "void glMultiTexCoord2sv (GLenum target, const GLshort *v)" \
  "void glMultiTexCoord3d (GLenum target, GLdouble s, GLdouble t, GLdouble r)" \
  "void glMultiTexCoord3dv (GLenum target, const GLdouble *v)" \
  "void glMultiTexCoord3f (GLenum target, GLfloat s, GLfloat t, GLfloat r)" \
  "void glMultiTexCoord3fv (GLenum target, const GLfloat *v)" \
  "void glMultiTexCoord3i (GLenum target, GLint s, GLint t, GLint r)" \
  "void glMultiTexCoord3iv (GLenum target, const GLint *v)" \
  "void glMultiTexCoord3s (GLenum target, GLshort s, GLshort t, GLshort r)" \
  "void glMultiTexCoord3sv (GLenum target, const GLshort *v)" \
  "void glMultiTexCoord4d (GLenum target, GLdouble s, GLdouble t, GLdouble r, GLdouble q)" \
  "void glMultiTexCoord4dv (GLenum target, const GLdouble *v)" \
  "void glMultiTexCoord4f (GLenum target, GLfloat s, GLfloat t, GLfloat r, GLfloat q)" \
  "void glMultiTexCoord4fv (GLenum target, const GLfloat *v)" \
  "void glMultiTexCoord4i (GLenum target, GLint s, GLint t, GLint r, GLint q)" \
  "void glMultiTexCoord4iv (GLenum target, const GLint *v)" \
  "void glMultiTexCoord4s (GLenum target, GLshort s, GLshort t, GLshort r, GLshort q)" \
  "void glMultiTexCoord4sv (GLenum target, const GLshort *v)" \
  "void glSampleCoverage (GLclampf value, GLboolean invert)" \
  "void glBlendColor (GLclampf red, GLclampf green, GLclampf blue, GLclampf alpha)" \
  "void glBlendEquation (GLenum mode)" \
  "void glBlendFuncSeparate (GLenum sfactorRGB, GLenum dfactorRGB, GLenum sfactorAlpha, GLenum dfactorAlpha)" \
  "void glFogCoordPointer (GLenum type, GLsizei stride, const void *pointer)" \
  "void glFogCoordd (GLdouble coord)" \
  "void glFogCoorddv (const GLdouble *coord)" \
  "void glFogCoordf (GLfloat coord)" \
  "void glFogCoordfv (const GLfloat *coord)" \
  "void glMultiDrawArrays (GLenum mode, const GLint *first, const GLsizei *count, GLsizei drawcount)" \
  "void glMultiDrawElements (GLenum mode, const GLsizei *count, GLenum type, const void *const* indices, GLsizei drawcount)" \
  "void glPointParameterf (GLenum pname, GLfloat param)" \
  "void glPointParameterfv (GLenum pname, const GLfloat *params)" \
  "void glPointParameteri (GLenum pname, GLint param)" \
  "void glPointParameteriv (GLenum pname, const GLint *params)" \
  "void glSecondaryColor3b (GLbyte red, GLbyte green, GLbyte blue)" \
  "void glSecondaryColor3bv (const GLbyte *v)" \
  "void glSecondaryColor3d (GLdouble red, GLdouble green, GLdouble blue)" \
  "void glSecondaryColor3dv (const GLdouble *v)" \
  "void glSecondaryColor3f (GLfloat red, GLfloat green, GLfloat blue)" \
  "void glSecondaryColor3fv (const GLfloat *v)" \
  "void glSecondaryColor3i (GLint red, GLint green, GLint blue)" \
  "void glSecondaryColor3iv (const GLint *v)" \
  "void glSecondaryColor3s (GLshort red, GLshort green, GLshort blue)" \
  "void glSecondaryColor3sv (const GLshort *v)" \
  "void glSecondaryColor3ub (GLubyte red, GLubyte green, GLubyte blue)" \
  "void glSecondaryColor3ubv (const GLubyte *v)" \
  "void glSecondaryColor3ui (GLuint red, GLuint green, GLuint blue)" \
  "void glSecondaryColor3uiv (const GLuint *v)" \
  "void glSecondaryColor3us (GLushort red, GLushort green, GLushort blue)" \
  "void glSecondaryColor3usv (const GLushort *v)" \
  "void glSecondaryColorPointer (GLint size, GLenum type, GLsizei stride, const void *pointer)" \
  "void glWindowPos2d (GLdouble x, GLdouble y)" \
  "void glWindowPos2dv (const GLdouble *p)" \
  "void glWindowPos2f (GLfloat x, GLfloat y)" \
  "void glWindowPos2fv (const GLfloat *p)" \
  "void glWindowPos2i (GLint x, GLint y)" \
  "void glWindowPos2iv (const GLint *p)" \
  "void glWindowPos2s (GLshort x, GLshort y)" \
  "void glWindowPos2sv (const GLshort *p)" \
  "void glWindowPos3d (GLdouble x, GLdouble y, GLdouble z)" \
  "void glWindowPos3dv (const GLdouble *p)" \
  "void glWindowPos3f (GLfloat x, GLfloat y, GLfloat z)" \
  "void glWindowPos3fv (const GLfloat *p)" \
  "void glWindowPos3i (GLint x, GLint y, GLint z)" \
  "void glWindowPos3iv (const GLint *p)" \
  "void glWindowPos3s (GLshort x, GLshort y, GLshort z)" \
  "void glWindowPos3sv (const GLshort *p)" \
  "void glBeginQuery (GLenum target, GLuint id)" \
  "void glBindBuffer (GLenum target, GLuint buffer)" \
  "void glBufferData (GLenum target, GLsizeiptr size, const void* data, GLenum usage)" \
  "void glBufferSubData (GLenum target, GLintptr offset, GLsizeiptr size, const void* data)" \
  "void glDeleteBuffers (GLsizei n, const GLuint* buffers)" \
  "void glDeleteQueries (GLsizei n, const GLuint* ids)" \
  "void glEndQuery (GLenum target)" \
  "void glGenBuffers (GLsizei n, GLuint* buffers)" \
  "void glGenQueries (GLsizei n, GLuint* ids)" \
  "void glGetBufferParameteriv (GLenum target, GLenum pname, GLint* params)" \
  "void glGetBufferPointerv (GLenum target, GLenum pname, void** params)" \
  "void glGetBufferSubData (GLenum target, GLintptr offset, GLsizeiptr size, void* data)" \
  "void glGetQueryObjectiv (GLuint id, GLenum pname, GLint* params)" \
  "void glGetQueryObjectuiv (GLuint id, GLenum pname, GLuint* params)" \
  "void glGetQueryiv (GLenum target, GLenum pname, GLint* params)" \
  "GLboolean glIsBuffer (GLuint buffer)" \
  "GLboolean glIsQuery (GLuint id)" \
  "void* glMapBuffer (GLenum target, GLenum access)" \
  "GLboolean glUnmapBuffer (GLenum target)" \
  "void glAttachShader (GLuint program, GLuint shader)" \
  "void glBindAttribLocation (GLuint program, GLuint index, const GLchar* name)" \
  "void glBlendEquationSeparate (GLenum modeRGB, GLenum modeAlpha)" \
  "void glCompileShader (GLuint shader)" \
  "GLuint glCreateProgram (void)" \
  "GLuint glCreateShader (GLenum type)" \
  "void glDeleteProgram (GLuint program)" \
  "void glDeleteShader (GLuint shader)" \
  "void glDetachShader (GLuint program, GLuint shader)" \
  "void glDisableVertexAttribArray (GLuint index)" \
  "void glDrawBuffers (GLsizei n, const GLenum* bufs)" \
  "void glEnableVertexAttribArray (GLuint index)" \
  "void glGetActiveAttrib (GLuint program, GLuint index, GLsizei maxLength, GLsizei* length, GLint* size, GLenum* type, GLchar* name)" \
  "void glGetActiveUniform (GLuint program, GLuint index, GLsizei maxLength, GLsizei* length, GLint* size, GLenum* type, GLchar* name)" \
  "void glGetAttachedShaders (GLuint program, GLsizei maxCount, GLsizei* count, GLuint* shaders)" \
  "GLint glGetAttribLocation (GLuint program, const GLchar* name)" \
  "void glGetProgramInfoLog (GLuint program, GLsizei bufSize, GLsizei* length, GLchar* infoLog)" \
  "void glGetProgramiv (GLuint program, GLenum pname, GLint* param)" \
  "void glGetShaderInfoLog (GLuint shader, GLsizei bufSize, GLsizei* length, GLchar* infoLog)" \
  "void glGetShaderSource (GLuint obj, GLsizei maxLength, GLsizei* length, GLchar* source)" \
  "void glGetShaderiv (GLuint shader, GLenum pname, GLint* param)" \
  "GLint glGetUniformLocation (GLuint program, const GLchar* name)" \
  "void glGetUniformfv (GLuint program, GLint location, GLfloat* params)" \
  "void glGetUniformiv (GLuint program, GLint location, GLint* params)" \
  "void glGetVertexAttribPointerv (GLuint index, GLenum pname, void** pointer)" \
  "void glGetVertexAttribdv (GLuint index, GLenum pname, GLdouble* params)" \
  "void glGetVertexAttribfv (GLuint index, GLenum pname, GLfloat* params)" \
  "void glGetVertexAttribiv (GLuint index, GLenum pname, GLint* params)" \
  "GLboolean glIsProgram (GLuint program)" \
  "GLboolean glIsShader (GLuint shader)" \
  "void glLinkProgram (GLuint program)" \
  "void glShaderSource (GLuint shader, GLsizei count, const GLchar *const* string, const GLint* length)" \
  "void glStencilFuncSeparate (GLenum face, GLenum func, GLint ref, GLuint mask)" \
  "void glStencilMaskSeparate (GLenum face, GLuint mask)" \
  "void glStencilOpSeparate (GLenum face, GLenum sfail, GLenum dpfail, GLenum dppass)" \
  "void glUniform1f (GLint location, GLfloat v0)" \
  "void glUniform1fv (GLint location, GLsizei count, const GLfloat* value)" \
  "void glUniform1i (GLint location, GLint v0)" \
  "void glUniform1iv (GLint location, GLsizei count, const GLint* value)" \
  "void glUniform2f (GLint location, GLfloat v0, GLfloat v1)" \
  "void glUniform2fv (GLint location, GLsizei count, const GLfloat* value)" \
  "void glUniform2i (GLint location, GLint v0, GLint v1)" \
  "void glUniform2iv (GLint location, GLsizei count, const GLint* value)" \
  "void glUniform3f (GLint location, GLfloat v0, GLfloat v1, GLfloat v2)" \
  "void glUniform3fv (GLint location, GLsizei count, const GLfloat* value)" \
  "void glUniform3i (GLint location, GLint v0, GLint v1, GLint v2)" \
  "void glUniform3iv (GLint location, GLsizei count, const GLint* value)" \
  "void glUniform4f (GLint location, GLfloat v0, GLfloat v1, GLfloat v2, GLfloat v3)" \
  "void glUniform4fv (GLint location, GLsizei count, const GLfloat* value)" \
  "void glUniform4i (GLint location, GLint v0, GLint v1, GLint v2, GLint v3)" \
  "void glUniform4iv (GLint location, GLsizei count, const GLint* value)" \
  "void glUniformMatrix2fv (GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glUniformMatrix3fv (GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glUniformMatrix4fv (GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glUseProgram (GLuint program)" \
  "void glValidateProgram (GLuint program)" \
  "void glVertexAttrib1d (GLuint index, GLdouble x)" \
  "void glVertexAttrib1dv (GLuint index, const GLdouble* v)" \
  "void glVertexAttrib1f (GLuint index, GLfloat x)" \
  "void glVertexAttrib1fv (GLuint index, const GLfloat* v)" \
  "void glVertexAttrib1s (GLuint index, GLshort x)" \
  "void glVertexAttrib1sv (GLuint index, const GLshort* v)" \
  "void glVertexAttrib2d (GLuint index, GLdouble x, GLdouble y)" \
  "void glVertexAttrib2dv (GLuint index, const GLdouble* v)" \
  "void glVertexAttrib2f (GLuint index, GLfloat x, GLfloat y)" \
  "void glVertexAttrib2fv (GLuint index, const GLfloat* v)" \
  "void glVertexAttrib2s (GLuint index, GLshort x, GLshort y)" \
  "void glVertexAttrib2sv (GLuint index, const GLshort* v)" \
  "void glVertexAttrib3d (GLuint index, GLdouble x, GLdouble y, GLdouble z)" \
  "void glVertexAttrib3dv (GLuint index, const GLdouble* v)" \
  "void glVertexAttrib3f (GLuint index, GLfloat x, GLfloat y, GLfloat z)" \
  "void glVertexAttrib3fv (GLuint index, const GLfloat* v)" \
  "void glVertexAttrib3s (GLuint index, GLshort x, GLshort y, GLshort z)" \
  "void glVertexAttrib3sv (GLuint index, const GLshort* v)" \
  "void glVertexAttrib4Nbv (GLuint index, const GLbyte* v)" \
  "void glVertexAttrib4Niv (GLuint index, const GLint* v)" \
  "void glVertexAttrib4Nsv (GLuint index, const GLshort* v)" \
  "void glVertexAttrib4Nub (GLuint index, GLubyte x, GLubyte y, GLubyte z, GLubyte w)" \
  "void glVertexAttrib4Nubv (GLuint index, const GLubyte* v)" \
  "void glVertexAttrib4Nuiv (GLuint index, const GLuint* v)" \
  "void glVertexAttrib4Nusv (GLuint index, const GLushort* v)" \
  "void glVertexAttrib4bv (GLuint index, const GLbyte* v)" \
  "void glVertexAttrib4d (GLuint index, GLdouble x, GLdouble y, GLdouble z, GLdouble w)" \
  "void glVertexAttrib4dv (GLuint index, const GLdouble* v)" \
  "void glVertexAttrib4f (GLuint index, GLfloat x, GLfloat y, GLfloat z, GLfloat w)" \
  "void glVertexAttrib4fv (GLuint index, const GLfloat* v)" \
  "void glVertexAttrib4iv (GLuint index, const GLint* v)" \
  "void glVertexAttrib4s (GLuint index, GLshort x, GLshort y, GLshort z, GLshort w)" \
  "void glVertexAttrib4sv (GLuint index, const GLshort* v)" \
  "void glVertexAttrib4ubv (GLuint index, const GLubyte* v)" \
  "void glVertexAttrib4uiv (GLuint index, const GLuint* v)" \
  "void glVertexAttrib4usv (GLuint index, const GLushort* v)" \
  "void glVertexAttribPointer (GLuint index, GLint size, GLenum type, GLboolean normalized, GLsizei stride, const void* pointer)" \
  "void glUniformMatrix2x3fv (GLint location, GLsizei count, GLboolean transpose, const GLfloat *value)" \
  "void glUniformMatrix2x4fv (GLint location, GLsizei count, GLboolean transpose, const GLfloat *value)" \
  "void glUniformMatrix3x2fv (GLint location, GLsizei count, GLboolean transpose, const GLfloat *value)" \
  "void glUniformMatrix3x4fv (GLint location, GLsizei count, GLboolean transpose, const GLfloat *value)" \
  "void glUniformMatrix4x2fv (GLint location, GLsizei count, GLboolean transpose, const GLfloat *value)" \
  "void glUniformMatrix4x3fv (GLint location, GLsizei count, GLboolean transpose, const GLfloat *value)" \
  "void glBeginConditionalRender (GLuint id, GLenum mode)" \
  "void glBeginTransformFeedback (GLenum primitiveMode)" \
  "void glBindFragDataLocation (GLuint program, GLuint colorNumber, const GLchar* name)" \
  "void glClampColor (GLenum target, GLenum clamp)" \
  "void glClearBufferfi (GLenum buffer, GLint drawBuffer, GLfloat depth, GLint stencil)" \
  "void glClearBufferfv (GLenum buffer, GLint drawBuffer, const GLfloat* value)" \
  "void glClearBufferiv (GLenum buffer, GLint drawBuffer, const GLint* value)" \
  "void glClearBufferuiv (GLenum buffer, GLint drawBuffer, const GLuint* value)" \
  "void glColorMaski (GLuint buf, GLboolean red, GLboolean green, GLboolean blue, GLboolean alpha)" \
  "void glDisablei (GLenum cap, GLuint index)" \
  "void glEnablei (GLenum cap, GLuint index)" \
  "void glEndConditionalRender (void)" \
  "void glEndTransformFeedback (void)" \
  "void glGetBooleani_v (GLenum pname, GLuint index, GLboolean* data)" \
  "GLint glGetFragDataLocation (GLuint program, const GLchar* name)" \
  "const GLubyte* glGetStringi (GLenum name, GLuint index)" \
  "void glGetTexParameterIiv (GLenum target, GLenum pname, GLint* params)" \
  "void glGetTexParameterIuiv (GLenum target, GLenum pname, GLuint* params)" \
  "void glGetTransformFeedbackVarying (GLuint program, GLuint index, GLsizei bufSize, GLsizei * length, GLsizei * size, GLenum * type, GLchar * name)" \
  "void glGetUniformuiv (GLuint program, GLint location, GLuint* params)" \
  "void glGetVertexAttribIiv (GLuint index, GLenum pname, GLint* params)" \
  "void glGetVertexAttribIuiv (GLuint index, GLenum pname, GLuint* params)" \
  "GLboolean glIsEnabledi (GLenum cap, GLuint index)" \
  "void glTexParameterIiv (GLenum target, GLenum pname, const GLint* params)" \
  "void glTexParameterIuiv (GLenum target, GLenum pname, const GLuint* params)" \
  "void glTransformFeedbackVaryings (GLuint program, GLsizei count, const GLchar *const* varyings, GLenum bufferMode)" \
  "void glUniform1ui (GLint location, GLuint v0)" \
  "void glUniform1uiv (GLint location, GLsizei count, const GLuint* value)" \
  "void glUniform2ui (GLint location, GLuint v0, GLuint v1)" \
  "void glUniform2uiv (GLint location, GLsizei count, const GLuint* value)" \
  "void glUniform3ui (GLint location, GLuint v0, GLuint v1, GLuint v2)" \
  "void glUniform3uiv (GLint location, GLsizei count, const GLuint* value)" \
  "void glUniform4ui (GLint location, GLuint v0, GLuint v1, GLuint v2, GLuint v3)" \
  "void glUniform4uiv (GLint location, GLsizei count, const GLuint* value)" \
  "void glVertexAttribI1i (GLuint index, GLint v0)" \
  "void glVertexAttribI1iv (GLuint index, const GLint* v0)" \
  "void glVertexAttribI1ui (GLuint index, GLuint v0)" \
  "void glVertexAttribI1uiv (GLuint index, const GLuint* v0)" \
  "void glVertexAttribI2i (GLuint index, GLint v0, GLint v1)" \
  "void glVertexAttribI2iv (GLuint index, const GLint* v0)" \
  "void glVertexAttribI2ui (GLuint index, GLuint v0, GLuint v1)" \
  "void glVertexAttribI2uiv (GLuint index, const GLuint* v0)" \
  "void glVertexAttribI3i (GLuint index, GLint v0, GLint v1, GLint v2)" \
  "void glVertexAttribI3iv (GLuint index, const GLint* v0)" \
  "void glVertexAttribI3ui (GLuint index, GLuint v0, GLuint v1, GLuint v2)" \
  "void glVertexAttribI3uiv (GLuint index, const GLuint* v0)" \
  "void glVertexAttribI4bv (GLuint index, const GLbyte* v0)" \
  "void glVertexAttribI4i (GLuint index, GLint v0, GLint v1, GLint v2, GLint v3)" \
  "void glVertexAttribI4iv (GLuint index, const GLint* v0)" \
  "void glVertexAttribI4sv (GLuint index, const GLshort* v0)" \
  "void glVertexAttribI4ubv (GLuint index, const GLubyte* v0)" \
  "void glVertexAttribI4ui (GLuint index, GLuint v0, GLuint v1, GLuint v2, GLuint v3)" \
  "void glVertexAttribI4uiv (GLuint index, const GLuint* v0)" \
  "void glVertexAttribI4usv (GLuint index, const GLushort* v0)" \
  "void glVertexAttribIPointer (GLuint index, GLint size, GLenum type, GLsizei stride, const void*pointer)" \
  "void glDrawArraysInstanced (GLenum mode, GLint first, GLsizei count, GLsizei primcount)" \
  "void glDrawElementsInstanced (GLenum mode, GLsizei count, GLenum type, const void* indices, GLsizei primcount)" \
  "void glPrimitiveRestartIndex (GLuint buffer)" \
  "void glTexBuffer (GLenum target, GLenum internalFormat, GLuint buffer)" \
  "void glFramebufferTexture (GLenum target, GLenum attachment, GLuint texture, GLint level)" \
  "void glGetBufferParameteri64v (GLenum target, GLenum value, GLint64 * data)" \
  "void glGetInteger64i_v (GLenum pname, GLuint index, GLint64 * data)" \
  "void glVertexAttribDivisor (GLuint index, GLuint divisor)" \
  "void glBlendEquationSeparatei (GLuint buf, GLenum modeRGB, GLenum modeAlpha)" \
  "void glBlendEquationi (GLuint buf, GLenum mode)" \
  "void glBlendFuncSeparatei (GLuint buf, GLenum srcRGB, GLenum dstRGB, GLenum srcAlpha, GLenum dstAlpha)" \
  "void glBlendFunci (GLuint buf, GLenum src, GLenum dst)" \
  "void glMinSampleShading (GLclampf value)" \
  "GLenum glGetGraphicsResetStatus (void)" \
  "void glGetnCompressedTexImage (GLenum target, GLint lod, GLsizei bufSize, GLvoid *pixels)" \
  "void glGetnTexImage (GLenum tex, GLint level, GLenum format, GLenum type, GLsizei bufSize, GLvoid *pixels)" \
  "void glGetnUniformdv (GLuint program, GLint location, GLsizei bufSize, GLdouble *params)" \
  "void glMultiDrawArraysIndirectCount (GLenum mode, const GLvoid *indirect, GLintptr drawcount, GLsizei maxdrawcount, GLsizei stride)" \
  "void glMultiDrawElementsIndirectCount (GLenum mode, GLenum type, const GLvoid *indirect, GLintptr drawcount, GLsizei maxdrawcount, GLsizei stride)" \
  "void glSpecializeShader (GLuint shader, const GLchar *pEntryPoint, GLuint numSpecializationConstants, const GLuint *pConstantIndex, const GLuint *pConstantValue)" \
  "void glTbufferMask3DFX (GLuint mask)" \
  "void glDebugMessageCallbackAMD (GLDEBUGPROCAMD callback, void *userParam)" \
  "void glDebugMessageEnableAMD (GLenum category, GLenum severity, GLsizei count, const GLuint* ids, GLboolean enabled)" \
  "void glDebugMessageInsertAMD (GLenum category, GLenum severity, GLuint id, GLsizei length, const GLchar* buf)" \
  "GLuint glGetDebugMessageLogAMD (GLuint count, GLsizei bufsize, GLenum* categories, GLuint* severities, GLuint* ids, GLsizei* lengths, GLchar* message)" \
  "void glBlendEquationIndexedAMD (GLuint buf, GLenum mode)" \
  "void glBlendEquationSeparateIndexedAMD (GLuint buf, GLenum modeRGB, GLenum modeAlpha)" \
  "void glBlendFuncIndexedAMD (GLuint buf, GLenum src, GLenum dst)" \
  "void glBlendFuncSeparateIndexedAMD (GLuint buf, GLenum srcRGB, GLenum dstRGB, GLenum srcAlpha, GLenum dstAlpha)" \
  "void glNamedRenderbufferStorageMultisampleAdvancedAMD (GLuint renderbuffer, GLsizei samples, GLsizei storageSamples, GLenum internalformat, GLsizei width, GLsizei height)" \
  "void glRenderbufferStorageMultisampleAdvancedAMD (GLenum target, GLsizei samples, GLsizei storageSamples, GLenum internalformat, GLsizei width, GLsizei height)" \
  "void glFramebufferSamplePositionsfvAMD (GLenum target, GLuint numsamples, GLuint pixelindex, const GLfloat* values)" \
  "void glGetFramebufferParameterfvAMD (GLenum target, GLenum pname, GLuint numsamples, GLuint pixelindex, GLsizei size, GLfloat* values)" \
  "void glGetNamedFramebufferParameterfvAMD (GLuint framebuffer, GLenum pname, GLuint numsamples, GLuint pixelindex, GLsizei size, GLfloat* values)" \
  "void glNamedFramebufferSamplePositionsfvAMD (GLuint framebuffer, GLuint numsamples, GLuint pixelindex, const GLfloat* values)" \
  "void glVertexAttribParameteriAMD (GLuint index, GLenum pname, GLint param)" \
  "void glMultiDrawArraysIndirectAMD (GLenum mode, const void *indirect, GLsizei primcount, GLsizei stride)" \
  "void glMultiDrawElementsIndirectAMD (GLenum mode, GLenum type, const void *indirect, GLsizei primcount, GLsizei stride)" \
  "void glDeleteNamesAMD (GLenum identifier, GLuint num, const GLuint* names)" \
  "void glGenNamesAMD (GLenum identifier, GLuint num, GLuint* names)" \
  "GLboolean glIsNameAMD (GLenum identifier, GLuint name)" \
  "void glQueryObjectParameteruiAMD (GLenum target, GLuint id, GLenum pname, GLuint param)" \
  "void glBeginPerfMonitorAMD (GLuint monitor)" \
  "void glDeletePerfMonitorsAMD (GLsizei n, GLuint* monitors)" \
  "void glEndPerfMonitorAMD (GLuint monitor)" \
  "void glGenPerfMonitorsAMD (GLsizei n, GLuint* monitors)" \
  "void glGetPerfMonitorCounterDataAMD (GLuint monitor, GLenum pname, GLsizei dataSize, GLuint* data, GLint *bytesWritten)" \
  "void glGetPerfMonitorCounterInfoAMD (GLuint group, GLuint counter, GLenum pname, void *data)" \
  "void glGetPerfMonitorCounterStringAMD (GLuint group, GLuint counter, GLsizei bufSize, GLsizei* length, GLchar *counterString)" \
  "void glGetPerfMonitorCountersAMD (GLuint group, GLint* numCounters, GLint *maxActiveCounters, GLsizei countersSize, GLuint *counters)" \
  "void glGetPerfMonitorGroupStringAMD (GLuint group, GLsizei bufSize, GLsizei* length, GLchar *groupString)" \
  "void glGetPerfMonitorGroupsAMD (GLint* numGroups, GLsizei groupsSize, GLuint *groups)" \
  "void glSelectPerfMonitorCountersAMD (GLuint monitor, GLboolean enable, GLuint group, GLint numCounters, GLuint* counterList)" \
  "void glSetMultisamplefvAMD (GLenum pname, GLuint index, const GLfloat* val)" \
  "void glTexStorageSparseAMD (GLenum target, GLenum internalFormat, GLsizei width, GLsizei height, GLsizei depth, GLsizei layers, GLbitfield flags)" \
  "void glTextureStorageSparseAMD (GLuint texture, GLenum target, GLenum internalFormat, GLsizei width, GLsizei height, GLsizei depth, GLsizei layers, GLbitfield flags)" \
  "void glStencilOpValueAMD (GLenum face, GLuint value)" \
  "void glTessellationFactorAMD (GLfloat factor)" \
  "void glTessellationModeAMD (GLenum mode)" \
  "void glBlitFramebufferANGLE (GLint srcX0, GLint srcY0, GLint srcX1, GLint srcY1, GLint dstX0, GLint dstY0, GLint dstX1, GLint dstY1, GLbitfield mask, GLenum filter)" \
  "void glRenderbufferStorageMultisampleANGLE (GLenum target, GLsizei samples, GLenum internalformat, GLsizei width, GLsizei height)" \
  "void glDrawArraysInstancedANGLE (GLenum mode, GLint first, GLsizei count, GLsizei primcount)" \
  "void glDrawElementsInstancedANGLE (GLenum mode, GLsizei count, GLenum type, const void *indices, GLsizei primcount)" \
  "void glVertexAttribDivisorANGLE (GLuint index, GLuint divisor)" \
  "void glBeginQueryANGLE (GLenum target, GLuint id)" \
  "void glDeleteQueriesANGLE (GLsizei n, const GLuint* ids)" \
  "void glEndQueryANGLE (GLenum target)" \
  "void glGenQueriesANGLE (GLsizei n, GLuint* ids)" \
  "void glGetQueryObjecti64vANGLE (GLuint id, GLenum pname, GLint64* params)" \
  "void glGetQueryObjectivANGLE (GLuint id, GLenum pname, GLint* params)" \
  "void glGetQueryObjectui64vANGLE (GLuint id, GLenum pname, GLuint64* params)" \
  "void glGetQueryObjectuivANGLE (GLuint id, GLenum pname, GLuint* params)" \
  "void glGetQueryivANGLE (GLenum target, GLenum pname, GLint* params)" \
  "GLboolean glIsQueryANGLE (GLuint id)" \
  "void glQueryCounterANGLE (GLuint id, GLenum target)" \
  "void glGetTranslatedShaderSourceANGLE (GLuint shader, GLsizei bufsize, GLsizei* length, GLchar* source)" \
  "void glCopyTextureLevelsAPPLE (GLuint destinationTexture, GLuint sourceTexture, GLint sourceBaseLevel, GLsizei sourceLevelCount)" \
  "void glDrawElementArrayAPPLE (GLenum mode, GLint first, GLsizei count)" \
  "void glDrawRangeElementArrayAPPLE (GLenum mode, GLuint start, GLuint end, GLint first, GLsizei count)" \
  "void glElementPointerAPPLE (GLenum type, const void *pointer)" \
  "void glMultiDrawElementArrayAPPLE (GLenum mode, const GLint* first, const GLsizei *count, GLsizei primcount)" \
  "void glMultiDrawRangeElementArrayAPPLE (GLenum mode, GLuint start, GLuint end, const GLint* first, const GLsizei *count, GLsizei primcount)" \
  "void glDeleteFencesAPPLE (GLsizei n, const GLuint* fences)" \
  "void glFinishFenceAPPLE (GLuint fence)" \
  "void glFinishObjectAPPLE (GLenum object, GLint name)" \
  "void glGenFencesAPPLE (GLsizei n, GLuint* fences)" \
  "GLboolean glIsFenceAPPLE (GLuint fence)" \
  "void glSetFenceAPPLE (GLuint fence)" \
  "GLboolean glTestFenceAPPLE (GLuint fence)" \
  "GLboolean glTestObjectAPPLE (GLenum object, GLuint name)" \
  "void glBufferParameteriAPPLE (GLenum target, GLenum pname, GLint param)" \
  "void glFlushMappedBufferRangeAPPLE (GLenum target, GLintptr offset, GLsizeiptr size)" \
  "void glRenderbufferStorageMultisampleAPPLE (GLenum target, GLsizei samples, GLenum internalformat, GLsizei width, GLsizei height)" \
  "void glResolveMultisampleFramebufferAPPLE (void)" \
  "void glGetObjectParameterivAPPLE (GLenum objectType, GLuint name, GLenum pname, GLint* params)" \
  "GLenum glObjectPurgeableAPPLE (GLenum objectType, GLuint name, GLenum option)" \
  "GLenum glObjectUnpurgeableAPPLE (GLenum objectType, GLuint name, GLenum option)" \
  "GLenum glClientWaitSyncAPPLE (GLsync GLsync, GLbitfield flags, GLuint64 timeout)" \
  "void glDeleteSyncAPPLE (GLsync GLsync)" \
  "GLsync glFenceSyncAPPLE (GLenum condition, GLbitfield flags)" \
  "void glGetInteger64vAPPLE (GLenum pname, GLint64* params)" \
  "void glGetSyncivAPPLE (GLsync GLsync, GLenum pname, GLsizei bufSize, GLsizei* length, GLint *values)" \
  "GLboolean glIsSyncAPPLE (GLsync GLsync)" \
  "void glWaitSyncAPPLE (GLsync GLsync, GLbitfield flags, GLuint64 timeout)" \
  "void glGetTexParameterPointervAPPLE (GLenum target, GLenum pname, void **params)" \
  "void glTextureRangeAPPLE (GLenum target, GLsizei length, void *pointer)" \
  "void glBindVertexArrayAPPLE (GLuint array)" \
  "void glDeleteVertexArraysAPPLE (GLsizei n, const GLuint* arrays)" \
  "void glGenVertexArraysAPPLE (GLsizei n, const GLuint* arrays)" \
  "GLboolean glIsVertexArrayAPPLE (GLuint array)" \
  "void glFlushVertexArrayRangeAPPLE (GLsizei length, void *pointer)" \
  "void glVertexArrayParameteriAPPLE (GLenum pname, GLint param)" \
  "void glVertexArrayRangeAPPLE (GLsizei length, void *pointer)" \
  "void glDisableVertexAttribAPPLE (GLuint index, GLenum pname)" \
  "void glEnableVertexAttribAPPLE (GLuint index, GLenum pname)" \
  "GLboolean glIsVertexAttribEnabledAPPLE (GLuint index, GLenum pname)" \
  "void glMapVertexAttrib1dAPPLE (GLuint index, GLuint size, GLdouble u1, GLdouble u2, GLint stride, GLint order, const GLdouble* points)" \
  "void glMapVertexAttrib1fAPPLE (GLuint index, GLuint size, GLfloat u1, GLfloat u2, GLint stride, GLint order, const GLfloat* points)" \
  "void glMapVertexAttrib2dAPPLE (GLuint index, GLuint size, GLdouble u1, GLdouble u2, GLint ustride, GLint uorder, GLdouble v1, GLdouble v2, GLint vstride, GLint vorder, const GLdouble* points)" \
  "void glMapVertexAttrib2fAPPLE (GLuint index, GLuint size, GLfloat u1, GLfloat u2, GLint ustride, GLint uorder, GLfloat v1, GLfloat v2, GLint vstride, GLint vorder, const GLfloat* points)" \
  "void glClearDepthf (GLclampf d)" \
  "void glDepthRangef (GLclampf n, GLclampf f)" \
  "void glGetShaderPrecisionFormat (GLenum shadertype, GLenum precisiontype, GLint* range, GLint *precision)" \
  "void glReleaseShaderCompiler (void)" \
  "void glShaderBinary (GLsizei count, const GLuint* shaders, GLenum binaryformat, const void*binary, GLsizei length)" \
  "void glMemoryBarrierByRegion (GLbitfield barriers)" \
  "void glPrimitiveBoundingBoxARB (GLfloat minX, GLfloat minY, GLfloat minZ, GLfloat minW, GLfloat maxX, GLfloat maxY, GLfloat maxZ, GLfloat maxW)" \
  "void glDrawArraysInstancedBaseInstance (GLenum mode, GLint first, GLsizei count, GLsizei primcount, GLuint baseinstance)" \
  "void glDrawElementsInstancedBaseInstance (GLenum mode, GLsizei count, GLenum type, const void *indices, GLsizei primcount, GLuint baseinstance)" \
  "void glDrawElementsInstancedBaseVertexBaseInstance (GLenum mode, GLsizei count, GLenum type, const void *indices, GLsizei primcount, GLint basevertex, GLuint baseinstance)" \
  "GLuint64 glGetImageHandleARB (GLuint texture, GLint level, GLboolean layered, GLint layer, GLenum format)" \
  "GLuint64 glGetTextureHandleARB (GLuint texture)" \
  "GLuint64 glGetTextureSamplerHandleARB (GLuint texture, GLuint sampler)" \
  "void glGetVertexAttribLui64vARB (GLuint index, GLenum pname, GLuint64EXT* params)" \
  "GLboolean glIsImageHandleResidentARB (GLuint64 handle)" \
  "GLboolean glIsTextureHandleResidentARB (GLuint64 handle)" \
  "void glMakeImageHandleNonResidentARB (GLuint64 handle)" \
  "void glMakeImageHandleResidentARB (GLuint64 handle, GLenum access)" \
  "void glMakeTextureHandleNonResidentARB (GLuint64 handle)" \
  "void glMakeTextureHandleResidentARB (GLuint64 handle)" \
  "void glProgramUniformHandleui64ARB (GLuint program, GLint location, GLuint64 value)" \
  "void glProgramUniformHandleui64vARB (GLuint program, GLint location, GLsizei count, const GLuint64* values)" \
  "void glUniformHandleui64ARB (GLint location, GLuint64 value)" \
  "void glUniformHandleui64vARB (GLint location, GLsizei count, const GLuint64* value)" \
  "void glVertexAttribL1ui64ARB (GLuint index, GLuint64EXT x)" \
  "void glVertexAttribL1ui64vARB (GLuint index, const GLuint64EXT* v)" \
  "void glBindFragDataLocationIndexed (GLuint program, GLuint colorNumber, GLuint index, const GLchar * name)" \
  "GLint glGetFragDataIndex (GLuint program, const GLchar * name)" \
  "void glBufferStorage (GLenum target, GLsizeiptr size, const void *data, GLbitfield flags)" \
  "GLsync glCreateSyncFromCLeventARB (cl_context context, cl_event event, GLbitfield flags)" \
  "void glClearBufferData (GLenum target, GLenum internalformat, GLenum format, GLenum type, const void *data)" \
  "void glClearBufferSubData (GLenum target, GLenum internalformat, GLintptr offset, GLsizeiptr size, GLenum format, GLenum type, const void *data)" \
  "void glClearNamedBufferDataEXT (GLuint buffer, GLenum internalformat, GLenum format, GLenum type, const void *data)" \
  "void glClearNamedBufferSubDataEXT (GLuint buffer, GLenum internalformat, GLintptr offset, GLsizeiptr size, GLenum format, GLenum type, const void *data)" \
  "void glClearTexImage (GLuint texture, GLint level, GLenum format, GLenum type, const void *data)" \
  "void glClearTexSubImage (GLuint texture, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, GLenum format, GLenum type, const void *data)" \
  "void glClipControl (GLenum origin, GLenum depth)" \
  "void glClampColorARB (GLenum target, GLenum clamp)" \
  "void glDispatchCompute (GLuint num_groups_x, GLuint num_groups_y, GLuint num_groups_z)" \
  "void glDispatchComputeIndirect (GLintptr indirect)" \
  "void glDispatchComputeGroupSizeARB (GLuint num_groups_x, GLuint num_groups_y, GLuint num_groups_z, GLuint group_size_x, GLuint group_size_y, GLuint group_size_z)" \
  "void glCopyBufferSubData (GLenum readtarget, GLenum writetarget, GLintptr readoffset, GLintptr writeoffset, GLsizeiptr size)" \
  "void glCopyImageSubData (GLuint srcName, GLenum srcTarget, GLint srcLevel, GLint srcX, GLint srcY, GLint srcZ, GLuint dstName, GLenum dstTarget, GLint dstLevel, GLint dstX, GLint dstY, GLint dstZ, GLsizei srcWidth, GLsizei srcHeight, GLsizei srcDepth)" \
  "void glDebugMessageCallbackARB (GLDEBUGPROCARB callback, const void *userParam)" \
  "void glDebugMessageControlARB (GLenum source, GLenum type, GLenum severity, GLsizei count, const GLuint* ids, GLboolean enabled)" \
  "void glDebugMessageInsertARB (GLenum source, GLenum type, GLuint id, GLenum severity, GLsizei length, const GLchar* buf)" \
  "GLuint glGetDebugMessageLogARB (GLuint count, GLsizei bufSize, GLenum* sources, GLenum* types, GLuint* ids, GLenum* severities, GLsizei* lengths, GLchar* messageLog)" \
  "void glBindTextureUnit (GLuint unit, GLuint texture)" \
  "void glBlitNamedFramebuffer (GLuint readFramebuffer, GLuint drawFramebuffer, GLint srcX0, GLint srcY0, GLint srcX1, GLint srcY1, GLint dstX0, GLint dstY0, GLint dstX1, GLint dstY1, GLbitfield mask, GLenum filter)" \
  "GLenum glCheckNamedFramebufferStatus (GLuint framebuffer, GLenum target)" \
  "void glClearNamedBufferData (GLuint buffer, GLenum internalformat, GLenum format, GLenum type, const void *data)" \
  "void glClearNamedBufferSubData (GLuint buffer, GLenum internalformat, GLintptr offset, GLsizeiptr size, GLenum format, GLenum type, const void *data)" \
  "void glClearNamedFramebufferfi (GLuint framebuffer, GLenum buffer, GLint drawbuffer, GLfloat depth, GLint stencil)" \
  "void glClearNamedFramebufferfv (GLuint framebuffer, GLenum buffer, GLint drawbuffer, const GLfloat* value)" \
  "void glClearNamedFramebufferiv (GLuint framebuffer, GLenum buffer, GLint drawbuffer, const GLint* value)" \
  "void glClearNamedFramebufferuiv (GLuint framebuffer, GLenum buffer, GLint drawbuffer, const GLuint* value)" \
  "void glCompressedTextureSubImage1D (GLuint texture, GLint level, GLint xoffset, GLsizei width, GLenum format, GLsizei imageSize, const void *data)" \
  "void glCompressedTextureSubImage2D (GLuint texture, GLint level, GLint xoffset, GLint yoffset, GLsizei width, GLsizei height, GLenum format, GLsizei imageSize, const void *data)" \
  "void glCompressedTextureSubImage3D (GLuint texture, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, GLenum format, GLsizei imageSize, const void *data)" \
  "void glCopyNamedBufferSubData (GLuint readBuffer, GLuint writeBuffer, GLintptr readOffset, GLintptr writeOffset, GLsizeiptr size)" \
  "void glCopyTextureSubImage1D (GLuint texture, GLint level, GLint xoffset, GLint x, GLint y, GLsizei width)" \
  "void glCopyTextureSubImage2D (GLuint texture, GLint level, GLint xoffset, GLint yoffset, GLint x, GLint y, GLsizei width, GLsizei height)" \
  "void glCopyTextureSubImage3D (GLuint texture, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLint x, GLint y, GLsizei width, GLsizei height)" \
  "void glCreateBuffers (GLsizei n, GLuint* buffers)" \
  "void glCreateFramebuffers (GLsizei n, GLuint* framebuffers)" \
  "void glCreateProgramPipelines (GLsizei n, GLuint* pipelines)" \
  "void glCreateQueries (GLenum target, GLsizei n, GLuint* ids)" \
  "void glCreateRenderbuffers (GLsizei n, GLuint* renderbuffers)" \
  "void glCreateSamplers (GLsizei n, GLuint* samplers)" \
  "void glCreateTextures (GLenum target, GLsizei n, GLuint* textures)" \
  "void glCreateTransformFeedbacks (GLsizei n, GLuint* ids)" \
  "void glCreateVertexArrays (GLsizei n, GLuint* arrays)" \
  "void glDisableVertexArrayAttrib (GLuint vaobj, GLuint index)" \
  "void glEnableVertexArrayAttrib (GLuint vaobj, GLuint index)" \
  "void glFlushMappedNamedBufferRange (GLuint buffer, GLintptr offset, GLsizeiptr length)" \
  "void glGenerateTextureMipmap (GLuint texture)" \
  "void glGetCompressedTextureImage (GLuint texture, GLint level, GLsizei bufSize, void *pixels)" \
  "void glGetNamedBufferParameteri64v (GLuint buffer, GLenum pname, GLint64* params)" \
  "void glGetNamedBufferParameteriv (GLuint buffer, GLenum pname, GLint* params)" \
  "void glGetNamedBufferPointerv (GLuint buffer, GLenum pname, void** params)" \
  "void glGetNamedBufferSubData (GLuint buffer, GLintptr offset, GLsizeiptr size, void *data)" \
  "void glGetNamedFramebufferAttachmentParameteriv (GLuint framebuffer, GLenum attachment, GLenum pname, GLint* params)" \
  "void glGetNamedFramebufferParameteriv (GLuint framebuffer, GLenum pname, GLint* param)" \
  "void glGetNamedRenderbufferParameteriv (GLuint renderbuffer, GLenum pname, GLint* params)" \
  "void glGetQueryBufferObjecti64v (GLuint id, GLuint buffer, GLenum pname, GLintptr offset)" \
  "void glGetQueryBufferObjectiv (GLuint id, GLuint buffer, GLenum pname, GLintptr offset)" \
  "void glGetQueryBufferObjectui64v (GLuint id, GLuint buffer, GLenum pname, GLintptr offset)" \
  "void glGetQueryBufferObjectuiv (GLuint id, GLuint buffer, GLenum pname, GLintptr offset)" \
  "void glGetTextureImage (GLuint texture, GLint level, GLenum format, GLenum type, GLsizei bufSize, void *pixels)" \
  "void glGetTextureLevelParameterfv (GLuint texture, GLint level, GLenum pname, GLfloat* params)" \
  "void glGetTextureLevelParameteriv (GLuint texture, GLint level, GLenum pname, GLint* params)" \
  "void glGetTextureParameterIiv (GLuint texture, GLenum pname, GLint* params)" \
  "void glGetTextureParameterIuiv (GLuint texture, GLenum pname, GLuint* params)" \
  "void glGetTextureParameterfv (GLuint texture, GLenum pname, GLfloat* params)" \
  "void glGetTextureParameteriv (GLuint texture, GLenum pname, GLint* params)" \
  "void glGetTransformFeedbacki64_v (GLuint xfb, GLenum pname, GLuint index, GLint64* param)" \
  "void glGetTransformFeedbacki_v (GLuint xfb, GLenum pname, GLuint index, GLint* param)" \
  "void glGetTransformFeedbackiv (GLuint xfb, GLenum pname, GLint* param)" \
  "void glGetVertexArrayIndexed64iv (GLuint vaobj, GLuint index, GLenum pname, GLint64* param)" \
  "void glGetVertexArrayIndexediv (GLuint vaobj, GLuint index, GLenum pname, GLint* param)" \
  "void glGetVertexArrayiv (GLuint vaobj, GLenum pname, GLint* param)" \
  "void glInvalidateNamedFramebufferData (GLuint framebuffer, GLsizei numAttachments, const GLenum* attachments)" \
  "void glInvalidateNamedFramebufferSubData (GLuint framebuffer, GLsizei numAttachments, const GLenum* attachments, GLint x, GLint y, GLsizei width, GLsizei height)" \
  "void * glMapNamedBuffer (GLuint buffer, GLenum access)" \
  "void * glMapNamedBufferRange (GLuint buffer, GLintptr offset, GLsizeiptr length, GLbitfield access)" \
  "void glNamedBufferData (GLuint buffer, GLsizeiptr size, const void *data, GLenum usage)" \
  "void glNamedBufferStorage (GLuint buffer, GLsizeiptr size, const void *data, GLbitfield flags)" \
  "void glNamedBufferSubData (GLuint buffer, GLintptr offset, GLsizeiptr size, const void *data)" \
  "void glNamedFramebufferDrawBuffer (GLuint framebuffer, GLenum mode)" \
  "void glNamedFramebufferDrawBuffers (GLuint framebuffer, GLsizei n, const GLenum* bufs)" \
  "void glNamedFramebufferParameteri (GLuint framebuffer, GLenum pname, GLint param)" \
  "void glNamedFramebufferReadBuffer (GLuint framebuffer, GLenum mode)" \
  "void glNamedFramebufferRenderbuffer (GLuint framebuffer, GLenum attachment, GLenum renderbuffertarget, GLuint renderbuffer)" \
  "void glNamedFramebufferTexture (GLuint framebuffer, GLenum attachment, GLuint texture, GLint level)" \
  "void glNamedFramebufferTextureLayer (GLuint framebuffer, GLenum attachment, GLuint texture, GLint level, GLint layer)" \
  "void glNamedRenderbufferStorage (GLuint renderbuffer, GLenum internalformat, GLsizei width, GLsizei height)" \
  "void glNamedRenderbufferStorageMultisample (GLuint renderbuffer, GLsizei samples, GLenum internalformat, GLsizei width, GLsizei height)" \
  "void glTextureBuffer (GLuint texture, GLenum internalformat, GLuint buffer)" \
  "void glTextureBufferRange (GLuint texture, GLenum internalformat, GLuint buffer, GLintptr offset, GLsizeiptr size)" \
  "void glTextureParameterIiv (GLuint texture, GLenum pname, const GLint* params)" \
  "void glTextureParameterIuiv (GLuint texture, GLenum pname, const GLuint* params)" \
  "void glTextureParameterf (GLuint texture, GLenum pname, GLfloat param)" \
  "void glTextureParameterfv (GLuint texture, GLenum pname, const GLfloat* param)" \
  "void glTextureParameteri (GLuint texture, GLenum pname, GLint param)" \
  "void glTextureParameteriv (GLuint texture, GLenum pname, const GLint* param)" \
  "void glTextureStorage1D (GLuint texture, GLsizei levels, GLenum internalformat, GLsizei width)" \
  "void glTextureStorage2D (GLuint texture, GLsizei levels, GLenum internalformat, GLsizei width, GLsizei height)" \
  "void glTextureStorage2DMultisample (GLuint texture, GLsizei samples, GLenum internalformat, GLsizei width, GLsizei height, GLboolean fixedsamplelocations)" \
  "void glTextureStorage3D (GLuint texture, GLsizei levels, GLenum internalformat, GLsizei width, GLsizei height, GLsizei depth)" \
  "void glTextureStorage3DMultisample (GLuint texture, GLsizei samples, GLenum internalformat, GLsizei width, GLsizei height, GLsizei depth, GLboolean fixedsamplelocations)" \
  "void glTextureSubImage1D (GLuint texture, GLint level, GLint xoffset, GLsizei width, GLenum format, GLenum type, const void *pixels)" \
  "void glTextureSubImage2D (GLuint texture, GLint level, GLint xoffset, GLint yoffset, GLsizei width, GLsizei height, GLenum format, GLenum type, const void *pixels)" \
  "void glTextureSubImage3D (GLuint texture, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, GLenum format, GLenum type, const void *pixels)" \
  "void glTransformFeedbackBufferBase (GLuint xfb, GLuint index, GLuint buffer)" \
  "void glTransformFeedbackBufferRange (GLuint xfb, GLuint index, GLuint buffer, GLintptr offset, GLsizeiptr size)" \
  "GLboolean glUnmapNamedBuffer (GLuint buffer)" \
  "void glVertexArrayAttribBinding (GLuint vaobj, GLuint attribindex, GLuint bindingindex)" \
  "void glVertexArrayAttribFormat (GLuint vaobj, GLuint attribindex, GLint size, GLenum type, GLboolean normalized, GLuint relativeoffset)" \
  "void glVertexArrayAttribIFormat (GLuint vaobj, GLuint attribindex, GLint size, GLenum type, GLuint relativeoffset)" \
  "void glVertexArrayAttribLFormat (GLuint vaobj, GLuint attribindex, GLint size, GLenum type, GLuint relativeoffset)" \
  "void glVertexArrayBindingDivisor (GLuint vaobj, GLuint bindingindex, GLuint divisor)" \
  "void glVertexArrayElementBuffer (GLuint vaobj, GLuint buffer)" \
  "void glVertexArrayVertexBuffer (GLuint vaobj, GLuint bindingindex, GLuint buffer, GLintptr offset, GLsizei stride)" \
  "void glVertexArrayVertexBuffers (GLuint vaobj, GLuint first, GLsizei count, const GLuint* buffers, const GLintptr *offsets, const GLsizei *strides)" \
  "void glDrawBuffersARB (GLsizei n, const GLenum* bufs)" \
  "void glBlendEquationSeparateiARB (GLuint buf, GLenum modeRGB, GLenum modeAlpha)" \
  "void glBlendEquationiARB (GLuint buf, GLenum mode)" \
  "void glBlendFuncSeparateiARB (GLuint buf, GLenum srcRGB, GLenum dstRGB, GLenum srcAlpha, GLenum dstAlpha)" \
  "void glBlendFunciARB (GLuint buf, GLenum src, GLenum dst)" \
  "void glDrawElementsBaseVertex (GLenum mode, GLsizei count, GLenum type, const void *indices, GLint basevertex)" \
  "void glDrawElementsInstancedBaseVertex (GLenum mode, GLsizei count, GLenum type, const void *indices, GLsizei instancecount, GLint basevertex)" \
  "void glDrawRangeElementsBaseVertex (GLenum mode, GLuint start, GLuint end, GLsizei count, GLenum type, const void *indices, GLint basevertex)" \
  "void glMultiDrawElementsBaseVertex (GLenum mode, const GLsizei* count, GLenum type, const void*const *indices, GLsizei drawcount, const GLint *basevertex)" \
  "void glDrawArraysIndirect (GLenum mode, const void *indirect)" \
  "void glDrawElementsIndirect (GLenum mode, GLenum type, const void *indirect)" \
  "void glFramebufferParameteri (GLenum target, GLenum pname, GLint param)" \
  "void glGetFramebufferParameteriv (GLenum target, GLenum pname, GLint* params)" \
  "void glGetNamedFramebufferParameterivEXT (GLuint framebuffer, GLenum pname, GLint* params)" \
  "void glNamedFramebufferParameteriEXT (GLuint framebuffer, GLenum pname, GLint param)" \
  "void glBindFramebuffer (GLenum target, GLuint framebuffer)" \
  "void glBindRenderbuffer (GLenum target, GLuint renderbuffer)" \
  "void glBlitFramebuffer (GLint srcX0, GLint srcY0, GLint srcX1, GLint srcY1, GLint dstX0, GLint dstY0, GLint dstX1, GLint dstY1, GLbitfield mask, GLenum filter)" \
  "GLenum glCheckFramebufferStatus (GLenum target)" \
  "void glDeleteFramebuffers (GLsizei n, const GLuint* framebuffers)" \
  "void glDeleteRenderbuffers (GLsizei n, const GLuint* renderbuffers)" \
  "void glFramebufferRenderbuffer (GLenum target, GLenum attachment, GLenum renderbuffertarget, GLuint renderbuffer)" \
  "void glFramebufferTexture1D (GLenum target, GLenum attachment, GLenum textarget, GLuint texture, GLint level)" \
  "void glFramebufferTexture2D (GLenum target, GLenum attachment, GLenum textarget, GLuint texture, GLint level)" \
  "void glFramebufferTexture3D (GLenum target, GLenum attachment, GLenum textarget, GLuint texture, GLint level, GLint layer)" \
  "void glFramebufferTextureLayer (GLenum target,GLenum attachment, GLuint texture,GLint level,GLint layer)" \
  "void glGenFramebuffers (GLsizei n, GLuint* framebuffers)" \
  "void glGenRenderbuffers (GLsizei n, GLuint* renderbuffers)" \
  "void glGenerateMipmap (GLenum target)" \
  "void glGetFramebufferAttachmentParameteriv (GLenum target, GLenum attachment, GLenum pname, GLint* params)" \
  "void glGetRenderbufferParameteriv (GLenum target, GLenum pname, GLint* params)" \
  "GLboolean glIsFramebuffer (GLuint framebuffer)" \
  "GLboolean glIsRenderbuffer (GLuint renderbuffer)" \
  "void glRenderbufferStorage (GLenum target, GLenum internalformat, GLsizei width, GLsizei height)" \
  "void glRenderbufferStorageMultisample (GLenum target, GLsizei samples, GLenum internalformat, GLsizei width, GLsizei height)" \
  "void glFramebufferTextureARB (GLenum target, GLenum attachment, GLuint texture, GLint level)" \
  "void glFramebufferTextureFaceARB (GLenum target, GLenum attachment, GLuint texture, GLint level, GLenum face)" \
  "void glFramebufferTextureLayerARB (GLenum target, GLenum attachment, GLuint texture, GLint level, GLint layer)" \
  "void glProgramParameteriARB (GLuint program, GLenum pname, GLint value)" \
  "void glGetProgramBinary (GLuint program, GLsizei bufSize, GLsizei* length, GLenum *binaryFormat, void*binary)" \
  "void glProgramBinary (GLuint program, GLenum binaryFormat, const void *binary, GLsizei length)" \
  "void glProgramParameteri (GLuint program, GLenum pname, GLint value)" \
  "void glGetCompressedTextureSubImage (GLuint texture, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, GLsizei bufSize, void *pixels)" \
  "void glGetTextureSubImage (GLuint texture, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, GLenum format, GLenum type, GLsizei bufSize, void *pixels)" \
  "void glSpecializeShaderARB (GLuint shader, const GLchar* pEntryPoint, GLuint numSpecializationConstants, const GLuint* pConstantIndex, const GLuint* pConstantValue)" \
  "void glGetUniformdv (GLuint program, GLint location, GLdouble* params)" \
  "void glUniform1d (GLint location, GLdouble x)" \
  "void glUniform1dv (GLint location, GLsizei count, const GLdouble* value)" \
  "void glUniform2d (GLint location, GLdouble x, GLdouble y)" \
  "void glUniform2dv (GLint location, GLsizei count, const GLdouble* value)" \
  "void glUniform3d (GLint location, GLdouble x, GLdouble y, GLdouble z)" \
  "void glUniform3dv (GLint location, GLsizei count, const GLdouble* value)" \
  "void glUniform4d (GLint location, GLdouble x, GLdouble y, GLdouble z, GLdouble w)" \
  "void glUniform4dv (GLint location, GLsizei count, const GLdouble* value)" \
  "void glUniformMatrix2dv (GLint location, GLsizei count, GLboolean transpose, const GLdouble* value)" \
  "void glUniformMatrix2x3dv (GLint location, GLsizei count, GLboolean transpose, const GLdouble* value)" \
  "void glUniformMatrix2x4dv (GLint location, GLsizei count, GLboolean transpose, const GLdouble* value)" \
  "void glUniformMatrix3dv (GLint location, GLsizei count, GLboolean transpose, const GLdouble* value)" \
  "void glUniformMatrix3x2dv (GLint location, GLsizei count, GLboolean transpose, const GLdouble* value)" \
  "void glUniformMatrix3x4dv (GLint location, GLsizei count, GLboolean transpose, const GLdouble* value)" \
  "void glUniformMatrix4dv (GLint location, GLsizei count, GLboolean transpose, const GLdouble* value)" \
  "void glUniformMatrix4x2dv (GLint location, GLsizei count, GLboolean transpose, const GLdouble* value)" \
  "void glUniformMatrix4x3dv (GLint location, GLsizei count, GLboolean transpose, const GLdouble* value)" \
  "void glGetUniformi64vARB (GLuint program, GLint location, GLint64* params)" \
  "void glGetUniformui64vARB (GLuint program, GLint location, GLuint64* params)" \
  "void glGetnUniformi64vARB (GLuint program, GLint location, GLsizei bufSize, GLint64* params)" \
  "void glGetnUniformui64vARB (GLuint program, GLint location, GLsizei bufSize, GLuint64* params)" \
  "void glProgramUniform1i64ARB (GLuint program, GLint location, GLint64 x)" \
  "void glProgramUniform1i64vARB (GLuint program, GLint location, GLsizei count, const GLint64* value)" \
  "void glProgramUniform1ui64ARB (GLuint program, GLint location, GLuint64 x)" \
  "void glProgramUniform1ui64vARB (GLuint program, GLint location, GLsizei count, const GLuint64* value)" \
  "void glProgramUniform2i64ARB (GLuint program, GLint location, GLint64 x, GLint64 y)" \
  "void glProgramUniform2i64vARB (GLuint program, GLint location, GLsizei count, const GLint64* value)" \
  "void glProgramUniform2ui64ARB (GLuint program, GLint location, GLuint64 x, GLuint64 y)" \
  "void glProgramUniform2ui64vARB (GLuint program, GLint location, GLsizei count, const GLuint64* value)" \
  "void glProgramUniform3i64ARB (GLuint program, GLint location, GLint64 x, GLint64 y, GLint64 z)" \
  "void glProgramUniform3i64vARB (GLuint program, GLint location, GLsizei count, const GLint64* value)" \
  "void glProgramUniform3ui64ARB (GLuint program, GLint location, GLuint64 x, GLuint64 y, GLuint64 z)" \
  "void glProgramUniform3ui64vARB (GLuint program, GLint location, GLsizei count, const GLuint64* value)" \
  "void glProgramUniform4i64ARB (GLuint program, GLint location, GLint64 x, GLint64 y, GLint64 z, GLint64 w)" \
  "void glProgramUniform4i64vARB (GLuint program, GLint location, GLsizei count, const GLint64* value)" \
  "void glProgramUniform4ui64ARB (GLuint program, GLint location, GLuint64 x, GLuint64 y, GLuint64 z, GLuint64 w)" \
  "void glProgramUniform4ui64vARB (GLuint program, GLint location, GLsizei count, const GLuint64* value)" \
  "void glUniform1i64ARB (GLint location, GLint64 x)" \
  "void glUniform1i64vARB (GLint location, GLsizei count, const GLint64* value)" \
  "void glUniform1ui64ARB (GLint location, GLuint64 x)" \
  "void glUniform1ui64vARB (GLint location, GLsizei count, const GLuint64* value)" \
  "void glUniform2i64ARB (GLint location, GLint64 x, GLint64 y)" \
  "void glUniform2i64vARB (GLint location, GLsizei count, const GLint64* value)" \
  "void glUniform2ui64ARB (GLint location, GLuint64 x, GLuint64 y)" \
  "void glUniform2ui64vARB (GLint location, GLsizei count, const GLuint64* value)" \
  "void glUniform3i64ARB (GLint location, GLint64 x, GLint64 y, GLint64 z)" \
  "void glUniform3i64vARB (GLint location, GLsizei count, const GLint64* value)" \
  "void glUniform3ui64ARB (GLint location, GLuint64 x, GLuint64 y, GLuint64 z)" \
  "void glUniform3ui64vARB (GLint location, GLsizei count, const GLuint64* value)" \
  "void glUniform4i64ARB (GLint location, GLint64 x, GLint64 y, GLint64 z, GLint64 w)" \
  "void glUniform4i64vARB (GLint location, GLsizei count, const GLint64* value)" \
  "void glUniform4ui64ARB (GLint location, GLuint64 x, GLuint64 y, GLuint64 z, GLuint64 w)" \
  "void glUniform4ui64vARB (GLint location, GLsizei count, const GLuint64* value)" \
  "void glColorSubTable (GLenum target, GLsizei start, GLsizei count, GLenum format, GLenum type, const void *data)" \
  "void glColorTable (GLenum target, GLenum internalformat, GLsizei width, GLenum format, GLenum type, const void *table)" \
  "void glColorTableParameterfv (GLenum target, GLenum pname, const GLfloat *params)" \
  "void glColorTableParameteriv (GLenum target, GLenum pname, const GLint *params)" \
  "void glConvolutionFilter1D (GLenum target, GLenum internalformat, GLsizei width, GLenum format, GLenum type, const void *image)" \
  "void glConvolutionFilter2D (GLenum target, GLenum internalformat, GLsizei width, GLsizei height, GLenum format, GLenum type, const void *image)" \
  "void glConvolutionParameterf (GLenum target, GLenum pname, GLfloat params)" \
  "void glConvolutionParameterfv (GLenum target, GLenum pname, const GLfloat *params)" \
  "void glConvolutionParameteri (GLenum target, GLenum pname, GLint params)" \
  "void glConvolutionParameteriv (GLenum target, GLenum pname, const GLint *params)" \
  "void glCopyColorSubTable (GLenum target, GLsizei start, GLint x, GLint y, GLsizei width)" \
  "void glCopyColorTable (GLenum target, GLenum internalformat, GLint x, GLint y, GLsizei width)" \
  "void glCopyConvolutionFilter1D (GLenum target, GLenum internalformat, GLint x, GLint y, GLsizei width)" \
  "void glCopyConvolutionFilter2D (GLenum target, GLenum internalformat, GLint x, GLint y, GLsizei width, GLsizei height)" \
  "void glGetColorTable (GLenum target, GLenum format, GLenum type, void *table)" \
  "void glGetColorTableParameterfv (GLenum target, GLenum pname, GLfloat *params)" \
  "void glGetColorTableParameteriv (GLenum target, GLenum pname, GLint *params)" \
  "void glGetConvolutionFilter (GLenum target, GLenum format, GLenum type, void *image)" \
  "void glGetConvolutionParameterfv (GLenum target, GLenum pname, GLfloat *params)" \
  "void glGetConvolutionParameteriv (GLenum target, GLenum pname, GLint *params)" \
  "void glGetHistogram (GLenum target, GLboolean reset, GLenum format, GLenum type, void *values)" \
  "void glGetHistogramParameterfv (GLenum target, GLenum pname, GLfloat *params)" \
  "void glGetHistogramParameteriv (GLenum target, GLenum pname, GLint *params)" \
  "void glGetMinmax (GLenum target, GLboolean reset, GLenum format, GLenum types, void *values)" \
  "void glGetMinmaxParameterfv (GLenum target, GLenum pname, GLfloat *params)" \
  "void glGetMinmaxParameteriv (GLenum target, GLenum pname, GLint *params)" \
  "void glGetSeparableFilter (GLenum target, GLenum format, GLenum type, void *row, void *column, void *span)" \
  "void glHistogram (GLenum target, GLsizei width, GLenum internalformat, GLboolean sink)" \
  "void glMinmax (GLenum target, GLenum internalformat, GLboolean sink)" \
  "void glResetHistogram (GLenum target)" \
  "void glResetMinmax (GLenum target)" \
  "void glSeparableFilter2D (GLenum target, GLenum internalformat, GLsizei width, GLsizei height, GLenum format, GLenum type, const void *row, const void *column)" \
  "void glMultiDrawArraysIndirectCountARB (GLenum mode, const void *indirect, GLintptr drawcount, GLsizei maxdrawcount, GLsizei stride)" \
  "void glMultiDrawElementsIndirectCountARB (GLenum mode, GLenum type, const void *indirect, GLintptr drawcount, GLsizei maxdrawcount, GLsizei stride)" \
  "void glDrawArraysInstancedARB (GLenum mode, GLint first, GLsizei count, GLsizei primcount)" \
  "void glDrawElementsInstancedARB (GLenum mode, GLsizei count, GLenum type, const void* indices, GLsizei primcount)" \
  "void glVertexAttribDivisorARB (GLuint index, GLuint divisor)" \
  "void glGetInternalformativ (GLenum target, GLenum internalformat, GLenum pname, GLsizei bufSize, GLint* params)" \
  "void glGetInternalformati64v (GLenum target, GLenum internalformat, GLenum pname, GLsizei bufSize, GLint64* params)" \
  "void glInvalidateBufferData (GLuint buffer)" \
  "void glInvalidateBufferSubData (GLuint buffer, GLintptr offset, GLsizeiptr length)" \
  "void glInvalidateFramebuffer (GLenum target, GLsizei numAttachments, const GLenum* attachments)" \
  "void glInvalidateSubFramebuffer (GLenum target, GLsizei numAttachments, const GLenum* attachments, GLint x, GLint y, GLsizei width, GLsizei height)" \
  "void glInvalidateTexImage (GLuint texture, GLint level)" \
  "void glInvalidateTexSubImage (GLuint texture, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLsizei width, GLsizei height, GLsizei depth)" \
  "void glFlushMappedBufferRange (GLenum target, GLintptr offset, GLsizeiptr length)" \
  "void * glMapBufferRange (GLenum target, GLintptr offset, GLsizeiptr length, GLbitfield access)" \
  "void glCurrentPaletteMatrixARB (GLint index)" \
  "void glMatrixIndexPointerARB (GLint size, GLenum type, GLsizei stride, void *pointer)" \
  "void glMatrixIndexubvARB (GLint size, GLubyte *indices)" \
  "void glMatrixIndexuivARB (GLint size, GLuint *indices)" \
  "void glMatrixIndexusvARB (GLint size, GLushort *indices)" \
  "void glBindBuffersBase (GLenum target, GLuint first, GLsizei count, const GLuint* buffers)" \
  "void glBindBuffersRange (GLenum target, GLuint first, GLsizei count, const GLuint* buffers, const GLintptr *offsets, const GLsizeiptr *sizes)" \
  "void glBindImageTextures (GLuint first, GLsizei count, const GLuint* textures)" \
  "void glBindSamplers (GLuint first, GLsizei count, const GLuint* samplers)" \
  "void glBindTextures (GLuint first, GLsizei count, const GLuint* textures)" \
  "void glBindVertexBuffers (GLuint first, GLsizei count, const GLuint* buffers, const GLintptr *offsets, const GLsizei *strides)" \
  "void glMultiDrawArraysIndirect (GLenum mode, const void *indirect, GLsizei primcount, GLsizei stride)" \
  "void glMultiDrawElementsIndirect (GLenum mode, GLenum type, const void *indirect, GLsizei primcount, GLsizei stride)" \
  "void glSampleCoverageARB (GLclampf value, GLboolean invert)" \
  "void glActiveTextureARB (GLenum texture)" \
  "void glClientActiveTextureARB (GLenum texture)" \
  "void glMultiTexCoord1dARB (GLenum target, GLdouble s)" \
  "void glMultiTexCoord1dvARB (GLenum target, const GLdouble *v)" \
  "void glMultiTexCoord1fARB (GLenum target, GLfloat s)" \
  "void glMultiTexCoord1fvARB (GLenum target, const GLfloat *v)" \
  "void glMultiTexCoord1iARB (GLenum target, GLint s)" \
  "void glMultiTexCoord1ivARB (GLenum target, const GLint *v)" \
  "void glMultiTexCoord1sARB (GLenum target, GLshort s)" \
  "void glMultiTexCoord1svARB (GLenum target, const GLshort *v)" \
  "void glMultiTexCoord2dARB (GLenum target, GLdouble s, GLdouble t)" \
  "void glMultiTexCoord2dvARB (GLenum target, const GLdouble *v)" \
  "void glMultiTexCoord2fARB (GLenum target, GLfloat s, GLfloat t)" \
  "void glMultiTexCoord2fvARB (GLenum target, const GLfloat *v)" \
  "void glMultiTexCoord2iARB (GLenum target, GLint s, GLint t)" \
  "void glMultiTexCoord2ivARB (GLenum target, const GLint *v)" \
  "void glMultiTexCoord2sARB (GLenum target, GLshort s, GLshort t)" \
  "void glMultiTexCoord2svARB (GLenum target, const GLshort *v)" \
  "void glMultiTexCoord3dARB (GLenum target, GLdouble s, GLdouble t, GLdouble r)" \
  "void glMultiTexCoord3dvARB (GLenum target, const GLdouble *v)" \
  "void glMultiTexCoord3fARB (GLenum target, GLfloat s, GLfloat t, GLfloat r)" \
  "void glMultiTexCoord3fvARB (GLenum target, const GLfloat *v)" \
  "void glMultiTexCoord3iARB (GLenum target, GLint s, GLint t, GLint r)" \
  "void glMultiTexCoord3ivARB (GLenum target, const GLint *v)" \
  "void glMultiTexCoord3sARB (GLenum target, GLshort s, GLshort t, GLshort r)" \
  "void glMultiTexCoord3svARB (GLenum target, const GLshort *v)" \
  "void glMultiTexCoord4dARB (GLenum target, GLdouble s, GLdouble t, GLdouble r, GLdouble q)" \
  "void glMultiTexCoord4dvARB (GLenum target, const GLdouble *v)" \
  "void glMultiTexCoord4fARB (GLenum target, GLfloat s, GLfloat t, GLfloat r, GLfloat q)" \
  "void glMultiTexCoord4fvARB (GLenum target, const GLfloat *v)" \
  "void glMultiTexCoord4iARB (GLenum target, GLint s, GLint t, GLint r, GLint q)" \
  "void glMultiTexCoord4ivARB (GLenum target, const GLint *v)" \
  "void glMultiTexCoord4sARB (GLenum target, GLshort s, GLshort t, GLshort r, GLshort q)" \
  "void glMultiTexCoord4svARB (GLenum target, const GLshort *v)" \
  "void glBeginQueryARB (GLenum target, GLuint id)" \
  "void glDeleteQueriesARB (GLsizei n, const GLuint* ids)" \
  "void glEndQueryARB (GLenum target)" \
  "void glGenQueriesARB (GLsizei n, GLuint* ids)" \
  "void glGetQueryObjectivARB (GLuint id, GLenum pname, GLint* params)" \
  "void glGetQueryObjectuivARB (GLuint id, GLenum pname, GLuint* params)" \
  "void glGetQueryivARB (GLenum target, GLenum pname, GLint* params)" \
  "GLboolean glIsQueryARB (GLuint id)" \
  "void glMaxShaderCompilerThreadsARB (GLuint count)" \
  "void glPointParameterfARB (GLenum pname, GLfloat param)" \
  "void glPointParameterfvARB (GLenum pname, const GLfloat* params)" \
  "void glPolygonOffsetClamp (GLfloat factor, GLfloat units, GLfloat clamp)" \
  "void glGetProgramInterfaceiv (GLuint program, GLenum programInterface, GLenum pname, GLint* params)" \
  "GLuint glGetProgramResourceIndex (GLuint program, GLenum programInterface, const GLchar* name)" \
  "GLint glGetProgramResourceLocation (GLuint program, GLenum programInterface, const GLchar* name)" \
  "GLint glGetProgramResourceLocationIndex (GLuint program, GLenum programInterface, const GLchar* name)" \
  "void glGetProgramResourceName (GLuint program, GLenum programInterface, GLuint index, GLsizei bufSize, GLsizei* length, GLchar *name)" \
  "void glGetProgramResourceiv (GLuint program, GLenum programInterface, GLuint index, GLsizei propCount, const GLenum* props, GLsizei bufSize, GLsizei *length, GLint *params)" \
  "void glProvokingVertex (GLenum mode)" \
  "GLenum glGetGraphicsResetStatusARB (void)" \
  "void glGetnColorTableARB (GLenum target, GLenum format, GLenum type, GLsizei bufSize, void* table)" \
  "void glGetnCompressedTexImageARB (GLenum target, GLint lod, GLsizei bufSize, void* img)" \
  "void glGetnConvolutionFilterARB (GLenum target, GLenum format, GLenum type, GLsizei bufSize, void* image)" \
  "void glGetnHistogramARB (GLenum target, GLboolean reset, GLenum format, GLenum type, GLsizei bufSize, void* values)" \
  "void glGetnMapdvARB (GLenum target, GLenum query, GLsizei bufSize, GLdouble* v)" \
  "void glGetnMapfvARB (GLenum target, GLenum query, GLsizei bufSize, GLfloat* v)" \
  "void glGetnMapivARB (GLenum target, GLenum query, GLsizei bufSize, GLint* v)" \
  "void glGetnMinmaxARB (GLenum target, GLboolean reset, GLenum format, GLenum type, GLsizei bufSize, void* values)" \
  "void glGetnPixelMapfvARB (GLenum map, GLsizei bufSize, GLfloat* values)" \
  "void glGetnPixelMapuivARB (GLenum map, GLsizei bufSize, GLuint* values)" \
  "void glGetnPixelMapusvARB (GLenum map, GLsizei bufSize, GLushort* values)" \
  "void glGetnPolygonStippleARB (GLsizei bufSize, GLubyte* pattern)" \
  "void glGetnSeparableFilterARB (GLenum target, GLenum format, GLenum type, GLsizei rowBufSize, void* row, GLsizei columnBufSize, void*column, void*span)" \
  "void glGetnTexImageARB (GLenum target, GLint level, GLenum format, GLenum type, GLsizei bufSize, void* img)" \
  "void glGetnUniformdvARB (GLuint program, GLint location, GLsizei bufSize, GLdouble* params)" \
  "void glGetnUniformfvARB (GLuint program, GLint location, GLsizei bufSize, GLfloat* params)" \
  "void glGetnUniformivARB (GLuint program, GLint location, GLsizei bufSize, GLint* params)" \
  "void glGetnUniformuivARB (GLuint program, GLint location, GLsizei bufSize, GLuint* params)" \
  "void glReadnPixelsARB (GLint x, GLint y, GLsizei width, GLsizei height, GLenum format, GLenum type, GLsizei bufSize, void* data)" \
  "void glFramebufferSampleLocationsfvARB (GLenum target, GLuint start, GLsizei count, const GLfloat* v)" \
  "void glNamedFramebufferSampleLocationsfvARB (GLuint framebuffer, GLuint start, GLsizei count, const GLfloat* v)" \
  "void glMinSampleShadingARB (GLclampf value)" \
  "void glBindSampler (GLuint unit, GLuint sampler)" \
  "void glDeleteSamplers (GLsizei count, const GLuint * samplers)" \
  "void glGenSamplers (GLsizei count, GLuint* samplers)" \
  "void glGetSamplerParameterIiv (GLuint sampler, GLenum pname, GLint* params)" \
  "void glGetSamplerParameterIuiv (GLuint sampler, GLenum pname, GLuint* params)" \
  "void glGetSamplerParameterfv (GLuint sampler, GLenum pname, GLfloat* params)" \
  "void glGetSamplerParameteriv (GLuint sampler, GLenum pname, GLint* params)" \
  "GLboolean glIsSampler (GLuint sampler)" \
  "void glSamplerParameterIiv (GLuint sampler, GLenum pname, const GLint* params)" \
  "void glSamplerParameterIuiv (GLuint sampler, GLenum pname, const GLuint* params)" \
  "void glSamplerParameterf (GLuint sampler, GLenum pname, GLfloat param)" \
  "void glSamplerParameterfv (GLuint sampler, GLenum pname, const GLfloat* params)" \
  "void glSamplerParameteri (GLuint sampler, GLenum pname, GLint param)" \
  "void glSamplerParameteriv (GLuint sampler, GLenum pname, const GLint* params)" \
  "void glActiveShaderProgram (GLuint pipeline, GLuint program)" \
  "void glBindProgramPipeline (GLuint pipeline)" \
  "GLuint glCreateShaderProgramv (GLenum type, GLsizei count, const GLchar * const * strings)" \
  "void glDeleteProgramPipelines (GLsizei n, const GLuint* pipelines)" \
  "void glGenProgramPipelines (GLsizei n, GLuint* pipelines)" \
  "void glGetProgramPipelineInfoLog (GLuint pipeline, GLsizei bufSize, GLsizei* length, GLchar *infoLog)" \
  "void glGetProgramPipelineiv (GLuint pipeline, GLenum pname, GLint* params)" \
  "GLboolean glIsProgramPipeline (GLuint pipeline)" \
  "void glProgramUniform1d (GLuint program, GLint location, GLdouble x)" \
  "void glProgramUniform1dv (GLuint program, GLint location, GLsizei count, const GLdouble* value)" \
  "void glProgramUniform1f (GLuint program, GLint location, GLfloat x)" \
  "void glProgramUniform1fv (GLuint program, GLint location, GLsizei count, const GLfloat* value)" \
  "void glProgramUniform1i (GLuint program, GLint location, GLint x)" \
  "void glProgramUniform1iv (GLuint program, GLint location, GLsizei count, const GLint* value)" \
  "void glProgramUniform1ui (GLuint program, GLint location, GLuint x)" \
  "void glProgramUniform1uiv (GLuint program, GLint location, GLsizei count, const GLuint* value)" \
  "void glProgramUniform2d (GLuint program, GLint location, GLdouble x, GLdouble y)" \
  "void glProgramUniform2dv (GLuint program, GLint location, GLsizei count, const GLdouble* value)" \
  "void glProgramUniform2f (GLuint program, GLint location, GLfloat x, GLfloat y)" \
  "void glProgramUniform2fv (GLuint program, GLint location, GLsizei count, const GLfloat* value)" \
  "void glProgramUniform2i (GLuint program, GLint location, GLint x, GLint y)" \
  "void glProgramUniform2iv (GLuint program, GLint location, GLsizei count, const GLint* value)" \
  "void glProgramUniform2ui (GLuint program, GLint location, GLuint x, GLuint y)" \
  "void glProgramUniform2uiv (GLuint program, GLint location, GLsizei count, const GLuint* value)" \
  "void glProgramUniform3d (GLuint program, GLint location, GLdouble x, GLdouble y, GLdouble z)" \
  "void glProgramUniform3dv (GLuint program, GLint location, GLsizei count, const GLdouble* value)" \
  "void glProgramUniform3f (GLuint program, GLint location, GLfloat x, GLfloat y, GLfloat z)" \
  "void glProgramUniform3fv (GLuint program, GLint location, GLsizei count, const GLfloat* value)" \
  "void glProgramUniform3i (GLuint program, GLint location, GLint x, GLint y, GLint z)" \
  "void glProgramUniform3iv (GLuint program, GLint location, GLsizei count, const GLint* value)" \
  "void glProgramUniform3ui (GLuint program, GLint location, GLuint x, GLuint y, GLuint z)" \
  "void glProgramUniform3uiv (GLuint program, GLint location, GLsizei count, const GLuint* value)" \
  "void glProgramUniform4d (GLuint program, GLint location, GLdouble x, GLdouble y, GLdouble z, GLdouble w)" \
  "void glProgramUniform4dv (GLuint program, GLint location, GLsizei count, const GLdouble* value)" \
  "void glProgramUniform4f (GLuint program, GLint location, GLfloat x, GLfloat y, GLfloat z, GLfloat w)" \
  "void glProgramUniform4fv (GLuint program, GLint location, GLsizei count, const GLfloat* value)" \
  "void glProgramUniform4i (GLuint program, GLint location, GLint x, GLint y, GLint z, GLint w)" \
  "void glProgramUniform4iv (GLuint program, GLint location, GLsizei count, const GLint* value)" \
  "void glProgramUniform4ui (GLuint program, GLint location, GLuint x, GLuint y, GLuint z, GLuint w)" \
  "void glProgramUniform4uiv (GLuint program, GLint location, GLsizei count, const GLuint* value)" \
  "void glProgramUniformMatrix2dv (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLdouble* value)" \
  "void glProgramUniformMatrix2fv (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glProgramUniformMatrix2x3dv (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLdouble* value)" \
  "void glProgramUniformMatrix2x3fv (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glProgramUniformMatrix2x4dv (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLdouble* value)" \
  "void glProgramUniformMatrix2x4fv (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glProgramUniformMatrix3dv (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLdouble* value)" \
  "void glProgramUniformMatrix3fv (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glProgramUniformMatrix3x2dv (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLdouble* value)" \
  "void glProgramUniformMatrix3x2fv (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glProgramUniformMatrix3x4dv (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLdouble* value)" \
  "void glProgramUniformMatrix3x4fv (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glProgramUniformMatrix4dv (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLdouble* value)" \
  "void glProgramUniformMatrix4fv (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glProgramUniformMatrix4x2dv (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLdouble* value)" \
  "void glProgramUniformMatrix4x2fv (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glProgramUniformMatrix4x3dv (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLdouble* value)" \
  "void glProgramUniformMatrix4x3fv (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glUseProgramStages (GLuint pipeline, GLbitfield stages, GLuint program)" \
  "void glValidateProgramPipeline (GLuint pipeline)" \
  "void glGetActiveAtomicCounterBufferiv (GLuint program, GLuint bufferIndex, GLenum pname, GLint* params)" \
  "void glBindImageTexture (GLuint unit, GLuint texture, GLint level, GLboolean layered, GLint layer, GLenum access, GLenum format)" \
  "void glMemoryBarrier (GLbitfield barriers)" \
  "void glAttachObjectARB (GLhandleARB containerObj, GLhandleARB obj)" \
  "void glCompileShaderARB (GLhandleARB shaderObj)" \
  "GLhandleARB glCreateProgramObjectARB (void)" \
  "GLhandleARB glCreateShaderObjectARB (GLenum shaderType)" \
  "void glDeleteObjectARB (GLhandleARB obj)" \
  "void glDetachObjectARB (GLhandleARB containerObj, GLhandleARB attachedObj)" \
  "void glGetActiveUniformARB (GLhandleARB programObj, GLuint index, GLsizei maxLength, GLsizei* length, GLint *size, GLenum *type, GLcharARB *name)" \
  "void glGetAttachedObjectsARB (GLhandleARB containerObj, GLsizei maxCount, GLsizei* count, GLhandleARB *obj)" \
  "GLhandleARB glGetHandleARB (GLenum pname)" \
  "void glGetInfoLogARB (GLhandleARB obj, GLsizei maxLength, GLsizei* length, GLcharARB *infoLog)" \
  "void glGetObjectParameterfvARB (GLhandleARB obj, GLenum pname, GLfloat* params)" \
  "void glGetObjectParameterivARB (GLhandleARB obj, GLenum pname, GLint* params)" \
  "void glGetShaderSourceARB (GLhandleARB obj, GLsizei maxLength, GLsizei* length, GLcharARB *source)" \
  "GLint glGetUniformLocationARB (GLhandleARB programObj, const GLcharARB* name)" \
  "void glGetUniformfvARB (GLhandleARB programObj, GLint location, GLfloat* params)" \
  "void glGetUniformivARB (GLhandleARB programObj, GLint location, GLint* params)" \
  "void glLinkProgramARB (GLhandleARB programObj)" \
  "void glShaderSourceARB (GLhandleARB shaderObj, GLsizei count, const GLcharARB ** string, const GLint *length)" \
  "void glUniform1fARB (GLint location, GLfloat v0)" \
  "void glUniform1fvARB (GLint location, GLsizei count, const GLfloat* value)" \
  "void glUniform1iARB (GLint location, GLint v0)" \
  "void glUniform1ivARB (GLint location, GLsizei count, const GLint* value)" \
  "void glUniform2fARB (GLint location, GLfloat v0, GLfloat v1)" \
  "void glUniform2fvARB (GLint location, GLsizei count, const GLfloat* value)" \
  "void glUniform2iARB (GLint location, GLint v0, GLint v1)" \
  "void glUniform2ivARB (GLint location, GLsizei count, const GLint* value)" \
  "void glUniform3fARB (GLint location, GLfloat v0, GLfloat v1, GLfloat v2)" \
  "void glUniform3fvARB (GLint location, GLsizei count, const GLfloat* value)" \
  "void glUniform3iARB (GLint location, GLint v0, GLint v1, GLint v2)" \
  "void glUniform3ivARB (GLint location, GLsizei count, const GLint* value)" \
  "void glUniform4fARB (GLint location, GLfloat v0, GLfloat v1, GLfloat v2, GLfloat v3)" \
  "void glUniform4fvARB (GLint location, GLsizei count, const GLfloat* value)" \
  "void glUniform4iARB (GLint location, GLint v0, GLint v1, GLint v2, GLint v3)" \
  "void glUniform4ivARB (GLint location, GLsizei count, const GLint* value)" \
  "void glUniformMatrix2fvARB (GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glUniformMatrix3fvARB (GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glUniformMatrix4fvARB (GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glUseProgramObjectARB (GLhandleARB programObj)" \
  "void glValidateProgramARB (GLhandleARB programObj)" \
  "void glShaderStorageBlockBinding (GLuint program, GLuint storageBlockIndex, GLuint storageBlockBinding)" \
  "void glGetActiveSubroutineName (GLuint program, GLenum shadertype, GLuint index, GLsizei bufsize, GLsizei* length, GLchar *name)" \
  "void glGetActiveSubroutineUniformName (GLuint program, GLenum shadertype, GLuint index, GLsizei bufsize, GLsizei* length, GLchar *name)" \
  "void glGetActiveSubroutineUniformiv (GLuint program, GLenum shadertype, GLuint index, GLenum pname, GLint* values)" \
  "void glGetProgramStageiv (GLuint program, GLenum shadertype, GLenum pname, GLint* values)" \
  "GLuint glGetSubroutineIndex (GLuint program, GLenum shadertype, const GLchar* name)" \
  "GLint glGetSubroutineUniformLocation (GLuint program, GLenum shadertype, const GLchar* name)" \
  "void glGetUniformSubroutineuiv (GLenum shadertype, GLint location, GLuint* params)" \
  "void glUniformSubroutinesuiv (GLenum shadertype, GLsizei count, const GLuint* indices)" \
  "void glCompileShaderIncludeARB (GLuint shader, GLsizei count, const GLchar* const *path, const GLint *length)" \
  "void glDeleteNamedStringARB (GLint namelen, const GLchar* name)" \
  "void glGetNamedStringARB (GLint namelen, const GLchar* name, GLsizei bufSize, GLint *stringlen, GLchar *string)" \
  "void glGetNamedStringivARB (GLint namelen, const GLchar* name, GLenum pname, GLint *params)" \
  "GLboolean glIsNamedStringARB (GLint namelen, const GLchar* name)" \
  "void glNamedStringARB (GLenum type, GLint namelen, const GLchar* name, GLint stringlen, const GLchar *string)" \
  "void glBufferPageCommitmentARB (GLenum target, GLintptr offset, GLsizeiptr size, GLboolean commit)" \
  "void glTexPageCommitmentARB (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, GLboolean commit)" \
  "GLenum glClientWaitSync (GLsync GLsync,GLbitfield flags,GLuint64 timeout)" \
  "void glDeleteSync (GLsync GLsync)" \
  "GLsync glFenceSync (GLenum condition,GLbitfield flags)" \
  "void glGetInteger64v (GLenum pname, GLint64* params)" \
  "void glGetSynciv (GLsync GLsync,GLenum pname,GLsizei bufSize,GLsizei* length, GLint *values)" \
  "GLboolean glIsSync (GLsync GLsync)" \
  "void glWaitSync (GLsync GLsync,GLbitfield flags,GLuint64 timeout)" \
  "void glPatchParameterfv (GLenum pname, const GLfloat* values)" \
  "void glPatchParameteri (GLenum pname, GLint value)" \
  "void glTextureBarrier (void)" \
  "void glTexBufferARB (GLenum target, GLenum internalformat, GLuint buffer)" \
  "void glTexBufferRange (GLenum target, GLenum internalformat, GLuint buffer, GLintptr offset, GLsizeiptr size)" \
  "void glTextureBufferRangeEXT (GLuint texture, GLenum target, GLenum internalformat, GLuint buffer, GLintptr offset, GLsizeiptr size)" \
  "void glCompressedTexImage1DARB (GLenum target, GLint level, GLenum internalformat, GLsizei width, GLint border, GLsizei imageSize, const void *data)" \
  "void glCompressedTexImage2DARB (GLenum target, GLint level, GLenum internalformat, GLsizei width, GLsizei height, GLint border, GLsizei imageSize, const void *data)" \
  "void glCompressedTexImage3DARB (GLenum target, GLint level, GLenum internalformat, GLsizei width, GLsizei height, GLsizei depth, GLint border, GLsizei imageSize, const void *data)" \
  "void glCompressedTexSubImage1DARB (GLenum target, GLint level, GLint xoffset, GLsizei width, GLenum format, GLsizei imageSize, const void *data)" \
  "void glCompressedTexSubImage2DARB (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLsizei width, GLsizei height, GLenum format, GLsizei imageSize, const void *data)" \
  "void glCompressedTexSubImage3DARB (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, GLenum format, GLsizei imageSize, const void *data)" \
  "void glGetCompressedTexImageARB (GLenum target, GLint lod, void *img)" \
  "void glGetMultisamplefv (GLenum pname, GLuint index, GLfloat* val)" \
  "void glSampleMaski (GLuint index, GLbitfield mask)" \
  "void glTexImage2DMultisample (GLenum target, GLsizei samples, GLenum internalformat, GLsizei width, GLsizei height, GLboolean fixedsamplelocations)" \
  "void glTexImage3DMultisample (GLenum target, GLsizei samples, GLenum internalformat, GLsizei width, GLsizei height, GLsizei depth, GLboolean fixedsamplelocations)" \
  "void glTexStorage1D (GLenum target, GLsizei levels, GLenum internalformat, GLsizei width)" \
  "void glTexStorage2D (GLenum target, GLsizei levels, GLenum internalformat, GLsizei width, GLsizei height)" \
  "void glTexStorage3D (GLenum target, GLsizei levels, GLenum internalformat, GLsizei width, GLsizei height, GLsizei depth)" \
  "void glTexStorage2DMultisample (GLenum target, GLsizei samples, GLenum internalformat, GLsizei width, GLsizei height, GLboolean fixedsamplelocations)" \
  "void glTexStorage3DMultisample (GLenum target, GLsizei samples, GLenum internalformat, GLsizei width, GLsizei height, GLsizei depth, GLboolean fixedsamplelocations)" \
  "void glTextureStorage2DMultisampleEXT (GLuint texture, GLenum target, GLsizei samples, GLenum internalformat, GLsizei width, GLsizei height, GLboolean fixedsamplelocations)" \
  "void glTextureStorage3DMultisampleEXT (GLuint texture, GLenum target, GLsizei samples, GLenum internalformat, GLsizei width, GLsizei height, GLsizei depth, GLboolean fixedsamplelocations)" \
  "void glTextureView (GLuint texture, GLenum target, GLuint origtexture, GLenum internalformat, GLuint minlevel, GLuint numlevels, GLuint minlayer, GLuint numlayers)" \
  "void glGetQueryObjecti64v (GLuint id, GLenum pname, GLint64* params)" \
  "void glGetQueryObjectui64v (GLuint id, GLenum pname, GLuint64* params)" \
  "void glQueryCounter (GLuint id, GLenum target)" \
  "void glBindTransformFeedback (GLenum target, GLuint id)" \
  "void glDeleteTransformFeedbacks (GLsizei n, const GLuint* ids)" \
  "void glDrawTransformFeedback (GLenum mode, GLuint id)" \
  "void glGenTransformFeedbacks (GLsizei n, GLuint* ids)" \
  "GLboolean glIsTransformFeedback (GLuint id)" \
  "void glPauseTransformFeedback (void)" \
  "void glResumeTransformFeedback (void)" \
  "void glBeginQueryIndexed (GLenum target, GLuint index, GLuint id)" \
  "void glDrawTransformFeedbackStream (GLenum mode, GLuint id, GLuint stream)" \
  "void glEndQueryIndexed (GLenum target, GLuint index)" \
  "void glGetQueryIndexediv (GLenum target, GLuint index, GLenum pname, GLint* params)" \
  "void glDrawTransformFeedbackInstanced (GLenum mode, GLuint id, GLsizei primcount)" \
  "void glDrawTransformFeedbackStreamInstanced (GLenum mode, GLuint id, GLuint stream, GLsizei primcount)" \
  "void glLoadTransposeMatrixdARB (GLdouble m\[16\])" \
  "void glLoadTransposeMatrixfARB (GLfloat m\[16\])" \
  "void glMultTransposeMatrixdARB (GLdouble m\[16\])" \
  "void glMultTransposeMatrixfARB (GLfloat m\[16\])" \
  "void glBindBufferBase (GLenum target, GLuint index, GLuint buffer)" \
  "void glBindBufferRange (GLenum target, GLuint index, GLuint buffer, GLintptr offset, GLsizeiptr size)" \
  "void glGetActiveUniformBlockName (GLuint program, GLuint uniformBlockIndex, GLsizei bufSize, GLsizei* length, GLchar* uniformBlockName)" \
  "void glGetActiveUniformBlockiv (GLuint program, GLuint uniformBlockIndex, GLenum pname, GLint* params)" \
  "void glGetActiveUniformName (GLuint program, GLuint uniformIndex, GLsizei bufSize, GLsizei* length, GLchar* uniformName)" \
  "void glGetActiveUniformsiv (GLuint program, GLsizei uniformCount, const GLuint* uniformIndices, GLenum pname, GLint* params)" \
  "void glGetIntegeri_v (GLenum target, GLuint index, GLint* data)" \
  "GLuint glGetUniformBlockIndex (GLuint program, const GLchar* uniformBlockName)" \
  "void glGetUniformIndices (GLuint program, GLsizei uniformCount, const GLchar* const * uniformNames, GLuint* uniformIndices)" \
  "void glUniformBlockBinding (GLuint program, GLuint uniformBlockIndex, GLuint uniformBlockBinding)" \
  "void glBindVertexArray (GLuint array)" \
  "void glDeleteVertexArrays (GLsizei n, const GLuint* arrays)" \
  "void glGenVertexArrays (GLsizei n, GLuint* arrays)" \
  "GLboolean glIsVertexArray (GLuint array)" \
  "void glGetVertexAttribLdv (GLuint index, GLenum pname, GLdouble* params)" \
  "void glVertexAttribL1d (GLuint index, GLdouble x)" \
  "void glVertexAttribL1dv (GLuint index, const GLdouble* v)" \
  "void glVertexAttribL2d (GLuint index, GLdouble x, GLdouble y)" \
  "void glVertexAttribL2dv (GLuint index, const GLdouble* v)" \
  "void glVertexAttribL3d (GLuint index, GLdouble x, GLdouble y, GLdouble z)" \
  "void glVertexAttribL3dv (GLuint index, const GLdouble* v)" \
  "void glVertexAttribL4d (GLuint index, GLdouble x, GLdouble y, GLdouble z, GLdouble w)" \
  "void glVertexAttribL4dv (GLuint index, const GLdouble* v)" \
  "void glVertexAttribLPointer (GLuint index, GLint size, GLenum type, GLsizei stride, const void* pointer)" \
  "void glBindVertexBuffer (GLuint bindingindex, GLuint buffer, GLintptr offset, GLsizei stride)" \
  "void glVertexArrayBindVertexBufferEXT (GLuint vaobj, GLuint bindingindex, GLuint buffer, GLintptr offset, GLsizei stride)" \
  "void glVertexArrayVertexAttribBindingEXT (GLuint vaobj, GLuint attribindex, GLuint bindingindex)" \
  "void glVertexArrayVertexAttribFormatEXT (GLuint vaobj, GLuint attribindex, GLint size, GLenum type, GLboolean normalized, GLuint relativeoffset)" \
  "void glVertexArrayVertexAttribIFormatEXT (GLuint vaobj, GLuint attribindex, GLint size, GLenum type, GLuint relativeoffset)" \
  "void glVertexArrayVertexAttribLFormatEXT (GLuint vaobj, GLuint attribindex, GLint size, GLenum type, GLuint relativeoffset)" \
  "void glVertexArrayVertexBindingDivisorEXT (GLuint vaobj, GLuint bindingindex, GLuint divisor)" \
  "void glVertexAttribBinding (GLuint attribindex, GLuint bindingindex)" \
  "void glVertexAttribFormat (GLuint attribindex, GLint size, GLenum type, GLboolean normalized, GLuint relativeoffset)" \
  "void glVertexAttribIFormat (GLuint attribindex, GLint size, GLenum type, GLuint relativeoffset)" \
  "void glVertexAttribLFormat (GLuint attribindex, GLint size, GLenum type, GLuint relativeoffset)" \
  "void glVertexBindingDivisor (GLuint bindingindex, GLuint divisor)" \
  "void glVertexBlendARB (GLint count)" \
  "void glWeightPointerARB (GLint size, GLenum type, GLsizei stride, void *pointer)" \
  "void glWeightbvARB (GLint size, GLbyte *weights)" \
  "void glWeightdvARB (GLint size, GLdouble *weights)" \
  "void glWeightfvARB (GLint size, GLfloat *weights)" \
  "void glWeightivARB (GLint size, GLint *weights)" \
  "void glWeightsvARB (GLint size, GLshort *weights)" \
  "void glWeightubvARB (GLint size, GLubyte *weights)" \
  "void glWeightuivARB (GLint size, GLuint *weights)" \
  "void glWeightusvARB (GLint size, GLushort *weights)" \
  "void glBindBufferARB (GLenum target, GLuint buffer)" \
  "void glBufferDataARB (GLenum target, GLsizeiptrARB size, const void *data, GLenum usage)" \
  "void glBufferSubDataARB (GLenum target, GLintptrARB offset, GLsizeiptrARB size, const void *data)" \
  "void glDeleteBuffersARB (GLsizei n, const GLuint* buffers)" \
  "void glGenBuffersARB (GLsizei n, GLuint* buffers)" \
  "void glGetBufferParameterivARB (GLenum target, GLenum pname, GLint* params)" \
  "void glGetBufferPointervARB (GLenum target, GLenum pname, void** params)" \
  "void glGetBufferSubDataARB (GLenum target, GLintptrARB offset, GLsizeiptrARB size, void *data)" \
  "GLboolean glIsBufferARB (GLuint buffer)" \
  "void * glMapBufferARB (GLenum target, GLenum access)" \
  "GLboolean glUnmapBufferARB (GLenum target)" \
  "void glBindProgramARB (GLenum target, GLuint program)" \
  "void glDeleteProgramsARB (GLsizei n, const GLuint* programs)" \
  "void glDisableVertexAttribArrayARB (GLuint index)" \
  "void glEnableVertexAttribArrayARB (GLuint index)" \
  "void glGenProgramsARB (GLsizei n, GLuint* programs)" \
  "void glGetProgramEnvParameterdvARB (GLenum target, GLuint index, GLdouble* params)" \
  "void glGetProgramEnvParameterfvARB (GLenum target, GLuint index, GLfloat* params)" \
  "void glGetProgramLocalParameterdvARB (GLenum target, GLuint index, GLdouble* params)" \
  "void glGetProgramLocalParameterfvARB (GLenum target, GLuint index, GLfloat* params)" \
  "void glGetProgramStringARB (GLenum target, GLenum pname, void *string)" \
  "void glGetProgramivARB (GLenum target, GLenum pname, GLint* params)" \
  "void glGetVertexAttribPointervARB (GLuint index, GLenum pname, void** pointer)" \
  "void glGetVertexAttribdvARB (GLuint index, GLenum pname, GLdouble* params)" \
  "void glGetVertexAttribfvARB (GLuint index, GLenum pname, GLfloat* params)" \
  "void glGetVertexAttribivARB (GLuint index, GLenum pname, GLint* params)" \
  "GLboolean glIsProgramARB (GLuint program)" \
  "void glProgramEnvParameter4dARB (GLenum target, GLuint index, GLdouble x, GLdouble y, GLdouble z, GLdouble w)" \
  "void glProgramEnvParameter4dvARB (GLenum target, GLuint index, const GLdouble* params)" \
  "void glProgramEnvParameter4fARB (GLenum target, GLuint index, GLfloat x, GLfloat y, GLfloat z, GLfloat w)" \
  "void glProgramEnvParameter4fvARB (GLenum target, GLuint index, const GLfloat* params)" \
  "void glProgramLocalParameter4dARB (GLenum target, GLuint index, GLdouble x, GLdouble y, GLdouble z, GLdouble w)" \
  "void glProgramLocalParameter4dvARB (GLenum target, GLuint index, const GLdouble* params)" \
  "void glProgramLocalParameter4fARB (GLenum target, GLuint index, GLfloat x, GLfloat y, GLfloat z, GLfloat w)" \
  "void glProgramLocalParameter4fvARB (GLenum target, GLuint index, const GLfloat* params)" \
  "void glProgramStringARB (GLenum target, GLenum format, GLsizei len, const void *string)" \
  "void glVertexAttrib1dARB (GLuint index, GLdouble x)" \
  "void glVertexAttrib1dvARB (GLuint index, const GLdouble* v)" \
  "void glVertexAttrib1fARB (GLuint index, GLfloat x)" \
  "void glVertexAttrib1fvARB (GLuint index, const GLfloat* v)" \
  "void glVertexAttrib1sARB (GLuint index, GLshort x)" \
  "void glVertexAttrib1svARB (GLuint index, const GLshort* v)" \
  "void glVertexAttrib2dARB (GLuint index, GLdouble x, GLdouble y)" \
  "void glVertexAttrib2dvARB (GLuint index, const GLdouble* v)" \
  "void glVertexAttrib2fARB (GLuint index, GLfloat x, GLfloat y)" \
  "void glVertexAttrib2fvARB (GLuint index, const GLfloat* v)" \
  "void glVertexAttrib2sARB (GLuint index, GLshort x, GLshort y)" \
  "void glVertexAttrib2svARB (GLuint index, const GLshort* v)" \
  "void glVertexAttrib3dARB (GLuint index, GLdouble x, GLdouble y, GLdouble z)" \
  "void glVertexAttrib3dvARB (GLuint index, const GLdouble* v)" \
  "void glVertexAttrib3fARB (GLuint index, GLfloat x, GLfloat y, GLfloat z)" \
  "void glVertexAttrib3fvARB (GLuint index, const GLfloat* v)" \
  "void glVertexAttrib3sARB (GLuint index, GLshort x, GLshort y, GLshort z)" \
  "void glVertexAttrib3svARB (GLuint index, const GLshort* v)" \
  "void glVertexAttrib4NbvARB (GLuint index, const GLbyte* v)" \
  "void glVertexAttrib4NivARB (GLuint index, const GLint* v)" \
  "void glVertexAttrib4NsvARB (GLuint index, const GLshort* v)" \
  "void glVertexAttrib4NubARB (GLuint index, GLubyte x, GLubyte y, GLubyte z, GLubyte w)" \
  "void glVertexAttrib4NubvARB (GLuint index, const GLubyte* v)" \
  "void glVertexAttrib4NuivARB (GLuint index, const GLuint* v)" \
  "void glVertexAttrib4NusvARB (GLuint index, const GLushort* v)" \
  "void glVertexAttrib4bvARB (GLuint index, const GLbyte* v)" \
  "void glVertexAttrib4dARB (GLuint index, GLdouble x, GLdouble y, GLdouble z, GLdouble w)" \
  "void glVertexAttrib4dvARB (GLuint index, const GLdouble* v)" \
  "void glVertexAttrib4fARB (GLuint index, GLfloat x, GLfloat y, GLfloat z, GLfloat w)" \
  "void glVertexAttrib4fvARB (GLuint index, const GLfloat* v)" \
  "void glVertexAttrib4ivARB (GLuint index, const GLint* v)" \
  "void glVertexAttrib4sARB (GLuint index, GLshort x, GLshort y, GLshort z, GLshort w)" \
  "void glVertexAttrib4svARB (GLuint index, const GLshort* v)" \
  "void glVertexAttrib4ubvARB (GLuint index, const GLubyte* v)" \
  "void glVertexAttrib4uivARB (GLuint index, const GLuint* v)" \
  "void glVertexAttrib4usvARB (GLuint index, const GLushort* v)" \
  "void glVertexAttribPointerARB (GLuint index, GLint size, GLenum type, GLboolean normalized, GLsizei stride, const void *pointer)" \
  "void glBindAttribLocationARB (GLhandleARB programObj, GLuint index, const GLcharARB* name)" \
  "void glGetActiveAttribARB (GLhandleARB programObj, GLuint index, GLsizei maxLength, GLsizei* length, GLint *size, GLenum *type, GLcharARB *name)" \
  "GLint glGetAttribLocationARB (GLhandleARB programObj, const GLcharARB* name)" \
  "void glColorP3ui (GLenum type, GLuint color)" \
  "void glColorP3uiv (GLenum type, const GLuint* color)" \
  "void glColorP4ui (GLenum type, GLuint color)" \
  "void glColorP4uiv (GLenum type, const GLuint* color)" \
  "void glMultiTexCoordP1ui (GLenum texture, GLenum type, GLuint coords)" \
  "void glMultiTexCoordP1uiv (GLenum texture, GLenum type, const GLuint* coords)" \
  "void glMultiTexCoordP2ui (GLenum texture, GLenum type, GLuint coords)" \
  "void glMultiTexCoordP2uiv (GLenum texture, GLenum type, const GLuint* coords)" \
  "void glMultiTexCoordP3ui (GLenum texture, GLenum type, GLuint coords)" \
  "void glMultiTexCoordP3uiv (GLenum texture, GLenum type, const GLuint* coords)" \
  "void glMultiTexCoordP4ui (GLenum texture, GLenum type, GLuint coords)" \
  "void glMultiTexCoordP4uiv (GLenum texture, GLenum type, const GLuint* coords)" \
  "void glNormalP3ui (GLenum type, GLuint coords)" \
  "void glNormalP3uiv (GLenum type, const GLuint* coords)" \
  "void glSecondaryColorP3ui (GLenum type, GLuint color)" \
  "void glSecondaryColorP3uiv (GLenum type, const GLuint* color)" \
  "void glTexCoordP1ui (GLenum type, GLuint coords)" \
  "void glTexCoordP1uiv (GLenum type, const GLuint* coords)" \
  "void glTexCoordP2ui (GLenum type, GLuint coords)" \
  "void glTexCoordP2uiv (GLenum type, const GLuint* coords)" \
  "void glTexCoordP3ui (GLenum type, GLuint coords)" \
  "void glTexCoordP3uiv (GLenum type, const GLuint* coords)" \
  "void glTexCoordP4ui (GLenum type, GLuint coords)" \
  "void glTexCoordP4uiv (GLenum type, const GLuint* coords)" \
  "void glVertexAttribP1ui (GLuint index, GLenum type, GLboolean normalized, GLuint value)" \
  "void glVertexAttribP1uiv (GLuint index, GLenum type, GLboolean normalized, const GLuint* value)" \
  "void glVertexAttribP2ui (GLuint index, GLenum type, GLboolean normalized, GLuint value)" \
  "void glVertexAttribP2uiv (GLuint index, GLenum type, GLboolean normalized, const GLuint* value)" \
  "void glVertexAttribP3ui (GLuint index, GLenum type, GLboolean normalized, GLuint value)" \
  "void glVertexAttribP3uiv (GLuint index, GLenum type, GLboolean normalized, const GLuint* value)" \
  "void glVertexAttribP4ui (GLuint index, GLenum type, GLboolean normalized, GLuint value)" \
  "void glVertexAttribP4uiv (GLuint index, GLenum type, GLboolean normalized, const GLuint* value)" \
  "void glVertexP2ui (GLenum type, GLuint value)" \
  "void glVertexP2uiv (GLenum type, const GLuint* value)" \
  "void glVertexP3ui (GLenum type, GLuint value)" \
  "void glVertexP3uiv (GLenum type, const GLuint* value)" \
  "void glVertexP4ui (GLenum type, GLuint value)" \
  "void glVertexP4uiv (GLenum type, const GLuint* value)" \
  "void glDepthRangeArrayv (GLuint first, GLsizei count, const GLclampd * v)" \
  "void glDepthRangeIndexed (GLuint index, GLclampd n, GLclampd f)" \
  "void glGetDoublei_v (GLenum target, GLuint index, GLdouble* data)" \
  "void glGetFloati_v (GLenum target, GLuint index, GLfloat* data)" \
  "void glScissorArrayv (GLuint first, GLsizei count, const GLint * v)" \
  "void glScissorIndexed (GLuint index, GLint left, GLint bottom, GLsizei width, GLsizei height)" \
  "void glScissorIndexedv (GLuint index, const GLint * v)" \
  "void glViewportArrayv (GLuint first, GLsizei count, const GLfloat * v)" \
  "void glViewportIndexedf (GLuint index, GLfloat x, GLfloat y, GLfloat w, GLfloat h)" \
  "void glViewportIndexedfv (GLuint index, const GLfloat * v)" \
  "void glWindowPos2dARB (GLdouble x, GLdouble y)" \
  "void glWindowPos2dvARB (const GLdouble* p)" \
  "void glWindowPos2fARB (GLfloat x, GLfloat y)" \
  "void glWindowPos2fvARB (const GLfloat* p)" \
  "void glWindowPos2iARB (GLint x, GLint y)" \
  "void glWindowPos2ivARB (const GLint* p)" \
  "void glWindowPos2sARB (GLshort x, GLshort y)" \
  "void glWindowPos2svARB (const GLshort* p)" \
  "void glWindowPos3dARB (GLdouble x, GLdouble y, GLdouble z)" \
  "void glWindowPos3dvARB (const GLdouble* p)" \
  "void glWindowPos3fARB (GLfloat x, GLfloat y, GLfloat z)" \
  "void glWindowPos3fvARB (const GLfloat* p)" \
  "void glWindowPos3iARB (GLint x, GLint y, GLint z)" \
  "void glWindowPos3ivARB (const GLint* p)" \
  "void glWindowPos3sARB (GLshort x, GLshort y, GLshort z)" \
  "void glWindowPos3svARB (const GLshort* p)" \
  "void glDrawBuffersATI (GLsizei n, const GLenum* bufs)" \
  "void glDrawElementArrayATI (GLenum mode, GLsizei count)" \
  "void glDrawRangeElementArrayATI (GLenum mode, GLuint start, GLuint end, GLsizei count)" \
  "void glElementPointerATI (GLenum type, const void *pointer)" \
  "void glGetTexBumpParameterfvATI (GLenum pname, GLfloat *param)" \
  "void glGetTexBumpParameterivATI (GLenum pname, GLint *param)" \
  "void glTexBumpParameterfvATI (GLenum pname, GLfloat *param)" \
  "void glTexBumpParameterivATI (GLenum pname, GLint *param)" \
  "void glAlphaFragmentOp1ATI (GLenum op, GLuint dst, GLuint dstMod, GLuint arg1, GLuint arg1Rep, GLuint arg1Mod)" \
  "void glAlphaFragmentOp2ATI (GLenum op, GLuint dst, GLuint dstMod, GLuint arg1, GLuint arg1Rep, GLuint arg1Mod, GLuint arg2, GLuint arg2Rep, GLuint arg2Mod)" \
  "void glAlphaFragmentOp3ATI (GLenum op, GLuint dst, GLuint dstMod, GLuint arg1, GLuint arg1Rep, GLuint arg1Mod, GLuint arg2, GLuint arg2Rep, GLuint arg2Mod, GLuint arg3, GLuint arg3Rep, GLuint arg3Mod)" \
  "void glBeginFragmentShaderATI (void)" \
  "void glBindFragmentShaderATI (GLuint id)" \
  "void glColorFragmentOp1ATI (GLenum op, GLuint dst, GLuint dstMask, GLuint dstMod, GLuint arg1, GLuint arg1Rep, GLuint arg1Mod)" \
  "void glColorFragmentOp2ATI (GLenum op, GLuint dst, GLuint dstMask, GLuint dstMod, GLuint arg1, GLuint arg1Rep, GLuint arg1Mod, GLuint arg2, GLuint arg2Rep, GLuint arg2Mod)" \
  "void glColorFragmentOp3ATI (GLenum op, GLuint dst, GLuint dstMask, GLuint dstMod, GLuint arg1, GLuint arg1Rep, GLuint arg1Mod, GLuint arg2, GLuint arg2Rep, GLuint arg2Mod, GLuint arg3, GLuint arg3Rep, GLuint arg3Mod)" \
  "void glDeleteFragmentShaderATI (GLuint id)" \
  "void glEndFragmentShaderATI (void)" \
  "GLuint glGenFragmentShadersATI (GLuint range)" \
  "void glPassTexCoordATI (GLuint dst, GLuint coord, GLenum swizzle)" \
  "void glSampleMapATI (GLuint dst, GLuint interp, GLenum swizzle)" \
  "void glSetFragmentShaderConstantATI (GLuint dst, const GLfloat* value)" \
  "void * glMapObjectBufferATI (GLuint buffer)" \
  "void glUnmapObjectBufferATI (GLuint buffer)" \
  "void glPNTrianglesfATI (GLenum pname, GLfloat param)" \
  "void glPNTrianglesiATI (GLenum pname, GLint param)" \
  "void glStencilFuncSeparateATI (GLenum frontfunc, GLenum backfunc, GLint ref, GLuint mask)" \
  "void glStencilOpSeparateATI (GLenum face, GLenum sfail, GLenum dpfail, GLenum dppass)" \
  "void glArrayObjectATI (GLenum array, GLint size, GLenum type, GLsizei stride, GLuint buffer, GLuint offset)" \
  "void glFreeObjectBufferATI (GLuint buffer)" \
  "void glGetArrayObjectfvATI (GLenum array, GLenum pname, GLfloat* params)" \
  "void glGetArrayObjectivATI (GLenum array, GLenum pname, GLint* params)" \
  "void glGetObjectBufferfvATI (GLuint buffer, GLenum pname, GLfloat* params)" \
  "void glGetObjectBufferivATI (GLuint buffer, GLenum pname, GLint* params)" \
  "void glGetVariantArrayObjectfvATI (GLuint id, GLenum pname, GLfloat* params)" \
  "void glGetVariantArrayObjectivATI (GLuint id, GLenum pname, GLint* params)" \
  "GLboolean glIsObjectBufferATI (GLuint buffer)" \
  "GLuint glNewObjectBufferATI (GLsizei size, const void *pointer, GLenum usage)" \
  "void glUpdateObjectBufferATI (GLuint buffer, GLuint offset, GLsizei size, const void *pointer, GLenum preserve)" \
  "void glVariantArrayObjectATI (GLuint id, GLenum type, GLsizei stride, GLuint buffer, GLuint offset)" \
  "void glGetVertexAttribArrayObjectfvATI (GLuint index, GLenum pname, GLfloat* params)" \
  "void glGetVertexAttribArrayObjectivATI (GLuint index, GLenum pname, GLint* params)" \
  "void glVertexAttribArrayObjectATI (GLuint index, GLint size, GLenum type, GLboolean normalized, GLsizei stride, GLuint buffer, GLuint offset)" \
  "void glClientActiveVertexStreamATI (GLenum stream)" \
  "void glNormalStream3bATI (GLenum stream, GLbyte x, GLbyte y, GLbyte z)" \
  "void glNormalStream3bvATI (GLenum stream, const GLbyte *coords)" \
  "void glNormalStream3dATI (GLenum stream, GLdouble x, GLdouble y, GLdouble z)" \
  "void glNormalStream3dvATI (GLenum stream, const GLdouble *coords)" \
  "void glNormalStream3fATI (GLenum stream, GLfloat x, GLfloat y, GLfloat z)" \
  "void glNormalStream3fvATI (GLenum stream, const GLfloat *coords)" \
  "void glNormalStream3iATI (GLenum stream, GLint x, GLint y, GLint z)" \
  "void glNormalStream3ivATI (GLenum stream, const GLint *coords)" \
  "void glNormalStream3sATI (GLenum stream, GLshort x, GLshort y, GLshort z)" \
  "void glNormalStream3svATI (GLenum stream, const GLshort *coords)" \
  "void glVertexBlendEnvfATI (GLenum pname, GLfloat param)" \
  "void glVertexBlendEnviATI (GLenum pname, GLint param)" \
  "void glVertexStream1dATI (GLenum stream, GLdouble x)" \
  "void glVertexStream1dvATI (GLenum stream, const GLdouble *coords)" \
  "void glVertexStream1fATI (GLenum stream, GLfloat x)" \
  "void glVertexStream1fvATI (GLenum stream, const GLfloat *coords)" \
  "void glVertexStream1iATI (GLenum stream, GLint x)" \
  "void glVertexStream1ivATI (GLenum stream, const GLint *coords)" \
  "void glVertexStream1sATI (GLenum stream, GLshort x)" \
  "void glVertexStream1svATI (GLenum stream, const GLshort *coords)" \
  "void glVertexStream2dATI (GLenum stream, GLdouble x, GLdouble y)" \
  "void glVertexStream2dvATI (GLenum stream, const GLdouble *coords)" \
  "void glVertexStream2fATI (GLenum stream, GLfloat x, GLfloat y)" \
  "void glVertexStream2fvATI (GLenum stream, const GLfloat *coords)" \
  "void glVertexStream2iATI (GLenum stream, GLint x, GLint y)" \
  "void glVertexStream2ivATI (GLenum stream, const GLint *coords)" \
  "void glVertexStream2sATI (GLenum stream, GLshort x, GLshort y)" \
  "void glVertexStream2svATI (GLenum stream, const GLshort *coords)" \
  "void glVertexStream3dATI (GLenum stream, GLdouble x, GLdouble y, GLdouble z)" \
  "void glVertexStream3dvATI (GLenum stream, const GLdouble *coords)" \
  "void glVertexStream3fATI (GLenum stream, GLfloat x, GLfloat y, GLfloat z)" \
  "void glVertexStream3fvATI (GLenum stream, const GLfloat *coords)" \
  "void glVertexStream3iATI (GLenum stream, GLint x, GLint y, GLint z)" \
  "void glVertexStream3ivATI (GLenum stream, const GLint *coords)" \
  "void glVertexStream3sATI (GLenum stream, GLshort x, GLshort y, GLshort z)" \
  "void glVertexStream3svATI (GLenum stream, const GLshort *coords)" \
  "void glVertexStream4dATI (GLenum stream, GLdouble x, GLdouble y, GLdouble z, GLdouble w)" \
  "void glVertexStream4dvATI (GLenum stream, const GLdouble *coords)" \
  "void glVertexStream4fATI (GLenum stream, GLfloat x, GLfloat y, GLfloat z, GLfloat w)" \
  "void glVertexStream4fvATI (GLenum stream, const GLfloat *coords)" \
  "void glVertexStream4iATI (GLenum stream, GLint x, GLint y, GLint z, GLint w)" \
  "void glVertexStream4ivATI (GLenum stream, const GLint *coords)" \
  "void glVertexStream4sATI (GLenum stream, GLshort x, GLshort y, GLshort z, GLshort w)" \
  "void glVertexStream4svATI (GLenum stream, const GLshort *coords)" \
  "void glEGLImageTargetTexStorageEXT (GLenum target, GLeglImageOES image, const GLint* attrib_list)" \
  "void glEGLImageTargetTextureStorageEXT (GLuint texture, GLeglImageOES image, const GLint* attrib_list)" \
  "void glDrawArraysInstancedBaseInstanceEXT (GLenum mode, GLint first, GLsizei count, GLsizei instancecount, GLuint baseinstance)" \
  "void glDrawElementsInstancedBaseInstanceEXT (GLenum mode, GLsizei count, GLenum type, const void *indices, GLsizei instancecount, GLuint baseinstance)" \
  "void glDrawElementsInstancedBaseVertexBaseInstanceEXT (GLenum mode, GLsizei count, GLenum type, const void *indices, GLsizei instancecount, GLint basevertex, GLuint baseinstance)" \
  "GLint glGetUniformBufferSizeEXT (GLuint program, GLint location)" \
  "GLintptr glGetUniformOffsetEXT (GLuint program, GLint location)" \
  "void glUniformBufferEXT (GLuint program, GLint location, GLuint buffer)" \
  "void glBlendColorEXT (GLclampf red, GLclampf green, GLclampf blue, GLclampf alpha)" \
  "void glBlendEquationSeparateEXT (GLenum modeRGB, GLenum modeAlpha)" \
  "void glBindFragDataLocationIndexedEXT (GLuint program, GLuint colorNumber, GLuint index, const GLchar * name)" \
  "GLint glGetFragDataIndexEXT (GLuint program, const GLchar * name)" \
  "GLint glGetProgramResourceLocationIndexEXT (GLuint program, GLenum programInterface, const GLchar* name)" \
  "void glBlendFuncSeparateEXT (GLenum sfactorRGB, GLenum dfactorRGB, GLenum sfactorAlpha, GLenum dfactorAlpha)" \
  "void glBlendEquationEXT (GLenum mode)" \
  "void glBufferStorageEXT (GLenum target, GLsizeiptr size, const void *data, GLbitfield flags)" \
  "void glNamedBufferStorageEXT (GLuint buffer, GLsizeiptr size, const void *data, GLbitfield flags)" \
  "void glClearTexImageEXT (GLuint texture, GLint level, GLenum format, GLenum type, const void *data)" \
  "void glClearTexSubImageEXT (GLuint texture, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, GLenum format, GLenum type, const void *data)" \
  "void glClipControlEXT (GLenum origin, GLenum depth)" \
  "void glColorSubTableEXT (GLenum target, GLsizei start, GLsizei count, GLenum format, GLenum type, const void *data)" \
  "void glCopyColorSubTableEXT (GLenum target, GLsizei start, GLint x, GLint y, GLsizei width)" \
  "void glLockArraysEXT (GLint first, GLsizei count)" \
  "void glUnlockArraysEXT (void)" \
  "void glConvolutionFilter1DEXT (GLenum target, GLenum internalformat, GLsizei width, GLenum format, GLenum type, const void *image)" \
  "void glConvolutionFilter2DEXT (GLenum target, GLenum internalformat, GLsizei width, GLsizei height, GLenum format, GLenum type, const void *image)" \
  "void glConvolutionParameterfEXT (GLenum target, GLenum pname, GLfloat param)" \
  "void glConvolutionParameterfvEXT (GLenum target, GLenum pname, const GLfloat* params)" \
  "void glConvolutionParameteriEXT (GLenum target, GLenum pname, GLint param)" \
  "void glConvolutionParameterivEXT (GLenum target, GLenum pname, const GLint* params)" \
  "void glCopyConvolutionFilter1DEXT (GLenum target, GLenum internalformat, GLint x, GLint y, GLsizei width)" \
  "void glCopyConvolutionFilter2DEXT (GLenum target, GLenum internalformat, GLint x, GLint y, GLsizei width, GLsizei height)" \
  "void glGetConvolutionFilterEXT (GLenum target, GLenum format, GLenum type, void *image)" \
  "void glGetConvolutionParameterfvEXT (GLenum target, GLenum pname, GLfloat* params)" \
  "void glGetConvolutionParameterivEXT (GLenum target, GLenum pname, GLint* params)" \
  "void glGetSeparableFilterEXT (GLenum target, GLenum format, GLenum type, void *row, void *column, void *span)" \
  "void glSeparableFilter2DEXT (GLenum target, GLenum internalformat, GLsizei width, GLsizei height, GLenum format, GLenum type, const void *row, const void *column)" \
  "void glBinormalPointerEXT (GLenum type, GLsizei stride, void *pointer)" \
  "void glTangentPointerEXT (GLenum type, GLsizei stride, void *pointer)" \
  "void glCopyImageSubDataEXT (GLuint srcName, GLenum srcTarget, GLint srcLevel, GLint srcX, GLint srcY, GLint srcZ, GLuint dstName, GLenum dstTarget, GLint dstLevel, GLint dstX, GLint dstY, GLint dstZ, GLsizei srcWidth, GLsizei srcHeight, GLsizei srcDepth)" \
  "void glCopyTexImage1DEXT (GLenum target, GLint level, GLenum internalformat, GLint x, GLint y, GLsizei width, GLint border)" \
  "void glCopyTexImage2DEXT (GLenum target, GLint level, GLenum internalformat, GLint x, GLint y, GLsizei width, GLsizei height, GLint border)" \
  "void glCopyTexSubImage1DEXT (GLenum target, GLint level, GLint xoffset, GLint x, GLint y, GLsizei width)" \
  "void glCopyTexSubImage2DEXT (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint x, GLint y, GLsizei width, GLsizei height)" \
  "void glCopyTexSubImage3DEXT (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLint x, GLint y, GLsizei width, GLsizei height)" \
  "void glCullParameterdvEXT (GLenum pname, GLdouble* params)" \
  "void glCullParameterfvEXT (GLenum pname, GLfloat* params)" \
  "void glGetObjectLabelEXT (GLenum type, GLuint object, GLsizei bufSize, GLsizei* length, GLchar *label)" \
  "void glLabelObjectEXT (GLenum type, GLuint object, GLsizei length, const GLchar* label)" \
  "void glInsertEventMarkerEXT (GLsizei length, const GLchar* marker)" \
  "void glPopGroupMarkerEXT (void)" \
  "void glPushGroupMarkerEXT (GLsizei length, const GLchar* marker)" \
  "void glDepthBoundsEXT (GLclampd zmin, GLclampd zmax)" \
  "void glBindMultiTextureEXT (GLenum texunit, GLenum target, GLuint texture)" \
  "GLenum glCheckNamedFramebufferStatusEXT (GLuint framebuffer, GLenum target)" \
  "void glClientAttribDefaultEXT (GLbitfield mask)" \
  "void glCompressedMultiTexImage1DEXT (GLenum texunit, GLenum target, GLint level, GLenum internalformat, GLsizei width, GLint border, GLsizei imageSize, const void *data)" \
  "void glCompressedMultiTexImage2DEXT (GLenum texunit, GLenum target, GLint level, GLenum internalformat, GLsizei width, GLsizei height, GLint border, GLsizei imageSize, const void *data)" \
  "void glCompressedMultiTexImage3DEXT (GLenum texunit, GLenum target, GLint level, GLenum internalformat, GLsizei width, GLsizei height, GLsizei depth, GLint border, GLsizei imageSize, const void *data)" \
  "void glCompressedMultiTexSubImage1DEXT (GLenum texunit, GLenum target, GLint level, GLint xoffset, GLsizei width, GLenum format, GLsizei imageSize, const void *data)" \
  "void glCompressedMultiTexSubImage2DEXT (GLenum texunit, GLenum target, GLint level, GLint xoffset, GLint yoffset, GLsizei width, GLsizei height, GLenum format, GLsizei imageSize, const void *data)" \
  "void glCompressedMultiTexSubImage3DEXT (GLenum texunit, GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, GLenum format, GLsizei imageSize, const void *data)" \
  "void glCompressedTextureImage1DEXT (GLuint texture, GLenum target, GLint level, GLenum internalformat, GLsizei width, GLint border, GLsizei imageSize, const void *data)" \
  "void glCompressedTextureImage2DEXT (GLuint texture, GLenum target, GLint level, GLenum internalformat, GLsizei width, GLsizei height, GLint border, GLsizei imageSize, const void *data)" \
  "void glCompressedTextureImage3DEXT (GLuint texture, GLenum target, GLint level, GLenum internalformat, GLsizei width, GLsizei height, GLsizei depth, GLint border, GLsizei imageSize, const void *data)" \
  "void glCompressedTextureSubImage1DEXT (GLuint texture, GLenum target, GLint level, GLint xoffset, GLsizei width, GLenum format, GLsizei imageSize, const void *data)" \
  "void glCompressedTextureSubImage2DEXT (GLuint texture, GLenum target, GLint level, GLint xoffset, GLint yoffset, GLsizei width, GLsizei height, GLenum format, GLsizei imageSize, const void *data)" \
  "void glCompressedTextureSubImage3DEXT (GLuint texture, GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, GLenum format, GLsizei imageSize, const void *data)" \
  "void glCopyMultiTexImage1DEXT (GLenum texunit, GLenum target, GLint level, GLenum internalformat, GLint x, GLint y, GLsizei width, GLint border)" \
  "void glCopyMultiTexImage2DEXT (GLenum texunit, GLenum target, GLint level, GLenum internalformat, GLint x, GLint y, GLsizei width, GLsizei height, GLint border)" \
  "void glCopyMultiTexSubImage1DEXT (GLenum texunit, GLenum target, GLint level, GLint xoffset, GLint x, GLint y, GLsizei width)" \
  "void glCopyMultiTexSubImage2DEXT (GLenum texunit, GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint x, GLint y, GLsizei width, GLsizei height)" \
  "void glCopyMultiTexSubImage3DEXT (GLenum texunit, GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLint x, GLint y, GLsizei width, GLsizei height)" \
  "void glCopyTextureImage1DEXT (GLuint texture, GLenum target, GLint level, GLenum internalformat, GLint x, GLint y, GLsizei width, GLint border)" \
  "void glCopyTextureImage2DEXT (GLuint texture, GLenum target, GLint level, GLenum internalformat, GLint x, GLint y, GLsizei width, GLsizei height, GLint border)" \
  "void glCopyTextureSubImage1DEXT (GLuint texture, GLenum target, GLint level, GLint xoffset, GLint x, GLint y, GLsizei width)" \
  "void glCopyTextureSubImage2DEXT (GLuint texture, GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint x, GLint y, GLsizei width, GLsizei height)" \
  "void glCopyTextureSubImage3DEXT (GLuint texture, GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLint x, GLint y, GLsizei width, GLsizei height)" \
  "void glDisableClientStateIndexedEXT (GLenum array, GLuint index)" \
  "void glDisableClientStateiEXT (GLenum array, GLuint index)" \
  "void glDisableVertexArrayAttribEXT (GLuint vaobj, GLuint index)" \
  "void glDisableVertexArrayEXT (GLuint vaobj, GLenum array)" \
  "void glEnableClientStateIndexedEXT (GLenum array, GLuint index)" \
  "void glEnableClientStateiEXT (GLenum array, GLuint index)" \
  "void glEnableVertexArrayAttribEXT (GLuint vaobj, GLuint index)" \
  "void glEnableVertexArrayEXT (GLuint vaobj, GLenum array)" \
  "void glFlushMappedNamedBufferRangeEXT (GLuint buffer, GLintptr offset, GLsizeiptr length)" \
  "void glFramebufferDrawBufferEXT (GLuint framebuffer, GLenum mode)" \
  "void glFramebufferDrawBuffersEXT (GLuint framebuffer, GLsizei n, const GLenum* bufs)" \
  "void glFramebufferReadBufferEXT (GLuint framebuffer, GLenum mode)" \
  "void glGenerateMultiTexMipmapEXT (GLenum texunit, GLenum target)" \
  "void glGenerateTextureMipmapEXT (GLuint texture, GLenum target)" \
  "void glGetCompressedMultiTexImageEXT (GLenum texunit, GLenum target, GLint level, void *img)" \
  "void glGetCompressedTextureImageEXT (GLuint texture, GLenum target, GLint level, void *img)" \
  "void glGetDoubleIndexedvEXT (GLenum target, GLuint index, GLdouble* params)" \
  "void glGetDoublei_vEXT (GLenum pname, GLuint index, GLdouble* params)" \
  "void glGetFloatIndexedvEXT (GLenum target, GLuint index, GLfloat* params)" \
  "void glGetFloati_vEXT (GLenum pname, GLuint index, GLfloat* params)" \
  "void glGetFramebufferParameterivEXT (GLuint framebuffer, GLenum pname, GLint* param)" \
  "void glGetMultiTexEnvfvEXT (GLenum texunit, GLenum target, GLenum pname, GLfloat* params)" \
  "void glGetMultiTexEnvivEXT (GLenum texunit, GLenum target, GLenum pname, GLint* params)" \
  "void glGetMultiTexGendvEXT (GLenum texunit, GLenum coord, GLenum pname, GLdouble* params)" \
  "void glGetMultiTexGenfvEXT (GLenum texunit, GLenum coord, GLenum pname, GLfloat* params)" \
  "void glGetMultiTexGenivEXT (GLenum texunit, GLenum coord, GLenum pname, GLint* params)" \
  "void glGetMultiTexImageEXT (GLenum texunit, GLenum target, GLint level, GLenum format, GLenum type, void *pixels)" \
  "void glGetMultiTexLevelParameterfvEXT (GLenum texunit, GLenum target, GLint level, GLenum pname, GLfloat* params)" \
  "void glGetMultiTexLevelParameterivEXT (GLenum texunit, GLenum target, GLint level, GLenum pname, GLint* params)" \
  "void glGetMultiTexParameterIivEXT (GLenum texunit, GLenum target, GLenum pname, GLint* params)" \
  "void glGetMultiTexParameterIuivEXT (GLenum texunit, GLenum target, GLenum pname, GLuint* params)" \
  "void glGetMultiTexParameterfvEXT (GLenum texunit, GLenum target, GLenum pname, GLfloat* params)" \
  "void glGetMultiTexParameterivEXT (GLenum texunit, GLenum target, GLenum pname, GLint* params)" \
  "void glGetNamedBufferParameterivEXT (GLuint buffer, GLenum pname, GLint* params)" \
  "void glGetNamedBufferPointervEXT (GLuint buffer, GLenum pname, void** params)" \
  "void glGetNamedBufferSubDataEXT (GLuint buffer, GLintptr offset, GLsizeiptr size, void *data)" \
  "void glGetNamedFramebufferAttachmentParameterivEXT (GLuint framebuffer, GLenum attachment, GLenum pname, GLint* params)" \
  "void glGetNamedProgramLocalParameterIivEXT (GLuint program, GLenum target, GLuint index, GLint* params)" \
  "void glGetNamedProgramLocalParameterIuivEXT (GLuint program, GLenum target, GLuint index, GLuint* params)" \
  "void glGetNamedProgramLocalParameterdvEXT (GLuint program, GLenum target, GLuint index, GLdouble* params)" \
  "void glGetNamedProgramLocalParameterfvEXT (GLuint program, GLenum target, GLuint index, GLfloat* params)" \
  "void glGetNamedProgramStringEXT (GLuint program, GLenum target, GLenum pname, void *string)" \
  "void glGetNamedProgramivEXT (GLuint program, GLenum target, GLenum pname, GLint* params)" \
  "void glGetNamedRenderbufferParameterivEXT (GLuint renderbuffer, GLenum pname, GLint* params)" \
  "void glGetPointerIndexedvEXT (GLenum target, GLuint index, void** params)" \
  "void glGetPointeri_vEXT (GLenum pname, GLuint index, void** params)" \
  "void glGetTextureImageEXT (GLuint texture, GLenum target, GLint level, GLenum format, GLenum type, void *pixels)" \
  "void glGetTextureLevelParameterfvEXT (GLuint texture, GLenum target, GLint level, GLenum pname, GLfloat* params)" \
  "void glGetTextureLevelParameterivEXT (GLuint texture, GLenum target, GLint level, GLenum pname, GLint* params)" \
  "void glGetTextureParameterIivEXT (GLuint texture, GLenum target, GLenum pname, GLint* params)" \
  "void glGetTextureParameterIuivEXT (GLuint texture, GLenum target, GLenum pname, GLuint* params)" \
  "void glGetTextureParameterfvEXT (GLuint texture, GLenum target, GLenum pname, GLfloat* params)" \
  "void glGetTextureParameterivEXT (GLuint texture, GLenum target, GLenum pname, GLint* params)" \
  "void glGetVertexArrayIntegeri_vEXT (GLuint vaobj, GLuint index, GLenum pname, GLint* param)" \
  "void glGetVertexArrayIntegervEXT (GLuint vaobj, GLenum pname, GLint* param)" \
  "void glGetVertexArrayPointeri_vEXT (GLuint vaobj, GLuint index, GLenum pname, void** param)" \
  "void glGetVertexArrayPointervEXT (GLuint vaobj, GLenum pname, void** param)" \
  "void * glMapNamedBufferEXT (GLuint buffer, GLenum access)" \
  "void * glMapNamedBufferRangeEXT (GLuint buffer, GLintptr offset, GLsizeiptr length, GLbitfield access)" \
  "void glMatrixFrustumEXT (GLenum matrixMode, GLdouble l, GLdouble r, GLdouble b, GLdouble t, GLdouble n, GLdouble f)" \
  "void glMatrixLoadIdentityEXT (GLenum matrixMode)" \
  "void glMatrixLoadTransposedEXT (GLenum matrixMode, const GLdouble* m)" \
  "void glMatrixLoadTransposefEXT (GLenum matrixMode, const GLfloat* m)" \
  "void glMatrixLoaddEXT (GLenum matrixMode, const GLdouble* m)" \
  "void glMatrixLoadfEXT (GLenum matrixMode, const GLfloat* m)" \
  "void glMatrixMultTransposedEXT (GLenum matrixMode, const GLdouble* m)" \
  "void glMatrixMultTransposefEXT (GLenum matrixMode, const GLfloat* m)" \
  "void glMatrixMultdEXT (GLenum matrixMode, const GLdouble* m)" \
  "void glMatrixMultfEXT (GLenum matrixMode, const GLfloat* m)" \
  "void glMatrixOrthoEXT (GLenum matrixMode, GLdouble l, GLdouble r, GLdouble b, GLdouble t, GLdouble n, GLdouble f)" \
  "void glMatrixPopEXT (GLenum matrixMode)" \
  "void glMatrixPushEXT (GLenum matrixMode)" \
  "void glMatrixRotatedEXT (GLenum matrixMode, GLdouble angle, GLdouble x, GLdouble y, GLdouble z)" \
  "void glMatrixRotatefEXT (GLenum matrixMode, GLfloat angle, GLfloat x, GLfloat y, GLfloat z)" \
  "void glMatrixScaledEXT (GLenum matrixMode, GLdouble x, GLdouble y, GLdouble z)" \
  "void glMatrixScalefEXT (GLenum matrixMode, GLfloat x, GLfloat y, GLfloat z)" \
  "void glMatrixTranslatedEXT (GLenum matrixMode, GLdouble x, GLdouble y, GLdouble z)" \
  "void glMatrixTranslatefEXT (GLenum matrixMode, GLfloat x, GLfloat y, GLfloat z)" \
  "void glMultiTexBufferEXT (GLenum texunit, GLenum target, GLenum internalformat, GLuint buffer)" \
  "void glMultiTexCoordPointerEXT (GLenum texunit, GLint size, GLenum type, GLsizei stride, const void *pointer)" \
  "void glMultiTexEnvfEXT (GLenum texunit, GLenum target, GLenum pname, GLfloat param)" \
  "void glMultiTexEnvfvEXT (GLenum texunit, GLenum target, GLenum pname, const GLfloat* params)" \
  "void glMultiTexEnviEXT (GLenum texunit, GLenum target, GLenum pname, GLint param)" \
  "void glMultiTexEnvivEXT (GLenum texunit, GLenum target, GLenum pname, const GLint* params)" \
  "void glMultiTexGendEXT (GLenum texunit, GLenum coord, GLenum pname, GLdouble param)" \
  "void glMultiTexGendvEXT (GLenum texunit, GLenum coord, GLenum pname, const GLdouble* params)" \
  "void glMultiTexGenfEXT (GLenum texunit, GLenum coord, GLenum pname, GLfloat param)" \
  "void glMultiTexGenfvEXT (GLenum texunit, GLenum coord, GLenum pname, const GLfloat* params)" \
  "void glMultiTexGeniEXT (GLenum texunit, GLenum coord, GLenum pname, GLint param)" \
  "void glMultiTexGenivEXT (GLenum texunit, GLenum coord, GLenum pname, const GLint* params)" \
  "void glMultiTexImage1DEXT (GLenum texunit, GLenum target, GLint level, GLint internalformat, GLsizei width, GLint border, GLenum format, GLenum type, const void *pixels)" \
  "void glMultiTexImage2DEXT (GLenum texunit, GLenum target, GLint level, GLint internalformat, GLsizei width, GLsizei height, GLint border, GLenum format, GLenum type, const void *pixels)" \
  "void glMultiTexImage3DEXT (GLenum texunit, GLenum target, GLint level, GLint internalformat, GLsizei width, GLsizei height, GLsizei depth, GLint border, GLenum format, GLenum type, const void *pixels)" \
  "void glMultiTexParameterIivEXT (GLenum texunit, GLenum target, GLenum pname, const GLint* params)" \
  "void glMultiTexParameterIuivEXT (GLenum texunit, GLenum target, GLenum pname, const GLuint* params)" \
  "void glMultiTexParameterfEXT (GLenum texunit, GLenum target, GLenum pname, GLfloat param)" \
  "void glMultiTexParameterfvEXT (GLenum texunit, GLenum target, GLenum pname, const GLfloat* param)" \
  "void glMultiTexParameteriEXT (GLenum texunit, GLenum target, GLenum pname, GLint param)" \
  "void glMultiTexParameterivEXT (GLenum texunit, GLenum target, GLenum pname, const GLint* param)" \
  "void glMultiTexRenderbufferEXT (GLenum texunit, GLenum target, GLuint renderbuffer)" \
  "void glMultiTexSubImage1DEXT (GLenum texunit, GLenum target, GLint level, GLint xoffset, GLsizei width, GLenum format, GLenum type, const void *pixels)" \
  "void glMultiTexSubImage2DEXT (GLenum texunit, GLenum target, GLint level, GLint xoffset, GLint yoffset, GLsizei width, GLsizei height, GLenum format, GLenum type, const void *pixels)" \
  "void glMultiTexSubImage3DEXT (GLenum texunit, GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, GLenum format, GLenum type, const void *pixels)" \
  "void glNamedBufferDataEXT (GLuint buffer, GLsizeiptr size, const void *data, GLenum usage)" \
  "void glNamedBufferSubDataEXT (GLuint buffer, GLintptr offset, GLsizeiptr size, const void *data)" \
  "void glNamedCopyBufferSubDataEXT (GLuint readBuffer, GLuint writeBuffer, GLintptr readOffset, GLintptr writeOffset, GLsizeiptr size)" \
  "void glNamedFramebufferRenderbufferEXT (GLuint framebuffer, GLenum attachment, GLenum renderbuffertarget, GLuint renderbuffer)" \
  "void glNamedFramebufferTexture1DEXT (GLuint framebuffer, GLenum attachment, GLenum textarget, GLuint texture, GLint level)" \
  "void glNamedFramebufferTexture2DEXT (GLuint framebuffer, GLenum attachment, GLenum textarget, GLuint texture, GLint level)" \
  "void glNamedFramebufferTexture3DEXT (GLuint framebuffer, GLenum attachment, GLenum textarget, GLuint texture, GLint level, GLint zoffset)" \
  "void glNamedFramebufferTextureEXT (GLuint framebuffer, GLenum attachment, GLuint texture, GLint level)" \
  "void glNamedFramebufferTextureFaceEXT (GLuint framebuffer, GLenum attachment, GLuint texture, GLint level, GLenum face)" \
  "void glNamedFramebufferTextureLayerEXT (GLuint framebuffer, GLenum attachment, GLuint texture, GLint level, GLint layer)" \
  "void glNamedProgramLocalParameter4dEXT (GLuint program, GLenum target, GLuint index, GLdouble x, GLdouble y, GLdouble z, GLdouble w)" \
  "void glNamedProgramLocalParameter4dvEXT (GLuint program, GLenum target, GLuint index, const GLdouble* params)" \
  "void glNamedProgramLocalParameter4fEXT (GLuint program, GLenum target, GLuint index, GLfloat x, GLfloat y, GLfloat z, GLfloat w)" \
  "void glNamedProgramLocalParameter4fvEXT (GLuint program, GLenum target, GLuint index, const GLfloat* params)" \
  "void glNamedProgramLocalParameterI4iEXT (GLuint program, GLenum target, GLuint index, GLint x, GLint y, GLint z, GLint w)" \
  "void glNamedProgramLocalParameterI4ivEXT (GLuint program, GLenum target, GLuint index, const GLint* params)" \
  "void glNamedProgramLocalParameterI4uiEXT (GLuint program, GLenum target, GLuint index, GLuint x, GLuint y, GLuint z, GLuint w)" \
  "void glNamedProgramLocalParameterI4uivEXT (GLuint program, GLenum target, GLuint index, const GLuint* params)" \
  "void glNamedProgramLocalParameters4fvEXT (GLuint program, GLenum target, GLuint index, GLsizei count, const GLfloat* params)" \
  "void glNamedProgramLocalParametersI4ivEXT (GLuint program, GLenum target, GLuint index, GLsizei count, const GLint* params)" \
  "void glNamedProgramLocalParametersI4uivEXT (GLuint program, GLenum target, GLuint index, GLsizei count, const GLuint* params)" \
  "void glNamedProgramStringEXT (GLuint program, GLenum target, GLenum format, GLsizei len, const void *string)" \
  "void glNamedRenderbufferStorageEXT (GLuint renderbuffer, GLenum internalformat, GLsizei width, GLsizei height)" \
  "void glNamedRenderbufferStorageMultisampleCoverageEXT (GLuint renderbuffer, GLsizei coverageSamples, GLsizei colorSamples, GLenum internalformat, GLsizei width, GLsizei height)" \
  "void glNamedRenderbufferStorageMultisampleEXT (GLuint renderbuffer, GLsizei samples, GLenum internalformat, GLsizei width, GLsizei height)" \
  "void glProgramUniform1fEXT (GLuint program, GLint location, GLfloat v0)" \
  "void glProgramUniform1fvEXT (GLuint program, GLint location, GLsizei count, const GLfloat* value)" \
  "void glProgramUniform1iEXT (GLuint program, GLint location, GLint v0)" \
  "void glProgramUniform1ivEXT (GLuint program, GLint location, GLsizei count, const GLint* value)" \
  "void glProgramUniform1uiEXT (GLuint program, GLint location, GLuint v0)" \
  "void glProgramUniform1uivEXT (GLuint program, GLint location, GLsizei count, const GLuint* value)" \
  "void glProgramUniform2fEXT (GLuint program, GLint location, GLfloat v0, GLfloat v1)" \
  "void glProgramUniform2fvEXT (GLuint program, GLint location, GLsizei count, const GLfloat* value)" \
  "void glProgramUniform2iEXT (GLuint program, GLint location, GLint v0, GLint v1)" \
  "void glProgramUniform2ivEXT (GLuint program, GLint location, GLsizei count, const GLint* value)" \
  "void glProgramUniform2uiEXT (GLuint program, GLint location, GLuint v0, GLuint v1)" \
  "void glProgramUniform2uivEXT (GLuint program, GLint location, GLsizei count, const GLuint* value)" \
  "void glProgramUniform3fEXT (GLuint program, GLint location, GLfloat v0, GLfloat v1, GLfloat v2)" \
  "void glProgramUniform3fvEXT (GLuint program, GLint location, GLsizei count, const GLfloat* value)" \
  "void glProgramUniform3iEXT (GLuint program, GLint location, GLint v0, GLint v1, GLint v2)" \
  "void glProgramUniform3ivEXT (GLuint program, GLint location, GLsizei count, const GLint* value)" \
  "void glProgramUniform3uiEXT (GLuint program, GLint location, GLuint v0, GLuint v1, GLuint v2)" \
  "void glProgramUniform3uivEXT (GLuint program, GLint location, GLsizei count, const GLuint* value)" \
  "void glProgramUniform4fEXT (GLuint program, GLint location, GLfloat v0, GLfloat v1, GLfloat v2, GLfloat v3)" \
  "void glProgramUniform4fvEXT (GLuint program, GLint location, GLsizei count, const GLfloat* value)" \
  "void glProgramUniform4iEXT (GLuint program, GLint location, GLint v0, GLint v1, GLint v2, GLint v3)" \
  "void glProgramUniform4ivEXT (GLuint program, GLint location, GLsizei count, const GLint* value)" \
  "void glProgramUniform4uiEXT (GLuint program, GLint location, GLuint v0, GLuint v1, GLuint v2, GLuint v3)" \
  "void glProgramUniform4uivEXT (GLuint program, GLint location, GLsizei count, const GLuint* value)" \
  "void glProgramUniformMatrix2fvEXT (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glProgramUniformMatrix2x3fvEXT (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glProgramUniformMatrix2x4fvEXT (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glProgramUniformMatrix3fvEXT (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glProgramUniformMatrix3x2fvEXT (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glProgramUniformMatrix3x4fvEXT (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glProgramUniformMatrix4fvEXT (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glProgramUniformMatrix4x2fvEXT (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glProgramUniformMatrix4x3fvEXT (GLuint program, GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glPushClientAttribDefaultEXT (GLbitfield mask)" \
  "void glTextureBufferEXT (GLuint texture, GLenum target, GLenum internalformat, GLuint buffer)" \
  "void glTextureImage1DEXT (GLuint texture, GLenum target, GLint level, GLint internalformat, GLsizei width, GLint border, GLenum format, GLenum type, const void *pixels)" \
  "void glTextureImage2DEXT (GLuint texture, GLenum target, GLint level, GLint internalformat, GLsizei width, GLsizei height, GLint border, GLenum format, GLenum type, const void *pixels)" \
  "void glTextureImage3DEXT (GLuint texture, GLenum target, GLint level, GLint internalformat, GLsizei width, GLsizei height, GLsizei depth, GLint border, GLenum format, GLenum type, const void *pixels)" \
  "void glTextureParameterIivEXT (GLuint texture, GLenum target, GLenum pname, const GLint* params)" \
  "void glTextureParameterIuivEXT (GLuint texture, GLenum target, GLenum pname, const GLuint* params)" \
  "void glTextureParameterfEXT (GLuint texture, GLenum target, GLenum pname, GLfloat param)" \
  "void glTextureParameterfvEXT (GLuint texture, GLenum target, GLenum pname, const GLfloat* param)" \
  "void glTextureParameteriEXT (GLuint texture, GLenum target, GLenum pname, GLint param)" \
  "void glTextureParameterivEXT (GLuint texture, GLenum target, GLenum pname, const GLint* param)" \
  "void glTextureRenderbufferEXT (GLuint texture, GLenum target, GLuint renderbuffer)" \
  "void glTextureSubImage1DEXT (GLuint texture, GLenum target, GLint level, GLint xoffset, GLsizei width, GLenum format, GLenum type, const void *pixels)" \
  "void glTextureSubImage2DEXT (GLuint texture, GLenum target, GLint level, GLint xoffset, GLint yoffset, GLsizei width, GLsizei height, GLenum format, GLenum type, const void *pixels)" \
  "void glTextureSubImage3DEXT (GLuint texture, GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, GLenum format, GLenum type, const void *pixels)" \
  "GLboolean glUnmapNamedBufferEXT (GLuint buffer)" \
  "void glVertexArrayColorOffsetEXT (GLuint vaobj, GLuint buffer, GLint size, GLenum type, GLsizei stride, GLintptr offset)" \
  "void glVertexArrayEdgeFlagOffsetEXT (GLuint vaobj, GLuint buffer, GLsizei stride, GLintptr offset)" \
  "void glVertexArrayFogCoordOffsetEXT (GLuint vaobj, GLuint buffer, GLenum type, GLsizei stride, GLintptr offset)" \
  "void glVertexArrayIndexOffsetEXT (GLuint vaobj, GLuint buffer, GLenum type, GLsizei stride, GLintptr offset)" \
  "void glVertexArrayMultiTexCoordOffsetEXT (GLuint vaobj, GLuint buffer, GLenum texunit, GLint size, GLenum type, GLsizei stride, GLintptr offset)" \
  "void glVertexArrayNormalOffsetEXT (GLuint vaobj, GLuint buffer, GLenum type, GLsizei stride, GLintptr offset)" \
  "void glVertexArraySecondaryColorOffsetEXT (GLuint vaobj, GLuint buffer, GLint size, GLenum type, GLsizei stride, GLintptr offset)" \
  "void glVertexArrayTexCoordOffsetEXT (GLuint vaobj, GLuint buffer, GLint size, GLenum type, GLsizei stride, GLintptr offset)" \
  "void glVertexArrayVertexAttribDivisorEXT (GLuint vaobj, GLuint index, GLuint divisor)" \
  "void glVertexArrayVertexAttribIOffsetEXT (GLuint vaobj, GLuint buffer, GLuint index, GLint size, GLenum type, GLsizei stride, GLintptr offset)" \
  "void glVertexArrayVertexAttribOffsetEXT (GLuint vaobj, GLuint buffer, GLuint index, GLint size, GLenum type, GLboolean normalized, GLsizei stride, GLintptr offset)" \
  "void glVertexArrayVertexOffsetEXT (GLuint vaobj, GLuint buffer, GLint size, GLenum type, GLsizei stride, GLintptr offset)" \
  "void glDiscardFramebufferEXT (GLenum target, GLsizei numAttachments, const GLenum* attachments)" \
  "void glBeginQueryEXT (GLenum target, GLuint id)" \
  "void glDeleteQueriesEXT (GLsizei n, const GLuint* ids)" \
  "void glEndQueryEXT (GLenum target)" \
  "void glGenQueriesEXT (GLsizei n, GLuint* ids)" \
  "void glGetInteger64vEXT (GLenum pname, GLint64* data)" \
  "void glGetQueryObjectivEXT (GLuint id, GLenum pname, GLint* params)" \
  "void glGetQueryObjectuivEXT (GLuint id, GLenum pname, GLuint* params)" \
  "void glGetQueryivEXT (GLenum target, GLenum pname, GLint* params)" \
  "GLboolean glIsQueryEXT (GLuint id)" \
  "void glQueryCounterEXT (GLuint id, GLenum target)" \
  "void glDrawBuffersEXT (GLsizei n, const GLenum* bufs)" \
  "void glColorMaskIndexedEXT (GLuint buf, GLboolean r, GLboolean g, GLboolean b, GLboolean a)" \
  "void glDisableIndexedEXT (GLenum target, GLuint index)" \
  "void glEnableIndexedEXT (GLenum target, GLuint index)" \
  "void glGetBooleanIndexedvEXT (GLenum value, GLuint index, GLboolean* data)" \
  "void glGetIntegerIndexedvEXT (GLenum value, GLuint index, GLint* data)" \
  "GLboolean glIsEnabledIndexedEXT (GLenum target, GLuint index)" \
  "void glBlendEquationSeparateiEXT (GLuint buf, GLenum modeRGB, GLenum modeAlpha)" \
  "void glBlendEquationiEXT (GLuint buf, GLenum mode)" \
  "void glBlendFuncSeparateiEXT (GLuint buf, GLenum srcRGB, GLenum dstRGB, GLenum srcAlpha, GLenum dstAlpha)" \
  "void glBlendFunciEXT (GLuint buf, GLenum src, GLenum dst)" \
  "void glColorMaskiEXT (GLuint buf, GLboolean r, GLboolean g, GLboolean b, GLboolean a)" \
  "void glDisableiEXT (GLenum target, GLuint index)" \
  "void glEnableiEXT (GLenum target, GLuint index)" \
  "GLboolean glIsEnablediEXT (GLenum target, GLuint index)" \
  "void glDrawElementsBaseVertexEXT (GLenum mode, GLsizei count, GLenum type, const void *indices, GLint basevertex)" \
  "void glDrawElementsInstancedBaseVertexEXT (GLenum mode, GLsizei count, GLenum type, const void *indices, GLsizei instancecount, GLint basevertex)" \
  "void glDrawRangeElementsBaseVertexEXT (GLenum mode, GLuint start, GLuint end, GLsizei count, GLenum type, const void *indices, GLint basevertex)" \
  "void glMultiDrawElementsBaseVertexEXT (GLenum mode, const GLsizei* count, GLenum type, const void *const *indices, GLsizei primcount, const GLint *basevertex)" \
  "void glDrawArraysInstancedEXT (GLenum mode, GLint start, GLsizei count, GLsizei primcount)" \
  "void glDrawElementsInstancedEXT (GLenum mode, GLsizei count, GLenum type, const void *indices, GLsizei primcount)" \
  "void glDrawRangeElementsEXT (GLenum mode, GLuint start, GLuint end, GLsizei count, GLenum type, const void *indices)" \
  "void glDrawTransformFeedbackEXT (GLenum mode, GLuint id)" \
  "void glDrawTransformFeedbackInstancedEXT (GLenum mode, GLuint id, GLsizei instancecount)" \
  "void glBufferStorageExternalEXT (GLenum target, GLintptr offset, GLsizeiptr size, GLeglClientBufferEXT clientBuffer, GLbitfield flags)" \
  "void glNamedBufferStorageExternalEXT (GLuint buffer, GLintptr offset, GLsizeiptr size, GLeglClientBufferEXT clientBuffer, GLbitfield flags)" \
  "void glFogCoordPointerEXT (GLenum type, GLsizei stride, const void *pointer)" \
  "void glFogCoorddEXT (GLdouble coord)" \
  "void glFogCoorddvEXT (const GLdouble *coord)" \
  "void glFogCoordfEXT (GLfloat coord)" \
  "void glFogCoordfvEXT (const GLfloat *coord)" \
  "void glFragmentColorMaterialEXT (GLenum face, GLenum mode)" \
  "void glFragmentLightModelfEXT (GLenum pname, GLfloat param)" \
  "void glFragmentLightModelfvEXT (GLenum pname, GLfloat* params)" \
  "void glFragmentLightModeliEXT (GLenum pname, GLint param)" \
  "void glFragmentLightModelivEXT (GLenum pname, GLint* params)" \
  "void glFragmentLightfEXT (GLenum light, GLenum pname, GLfloat param)" \
  "void glFragmentLightfvEXT (GLenum light, GLenum pname, GLfloat* params)" \
  "void glFragmentLightiEXT (GLenum light, GLenum pname, GLint param)" \
  "void glFragmentLightivEXT (GLenum light, GLenum pname, GLint* params)" \
  "void glFragmentMaterialfEXT (GLenum face, GLenum pname, const GLfloat param)" \
  "void glFragmentMaterialfvEXT (GLenum face, GLenum pname, const GLfloat* params)" \
  "void glFragmentMaterialiEXT (GLenum face, GLenum pname, const GLint param)" \
  "void glFragmentMaterialivEXT (GLenum face, GLenum pname, const GLint* params)" \
  "void glGetFragmentLightfvEXT (GLenum light, GLenum pname, GLfloat* params)" \
  "void glGetFragmentLightivEXT (GLenum light, GLenum pname, GLint* params)" \
  "void glGetFragmentMaterialfvEXT (GLenum face, GLenum pname, const GLfloat* params)" \
  "void glGetFragmentMaterialivEXT (GLenum face, GLenum pname, const GLint* params)" \
  "void glLightEnviEXT (GLenum pname, GLint param)" \
  "void glBlitFramebufferEXT (GLint srcX0, GLint srcY0, GLint srcX1, GLint srcY1, GLint dstX0, GLint dstY0, GLint dstX1, GLint dstY1, GLbitfield mask, GLenum filter)" \
  "void glRenderbufferStorageMultisampleEXT (GLenum target, GLsizei samples, GLenum internalformat, GLsizei width, GLsizei height)" \
  "void glBindFramebufferEXT (GLenum target, GLuint framebuffer)" \
  "void glBindRenderbufferEXT (GLenum target, GLuint renderbuffer)" \
  "GLenum glCheckFramebufferStatusEXT (GLenum target)" \
  "void glDeleteFramebuffersEXT (GLsizei n, const GLuint* framebuffers)" \
  "void glDeleteRenderbuffersEXT (GLsizei n, const GLuint* renderbuffers)" \
  "void glFramebufferRenderbufferEXT (GLenum target, GLenum attachment, GLenum renderbuffertarget, GLuint renderbuffer)" \
  "void glFramebufferTexture1DEXT (GLenum target, GLenum attachment, GLenum textarget, GLuint texture, GLint level)" \
  "void glFramebufferTexture2DEXT (GLenum target, GLenum attachment, GLenum textarget, GLuint texture, GLint level)" \
  "void glFramebufferTexture3DEXT (GLenum target, GLenum attachment, GLenum textarget, GLuint texture, GLint level, GLint zoffset)" \
  "void glGenFramebuffersEXT (GLsizei n, GLuint* framebuffers)" \
  "void glGenRenderbuffersEXT (GLsizei n, GLuint* renderbuffers)" \
  "void glGenerateMipmapEXT (GLenum target)" \
  "void glGetFramebufferAttachmentParameterivEXT (GLenum target, GLenum attachment, GLenum pname, GLint* params)" \
  "void glGetRenderbufferParameterivEXT (GLenum target, GLenum pname, GLint* params)" \
  "GLboolean glIsFramebufferEXT (GLuint framebuffer)" \
  "GLboolean glIsRenderbufferEXT (GLuint renderbuffer)" \
  "void glRenderbufferStorageEXT (GLenum target, GLenum internalformat, GLsizei width, GLsizei height)" \
  "void glFramebufferTextureEXT (GLenum target, GLenum attachment, GLuint texture, GLint level)" \
  "void glFramebufferTextureFaceEXT (GLenum target, GLenum attachment, GLuint texture, GLint level, GLenum face)" \
  "void glProgramParameteriEXT (GLuint program, GLenum pname, GLint value)" \
  "void glProgramEnvParameters4fvEXT (GLenum target, GLuint index, GLsizei count, const GLfloat* params)" \
  "void glProgramLocalParameters4fvEXT (GLenum target, GLuint index, GLsizei count, const GLfloat* params)" \
  "void glBindFragDataLocationEXT (GLuint program, GLuint color, const GLchar *name)" \
  "GLint glGetFragDataLocationEXT (GLuint program, const GLchar *name)" \
  "void glGetUniformuivEXT (GLuint program, GLint location, GLuint *params)" \
  "void glGetVertexAttribIivEXT (GLuint index, GLenum pname, GLint *params)" \
  "void glGetVertexAttribIuivEXT (GLuint index, GLenum pname, GLuint *params)" \
  "void glUniform1uiEXT (GLint location, GLuint v0)" \
  "void glUniform1uivEXT (GLint location, GLsizei count, const GLuint *value)" \
  "void glUniform2uiEXT (GLint location, GLuint v0, GLuint v1)" \
  "void glUniform2uivEXT (GLint location, GLsizei count, const GLuint *value)" \
  "void glUniform3uiEXT (GLint location, GLuint v0, GLuint v1, GLuint v2)" \
  "void glUniform3uivEXT (GLint location, GLsizei count, const GLuint *value)" \
  "void glUniform4uiEXT (GLint location, GLuint v0, GLuint v1, GLuint v2, GLuint v3)" \
  "void glUniform4uivEXT (GLint location, GLsizei count, const GLuint *value)" \
  "void glVertexAttribI1iEXT (GLuint index, GLint x)" \
  "void glVertexAttribI1ivEXT (GLuint index, const GLint *v)" \
  "void glVertexAttribI1uiEXT (GLuint index, GLuint x)" \
  "void glVertexAttribI1uivEXT (GLuint index, const GLuint *v)" \
  "void glVertexAttribI2iEXT (GLuint index, GLint x, GLint y)" \
  "void glVertexAttribI2ivEXT (GLuint index, const GLint *v)" \
  "void glVertexAttribI2uiEXT (GLuint index, GLuint x, GLuint y)" \
  "void glVertexAttribI2uivEXT (GLuint index, const GLuint *v)" \
  "void glVertexAttribI3iEXT (GLuint index, GLint x, GLint y, GLint z)" \
  "void glVertexAttribI3ivEXT (GLuint index, const GLint *v)" \
  "void glVertexAttribI3uiEXT (GLuint index, GLuint x, GLuint y, GLuint z)" \
  "void glVertexAttribI3uivEXT (GLuint index, const GLuint *v)" \
  "void glVertexAttribI4bvEXT (GLuint index, const GLbyte *v)" \
  "void glVertexAttribI4iEXT (GLuint index, GLint x, GLint y, GLint z, GLint w)" \
  "void glVertexAttribI4ivEXT (GLuint index, const GLint *v)" \
  "void glVertexAttribI4svEXT (GLuint index, const GLshort *v)" \
  "void glVertexAttribI4ubvEXT (GLuint index, const GLubyte *v)" \
  "void glVertexAttribI4uiEXT (GLuint index, GLuint x, GLuint y, GLuint z, GLuint w)" \
  "void glVertexAttribI4uivEXT (GLuint index, const GLuint *v)" \
  "void glVertexAttribI4usvEXT (GLuint index, const GLushort *v)" \
  "void glVertexAttribIPointerEXT (GLuint index, GLint size, GLenum type, GLsizei stride, const void *pointer)" \
  "void glGetHistogramEXT (GLenum target, GLboolean reset, GLenum format, GLenum type, void *values)" \
  "void glGetHistogramParameterfvEXT (GLenum target, GLenum pname, GLfloat* params)" \
  "void glGetHistogramParameterivEXT (GLenum target, GLenum pname, GLint* params)" \
  "void glGetMinmaxEXT (GLenum target, GLboolean reset, GLenum format, GLenum type, void *values)" \
  "void glGetMinmaxParameterfvEXT (GLenum target, GLenum pname, GLfloat* params)" \
  "void glGetMinmaxParameterivEXT (GLenum target, GLenum pname, GLint* params)" \
  "void glHistogramEXT (GLenum target, GLsizei width, GLenum internalformat, GLboolean sink)" \
  "void glMinmaxEXT (GLenum target, GLenum internalformat, GLboolean sink)" \
  "void glResetHistogramEXT (GLenum target)" \
  "void glResetMinmaxEXT (GLenum target)" \
  "void glIndexFuncEXT (GLenum func, GLfloat ref)" \
  "void glIndexMaterialEXT (GLenum face, GLenum mode)" \
  "void glVertexAttribDivisorEXT (GLuint index, GLuint divisor)" \
  "void glApplyTextureEXT (GLenum mode)" \
  "void glTextureLightEXT (GLenum pname)" \
  "void glTextureMaterialEXT (GLenum face, GLenum mode)" \
  "void glFlushMappedBufferRangeEXT (GLenum target, GLintptr offset, GLsizeiptr length)" \
  "void * glMapBufferRangeEXT (GLenum target, GLintptr offset, GLsizeiptr length, GLbitfield access)" \
  "void glBufferStorageMemEXT (GLenum target, GLsizeiptr size, GLuint memory, GLuint64 offset)" \
  "void glCreateMemoryObjectsEXT (GLsizei n, GLuint* memoryObjects)" \
  "void glDeleteMemoryObjectsEXT (GLsizei n, const GLuint* memoryObjects)" \
  "void glGetMemoryObjectParameterivEXT (GLuint memoryObject, GLenum pname, GLint* params)" \
  "void glGetUnsignedBytei_vEXT (GLenum target, GLuint index, GLubyte* data)" \
  "void glGetUnsignedBytevEXT (GLenum pname, GLubyte* data)" \
  "GLboolean glIsMemoryObjectEXT (GLuint memoryObject)" \
  "void glMemoryObjectParameterivEXT (GLuint memoryObject, GLenum pname, const GLint* params)" \
  "void glNamedBufferStorageMemEXT (GLuint buffer, GLsizeiptr size, GLuint memory, GLuint64 offset)" \
  "void glTexStorageMem1DEXT (GLenum target, GLsizei levels, GLenum internalFormat, GLsizei width, GLuint memory, GLuint64 offset)" \
  "void glTexStorageMem2DEXT (GLenum target, GLsizei levels, GLenum internalFormat, GLsizei width, GLsizei height, GLuint memory, GLuint64 offset)" \
  "void glTexStorageMem2DMultisampleEXT (GLenum target, GLsizei samples, GLenum internalFormat, GLsizei width, GLsizei height, GLboolean fixedSampleLocations, GLuint memory, GLuint64 offset)" \
  "void glTexStorageMem3DEXT (GLenum target, GLsizei levels, GLenum internalFormat, GLsizei width, GLsizei height, GLsizei depth, GLuint memory, GLuint64 offset)" \
  "void glTexStorageMem3DMultisampleEXT (GLenum target, GLsizei samples, GLenum internalFormat, GLsizei width, GLsizei height, GLsizei depth, GLboolean fixedSampleLocations, GLuint memory, GLuint64 offset)" \
  "void glTextureStorageMem1DEXT (GLuint texture, GLsizei levels, GLenum internalFormat, GLsizei width, GLuint memory, GLuint64 offset)" \
  "void glTextureStorageMem2DEXT (GLuint texture, GLsizei levels, GLenum internalFormat, GLsizei width, GLsizei height, GLuint memory, GLuint64 offset)" \
  "void glTextureStorageMem2DMultisampleEXT (GLuint texture, GLsizei samples, GLenum internalFormat, GLsizei width, GLsizei height, GLboolean fixedSampleLocations, GLuint memory, GLuint64 offset)" \
  "void glTextureStorageMem3DEXT (GLuint texture, GLsizei levels, GLenum internalFormat, GLsizei width, GLsizei height, GLsizei depth, GLuint memory, GLuint64 offset)" \
  "void glTextureStorageMem3DMultisampleEXT (GLuint texture, GLsizei samples, GLenum internalFormat, GLsizei width, GLsizei height, GLsizei depth, GLboolean fixedSampleLocations, GLuint memory, GLuint64 offset)" \
  "void glImportMemoryFdEXT (GLuint memory, GLuint64 size, GLenum handleType, GLint fd)" \
  "void glImportMemoryWin32HandleEXT (GLuint memory, GLuint64 size, GLenum handleType, void *handle)" \
  "void glImportMemoryWin32NameEXT (GLuint memory, GLuint64 size, GLenum handleType, const void *name)" \
  "void glMultiDrawArraysEXT (GLenum mode, const GLint* first, const GLsizei *count, GLsizei primcount)" \
  "void glMultiDrawElementsEXT (GLenum mode, GLsizei* count, GLenum type, const void *const *indices, GLsizei primcount)" \
  "void glMultiDrawArraysIndirectEXT (GLenum mode, const void *indirect, GLsizei drawcount, GLsizei stride)" \
  "void glMultiDrawElementsIndirectEXT (GLenum mode, GLenum type, const void *indirect, GLsizei drawcount, GLsizei stride)" \
  "void glSampleMaskEXT (GLclampf value, GLboolean invert)" \
  "void glSamplePatternEXT (GLenum pattern)" \
  "void glFramebufferTexture2DMultisampleEXT (GLenum target, GLenum attachment, GLenum textarget, GLuint texture, GLint level, GLsizei samples)" \
  "void glDrawBuffersIndexedEXT (GLint n, const GLenum* location, const GLint *indices)" \
  "void glGetIntegeri_vEXT (GLenum target, GLuint index, GLint* data)" \
  "void glReadBufferIndexedEXT (GLenum src, GLint index)" \
  "void glColorTableEXT (GLenum target, GLenum internalFormat, GLsizei width, GLenum format, GLenum type, const void *data)" \
  "void glGetColorTableEXT (GLenum target, GLenum format, GLenum type, void *data)" \
  "void glGetColorTableParameterfvEXT (GLenum target, GLenum pname, GLfloat* params)" \
  "void glGetColorTableParameterivEXT (GLenum target, GLenum pname, GLint* params)" \
  "void glGetPixelTransformParameterfvEXT (GLenum target, GLenum pname, const GLfloat* params)" \
  "void glGetPixelTransformParameterivEXT (GLenum target, GLenum pname, const GLint* params)" \
  "void glPixelTransformParameterfEXT (GLenum target, GLenum pname, const GLfloat param)" \
  "void glPixelTransformParameterfvEXT (GLenum target, GLenum pname, const GLfloat* params)" \
  "void glPixelTransformParameteriEXT (GLenum target, GLenum pname, const GLint param)" \
  "void glPixelTransformParameterivEXT (GLenum target, GLenum pname, const GLint* params)" \
  "void glPointParameterfEXT (GLenum pname, GLfloat param)" \
  "void glPointParameterfvEXT (GLenum pname, const GLfloat* params)" \
  "void glPolygonOffsetEXT (GLfloat factor, GLfloat bias)" \
  "void glPolygonOffsetClampEXT (GLfloat factor, GLfloat units, GLfloat clamp)" \
  "void glPrimitiveBoundingBoxEXT (GLfloat minX, GLfloat minY, GLfloat minZ, GLfloat minW, GLfloat maxX, GLfloat maxY, GLfloat maxZ, GLfloat maxW)" \
  "void glProvokingVertexEXT (GLenum mode)" \
  "void glCoverageModulationNV (GLenum components)" \
  "void glCoverageModulationTableNV (GLsizei n, const GLfloat* v)" \
  "void glGetCoverageModulationTableNV (GLsizei bufsize, GLfloat* v)" \
  "void glRasterSamplesEXT (GLuint samples, GLboolean fixedsamplelocations)" \
  "void glGetnUniformfvEXT (GLuint program, GLint location, GLsizei bufSize, GLfloat* params)" \
  "void glGetnUniformivEXT (GLuint program, GLint location, GLsizei bufSize, GLint* params)" \
  "void glReadnPixelsEXT (GLint x, GLint y, GLsizei width, GLsizei height, GLenum format, GLenum type, GLsizei bufSize, void *data)" \
  "void glBeginSceneEXT (void)" \
  "void glEndSceneEXT (void)" \
  "void glSecondaryColor3bEXT (GLbyte red, GLbyte green, GLbyte blue)" \
  "void glSecondaryColor3bvEXT (const GLbyte *v)" \
  "void glSecondaryColor3dEXT (GLdouble red, GLdouble green, GLdouble blue)" \
  "void glSecondaryColor3dvEXT (const GLdouble *v)" \
  "void glSecondaryColor3fEXT (GLfloat red, GLfloat green, GLfloat blue)" \
  "void glSecondaryColor3fvEXT (const GLfloat *v)" \
  "void glSecondaryColor3iEXT (GLint red, GLint green, GLint blue)" \
  "void glSecondaryColor3ivEXT (const GLint *v)" \
  "void glSecondaryColor3sEXT (GLshort red, GLshort green, GLshort blue)" \
  "void glSecondaryColor3svEXT (const GLshort *v)" \
  "void glSecondaryColor3ubEXT (GLubyte red, GLubyte green, GLubyte blue)" \
  "void glSecondaryColor3ubvEXT (const GLubyte *v)" \
  "void glSecondaryColor3uiEXT (GLuint red, GLuint green, GLuint blue)" \
  "void glSecondaryColor3uivEXT (const GLuint *v)" \
  "void glSecondaryColor3usEXT (GLushort red, GLushort green, GLushort blue)" \
  "void glSecondaryColor3usvEXT (const GLushort *v)" \
  "void glSecondaryColorPointerEXT (GLint size, GLenum type, GLsizei stride, const void *pointer)" \
  "void glDeleteSemaphoresEXT (GLsizei n, const GLuint* semaphores)" \
  "void glGenSemaphoresEXT (GLsizei n, GLuint* semaphores)" \
  "void glGetSemaphoreParameterui64vEXT (GLuint semaphore, GLenum pname, GLuint64* params)" \
  "GLboolean glIsSemaphoreEXT (GLuint semaphore)" \
  "void glSemaphoreParameterui64vEXT (GLuint semaphore, GLenum pname, const GLuint64* params)" \
  "void glSignalSemaphoreEXT (GLuint semaphore, GLuint numBufferBarriers, const GLuint* buffers, GLuint numTextureBarriers, const GLuint *textures, const GLenum *dstLayouts)" \
  "void glWaitSemaphoreEXT (GLuint semaphore, GLuint numBufferBarriers, const GLuint* buffers, GLuint numTextureBarriers, const GLuint *textures, const GLenum *srcLayouts)" \
  "void glImportSemaphoreFdEXT (GLuint semaphore, GLenum handleType, GLint fd)" \
  "void glImportSemaphoreWin32HandleEXT (GLuint semaphore, GLenum handleType, void *handle)" \
  "void glImportSemaphoreWin32NameEXT (GLuint semaphore, GLenum handleType, const void *name)" \
  "void glActiveProgramEXT (GLuint program)" \
  "GLuint glCreateShaderProgramEXT (GLenum type, const GLchar* string)" \
  "void glUseShaderProgramEXT (GLenum type, GLuint program)" \
  "void glFramebufferFetchBarrierEXT (void)" \
  "void glBindImageTextureEXT (GLuint index, GLuint texture, GLint level, GLboolean layered, GLint layer, GLenum access, GLint format)" \
  "void glMemoryBarrierEXT (GLbitfield barriers)" \
  "void glClearPixelLocalStorageuiEXT (GLsizei offset, GLsizei n, const GLuint* values)" \
  "void glFramebufferPixelLocalStorageSizeEXT (GLuint target, GLsizei size)" \
  "GLsizei glGetFramebufferPixelLocalStorageSizeEXT (GLuint target)" \
  "void glTexPageCommitmentEXT (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, GLboolean commit)" \
  "void glTexturePageCommitmentEXT (GLuint texture, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, GLboolean commit)" \
  "void glActiveStencilFaceEXT (GLenum face)" \
  "void glTexSubImage1DEXT (GLenum target, GLint level, GLint xoffset, GLsizei width, GLenum format, GLenum type, const void *pixels)" \
  "void glTexSubImage2DEXT (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLsizei width, GLsizei height, GLenum format, GLenum type, const void *pixels)" \
  "void glTexSubImage3DEXT (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, GLenum format, GLenum type, const void *pixels)" \
  "void glPatchParameteriEXT (GLenum pname, GLint value)" \
  "void glTexImage3DEXT (GLenum target, GLint level, GLenum internalformat, GLsizei width, GLsizei height, GLsizei depth, GLint border, GLenum format, GLenum type, const void *pixels)" \
  "void glFramebufferTextureLayerEXT (GLenum target, GLenum attachment, GLuint texture, GLint level, GLint layer)" \
  "void glGetSamplerParameterIivEXT (GLuint sampler, GLenum pname, GLint* params)" \
  "void glGetSamplerParameterIuivEXT (GLuint sampler, GLenum pname, GLuint* params)" \
  "void glSamplerParameterIivEXT (GLuint sampler, GLenum pname, const GLint* params)" \
  "void glSamplerParameterIuivEXT (GLuint sampler, GLenum pname, const GLuint* params)" \
  "void glTexBufferEXT (GLenum target, GLenum internalformat, GLuint buffer)" \
  "void glClearColorIiEXT (GLint red, GLint green, GLint blue, GLint alpha)" \
  "void glClearColorIuiEXT (GLuint red, GLuint green, GLuint blue, GLuint alpha)" \
  "void glGetTexParameterIivEXT (GLenum target, GLenum pname, GLint *params)" \
  "void glGetTexParameterIuivEXT (GLenum target, GLenum pname, GLuint *params)" \
  "void glTexParameterIivEXT (GLenum target, GLenum pname, const GLint *params)" \
  "void glTexParameterIuivEXT (GLenum target, GLenum pname, const GLuint *params)" \
  "GLboolean glAreTexturesResidentEXT (GLsizei n, const GLuint* textures, GLboolean* residences)" \
  "void glBindTextureEXT (GLenum target, GLuint texture)" \
  "void glDeleteTexturesEXT (GLsizei n, const GLuint* textures)" \
  "void glGenTexturesEXT (GLsizei n, GLuint* textures)" \
  "GLboolean glIsTextureEXT (GLuint texture)" \
  "void glPrioritizeTexturesEXT (GLsizei n, const GLuint* textures, const GLclampf* priorities)" \
  "void glTextureNormalEXT (GLenum mode)" \
  "void glTexStorage1DEXT (GLenum target, GLsizei levels, GLenum internalformat, GLsizei width)" \
  "void glTexStorage2DEXT (GLenum target, GLsizei levels, GLenum internalformat, GLsizei width, GLsizei height)" \
  "void glTexStorage3DEXT (GLenum target, GLsizei levels, GLenum internalformat, GLsizei width, GLsizei height, GLsizei depth)" \
  "void glTextureStorage1DEXT (GLuint texture, GLenum target, GLsizei levels, GLenum internalformat, GLsizei width)" \
  "void glTextureStorage2DEXT (GLuint texture, GLenum target, GLsizei levels, GLenum internalformat, GLsizei width, GLsizei height)" \
  "void glTextureStorage3DEXT (GLuint texture, GLenum target, GLsizei levels, GLenum internalformat, GLsizei width, GLsizei height, GLsizei depth)" \
  "void glTextureViewEXT (GLuint texture, GLenum target, GLuint origtexture, GLenum internalformat, GLuint minlevel, GLuint numlevels, GLuint minlayer, GLuint numlayers)" \
  "void glGetQueryObjecti64vEXT (GLuint id, GLenum pname, GLint64EXT *params)" \
  "void glGetQueryObjectui64vEXT (GLuint id, GLenum pname, GLuint64EXT *params)" \
  "void glBeginTransformFeedbackEXT (GLenum primitiveMode)" \
  "void glBindBufferBaseEXT (GLenum target, GLuint index, GLuint buffer)" \
  "void glBindBufferOffsetEXT (GLenum target, GLuint index, GLuint buffer, GLintptr offset)" \
  "void glBindBufferRangeEXT (GLenum target, GLuint index, GLuint buffer, GLintptr offset, GLsizeiptr size)" \
  "void glEndTransformFeedbackEXT (void)" \
  "void glGetTransformFeedbackVaryingEXT (GLuint program, GLuint index, GLsizei bufSize, GLsizei* length, GLsizei *size, GLenum *type, GLchar *name)" \
  "void glTransformFeedbackVaryingsEXT (GLuint program, GLsizei count, const GLchar * const* varyings, GLenum bufferMode)" \
  "void glArrayElementEXT (GLint i)" \
  "void glColorPointerEXT (GLint size, GLenum type, GLsizei stride, GLsizei count, const void *pointer)" \
  "void glDrawArraysEXT (GLenum mode, GLint first, GLsizei count)" \
  "void glEdgeFlagPointerEXT (GLsizei stride, GLsizei count, const GLboolean* pointer)" \
  "void glIndexPointerEXT (GLenum type, GLsizei stride, GLsizei count, const void *pointer)" \
  "void glNormalPointerEXT (GLenum type, GLsizei stride, GLsizei count, const void *pointer)" \
  "void glTexCoordPointerEXT (GLint size, GLenum type, GLsizei stride, GLsizei count, const void *pointer)" \
  "void glVertexPointerEXT (GLint size, GLenum type, GLsizei stride, GLsizei count, const void *pointer)" \
  "void glBindArraySetEXT (const void *arrayset)" \
  "const void * glCreateArraySetExt (void)" \
  "void glDeleteArraySetsEXT (GLsizei n, const void **arrayset)" \
  "void glGetVertexAttribLdvEXT (GLuint index, GLenum pname, GLdouble* params)" \
  "void glVertexArrayVertexAttribLOffsetEXT (GLuint vaobj, GLuint buffer, GLuint index, GLint size, GLenum type, GLsizei stride, GLintptr offset)" \
  "void glVertexAttribL1dEXT (GLuint index, GLdouble x)" \
  "void glVertexAttribL1dvEXT (GLuint index, const GLdouble* v)" \
  "void glVertexAttribL2dEXT (GLuint index, GLdouble x, GLdouble y)" \
  "void glVertexAttribL2dvEXT (GLuint index, const GLdouble* v)" \
  "void glVertexAttribL3dEXT (GLuint index, GLdouble x, GLdouble y, GLdouble z)" \
  "void glVertexAttribL3dvEXT (GLuint index, const GLdouble* v)" \
  "void glVertexAttribL4dEXT (GLuint index, GLdouble x, GLdouble y, GLdouble z, GLdouble w)" \
  "void glVertexAttribL4dvEXT (GLuint index, const GLdouble* v)" \
  "void glVertexAttribLPointerEXT (GLuint index, GLint size, GLenum type, GLsizei stride, const void *pointer)" \
  "void glBeginVertexShaderEXT (void)" \
  "GLuint glBindLightParameterEXT (GLenum light, GLenum value)" \
  "GLuint glBindMaterialParameterEXT (GLenum face, GLenum value)" \
  "GLuint glBindParameterEXT (GLenum value)" \
  "GLuint glBindTexGenParameterEXT (GLenum unit, GLenum coord, GLenum value)" \
  "GLuint glBindTextureUnitParameterEXT (GLenum unit, GLenum value)" \
  "void glBindVertexShaderEXT (GLuint id)" \
  "void glDeleteVertexShaderEXT (GLuint id)" \
  "void glDisableVariantClientStateEXT (GLuint id)" \
  "void glEnableVariantClientStateEXT (GLuint id)" \
  "void glEndVertexShaderEXT (void)" \
  "void glExtractComponentEXT (GLuint res, GLuint src, GLuint num)" \
  "GLuint glGenSymbolsEXT (GLenum dataType, GLenum storageType, GLenum range, GLuint components)" \
  "GLuint glGenVertexShadersEXT (GLuint range)" \
  "void glGetInvariantBooleanvEXT (GLuint id, GLenum value, GLboolean *data)" \
  "void glGetInvariantFloatvEXT (GLuint id, GLenum value, GLfloat *data)" \
  "void glGetInvariantIntegervEXT (GLuint id, GLenum value, GLint *data)" \
  "void glGetLocalConstantBooleanvEXT (GLuint id, GLenum value, GLboolean *data)" \
  "void glGetLocalConstantFloatvEXT (GLuint id, GLenum value, GLfloat *data)" \
  "void glGetLocalConstantIntegervEXT (GLuint id, GLenum value, GLint *data)" \
  "void glGetVariantBooleanvEXT (GLuint id, GLenum value, GLboolean *data)" \
  "void glGetVariantFloatvEXT (GLuint id, GLenum value, GLfloat *data)" \
  "void glGetVariantIntegervEXT (GLuint id, GLenum value, GLint *data)" \
  "void glGetVariantPointervEXT (GLuint id, GLenum value, void **data)" \
  "void glInsertComponentEXT (GLuint res, GLuint src, GLuint num)" \
  "GLboolean glIsVariantEnabledEXT (GLuint id, GLenum cap)" \
  "void glSetInvariantEXT (GLuint id, GLenum type, void *addr)" \
  "void glSetLocalConstantEXT (GLuint id, GLenum type, void *addr)" \
  "void glShaderOp1EXT (GLenum op, GLuint res, GLuint arg1)" \
  "void glShaderOp2EXT (GLenum op, GLuint res, GLuint arg1, GLuint arg2)" \
  "void glShaderOp3EXT (GLenum op, GLuint res, GLuint arg1, GLuint arg2, GLuint arg3)" \
  "void glSwizzleEXT (GLuint res, GLuint in, GLenum outX, GLenum outY, GLenum outZ, GLenum outW)" \
  "void glVariantPointerEXT (GLuint id, GLenum type, GLuint stride, void *addr)" \
  "void glVariantbvEXT (GLuint id, GLbyte *addr)" \
  "void glVariantdvEXT (GLuint id, GLdouble *addr)" \
  "void glVariantfvEXT (GLuint id, GLfloat *addr)" \
  "void glVariantivEXT (GLuint id, GLint *addr)" \
  "void glVariantsvEXT (GLuint id, GLshort *addr)" \
  "void glVariantubvEXT (GLuint id, GLubyte *addr)" \
  "void glVariantuivEXT (GLuint id, GLuint *addr)" \
  "void glVariantusvEXT (GLuint id, GLushort *addr)" \
  "void glWriteMaskEXT (GLuint res, GLuint in, GLenum outX, GLenum outY, GLenum outZ, GLenum outW)" \
  "void glVertexWeightPointerEXT (GLint size, GLenum type, GLsizei stride, void *pointer)" \
  "void glVertexWeightfEXT (GLfloat weight)" \
  "void glVertexWeightfvEXT (GLfloat* weight)" \
  "GLboolean glAcquireKeyedMutexWin32EXT (GLuint memory, GLuint64 key, GLuint timeout)" \
  "GLboolean glReleaseKeyedMutexWin32EXT (GLuint memory, GLuint64 key)" \
  "void glWindowRectanglesEXT (GLenum mode, GLsizei count, const GLint *box)" \
  "GLsync glImportSyncEXT (GLenum external_sync_type, GLintptr external_sync, GLbitfield flags)" \
  "void glFrameTerminatorGREMEDY (void)" \
  "void glStringMarkerGREMEDY (GLsizei len, const void *string)" \
  "void glGetImageTransformParameterfvHP (GLenum target, GLenum pname, const GLfloat* params)" \
  "void glGetImageTransformParameterivHP (GLenum target, GLenum pname, const GLint* params)" \
  "void glImageTransformParameterfHP (GLenum target, GLenum pname, const GLfloat param)" \
  "void glImageTransformParameterfvHP (GLenum target, GLenum pname, const GLfloat* params)" \
  "void glImageTransformParameteriHP (GLenum target, GLenum pname, const GLint param)" \
  "void glImageTransformParameterivHP (GLenum target, GLenum pname, const GLint* params)" \
  "void glMultiModeDrawArraysIBM (const GLenum* mode, const GLint *first, const GLsizei *count, GLsizei primcount, GLint modestride)" \
  "void glMultiModeDrawElementsIBM (const GLenum* mode, const GLsizei *count, GLenum type, const void *const *indices, GLsizei primcount, GLint modestride)" \
  "void glColorPointerListIBM (GLint size, GLenum type, GLint stride, const void** pointer, GLint ptrstride)" \
  "void glEdgeFlagPointerListIBM (GLint stride, const GLboolean ** pointer, GLint ptrstride)" \
  "void glFogCoordPointerListIBM (GLenum type, GLint stride, const void** pointer, GLint ptrstride)" \
  "void glIndexPointerListIBM (GLenum type, GLint stride, const void** pointer, GLint ptrstride)" \
  "void glNormalPointerListIBM (GLenum type, GLint stride, const void** pointer, GLint ptrstride)" \
  "void glSecondaryColorPointerListIBM (GLint size, GLenum type, GLint stride, const void** pointer, GLint ptrstride)" \
  "void glTexCoordPointerListIBM (GLint size, GLenum type, GLint stride, const void** pointer, GLint ptrstride)" \
  "void glVertexPointerListIBM (GLint size, GLenum type, GLint stride, const void** pointer, GLint ptrstride)" \
  "GLuint64 glGetTextureHandleIMG (GLuint texture)" \
  "GLuint64 glGetTextureSamplerHandleIMG (GLuint texture, GLuint sampler)" \
  "void glProgramUniformHandleui64IMG (GLuint program, GLint location, GLuint64 value)" \
  "void glProgramUniformHandleui64vIMG (GLuint program, GLint location, GLsizei count, const GLuint64* values)" \
  "void glUniformHandleui64IMG (GLint location, GLuint64 value)" \
  "void glUniformHandleui64vIMG (GLint location, GLsizei count, const GLuint64* value)" \
  "void glFramebufferTexture2DDownsampleIMG (GLenum target, GLenum attachment, GLenum textarget, GLuint texture, GLint level, GLint xscale, GLint yscale)" \
  "void glFramebufferTextureLayerDownsampleIMG (GLenum target, GLenum attachment, GLuint texture, GLint level, GLint layer, GLint xscale, GLint yscale)" \
  "void glFramebufferTexture2DMultisampleIMG (GLenum target, GLenum attachment, GLenum textarget, GLuint texture, GLint level, GLsizei samples)" \
  "void glRenderbufferStorageMultisampleIMG (GLenum target, GLsizei samples, GLenum internalformat, GLsizei width, GLsizei height)" \
  "void * glMapTexture2DINTEL (GLuint texture, GLint level, GLbitfield access, GLint* stride, GLenum *layout)" \
  "void glSyncTextureINTEL (GLuint texture)" \
  "void glUnmapTexture2DINTEL (GLuint texture, GLint level)" \
  "void glColorPointervINTEL (GLint size, GLenum type, const void** pointer)" \
  "void glNormalPointervINTEL (GLenum type, const void** pointer)" \
  "void glTexCoordPointervINTEL (GLint size, GLenum type, const void** pointer)" \
  "void glVertexPointervINTEL (GLint size, GLenum type, const void** pointer)" \
  "void glBeginPerfQueryINTEL (GLuint queryHandle)" \
  "void glCreatePerfQueryINTEL (GLuint queryId, GLuint* queryHandle)" \
  "void glDeletePerfQueryINTEL (GLuint queryHandle)" \
  "void glEndPerfQueryINTEL (GLuint queryHandle)" \
  "void glGetFirstPerfQueryIdINTEL (GLuint* queryId)" \
  "void glGetNextPerfQueryIdINTEL (GLuint queryId, GLuint* nextQueryId)" \
  "void glGetPerfCounterInfoINTEL (GLuint queryId, GLuint counterId, GLuint counterNameLength, GLchar* counterName, GLuint counterDescLength, GLchar *counterDesc, GLuint *counterOffset, GLuint *counterDataSize, GLuint *counterTypeEnum, GLuint *counterDataTypeEnum, GLuint64 *rawCounterMaxValue)" \
  "void glGetPerfQueryDataINTEL (GLuint queryHandle, GLuint flags, GLsizei dataSize, void *data, GLuint *bytesWritten)" \
  "void glGetPerfQueryIdByNameINTEL (GLchar* queryName, GLuint *queryId)" \
  "void glGetPerfQueryInfoINTEL (GLuint queryId, GLuint queryNameLength, GLchar* queryName, GLuint *dataSize, GLuint *noCounters, GLuint *noInstances, GLuint *capsMask)" \
  "void glTexScissorFuncINTEL (GLenum target, GLenum lfunc, GLenum hfunc)" \
  "void glTexScissorINTEL (GLenum target, GLclampf tlow, GLclampf thigh)" \
  "void glBlendBarrierKHR (void)" \
  "void glDebugMessageCallback (GLDEBUGPROC callback, const void *userParam)" \
  "void glDebugMessageControl (GLenum source, GLenum type, GLenum severity, GLsizei count, const GLuint* ids, GLboolean enabled)" \
  "void glDebugMessageInsert (GLenum source, GLenum type, GLuint id, GLenum severity, GLsizei length, const GLchar* buf)" \
  "GLuint glGetDebugMessageLog (GLuint count, GLsizei bufSize, GLenum* sources, GLenum* types, GLuint* ids, GLenum* severities, GLsizei* lengths, GLchar* messageLog)" \
  "void glGetObjectLabel (GLenum identifier, GLuint name, GLsizei bufSize, GLsizei* length, GLchar *label)" \
  "void glGetObjectPtrLabel (void* ptr, GLsizei bufSize, GLsizei* length, GLchar *label)" \
  "void glObjectLabel (GLenum identifier, GLuint name, GLsizei length, const GLchar* label)" \
  "void glObjectPtrLabel (void* ptr, GLsizei length, const GLchar* label)" \
  "void glPopDebugGroup (void)" \
  "void glPushDebugGroup (GLenum source, GLuint id, GLsizei length, const GLchar * message)" \
  "void glMaxShaderCompilerThreadsKHR (GLuint count)" \
  "void glGetnUniformfv (GLuint program, GLint location, GLsizei bufSize, GLfloat* params)" \
  "void glGetnUniformiv (GLuint program, GLint location, GLsizei bufSize, GLint* params)" \
  "void glGetnUniformuiv (GLuint program, GLint location, GLsizei bufSize, GLuint* params)" \
  "void glReadnPixels (GLint x, GLint y, GLsizei width, GLsizei height, GLenum format, GLenum type, GLsizei bufSize, void *data)" \
  "GLuint glBufferRegionEnabled (void)" \
  "void glDeleteBufferRegion (GLenum region)" \
  "void glDrawBufferRegion (GLuint region, GLint x, GLint y, GLsizei width, GLsizei height, GLint xDest, GLint yDest)" \
  "GLuint glNewBufferRegion (GLenum region)" \
  "void glReadBufferRegion (GLuint region, GLint x, GLint y, GLsizei width, GLsizei height)" \
  "void glFramebufferParameteriMESA (GLenum target, GLenum pname, GLint param)" \
  "void glGetFramebufferParameterivMESA (GLenum target, GLenum pname, GLint* params)" \
  "void glResizeBuffersMESA (void)" \
  "void glWindowPos2dMESA (GLdouble x, GLdouble y)" \
  "void glWindowPos2dvMESA (const GLdouble* p)" \
  "void glWindowPos2fMESA (GLfloat x, GLfloat y)" \
  "void glWindowPos2fvMESA (const GLfloat* p)" \
  "void glWindowPos2iMESA (GLint x, GLint y)" \
  "void glWindowPos2ivMESA (const GLint* p)" \
  "void glWindowPos2sMESA (GLshort x, GLshort y)" \
  "void glWindowPos2svMESA (const GLshort* p)" \
  "void glWindowPos3dMESA (GLdouble x, GLdouble y, GLdouble z)" \
  "void glWindowPos3dvMESA (const GLdouble* p)" \
  "void glWindowPos3fMESA (GLfloat x, GLfloat y, GLfloat z)" \
  "void glWindowPos3fvMESA (const GLfloat* p)" \
  "void glWindowPos3iMESA (GLint x, GLint y, GLint z)" \
  "void glWindowPos3ivMESA (const GLint* p)" \
  "void glWindowPos3sMESA (GLshort x, GLshort y, GLshort z)" \
  "void glWindowPos3svMESA (const GLshort* p)" \
  "void glWindowPos4dMESA (GLdouble x, GLdouble y, GLdouble z, GLdouble)" \
  "void glWindowPos4dvMESA (const GLdouble* p)" \
  "void glWindowPos4fMESA (GLfloat x, GLfloat y, GLfloat z, GLfloat w)" \
  "void glWindowPos4fvMESA (const GLfloat* p)" \
  "void glWindowPos4iMESA (GLint x, GLint y, GLint z, GLint w)" \
  "void glWindowPos4ivMESA (const GLint* p)" \
  "void glWindowPos4sMESA (GLshort x, GLshort y, GLshort z, GLshort w)" \
  "void glWindowPos4svMESA (const GLshort* p)" \
  "void glBeginConditionalRenderNVX (GLuint id)" \
  "void glEndConditionalRenderNVX (void)" \
  "GLsync glAsyncCopyBufferSubDataNVX (GLsizei waitSemaphoreCount, const GLuint* waitSemaphoreArray, const GLuint64 *fenceValueArray, GLuint readGpu, GLbitfield writeGpuMask, GLuint readBuffer, GLuint writeBuffer, GLintptr readOffset, GLintptr writeOffset, GLsizeiptr size, GLsizei signalSemaphoreCount, const GLuint *signalSemaphoreArray, const GLuint64 *signalValueArray)" \
  "GLuint glAsyncCopyImageSubDataNVX (GLsizei waitSemaphoreCount, const GLuint* waitSemaphoreArray, const GLuint64 *waitValueArray, GLuint srcGpu, GLbitfield dstGpuMask, GLuint srcName, GLenum srcTarget, GLint srcLevel, GLint srcX, GLint srcY, GLint srcZ, GLuint dstName, GLenum dstTarget, GLint dstLevel, GLint dstX, GLint dstY, GLint dstZ, GLsizei srcWidth, GLsizei srcHeight, GLsizei srcDepth, GLsizei signalSemaphoreCount, const GLuint *signalSemaphoreArray, const GLuint64 *signalValueArray)" \
  "void glMulticastScissorArrayvNVX (GLuint gpu, GLuint first, GLsizei count, const GLint* v)" \
  "void glMulticastViewportArrayvNVX (GLuint gpu, GLuint first, GLsizei count, const GLfloat* v)" \
  "void glMulticastViewportPositionWScaleNVX (GLuint gpu, GLuint index, GLfloat xcoeff, GLfloat ycoeff)" \
  "void glUploadGpuMaskNVX (GLbitfield mask)" \
  "void glLGPUCopyImageSubDataNVX (GLuint sourceGpu, GLbitfield destinationGpuMask, GLuint srcName, GLenum srcTarget, GLint srcLevel, GLint srcX, GLint srxY, GLint srcZ, GLuint dstName, GLenum dstTarget, GLint dstLevel, GLint dstX, GLint dstY, GLint dstZ, GLsizei width, GLsizei height, GLsizei depth)" \
  "void glLGPUInterlockNVX (void)" \
  "void glLGPUNamedBufferSubDataNVX (GLbitfield gpuMask, GLuint buffer, GLintptr offset, GLsizeiptr size, const void *data)" \
  "void glClientWaitSemaphoreui64NVX (GLsizei fenceObjectCount, const GLuint* semaphoreArray, const GLuint64 *fenceValueArray)" \
  "void glSignalSemaphoreui64NVX (GLuint signalGpu, GLsizei fenceObjectCount, const GLuint* semaphoreArray, const GLuint64 *fenceValueArray)" \
  "void glWaitSemaphoreui64NVX (GLuint waitGpu, GLsizei fenceObjectCount, const GLuint* semaphoreArray, const GLuint64 *fenceValueArray)" \
  "void glStereoParameterfNV (GLenum pname, GLfloat param)" \
  "void glStereoParameteriNV (GLenum pname, GLint param)" \
  "void glAlphaToCoverageDitherControlNV (GLenum mode)" \
  "void glMultiDrawArraysIndirectBindlessNV (GLenum mode, const void *indirect, GLsizei drawCount, GLsizei stride, GLint vertexBufferCount)" \
  "void glMultiDrawElementsIndirectBindlessNV (GLenum mode, GLenum type, const void *indirect, GLsizei drawCount, GLsizei stride, GLint vertexBufferCount)" \
  "void glMultiDrawArraysIndirectBindlessCountNV (GLenum mode, const void *indirect, GLintptr drawCount, GLsizei maxDrawCount, GLsizei stride, GLint vertexBufferCount)" \
  "void glMultiDrawElementsIndirectBindlessCountNV (GLenum mode, GLenum type, const void *indirect, GLintptr drawCount, GLsizei maxDrawCount, GLsizei stride, GLint vertexBufferCount)" \
  "GLuint64 glGetImageHandleNV (GLuint texture, GLint level, GLboolean layered, GLint layer, GLenum format)" \
  "GLuint64 glGetTextureHandleNV (GLuint texture)" \
  "GLuint64 glGetTextureSamplerHandleNV (GLuint texture, GLuint sampler)" \
  "GLboolean glIsImageHandleResidentNV (GLuint64 handle)" \
  "GLboolean glIsTextureHandleResidentNV (GLuint64 handle)" \
  "void glMakeImageHandleNonResidentNV (GLuint64 handle)" \
  "void glMakeImageHandleResidentNV (GLuint64 handle, GLenum access)" \
  "void glMakeTextureHandleNonResidentNV (GLuint64 handle)" \
  "void glMakeTextureHandleResidentNV (GLuint64 handle)" \
  "void glProgramUniformHandleui64NV (GLuint program, GLint location, GLuint64 value)" \
  "void glProgramUniformHandleui64vNV (GLuint program, GLint location, GLsizei count, const GLuint64* values)" \
  "void glUniformHandleui64NV (GLint location, GLuint64 value)" \
  "void glUniformHandleui64vNV (GLint location, GLsizei count, const GLuint64* value)" \
  "void glBlendBarrierNV (void)" \
  "void glBlendParameteriNV (GLenum pname, GLint value)" \
  "void glViewportPositionWScaleNV (GLuint index, GLfloat xcoeff, GLfloat ycoeff)" \
  "void glCallCommandListNV (GLuint list)" \
  "void glCommandListSegmentsNV (GLuint list, GLuint segments)" \
  "void glCompileCommandListNV (GLuint list)" \
  "void glCreateCommandListsNV (GLsizei n, GLuint* lists)" \
  "void glCreateStatesNV (GLsizei n, GLuint* states)" \
  "void glDeleteCommandListsNV (GLsizei n, const GLuint* lists)" \
  "void glDeleteStatesNV (GLsizei n, const GLuint* states)" \
  "void glDrawCommandsAddressNV (GLenum primitiveMode, const GLuint64* indirects, const GLsizei* sizes, GLuint count)" \
  "void glDrawCommandsNV (GLenum primitiveMode, GLuint buffer, const GLintptr* indirects, const GLsizei* sizes, GLuint count)" \
  "void glDrawCommandsStatesAddressNV (const GLuint64* indirects, const GLsizei* sizes, const GLuint* states, const GLuint* fbos, GLuint count)" \
  "void glDrawCommandsStatesNV (GLuint buffer, const GLintptr* indirects, const GLsizei* sizes, const GLuint* states, const GLuint* fbos, GLuint count)" \
  "GLuint glGetCommandHeaderNV (GLenum tokenID, GLuint size)" \
  "GLushort glGetStageIndexNV (GLenum shadertype)" \
  "GLboolean glIsCommandListNV (GLuint list)" \
  "GLboolean glIsStateNV (GLuint state)" \
  "void glListDrawCommandsStatesClientNV (GLuint list, GLuint segment, const void** indirects, const GLsizei* sizes, const GLuint* states, const GLuint* fbos, GLuint count)" \
  "void glStateCaptureNV (GLuint state, GLenum mode)" \
  "void glBeginConditionalRenderNV (GLuint id, GLenum mode)" \
  "void glEndConditionalRenderNV (void)" \
  "void glSubpixelPrecisionBiasNV (GLuint xbits, GLuint ybits)" \
  "void glConservativeRasterParameterfNV (GLenum pname, GLfloat value)" \
  "void glConservativeRasterParameteriNV (GLenum pname, GLint param)" \
  "void glCopyBufferSubDataNV (GLenum readtarget, GLenum writetarget, GLintptr readoffset, GLintptr writeoffset, GLsizeiptr size)" \
  "void glCopyImageSubDataNV (GLuint srcName, GLenum srcTarget, GLint srcLevel, GLint srcX, GLint srcY, GLint srcZ, GLuint dstName, GLenum dstTarget, GLint dstLevel, GLint dstX, GLint dstY, GLint dstZ, GLsizei width, GLsizei height, GLsizei depth)" \
  "void glClearDepthdNV (GLdouble depth)" \
  "void glDepthBoundsdNV (GLdouble zmin, GLdouble zmax)" \
  "void glDepthRangedNV (GLdouble zNear, GLdouble zFar)" \
  "void glDrawBuffersNV (GLsizei n, const GLenum* bufs)" \
  "void glDrawArraysInstancedNV (GLenum mode, GLint first, GLsizei count, GLsizei primcount)" \
  "void glDrawElementsInstancedNV (GLenum mode, GLsizei count, GLenum type, const void *indices, GLsizei primcount)" \
  "void glDrawTextureNV (GLuint texture, GLuint sampler, GLfloat x0, GLfloat y0, GLfloat x1, GLfloat y1, GLfloat z, GLfloat s0, GLfloat t0, GLfloat s1, GLfloat t1)" \
  "void glDrawVkImageNV (GLuint64 vkImage, GLuint sampler, GLfloat x0, GLfloat y0, GLfloat x1, GLfloat y1, GLfloat z, GLfloat s0, GLfloat t0, GLfloat s1, GLfloat t1)" \
  "void glSignalVkFenceNV (GLuint64 vkFence)" \
  "void glSignalVkSemaphoreNV (GLuint64 vkSemaphore)" \
  "void glWaitVkSemaphoreNV (GLuint64 vkSemaphore)" \
  "void glEvalMapsNV (GLenum target, GLenum mode)" \
  "void glGetMapAttribParameterfvNV (GLenum target, GLuint index, GLenum pname, GLfloat* params)" \
  "void glGetMapAttribParameterivNV (GLenum target, GLuint index, GLenum pname, GLint* params)" \
  "void glGetMapControlPointsNV (GLenum target, GLuint index, GLenum type, GLsizei ustride, GLsizei vstride, GLboolean packed, void *points)" \
  "void glGetMapParameterfvNV (GLenum target, GLenum pname, GLfloat* params)" \
  "void glGetMapParameterivNV (GLenum target, GLenum pname, GLint* params)" \
  "void glMapControlPointsNV (GLenum target, GLuint index, GLenum type, GLsizei ustride, GLsizei vstride, GLint uorder, GLint vorder, GLboolean packed, const void *points)" \
  "void glMapParameterfvNV (GLenum target, GLenum pname, const GLfloat* params)" \
  "void glMapParameterivNV (GLenum target, GLenum pname, const GLint* params)" \
  "void glGetMultisamplefvNV (GLenum pname, GLuint index, GLfloat* val)" \
  "void glSampleMaskIndexedNV (GLuint index, GLbitfield mask)" \
  "void glTexRenderbufferNV (GLenum target, GLuint renderbuffer)" \
  "void glDeleteFencesNV (GLsizei n, const GLuint* fences)" \
  "void glFinishFenceNV (GLuint fence)" \
  "void glGenFencesNV (GLsizei n, GLuint* fences)" \
  "void glGetFenceivNV (GLuint fence, GLenum pname, GLint* params)" \
  "GLboolean glIsFenceNV (GLuint fence)" \
  "void glSetFenceNV (GLuint fence, GLenum condition)" \
  "GLboolean glTestFenceNV (GLuint fence)" \
  "void glFragmentCoverageColorNV (GLuint color)" \
  "void glGetProgramNamedParameterdvNV (GLuint id, GLsizei len, const GLubyte* name, GLdouble *params)" \
  "void glGetProgramNamedParameterfvNV (GLuint id, GLsizei len, const GLubyte* name, GLfloat *params)" \
  "void glProgramNamedParameter4dNV (GLuint id, GLsizei len, const GLubyte* name, GLdouble x, GLdouble y, GLdouble z, GLdouble w)" \
  "void glProgramNamedParameter4dvNV (GLuint id, GLsizei len, const GLubyte* name, const GLdouble *v)" \
  "void glProgramNamedParameter4fNV (GLuint id, GLsizei len, const GLubyte* name, GLfloat x, GLfloat y, GLfloat z, GLfloat w)" \
  "void glProgramNamedParameter4fvNV (GLuint id, GLsizei len, const GLubyte* name, const GLfloat *v)" \
  "void glBlitFramebufferNV (GLint srcX0, GLint srcY0, GLint srcX1, GLint srcY1, GLint dstX0, GLint dstY0, GLint dstX1, GLint dstY1, GLbitfield mask, GLenum filter)" \
  "void glRenderbufferStorageMultisampleNV (GLenum target, GLsizei samples, GLenum internalformat, GLsizei width, GLsizei height)" \
  "void glRenderbufferStorageMultisampleCoverageNV (GLenum target, GLsizei coverageSamples, GLsizei colorSamples, GLenum internalformat, GLsizei width, GLsizei height)" \
  "void glProgramVertexLimitNV (GLenum target, GLint limit)" \
  "void glMulticastBarrierNV (void)" \
  "void glMulticastBlitFramebufferNV (GLuint srcGpu, GLuint dstGpu, GLint srcX0, GLint srcY0, GLint srcX1, GLint srcY1, GLint dstX0, GLint dstY0, GLint dstX1, GLint dstY1, GLbitfield mask, GLenum filter)" \
  "void glMulticastBufferSubDataNV (GLbitfield gpuMask, GLuint buffer, GLintptr offset, GLsizeiptr size, const void *data)" \
  "void glMulticastCopyBufferSubDataNV (GLuint readGpu, GLbitfield writeGpuMask, GLuint readBuffer, GLuint writeBuffer, GLintptr readOffset, GLintptr writeOffset, GLsizeiptr size)" \
  "void glMulticastCopyImageSubDataNV (GLuint srcGpu, GLbitfield dstGpuMask, GLuint srcName, GLenum srcTarget, GLint srcLevel, GLint srcX, GLint srcY, GLint srcZ, GLuint dstName, GLenum dstTarget, GLint dstLevel, GLint dstX, GLint dstY, GLint dstZ, GLsizei srcWidth, GLsizei srcHeight, GLsizei srcDepth)" \
  "void glMulticastFramebufferSampleLocationsfvNV (GLuint gpu, GLuint framebuffer, GLuint start, GLsizei count, const GLfloat* v)" \
  "void glMulticastGetQueryObjecti64vNV (GLuint gpu, GLuint id, GLenum pname, GLint64* params)" \
  "void glMulticastGetQueryObjectivNV (GLuint gpu, GLuint id, GLenum pname, GLint* params)" \
  "void glMulticastGetQueryObjectui64vNV (GLuint gpu, GLuint id, GLenum pname, GLuint64* params)" \
  "void glMulticastGetQueryObjectuivNV (GLuint gpu, GLuint id, GLenum pname, GLuint* params)" \
  "void glMulticastWaitSyncNV (GLuint signalGpu, GLbitfield waitGpuMask)" \
  "void glRenderGpuMaskNV (GLbitfield mask)" \
  "void glProgramEnvParameterI4iNV (GLenum target, GLuint index, GLint x, GLint y, GLint z, GLint w)" \
  "void glProgramEnvParameterI4ivNV (GLenum target, GLuint index, const GLint *params)" \
  "void glProgramEnvParameterI4uiNV (GLenum target, GLuint index, GLuint x, GLuint y, GLuint z, GLuint w)" \
  "void glProgramEnvParameterI4uivNV (GLenum target, GLuint index, const GLuint *params)" \
  "void glProgramEnvParametersI4ivNV (GLenum target, GLuint index, GLsizei count, const GLint *params)" \
  "void glProgramEnvParametersI4uivNV (GLenum target, GLuint index, GLsizei count, const GLuint *params)" \
  "void glProgramLocalParameterI4iNV (GLenum target, GLuint index, GLint x, GLint y, GLint z, GLint w)" \
  "void glProgramLocalParameterI4ivNV (GLenum target, GLuint index, const GLint *params)" \
  "void glProgramLocalParameterI4uiNV (GLenum target, GLuint index, GLuint x, GLuint y, GLuint z, GLuint w)" \
  "void glProgramLocalParameterI4uivNV (GLenum target, GLuint index, const GLuint *params)" \
  "void glProgramLocalParametersI4ivNV (GLenum target, GLuint index, GLsizei count, const GLint *params)" \
  "void glProgramLocalParametersI4uivNV (GLenum target, GLuint index, GLsizei count, const GLuint *params)" \
  "void glGetUniformi64vNV (GLuint program, GLint location, GLint64EXT* params)" \
  "void glGetUniformui64vNV (GLuint program, GLint location, GLuint64EXT* params)" \
  "void glProgramUniform1i64NV (GLuint program, GLint location, GLint64EXT x)" \
  "void glProgramUniform1i64vNV (GLuint program, GLint location, GLsizei count, const GLint64EXT* value)" \
  "void glProgramUniform1ui64NV (GLuint program, GLint location, GLuint64EXT x)" \
  "void glProgramUniform1ui64vNV (GLuint program, GLint location, GLsizei count, const GLuint64EXT* value)" \
  "void glProgramUniform2i64NV (GLuint program, GLint location, GLint64EXT x, GLint64EXT y)" \
  "void glProgramUniform2i64vNV (GLuint program, GLint location, GLsizei count, const GLint64EXT* value)" \
  "void glProgramUniform2ui64NV (GLuint program, GLint location, GLuint64EXT x, GLuint64EXT y)" \
  "void glProgramUniform2ui64vNV (GLuint program, GLint location, GLsizei count, const GLuint64EXT* value)" \
  "void glProgramUniform3i64NV (GLuint program, GLint location, GLint64EXT x, GLint64EXT y, GLint64EXT z)" \
  "void glProgramUniform3i64vNV (GLuint program, GLint location, GLsizei count, const GLint64EXT* value)" \
  "void glProgramUniform3ui64NV (GLuint program, GLint location, GLuint64EXT x, GLuint64EXT y, GLuint64EXT z)" \
  "void glProgramUniform3ui64vNV (GLuint program, GLint location, GLsizei count, const GLuint64EXT* value)" \
  "void glProgramUniform4i64NV (GLuint program, GLint location, GLint64EXT x, GLint64EXT y, GLint64EXT z, GLint64EXT w)" \
  "void glProgramUniform4i64vNV (GLuint program, GLint location, GLsizei count, const GLint64EXT* value)" \
  "void glProgramUniform4ui64NV (GLuint program, GLint location, GLuint64EXT x, GLuint64EXT y, GLuint64EXT z, GLuint64EXT w)" \
  "void glProgramUniform4ui64vNV (GLuint program, GLint location, GLsizei count, const GLuint64EXT* value)" \
  "void glUniform1i64NV (GLint location, GLint64EXT x)" \
  "void glUniform1i64vNV (GLint location, GLsizei count, const GLint64EXT* value)" \
  "void glUniform1ui64NV (GLint location, GLuint64EXT x)" \
  "void glUniform1ui64vNV (GLint location, GLsizei count, const GLuint64EXT* value)" \
  "void glUniform2i64NV (GLint location, GLint64EXT x, GLint64EXT y)" \
  "void glUniform2i64vNV (GLint location, GLsizei count, const GLint64EXT* value)" \
  "void glUniform2ui64NV (GLint location, GLuint64EXT x, GLuint64EXT y)" \
  "void glUniform2ui64vNV (GLint location, GLsizei count, const GLuint64EXT* value)" \
  "void glUniform3i64NV (GLint location, GLint64EXT x, GLint64EXT y, GLint64EXT z)" \
  "void glUniform3i64vNV (GLint location, GLsizei count, const GLint64EXT* value)" \
  "void glUniform3ui64NV (GLint location, GLuint64EXT x, GLuint64EXT y, GLuint64EXT z)" \
  "void glUniform3ui64vNV (GLint location, GLsizei count, const GLuint64EXT* value)" \
  "void glUniform4i64NV (GLint location, GLint64EXT x, GLint64EXT y, GLint64EXT z, GLint64EXT w)" \
  "void glUniform4i64vNV (GLint location, GLsizei count, const GLint64EXT* value)" \
  "void glUniform4ui64NV (GLint location, GLuint64EXT x, GLuint64EXT y, GLuint64EXT z, GLuint64EXT w)" \
  "void glUniform4ui64vNV (GLint location, GLsizei count, const GLuint64EXT* value)" \
  "void glColor3hNV (GLhalf red, GLhalf green, GLhalf blue)" \
  "void glColor3hvNV (const GLhalf* v)" \
  "void glColor4hNV (GLhalf red, GLhalf green, GLhalf blue, GLhalf alpha)" \
  "void glColor4hvNV (const GLhalf* v)" \
  "void glFogCoordhNV (GLhalf fog)" \
  "void glFogCoordhvNV (const GLhalf* fog)" \
  "void glMultiTexCoord1hNV (GLenum target, GLhalf s)" \
  "void glMultiTexCoord1hvNV (GLenum target, const GLhalf* v)" \
  "void glMultiTexCoord2hNV (GLenum target, GLhalf s, GLhalf t)" \
  "void glMultiTexCoord2hvNV (GLenum target, const GLhalf* v)" \
  "void glMultiTexCoord3hNV (GLenum target, GLhalf s, GLhalf t, GLhalf r)" \
  "void glMultiTexCoord3hvNV (GLenum target, const GLhalf* v)" \
  "void glMultiTexCoord4hNV (GLenum target, GLhalf s, GLhalf t, GLhalf r, GLhalf q)" \
  "void glMultiTexCoord4hvNV (GLenum target, const GLhalf* v)" \
  "void glNormal3hNV (GLhalf nx, GLhalf ny, GLhalf nz)" \
  "void glNormal3hvNV (const GLhalf* v)" \
  "void glSecondaryColor3hNV (GLhalf red, GLhalf green, GLhalf blue)" \
  "void glSecondaryColor3hvNV (const GLhalf* v)" \
  "void glTexCoord1hNV (GLhalf s)" \
  "void glTexCoord1hvNV (const GLhalf* v)" \
  "void glTexCoord2hNV (GLhalf s, GLhalf t)" \
  "void glTexCoord2hvNV (const GLhalf* v)" \
  "void glTexCoord3hNV (GLhalf s, GLhalf t, GLhalf r)" \
  "void glTexCoord3hvNV (const GLhalf* v)" \
  "void glTexCoord4hNV (GLhalf s, GLhalf t, GLhalf r, GLhalf q)" \
  "void glTexCoord4hvNV (const GLhalf* v)" \
  "void glVertex2hNV (GLhalf x, GLhalf y)" \
  "void glVertex2hvNV (const GLhalf* v)" \
  "void glVertex3hNV (GLhalf x, GLhalf y, GLhalf z)" \
  "void glVertex3hvNV (const GLhalf* v)" \
  "void glVertex4hNV (GLhalf x, GLhalf y, GLhalf z, GLhalf w)" \
  "void glVertex4hvNV (const GLhalf* v)" \
  "void glVertexAttrib1hNV (GLuint index, GLhalf x)" \
  "void glVertexAttrib1hvNV (GLuint index, const GLhalf* v)" \
  "void glVertexAttrib2hNV (GLuint index, GLhalf x, GLhalf y)" \
  "void glVertexAttrib2hvNV (GLuint index, const GLhalf* v)" \
  "void glVertexAttrib3hNV (GLuint index, GLhalf x, GLhalf y, GLhalf z)" \
  "void glVertexAttrib3hvNV (GLuint index, const GLhalf* v)" \
  "void glVertexAttrib4hNV (GLuint index, GLhalf x, GLhalf y, GLhalf z, GLhalf w)" \
  "void glVertexAttrib4hvNV (GLuint index, const GLhalf* v)" \
  "void glVertexAttribs1hvNV (GLuint index, GLsizei n, const GLhalf* v)" \
  "void glVertexAttribs2hvNV (GLuint index, GLsizei n, const GLhalf* v)" \
  "void glVertexAttribs3hvNV (GLuint index, GLsizei n, const GLhalf* v)" \
  "void glVertexAttribs4hvNV (GLuint index, GLsizei n, const GLhalf* v)" \
  "void glVertexWeighthNV (GLhalf weight)" \
  "void glVertexWeighthvNV (const GLhalf* weight)" \
  "void glVertexAttribDivisorNV (GLuint index, GLuint divisor)" \
  "void glGetInternalformatSampleivNV (GLenum target, GLenum internalformat, GLsizei samples, GLenum pname, GLsizei bufSize, GLint* params)" \
  "void glBufferAttachMemoryNV (GLenum target, GLuint memory, GLuint64 offset)" \
  "void glGetMemoryObjectDetachedResourcesuivNV (GLuint memory, GLenum pname, GLint first, GLsizei count, GLuint* params)" \
  "void glNamedBufferAttachMemoryNV (GLuint buffer, GLuint memory, GLuint64 offset)" \
  "void glResetMemoryObjectParameterNV (GLuint memory, GLenum pname)" \
  "void glTexAttachMemoryNV (GLenum target, GLuint memory, GLuint64 offset)" \
  "void glTextureAttachMemoryNV (GLuint texture, GLuint memory, GLuint64 offset)" \
  "void glDrawMeshTasksIndirectNV (GLintptr indirect)" \
  "void glDrawMeshTasksNV (GLuint first, GLuint count)" \
  "void glMultiDrawMeshTasksIndirectCountNV (GLintptr indirect, GLintptr drawcount, GLsizei maxdrawcount, GLsizei stride)" \
  "void glMultiDrawMeshTasksIndirectNV (GLintptr indirect, GLsizei drawcount, GLsizei stride)" \
  "void glUniformMatrix2x3fvNV (GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glUniformMatrix2x4fvNV (GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glUniformMatrix3x2fvNV (GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glUniformMatrix3x4fvNV (GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glUniformMatrix4x2fvNV (GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glUniformMatrix4x3fvNV (GLint location, GLsizei count, GLboolean transpose, const GLfloat* value)" \
  "void glBeginOcclusionQueryNV (GLuint id)" \
  "void glDeleteOcclusionQueriesNV (GLsizei n, const GLuint* ids)" \
  "void glEndOcclusionQueryNV (void)" \
  "void glGenOcclusionQueriesNV (GLsizei n, GLuint* ids)" \
  "void glGetOcclusionQueryivNV (GLuint id, GLenum pname, GLint* params)" \
  "void glGetOcclusionQueryuivNV (GLuint id, GLenum pname, GLuint* params)" \
  "GLboolean glIsOcclusionQueryNV (GLuint id)" \
  "void glProgramBufferParametersIivNV (GLenum target, GLuint buffer, GLuint index, GLsizei count, const GLint *params)" \
  "void glProgramBufferParametersIuivNV (GLenum target, GLuint buffer, GLuint index, GLsizei count, const GLuint *params)" \
  "void glProgramBufferParametersfvNV (GLenum target, GLuint buffer, GLuint index, GLsizei count, const GLfloat *params)" \
  "void glCopyPathNV (GLuint resultPath, GLuint srcPath)" \
  "void glCoverFillPathInstancedNV (GLsizei numPaths, GLenum pathNameType, const void *paths, GLuint pathBase, GLenum coverMode, GLenum transformType, const GLfloat *transformValues)" \
  "void glCoverFillPathNV (GLuint path, GLenum coverMode)" \
  "void glCoverStrokePathInstancedNV (GLsizei numPaths, GLenum pathNameType, const void *paths, GLuint pathBase, GLenum coverMode, GLenum transformType, const GLfloat *transformValues)" \
  "void glCoverStrokePathNV (GLuint path, GLenum coverMode)" \
  "void glDeletePathsNV (GLuint path, GLsizei range)" \
  "GLuint glGenPathsNV (GLsizei range)" \
  "void glGetPathColorGenfvNV (GLenum color, GLenum pname, GLfloat* value)" \
  "void glGetPathColorGenivNV (GLenum color, GLenum pname, GLint* value)" \
  "void glGetPathCommandsNV (GLuint path, GLubyte* commands)" \
  "void glGetPathCoordsNV (GLuint path, GLfloat* coords)" \
  "void glGetPathDashArrayNV (GLuint path, GLfloat* dashArray)" \
  "GLfloat glGetPathLengthNV (GLuint path, GLsizei startSegment, GLsizei numSegments)" \
  "void glGetPathMetricRangeNV (GLbitfield metricQueryMask, GLuint firstPathName, GLsizei numPaths, GLsizei stride, GLfloat* metrics)" \
  "void glGetPathMetricsNV (GLbitfield metricQueryMask, GLsizei numPaths, GLenum pathNameType, const void *paths, GLuint pathBase, GLsizei stride, GLfloat *metrics)" \
  "void glGetPathParameterfvNV (GLuint path, GLenum pname, GLfloat* value)" \
  "void glGetPathParameterivNV (GLuint path, GLenum pname, GLint* value)" \
  "void glGetPathSpacingNV (GLenum pathListMode, GLsizei numPaths, GLenum pathNameType, const void *paths, GLuint pathBase, GLfloat advanceScale, GLfloat kerningScale, GLenum transformType, GLfloat *returnedSpacing)" \
  "void glGetPathTexGenfvNV (GLenum texCoordSet, GLenum pname, GLfloat* value)" \
  "void glGetPathTexGenivNV (GLenum texCoordSet, GLenum pname, GLint* value)" \
  "void glGetProgramResourcefvNV (GLuint program, GLenum programInterface, GLuint index, GLsizei propCount, const GLenum* props, GLsizei bufSize, GLsizei *length, GLfloat *params)" \
  "void glInterpolatePathsNV (GLuint resultPath, GLuint pathA, GLuint pathB, GLfloat weight)" \
  "GLboolean glIsPathNV (GLuint path)" \
  "GLboolean glIsPointInFillPathNV (GLuint path, GLuint mask, GLfloat x, GLfloat y)" \
  "GLboolean glIsPointInStrokePathNV (GLuint path, GLfloat x, GLfloat y)" \
  "void glMatrixLoad3x2fNV (GLenum matrixMode, const GLfloat* m)" \
  "void glMatrixLoad3x3fNV (GLenum matrixMode, const GLfloat* m)" \
  "void glMatrixLoadTranspose3x3fNV (GLenum matrixMode, const GLfloat* m)" \
  "void glMatrixMult3x2fNV (GLenum matrixMode, const GLfloat* m)" \
  "void glMatrixMult3x3fNV (GLenum matrixMode, const GLfloat* m)" \
  "void glMatrixMultTranspose3x3fNV (GLenum matrixMode, const GLfloat* m)" \
  "void glPathColorGenNV (GLenum color, GLenum genMode, GLenum colorFormat, const GLfloat* coeffs)" \
  "void glPathCommandsNV (GLuint path, GLsizei numCommands, const GLubyte* commands, GLsizei numCoords, GLenum coordType, const void*coords)" \
  "void glPathCoordsNV (GLuint path, GLsizei numCoords, GLenum coordType, const void *coords)" \
  "void glPathCoverDepthFuncNV (GLenum zfunc)" \
  "void glPathDashArrayNV (GLuint path, GLsizei dashCount, const GLfloat* dashArray)" \
  "void glPathFogGenNV (GLenum genMode)" \
  "GLenum glPathGlyphIndexArrayNV (GLuint firstPathName, GLenum fontTarget, const void *fontName, GLbitfield fontStyle, GLuint firstGlyphIndex, GLsizei numGlyphs, GLuint pathParameterTemplate, GLfloat emScale)" \
  "GLenum glPathGlyphIndexRangeNV (GLenum fontTarget, const void *fontName, GLbitfield fontStyle, GLuint pathParameterTemplate, GLfloat emScale, GLuint baseAndCount\[2\])" \
  "void glPathGlyphRangeNV (GLuint firstPathName, GLenum fontTarget, const void *fontName, GLbitfield fontStyle, GLuint firstGlyph, GLsizei numGlyphs, GLenum handleMissingGlyphs, GLuint pathParameterTemplate, GLfloat emScale)" \
  "void glPathGlyphsNV (GLuint firstPathName, GLenum fontTarget, const void *fontName, GLbitfield fontStyle, GLsizei numGlyphs, GLenum type, const void*charcodes, GLenum handleMissingGlyphs, GLuint pathParameterTemplate, GLfloat emScale)" \
  "GLenum glPathMemoryGlyphIndexArrayNV (GLuint firstPathName, GLenum fontTarget, GLsizeiptr fontSize, const void *fontData, GLsizei faceIndex, GLuint firstGlyphIndex, GLsizei numGlyphs, GLuint pathParameterTemplate, GLfloat emScale)" \
  "void glPathParameterfNV (GLuint path, GLenum pname, GLfloat value)" \
  "void glPathParameterfvNV (GLuint path, GLenum pname, const GLfloat* value)" \
  "void glPathParameteriNV (GLuint path, GLenum pname, GLint value)" \
  "void glPathParameterivNV (GLuint path, GLenum pname, const GLint* value)" \
  "void glPathStencilDepthOffsetNV (GLfloat factor, GLfloat units)" \
  "void glPathStencilFuncNV (GLenum func, GLint ref, GLuint mask)" \
  "void glPathStringNV (GLuint path, GLenum format, GLsizei length, const void *pathString)" \
  "void glPathSubCommandsNV (GLuint path, GLsizei commandStart, GLsizei commandsToDelete, GLsizei numCommands, const GLubyte* commands, GLsizei numCoords, GLenum coordType, const void*coords)" \
  "void glPathSubCoordsNV (GLuint path, GLsizei coordStart, GLsizei numCoords, GLenum coordType, const void *coords)" \
  "void glPathTexGenNV (GLenum texCoordSet, GLenum genMode, GLint components, const GLfloat* coeffs)" \
  "GLboolean glPointAlongPathNV (GLuint path, GLsizei startSegment, GLsizei numSegments, GLfloat distance, GLfloat* x, GLfloat *y, GLfloat *tangentX, GLfloat *tangentY)" \
  "void glProgramPathFragmentInputGenNV (GLuint program, GLint location, GLenum genMode, GLint components, const GLfloat* coeffs)" \
  "void glStencilFillPathInstancedNV (GLsizei numPaths, GLenum pathNameType, const void *paths, GLuint pathBase, GLenum fillMode, GLuint mask, GLenum transformType, const GLfloat *transformValues)" \
  "void glStencilFillPathNV (GLuint path, GLenum fillMode, GLuint mask)" \
  "void glStencilStrokePathInstancedNV (GLsizei numPaths, GLenum pathNameType, const void *paths, GLuint pathBase, GLint reference, GLuint mask, GLenum transformType, const GLfloat *transformValues)" \
  "void glStencilStrokePathNV (GLuint path, GLint reference, GLuint mask)" \
  "void glStencilThenCoverFillPathInstancedNV (GLsizei numPaths, GLenum pathNameType, const void *paths, GLuint pathBase, GLenum fillMode, GLuint mask, GLenum coverMode, GLenum transformType, const GLfloat *transformValues)" \
  "void glStencilThenCoverFillPathNV (GLuint path, GLenum fillMode, GLuint mask, GLenum coverMode)" \
  "void glStencilThenCoverStrokePathInstancedNV (GLsizei numPaths, GLenum pathNameType, const void *paths, GLuint pathBase, GLint reference, GLuint mask, GLenum coverMode, GLenum transformType, const GLfloat *transformValues)" \
  "void glStencilThenCoverStrokePathNV (GLuint path, GLint reference, GLuint mask, GLenum coverMode)" \
  "void glTransformPathNV (GLuint resultPath, GLuint srcPath, GLenum transformType, const GLfloat* transformValues)" \
  "void glWeightPathsNV (GLuint resultPath, GLsizei numPaths, const GLuint *paths, const GLfloat *weights)" \
  "void glFlushPixelDataRangeNV (GLenum target)" \
  "void glPixelDataRangeNV (GLenum target, GLsizei length, void *pointer)" \
  "void glPointParameteriNV (GLenum pname, GLint param)" \
  "void glPointParameterivNV (GLenum pname, const GLint* params)" \
  "void glPolygonModeNV (GLenum face, GLenum mode)" \
  "void glGetVideoi64vNV (GLuint video_slot, GLenum pname, GLint64EXT* params)" \
  "void glGetVideoivNV (GLuint video_slot, GLenum pname, GLint* params)" \
  "void glGetVideoui64vNV (GLuint video_slot, GLenum pname, GLuint64EXT* params)" \
  "void glGetVideouivNV (GLuint video_slot, GLenum pname, GLuint* params)" \
  "void glPresentFrameDualFillNV (GLuint video_slot, GLuint64EXT minPresentTime, GLuint beginPresentTimeId, GLuint presentDurationId, GLenum type, GLenum target0, GLuint fill0, GLenum target1, GLuint fill1, GLenum target2, GLuint fill2, GLenum target3, GLuint fill3)" \
  "void glPresentFrameKeyedNV (GLuint video_slot, GLuint64EXT minPresentTime, GLuint beginPresentTimeId, GLuint presentDurationId, GLenum type, GLenum target0, GLuint fill0, GLuint key0, GLenum target1, GLuint fill1, GLuint key1)" \
  "void glPrimitiveRestartIndexNV (GLuint index)" \
  "void glPrimitiveRestartNV (void)" \
  "void glReadBufferNV (GLenum mode)" \
  "void glCombinerInputNV (GLenum stage, GLenum portion, GLenum variable, GLenum input, GLenum mapping, GLenum componentUsage)" \
  "void glCombinerOutputNV (GLenum stage, GLenum portion, GLenum abOutput, GLenum cdOutput, GLenum sumOutput, GLenum scale, GLenum bias, GLboolean abDotProduct, GLboolean cdDotProduct, GLboolean muxSum)" \
  "void glCombinerParameterfNV (GLenum pname, GLfloat param)" \
  "void glCombinerParameterfvNV (GLenum pname, const GLfloat* params)" \
  "void glCombinerParameteriNV (GLenum pname, GLint param)" \
  "void glCombinerParameterivNV (GLenum pname, const GLint* params)" \
  "void glFinalCombinerInputNV (GLenum variable, GLenum input, GLenum mapping, GLenum componentUsage)" \
  "void glGetCombinerInputParameterfvNV (GLenum stage, GLenum portion, GLenum variable, GLenum pname, GLfloat* params)" \
  "void glGetCombinerInputParameterivNV (GLenum stage, GLenum portion, GLenum variable, GLenum pname, GLint* params)" \
  "void glGetCombinerOutputParameterfvNV (GLenum stage, GLenum portion, GLenum pname, GLfloat* params)" \
  "void glGetCombinerOutputParameterivNV (GLenum stage, GLenum portion, GLenum pname, GLint* params)" \
  "void glGetFinalCombinerInputParameterfvNV (GLenum variable, GLenum pname, GLfloat* params)" \
  "void glGetFinalCombinerInputParameterivNV (GLenum variable, GLenum pname, GLint* params)" \
  "void glCombinerStageParameterfvNV (GLenum stage, GLenum pname, const GLfloat* params)" \
  "void glGetCombinerStageParameterfvNV (GLenum stage, GLenum pname, GLfloat* params)" \
  "void glFramebufferSampleLocationsfvNV (GLenum target, GLuint start, GLsizei count, const GLfloat* v)" \
  "void glNamedFramebufferSampleLocationsfvNV (GLuint framebuffer, GLuint start, GLsizei count, const GLfloat* v)" \
  "void glResolveDepthValuesNV (void)" \
  "void glScissorExclusiveArrayvNV (GLuint first, GLsizei count, const GLint* v)" \
  "void glScissorExclusiveNV (GLint x, GLint y, GLsizei width, GLsizei height)" \
  "void glGetBufferParameterui64vNV (GLenum target, GLenum pname, GLuint64EXT* params)" \
  "void glGetIntegerui64vNV (GLenum value, GLuint64EXT* result)" \
  "void glGetNamedBufferParameterui64vNV (GLuint buffer, GLenum pname, GLuint64EXT* params)" \
  "GLboolean glIsBufferResidentNV (GLenum target)" \
  "GLboolean glIsNamedBufferResidentNV (GLuint buffer)" \
  "void glMakeBufferNonResidentNV (GLenum target)" \
  "void glMakeBufferResidentNV (GLenum target, GLenum access)" \
  "void glMakeNamedBufferNonResidentNV (GLuint buffer)" \
  "void glMakeNamedBufferResidentNV (GLuint buffer, GLenum access)" \
  "void glProgramUniformui64NV (GLuint program, GLint location, GLuint64EXT value)" \
  "void glProgramUniformui64vNV (GLuint program, GLint location, GLsizei count, const GLuint64EXT* value)" \
  "void glUniformui64NV (GLint location, GLuint64EXT value)" \
  "void glUniformui64vNV (GLint location, GLsizei count, const GLuint64EXT* value)" \
  "void glBindShadingRateImageNV (GLuint texture)" \
  "void glGetShadingRateImagePaletteNV (GLuint viewport, GLuint entry, GLenum* rate)" \
  "void glGetShadingRateSampleLocationivNV (GLenum rate, GLuint samples, GLuint index, GLint* location)" \
  "void glShadingRateImageBarrierNV (GLenum order)" \
  "void glShadingRateImagePaletteNV (GLuint viewport, GLuint first, GLsizei count, const GLenum* rates)" \
  "void glShadingRateSampleOrderCustomNV (GLenum rate, GLuint samples, const GLint* locations)" \
  "void glCompressedTexImage3DNV (GLenum target, GLint level, GLenum internalformat, GLsizei width, GLsizei height, GLsizei depth, GLint border, GLsizei imageSize, const void *data)" \
  "void glCompressedTexSubImage3DNV (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, GLenum format, GLsizei imageSize, const void *data)" \
  "void glCopyTexSubImage3DNV (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLint x, GLint y, GLsizei width, GLsizei height)" \
  "void glFramebufferTextureLayerNV (GLenum target, GLenum attachment, GLuint texture, GLint level, GLint layer)" \
  "void glTexImage3DNV (GLenum target, GLint level, GLenum internalFormat, GLsizei width, GLsizei height, GLsizei depth, GLint border, GLenum format, GLenum type, const void *pixels)" \
  "void glTexSubImage3DNV (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, GLenum format, GLenum type, const void *pixels)" \
  "void glTextureBarrierNV (void)" \
  "void glTexImage2DMultisampleCoverageNV (GLenum target, GLsizei coverageSamples, GLsizei colorSamples, GLint internalFormat, GLsizei width, GLsizei height, GLboolean fixedSampleLocations)" \
  "void glTexImage3DMultisampleCoverageNV (GLenum target, GLsizei coverageSamples, GLsizei colorSamples, GLint internalFormat, GLsizei width, GLsizei height, GLsizei depth, GLboolean fixedSampleLocations)" \
  "void glTextureImage2DMultisampleCoverageNV (GLuint texture, GLenum target, GLsizei coverageSamples, GLsizei colorSamples, GLint internalFormat, GLsizei width, GLsizei height, GLboolean fixedSampleLocations)" \
  "void glTextureImage2DMultisampleNV (GLuint texture, GLenum target, GLsizei samples, GLint internalFormat, GLsizei width, GLsizei height, GLboolean fixedSampleLocations)" \
  "void glTextureImage3DMultisampleCoverageNV (GLuint texture, GLenum target, GLsizei coverageSamples, GLsizei colorSamples, GLint internalFormat, GLsizei width, GLsizei height, GLsizei depth, GLboolean fixedSampleLocations)" \
  "void glTextureImage3DMultisampleNV (GLuint texture, GLenum target, GLsizei samples, GLint internalFormat, GLsizei width, GLsizei height, GLsizei depth, GLboolean fixedSampleLocations)" \
  "void glActiveVaryingNV (GLuint program, const GLchar *name)" \
  "void glBeginTransformFeedbackNV (GLenum primitiveMode)" \
  "void glBindBufferBaseNV (GLenum target, GLuint index, GLuint buffer)" \
  "void glBindBufferOffsetNV (GLenum target, GLuint index, GLuint buffer, GLintptr offset)" \
  "void glBindBufferRangeNV (GLenum target, GLuint index, GLuint buffer, GLintptr offset, GLsizeiptr size)" \
  "void glEndTransformFeedbackNV (void)" \
  "void glGetActiveVaryingNV (GLuint program, GLuint index, GLsizei bufSize, GLsizei *length, GLsizei *size, GLenum *type, GLchar *name)" \
  "void glGetTransformFeedbackVaryingNV (GLuint program, GLuint index, GLint *location)" \
  "GLint glGetVaryingLocationNV (GLuint program, const GLchar *name)" \
  "void glTransformFeedbackAttribsNV (GLuint count, const GLint *attribs, GLenum bufferMode)" \
  "void glTransformFeedbackVaryingsNV (GLuint program, GLsizei count, const GLint *locations, GLenum bufferMode)" \
  "void glBindTransformFeedbackNV (GLenum target, GLuint id)" \
  "void glDeleteTransformFeedbacksNV (GLsizei n, const GLuint* ids)" \
  "void glDrawTransformFeedbackNV (GLenum mode, GLuint id)" \
  "void glGenTransformFeedbacksNV (GLsizei n, GLuint* ids)" \
  "GLboolean glIsTransformFeedbackNV (GLuint id)" \
  "void glPauseTransformFeedbackNV (void)" \
  "void glResumeTransformFeedbackNV (void)" \
  "void glVDPAUFiniNV (void)" \
  "void glVDPAUGetSurfaceivNV (GLvdpauSurfaceNV surface, GLenum pname, GLsizei bufSize, GLsizei* length, GLint *values)" \
  "void glVDPAUInitNV (const void* vdpDevice, const void*getProcAddress)" \
  "void glVDPAUIsSurfaceNV (GLvdpauSurfaceNV surface)" \
  "void glVDPAUMapSurfacesNV (GLsizei numSurfaces, const GLvdpauSurfaceNV* surfaces)" \
  "GLvdpauSurfaceNV glVDPAURegisterOutputSurfaceNV (const void* vdpSurface, GLenum target, GLsizei numTextureNames, const GLuint *textureNames)" \
  "GLvdpauSurfaceNV glVDPAURegisterVideoSurfaceNV (const void* vdpSurface, GLenum target, GLsizei numTextureNames, const GLuint *textureNames)" \
  "void glVDPAUSurfaceAccessNV (GLvdpauSurfaceNV surface, GLenum access)" \
  "void glVDPAUUnmapSurfacesNV (GLsizei numSurface, const GLvdpauSurfaceNV* surfaces)" \
  "void glVDPAUUnregisterSurfaceNV (GLvdpauSurfaceNV surface)" \
  "GLvdpauSurfaceNV glVDPAURegisterVideoSurfaceWithPictureStructureNV (const void *vdpSurface, GLenum target, GLsizei numTextureNames, const GLuint *textureNames, GLboolean isFrameStructure)" \
  "void glFlushVertexArrayRangeNV (void)" \
  "void glVertexArrayRangeNV (GLsizei length, void *pointer)" \
  "void glGetVertexAttribLi64vNV (GLuint index, GLenum pname, GLint64EXT* params)" \
  "void glGetVertexAttribLui64vNV (GLuint index, GLenum pname, GLuint64EXT* params)" \
  "void glVertexAttribL1i64NV (GLuint index, GLint64EXT x)" \
  "void glVertexAttribL1i64vNV (GLuint index, const GLint64EXT* v)" \
  "void glVertexAttribL1ui64NV (GLuint index, GLuint64EXT x)" \
  "void glVertexAttribL1ui64vNV (GLuint index, const GLuint64EXT* v)" \
  "void glVertexAttribL2i64NV (GLuint index, GLint64EXT x, GLint64EXT y)" \
  "void glVertexAttribL2i64vNV (GLuint index, const GLint64EXT* v)" \
  "void glVertexAttribL2ui64NV (GLuint index, GLuint64EXT x, GLuint64EXT y)" \
  "void glVertexAttribL2ui64vNV (GLuint index, const GLuint64EXT* v)" \
  "void glVertexAttribL3i64NV (GLuint index, GLint64EXT x, GLint64EXT y, GLint64EXT z)" \
  "void glVertexAttribL3i64vNV (GLuint index, const GLint64EXT* v)" \
  "void glVertexAttribL3ui64NV (GLuint index, GLuint64EXT x, GLuint64EXT y, GLuint64EXT z)" \
  "void glVertexAttribL3ui64vNV (GLuint index, const GLuint64EXT* v)" \
  "void glVertexAttribL4i64NV (GLuint index, GLint64EXT x, GLint64EXT y, GLint64EXT z, GLint64EXT w)" \
  "void glVertexAttribL4i64vNV (GLuint index, const GLint64EXT* v)" \
  "void glVertexAttribL4ui64NV (GLuint index, GLuint64EXT x, GLuint64EXT y, GLuint64EXT z, GLuint64EXT w)" \
  "void glVertexAttribL4ui64vNV (GLuint index, const GLuint64EXT* v)" \
  "void glVertexAttribLFormatNV (GLuint index, GLint size, GLenum type, GLsizei stride)" \
  "void glBufferAddressRangeNV (GLenum pname, GLuint index, GLuint64EXT address, GLsizeiptr length)" \
  "void glColorFormatNV (GLint size, GLenum type, GLsizei stride)" \
  "void glEdgeFlagFormatNV (GLsizei stride)" \
  "void glFogCoordFormatNV (GLenum type, GLsizei stride)" \
  "void glGetIntegerui64i_vNV (GLenum value, GLuint index, GLuint64EXT *result)" \
  "void glIndexFormatNV (GLenum type, GLsizei stride)" \
  "void glNormalFormatNV (GLenum type, GLsizei stride)" \
  "void glSecondaryColorFormatNV (GLint size, GLenum type, GLsizei stride)" \
  "void glTexCoordFormatNV (GLint size, GLenum type, GLsizei stride)" \
  "void glVertexAttribFormatNV (GLuint index, GLint size, GLenum type, GLboolean normalized, GLsizei stride)" \
  "void glVertexAttribIFormatNV (GLuint index, GLint size, GLenum type, GLsizei stride)" \
  "void glVertexFormatNV (GLint size, GLenum type, GLsizei stride)" \
  "GLboolean glAreProgramsResidentNV (GLsizei n, const GLuint* ids, GLboolean *residences)" \
  "void glBindProgramNV (GLenum target, GLuint id)" \
  "void glDeleteProgramsNV (GLsizei n, const GLuint* ids)" \
  "void glExecuteProgramNV (GLenum target, GLuint id, const GLfloat* params)" \
  "void glGenProgramsNV (GLsizei n, GLuint* ids)" \
  "void glGetProgramParameterdvNV (GLenum target, GLuint index, GLenum pname, GLdouble* params)" \
  "void glGetProgramParameterfvNV (GLenum target, GLuint index, GLenum pname, GLfloat* params)" \
  "void glGetProgramStringNV (GLuint id, GLenum pname, GLubyte* program)" \
  "void glGetProgramivNV (GLuint id, GLenum pname, GLint* params)" \
  "void glGetTrackMatrixivNV (GLenum target, GLuint address, GLenum pname, GLint* params)" \
  "void glGetVertexAttribPointervNV (GLuint index, GLenum pname, void** pointer)" \
  "void glGetVertexAttribdvNV (GLuint index, GLenum pname, GLdouble* params)" \
  "void glGetVertexAttribfvNV (GLuint index, GLenum pname, GLfloat* params)" \
  "void glGetVertexAttribivNV (GLuint index, GLenum pname, GLint* params)" \
  "GLboolean glIsProgramNV (GLuint id)" \
  "void glLoadProgramNV (GLenum target, GLuint id, GLsizei len, const GLubyte* program)" \
  "void glProgramParameter4dNV (GLenum target, GLuint index, GLdouble x, GLdouble y, GLdouble z, GLdouble w)" \
  "void glProgramParameter4dvNV (GLenum target, GLuint index, const GLdouble* params)" \
  "void glProgramParameter4fNV (GLenum target, GLuint index, GLfloat x, GLfloat y, GLfloat z, GLfloat w)" \
  "void glProgramParameter4fvNV (GLenum target, GLuint index, const GLfloat* params)" \
  "void glProgramParameters4dvNV (GLenum target, GLuint index, GLsizei num, const GLdouble* params)" \
  "void glProgramParameters4fvNV (GLenum target, GLuint index, GLsizei num, const GLfloat* params)" \
  "void glRequestResidentProgramsNV (GLsizei n, GLuint* ids)" \
  "void glTrackMatrixNV (GLenum target, GLuint address, GLenum matrix, GLenum transform)" \
  "void glVertexAttrib1dNV (GLuint index, GLdouble x)" \
  "void glVertexAttrib1dvNV (GLuint index, const GLdouble* v)" \
  "void glVertexAttrib1fNV (GLuint index, GLfloat x)" \
  "void glVertexAttrib1fvNV (GLuint index, const GLfloat* v)" \
  "void glVertexAttrib1sNV (GLuint index, GLshort x)" \
  "void glVertexAttrib1svNV (GLuint index, const GLshort* v)" \
  "void glVertexAttrib2dNV (GLuint index, GLdouble x, GLdouble y)" \
  "void glVertexAttrib2dvNV (GLuint index, const GLdouble* v)" \
  "void glVertexAttrib2fNV (GLuint index, GLfloat x, GLfloat y)" \
  "void glVertexAttrib2fvNV (GLuint index, const GLfloat* v)" \
  "void glVertexAttrib2sNV (GLuint index, GLshort x, GLshort y)" \
  "void glVertexAttrib2svNV (GLuint index, const GLshort* v)" \
  "void glVertexAttrib3dNV (GLuint index, GLdouble x, GLdouble y, GLdouble z)" \
  "void glVertexAttrib3dvNV (GLuint index, const GLdouble* v)" \
  "void glVertexAttrib3fNV (GLuint index, GLfloat x, GLfloat y, GLfloat z)" \
  "void glVertexAttrib3fvNV (GLuint index, const GLfloat* v)" \
  "void glVertexAttrib3sNV (GLuint index, GLshort x, GLshort y, GLshort z)" \
  "void glVertexAttrib3svNV (GLuint index, const GLshort* v)" \
  "void glVertexAttrib4dNV (GLuint index, GLdouble x, GLdouble y, GLdouble z, GLdouble w)" \
  "void glVertexAttrib4dvNV (GLuint index, const GLdouble* v)" \
  "void glVertexAttrib4fNV (GLuint index, GLfloat x, GLfloat y, GLfloat z, GLfloat w)" \
  "void glVertexAttrib4fvNV (GLuint index, const GLfloat* v)" \
  "void glVertexAttrib4sNV (GLuint index, GLshort x, GLshort y, GLshort z, GLshort w)" \
  "void glVertexAttrib4svNV (GLuint index, const GLshort* v)" \
  "void glVertexAttrib4ubNV (GLuint index, GLubyte x, GLubyte y, GLubyte z, GLubyte w)" \
  "void glVertexAttrib4ubvNV (GLuint index, const GLubyte* v)" \
  "void glVertexAttribPointerNV (GLuint index, GLint size, GLenum type, GLsizei stride, const void *pointer)" \
  "void glVertexAttribs1dvNV (GLuint index, GLsizei n, const GLdouble* v)" \
  "void glVertexAttribs1fvNV (GLuint index, GLsizei n, const GLfloat* v)" \
  "void glVertexAttribs1svNV (GLuint index, GLsizei n, const GLshort* v)" \
  "void glVertexAttribs2dvNV (GLuint index, GLsizei n, const GLdouble* v)" \
  "void glVertexAttribs2fvNV (GLuint index, GLsizei n, const GLfloat* v)" \
  "void glVertexAttribs2svNV (GLuint index, GLsizei n, const GLshort* v)" \
  "void glVertexAttribs3dvNV (GLuint index, GLsizei n, const GLdouble* v)" \
  "void glVertexAttribs3fvNV (GLuint index, GLsizei n, const GLfloat* v)" \
  "void glVertexAttribs3svNV (GLuint index, GLsizei n, const GLshort* v)" \
  "void glVertexAttribs4dvNV (GLuint index, GLsizei n, const GLdouble* v)" \
  "void glVertexAttribs4fvNV (GLuint index, GLsizei n, const GLfloat* v)" \
  "void glVertexAttribs4svNV (GLuint index, GLsizei n, const GLshort* v)" \
  "void glVertexAttribs4ubvNV (GLuint index, GLsizei n, const GLubyte* v)" \
  "void glBeginVideoCaptureNV (GLuint video_capture_slot)" \
  "void glBindVideoCaptureStreamBufferNV (GLuint video_capture_slot, GLuint stream, GLenum frame_region, GLintptrARB offset)" \
  "void glBindVideoCaptureStreamTextureNV (GLuint video_capture_slot, GLuint stream, GLenum frame_region, GLenum target, GLuint texture)" \
  "void glEndVideoCaptureNV (GLuint video_capture_slot)" \
  "void glGetVideoCaptureStreamdvNV (GLuint video_capture_slot, GLuint stream, GLenum pname, GLdouble* params)" \
  "void glGetVideoCaptureStreamfvNV (GLuint video_capture_slot, GLuint stream, GLenum pname, GLfloat* params)" \
  "void glGetVideoCaptureStreamivNV (GLuint video_capture_slot, GLuint stream, GLenum pname, GLint* params)" \
  "void glGetVideoCaptureivNV (GLuint video_capture_slot, GLenum pname, GLint* params)" \
  "GLenum glVideoCaptureNV (GLuint video_capture_slot, GLuint* sequence_num, GLuint64EXT *capture_time)" \
  "void glVideoCaptureStreamParameterdvNV (GLuint video_capture_slot, GLuint stream, GLenum pname, const GLdouble* params)" \
  "void glVideoCaptureStreamParameterfvNV (GLuint video_capture_slot, GLuint stream, GLenum pname, const GLfloat* params)" \
  "void glVideoCaptureStreamParameterivNV (GLuint video_capture_slot, GLuint stream, GLenum pname, const GLint* params)" \
  "void glDepthRangeArrayfvNV (GLuint first, GLsizei count, const GLfloat * v)" \
  "void glDepthRangeIndexedfNV (GLuint index, GLfloat n, GLfloat f)" \
  "void glDisableiNV (GLenum target, GLuint index)" \
  "void glEnableiNV (GLenum target, GLuint index)" \
  "void glGetFloati_vNV (GLenum target, GLuint index, GLfloat* data)" \
  "GLboolean glIsEnablediNV (GLenum target, GLuint index)" \
  "void glScissorArrayvNV (GLuint first, GLsizei count, const GLint * v)" \
  "void glScissorIndexedNV (GLuint index, GLint left, GLint bottom, GLsizei width, GLsizei height)" \
  "void glScissorIndexedvNV (GLuint index, const GLint * v)" \
  "void glViewportArrayvNV (GLuint first, GLsizei count, const GLfloat * v)" \
  "void glViewportIndexedfNV (GLuint index, GLfloat x, GLfloat y, GLfloat w, GLfloat h)" \
  "void glViewportIndexedfvNV (GLuint index, const GLfloat * v)" \
  "void glViewportSwizzleNV (GLuint index, GLenum swizzlex, GLenum swizzley, GLenum swizzlez, GLenum swizzlew)" \
  "void glEGLImageTargetRenderbufferStorageOES (GLenum target, GLeglImageOES image)" \
  "void glEGLImageTargetTexture2DOES (GLenum target, GLeglImageOES image)" \
  "void glBlendEquationSeparateOES (GLenum modeRGB, GLenum modeAlpha)" \
  "void glBlendFuncSeparateOES (GLenum sfactorRGB, GLenum dfactorRGB, GLenum sfactorAlpha, GLenum dfactorAlpha)" \
  "void glBlendEquationOES (GLenum mode)" \
  "void glCopyImageSubDataOES (GLuint srcName, GLenum srcTarget, GLint srcLevel, GLint srcX, GLint srcY, GLint srcZ, GLuint dstName, GLenum dstTarget, GLint dstLevel, GLint dstX, GLint dstY, GLint dstZ, GLsizei srcWidth, GLsizei srcHeight, GLsizei srcDepth)" \
  "void glBlendEquationSeparateiOES (GLuint buf, GLenum modeRGB, GLenum modeAlpha)" \
  "void glBlendEquationiOES (GLuint buf, GLenum mode)" \
  "void glBlendFuncSeparateiOES (GLuint buf, GLenum srcRGB, GLenum dstRGB, GLenum srcAlpha, GLenum dstAlpha)" \
  "void glBlendFunciOES (GLuint buf, GLenum src, GLenum dst)" \
  "void glColorMaskiOES (GLuint buf, GLboolean r, GLboolean g, GLboolean b, GLboolean a)" \
  "void glDisableiOES (GLenum target, GLuint index)" \
  "void glEnableiOES (GLenum target, GLuint index)" \
  "GLboolean glIsEnablediOES (GLenum target, GLuint index)" \
  "void glBindFramebufferOES (GLenum target, GLuint framebuffer)" \
  "void glBindRenderbufferOES (GLenum target, GLuint renderbuffer)" \
  "GLenum glCheckFramebufferStatusOES (GLenum target)" \
  "void glDeleteFramebuffersOES (GLsizei n, const GLuint* framebuffers)" \
  "void glDeleteRenderbuffersOES (GLsizei n, const GLuint* renderbuffers)" \
  "void glFramebufferRenderbufferOES (GLenum target, GLenum attachment, GLenum renderbuffertarget, GLuint renderbuffer)" \
  "void glFramebufferTexture2DOES (GLenum target, GLenum attachment, GLenum textarget, GLuint texture, GLint level)" \
  "void glGenFramebuffersOES (GLsizei n, GLuint* framebuffers)" \
  "void glGenRenderbuffersOES (GLsizei n, GLuint* renderbuffers)" \
  "void glGenerateMipmapOES (GLenum target)" \
  "void glGetFramebufferAttachmentParameterivOES (GLenum target, GLenum attachment, GLenum pname, GLint* params)" \
  "void glGetRenderbufferParameterivOES (GLenum target, GLenum pname, GLint* params)" \
  "GLboolean glIsFramebufferOES (GLuint framebuffer)" \
  "GLboolean glIsRenderbufferOES (GLuint renderbuffer)" \
  "void glRenderbufferStorageOES (GLenum target, GLenum internalformat, GLsizei width, GLsizei height)" \
  "void glGetProgramBinaryOES (GLuint program, GLsizei bufSize, GLsizei* length, GLenum *binaryFormat, void*binary)" \
  "void glProgramBinaryOES (GLuint program, GLenum binaryFormat, const void *binary, GLint length)" \
  "void glGetBufferPointervOES (GLenum target, GLenum pname, void** params)" \
  "void * glMapBufferOES (GLenum target, GLenum access)" \
  "GLboolean glUnmapBufferOES (GLenum target)" \
  "void glCurrentPaletteMatrixOES (GLuint index)" \
  "void glMatrixIndexPointerOES (GLint size, GLenum type, GLsizei stride, void *pointer)" \
  "void glWeightPointerOES (GLint size, GLenum type, GLsizei stride, void *pointer)" \
  "void glMinSampleShadingOES (GLfloat value)" \
  "void glClearDepthfOES (GLclampf depth)" \
  "void glClipPlanefOES (GLenum plane, const GLfloat* equation)" \
  "void glDepthRangefOES (GLclampf n, GLclampf f)" \
  "void glFrustumfOES (GLfloat l, GLfloat r, GLfloat b, GLfloat t, GLfloat n, GLfloat f)" \
  "void glGetClipPlanefOES (GLenum plane, GLfloat* equation)" \
  "void glOrthofOES (GLfloat l, GLfloat r, GLfloat b, GLfloat t, GLfloat n, GLfloat f)" \
  "void glCompressedTexImage3DOES (GLenum target, GLint level, GLenum internalformat, GLsizei width, GLsizei height, GLsizei depth, GLint border, GLsizei imageSize, const void *data)" \
  "void glCompressedTexSubImage3DOES (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, GLenum format, GLsizei imageSize, const void *data)" \
  "void glCopyTexSubImage3DOES (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLint x, GLint y, GLsizei width, GLsizei height)" \
  "void glFramebufferTexture3DOES (GLenum target, GLenum attachment, GLenum textarget, GLuint texture, GLint level, GLint zoffset)" \
  "void glTexImage3DOES (GLenum target, GLint level, GLenum internalFormat, GLsizei width, GLsizei height, GLsizei depth, GLint border, GLenum format, GLenum type, const void *pixels)" \
  "void glTexSubImage3DOES (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, GLenum format, GLenum type, const void *pixels)" \
  "void glGetSamplerParameterIivOES (GLuint sampler, GLenum pname, GLint* params)" \
  "void glGetSamplerParameterIuivOES (GLuint sampler, GLenum pname, GLuint* params)" \
  "void glGetTexParameterIivOES (GLenum target, GLenum pname, GLint* params)" \
  "void glGetTexParameterIuivOES (GLenum target, GLenum pname, GLuint* params)" \
  "void glSamplerParameterIivOES (GLuint sampler, GLenum pname, const GLint* params)" \
  "void glSamplerParameterIuivOES (GLuint sampler, GLenum pname, const GLuint* params)" \
  "void glTexParameterIivOES (GLenum target, GLenum pname, const GLint* params)" \
  "void glTexParameterIuivOES (GLenum target, GLenum pname, const GLuint* params)" \
  "void glTexBufferOES (GLenum target, GLenum internalformat, GLuint buffer)" \
  "void glTexBufferRangeOES (GLenum target, GLenum internalformat, GLuint buffer, GLintptr offset, GLsizeiptr size)" \
  "void glGetTexGenfvOES (GLenum coord, GLenum pname, GLfloat* params)" \
  "void glGetTexGenivOES (GLenum coord, GLenum pname, GLint* params)" \
  "void glGetTexGenxvOES (GLenum coord, GLenum pname, GLfixed* params)" \
  "void glTexGenfOES (GLenum coord, GLenum pname, GLfloat param)" \
  "void glTexGenfvOES (GLenum coord, GLenum pname, const GLfloat* params)" \
  "void glTexGeniOES (GLenum coord, GLenum pname, GLint param)" \
  "void glTexGenivOES (GLenum coord, GLenum pname, const GLint* params)" \
  "void glTexGenxOES (GLenum coord, GLenum pname, GLfixed param)" \
  "void glTexGenxvOES (GLenum coord, GLenum pname, const GLfixed* params)" \
  "void glTexStorage3DMultisampleOES (GLenum target, GLsizei samples, GLenum internalformat, GLsizei width, GLsizei height, GLsizei depth, GLboolean fixedsamplelocations)" \
  "void glTextureViewOES (GLuint texture, GLenum target, GLuint origtexture, GLenum internalformat, GLuint minlevel, GLuint numlevels, GLuint minlayer, GLuint numlayers)" \
  "void glBindVertexArrayOES (GLuint array)" \
  "void glDeleteVertexArraysOES (GLsizei n, const GLuint* arrays)" \
  "void glGenVertexArraysOES (GLsizei n, GLuint* arrays)" \
  "GLboolean glIsVertexArrayOES (GLuint array)" \
  "void glFramebufferTextureMultiviewOVR (GLenum target, GLenum attachment, GLuint texture, GLint level, GLint baseViewIndex, GLsizei numViews)" \
  "void glNamedFramebufferTextureMultiviewOVR (GLuint framebuffer, GLenum attachment, GLuint texture, GLint level, GLint baseViewIndex, GLsizei numViews)" \
  "void glFramebufferTextureMultisampleMultiviewOVR (GLenum target, GLenum attachment, GLuint texture, GLint level, GLsizei samples, GLint baseViewIndex, GLsizei numViews)" \
  "void glAlphaFuncQCOM (GLenum func, GLclampf ref)" \
  "void glDisableDriverControlQCOM (GLuint driverControl)" \
  "void glEnableDriverControlQCOM (GLuint driverControl)" \
  "void glGetDriverControlStringQCOM (GLuint driverControl, GLsizei bufSize, GLsizei* length, GLchar *driverControlString)" \
  "void glGetDriverControlsQCOM (GLint* num, GLsizei size, GLuint *driverControls)" \
  "void glExtGetBufferPointervQCOM (GLenum target, void** params)" \
  "void glExtGetBuffersQCOM (GLuint* buffers, GLint maxBuffers, GLint* numBuffers)" \
  "void glExtGetFramebuffersQCOM (GLuint* framebuffers, GLint maxFramebuffers, GLint* numFramebuffers)" \
  "void glExtGetRenderbuffersQCOM (GLuint* renderbuffers, GLint maxRenderbuffers, GLint* numRenderbuffers)" \
  "void glExtGetTexLevelParameterivQCOM (GLuint texture, GLenum face, GLint level, GLenum pname, GLint* params)" \
  "void glExtGetTexSubImageQCOM (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLsizei width, GLsizei height, GLsizei depth, GLenum format, GLenum type, void *texels)" \
  "void glExtGetTexturesQCOM (GLuint* textures, GLint maxTextures, GLint* numTextures)" \
  "void glExtTexObjectStateOverrideiQCOM (GLenum target, GLenum pname, GLint param)" \
  "void glExtGetProgramBinarySourceQCOM (GLuint program, GLenum shadertype, GLchar* source, GLint* length)" \
  "void glExtGetProgramsQCOM (GLuint* programs, GLint maxPrograms, GLint* numPrograms)" \
  "void glExtGetShadersQCOM (GLuint* shaders, GLint maxShaders, GLint* numShaders)" \
  "GLboolean glExtIsProgramBinaryQCOM (GLuint program)" \
  "void glFramebufferFoveationConfigQCOM (GLuint fbo, GLuint numLayers, GLuint focalPointsPerLayer, GLuint requestedFeatures, GLuint* providedFeatures)" \
  "void glFramebufferFoveationParametersQCOM (GLuint fbo, GLuint layer, GLuint focalPoint, GLfloat focalX, GLfloat focalY, GLfloat gainX, GLfloat gainY, GLfloat foveaArea)" \
  "void glFramebufferFetchBarrierQCOM (void)" \
  "void glTextureFoveationParametersQCOM (GLuint texture, GLuint layer, GLuint focalPoint, GLfloat focalX, GLfloat focalY, GLfloat gainX, GLfloat gainY, GLfloat foveaArea)" \
  "void glEndTilingQCOM (GLbitfield preserveMask)" \
  "void glStartTilingQCOM (GLuint x, GLuint y, GLuint width, GLuint height, GLbitfield preserveMask)" \
  "void glAlphaFuncx (GLenum func, GLclampx ref)" \
  "void glClearColorx (GLclampx red, GLclampx green, GLclampx blue, GLclampx alpha)" \
  "void glClearDepthx (GLclampx depth)" \
  "void glColor4x (GLfixed red, GLfixed green, GLfixed blue, GLfixed alpha)" \
  "void glDepthRangex (GLclampx zNear, GLclampx zFar)" \
  "void glFogx (GLenum pname, GLfixed param)" \
  "void glFogxv (GLenum pname, const GLfixed* params)" \
  "void glFrustumf (GLfloat left, GLfloat right, GLfloat bottom, GLfloat top, GLfloat zNear, GLfloat zFar)" \
  "void glFrustumx (GLfixed left, GLfixed right, GLfixed bottom, GLfixed top, GLfixed zNear, GLfixed zFar)" \
  "void glLightModelx (GLenum pname, GLfixed param)" \
  "void glLightModelxv (GLenum pname, const GLfixed* params)" \
  "void glLightx (GLenum light, GLenum pname, GLfixed param)" \
  "void glLightxv (GLenum light, GLenum pname, const GLfixed* params)" \
  "void glLineWidthx (GLfixed width)" \
  "void glLoadMatrixx (const GLfixed* m)" \
  "void glMaterialx (GLenum face, GLenum pname, GLfixed param)" \
  "void glMaterialxv (GLenum face, GLenum pname, const GLfixed* params)" \
  "void glMultMatrixx (const GLfixed* m)" \
  "void glMultiTexCoord4x (GLenum target, GLfixed s, GLfixed t, GLfixed r, GLfixed q)" \
  "void glNormal3x (GLfixed nx, GLfixed ny, GLfixed nz)" \
  "void glOrthof (GLfloat left, GLfloat right, GLfloat bottom, GLfloat top, GLfloat zNear, GLfloat zFar)" \
  "void glOrthox (GLfixed left, GLfixed right, GLfixed bottom, GLfixed top, GLfixed zNear, GLfixed zFar)" \
  "void glPointSizex (GLfixed size)" \
  "void glPolygonOffsetx (GLfixed factor, GLfixed units)" \
  "void glRotatex (GLfixed angle, GLfixed x, GLfixed y, GLfixed z)" \
  "void glSampleCoveragex (GLclampx value, GLboolean invert)" \
  "void glScalex (GLfixed x, GLfixed y, GLfixed z)" \
  "void glTexEnvx (GLenum target, GLenum pname, GLfixed param)" \
  "void glTexEnvxv (GLenum target, GLenum pname, const GLfixed* params)" \
  "void glTexParameterx (GLenum target, GLenum pname, GLfixed param)" \
  "void glTranslatex (GLfixed x, GLfixed y, GLfixed z)" \
  "void glClipPlanef (GLenum plane, const GLfloat* equation)" \
  "void glClipPlanex (GLenum plane, const GLfixed* equation)" \
  "void glGetClipPlanef (GLenum pname, GLfloat eqn\[4\])" \
  "void glGetClipPlanex (GLenum pname, GLfixed eqn\[4\])" \
  "void glGetFixedv (GLenum pname, GLfixed* params)" \
  "void glGetLightxv (GLenum light, GLenum pname, GLfixed* params)" \
  "void glGetMaterialxv (GLenum face, GLenum pname, GLfixed* params)" \
  "void glGetTexEnvxv (GLenum env, GLenum pname, GLfixed* params)" \
  "void glGetTexParameterxv (GLenum target, GLenum pname, GLfixed* params)" \
  "void glPointParameterx (GLenum pname, GLfixed param)" \
  "void glPointParameterxv (GLenum pname, const GLfixed* params)" \
  "void glPointSizePointerOES (GLenum type, GLsizei stride, const void *pointer)" \
  "void glTexParameterxv (GLenum target, GLenum pname, const GLfixed* params)" \
  "const GLchar* glErrorStringREGAL (GLenum error)" \
  "GLboolean glGetExtensionREGAL (const GLchar* ext)" \
  "GLboolean glIsSupportedREGAL (const GLchar* ext)" \
  "void glLogMessageCallbackREGAL (GLLOGPROCREGAL callback)" \
  "void glDetailTexFuncSGIS (GLenum target, GLsizei n, const GLfloat* points)" \
  "void glGetDetailTexFuncSGIS (GLenum target, GLfloat* points)" \
  "void glFogFuncSGIS (GLsizei n, const GLfloat* points)" \
  "void glGetFogFuncSGIS (GLfloat* points)" \
  "void glSampleMaskSGIS (GLclampf value, GLboolean invert)" \
  "void glSamplePatternSGIS (GLenum pattern)" \
  "void glInterleavedTextureCoordSetsSGIS (GLint factor)" \
  "void glSelectTextureCoordSetSGIS (GLenum target)" \
  "void glSelectTextureSGIS (GLenum target)" \
  "void glSelectTextureTransformSGIS (GLenum target)" \
  "void glMultisampleSubRectPosSGIS (GLint x, GLint y)" \
  "void glGetSharpenTexFuncSGIS (GLenum target, GLfloat* points)" \
  "void glSharpenTexFuncSGIS (GLenum target, GLsizei n, const GLfloat* points)" \
  "void glTexImage4DSGIS (GLenum target, GLint level, GLenum internalformat, GLsizei width, GLsizei height, GLsizei depth, GLsizei extent, GLint border, GLenum format, GLenum type, const void *pixels)" \
  "void glTexSubImage4DSGIS (GLenum target, GLint level, GLint xoffset, GLint yoffset, GLint zoffset, GLint woffset, GLsizei width, GLsizei height, GLsizei depth, GLsizei extent, GLenum format, GLenum type, const void *pixels)" \
  "void glGetTexFilterFuncSGIS (GLenum target, GLenum filter, GLfloat* weights)" \
  "void glTexFilterFuncSGIS (GLenum target, GLenum filter, GLsizei n, const GLfloat* weights)" \
  "void glAsyncMarkerSGIX (GLuint marker)" \
  "void glDeleteAsyncMarkersSGIX (GLuint marker, GLsizei range)" \
  "GLint glFinishAsyncSGIX (GLuint* markerp)" \
  "GLuint glGenAsyncMarkersSGIX (GLsizei range)" \
  "GLboolean glIsAsyncMarkerSGIX (GLuint marker)" \
  "GLint glPollAsyncSGIX (GLuint* markerp)" \
  "void glAddressSpace (GLenum space, GLbitfield mask)" \
  "GLint glDataPipe (GLenum space)" \
  "void glFlushRasterSGIX (void)" \
  "void glFogLayersSGIX (GLsizei n, const GLfloat* points)" \
  "void glGetFogLayersSGIX (GLfloat* points)" \
  "void glTextureFogSGIX (GLenum pname)" \
  "void glFragmentColorMaterialSGIX (GLenum face, GLenum mode)" \
  "void glFragmentLightModelfSGIX (GLenum pname, GLfloat param)" \
  "void glFragmentLightModelfvSGIX (GLenum pname, GLfloat* params)" \
  "void glFragmentLightModeliSGIX (GLenum pname, GLint param)" \
  "void glFragmentLightModelivSGIX (GLenum pname, GLint* params)" \
  "void glFragmentLightfSGIX (GLenum light, GLenum pname, GLfloat param)" \
  "void glFragmentLightfvSGIX (GLenum light, GLenum pname, GLfloat* params)" \
  "void glFragmentLightiSGIX (GLenum light, GLenum pname, GLint param)" \
  "void glFragmentLightivSGIX (GLenum light, GLenum pname, GLint* params)" \
  "void glFragmentMaterialfSGIX (GLenum face, GLenum pname, const GLfloat param)" \
  "void glFragmentMaterialfvSGIX (GLenum face, GLenum pname, const GLfloat* params)" \
  "void glFragmentMaterialiSGIX (GLenum face, GLenum pname, const GLint param)" \
  "void glFragmentMaterialivSGIX (GLenum face, GLenum pname, const GLint* params)" \
  "void glGetFragmentLightfvSGIX (GLenum light, GLenum value, GLfloat* data)" \
  "void glGetFragmentLightivSGIX (GLenum light, GLenum value, GLint* data)" \
  "void glGetFragmentMaterialfvSGIX (GLenum face, GLenum pname, GLfloat* data)" \
  "void glGetFragmentMaterialivSGIX (GLenum face, GLenum pname, GLint* data)" \
  "void glFrameZoomSGIX (GLint factor)" \
  "void glIglooInterfaceSGIX (GLenum pname, void *param)" \
  "void glAllocMPEGPredictorsSGIX (GLsizei width, GLsizei height, GLsizei n, GLuint* predictors)" \
  "void glDeleteMPEGPredictorsSGIX (GLsizei n, GLuint* predictors)" \
  "void glGenMPEGPredictorsSGIX (GLsizei n, GLuint* predictors)" \
  "void glGetMPEGParameterfvSGIX (GLenum target, GLenum pname, GLfloat* params)" \
  "void glGetMPEGParameterivSGIX (GLenum target, GLenum pname, GLint* params)" \
  "void glGetMPEGPredictorSGIX (GLenum target, GLenum format, GLenum type, void *pixels)" \
  "void glGetMPEGQuantTableubv (GLenum target, GLubyte* values)" \
  "GLboolean glIsMPEGPredictorSGIX (GLuint predictor)" \
  "void glMPEGPredictorSGIX (GLenum target, GLenum format, GLenum type, void *pixels)" \
  "void glMPEGQuantTableubv (GLenum target, GLubyte* values)" \
  "void glSwapMPEGPredictorsSGIX (GLenum target0, GLenum target1)" \
  "void glGetNonlinLightfvSGIX (GLenum light, GLenum pname, GLint* terms, GLfloat *data)" \
  "void glGetNonlinMaterialfvSGIX (GLenum face, GLenum pname, GLint* terms, const GLfloat *data)" \
  "void glNonlinLightfvSGIX (GLenum light, GLenum pname, GLint terms, GLfloat* params)" \
  "void glNonlinMaterialfvSGIX (GLenum face, GLenum pname, GLint terms, const GLfloat* params)" \
  "void glPixelTexGenSGIX (GLenum mode)" \
  "void glDeformSGIX (GLbitfield mask)" \
  "void glLoadIdentityDeformationMapSGIX (GLbitfield mask)" \
  "void glMeshBreadthSGIX (GLint breadth)" \
  "void glMeshStrideSGIX (GLint stride)" \
  "void glReferencePlaneSGIX (const GLdouble* equation)" \
  "void glSpriteParameterfSGIX (GLenum pname, GLfloat param)" \
  "void glSpriteParameterfvSGIX (GLenum pname, GLfloat* params)" \
  "void glSpriteParameteriSGIX (GLenum pname, GLint param)" \
  "void glSpriteParameterivSGIX (GLenum pname, GLint* params)" \
  "void glTagSampleBufferSGIX (void)" \
  "void glGetVectorOperationSGIX (GLenum operation)" \
  "void glVectorOperationSGIX (GLenum operation)" \
  "GLboolean glAreVertexArraysResidentSGIX (GLsizei n, const GLuint* arrays, GLboolean* residences)" \
  "void glBindVertexArraySGIX (GLuint array)" \
  "void glDeleteVertexArraysSGIX (GLsizei n, const GLuint* arrays)" \
  "void glGenVertexArraysSGIX (GLsizei n, GLuint* arrays)" \
  "GLboolean glIsVertexArraySGIX (GLuint array)" \
  "void glPrioritizeVertexArraysSGIX (GLsizei n, const GLuint* arrays, const GLclampf* priorities)" \
  "void glColorTableParameterfvSGI (GLenum target, GLenum pname, const GLfloat* params)" \
  "void glColorTableParameterivSGI (GLenum target, GLenum pname, const GLint* params)" \
  "void glColorTableSGI (GLenum target, GLenum internalformat, GLsizei width, GLenum format, GLenum type, const void *table)" \
  "void glCopyColorTableSGI (GLenum target, GLenum internalformat, GLint x, GLint y, GLsizei width)" \
  "void glGetColorTableParameterfvSGI (GLenum target, GLenum pname, GLfloat* params)" \
  "void glGetColorTableParameterivSGI (GLenum target, GLenum pname, GLint* params)" \
  "void glGetColorTableSGI (GLenum target, GLenum format, GLenum type, void *table)" \
  "void glGetPixelTransformParameterfvSGI (GLenum target, GLenum pname, GLfloat* params)" \
  "void glGetPixelTransformParameterivSGI (GLenum target, GLenum pname, GLint* params)" \
  "void glPixelTransformParameterfSGI (GLenum target, GLenum pname, GLfloat param)" \
  "void glPixelTransformParameterfvSGI (GLenum target, GLenum pname, const GLfloat* params)" \
  "void glPixelTransformParameteriSGI (GLenum target, GLenum pname, GLint param)" \
  "void glPixelTransformParameterivSGI (GLenum target, GLenum pname, const GLint* params)" \
  "void glPixelTransformSGI (GLenum target)" \
  "void glFinishTextureSUNX (void)" \
  "void glGlobalAlphaFactorbSUN (GLbyte factor)" \
  "void glGlobalAlphaFactordSUN (GLdouble factor)" \
  "void glGlobalAlphaFactorfSUN (GLfloat factor)" \
  "void glGlobalAlphaFactoriSUN (GLint factor)" \
  "void glGlobalAlphaFactorsSUN (GLshort factor)" \
  "void glGlobalAlphaFactorubSUN (GLubyte factor)" \
  "void glGlobalAlphaFactoruiSUN (GLuint factor)" \
  "void glGlobalAlphaFactorusSUN (GLushort factor)" \
  "void glReadVideoPixelsSUN (GLint x, GLint y, GLsizei width, GLsizei height, GLenum format, GLenum type, void* pixels)" \
  "void glReplacementCodePointerSUN (GLenum type, GLsizei stride, const void *pointer)" \
  "void glReplacementCodeubSUN (GLubyte code)" \
  "void glReplacementCodeubvSUN (const GLubyte* code)" \
  "void glReplacementCodeuiSUN (GLuint code)" \
  "void glReplacementCodeuivSUN (const GLuint* code)" \
  "void glReplacementCodeusSUN (GLushort code)" \
  "void glReplacementCodeusvSUN (const GLushort* code)" \
  "void glColor3fVertex3fSUN (GLfloat r, GLfloat g, GLfloat b, GLfloat x, GLfloat y, GLfloat z)" \
  "void glColor3fVertex3fvSUN (const GLfloat* c, const GLfloat *v)" \
  "void glColor4fNormal3fVertex3fSUN (GLfloat r, GLfloat g, GLfloat b, GLfloat a, GLfloat nx, GLfloat ny, GLfloat nz, GLfloat x, GLfloat y, GLfloat z)" \
  "void glColor4fNormal3fVertex3fvSUN (const GLfloat* c, const GLfloat *n, const GLfloat *v)" \
  "void glColor4ubVertex2fSUN (GLubyte r, GLubyte g, GLubyte b, GLubyte a, GLfloat x, GLfloat y)" \
  "void glColor4ubVertex2fvSUN (const GLubyte* c, const GLfloat *v)" \
  "void glColor4ubVertex3fSUN (GLubyte r, GLubyte g, GLubyte b, GLubyte a, GLfloat x, GLfloat y, GLfloat z)" \
  "void glColor4ubVertex3fvSUN (const GLubyte* c, const GLfloat *v)" \
  "void glNormal3fVertex3fSUN (GLfloat nx, GLfloat ny, GLfloat nz, GLfloat x, GLfloat y, GLfloat z)" \
  "void glNormal3fVertex3fvSUN (const GLfloat* n, const GLfloat *v)" \
  "void glReplacementCodeuiColor3fVertex3fSUN (GLuint rc, GLfloat r, GLfloat g, GLfloat b, GLfloat x, GLfloat y, GLfloat z)" \
  "void glReplacementCodeuiColor3fVertex3fvSUN (const GLuint* rc, const GLfloat *c, const GLfloat *v)" \
  "void glReplacementCodeuiColor4fNormal3fVertex3fSUN (GLuint rc, GLfloat r, GLfloat g, GLfloat b, GLfloat a, GLfloat nx, GLfloat ny, GLfloat nz, GLfloat x, GLfloat y, GLfloat z)" \
  "void glReplacementCodeuiColor4fNormal3fVertex3fvSUN (const GLuint* rc, const GLfloat *c, const GLfloat *n, const GLfloat *v)" \
  "void glReplacementCodeuiColor4ubVertex3fSUN (GLuint rc, GLubyte r, GLubyte g, GLubyte b, GLubyte a, GLfloat x, GLfloat y, GLfloat z)" \
  "void glReplacementCodeuiColor4ubVertex3fvSUN (const GLuint* rc, const GLubyte *c, const GLfloat *v)" \
  "void glReplacementCodeuiNormal3fVertex3fSUN (GLuint rc, GLfloat nx, GLfloat ny, GLfloat nz, GLfloat x, GLfloat y, GLfloat z)" \
  "void glReplacementCodeuiNormal3fVertex3fvSUN (const GLuint* rc, const GLfloat *n, const GLfloat *v)" \
  "void glReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fSUN (GLuint rc, GLfloat s, GLfloat t, GLfloat r, GLfloat g, GLfloat b, GLfloat a, GLfloat nx, GLfloat ny, GLfloat nz, GLfloat x, GLfloat y, GLfloat z)" \
  "void glReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fvSUN (const GLuint* rc, const GLfloat *tc, const GLfloat *c, const GLfloat *n, const GLfloat *v)" \
  "void glReplacementCodeuiTexCoord2fNormal3fVertex3fSUN (GLuint rc, GLfloat s, GLfloat t, GLfloat nx, GLfloat ny, GLfloat nz, GLfloat x, GLfloat y, GLfloat z)" \
  "void glReplacementCodeuiTexCoord2fNormal3fVertex3fvSUN (const GLuint* rc, const GLfloat *tc, const GLfloat *n, const GLfloat *v)" \
  "void glReplacementCodeuiTexCoord2fVertex3fSUN (GLuint rc, GLfloat s, GLfloat t, GLfloat x, GLfloat y, GLfloat z)" \
  "void glReplacementCodeuiTexCoord2fVertex3fvSUN (const GLuint* rc, const GLfloat *tc, const GLfloat *v)" \
  "void glReplacementCodeuiVertex3fSUN (GLuint rc, GLfloat x, GLfloat y, GLfloat z)" \
  "void glReplacementCodeuiVertex3fvSUN (const GLuint* rc, const GLfloat *v)" \
  "void glTexCoord2fColor3fVertex3fSUN (GLfloat s, GLfloat t, GLfloat r, GLfloat g, GLfloat b, GLfloat x, GLfloat y, GLfloat z)" \
  "void glTexCoord2fColor3fVertex3fvSUN (const GLfloat* tc, const GLfloat *c, const GLfloat *v)" \
  "void glTexCoord2fColor4fNormal3fVertex3fSUN (GLfloat s, GLfloat t, GLfloat r, GLfloat g, GLfloat b, GLfloat a, GLfloat nx, GLfloat ny, GLfloat nz, GLfloat x, GLfloat y, GLfloat z)" \
  "void glTexCoord2fColor4fNormal3fVertex3fvSUN (const GLfloat* tc, const GLfloat *c, const GLfloat *n, const GLfloat *v)" \
  "void glTexCoord2fColor4ubVertex3fSUN (GLfloat s, GLfloat t, GLubyte r, GLubyte g, GLubyte b, GLubyte a, GLfloat x, GLfloat y, GLfloat z)" \
  "void glTexCoord2fColor4ubVertex3fvSUN (const GLfloat* tc, const GLubyte *c, const GLfloat *v)" \
  "void glTexCoord2fNormal3fVertex3fSUN (GLfloat s, GLfloat t, GLfloat nx, GLfloat ny, GLfloat nz, GLfloat x, GLfloat y, GLfloat z)" \
  "void glTexCoord2fNormal3fVertex3fvSUN (const GLfloat* tc, const GLfloat *n, const GLfloat *v)" \
  "void glTexCoord2fVertex3fSUN (GLfloat s, GLfloat t, GLfloat x, GLfloat y, GLfloat z)" \
  "void glTexCoord2fVertex3fvSUN (const GLfloat* tc, const GLfloat *v)" \
  "void glTexCoord4fColor4fNormal3fVertex4fSUN (GLfloat s, GLfloat t, GLfloat p, GLfloat q, GLfloat r, GLfloat g, GLfloat b, GLfloat a, GLfloat nx, GLfloat ny, GLfloat nz, GLfloat x, GLfloat y, GLfloat z, GLfloat w)" \
  "void glTexCoord4fColor4fNormal3fVertex4fvSUN (const GLfloat* tc, const GLfloat *c, const GLfloat *n, const GLfloat *v)" \
  "void glTexCoord4fVertex4fSUN (GLfloat s, GLfloat t, GLfloat p, GLfloat q, GLfloat x, GLfloat y, GLfloat z, GLfloat w)" \
  "void glTexCoord4fVertex4fvSUN (const GLfloat* tc, const GLfloat *v)" \
  "void glAddSwapHintRectWIN (GLint x, GLint y, GLsizei width, GLsizei height)" \
]

# List of the OpenGL versions or extensions of all wrapped OpenGL functions.
set ::__tcl3dOglFuncVersionList [list \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_1 \
  GL_VERSION_1_2 \
  GL_VERSION_1_2 \
  GL_VERSION_1_2 \
  GL_VERSION_1_2 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_3 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_4 \
  GL_VERSION_1_5 \
  GL_VERSION_1_5 \
  GL_VERSION_1_5 \
  GL_VERSION_1_5 \
  GL_VERSION_1_5 \
  GL_VERSION_1_5 \
  GL_VERSION_1_5 \
  GL_VERSION_1_5 \
  GL_VERSION_1_5 \
  GL_VERSION_1_5 \
  GL_VERSION_1_5 \
  GL_VERSION_1_5 \
  GL_VERSION_1_5 \
  GL_VERSION_1_5 \
  GL_VERSION_1_5 \
  GL_VERSION_1_5 \
  GL_VERSION_1_5 \
  GL_VERSION_1_5 \
  GL_VERSION_1_5 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_0 \
  GL_VERSION_2_1 \
  GL_VERSION_2_1 \
  GL_VERSION_2_1 \
  GL_VERSION_2_1 \
  GL_VERSION_2_1 \
  GL_VERSION_2_1 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_0 \
  GL_VERSION_3_1 \
  GL_VERSION_3_1 \
  GL_VERSION_3_1 \
  GL_VERSION_3_1 \
  GL_VERSION_3_2 \
  GL_VERSION_3_2 \
  GL_VERSION_3_2 \
  GL_VERSION_3_3 \
  GL_VERSION_4_0 \
  GL_VERSION_4_0 \
  GL_VERSION_4_0 \
  GL_VERSION_4_0 \
  GL_VERSION_4_0 \
  GL_VERSION_4_5 \
  GL_VERSION_4_5 \
  GL_VERSION_4_5 \
  GL_VERSION_4_5 \
  GL_VERSION_4_6 \
  GL_VERSION_4_6 \
  GL_VERSION_4_6 \
  GL_3DFX_tbuffer \
  GL_AMD_debug_output \
  GL_AMD_debug_output \
  GL_AMD_debug_output \
  GL_AMD_debug_output \
  GL_AMD_draw_buffers_blend \
  GL_AMD_draw_buffers_blend \
  GL_AMD_draw_buffers_blend \
  GL_AMD_draw_buffers_blend \
  GL_AMD_framebuffer_multisample_advanced \
  GL_AMD_framebuffer_multisample_advanced \
  GL_AMD_framebuffer_sample_positions \
  GL_AMD_framebuffer_sample_positions \
  GL_AMD_framebuffer_sample_positions \
  GL_AMD_framebuffer_sample_positions \
  GL_AMD_interleaved_elements \
  GL_AMD_multi_draw_indirect \
  GL_AMD_multi_draw_indirect \
  GL_AMD_name_gen_delete \
  GL_AMD_name_gen_delete \
  GL_AMD_name_gen_delete \
  GL_AMD_occlusion_query_event \
  GL_AMD_performance_monitor \
  GL_AMD_performance_monitor \
  GL_AMD_performance_monitor \
  GL_AMD_performance_monitor \
  GL_AMD_performance_monitor \
  GL_AMD_performance_monitor \
  GL_AMD_performance_monitor \
  GL_AMD_performance_monitor \
  GL_AMD_performance_monitor \
  GL_AMD_performance_monitor \
  GL_AMD_performance_monitor \
  GL_AMD_sample_positions \
  GL_AMD_sparse_texture \
  GL_AMD_sparse_texture \
  GL_AMD_stencil_operation_extended \
  GL_AMD_vertex_shader_tessellator \
  GL_AMD_vertex_shader_tessellator \
  GL_ANGLE_framebuffer_blit \
  GL_ANGLE_framebuffer_multisample \
  GL_ANGLE_instanced_arrays \
  GL_ANGLE_instanced_arrays \
  GL_ANGLE_instanced_arrays \
  GL_ANGLE_timer_query \
  GL_ANGLE_timer_query \
  GL_ANGLE_timer_query \
  GL_ANGLE_timer_query \
  GL_ANGLE_timer_query \
  GL_ANGLE_timer_query \
  GL_ANGLE_timer_query \
  GL_ANGLE_timer_query \
  GL_ANGLE_timer_query \
  GL_ANGLE_timer_query \
  GL_ANGLE_timer_query \
  GL_ANGLE_translated_shader_source \
  GL_APPLE_copy_texture_levels \
  GL_APPLE_element_array \
  GL_APPLE_element_array \
  GL_APPLE_element_array \
  GL_APPLE_element_array \
  GL_APPLE_element_array \
  GL_APPLE_fence \
  GL_APPLE_fence \
  GL_APPLE_fence \
  GL_APPLE_fence \
  GL_APPLE_fence \
  GL_APPLE_fence \
  GL_APPLE_fence \
  GL_APPLE_fence \
  GL_APPLE_flush_buffer_range \
  GL_APPLE_flush_buffer_range \
  GL_APPLE_framebuffer_multisample \
  GL_APPLE_framebuffer_multisample \
  GL_APPLE_object_purgeable \
  GL_APPLE_object_purgeable \
  GL_APPLE_object_purgeable \
  GL_APPLE_sync \
  GL_APPLE_sync \
  GL_APPLE_sync \
  GL_APPLE_sync \
  GL_APPLE_sync \
  GL_APPLE_sync \
  GL_APPLE_sync \
  GL_APPLE_texture_range \
  GL_APPLE_texture_range \
  GL_APPLE_vertex_array_object \
  GL_APPLE_vertex_array_object \
  GL_APPLE_vertex_array_object \
  GL_APPLE_vertex_array_object \
  GL_APPLE_vertex_array_range \
  GL_APPLE_vertex_array_range \
  GL_APPLE_vertex_array_range \
  GL_APPLE_vertex_program_evaluators \
  GL_APPLE_vertex_program_evaluators \
  GL_APPLE_vertex_program_evaluators \
  GL_APPLE_vertex_program_evaluators \
  GL_APPLE_vertex_program_evaluators \
  GL_APPLE_vertex_program_evaluators \
  GL_APPLE_vertex_program_evaluators \
  GL_ARB_ES2_compatibility \
  GL_ARB_ES2_compatibility \
  GL_ARB_ES2_compatibility \
  GL_ARB_ES2_compatibility \
  GL_ARB_ES2_compatibility \
  GL_ARB_ES3_1_compatibility \
  GL_ARB_ES3_2_compatibility \
  GL_ARB_base_instance \
  GL_ARB_base_instance \
  GL_ARB_base_instance \
  GL_ARB_bindless_texture \
  GL_ARB_bindless_texture \
  GL_ARB_bindless_texture \
  GL_ARB_bindless_texture \
  GL_ARB_bindless_texture \
  GL_ARB_bindless_texture \
  GL_ARB_bindless_texture \
  GL_ARB_bindless_texture \
  GL_ARB_bindless_texture \
  GL_ARB_bindless_texture \
  GL_ARB_bindless_texture \
  GL_ARB_bindless_texture \
  GL_ARB_bindless_texture \
  GL_ARB_bindless_texture \
  GL_ARB_bindless_texture \
  GL_ARB_bindless_texture \
  GL_ARB_blend_func_extended \
  GL_ARB_blend_func_extended \
  GL_ARB_buffer_storage \
  GL_ARB_cl_event \
  GL_ARB_clear_buffer_object \
  GL_ARB_clear_buffer_object \
  GL_ARB_clear_buffer_object \
  GL_ARB_clear_buffer_object \
  GL_ARB_clear_texture \
  GL_ARB_clear_texture \
  GL_ARB_clip_control \
  GL_ARB_color_buffer_float \
  GL_ARB_compute_shader \
  GL_ARB_compute_shader \
  GL_ARB_compute_variable_group_size \
  GL_ARB_copy_buffer \
  GL_ARB_copy_image \
  GL_ARB_debug_output \
  GL_ARB_debug_output \
  GL_ARB_debug_output \
  GL_ARB_debug_output \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_direct_state_access \
  GL_ARB_draw_buffers \
  GL_ARB_draw_buffers_blend \
  GL_ARB_draw_buffers_blend \
  GL_ARB_draw_buffers_blend \
  GL_ARB_draw_buffers_blend \
  GL_ARB_draw_elements_base_vertex \
  GL_ARB_draw_elements_base_vertex \
  GL_ARB_draw_elements_base_vertex \
  GL_ARB_draw_elements_base_vertex \
  GL_ARB_draw_indirect \
  GL_ARB_draw_indirect \
  GL_ARB_framebuffer_no_attachments \
  GL_ARB_framebuffer_no_attachments \
  GL_ARB_framebuffer_no_attachments \
  GL_ARB_framebuffer_no_attachments \
  GL_ARB_framebuffer_object \
  GL_ARB_framebuffer_object \
  GL_ARB_framebuffer_object \
  GL_ARB_framebuffer_object \
  GL_ARB_framebuffer_object \
  GL_ARB_framebuffer_object \
  GL_ARB_framebuffer_object \
  GL_ARB_framebuffer_object \
  GL_ARB_framebuffer_object \
  GL_ARB_framebuffer_object \
  GL_ARB_framebuffer_object \
  GL_ARB_framebuffer_object \
  GL_ARB_framebuffer_object \
  GL_ARB_framebuffer_object \
  GL_ARB_framebuffer_object \
  GL_ARB_framebuffer_object \
  GL_ARB_framebuffer_object \
  GL_ARB_framebuffer_object \
  GL_ARB_framebuffer_object \
  GL_ARB_framebuffer_object \
  GL_ARB_geometry_shader4 \
  GL_ARB_geometry_shader4 \
  GL_ARB_geometry_shader4 \
  GL_ARB_geometry_shader4 \
  GL_ARB_get_program_binary \
  GL_ARB_get_program_binary \
  GL_ARB_get_program_binary \
  GL_ARB_get_texture_sub_image \
  GL_ARB_get_texture_sub_image \
  GL_ARB_gl_spirv \
  GL_ARB_gpu_shader_fp64 \
  GL_ARB_gpu_shader_fp64 \
  GL_ARB_gpu_shader_fp64 \
  GL_ARB_gpu_shader_fp64 \
  GL_ARB_gpu_shader_fp64 \
  GL_ARB_gpu_shader_fp64 \
  GL_ARB_gpu_shader_fp64 \
  GL_ARB_gpu_shader_fp64 \
  GL_ARB_gpu_shader_fp64 \
  GL_ARB_gpu_shader_fp64 \
  GL_ARB_gpu_shader_fp64 \
  GL_ARB_gpu_shader_fp64 \
  GL_ARB_gpu_shader_fp64 \
  GL_ARB_gpu_shader_fp64 \
  GL_ARB_gpu_shader_fp64 \
  GL_ARB_gpu_shader_fp64 \
  GL_ARB_gpu_shader_fp64 \
  GL_ARB_gpu_shader_fp64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_gpu_shader_int64 \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_imaging \
  GL_ARB_indirect_parameters \
  GL_ARB_indirect_parameters \
  GL_ARB_instanced_arrays \
  GL_ARB_instanced_arrays \
  GL_ARB_instanced_arrays \
  GL_ARB_internalformat_query \
  GL_ARB_internalformat_query2 \
  GL_ARB_invalidate_subdata \
  GL_ARB_invalidate_subdata \
  GL_ARB_invalidate_subdata \
  GL_ARB_invalidate_subdata \
  GL_ARB_invalidate_subdata \
  GL_ARB_invalidate_subdata \
  GL_ARB_map_buffer_range \
  GL_ARB_map_buffer_range \
  GL_ARB_matrix_palette \
  GL_ARB_matrix_palette \
  GL_ARB_matrix_palette \
  GL_ARB_matrix_palette \
  GL_ARB_matrix_palette \
  GL_ARB_multi_bind \
  GL_ARB_multi_bind \
  GL_ARB_multi_bind \
  GL_ARB_multi_bind \
  GL_ARB_multi_bind \
  GL_ARB_multi_bind \
  GL_ARB_multi_draw_indirect \
  GL_ARB_multi_draw_indirect \
  GL_ARB_multisample \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_multitexture \
  GL_ARB_occlusion_query \
  GL_ARB_occlusion_query \
  GL_ARB_occlusion_query \
  GL_ARB_occlusion_query \
  GL_ARB_occlusion_query \
  GL_ARB_occlusion_query \
  GL_ARB_occlusion_query \
  GL_ARB_occlusion_query \
  GL_ARB_parallel_shader_compile \
  GL_ARB_point_parameters \
  GL_ARB_point_parameters \
  GL_ARB_polygon_offset_clamp \
  GL_ARB_program_interface_query \
  GL_ARB_program_interface_query \
  GL_ARB_program_interface_query \
  GL_ARB_program_interface_query \
  GL_ARB_program_interface_query \
  GL_ARB_program_interface_query \
  GL_ARB_provoking_vertex \
  GL_ARB_robustness \
  GL_ARB_robustness \
  GL_ARB_robustness \
  GL_ARB_robustness \
  GL_ARB_robustness \
  GL_ARB_robustness \
  GL_ARB_robustness \
  GL_ARB_robustness \
  GL_ARB_robustness \
  GL_ARB_robustness \
  GL_ARB_robustness \
  GL_ARB_robustness \
  GL_ARB_robustness \
  GL_ARB_robustness \
  GL_ARB_robustness \
  GL_ARB_robustness \
  GL_ARB_robustness \
  GL_ARB_robustness \
  GL_ARB_robustness \
  GL_ARB_robustness \
  GL_ARB_sample_locations \
  GL_ARB_sample_locations \
  GL_ARB_sample_shading \
  GL_ARB_sampler_objects \
  GL_ARB_sampler_objects \
  GL_ARB_sampler_objects \
  GL_ARB_sampler_objects \
  GL_ARB_sampler_objects \
  GL_ARB_sampler_objects \
  GL_ARB_sampler_objects \
  GL_ARB_sampler_objects \
  GL_ARB_sampler_objects \
  GL_ARB_sampler_objects \
  GL_ARB_sampler_objects \
  GL_ARB_sampler_objects \
  GL_ARB_sampler_objects \
  GL_ARB_sampler_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_separate_shader_objects \
  GL_ARB_shader_atomic_counters \
  GL_ARB_shader_image_load_store \
  GL_ARB_shader_image_load_store \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_objects \
  GL_ARB_shader_storage_buffer_object \
  GL_ARB_shader_subroutine \
  GL_ARB_shader_subroutine \
  GL_ARB_shader_subroutine \
  GL_ARB_shader_subroutine \
  GL_ARB_shader_subroutine \
  GL_ARB_shader_subroutine \
  GL_ARB_shader_subroutine \
  GL_ARB_shader_subroutine \
  GL_ARB_shading_language_include \
  GL_ARB_shading_language_include \
  GL_ARB_shading_language_include \
  GL_ARB_shading_language_include \
  GL_ARB_shading_language_include \
  GL_ARB_shading_language_include \
  GL_ARB_sparse_buffer \
  GL_ARB_sparse_texture \
  GL_ARB_sync \
  GL_ARB_sync \
  GL_ARB_sync \
  GL_ARB_sync \
  GL_ARB_sync \
  GL_ARB_sync \
  GL_ARB_sync \
  GL_ARB_tessellation_shader \
  GL_ARB_tessellation_shader \
  GL_ARB_texture_barrier \
  GL_ARB_texture_buffer_object \
  GL_ARB_texture_buffer_range \
  GL_ARB_texture_buffer_range \
  GL_ARB_texture_compression \
  GL_ARB_texture_compression \
  GL_ARB_texture_compression \
  GL_ARB_texture_compression \
  GL_ARB_texture_compression \
  GL_ARB_texture_compression \
  GL_ARB_texture_compression \
  GL_ARB_texture_multisample \
  GL_ARB_texture_multisample \
  GL_ARB_texture_multisample \
  GL_ARB_texture_multisample \
  GL_ARB_texture_storage \
  GL_ARB_texture_storage \
  GL_ARB_texture_storage \
  GL_ARB_texture_storage_multisample \
  GL_ARB_texture_storage_multisample \
  GL_ARB_texture_storage_multisample \
  GL_ARB_texture_storage_multisample \
  GL_ARB_texture_view \
  GL_ARB_timer_query \
  GL_ARB_timer_query \
  GL_ARB_timer_query \
  GL_ARB_transform_feedback2 \
  GL_ARB_transform_feedback2 \
  GL_ARB_transform_feedback2 \
  GL_ARB_transform_feedback2 \
  GL_ARB_transform_feedback2 \
  GL_ARB_transform_feedback2 \
  GL_ARB_transform_feedback2 \
  GL_ARB_transform_feedback3 \
  GL_ARB_transform_feedback3 \
  GL_ARB_transform_feedback3 \
  GL_ARB_transform_feedback3 \
  GL_ARB_transform_feedback_instanced \
  GL_ARB_transform_feedback_instanced \
  GL_ARB_transpose_matrix \
  GL_ARB_transpose_matrix \
  GL_ARB_transpose_matrix \
  GL_ARB_transpose_matrix \
  GL_ARB_uniform_buffer_object \
  GL_ARB_uniform_buffer_object \
  GL_ARB_uniform_buffer_object \
  GL_ARB_uniform_buffer_object \
  GL_ARB_uniform_buffer_object \
  GL_ARB_uniform_buffer_object \
  GL_ARB_uniform_buffer_object \
  GL_ARB_uniform_buffer_object \
  GL_ARB_uniform_buffer_object \
  GL_ARB_uniform_buffer_object \
  GL_ARB_vertex_array_object \
  GL_ARB_vertex_array_object \
  GL_ARB_vertex_array_object \
  GL_ARB_vertex_array_object \
  GL_ARB_vertex_attrib_64bit \
  GL_ARB_vertex_attrib_64bit \
  GL_ARB_vertex_attrib_64bit \
  GL_ARB_vertex_attrib_64bit \
  GL_ARB_vertex_attrib_64bit \
  GL_ARB_vertex_attrib_64bit \
  GL_ARB_vertex_attrib_64bit \
  GL_ARB_vertex_attrib_64bit \
  GL_ARB_vertex_attrib_64bit \
  GL_ARB_vertex_attrib_64bit \
  GL_ARB_vertex_attrib_binding \
  GL_ARB_vertex_attrib_binding \
  GL_ARB_vertex_attrib_binding \
  GL_ARB_vertex_attrib_binding \
  GL_ARB_vertex_attrib_binding \
  GL_ARB_vertex_attrib_binding \
  GL_ARB_vertex_attrib_binding \
  GL_ARB_vertex_attrib_binding \
  GL_ARB_vertex_attrib_binding \
  GL_ARB_vertex_attrib_binding \
  GL_ARB_vertex_attrib_binding \
  GL_ARB_vertex_attrib_binding \
  GL_ARB_vertex_blend \
  GL_ARB_vertex_blend \
  GL_ARB_vertex_blend \
  GL_ARB_vertex_blend \
  GL_ARB_vertex_blend \
  GL_ARB_vertex_blend \
  GL_ARB_vertex_blend \
  GL_ARB_vertex_blend \
  GL_ARB_vertex_blend \
  GL_ARB_vertex_blend \
  GL_ARB_vertex_buffer_object \
  GL_ARB_vertex_buffer_object \
  GL_ARB_vertex_buffer_object \
  GL_ARB_vertex_buffer_object \
  GL_ARB_vertex_buffer_object \
  GL_ARB_vertex_buffer_object \
  GL_ARB_vertex_buffer_object \
  GL_ARB_vertex_buffer_object \
  GL_ARB_vertex_buffer_object \
  GL_ARB_vertex_buffer_object \
  GL_ARB_vertex_buffer_object \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_program \
  GL_ARB_vertex_shader \
  GL_ARB_vertex_shader \
  GL_ARB_vertex_shader \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_vertex_type_2_10_10_10_rev \
  GL_ARB_viewport_array \
  GL_ARB_viewport_array \
  GL_ARB_viewport_array \
  GL_ARB_viewport_array \
  GL_ARB_viewport_array \
  GL_ARB_viewport_array \
  GL_ARB_viewport_array \
  GL_ARB_viewport_array \
  GL_ARB_viewport_array \
  GL_ARB_viewport_array \
  GL_ARB_window_pos \
  GL_ARB_window_pos \
  GL_ARB_window_pos \
  GL_ARB_window_pos \
  GL_ARB_window_pos \
  GL_ARB_window_pos \
  GL_ARB_window_pos \
  GL_ARB_window_pos \
  GL_ARB_window_pos \
  GL_ARB_window_pos \
  GL_ARB_window_pos \
  GL_ARB_window_pos \
  GL_ARB_window_pos \
  GL_ARB_window_pos \
  GL_ARB_window_pos \
  GL_ARB_window_pos \
  GL_ATI_draw_buffers \
  GL_ATI_element_array \
  GL_ATI_element_array \
  GL_ATI_element_array \
  GL_ATI_envmap_bumpmap \
  GL_ATI_envmap_bumpmap \
  GL_ATI_envmap_bumpmap \
  GL_ATI_envmap_bumpmap \
  GL_ATI_fragment_shader \
  GL_ATI_fragment_shader \
  GL_ATI_fragment_shader \
  GL_ATI_fragment_shader \
  GL_ATI_fragment_shader \
  GL_ATI_fragment_shader \
  GL_ATI_fragment_shader \
  GL_ATI_fragment_shader \
  GL_ATI_fragment_shader \
  GL_ATI_fragment_shader \
  GL_ATI_fragment_shader \
  GL_ATI_fragment_shader \
  GL_ATI_fragment_shader \
  GL_ATI_fragment_shader \
  GL_ATI_map_object_buffer \
  GL_ATI_map_object_buffer \
  GL_ATI_pn_triangles \
  GL_ATI_pn_triangles \
  GL_ATI_separate_stencil \
  GL_ATI_separate_stencil \
  GL_ATI_vertex_array_object \
  GL_ATI_vertex_array_object \
  GL_ATI_vertex_array_object \
  GL_ATI_vertex_array_object \
  GL_ATI_vertex_array_object \
  GL_ATI_vertex_array_object \
  GL_ATI_vertex_array_object \
  GL_ATI_vertex_array_object \
  GL_ATI_vertex_array_object \
  GL_ATI_vertex_array_object \
  GL_ATI_vertex_array_object \
  GL_ATI_vertex_array_object \
  GL_ATI_vertex_attrib_array_object \
  GL_ATI_vertex_attrib_array_object \
  GL_ATI_vertex_attrib_array_object \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_ATI_vertex_streams \
  GL_EXT_EGL_image_storage \
  GL_EXT_EGL_image_storage \
  GL_EXT_base_instance \
  GL_EXT_base_instance \
  GL_EXT_base_instance \
  GL_EXT_bindable_uniform \
  GL_EXT_bindable_uniform \
  GL_EXT_bindable_uniform \
  GL_EXT_blend_color \
  GL_EXT_blend_equation_separate \
  GL_EXT_blend_func_extended \
  GL_EXT_blend_func_extended \
  GL_EXT_blend_func_extended \
  GL_EXT_blend_func_separate \
  GL_EXT_blend_minmax \
  GL_EXT_buffer_storage \
  GL_EXT_buffer_storage \
  GL_EXT_clear_texture \
  GL_EXT_clear_texture \
  GL_EXT_clip_control \
  GL_EXT_color_subtable \
  GL_EXT_color_subtable \
  GL_EXT_compiled_vertex_array \
  GL_EXT_compiled_vertex_array \
  GL_EXT_convolution \
  GL_EXT_convolution \
  GL_EXT_convolution \
  GL_EXT_convolution \
  GL_EXT_convolution \
  GL_EXT_convolution \
  GL_EXT_convolution \
  GL_EXT_convolution \
  GL_EXT_convolution \
  GL_EXT_convolution \
  GL_EXT_convolution \
  GL_EXT_convolution \
  GL_EXT_convolution \
  GL_EXT_coordinate_frame \
  GL_EXT_coordinate_frame \
  GL_EXT_copy_image \
  GL_EXT_copy_texture \
  GL_EXT_copy_texture \
  GL_EXT_copy_texture \
  GL_EXT_copy_texture \
  GL_EXT_copy_texture \
  GL_EXT_cull_vertex \
  GL_EXT_cull_vertex \
  GL_EXT_debug_label \
  GL_EXT_debug_label \
  GL_EXT_debug_marker \
  GL_EXT_debug_marker \
  GL_EXT_debug_marker \
  GL_EXT_depth_bounds_test \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_direct_state_access \
  GL_EXT_discard_framebuffer \
  GL_EXT_disjoint_timer_query \
  GL_EXT_disjoint_timer_query \
  GL_EXT_disjoint_timer_query \
  GL_EXT_disjoint_timer_query \
  GL_EXT_disjoint_timer_query \
  GL_EXT_disjoint_timer_query \
  GL_EXT_disjoint_timer_query \
  GL_EXT_disjoint_timer_query \
  GL_EXT_disjoint_timer_query \
  GL_EXT_disjoint_timer_query \
  GL_EXT_draw_buffers \
  GL_EXT_draw_buffers2 \
  GL_EXT_draw_buffers2 \
  GL_EXT_draw_buffers2 \
  GL_EXT_draw_buffers2 \
  GL_EXT_draw_buffers2 \
  GL_EXT_draw_buffers2 \
  GL_EXT_draw_buffers_indexed \
  GL_EXT_draw_buffers_indexed \
  GL_EXT_draw_buffers_indexed \
  GL_EXT_draw_buffers_indexed \
  GL_EXT_draw_buffers_indexed \
  GL_EXT_draw_buffers_indexed \
  GL_EXT_draw_buffers_indexed \
  GL_EXT_draw_buffers_indexed \
  GL_EXT_draw_elements_base_vertex \
  GL_EXT_draw_elements_base_vertex \
  GL_EXT_draw_elements_base_vertex \
  GL_EXT_draw_elements_base_vertex \
  GL_EXT_draw_instanced \
  GL_EXT_draw_instanced \
  GL_EXT_draw_range_elements \
  GL_EXT_draw_transform_feedback \
  GL_EXT_draw_transform_feedback \
  GL_EXT_external_buffer \
  GL_EXT_external_buffer \
  GL_EXT_fog_coord \
  GL_EXT_fog_coord \
  GL_EXT_fog_coord \
  GL_EXT_fog_coord \
  GL_EXT_fog_coord \
  GL_EXT_fragment_lighting \
  GL_EXT_fragment_lighting \
  GL_EXT_fragment_lighting \
  GL_EXT_fragment_lighting \
  GL_EXT_fragment_lighting \
  GL_EXT_fragment_lighting \
  GL_EXT_fragment_lighting \
  GL_EXT_fragment_lighting \
  GL_EXT_fragment_lighting \
  GL_EXT_fragment_lighting \
  GL_EXT_fragment_lighting \
  GL_EXT_fragment_lighting \
  GL_EXT_fragment_lighting \
  GL_EXT_fragment_lighting \
  GL_EXT_fragment_lighting \
  GL_EXT_fragment_lighting \
  GL_EXT_fragment_lighting \
  GL_EXT_fragment_lighting \
  GL_EXT_framebuffer_blit \
  GL_EXT_framebuffer_multisample \
  GL_EXT_framebuffer_object \
  GL_EXT_framebuffer_object \
  GL_EXT_framebuffer_object \
  GL_EXT_framebuffer_object \
  GL_EXT_framebuffer_object \
  GL_EXT_framebuffer_object \
  GL_EXT_framebuffer_object \
  GL_EXT_framebuffer_object \
  GL_EXT_framebuffer_object \
  GL_EXT_framebuffer_object \
  GL_EXT_framebuffer_object \
  GL_EXT_framebuffer_object \
  GL_EXT_framebuffer_object \
  GL_EXT_framebuffer_object \
  GL_EXT_framebuffer_object \
  GL_EXT_framebuffer_object \
  GL_EXT_framebuffer_object \
  GL_EXT_geometry_shader4 \
  GL_EXT_geometry_shader4 \
  GL_EXT_geometry_shader4 \
  GL_EXT_gpu_program_parameters \
  GL_EXT_gpu_program_parameters \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_gpu_shader4 \
  GL_EXT_histogram \
  GL_EXT_histogram \
  GL_EXT_histogram \
  GL_EXT_histogram \
  GL_EXT_histogram \
  GL_EXT_histogram \
  GL_EXT_histogram \
  GL_EXT_histogram \
  GL_EXT_histogram \
  GL_EXT_histogram \
  GL_EXT_index_func \
  GL_EXT_index_material \
  GL_EXT_instanced_arrays \
  GL_EXT_light_texture \
  GL_EXT_light_texture \
  GL_EXT_light_texture \
  GL_EXT_map_buffer_range \
  GL_EXT_map_buffer_range \
  GL_EXT_memory_object \
  GL_EXT_memory_object \
  GL_EXT_memory_object \
  GL_EXT_memory_object \
  GL_EXT_memory_object \
  GL_EXT_memory_object \
  GL_EXT_memory_object \
  GL_EXT_memory_object \
  GL_EXT_memory_object \
  GL_EXT_memory_object \
  GL_EXT_memory_object \
  GL_EXT_memory_object \
  GL_EXT_memory_object \
  GL_EXT_memory_object \
  GL_EXT_memory_object \
  GL_EXT_memory_object \
  GL_EXT_memory_object \
  GL_EXT_memory_object \
  GL_EXT_memory_object \
  GL_EXT_memory_object_fd \
  GL_EXT_memory_object_win32 \
  GL_EXT_memory_object_win32 \
  GL_EXT_multi_draw_arrays \
  GL_EXT_multi_draw_arrays \
  GL_EXT_multi_draw_indirect \
  GL_EXT_multi_draw_indirect \
  GL_EXT_multisample \
  GL_EXT_multisample \
  GL_EXT_multisampled_render_to_texture \
  GL_EXT_multiview_draw_buffers \
  GL_EXT_multiview_draw_buffers \
  GL_EXT_multiview_draw_buffers \
  GL_EXT_paletted_texture \
  GL_EXT_paletted_texture \
  GL_EXT_paletted_texture \
  GL_EXT_paletted_texture \
  GL_EXT_pixel_transform \
  GL_EXT_pixel_transform \
  GL_EXT_pixel_transform \
  GL_EXT_pixel_transform \
  GL_EXT_pixel_transform \
  GL_EXT_pixel_transform \
  GL_EXT_point_parameters \
  GL_EXT_point_parameters \
  GL_EXT_polygon_offset \
  GL_EXT_polygon_offset_clamp \
  GL_EXT_primitive_bounding_box \
  GL_EXT_provoking_vertex \
  GL_EXT_raster_multisample \
  GL_EXT_raster_multisample \
  GL_EXT_raster_multisample \
  GL_EXT_raster_multisample \
  GL_EXT_robustness \
  GL_EXT_robustness \
  GL_EXT_robustness \
  GL_EXT_scene_marker \
  GL_EXT_scene_marker \
  GL_EXT_secondary_color \
  GL_EXT_secondary_color \
  GL_EXT_secondary_color \
  GL_EXT_secondary_color \
  GL_EXT_secondary_color \
  GL_EXT_secondary_color \
  GL_EXT_secondary_color \
  GL_EXT_secondary_color \
  GL_EXT_secondary_color \
  GL_EXT_secondary_color \
  GL_EXT_secondary_color \
  GL_EXT_secondary_color \
  GL_EXT_secondary_color \
  GL_EXT_secondary_color \
  GL_EXT_secondary_color \
  GL_EXT_secondary_color \
  GL_EXT_secondary_color \
  GL_EXT_semaphore \
  GL_EXT_semaphore \
  GL_EXT_semaphore \
  GL_EXT_semaphore \
  GL_EXT_semaphore \
  GL_EXT_semaphore \
  GL_EXT_semaphore \
  GL_EXT_semaphore_fd \
  GL_EXT_semaphore_win32 \
  GL_EXT_semaphore_win32 \
  GL_EXT_separate_shader_objects \
  GL_EXT_separate_shader_objects \
  GL_EXT_separate_shader_objects \
  GL_EXT_shader_framebuffer_fetch \
  GL_EXT_shader_image_load_store \
  GL_EXT_shader_image_load_store \
  GL_EXT_shader_pixel_local_storage2 \
  GL_EXT_shader_pixel_local_storage2 \
  GL_EXT_shader_pixel_local_storage2 \
  GL_EXT_sparse_texture \
  GL_EXT_sparse_texture \
  GL_EXT_stencil_two_side \
  GL_EXT_subtexture \
  GL_EXT_subtexture \
  GL_EXT_subtexture \
  GL_EXT_tessellation_point_size \
  GL_EXT_texture3D \
  GL_EXT_texture_array \
  GL_EXT_texture_border_clamp \
  GL_EXT_texture_border_clamp \
  GL_EXT_texture_border_clamp \
  GL_EXT_texture_border_clamp \
  GL_EXT_texture_buffer_object \
  GL_EXT_texture_integer \
  GL_EXT_texture_integer \
  GL_EXT_texture_integer \
  GL_EXT_texture_integer \
  GL_EXT_texture_integer \
  GL_EXT_texture_integer \
  GL_EXT_texture_object \
  GL_EXT_texture_object \
  GL_EXT_texture_object \
  GL_EXT_texture_object \
  GL_EXT_texture_object \
  GL_EXT_texture_object \
  GL_EXT_texture_perturb_normal \
  GL_EXT_texture_storage \
  GL_EXT_texture_storage \
  GL_EXT_texture_storage \
  GL_EXT_texture_storage \
  GL_EXT_texture_storage \
  GL_EXT_texture_storage \
  GL_EXT_texture_view \
  GL_EXT_timer_query \
  GL_EXT_timer_query \
  GL_EXT_transform_feedback \
  GL_EXT_transform_feedback \
  GL_EXT_transform_feedback \
  GL_EXT_transform_feedback \
  GL_EXT_transform_feedback \
  GL_EXT_transform_feedback \
  GL_EXT_transform_feedback \
  GL_EXT_vertex_array \
  GL_EXT_vertex_array \
  GL_EXT_vertex_array \
  GL_EXT_vertex_array \
  GL_EXT_vertex_array \
  GL_EXT_vertex_array \
  GL_EXT_vertex_array \
  GL_EXT_vertex_array \
  GL_EXT_vertex_array_setXXX \
  GL_EXT_vertex_array_setXXX \
  GL_EXT_vertex_array_setXXX \
  GL_EXT_vertex_attrib_64bit \
  GL_EXT_vertex_attrib_64bit \
  GL_EXT_vertex_attrib_64bit \
  GL_EXT_vertex_attrib_64bit \
  GL_EXT_vertex_attrib_64bit \
  GL_EXT_vertex_attrib_64bit \
  GL_EXT_vertex_attrib_64bit \
  GL_EXT_vertex_attrib_64bit \
  GL_EXT_vertex_attrib_64bit \
  GL_EXT_vertex_attrib_64bit \
  GL_EXT_vertex_attrib_64bit \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_shader \
  GL_EXT_vertex_weighting \
  GL_EXT_vertex_weighting \
  GL_EXT_vertex_weighting \
  GL_EXT_win32_keyed_mutex \
  GL_EXT_win32_keyed_mutex \
  GL_EXT_window_rectangles \
  GL_EXT_x11_sync_object \
  GL_GREMEDY_frame_terminator \
  GL_GREMEDY_string_marker \
  GL_HP_image_transform \
  GL_HP_image_transform \
  GL_HP_image_transform \
  GL_HP_image_transform \
  GL_HP_image_transform \
  GL_HP_image_transform \
  GL_IBM_multimode_draw_arrays \
  GL_IBM_multimode_draw_arrays \
  GL_IBM_vertex_array_lists \
  GL_IBM_vertex_array_lists \
  GL_IBM_vertex_array_lists \
  GL_IBM_vertex_array_lists \
  GL_IBM_vertex_array_lists \
  GL_IBM_vertex_array_lists \
  GL_IBM_vertex_array_lists \
  GL_IBM_vertex_array_lists \
  GL_IMG_bindless_texture \
  GL_IMG_bindless_texture \
  GL_IMG_bindless_texture \
  GL_IMG_bindless_texture \
  GL_IMG_bindless_texture \
  GL_IMG_bindless_texture \
  GL_IMG_framebuffer_downsample \
  GL_IMG_framebuffer_downsample \
  GL_IMG_multisampled_render_to_texture \
  GL_IMG_multisampled_render_to_texture \
  GL_INTEL_map_texture \
  GL_INTEL_map_texture \
  GL_INTEL_map_texture \
  GL_INTEL_parallel_arrays \
  GL_INTEL_parallel_arrays \
  GL_INTEL_parallel_arrays \
  GL_INTEL_parallel_arrays \
  GL_INTEL_performance_query \
  GL_INTEL_performance_query \
  GL_INTEL_performance_query \
  GL_INTEL_performance_query \
  GL_INTEL_performance_query \
  GL_INTEL_performance_query \
  GL_INTEL_performance_query \
  GL_INTEL_performance_query \
  GL_INTEL_performance_query \
  GL_INTEL_performance_query \
  GL_INTEL_texture_scissor \
  GL_INTEL_texture_scissor \
  GL_KHR_blend_equation_advanced \
  GL_KHR_debug \
  GL_KHR_debug \
  GL_KHR_debug \
  GL_KHR_debug \
  GL_KHR_debug \
  GL_KHR_debug \
  GL_KHR_debug \
  GL_KHR_debug \
  GL_KHR_debug \
  GL_KHR_debug \
  GL_KHR_parallel_shader_compile \
  GL_KHR_robustness \
  GL_KHR_robustness \
  GL_KHR_robustness \
  GL_KHR_robustness \
  GL_KTX_buffer_region \
  GL_KTX_buffer_region \
  GL_KTX_buffer_region \
  GL_KTX_buffer_region \
  GL_KTX_buffer_region \
  GL_MESA_framebuffer_flip_y \
  GL_MESA_framebuffer_flip_y \
  GL_MESA_resize_buffers \
  GL_MESA_window_pos \
  GL_MESA_window_pos \
  GL_MESA_window_pos \
  GL_MESA_window_pos \
  GL_MESA_window_pos \
  GL_MESA_window_pos \
  GL_MESA_window_pos \
  GL_MESA_window_pos \
  GL_MESA_window_pos \
  GL_MESA_window_pos \
  GL_MESA_window_pos \
  GL_MESA_window_pos \
  GL_MESA_window_pos \
  GL_MESA_window_pos \
  GL_MESA_window_pos \
  GL_MESA_window_pos \
  GL_MESA_window_pos \
  GL_MESA_window_pos \
  GL_MESA_window_pos \
  GL_MESA_window_pos \
  GL_MESA_window_pos \
  GL_MESA_window_pos \
  GL_MESA_window_pos \
  GL_MESA_window_pos \
  GL_NVX_conditional_render \
  GL_NVX_conditional_render \
  GL_NVX_gpu_multicast2 \
  GL_NVX_gpu_multicast2 \
  GL_NVX_gpu_multicast2 \
  GL_NVX_gpu_multicast2 \
  GL_NVX_gpu_multicast2 \
  GL_NVX_gpu_multicast2 \
  GL_NVX_linked_gpu_multicast \
  GL_NVX_linked_gpu_multicast \
  GL_NVX_linked_gpu_multicast \
  GL_NVX_progress_fence \
  GL_NVX_progress_fence \
  GL_NVX_progress_fence \
  GL_NV_3dvision_settings \
  GL_NV_3dvision_settings \
  GL_NV_alpha_to_coverage_dither_control \
  GL_NV_bindless_multi_draw_indirect \
  GL_NV_bindless_multi_draw_indirect \
  GL_NV_bindless_multi_draw_indirect_count \
  GL_NV_bindless_multi_draw_indirect_count \
  GL_NV_bindless_texture \
  GL_NV_bindless_texture \
  GL_NV_bindless_texture \
  GL_NV_bindless_texture \
  GL_NV_bindless_texture \
  GL_NV_bindless_texture \
  GL_NV_bindless_texture \
  GL_NV_bindless_texture \
  GL_NV_bindless_texture \
  GL_NV_bindless_texture \
  GL_NV_bindless_texture \
  GL_NV_bindless_texture \
  GL_NV_bindless_texture \
  GL_NV_blend_equation_advanced \
  GL_NV_blend_equation_advanced \
  GL_NV_clip_space_w_scaling \
  GL_NV_command_list \
  GL_NV_command_list \
  GL_NV_command_list \
  GL_NV_command_list \
  GL_NV_command_list \
  GL_NV_command_list \
  GL_NV_command_list \
  GL_NV_command_list \
  GL_NV_command_list \
  GL_NV_command_list \
  GL_NV_command_list \
  GL_NV_command_list \
  GL_NV_command_list \
  GL_NV_command_list \
  GL_NV_command_list \
  GL_NV_command_list \
  GL_NV_command_list \
  GL_NV_conditional_render \
  GL_NV_conditional_render \
  GL_NV_conservative_raster \
  GL_NV_conservative_raster_dilate \
  GL_NV_conservative_raster_pre_snap_triangles \
  GL_NV_copy_buffer \
  GL_NV_copy_image \
  GL_NV_depth_buffer_float \
  GL_NV_depth_buffer_float \
  GL_NV_depth_buffer_float \
  GL_NV_draw_buffers \
  GL_NV_draw_instanced \
  GL_NV_draw_instanced \
  GL_NV_draw_texture \
  GL_NV_draw_vulkan_image \
  GL_NV_draw_vulkan_image \
  GL_NV_draw_vulkan_image \
  GL_NV_draw_vulkan_image \
  GL_NV_evaluators \
  GL_NV_evaluators \
  GL_NV_evaluators \
  GL_NV_evaluators \
  GL_NV_evaluators \
  GL_NV_evaluators \
  GL_NV_evaluators \
  GL_NV_evaluators \
  GL_NV_evaluators \
  GL_NV_explicit_multisample \
  GL_NV_explicit_multisample \
  GL_NV_explicit_multisample \
  GL_NV_fence \
  GL_NV_fence \
  GL_NV_fence \
  GL_NV_fence \
  GL_NV_fence \
  GL_NV_fence \
  GL_NV_fence \
  GL_NV_fragment_coverage_to_color \
  GL_NV_fragment_program \
  GL_NV_fragment_program \
  GL_NV_fragment_program \
  GL_NV_fragment_program \
  GL_NV_fragment_program \
  GL_NV_fragment_program \
  GL_NV_framebuffer_blit \
  GL_NV_framebuffer_multisample \
  GL_NV_framebuffer_multisample_coverage \
  GL_NV_geometry_program4 \
  GL_NV_gpu_multicast \
  GL_NV_gpu_multicast \
  GL_NV_gpu_multicast \
  GL_NV_gpu_multicast \
  GL_NV_gpu_multicast \
  GL_NV_gpu_multicast \
  GL_NV_gpu_multicast \
  GL_NV_gpu_multicast \
  GL_NV_gpu_multicast \
  GL_NV_gpu_multicast \
  GL_NV_gpu_multicast \
  GL_NV_gpu_multicast \
  GL_NV_gpu_program4 \
  GL_NV_gpu_program4 \
  GL_NV_gpu_program4 \
  GL_NV_gpu_program4 \
  GL_NV_gpu_program4 \
  GL_NV_gpu_program4 \
  GL_NV_gpu_program4 \
  GL_NV_gpu_program4 \
  GL_NV_gpu_program4 \
  GL_NV_gpu_program4 \
  GL_NV_gpu_program4 \
  GL_NV_gpu_program4 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_gpu_shader5 \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_half_float \
  GL_NV_instanced_arrays \
  GL_NV_internalformat_sample_query \
  GL_NV_memory_attachment \
  GL_NV_memory_attachment \
  GL_NV_memory_attachment \
  GL_NV_memory_attachment \
  GL_NV_memory_attachment \
  GL_NV_memory_attachment \
  GL_NV_mesh_shader \
  GL_NV_mesh_shader \
  GL_NV_mesh_shader \
  GL_NV_mesh_shader \
  GL_NV_non_square_matrices \
  GL_NV_non_square_matrices \
  GL_NV_non_square_matrices \
  GL_NV_non_square_matrices \
  GL_NV_non_square_matrices \
  GL_NV_non_square_matrices \
  GL_NV_occlusion_query \
  GL_NV_occlusion_query \
  GL_NV_occlusion_query \
  GL_NV_occlusion_query \
  GL_NV_occlusion_query \
  GL_NV_occlusion_query \
  GL_NV_occlusion_query \
  GL_NV_parameter_buffer_object \
  GL_NV_parameter_buffer_object \
  GL_NV_parameter_buffer_object \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_path_rendering \
  GL_NV_pixel_data_range \
  GL_NV_pixel_data_range \
  GL_NV_point_sprite \
  GL_NV_point_sprite \
  GL_NV_polygon_mode \
  GL_NV_present_video \
  GL_NV_present_video \
  GL_NV_present_video \
  GL_NV_present_video \
  GL_NV_present_video \
  GL_NV_present_video \
  GL_NV_primitive_restart \
  GL_NV_primitive_restart \
  GL_NV_read_buffer \
  GL_NV_register_combiners \
  GL_NV_register_combiners \
  GL_NV_register_combiners \
  GL_NV_register_combiners \
  GL_NV_register_combiners \
  GL_NV_register_combiners \
  GL_NV_register_combiners \
  GL_NV_register_combiners \
  GL_NV_register_combiners \
  GL_NV_register_combiners \
  GL_NV_register_combiners \
  GL_NV_register_combiners \
  GL_NV_register_combiners \
  GL_NV_register_combiners2 \
  GL_NV_register_combiners2 \
  GL_NV_sample_locations \
  GL_NV_sample_locations \
  GL_NV_sample_locations \
  GL_NV_scissor_exclusive \
  GL_NV_scissor_exclusive \
  GL_NV_shader_buffer_load \
  GL_NV_shader_buffer_load \
  GL_NV_shader_buffer_load \
  GL_NV_shader_buffer_load \
  GL_NV_shader_buffer_load \
  GL_NV_shader_buffer_load \
  GL_NV_shader_buffer_load \
  GL_NV_shader_buffer_load \
  GL_NV_shader_buffer_load \
  GL_NV_shader_buffer_load \
  GL_NV_shader_buffer_load \
  GL_NV_shader_buffer_load \
  GL_NV_shader_buffer_load \
  GL_NV_shading_rate_image \
  GL_NV_shading_rate_image \
  GL_NV_shading_rate_image \
  GL_NV_shading_rate_image \
  GL_NV_shading_rate_image \
  GL_NV_shading_rate_image \
  GL_NV_texture_array \
  GL_NV_texture_array \
  GL_NV_texture_array \
  GL_NV_texture_array \
  GL_NV_texture_array \
  GL_NV_texture_array \
  GL_NV_texture_barrier \
  GL_NV_texture_multisample \
  GL_NV_texture_multisample \
  GL_NV_texture_multisample \
  GL_NV_texture_multisample \
  GL_NV_texture_multisample \
  GL_NV_texture_multisample \
  GL_NV_transform_feedback \
  GL_NV_transform_feedback \
  GL_NV_transform_feedback \
  GL_NV_transform_feedback \
  GL_NV_transform_feedback \
  GL_NV_transform_feedback \
  GL_NV_transform_feedback \
  GL_NV_transform_feedback \
  GL_NV_transform_feedback \
  GL_NV_transform_feedback \
  GL_NV_transform_feedback \
  GL_NV_transform_feedback2 \
  GL_NV_transform_feedback2 \
  GL_NV_transform_feedback2 \
  GL_NV_transform_feedback2 \
  GL_NV_transform_feedback2 \
  GL_NV_transform_feedback2 \
  GL_NV_transform_feedback2 \
  GL_NV_vdpau_interop \
  GL_NV_vdpau_interop \
  GL_NV_vdpau_interop \
  GL_NV_vdpau_interop \
  GL_NV_vdpau_interop \
  GL_NV_vdpau_interop \
  GL_NV_vdpau_interop \
  GL_NV_vdpau_interop \
  GL_NV_vdpau_interop \
  GL_NV_vdpau_interop \
  GL_NV_vdpau_interop2 \
  GL_NV_vertex_array_range \
  GL_NV_vertex_array_range \
  GL_NV_vertex_attrib_integer_64bit \
  GL_NV_vertex_attrib_integer_64bit \
  GL_NV_vertex_attrib_integer_64bit \
  GL_NV_vertex_attrib_integer_64bit \
  GL_NV_vertex_attrib_integer_64bit \
  GL_NV_vertex_attrib_integer_64bit \
  GL_NV_vertex_attrib_integer_64bit \
  GL_NV_vertex_attrib_integer_64bit \
  GL_NV_vertex_attrib_integer_64bit \
  GL_NV_vertex_attrib_integer_64bit \
  GL_NV_vertex_attrib_integer_64bit \
  GL_NV_vertex_attrib_integer_64bit \
  GL_NV_vertex_attrib_integer_64bit \
  GL_NV_vertex_attrib_integer_64bit \
  GL_NV_vertex_attrib_integer_64bit \
  GL_NV_vertex_attrib_integer_64bit \
  GL_NV_vertex_attrib_integer_64bit \
  GL_NV_vertex_attrib_integer_64bit \
  GL_NV_vertex_attrib_integer_64bit \
  GL_NV_vertex_buffer_unified_memory \
  GL_NV_vertex_buffer_unified_memory \
  GL_NV_vertex_buffer_unified_memory \
  GL_NV_vertex_buffer_unified_memory \
  GL_NV_vertex_buffer_unified_memory \
  GL_NV_vertex_buffer_unified_memory \
  GL_NV_vertex_buffer_unified_memory \
  GL_NV_vertex_buffer_unified_memory \
  GL_NV_vertex_buffer_unified_memory \
  GL_NV_vertex_buffer_unified_memory \
  GL_NV_vertex_buffer_unified_memory \
  GL_NV_vertex_buffer_unified_memory \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_vertex_program \
  GL_NV_video_capture \
  GL_NV_video_capture \
  GL_NV_video_capture \
  GL_NV_video_capture \
  GL_NV_video_capture \
  GL_NV_video_capture \
  GL_NV_video_capture \
  GL_NV_video_capture \
  GL_NV_video_capture \
  GL_NV_video_capture \
  GL_NV_video_capture \
  GL_NV_video_capture \
  GL_NV_viewport_array \
  GL_NV_viewport_array \
  GL_NV_viewport_array \
  GL_NV_viewport_array \
  GL_NV_viewport_array \
  GL_NV_viewport_array \
  GL_NV_viewport_array \
  GL_NV_viewport_array \
  GL_NV_viewport_array \
  GL_NV_viewport_array \
  GL_NV_viewport_array \
  GL_NV_viewport_array \
  GL_NV_viewport_swizzle \
  GL_OES_EGL_image \
  GL_OES_EGL_image \
  GL_OES_blend_equation_separate \
  GL_OES_blend_func_separate \
  GL_OES_blend_subtract \
  GL_OES_copy_image \
  GL_OES_draw_buffers_indexed \
  GL_OES_draw_buffers_indexed \
  GL_OES_draw_buffers_indexed \
  GL_OES_draw_buffers_indexed \
  GL_OES_draw_buffers_indexed \
  GL_OES_draw_buffers_indexed \
  GL_OES_draw_buffers_indexed \
  GL_OES_draw_buffers_indexed \
  GL_OES_framebuffer_object \
  GL_OES_framebuffer_object \
  GL_OES_framebuffer_object \
  GL_OES_framebuffer_object \
  GL_OES_framebuffer_object \
  GL_OES_framebuffer_object \
  GL_OES_framebuffer_object \
  GL_OES_framebuffer_object \
  GL_OES_framebuffer_object \
  GL_OES_framebuffer_object \
  GL_OES_framebuffer_object \
  GL_OES_framebuffer_object \
  GL_OES_framebuffer_object \
  GL_OES_framebuffer_object \
  GL_OES_framebuffer_object \
  GL_OES_get_program_binary \
  GL_OES_get_program_binary \
  GL_OES_mapbuffer \
  GL_OES_mapbuffer \
  GL_OES_mapbuffer \
  GL_OES_matrix_palette \
  GL_OES_matrix_palette \
  GL_OES_matrix_palette \
  GL_OES_sample_shading \
  GL_OES_single_precision \
  GL_OES_single_precision \
  GL_OES_single_precision \
  GL_OES_single_precision \
  GL_OES_single_precision \
  GL_OES_single_precision \
  GL_OES_texture_3D \
  GL_OES_texture_3D \
  GL_OES_texture_3D \
  GL_OES_texture_3D \
  GL_OES_texture_3D \
  GL_OES_texture_3D \
  GL_OES_texture_border_clamp \
  GL_OES_texture_border_clamp \
  GL_OES_texture_border_clamp \
  GL_OES_texture_border_clamp \
  GL_OES_texture_border_clamp \
  GL_OES_texture_border_clamp \
  GL_OES_texture_border_clamp \
  GL_OES_texture_border_clamp \
  GL_OES_texture_buffer \
  GL_OES_texture_buffer \
  GL_OES_texture_cube_map \
  GL_OES_texture_cube_map \
  GL_OES_texture_cube_map \
  GL_OES_texture_cube_map \
  GL_OES_texture_cube_map \
  GL_OES_texture_cube_map \
  GL_OES_texture_cube_map \
  GL_OES_texture_cube_map \
  GL_OES_texture_cube_map \
  GL_OES_texture_storage_multisample_2d_array \
  GL_OES_texture_view \
  GL_OES_vertex_array_object \
  GL_OES_vertex_array_object \
  GL_OES_vertex_array_object \
  GL_OES_vertex_array_object \
  GL_OVR_multiview \
  GL_OVR_multiview \
  GL_OVR_multiview_multisampled_render_to_texture \
  GL_QCOM_alpha_test \
  GL_QCOM_driver_control \
  GL_QCOM_driver_control \
  GL_QCOM_driver_control \
  GL_QCOM_driver_control \
  GL_QCOM_extended_get \
  GL_QCOM_extended_get \
  GL_QCOM_extended_get \
  GL_QCOM_extended_get \
  GL_QCOM_extended_get \
  GL_QCOM_extended_get \
  GL_QCOM_extended_get \
  GL_QCOM_extended_get \
  GL_QCOM_extended_get2 \
  GL_QCOM_extended_get2 \
  GL_QCOM_extended_get2 \
  GL_QCOM_extended_get2 \
  GL_QCOM_framebuffer_foveated \
  GL_QCOM_framebuffer_foveated \
  GL_QCOM_shader_framebuffer_fetch_noncoherent \
  GL_QCOM_texture_foveated \
  GL_QCOM_tiled_rendering \
  GL_QCOM_tiled_rendering \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_0_compatibility \
  GL_REGAL_ES1_1_compatibility \
  GL_REGAL_ES1_1_compatibility \
  GL_REGAL_ES1_1_compatibility \
  GL_REGAL_ES1_1_compatibility \
  GL_REGAL_ES1_1_compatibility \
  GL_REGAL_ES1_1_compatibility \
  GL_REGAL_ES1_1_compatibility \
  GL_REGAL_ES1_1_compatibility \
  GL_REGAL_ES1_1_compatibility \
  GL_REGAL_ES1_1_compatibility \
  GL_REGAL_ES1_1_compatibility \
  GL_REGAL_ES1_1_compatibility \
  GL_REGAL_ES1_1_compatibility \
  GL_REGAL_error_string \
  GL_REGAL_extension_query \
  GL_REGAL_extension_query \
  GL_REGAL_log \
  GL_SGIS_detail_texture \
  GL_SGIS_detail_texture \
  GL_SGIS_fog_function \
  GL_SGIS_fog_function \
  GL_SGIS_multisample \
  GL_SGIS_multisample \
  GL_SGIS_multitexture \
  GL_SGIS_multitexture \
  GL_SGIS_multitexture \
  GL_SGIS_multitexture \
  GL_SGIS_shared_multisample \
  GL_SGIS_sharpen_texture \
  GL_SGIS_sharpen_texture \
  GL_SGIS_texture4D \
  GL_SGIS_texture4D \
  GL_SGIS_texture_filter4 \
  GL_SGIS_texture_filter4 \
  GL_SGIX_async \
  GL_SGIX_async \
  GL_SGIX_async \
  GL_SGIX_async \
  GL_SGIX_async \
  GL_SGIX_async \
  GL_SGIX_datapipe \
  GL_SGIX_datapipe \
  GL_SGIX_flush_raster \
  GL_SGIX_fog_layers \
  GL_SGIX_fog_layers \
  GL_SGIX_fog_texture \
  GL_SGIX_fragment_specular_lighting \
  GL_SGIX_fragment_specular_lighting \
  GL_SGIX_fragment_specular_lighting \
  GL_SGIX_fragment_specular_lighting \
  GL_SGIX_fragment_specular_lighting \
  GL_SGIX_fragment_specular_lighting \
  GL_SGIX_fragment_specular_lighting \
  GL_SGIX_fragment_specular_lighting \
  GL_SGIX_fragment_specular_lighting \
  GL_SGIX_fragment_specular_lighting \
  GL_SGIX_fragment_specular_lighting \
  GL_SGIX_fragment_specular_lighting \
  GL_SGIX_fragment_specular_lighting \
  GL_SGIX_fragment_specular_lighting \
  GL_SGIX_fragment_specular_lighting \
  GL_SGIX_fragment_specular_lighting \
  GL_SGIX_fragment_specular_lighting \
  GL_SGIX_framezoom \
  GL_SGIX_igloo_interface \
  GL_SGIX_mpeg1 \
  GL_SGIX_mpeg1 \
  GL_SGIX_mpeg1 \
  GL_SGIX_mpeg1 \
  GL_SGIX_mpeg1 \
  GL_SGIX_mpeg1 \
  GL_SGIX_mpeg1 \
  GL_SGIX_mpeg1 \
  GL_SGIX_mpeg1 \
  GL_SGIX_mpeg1 \
  GL_SGIX_mpeg1 \
  GL_SGIX_nonlinear_lighting_pervertex \
  GL_SGIX_nonlinear_lighting_pervertex \
  GL_SGIX_nonlinear_lighting_pervertex \
  GL_SGIX_nonlinear_lighting_pervertex \
  GL_SGIX_pixel_texture \
  GL_SGIX_polynomial_ffd \
  GL_SGIX_polynomial_ffd \
  GL_SGIX_quad_mesh \
  GL_SGIX_quad_mesh \
  GL_SGIX_reference_plane \
  GL_SGIX_sprite \
  GL_SGIX_sprite \
  GL_SGIX_sprite \
  GL_SGIX_sprite \
  GL_SGIX_tag_sample_buffer \
  GL_SGIX_vector_ops \
  GL_SGIX_vector_ops \
  GL_SGIX_vertex_array_object \
  GL_SGIX_vertex_array_object \
  GL_SGIX_vertex_array_object \
  GL_SGIX_vertex_array_object \
  GL_SGIX_vertex_array_object \
  GL_SGIX_vertex_array_object \
  GL_SGI_color_table \
  GL_SGI_color_table \
  GL_SGI_color_table \
  GL_SGI_color_table \
  GL_SGI_color_table \
  GL_SGI_color_table \
  GL_SGI_color_table \
  GL_SGI_fft \
  GL_SGI_fft \
  GL_SGI_fft \
  GL_SGI_fft \
  GL_SGI_fft \
  GL_SGI_fft \
  GL_SGI_fft \
  GL_SUNX_constant_data \
  GL_SUN_global_alpha \
  GL_SUN_global_alpha \
  GL_SUN_global_alpha \
  GL_SUN_global_alpha \
  GL_SUN_global_alpha \
  GL_SUN_global_alpha \
  GL_SUN_global_alpha \
  GL_SUN_global_alpha \
  GL_SUN_read_video_pixels \
  GL_SUN_triangle_list \
  GL_SUN_triangle_list \
  GL_SUN_triangle_list \
  GL_SUN_triangle_list \
  GL_SUN_triangle_list \
  GL_SUN_triangle_list \
  GL_SUN_triangle_list \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_SUN_vertex \
  GL_WIN_swap_hint \
]

# List of the deprecation version of all wrapped OpenGL functions.
set ::__tcl3dOglFuncDeprecatedList [list \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  0.0 \
  3.1 \
  0.0 \
  3.1 \
  3.1 \
  0.0 \
  3.1 \
  0.0 \
  0.0 \
  3.1 \
  0.0 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  0.0 \
  3.1 \
  3.1 \
  3.1 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  3.1 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  3.1 \
  0.0 \
  0.0 \
  0.0 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  0.0 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  0.0 \
  0.0 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  0.0 \
  3.1 \
  3.1 \
  0.0 \
  0.0 \
  3.1 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  0.0 \
  3.1 \
  0.0 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  0.0 \
  0.0 \
  3.1 \
  3.1 \
  0.0 \
  3.1 \
  0.0 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  0.0 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  0.0 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  0.0 \
  0.0 \
  3.1 \
  3.1 \
  3.1 \
  0.0 \
  0.0 \
  0.0 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  0.0 \
  0.0 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  0.0 \
  3.1 \
  3.1 \
  0.0 \
  0.0 \
  0.0 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  3.1 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  3.1 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
  0.0 \
]

# List of the reference URLs of all wrapped OpenGL functions.
set ::__tcl3dOglFuncUrlList [list \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glAccum.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glAlphaFunc.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glAreTexturesResident.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glArrayElement.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glBegin.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBindTexture.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glBitmap.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBlendFunc.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glCallList.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glCallLists.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glClear.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glClearAccum.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glClearColor.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glClearDepth.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glClearIndex.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glClearStencil.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glClipPlane.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColor4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glColorMask.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColorMaterial.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColorPointer.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glCopyPixels.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glCopyTexImage1D.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glCopyTexImage2D.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glCopyTexSubImage1D.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glCopyTexSubImage2D.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glCullFace.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glDeleteLists.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDeleteTextures.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDepthFunc.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDepthMask.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDepthRange.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glDisable.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glDisableClientState.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDrawArrays.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDrawBuffer.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDrawElements.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glDrawPixels.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glEdgeFlag.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glEdgeFlagPointer.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glEdgeFlag.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glEnable.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glEnableClientState.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glEnd.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glEndList.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glEvalCoord1.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glEvalCoord1.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glEvalCoord1.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glEvalCoord1.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glEvalCoord2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glEvalCoord2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glEvalCoord2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glEvalCoord2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glEvalMesh1.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glEvalMesh2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glEvalPoint1.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glEvalPoint2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glFeedbackBuffer.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glFinish.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glFlush.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glFog.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glFog.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glFog.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glFog.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glFrontFace.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glFrustum.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGenLists.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGenTextures.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGet.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetClipPlane.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetDoublev.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetError.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetFloatv.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetIntegerv.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetLight.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetLight.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetMap.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetMap.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetMap.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetMaterial.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetMaterial.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetPixelMap.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetPixelMap.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetPixelMap.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetPointerv.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetPolygonStipple.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetString.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetTexEnv.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetTexEnv.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetTexGen.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetTexGen.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetTexGen.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetTexImage.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetTexLevelParameter.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetTexLevelParameter.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetTexParameter.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetTexParameter.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glHint.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glIndexMask.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glIndexPointer.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glIndex.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glIndex.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glIndex.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glIndex.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glIndex.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glIndex.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glIndex.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glIndex.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glIndex.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glIndex.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glInitNames.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glInterleavedArrays.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glIsEnabled.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glIsList.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glIsTexture.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glLightModel.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glLightModel.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glLightModel.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glLightModel.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glLight.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glLight.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glLight.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glLight.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glLineStipple.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glLineWidth.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glListBase.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glLoadIdentity.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glLoadMatrix.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glLoadMatrix.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glLoadName.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glLogicOp.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMap1.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMap1.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMap2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMap2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMapGrid1.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMapGrid1.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMapGrid2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMapGrid2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMaterial.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMaterial.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMaterial.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMaterial.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMatrixMode.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultMatrix.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultMatrix.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glNewList.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glNormal.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glNormal.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glNormal.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glNormal.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glNormal.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glNormal.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glNormal.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glNormal.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glNormal.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glNormal.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glNormalPointer.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glOrtho.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glPassThrough.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glPixelMap.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glPixelMap.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glPixelMap.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glPixelStore.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glPixelStore.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glPixelTransfer.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glPixelTransfer.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glPixelZoom.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glPointSize.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glPolygonMode.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glPolygonOffset.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glPolygonStipple.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glPopAttrib.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glPopClientAttrib.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glPopMatrix.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glPopName.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glPrioritizeTextures.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glPushAttrib.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glPushClientAttrib.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glPushMatrix.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glPushName.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRasterPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRasterPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRasterPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRasterPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRasterPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRasterPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRasterPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRasterPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRasterPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRasterPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRasterPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRasterPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRasterPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRasterPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRasterPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRasterPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRasterPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRasterPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRasterPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRasterPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRasterPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRasterPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRasterPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRasterPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glReadBuffer.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glReadPixels.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRect.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRect.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRect.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRect.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRect.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRect.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRect.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRect.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRenderMode.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRotate.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glRotate.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glScale.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glScale.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glScissor.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glSelectBuffer.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glShadeModel.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glStencilFunc.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glStencilMask.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glStencilOp.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord1.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord1.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord1.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord1.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord1.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord1.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord1.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord1.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord3.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord3.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord3.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord3.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord3.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord3.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord3.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord3.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoord4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexCoordPointer.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexEnv.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexEnv.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexEnv.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexEnv.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexGen.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexGen.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexGen.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexGen.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexGen.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTexGen.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glTexImage1D.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glTexImage2D.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glTexParameter.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glTexParameter.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glTexParameter.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glTexParameter.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glTexSubImage1D.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glTexSubImage2D.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTranslate.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glTranslate.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertex2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertex2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertex2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertex2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertex2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertex2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertex2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertex2.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertex3.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertex3.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertex3.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertex3.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertex3.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertex3.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertex3.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertex3.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertex4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertex4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertex4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertex4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertex4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertex4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertex4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertex4.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glVertexPointer.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glViewport.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glCopyTexSubImage3D.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDrawRangeElements.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glTexImage3D.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glTexSubImage3D.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glActiveTexture.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glClientActiveTexture.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glCompressedTexImage1D.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glCompressedTexImage2D.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glCompressedTexImage3D.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glCompressedTexSubImage1D.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glCompressedTexSubImage2D.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glCompressedTexSubImage3D.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetCompressedTexImage.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glLoadTransposeMatrix.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glLoadTransposeMatrix.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultTransposeMatrix.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultTransposeMatrix.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMultiTexCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glSampleCoverage.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBlendColor.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBlendEquation.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBlendFuncSeparate.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glFogCoordPointer.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glFogCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glFogCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glFogCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glFogCoord.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glMultiDrawArrays.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glMultiDrawElements.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glPointParameter.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glPointParameter.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glPointParameter.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glPointParameter.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glSecondaryColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glSecondaryColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glSecondaryColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glSecondaryColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glSecondaryColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glSecondaryColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glSecondaryColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glSecondaryColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glSecondaryColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glSecondaryColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glSecondaryColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glSecondaryColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glSecondaryColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glSecondaryColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glSecondaryColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glSecondaryColor.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glSecondaryColorPointer.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glWindowPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glWindowPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glWindowPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glWindowPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glWindowPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glWindowPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glWindowPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glWindowPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glWindowPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glWindowPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glWindowPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glWindowPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glWindowPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glWindowPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glWindowPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glWindowPos.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBeginQuery.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBindBuffer.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBufferData.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBufferSubData.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDeleteBuffers.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDeleteQueries.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBeginQuery.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGenBuffers.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGenQueries.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetBufferParameteriv.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetBufferPointerv.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetBufferSubData.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetQueryObject.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetQueryObject.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetQueryiv.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glIsBuffer.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glIsQuery.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glMapBuffer.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUnmapBuffer.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glAttachShader.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBindAttribLocation.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBlendEquationSeparate.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glCompileShader.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glCreateProgram.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glCreateShader.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDeleteProgram.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDeleteShader.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDetachShader.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glDisableVertexAttribArray.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDrawBuffers.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glEnableVertexAttribArray.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetActiveAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetActiveUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetAttachedShaders.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetAttribLocation.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetProgramInfoLog.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetProgram.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetShaderInfoLog.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetShaderSource.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetShader.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetUniformLocation.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetVertexAttribPointerv.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glIsProgram.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glIsShader.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glLinkProgram.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glShaderSource.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glStencilFuncSeparate.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glStencilMaskSeparate.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glStencilOpSeparate.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUseProgram.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glValidateProgram.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttribPointer.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBeginConditionalRender.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBeginTransformFeedback.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBindFragDataLocation.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glClampColor.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glClearBuffer.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glClearBuffer.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glClearBuffer.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glClearBuffer.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glColorMask.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glDisable.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glEnable.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBeginConditionalRender.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBeginTransformFeedback.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGet.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetFragDataLocation.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetString.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetTexParameter.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetTexParameter.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetTransformFeedbackVarying.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glIsEnabled.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glTexParameter.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glTexParameter.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glTransformFeedbackVaryings.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glUniform.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttrib.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDrawArraysInstanced.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glDrawElementsInstanced.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glPrimitiveRestartIndex.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glTexBuffer.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glFramebufferTexture.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetBufferParameter.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGet.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glVertexAttribDivisor.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBlendEquationSeparate.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBlendEquation.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBlendFuncSeparate.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glBlendFunc.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glMinSampleShading.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGetGraphicsResetStatus.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGet.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGet.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glGet.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glMultiDrawArraysIndirect.xhtml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl4/html/glMultiDrawElementsIndirect.xhtml" \
  "http://www.google.com/search?q=glSpecializeShader" \
  "https://www.khronos.org/registry/OpenGL/extensions/3DFX/3DFX_tbuffer.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_debug_output.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_debug_output.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_debug_output.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_debug_output.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_draw_buffers_blend.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_draw_buffers_blend.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_draw_buffers_blend.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_draw_buffers_blend.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_framebuffer_multisample_advanced.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_framebuffer_multisample_advanced.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_framebuffer_sample_positions.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_framebuffer_sample_positions.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_framebuffer_sample_positions.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_framebuffer_sample_positions.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_interleaved_elements.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_multi_draw_indirect.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_multi_draw_indirect.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_name_gen_delete.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_name_gen_delete.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_name_gen_delete.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_occlusion_query_event.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_performance_monitor.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_performance_monitor.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_performance_monitor.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_performance_monitor.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_performance_monitor.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_performance_monitor.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_performance_monitor.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_performance_monitor.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_performance_monitor.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_performance_monitor.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_performance_monitor.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_sample_positions.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_sparse_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_sparse_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_stencil_operation_extended.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_vertex_shader_tessellator.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_vertex_shader_tessellator.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ANGLE/ANGLE_framebuffer_blit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ANGLE/ANGLE_framebuffer_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ANGLE/ANGLE_instanced_arrays.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ANGLE/ANGLE_instanced_arrays.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ANGLE/ANGLE_instanced_arrays.txt" \
  "http://www.google.com/search?q=glBeginQueryANGLE" \
  "http://www.google.com/search?q=glDeleteQueriesANGLE" \
  "http://www.google.com/search?q=glEndQueryANGLE" \
  "http://www.google.com/search?q=glGenQueriesANGLE" \
  "http://www.google.com/search?q=glGetQueryObjecti64vANGLE" \
  "http://www.google.com/search?q=glGetQueryObjectivANGLE" \
  "http://www.google.com/search?q=glGetQueryObjectui64vANGLE" \
  "http://www.google.com/search?q=glGetQueryObjectuivANGLE" \
  "http://www.google.com/search?q=glGetQueryivANGLE" \
  "http://www.google.com/search?q=glIsQueryANGLE" \
  "http://www.google.com/search?q=glQueryCounterANGLE" \
  "https://www.khronos.org/registry/OpenGL/extensions/ANGLE/ANGLE_translated_shader_source.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_copy_texture_levels.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_element_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_element_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_element_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_element_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_element_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_fence.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_fence.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_fence.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_fence.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_fence.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_fence.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_fence.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_fence.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_flush_buffer_range.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_flush_buffer_range.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_framebuffer_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_framebuffer_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_object_purgeable.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_object_purgeable.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_object_purgeable.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_sync.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_sync.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_sync.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_sync.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_sync.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_sync.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_sync.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_texture_range.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_texture_range.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_vertex_array_range.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_vertex_array_range.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_vertex_array_range.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_vertex_program_evaluators.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_vertex_program_evaluators.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_vertex_program_evaluators.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_vertex_program_evaluators.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_vertex_program_evaluators.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_vertex_program_evaluators.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_vertex_program_evaluators.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_ES2_compatibility.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_ES2_compatibility.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_ES2_compatibility.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_ES2_compatibility.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_ES2_compatibility.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_ES3_1_compatibility.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_ES3_2_compatibility.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_base_instance.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_base_instance.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_base_instance.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_blend_func_extended.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_blend_func_extended.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_buffer_storage.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_cl_event.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_clear_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_clear_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_clear_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_clear_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_clear_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_clear_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_clip_control.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_color_buffer_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_compute_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_compute_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_compute_variable_group_size.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_copy_buffer.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_copy_image.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_debug_output.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_debug_output.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_debug_output.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_debug_output.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_draw_buffers.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_draw_buffers_blend.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_draw_buffers_blend.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_draw_buffers_blend.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_draw_buffers_blend.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_draw_elements_base_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_draw_elements_base_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_draw_elements_base_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_draw_elements_base_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_draw_indirect.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_draw_indirect.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_no_attachments.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_no_attachments.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_no_attachments.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_no_attachments.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_geometry_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_geometry_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_geometry_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_geometry_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_get_program_binary.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_get_program_binary.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_get_program_binary.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_get_texture_sub_image.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_get_texture_sub_image.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gl_spirv.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_fp64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_fp64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_fp64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_fp64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_fp64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_fp64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_fp64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_fp64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_fp64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_fp64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_fp64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_fp64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_fp64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_fp64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_fp64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_fp64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_fp64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_fp64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColorSubTable.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColorTable.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColorTableParameter.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glColorTableParameter.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glConvolutionFilter1D.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glConvolutionFilter2D.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glConvolutionParameter.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glConvolutionParameter.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glConvolutionParameter.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glConvolutionParameter.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glCopyColorSubTable.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glCopyColorTable.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glCopyConvolutionFilter1D.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glCopyConvolutionFilter2D.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetColorTable.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetColorTableParameter.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetColorTableParameter.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetConvolutionFilter.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetConvolutionParameter.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetConvolutionParameter.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetHistogram.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetHistogramParameter.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetHistogramParameter.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetMinmax.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetMinmaxParameter.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetMinmaxParameter.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glGetSeparableFilter.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glHistogram.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glMinmax.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glResetHistogram.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glResetMinmax.xml" \
  "https://www.khronos.org/registry/OpenGL-Refpages/gl2.1/xhtml/glSeparableFilter2D.xml" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_indirect_parameters.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_indirect_parameters.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_instanced_arrays.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_instanced_arrays.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_instanced_arrays.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_internalformat_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_internalformat_query2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_invalidate_subdata.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_invalidate_subdata.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_invalidate_subdata.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_invalidate_subdata.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_invalidate_subdata.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_invalidate_subdata.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_map_buffer_range.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_map_buffer_range.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_matrix_palette.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_matrix_palette.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_matrix_palette.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_matrix_palette.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_matrix_palette.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multi_bind.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multi_bind.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multi_bind.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multi_bind.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multi_bind.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multi_bind.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multi_draw_indirect.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multi_draw_indirect.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_occlusion_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_occlusion_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_occlusion_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_occlusion_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_occlusion_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_occlusion_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_occlusion_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_occlusion_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_parallel_shader_compile.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_point_parameters.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_point_parameters.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_polygon_offset_clamp.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_program_interface_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_program_interface_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_program_interface_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_program_interface_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_program_interface_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_program_interface_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_provoking_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sample_locations.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sample_locations.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sample_shading.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sampler_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sampler_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sampler_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sampler_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sampler_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sampler_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sampler_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sampler_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sampler_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sampler_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sampler_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sampler_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sampler_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sampler_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_atomic_counters.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_image_load_store.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_image_load_store.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_storage_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_subroutine.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_subroutine.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_subroutine.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_subroutine.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_subroutine.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_subroutine.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_subroutine.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_subroutine.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shading_language_include.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shading_language_include.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shading_language_include.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shading_language_include.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shading_language_include.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shading_language_include.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sparse_buffer.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sparse_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sync.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sync.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sync.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sync.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sync.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sync.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sync.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_tessellation_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_tessellation_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_barrier.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_buffer_range.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_buffer_range.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_compression.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_compression.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_compression.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_compression.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_compression.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_compression.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_compression.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_storage.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_storage.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_storage.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_storage_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_storage_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_storage_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_storage_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_view.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_timer_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_timer_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_timer_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_transform_feedback2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_transform_feedback2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_transform_feedback2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_transform_feedback2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_transform_feedback2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_transform_feedback2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_transform_feedback2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_transform_feedback3.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_transform_feedback3.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_transform_feedback3.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_transform_feedback3.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_transform_feedback_instanced.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_transform_feedback_instanced.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_transpose_matrix.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_transpose_matrix.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_transpose_matrix.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_transpose_matrix.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_uniform_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_uniform_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_uniform_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_uniform_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_uniform_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_uniform_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_uniform_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_uniform_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_uniform_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_uniform_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_attrib_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_attrib_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_attrib_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_attrib_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_attrib_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_attrib_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_attrib_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_attrib_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_attrib_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_attrib_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_attrib_binding.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_attrib_binding.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_attrib_binding.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_attrib_binding.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_attrib_binding.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_attrib_binding.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_attrib_binding.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_attrib_binding.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_attrib_binding.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_attrib_binding.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_attrib_binding.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_attrib_binding.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_blend.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_blend.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_blend.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_blend.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_blend.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_blend.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_blend.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_blend.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_blend.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_blend.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_viewport_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_viewport_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_viewport_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_viewport_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_viewport_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_viewport_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_viewport_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_viewport_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_viewport_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_viewport_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_draw_buffers.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_element_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_element_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_element_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_envmap_bumpmap.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_envmap_bumpmap.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_envmap_bumpmap.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_envmap_bumpmap.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_fragment_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_fragment_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_fragment_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_fragment_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_fragment_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_fragment_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_fragment_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_fragment_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_fragment_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_fragment_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_fragment_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_fragment_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_fragment_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_fragment_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_map_object_buffer.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_map_object_buffer.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_pn_triangles.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_pn_triangles.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_separate_stencil.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_separate_stencil.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_attrib_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_attrib_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_attrib_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_EGL_image_storage.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_EGL_image_storage.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_base_instance.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_base_instance.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_base_instance.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_bindable_uniform.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_bindable_uniform.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_bindable_uniform.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_blend_color.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_blend_equation_separate.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_blend_func_extended.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_blend_func_extended.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_blend_func_extended.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_blend_func_separate.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_blend_minmax.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_buffer_storage.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_buffer_storage.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_clear_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_clear_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_clip_control.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_color_subtable.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_color_subtable.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_compiled_vertex_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_compiled_vertex_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_convolution.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_convolution.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_convolution.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_convolution.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_convolution.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_convolution.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_convolution.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_convolution.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_convolution.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_convolution.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_convolution.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_convolution.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_convolution.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_coordinate_frame.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_coordinate_frame.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_copy_image.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_copy_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_copy_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_copy_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_copy_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_copy_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_cull_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_cull_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_debug_label.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_debug_label.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_debug_marker.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_debug_marker.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_debug_marker.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_depth_bounds_test.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_discard_framebuffer.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_disjoint_timer_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_disjoint_timer_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_disjoint_timer_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_disjoint_timer_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_disjoint_timer_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_disjoint_timer_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_disjoint_timer_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_disjoint_timer_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_disjoint_timer_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_disjoint_timer_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_buffers.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_buffers2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_buffers2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_buffers2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_buffers2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_buffers2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_buffers2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_buffers_indexed.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_buffers_indexed.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_buffers_indexed.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_buffers_indexed.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_buffers_indexed.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_buffers_indexed.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_buffers_indexed.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_buffers_indexed.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_elements_base_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_elements_base_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_elements_base_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_elements_base_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_instanced.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_instanced.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_range_elements.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_transform_feedback.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_transform_feedback.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_external_buffer.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_external_buffer.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fog_coord.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fog_coord.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fog_coord.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fog_coord.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fog_coord.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fragment_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fragment_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fragment_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fragment_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fragment_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fragment_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fragment_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fragment_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fragment_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fragment_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fragment_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fragment_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fragment_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fragment_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fragment_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fragment_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fragment_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fragment_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_blit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_geometry_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_geometry_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_geometry_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_program_parameters.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_program_parameters.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_histogram.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_histogram.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_histogram.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_histogram.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_histogram.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_histogram.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_histogram.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_histogram.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_histogram.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_histogram.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_index_func.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_index_material.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_instanced_arrays.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_light_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_light_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_light_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_map_buffer_range.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_map_buffer_range.txt" \
  "http://www.google.com/search?q=glBufferStorageMemEXT" \
  "http://www.google.com/search?q=glCreateMemoryObjectsEXT" \
  "http://www.google.com/search?q=glDeleteMemoryObjectsEXT" \
  "http://www.google.com/search?q=glGetMemoryObjectParameterivEXT" \
  "http://www.google.com/search?q=glGetUnsignedBytei_vEXT" \
  "http://www.google.com/search?q=glGetUnsignedBytevEXT" \
  "http://www.google.com/search?q=glIsMemoryObjectEXT" \
  "http://www.google.com/search?q=glMemoryObjectParameterivEXT" \
  "http://www.google.com/search?q=glNamedBufferStorageMemEXT" \
  "http://www.google.com/search?q=glTexStorageMem1DEXT" \
  "http://www.google.com/search?q=glTexStorageMem2DEXT" \
  "http://www.google.com/search?q=glTexStorageMem2DMultisampleEXT" \
  "http://www.google.com/search?q=glTexStorageMem3DEXT" \
  "http://www.google.com/search?q=glTexStorageMem3DMultisampleEXT" \
  "http://www.google.com/search?q=glTextureStorageMem1DEXT" \
  "http://www.google.com/search?q=glTextureStorageMem2DEXT" \
  "http://www.google.com/search?q=glTextureStorageMem2DMultisampleEXT" \
  "http://www.google.com/search?q=glTextureStorageMem3DEXT" \
  "http://www.google.com/search?q=glTextureStorageMem3DMultisampleEXT" \
  "http://www.google.com/search?q=glImportMemoryFdEXT" \
  "http://www.google.com/search?q=glImportMemoryWin32HandleEXT" \
  "http://www.google.com/search?q=glImportMemoryWin32NameEXT" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_multi_draw_arrays.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_multi_draw_arrays.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_multi_draw_indirect.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_multi_draw_indirect.txt" \
  "http://www.google.com/search?q=glSampleMaskEXT" \
  "http://www.google.com/search?q=glSamplePatternEXT" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_multisampled_render_to_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_multiview_draw_buffers.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_multiview_draw_buffers.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_multiview_draw_buffers.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_paletted_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_paletted_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_paletted_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_paletted_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_pixel_transform.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_pixel_transform.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_pixel_transform.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_pixel_transform.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_pixel_transform.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_pixel_transform.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_point_parameters.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_point_parameters.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_polygon_offset.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_polygon_offset_clamp.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_primitive_bounding_box.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_provoking_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_raster_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_raster_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_raster_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_raster_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_scene_marker.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_scene_marker.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_secondary_color.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_secondary_color.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_secondary_color.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_secondary_color.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_secondary_color.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_secondary_color.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_secondary_color.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_secondary_color.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_secondary_color.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_secondary_color.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_secondary_color.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_secondary_color.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_secondary_color.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_secondary_color.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_secondary_color.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_secondary_color.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_secondary_color.txt" \
  "http://www.google.com/search?q=glDeleteSemaphoresEXT" \
  "http://www.google.com/search?q=glGenSemaphoresEXT" \
  "http://www.google.com/search?q=glGetSemaphoreParameterui64vEXT" \
  "http://www.google.com/search?q=glIsSemaphoreEXT" \
  "http://www.google.com/search?q=glSemaphoreParameterui64vEXT" \
  "http://www.google.com/search?q=glSignalSemaphoreEXT" \
  "http://www.google.com/search?q=glWaitSemaphoreEXT" \
  "http://www.google.com/search?q=glImportSemaphoreFdEXT" \
  "http://www.google.com/search?q=glImportSemaphoreWin32HandleEXT" \
  "http://www.google.com/search?q=glImportSemaphoreWin32NameEXT" \
  "http://www.google.com/search?q=glActiveProgramEXT" \
  "http://www.google.com/search?q=glCreateShaderProgramEXT" \
  "http://www.google.com/search?q=glUseShaderProgramEXT" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_shader_framebuffer_fetch.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_shader_image_load_store.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_shader_image_load_store.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_shader_pixel_local_storage2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_shader_pixel_local_storage2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_shader_pixel_local_storage2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_sparse_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_sparse_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_stencil_two_side.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_subtexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_subtexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_subtexture.txt" \
  "http://www.google.com/search?q=glPatchParameteriEXT" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture3D.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_border_clamp.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_border_clamp.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_border_clamp.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_border_clamp.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_integer.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_integer.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_integer.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_integer.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_integer.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_integer.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_perturb_normal.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_storage.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_storage.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_storage.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_storage.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_storage.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_storage.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_view.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_timer_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_timer_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_transform_feedback.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_transform_feedback.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_transform_feedback.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_transform_feedback.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_transform_feedback.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_transform_feedback.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_transform_feedback.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_array_setXXX.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_array_setXXX.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_array_setXXX.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_attrib_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_attrib_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_attrib_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_attrib_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_attrib_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_attrib_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_attrib_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_attrib_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_attrib_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_attrib_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_attrib_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_weighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_weighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_weighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_win32_keyed_mutex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_win32_keyed_mutex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_window_rectangles.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_x11_sync_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/GREMEDY/GREMEDY_frame_terminator.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/GREMEDY/GREMEDY_string_marker.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/HP/HP_image_transform.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/HP/HP_image_transform.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/HP/HP_image_transform.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/HP/HP_image_transform.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/HP/HP_image_transform.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/HP/HP_image_transform.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/IBM/IBM_multimode_draw_arrays.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/IBM/IBM_multimode_draw_arrays.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/IBM/IBM_vertex_array_lists.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/IBM/IBM_vertex_array_lists.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/IBM/IBM_vertex_array_lists.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/IBM/IBM_vertex_array_lists.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/IBM/IBM_vertex_array_lists.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/IBM/IBM_vertex_array_lists.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/IBM/IBM_vertex_array_lists.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/IBM/IBM_vertex_array_lists.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/IMG/IMG_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/IMG/IMG_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/IMG/IMG_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/IMG/IMG_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/IMG/IMG_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/IMG/IMG_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/IMG/IMG_framebuffer_downsample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/IMG/IMG_framebuffer_downsample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/IMG/IMG_multisampled_render_to_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/IMG/IMG_multisampled_render_to_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_map_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_map_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_map_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_parallel_arrays.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_parallel_arrays.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_parallel_arrays.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_parallel_arrays.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_performance_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_performance_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_performance_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_performance_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_performance_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_performance_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_performance_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_performance_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_performance_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_performance_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_texture_scissor.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_texture_scissor.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_blend_equation_advanced.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_debug.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_debug.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_debug.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_debug.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_debug.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_debug.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_debug.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_debug.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_debug.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_debug.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_parallel_shader_compile.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_robustness.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_robustness.txt" \
  "http://www.google.com/search?q=glBufferRegionEnabled" \
  "http://www.google.com/search?q=glDeleteBufferRegion" \
  "http://www.google.com/search?q=glDrawBufferRegion" \
  "http://www.google.com/search?q=glNewBufferRegion" \
  "http://www.google.com/search?q=glReadBufferRegion" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_framebuffer_flip_y.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_framebuffer_flip_y.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_resize_buffers.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NVX/NVX_conditional_render.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NVX/NVX_conditional_render.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NVX/NVX_gpu_multicast2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NVX/NVX_gpu_multicast2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NVX/NVX_gpu_multicast2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NVX/NVX_gpu_multicast2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NVX/NVX_gpu_multicast2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NVX/NVX_gpu_multicast2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NVX/NVX_linked_gpu_multicast.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NVX/NVX_linked_gpu_multicast.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NVX/NVX_linked_gpu_multicast.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NVX/NVX_progress_fence.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NVX/NVX_progress_fence.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NVX/NVX_progress_fence.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_3dvision_settings.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_3dvision_settings.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_alpha_to_coverage_dither_control.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_bindless_multi_draw_indirect.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_bindless_multi_draw_indirect.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_bindless_multi_draw_indirect_count.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_bindless_multi_draw_indirect_count.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_bindless_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_blend_equation_advanced.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_blend_equation_advanced.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_clip_space_w_scaling.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_command_list.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_command_list.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_command_list.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_command_list.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_command_list.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_command_list.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_command_list.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_command_list.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_command_list.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_command_list.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_command_list.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_command_list.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_command_list.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_command_list.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_command_list.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_command_list.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_command_list.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_conditional_render.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_conditional_render.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_conservative_raster.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_conservative_raster_dilate.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_conservative_raster_pre_snap_triangles.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_copy_buffer.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_copy_image.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_depth_buffer_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_depth_buffer_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_depth_buffer_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_draw_buffers.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_draw_instanced.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_draw_instanced.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_draw_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_draw_vulkan_image.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_draw_vulkan_image.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_draw_vulkan_image.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_draw_vulkan_image.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_evaluators.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_evaluators.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_evaluators.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_evaluators.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_evaluators.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_evaluators.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_evaluators.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_evaluators.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_evaluators.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_explicit_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_explicit_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_explicit_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_fence.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_fence.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_fence.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_fence.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_fence.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_fence.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_fence.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_fragment_coverage_to_color.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_fragment_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_fragment_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_fragment_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_fragment_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_fragment_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_fragment_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_framebuffer_blit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_framebuffer_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_framebuffer_multisample_coverage.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_geometry_program4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_multicast.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_multicast.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_multicast.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_multicast.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_multicast.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_multicast.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_multicast.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_multicast.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_multicast.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_multicast.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_multicast.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_multicast.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_program4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_program4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_program4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_program4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_program4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_program4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_program4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_program4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_program4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_program4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_program4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_program4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_instanced_arrays.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_internalformat_sample_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_memory_attachment.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_memory_attachment.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_memory_attachment.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_memory_attachment.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_memory_attachment.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_memory_attachment.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_mesh_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_mesh_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_mesh_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_mesh_shader.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_non_square_matrices.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_non_square_matrices.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_non_square_matrices.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_non_square_matrices.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_non_square_matrices.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_non_square_matrices.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_occlusion_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_occlusion_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_occlusion_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_occlusion_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_occlusion_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_occlusion_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_occlusion_query.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_parameter_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_parameter_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_parameter_buffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_pixel_data_range.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_pixel_data_range.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_point_sprite.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_point_sprite.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_polygon_mode.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_present_video.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_present_video.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_present_video.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_present_video.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_present_video.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_present_video.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_primitive_restart.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_primitive_restart.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_read_buffer.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_register_combiners.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_register_combiners.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_register_combiners.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_register_combiners.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_register_combiners.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_register_combiners.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_register_combiners.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_register_combiners.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_register_combiners.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_register_combiners.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_register_combiners.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_register_combiners.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_register_combiners.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_register_combiners2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_register_combiners2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_sample_locations.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_sample_locations.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_sample_locations.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_scissor_exclusive.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_scissor_exclusive.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_buffer_load.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_buffer_load.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_buffer_load.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_buffer_load.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_buffer_load.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_buffer_load.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_buffer_load.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_buffer_load.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_buffer_load.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_buffer_load.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_buffer_load.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_buffer_load.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_buffer_load.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shading_rate_image.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shading_rate_image.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shading_rate_image.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shading_rate_image.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shading_rate_image.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shading_rate_image.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texture_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texture_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texture_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texture_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texture_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texture_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texture_barrier.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texture_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texture_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texture_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texture_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texture_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texture_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_transform_feedback.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_transform_feedback.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_transform_feedback.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_transform_feedback.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_transform_feedback.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_transform_feedback.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_transform_feedback.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_transform_feedback.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_transform_feedback.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_transform_feedback.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_transform_feedback.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_transform_feedback2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_transform_feedback2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_transform_feedback2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_transform_feedback2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_transform_feedback2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_transform_feedback2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_transform_feedback2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vdpau_interop.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vdpau_interop.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vdpau_interop.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vdpau_interop.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vdpau_interop.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vdpau_interop.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vdpau_interop.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vdpau_interop.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vdpau_interop.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vdpau_interop.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vdpau_interop2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_array_range.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_array_range.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_attrib_integer_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_attrib_integer_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_attrib_integer_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_attrib_integer_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_attrib_integer_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_attrib_integer_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_attrib_integer_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_attrib_integer_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_attrib_integer_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_attrib_integer_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_attrib_integer_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_attrib_integer_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_attrib_integer_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_attrib_integer_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_attrib_integer_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_attrib_integer_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_attrib_integer_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_attrib_integer_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_attrib_integer_64bit.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_buffer_unified_memory.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_buffer_unified_memory.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_buffer_unified_memory.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_buffer_unified_memory.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_buffer_unified_memory.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_buffer_unified_memory.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_buffer_unified_memory.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_buffer_unified_memory.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_buffer_unified_memory.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_buffer_unified_memory.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_buffer_unified_memory.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_buffer_unified_memory.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_video_capture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_video_capture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_video_capture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_video_capture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_video_capture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_video_capture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_video_capture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_video_capture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_video_capture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_video_capture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_video_capture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_video_capture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_viewport_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_viewport_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_viewport_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_viewport_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_viewport_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_viewport_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_viewport_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_viewport_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_viewport_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_viewport_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_viewport_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_viewport_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_viewport_swizzle.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_EGL_image.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_EGL_image.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_blend_equation_separate.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_blend_func_separate.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_blend_subtract.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_copy_image.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_draw_buffers_indexed.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_draw_buffers_indexed.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_draw_buffers_indexed.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_draw_buffers_indexed.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_draw_buffers_indexed.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_draw_buffers_indexed.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_draw_buffers_indexed.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_draw_buffers_indexed.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_framebuffer_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_get_program_binary.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_get_program_binary.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_mapbuffer.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_mapbuffer.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_mapbuffer.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_matrix_palette.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_matrix_palette.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_matrix_palette.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_sample_shading.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_single_precision.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_single_precision.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_single_precision.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_single_precision.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_single_precision.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_single_precision.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_3D.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_3D.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_3D.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_3D.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_3D.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_3D.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_border_clamp.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_border_clamp.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_border_clamp.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_border_clamp.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_border_clamp.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_border_clamp.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_border_clamp.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_border_clamp.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_buffer.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_buffer.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_cube_map.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_cube_map.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_cube_map.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_cube_map.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_cube_map.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_cube_map.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_cube_map.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_cube_map.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_cube_map.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_storage_multisample_2d_array.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_texture_view.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OVR/OVR_multiview.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OVR/OVR_multiview.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/OVR/OVR_multiview_multisampled_render_to_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/QCOM/QCOM_alpha_test.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/QCOM/QCOM_driver_control.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/QCOM/QCOM_driver_control.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/QCOM/QCOM_driver_control.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/QCOM/QCOM_driver_control.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/QCOM/QCOM_extended_get.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/QCOM/QCOM_extended_get.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/QCOM/QCOM_extended_get.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/QCOM/QCOM_extended_get.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/QCOM/QCOM_extended_get.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/QCOM/QCOM_extended_get.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/QCOM/QCOM_extended_get.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/QCOM/QCOM_extended_get.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/QCOM/QCOM_extended_get2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/QCOM/QCOM_extended_get2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/QCOM/QCOM_extended_get2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/QCOM/QCOM_extended_get2.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/QCOM/QCOM_framebuffer_foveated.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/QCOM/QCOM_framebuffer_foveated.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/QCOM/QCOM_shader_framebuffer_fetch_noncoherent.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/QCOM/QCOM_texture_foveated.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/QCOM/QCOM_tiled_rendering.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/QCOM/QCOM_tiled_rendering.txt" \
  "http://www.google.com/search?q=glAlphaFuncx" \
  "http://www.google.com/search?q=glClearColorx" \
  "http://www.google.com/search?q=glClearDepthx" \
  "http://www.google.com/search?q=glColor4x" \
  "http://www.google.com/search?q=glDepthRangex" \
  "http://www.google.com/search?q=glFogx" \
  "http://www.google.com/search?q=glFogxv" \
  "http://www.google.com/search?q=glFrustumf" \
  "http://www.google.com/search?q=glFrustumx" \
  "http://www.google.com/search?q=glLightModelx" \
  "http://www.google.com/search?q=glLightModelxv" \
  "http://www.google.com/search?q=glLightx" \
  "http://www.google.com/search?q=glLightxv" \
  "http://www.google.com/search?q=glLineWidthx" \
  "http://www.google.com/search?q=glLoadMatrixx" \
  "http://www.google.com/search?q=glMaterialx" \
  "http://www.google.com/search?q=glMaterialxv" \
  "http://www.google.com/search?q=glMultMatrixx" \
  "http://www.google.com/search?q=glMultiTexCoord4x" \
  "http://www.google.com/search?q=glNormal3x" \
  "http://www.google.com/search?q=glOrthof" \
  "http://www.google.com/search?q=glOrthox" \
  "http://www.google.com/search?q=glPointSizex" \
  "http://www.google.com/search?q=glPolygonOffsetx" \
  "http://www.google.com/search?q=glRotatex" \
  "http://www.google.com/search?q=glSampleCoveragex" \
  "http://www.google.com/search?q=glScalex" \
  "http://www.google.com/search?q=glTexEnvx" \
  "http://www.google.com/search?q=glTexEnvxv" \
  "http://www.google.com/search?q=glTexParameterx" \
  "http://www.google.com/search?q=glTranslatex" \
  "http://www.google.com/search?q=glClipPlanef" \
  "http://www.google.com/search?q=glClipPlanex" \
  "http://www.google.com/search?q=glGetClipPlanef" \
  "http://www.google.com/search?q=glGetClipPlanex" \
  "http://www.google.com/search?q=glGetFixedv" \
  "http://www.google.com/search?q=glGetLightxv" \
  "http://www.google.com/search?q=glGetMaterialxv" \
  "http://www.google.com/search?q=glGetTexEnvxv" \
  "http://www.google.com/search?q=glGetTexParameterxv" \
  "http://www.google.com/search?q=glPointParameterx" \
  "http://www.google.com/search?q=glPointParameterxv" \
  "http://www.google.com/search?q=glPointSizePointerOES" \
  "http://www.google.com/search?q=glTexParameterxv" \
  "http://www.google.com/search?q=glErrorStringREGAL" \
  "http://www.google.com/search?q=glGetExtensionREGAL" \
  "http://www.google.com/search?q=glIsSupportedREGAL" \
  "http://www.google.com/search?q=glLogMessageCallbackREGAL" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_detail_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_detail_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_fog_function.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_fog_function.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_multitexture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_shared_multisample.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_sharpen_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_sharpen_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_texture4D.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_texture4D.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_texture_filter4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_texture_filter4.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_async.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_async.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_async.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_async.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_async.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_async.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_datapipe.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_datapipe.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_flush_raster.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_fog_layers.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_fog_layers.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_fog_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_fragment_specular_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_fragment_specular_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_fragment_specular_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_fragment_specular_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_fragment_specular_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_fragment_specular_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_fragment_specular_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_fragment_specular_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_fragment_specular_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_fragment_specular_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_fragment_specular_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_fragment_specular_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_fragment_specular_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_fragment_specular_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_fragment_specular_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_fragment_specular_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_fragment_specular_lighting.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_framezoom.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_igloo_interface.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_mpeg1.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_mpeg1.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_mpeg1.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_mpeg1.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_mpeg1.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_mpeg1.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_mpeg1.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_mpeg1.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_mpeg1.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_mpeg1.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_mpeg1.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_nonlinear_lighting_pervertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_nonlinear_lighting_pervertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_nonlinear_lighting_pervertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_nonlinear_lighting_pervertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_pixel_texture.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_polynomial_ffd.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_polynomial_ffd.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_quad_mesh.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_quad_mesh.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_reference_plane.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_sprite.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_sprite.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_sprite.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_sprite.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_tag_sample_buffer.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_vector_ops.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_vector_ops.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_vertex_array_object.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGI/SGI_color_table.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGI/SGI_color_table.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGI/SGI_color_table.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGI/SGI_color_table.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGI/SGI_color_table.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGI/SGI_color_table.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGI/SGI_color_table.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGI/SGI_fft.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGI/SGI_fft.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGI/SGI_fft.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGI/SGI_fft.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGI/SGI_fft.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGI/SGI_fft.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SGI/SGI_fft.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUNX/SUNX_constant_data.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_global_alpha.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_global_alpha.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_global_alpha.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_global_alpha.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_global_alpha.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_global_alpha.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_global_alpha.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_global_alpha.txt" \
  "http://www.google.com/search?q=glReadVideoPixelsSUN" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_triangle_list.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_triangle_list.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_triangle_list.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_triangle_list.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_triangle_list.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_triangle_list.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_triangle_list.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt" \
  "http://www.google.com/search?q=glAddSwapHintRectWIN" \
]

# Array of extension names and corresponding URLs
array set ::__tcl3dOglExtensionList {
    "WGL_NV_DX_interop" "https://www.khronos.org/registry/OpenGL/extensions/NV/WGL_NV_DX_interop.txt"
    "GL_EXT_texture_sRGB_R8" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_sRGB_R8.txt"
    "GL_NV_internalformat_sample_query" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_internalformat_sample_query.txt"
    "GLX_EXT_scene_marker" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_scene_marker.txt"
    "WGL_NV_swap_group" "https://www.khronos.org/registry/OpenGL/extensions/NV/WGL_NV_swap_group.txt"
    "WGL_NV_present_video" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_present_video.txt"
    "GL_ARB_vertex_program" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_program.txt"
    "GL_NVX_linked_gpu_multicast" "https://www.khronos.org/registry/OpenGL/extensions/NVX/NVX_linked_gpu_multicast.txt"
    "GL_INTEL_conservative_rasterization" "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_conservative_rasterization.txt"
    "GL_ARB_gl_spirv" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gl_spirv.txt"
    "WGL_ATI_pixel_format_float" "https://www.khronos.org/registry/OpenGL/extensions/ATI/WGL_ATI_pixel_format_float.txt"
    "GL_EXT_blend_color" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_blend_color.txt"
    "GL_ARB_multi_bind" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multi_bind.txt"
    "GL_NV_primitive_shading_rate" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_primitive_shading_rate.txt"
    "GL_EXT_external_buffer" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_external_buffer.txt"
    "GL_EXT_direct_state_access" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_direct_state_access.txt"
    "GL_APPLE_client_storage" "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_client_storage.txt"
    "GL_EXT_texture_perturb_normal" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_perturb_normal.txt"
    "GLX_ARB_get_proc_address" "https://www.khronos.org/registry/OpenGL/extensions/ARB/GLX_ARB_get_proc_address.txt"
    "GL_NV_vdpau_interop2" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vdpau_interop2.txt"
    "GL_NV_shader_storage_buffer_object" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_storage_buffer_object.txt"
    "GL_SGIX_async_histogram" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_async_histogram.txt"
    "GL_HP_texture_lighting" "https://www.khronos.org/registry/OpenGL/extensions/HP/HP_texture_lighting.txt"
    "GL_SGIS_texture_select" "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_texture_select.txt"
    "GL_ARB_texture_rg" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_rg.txt"
    "GLX_EXT_no_config_context" "https://www.khronos.org/registry/OpenGL/extensions/EXT/GLX_EXT_no_config_context.txt"
    "GL_AMD_occlusion_query_event" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_occlusion_query_event.txt"
    "GL_ARB_shader_precision" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_precision.txt"
    "GL_NV_shader_buffer_store" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_buffer_store.txt"
    "WGL_NV_copy_image" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_copy_image.txt"
    "GL_SGIS_texture_color_mask" "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_texture_color_mask.txt"
    "GL_ARB_shader_bit_encoding" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_bit_encoding.txt"
    "GL_ARB_compute_shader" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_compute_shader.txt"
    "GL_OES_query_matrix" "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_query_matrix.txt"
    "GL_EXT_convolution" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_convolution.txt"
    "GL_EXT_memory_object_win32" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_external_objects_win32.txt"
    "GL_INTEL_parallel_arrays" "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_parallel_arrays.txt"
    "GL_ARB_point_parameters" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_point_parameters.txt"
    "GL_NV_query_resource_tag" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_query_resource_tag.txt"
    "WGL_3DL_stereo_control" "https://www.khronos.org/registry/OpenGL/extensions/3DL/WGL_3DL_stereo_control.txt"
    "GL_MESA_resize_buffers" "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_resize_buffers.txt"
    "GL_ARB_vertex_array_bgra" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_array_bgra.txt"
    "GL_ARB_gpu_shader5" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader5.txt"
    "GL_ARB_arrays_of_arrays" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_arrays_of_arrays.txt"
    "GL_ARB_clear_texture" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_clear_texture.txt"
    "GL_ATI_texture_mirror_once" "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_texture_mirror_once.txt"
    "GL_IBM_cull_vertex" "https://www.khronos.org/registry/OpenGL/extensions/IBM/IBM_cull_vertex.txt"
    "GL_ARB_shader_group_vote" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_group_vote.txt"
    "GLX_SUN_get_transparent_index" "https://www.khronos.org/registry/OpenGL/extensions/SUN/GLX_SUN_get_transparent_index.txt"
    "GL_EXT_cmyka" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_cmyka.txt"
    "GL_ARB_pixel_buffer_object" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_pixel_buffer_object.txt"
    "GL_AMD_query_buffer_object" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_query_buffer_object.txt"
    "GL_EXT_framebuffer_multisample_blit_scaled" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_multisample_blit_scaled.txt"
    "GL_SGIX_instruments" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_instruments.txt"
    "GL_SGIS_texture4D" "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_texture4D.txt"
    "GL_SGIX_depth_texture" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_depth_texture.txt"
    "GLX_ARB_fbconfig_float" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_color_buffer_float.txt"
    "GL_ARB_shader_image_size" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_image_size.txt"
    "GL_EXT_sparse_texture2" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_sparse_texture2.txt"
    "GL_NV_blend_square" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_blend_square.txt"
    "GLU_EXT_nurbs_tessellator" "https://www.khronos.org/registry/OpenGL/extensions/EXT/GLU_EXT_nurbs_tessellator.txt"
    "GL_SGIX_shadow" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_shadow.txt"
    "GL_NV_vertex_program1_1" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program1_1.txt"
    "GL_ARB_texture_env_combine" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_env_combine.txt"
    "GL_ARB_tessellation_shader" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_tessellation_shader.txt"
    "GL_S3_s3tc" "https://www.khronos.org/registry/OpenGL/extensions/S3/S3_s3tc.txt"
    "GLX_SGIS_multisample" "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_multisample.txt"
    "GL_ARB_timer_query" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_timer_query.txt"
    "GL_KHR_blend_equation_advanced_coherent" "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_blend_equation_advanced.txt"
    "GL_NV_draw_vulkan_image" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_draw_vulkan_image.txt"
    "GL_IBM_texture_mirrored_repeat" "https://www.khronos.org/registry/OpenGL/extensions/IBM/IBM_texture_mirrored_repeat.txt"
    "GL_KHR_debug" "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_debug.txt"
    "GL_EXT_texture_shadow_lod" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_shadow_lod.txt"
    "GL_ARB_robustness" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_robustness.txt"
    "GL_KHR_texture_compression_astc_hdr" "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_texture_compression_astc_hdr.txt"
    "GL_OVR_multiview2" "https://www.khronos.org/registry/OpenGL/extensions/OVR/OVR_multiview2.txt"
    "GL_INTEL_fragment_shader_ordering" "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_fragment_shader_ordering.txt"
    "GL_EXT_gpu_shader4" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_shader4.txt"
    "GL_SUN_multi_draw_arrays" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_multi_draw_arrays.txt"
    "GL_ARB_cull_distance" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_cull_distance.txt"
    "GL_ARB_ES2_compatibility" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_ES2_compatibility.txt"
    "GL_NV_vertex_program" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program.txt"
    "GL_SUNX_constant_data" "https://www.khronos.org/registry/OpenGL/extensions/SUNX/SUNX_constant_data.txt"
    "GL_EXT_light_texture" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_light_texture.txt"
    "GL_EXT_blend_logic_op" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_blend_logic_op.txt"
    "GL_ARB_shader_ballot" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_ballot.txt"
    "GL_ARB_occlusion_query" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_occlusion_query.txt"
    "GL_ARB_copy_buffer" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_copy_buffer.txt"
    "GL_ARB_texture_query_lod" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_query_lod.txt"
    "GL_NV_explicit_multisample" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_explicit_multisample.txt"
    "GL_EXT_texture_compression_rgtc" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_compression_rgtc.txt"
    "GL_EXT_pixel_buffer_object" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_pixel_buffer_object.txt"
    "GL_EXT_fragment_lighting" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fragment_lighting.txt"
    "GL_EXT_index_array_formats" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_index_array_formats.txt"
    "GL_NV_sample_mask_override_coverage" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_sample_mask_override_coverage.txt"
    "GL_NVX_gpu_memory_info" "https://www.khronos.org/registry/OpenGL/extensions/NVX/NVX_gpu_memory_info.txt"
    "GL_NV_vertex_program2" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program2.txt"
    "GL_ARB_occlusion_query2" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_occlusion_query2.txt"
    "GL_NV_vertex_program3" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program3.txt"
    "GL_OML_subsample" "https://www.khronos.org/registry/OpenGL/extensions/OML/OML_subsample.txt"
    "GL_EXT_provoking_vertex" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_provoking_vertex.txt"
    "GL_NV_vertex_program4" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program4.txt"
    "GL_EXT_texture_compression_latc" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_compression_latc.txt"
    "GL_EXT_texture_lod_bias" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_lod_bias.txt"
    "GL_EXT_texture_env_combine" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_env_combine.txt"
    "GL_AMD_transform_feedback3_lines_triangles" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_transform_feedback3_lines_triangles.txt"
    "GL_EXT_timer_query" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_timer_query.txt"
    "GL_SGIX_vertex_preclip" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_vertex_preclip.txt"
    "GL_EXT_histogram" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_histogram.txt"
    "GL_EXT_polygon_offset" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_polygon_offset.txt"
    "GL_ARB_transform_feedback2" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_transform_feedback2.txt"
    "GL_ATI_draw_buffers" "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_draw_buffers.txt"
    "GLX_EXT_import_context" "https://www.khronos.org/registry/OpenGL/extensions/EXT/GLX_EXT_import_context.txt"
    "GL_SGIS_texture_filter4" "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_texture_filter4.txt"
    "GL_KHR_parallel_shader_compile" "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_parallel_shader_compile.txt"
    "GL_ARB_fragment_program" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_fragment_program.txt"
    "GL_ARB_transform_feedback3" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_transform_feedback3.txt"
    "GL_NV_primitive_restart" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_primitive_restart.txt"
    "GL_NV_packed_depth_stencil" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_packed_depth_stencil.txt"
    "GL_EXT_texture_filter_anisotropic" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_filter_anisotropic.txt"
    "GL_SGIS_color_range" "https://www.khronos.org/registry/OpenGL/extensions/SGIS/GLX_SGIS_color_range.txt"
    "GLX_ARB_multisample" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multisample.txt"
    "GL_NV_fragment_coverage_to_color" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_fragment_coverage_to_color.txt"
    "GL_EXT_framebuffer_sRGB" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_sRGB.txt"
    "GLX_EXT_fbconfig_packed_float" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_packed_float.txt"
    "GL_ATI_map_object_buffer" "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_map_object_buffer.txt"
    "WGL_I3D_digital_video_control" "https://www.khronos.org/registry/OpenGL/extensions/I3D/WGL_I3D_digital_video_control.txt"
    "GL_EXT_fog_coord" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_fog_coord.txt"
    "GL_MESA_framebuffer_flip_x" "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_framebuffer_flip_x.txt"
    "GL_ARB_texture_filter_minmax" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_filter_minmax.txt"
    "GL_ARB_texture_barrier" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_barrier.txt"
    "GL_MESA_framebuffer_flip_y" "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_framebuffer_flip_y.txt"
    "GL_NV_shader_atomic_float64" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_atomic_float64.txt"
    "GL_AMD_name_gen_delete" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_name_gen_delete.txt"
    "GL_HP_image_transform" "https://www.khronos.org/registry/OpenGL/extensions/HP/HP_image_transform.txt"
    "GL_ARB_shader_stencil_export" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_stencil_export.txt"
    "GL_EXT_draw_range_elements" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_range_elements.txt"
    "GL_SGIS_fog_function" "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_fog_function.txt"
    "GL_ARB_instanced_arrays" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_instanced_arrays.txt"
    "GL_APPLE_row_bytes" "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_row_bytes.txt"
    "GL_ATI_fragment_shader" "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_fragment_shader.txt"
    "GL_ARB_shader_atomic_counter_ops" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_atomic_counter_ops.txt"
    "GL_SGI_texture_color_table" "https://www.khronos.org/registry/OpenGL/extensions/SGI/SGI_texture_color_table.txt"
    "GL_ARB_bindless_texture" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_bindless_texture.txt"
    "GL_ARB_sample_locations" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sample_locations.txt"
    "GL_ARB_texture_rectangle" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_rectangle.txt"
    "GL_ARB_shader_texture_lod" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_texture_lod.txt"
    "GL_EXT_multiview_texture_multisample" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_multiview_texture_multisample.txt"
    "GL_NV_framebuffer_mixed_samples" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_framebuffer_mixed_samples.txt"
    "GL_AMD_shader_stencil_export" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_shader_stencil_export.txt"
    "GLX_MESA_copy_sub_buffer" "https://www.khronos.org/registry/OpenGL/extensions/MESA/GLX_MESA_copy_sub_buffer.txt"
    "GL_3DFX_tbuffer" "https://www.khronos.org/registry/OpenGL/extensions/3DFX/3DFX_tbuffer.txt"
    "GLX_SGIX_video_resize" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/GLX_SGIX_video_resize.txt"
    "GL_ARB_shadow" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shadow.txt"
    "GL_EXT_swap_control" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_swap_control.txt"
    "WGL_EXT_swap_control_tear" "https://www.khronos.org/registry/OpenGL/extensions/EXT/WGL_EXT_swap_control_tear.txt"
    "GL_ARB_texture_compression_rgtc" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_compression_rgtc.txt"
    "GL_AMD_framebuffer_sample_positions" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_framebuffer_sample_positions.txt"
    "GL_AMD_shader_stencil_value_export" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_shader_stencil_value_export.txt"
    "GL_NV_fragment_program_option" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_fragment_program_option.txt"
    "GL_ARB_polygon_offset_clamp" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_polygon_offset_clamp.txt"
    "GL_ATI_vertex_array_object" "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_array_object.txt"
    "GL_EXT_texture_object" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_object.txt"
    "WGL_NV_DX_interop2" "https://www.khronos.org/registry/OpenGL/extensions/NV/WGL_NV_DX_interop2.txt"
    "GL_NV_copy_image" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_copy_image.txt"
    "GL_EXT_texture3D" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture3D.txt"
    "GL_ARB_provoking_vertex" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_provoking_vertex.txt"
    "GL_MESA_shader_integer_functions" "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_shader_integer_functions.txt"
    "WGL_EXT_pbuffer" "https://www.khronos.org/registry/OpenGL/extensions/EXT/WGL_EXT_pbuffer.txt"
    "GL_SGIX_sprite" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_sprite.txt"
    "GL_EXT_texture" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture.txt"
    "GL_ARB_texture_multisample" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_multisample.txt"
    "GLX_MESA_release_buffers" "https://www.khronos.org/registry/OpenGL/extensions/MESA/GLX_MESA_release_buffers.txt"
    "GL_APPLE_specular_vector" "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_specular_vector.txt"
    "GL_NV_viewport_array2" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_viewport_array2.txt"
    "GL_APPLE_vertex_program_evaluators" "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_vertex_program_evaluators.txt"
    "WGL_I3D_swap_frame_lock" "https://www.khronos.org/registry/OpenGL/extensions/I3D/WGL_I3D_swap_frame_lock.txt"
    "GL_EXT_paletted_texture" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_paletted_texture.txt"
    "GL_AMD_gpu_shader_int16" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_gpu_shader_int16.txt"
    "GL_OVR_multiview" "https://www.khronos.org/registry/OpenGL/extensions/OVR/OVR_multiview.txt"
    "GL_EXT_shader_image_load_store" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_shader_image_load_store.txt"
    "GL_ATI_vertex_attrib_array_object" "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_attrib_array_object.txt"
    "GLX_SGIS_blended_overlay" "https://www.khronos.org/registry/OpenGL/extensions/SGIS/GLX_SGIS_blended_overlay.txt"
    "GL_EXT_packed_pixels" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_packed_pixels.txt"
    "GL_ARB_texture_query_levels" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_query_levels.txt"
    "GL_ATI_element_array" "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_element_array.txt"
    "GLX_SGI_swap_control" "https://www.khronos.org/registry/OpenGL/extensions/SGI/GLX_SGI_swap_control.txt"
    "GL_EXT_secondary_color" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_secondary_color.txt"
    "GL_SGIX_ir_instrument1" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_ir_instrument1.txt"
    "GL_ARB_framebuffer_sRGB" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_sRGB.txt"
    "GL_EXT_texture_shared_exponent" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_shared_exponent.txt"
    "GL_NVX_progress_fence" "https://www.khronos.org/registry/OpenGL/extensions/NVX/NVX_progress_fence.txt"
    "GL_ARB_shading_language_include" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shading_language_include.txt"
    "GL_ARB_texture_rgb10_a2ui" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_rgb10_a2ui.txt"
    "GLX_EXT_stereo_tree" "https://www.khronos.org/registry/OpenGL/extensions/EXT/GLX_EXT_stereo_tree.txt"
    "GL_EXT_multi_draw_arrays" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_multi_draw_arrays.txt"
    "GL_SGIX_interlace" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_interlace.txt"
    "GLX_ARB_create_context_no_error" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_create_context_no_error.txt"
    "GL_ARB_sampler_objects" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sampler_objects.txt"
    "GL_INTEL_performance_query" "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_performance_query.txt"
    "GLX_EXT_texture_from_pixmap" "https://www.khronos.org/registry/OpenGL/extensions/EXT/GLX_EXT_texture_from_pixmap.txt"
    "GL_NV_geometry_program4" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_geometry_program4.txt"
    "GL_EXT_pixel_transform_color_table" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_pixel_transform_color_table.txt"
    "GL_AMD_gpu_shader_half_float_fetch" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_gpu_shader_half_float_fetch.txt"
    "GL_ATI_separate_stencil" "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_separate_stencil.txt"
    "GL_NV_vertex_array_range" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_array_range.txt"
    "GL_ARB_explicit_attrib_location" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_explicit_attrib_location.txt"
    "GL_MESA_framebuffer_swap_xy" "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_framebuffer_swap_xy.txt"
    "GL_NV_memory_attachment" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_memory_attachment.txt"
    "GL_NV_conservative_raster" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_conservative_raster.txt"
    "GL_ARB_clip_control" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_clip_control.txt"
    "GL_ARB_robustness_isolation" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_robustness_application_isolation.txt"
    "GL_NV_mesh_shader" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_mesh_shader.txt"
    "GL_NV_conservative_raster_underestimation" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_conservative_raster_underestimation.txt"
    "WGL_EXT_create_context_es_profile" "https://www.khronos.org/registry/OpenGL/extensions/EXT/WGL_EXT_create_context_es2_profile.txt"
    "GL_EXT_depth_bounds_test" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_depth_bounds_test.txt"
    "GL_ARB_vertex_blend" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_blend.txt"
    "GL_INTEL_blackhole_render" "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_blackhole_render.txt"
    "GL_APPLE_transform_hint" "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_transform_hint.txt"
    "GL_WIN_specular_fog" "https://www.khronos.org/registry/OpenGL/extensions/WIN/WIN_specular_fog.txt"
    "GL_ARB_parallel_shader_compile" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_parallel_shader_compile.txt"
    "GL_ARB_texture_view" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_view.txt"
    "GL_EXT_texture_buffer_object" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_buffer_object.txt"
    "GL_NV_texgen_reflection" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texgen_reflection.txt"
    "GL_NV_gpu_program5_mem_extended" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_program5_mem_extended.txt"
    "GL_ARB_shadow_ambient" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shadow_ambient.txt"
    "GL_ARB_sync" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sync.txt"
    "GL_NV_vertex_program2_option" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_program2_option.txt"
    "GL_NV_point_sprite" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_point_sprite.txt"
    "GL_SUN_global_alpha" "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_global_alpha.txt"
    "GL_ARB_fragment_coord_conventions" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_fragment_coord_conventions.txt"
    "GL_ARB_seamless_cubemap_per_texture" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_seamless_cubemap_per_texture.txt"
    "GL_SGIX_pixel_texture" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_pixel_texture.txt"
    "GLX_SGIX_pbuffer" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/GLX_SGIX_pbuffer.txt"
    "GL_ARB_shader_objects" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_objects.txt"
    "GL_EXT_raster_multisample" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_raster_multisample.txt"
    "GL_EXT_shader_image_load_formatted" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_shader_image_load_formatted.txt"
    "GLX_EXT_context_priority" "https://www.khronos.org/registry/OpenGL/extensions/EXT/GLX_EXT_context_priority.txt"
    "GLX_EXT_swap_control_tear" "https://www.khronos.org/registry/OpenGL/extensions/EXT/GLX_EXT_swap_control_tear.txt"
    "GL_AMD_gcn_shader" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_gcn_shader.txt"
    "GL_OML_resample" "https://www.khronos.org/registry/OpenGL/extensions/OML/OML_resample.txt"
    "GL_SGIX_texture_scale_bias" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_texture_scale_bias.txt"
    "GL_ARB_ES3_compatibility" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_ES3_compatibility.txt"
    "GL_ARB_framebuffer_no_attachments" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_no_attachments.txt"
    "GL_ARB_invalidate_subdata" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_invalidate_subdata.txt"
    "GLX_NV_video_out" "https://www.khronos.org/registry/OpenGL/extensions/NV/GLX_NV_video_out.txt"
    "WGL_EXT_pixel_format" "https://www.khronos.org/registry/OpenGL/extensions/EXT/WGL_EXT_pixel_format.txt"
    "GL_IBM_rasterpos_clip" "https://www.khronos.org/registry/OpenGL/extensions/IBM/IBM_rasterpos_clip.txt"
    "GL_ARB_texture_storage" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_storage.txt"
    "GL_NV_gpu_program4" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_program4.txt"
    "GL_ARB_draw_instanced" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_draw_instanced.txt"
    "GL_NV_gpu_program5" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_program5.txt"
    "GL_NV_parameter_buffer_object" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_parameter_buffer_object.txt"
    "GL_EXT_stencil_wrap" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_stencil_wrap.txt"
    "GL_ARB_base_instance" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_base_instance.txt"
    "GL_EXT_EGL_sync" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_EGL_sync.txt"
    "GL_AMD_shader_atomic_counter_ops" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_shader_atomic_counter_ops.txt"
    "GL_ARB_transform_feedback_overflow_query" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_transform_feedback_overflow_query.txt"
    "GL_NV_transform_feedback2" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_transform_feedback2.txt"
    "WGL_EXT_pixel_format_packed_float" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_packed_float.txt"
    "GL_SGIX_blend_alpha_minmax" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_blend_alpha_minmax.txt"
    "GL_NV_depth_buffer_float" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_depth_buffer_float.txt"
    "GL_ARB_texture_stencil8" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_stencil8.txt"
    "GL_NV_blend_minmax_factor" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_blend_minmax_factor.txt"
    "GL_EXT_texture_array" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_array.txt"
    "GL_SGIS_pixel_texture" "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_pixel_texture.txt"
    "GL_ARB_transpose_matrix" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_transpose_matrix.txt"
    "GLX_SGIS_color_range" "https://www.khronos.org/registry/OpenGL/extensions/SGIS/GLX_SGIS_color_range.txt"
    "GL_SGIS_detail_texture" "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_detail_texture.txt"
    "GL_INGR_interlace_read" "https://www.khronos.org/registry/OpenGL/extensions/INGR/INGR_interlace_read.txt"
    "GL_ARB_vertex_attrib_binding" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_attrib_binding.txt"
    "GLX_EXT_framebuffer_sRGB" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_sRGB.txt"
    "GL_NV_float_buffer" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_float_buffer.txt"
    "GL_REND_screen_coordinates" "https://www.khronos.org/registry/OpenGL/extensions/REND/REND_screen_coordinates.txt"
    "GL_SGIX_framezoom" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_framezoom.txt"
    "WGL_ARB_pixel_format_float" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_color_buffer_float.txt"
    "GLX_ARB_context_flush_control" "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_context_flush_control.txt"
    "GL_NV_robustness_video_memory_purge" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_robustness_video_memory_purge.txt"
    "GL_ARB_robustness_share_group_isolation" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_robustness_application_isolation.txt"
    "GL_APPLE_aux_depth_stencil" "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_aux_depth_stencil.txt"
    "GL_ARB_compute_variable_group_size" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_compute_variable_group_size.txt"
    "WGL_NV_multigpu_context" "https://www.khronos.org/registry/OpenGL/extensions/NV/WGL_NV_multigpu_context.txt"
    "GL_NV_vertex_buffer_unified_memory" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_buffer_unified_memory.txt"
    "GL_NV_vertex_array_range2" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_array_range2.txt"
    "GL_SGIX_ycrcb" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_ycrcb.txt"
    "GL_ARB_half_float_pixel" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_half_float_pixel.txt"
    "GL_ARB_texture_env_add" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_env_add.txt"
    "GL_ARB_texture_storage_multisample" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_storage_multisample.txt"
    "GL_SGIS_sharpen_texture" "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_sharpen_texture.txt"
    "WGL_ARB_make_current_read" "https://www.khronos.org/registry/OpenGL/extensions/ARB/WGL_ARB_make_current_read.txt"
    "GL_EXT_vertex_array_bgra" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_array_bgra.txt"
    "GL_EXT_framebuffer_blit" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_blit.txt"
    "GL_EXT_debug_marker" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_debug_marker.txt"
    "GL_NV_deep_texture3D" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_deep_texture3D.txt"
    "WGL_AMD_gpu_association" "https://www.khronos.org/registry/OpenGL/extensions/AMD/WGL_AMD_gpu_association.txt"
    "GLX_MESA_pixmap_colormap" "https://www.khronos.org/registry/OpenGL/extensions/MESA/GLX_MESA_pixmap_colormap.txt"
    "GL_NV_light_max_exponent" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_light_max_exponent.txt"
    "GL_AMD_vertex_shader_layer" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_vertex_shader_layer.txt"
    "WGL_NV_float_buffer" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_float_buffer.txt"
    "GL_ARB_depth_texture" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_depth_texture.txt"
    "GL_KHR_texture_compression_astc_ldr" "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_texture_compression_astc_hdr.txt"
    "GL_OES_single_precision" "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_single_precision.txt"
    "GL_EXT_copy_texture" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_copy_texture.txt"
    "GL_NV_texture_rectangle_compressed" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texture_rectangle_compressed.txt"
    "GL_EXT_draw_buffers2" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_buffers2.txt"
    "GL_ARB_point_sprite" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_point_sprite.txt"
    "GL_ARB_texture_swizzle" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_swizzle.txt"
    "GL_NV_command_list" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_command_list.txt"
    "GL_ATI_envmap_bumpmap" "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_envmap_bumpmap.txt"
    "GL_SGI_color_matrix" "https://www.khronos.org/registry/OpenGL/extensions/SGI/SGI_color_matrix.txt"
    "GL_ARB_explicit_uniform_location" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_explicit_uniform_location.txt"
    "GL_EXT_blend_equation_separate" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_blend_equation_separate.txt"
    "GL_ARB_draw_buffers" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_draw_buffers.txt"
    "GL_ARB_texture_border_clamp" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_border_clamp.txt"
    "GL_NV_bindless_multi_draw_indirect_count" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_bindless_multi_draw_indirect_count.txt"
    "GL_AMD_gpu_shader_int64" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_gpu_shader_int64.txt"
    "GL_SGIX_texture_range" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_texture_range.txt"
    "GL_NV_texture_multisample" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texture_multisample.txt"
    "GLX_SGIX_visual_select_group" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/GLX_SGIX_visual_select_group.txt"
    "WGL_EXT_depth_float" "https://www.khronos.org/registry/OpenGL/extensions/EXT/WGL_EXT_depth_float.txt"
    "GL_EXT_separate_specular_color" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_separate_specular_color.txt"
    "GL_ARB_half_float_vertex" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_half_float_vertex.txt"
    "GL_ARB_pipeline_statistics_query" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_pipeline_statistics_query.txt"
    "GL_NV_viewport_swizzle" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_viewport_swizzle.txt"
    "GL_NV_shader_atomic_int64" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_atomic_int64.txt"
    "GL_AMD_performance_monitor" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_performance_monitor.txt"
    "GL_NV_copy_depth_to_color" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_copy_depth_to_color.txt"
    "WGL_ARB_create_context_robustness" "https://www.khronos.org/registry/OpenGL/extensions/ARB/WGL_ARB_create_context_robustness.txt"
    "GL_ARB_gpu_shader_int64" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_int64.txt"
    "GL_ATI_texture_float" "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_texture_float.txt"
    "WGL_ARB_create_context" "https://www.khronos.org/registry/OpenGL/extensions/ARB/WGL_ARB_create_context.txt"
    "GL_EXT_shader_framebuffer_fetch" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_shader_framebuffer_fetch.txt"
    "GL_ARB_depth_clamp" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_depth_clamp.txt"
    "GL_EXT_semaphore_win32" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_external_objects_win32.txt"
    "GL_NV_fragment_program" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_fragment_program.txt"
    "GL_SUN_triangle_list" "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_triangle_list.txt"
    "GLX_ARB_framebuffer_sRGB" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_sRGB.txt"
    "GL_NV_shader_atomic_counters" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_atomic_counters.txt"
    "GL_SGIX_async" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_async.txt"
    "GL_EXT_index_texture" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_index_texture.txt"
    "GL_ARB_separate_shader_objects" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_separate_shader_objects.txt"
    "GL_EXT_framebuffer_multisample" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_multisample.txt"
    "GL_EXT_texture_compression_dxt1" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_compression_dxt1.txt"
    "WGL_EXT_swap_control" "https://www.khronos.org/registry/OpenGL/extensions/EXT/WGL_EXT_swap_control.txt"
    "GL_NV_vdpau_interop" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vdpau_interop.txt"
    "GLX_SGIX_swap_barrier" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/GLX_SGIX_swap_barrier.txt"
    "GL_SGIS_generate_mipmap" "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_generate_mipmap.txt"
    "GLX_SGIX_dm_buffer" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/GLX_SGIX_dm_buffer.txt"
    "GLX_ARB_create_context_profile" "https://www.khronos.org/registry/OpenGL/extensions/ARB/GLX_ARB_create_context.txt"
    "GL_NV_shader_thread_group" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_thread_group.txt"
    "GL_EXT_win32_keyed_mutex" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_win32_keyed_mutex.txt"
    "GL_NVX_conditional_render" "https://www.khronos.org/registry/OpenGL/extensions/NVX/NVX_conditional_render.txt"
    "GL_APPLE_object_purgeable" "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_object_purgeable.txt"
    "GL_NV_path_rendering_shared_edge" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering_shared_edge.txt"
    "GL_NV_bindless_texture" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_bindless_texture.txt"
    "GL_ARB_get_program_binary" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_get_program_binary.txt"
    "GL_NV_scissor_exclusive" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_scissor_exclusive.txt"
    "GL_NV_sample_locations" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_sample_locations.txt"
    "GL_SGIX_tag_sample_buffer" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_tag_sample_buffer.txt"
    "GL_SGIX_clipmap" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_clipmap.txt"
    "GL_ARB_fragment_shader_interlock" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_fragment_shader_interlock.txt"
    "GL_INTEL_framebuffer_CMAA" "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_framebuffer_CMAA.txt"
    "GL_SUN_convolution_border_modes" "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_convolution_border_modes.txt"
    "GL_ARB_multisample" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multisample.txt"
    "GL_AMD_multi_draw_indirect" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_multi_draw_indirect.txt"
    "GL_NV_shader_buffer_load" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_buffer_load.txt"
    "WGL_ARB_robustness_share_group_isolation" "https://www.khronos.org/registry/OpenGL/extensions/ARB/WGL_ARB_robustness_application_isolation.txt"
    "GL_APPLE_float_pixels" "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_float_pixels.txt"
    "GL_NV_transform_feedback" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_transform_feedback.txt"
    "GL_APPLE_vertex_array_object" "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_vertex_array_object.txt"
    "GL_ARB_draw_indirect" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_draw_indirect.txt"
    "GL_SGIS_texture_lod" "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_texture_lod.txt"
    "GL_ARB_texture_filter_anisotropic" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_filter_anisotropic.txt"
    "GL_ARB_texture_buffer_object" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_buffer_object.txt"
    "GL_EXT_cull_vertex" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_cull_vertex.txt"
    "GLX_NV_delay_before_swap" "https://www.khronos.org/registry/OpenGL/extensions/NV/GLX_NV_delay_before_swap.txt"
    "GL_EXT_texture_integer" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_integer.txt"
    "GL_EXT_draw_instanced" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_draw_instanced.txt"
    "GL_EXT_index_material" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_index_material.txt"
    "GL_ARB_texture_compression_bptc" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_compression_bptc.txt"
    "GL_EXT_window_rectangles" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_window_rectangles.txt"
    "GL_SGIX_texture_lod_bias" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_texture_lod_bias.txt"
    "GLX_MESA_swap_control" "https://www.khronos.org/registry/OpenGL/extensions/MESA/GLX_MESA_swap_control.txt"
    "GL_EXT_bindable_uniform" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_bindable_uniform.txt"
    "GL_IBM_static_data" "https://www.khronos.org/registry/OpenGL/extensions/IBM/IBM_static_data.txt"
    "GL_3DFX_texture_compression_FXT1" "https://www.khronos.org/registry/OpenGL/extensions/3DFX/3DFX_texture_compression_FXT1.txt"
    "WGL_ARB_multisample" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multisample.txt"
    "GL_ATI_vertex_streams" "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_vertex_streams.txt"
    "GL_ARB_copy_image" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_copy_image.txt"
    "GL_NV_gpu_shader5" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_shader5.txt"
    "GL_ARB_draw_buffers_blend" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_draw_buffers_blend.txt"
    "GL_EXT_framebuffer_object" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_object.txt"
    "GL_MESA_pack_invert" "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_pack_invert.txt"
    "GL_ARB_texture_env_crossbar" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_env_crossbar.txt"
    "GL_ARB_uniform_buffer_object" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_uniform_buffer_object.txt"
    "GL_ARB_fragment_layer_viewport" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_fragment_layer_viewport.txt"
    "GL_NV_geometry_shader_passthrough" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_geometry_shader_passthrough.txt"
    "GL_EXT_multisample" "https://www.khronos.org/registry/OpenGL/extensions/EXT/WGL_EXT_multisample.txt"
    "GL_NV_conservative_raster_dilate" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_conservative_raster_dilate.txt"
    "GL_EXT_texture_compression_s3tc" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_compression_s3tc.txt"
    "GLX_EXT_visual_info" "https://www.khronos.org/registry/OpenGL/extensions/EXT/GLX_EXT_visual_info.txt"
    "GL_ARB_vertex_attrib_64bit" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_attrib_64bit.txt"
    "GL_SGIS_point_line_texgen" "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_point_line_texgen.txt"
    "GL_INTEL_map_texture" "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_map_texture.txt"
    "GL_AMD_shader_trinary_minmax" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_shader_trinary_minmax.txt"
    "GL_NV_evaluators" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_evaluators.txt"
    "GL_NV_register_combiners" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_register_combiners.txt"
    "WGL_ARB_extensions_string" "https://www.khronos.org/registry/OpenGL/extensions/ARB/WGL_ARB_extensions_string.txt"
    "GL_ARB_program_interface_query" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_program_interface_query.txt"
    "GL_AMD_sample_positions" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_sample_positions.txt"
    "GL_ARB_buffer_storage" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_buffer_storage.txt"
    "GL_KHR_context_flush_control" "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_context_flush_control.txt"
    "GL_GREMEDY_frame_terminator" "https://www.khronos.org/registry/OpenGL/extensions/GREMEDY/GREMEDY_frame_terminator.txt"
    "GL_EXT_stencil_clear_tag" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_stencil_clear_tag.txt"
    "GLX_ARB_robustness_share_group_isolation" "https://www.khronos.org/registry/OpenGL/extensions/ARB/GLX_ARB_robustness_application_isolation.txt"
    "GL_ARB_shader_draw_parameters" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_draw_parameters.txt"
    "GL_ARB_sparse_buffer" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sparse_buffer.txt"
    "WGL_EXT_multisample" "https://www.khronos.org/registry/OpenGL/extensions/EXT/WGL_EXT_multisample.txt"
    "GL_NV_texture_env_combine4" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texture_env_combine4.txt"
    "GL_EXT_422_pixels" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_422_pixels.txt"
    "GL_ARB_sparse_texture2" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sparse_texture2.txt"
    "GL_ARB_window_pos" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_window_pos.txt"
    "GL_NV_shading_rate_image" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shading_rate_image.txt"
    "GL_NV_shader_texture_footprint" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_texture_footprint.txt"
    "GL_ARB_shader_clock" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_clock.txt"
    "GL_NV_bindless_multi_draw_indirect" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_bindless_multi_draw_indirect.txt"
    "GLX_INTEL_swap_event" "https://www.khronos.org/registry/OpenGL/extensions/INTEL/GLX_INTEL_swap_event.txt"
    "GL_AMD_draw_buffers_blend" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_draw_buffers_blend.txt"
    "GL_NV_framebuffer_multisample_coverage" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_framebuffer_multisample_coverage.txt"
    "WGL_I3D_swap_frame_usage" "https://www.khronos.org/registry/OpenGL/extensions/I3D/WGL_I3D_swap_frame_usage.txt"
    "WGL_ARB_robustness_application_isolation" "https://www.khronos.org/registry/OpenGL/extensions/ARB/WGL_ARB_robustness_application_isolation.txt"
    "GL_AMD_debug_output" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_debug_output.txt"
    "GLX_MESA_query_renderer" "https://www.khronos.org/registry/OpenGL/extensions/MESA/GLX_MESA_query_renderer.txt"
    "GLX_MESA_set_3dfx_mode" "https://www.khronos.org/registry/OpenGL/extensions/MESA/GLX_MESA_set_3dfx_mode.txt"
    "GL_ARB_indirect_parameters" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_indirect_parameters.txt"
    "GL_EXT_vertex_attrib_64bit" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_attrib_64bit.txt"
    "GL_APPLE_rgb_422" "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_rgb_422.txt"
    "GLX_SGI_cushion" "https://www.khronos.org/registry/OpenGL/extensions/SGI/GLX_SGI_cushion.txt"
    "GL_EXT_blend_func_separate" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_blend_func_separate.txt"
    "GL_HP_occlusion_test" "https://www.khronos.org/registry/OpenGL/extensions/HP/HP_occlusion_test.txt"
    "GL_EXT_geometry_shader4" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_geometry_shader4.txt"
    "GL_ARB_fragment_shader" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_fragment_shader.txt"
    "GL_ARB_compatibility" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_compatibility.txt"
    "GL_ATI_meminfo" "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_meminfo.txt"
    "GL_SGIX_vertex_preclip_hint" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_vertex_preclip.txt"
    "GL_EXT_color_subtable" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_color_subtable.txt"
    "GL_ARB_texture_non_power_of_two" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_non_power_of_two.txt"
    "GL_ARB_shader_atomic_counters" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_atomic_counters.txt"
    "GL_INGR_color_clamp" "https://www.khronos.org/registry/OpenGL/extensions/INGR/INGR_color_clamp.txt"
    "GL_ARB_viewport_array" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_viewport_array.txt"
    "GL_MESAX_texture_stack" "https://www.khronos.org/registry/OpenGL/extensions/MESAX/MESAX_texture_stack.txt"
    "GL_ARB_fragment_program_shadow" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_fragment_program_shadow.txt"
    "GL_AMD_transform_feedback4" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_transform_feedback4.txt"
    "GL_AMD_blend_minmax_factor" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_blend_minmax_factor.txt"
    "GL_ARB_gpu_shader_fp64" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_gpu_shader_fp64.txt"
    "GL_EXT_EGL_image_storage" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_EGL_image_storage.txt"
    "GL_EXT_texture_sRGB" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_sRGB.txt"
    "GL_AMD_framebuffer_multisample_advanced" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_framebuffer_multisample_advanced.txt"
    "GLX_EXT_create_context_es_profile" "https://www.khronos.org/registry/OpenGL/extensions/EXT/GLX_EXT_create_context_es2_profile.txt"
    "GL_SGIS_texture_border_clamp" "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_texture_border_clamp.txt"
    "GL_ARB_map_buffer_range" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_map_buffer_range.txt"
    "GL_ARB_texture_gather" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_gather.txt"
    "GLX_ARB_robustness_application_isolation" "https://www.khronos.org/registry/OpenGL/extensions/ARB/GLX_ARB_robustness_application_isolation.txt"
    "GL_APPLE_vertex_array_range" "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_vertex_array_range.txt"
    "WGL_I3D_gamma" "https://www.khronos.org/registry/OpenGL/extensions/I3D/WGL_I3D_gamma.txt"
    "GL_ARB_color_buffer_float" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_color_buffer_float.txt"
    "GL_ARB_conservative_depth" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_conservative_depth.txt"
    "GL_ARB_query_buffer_object" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_query_buffer_object.txt"
    "WGL_ARB_context_flush_control" "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_context_flush_control.txt"
    "GL_NV_fragment_shader_barycentric" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_fragment_shader_barycentric.txt"
    "GLX_EXT_libglvnd" "https://www.khronos.org/registry/OpenGL/extensions/EXT/GLX_EXT_libglvnd.txt"
    "GL_EXT_polygon_offset_clamp" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_polygon_offset_clamp.txt"
    "GL_NV_blend_equation_advanced" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_blend_equation_advanced.txt"
    "GL_ARB_texture_float" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_float.txt"
    "GL_NV_conservative_raster_pre_snap" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_conservative_raster_pre_snap.txt"
    "GL_AMD_shader_image_load_store_lod" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_shader_image_load_store_lod.txt"
    "GL_NV_multisample_coverage" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_multisample_coverage.txt"
    "GL_ARB_vertex_type_10f_11f_11f_rev" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_10f_11f_11f_rev.txt"
    "GL_EXT_shadow_funcs" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_shadow_funcs.txt"
    "GL_EXT_blend_subtract" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_blend_subtract.txt"
    "GL_NV_path_rendering" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_path_rendering.txt"
    "GLX_NV_multigpu_context" "https://www.khronos.org/registry/OpenGL/extensions/NV/GLX_NV_multigpu_context.txt"
    "GL_NV_fill_rectangle" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_fill_rectangle.txt"
    "GL_WIN_phong_shading" "https://www.khronos.org/registry/OpenGL/extensions/WIN/WIN_phong_shading.txt"
    "GL_ARB_shader_storage_buffer_object" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_storage_buffer_object.txt"
    "GL_AMD_sparse_texture" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_sparse_texture.txt"
    "GL_AMD_pinned_memory" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_pinned_memory.txt"
    "GL_ARB_internalformat_query" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_internalformat_query.txt"
    "GL_ARB_stencil_texturing" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_stencil_texturing.txt"
    "GL_EXT_memory_object_fd" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_external_objects_fd.txt"
    "GL_AMD_seamless_cubemap_per_texture" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_seamless_cubemap_per_texture.txt"
    "GL_ARB_multitexture" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multitexture.txt"
    "GL_ARB_ES3_1_compatibility" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_ES3_1_compatibility.txt"
    "GLX_AMD_gpu_association" "https://www.khronos.org/registry/OpenGL/extensions/AMD/GLX_AMD_gpu_association.txt"
    "GL_NV_texture_compression_vtc" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texture_compression_vtc.txt"
    "GL_KHR_texture_compression_astc_sliced_3d" "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_texture_compression_astc_sliced_3d.txt"
    "GL_ARB_vertex_buffer_object" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_buffer_object.txt"
    "WGL_EXT_colorspace" "https://www.khronos.org/registry/OpenGL/extensions/EXT/WGL_EXT_colorspace.txt"
    "WGL_I3D_image_buffer" "https://www.khronos.org/registry/OpenGL/extensions/I3D/WGL_I3D_image_buffer.txt"
    "GL_NV_texture_shader" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texture_shader.txt"
    "WGL_EXT_make_current_read" "https://www.khronos.org/registry/OpenGL/extensions/EXT/WGL_EXT_make_current_read.txt"
    "GL_ARB_geometry_shader4" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_geometry_shader4.txt"
    "GL_ARB_shader_subroutine" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_subroutine.txt"
    "GL_KHR_no_error" "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_no_error.txt"
    "GL_ARB_sample_shading" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sample_shading.txt"
    "GLX_ARB_create_context" "https://www.khronos.org/registry/OpenGL/extensions/ARB/GLX_ARB_create_context.txt"
    "GLX_SGI_make_current_read" "https://www.khronos.org/registry/OpenGL/extensions/SGI/GLX_SGI_make_current_read.txt"
    "GLX_SGI_video_sync" "https://www.khronos.org/registry/OpenGL/extensions/SGI/GLX_SGI_video_sync.txt"
    "GL_ARB_shading_language_100" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shading_language_100.txt"
    "WGL_ARB_pbuffer" "https://www.khronos.org/registry/OpenGL/extensions/ARB/WGL_ARB_pbuffer.txt"
    "GL_AMD_shader_explicit_vertex_parameter" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_shader_explicit_vertex_parameter.txt"
    "GL_AMD_conservative_depth" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_conservative_depth.txt"
    "GL_NV_conditional_render" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_conditional_render.txt"
    "GL_SGIX_texture_coordinate_clamp" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_texture_coordinate_clamp.txt"
    "GL_NV_multisample_filter_hint" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_multisample_filter_hint.txt"
    "GL_SGIX_shadow_ambient" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_shadow_ambient.txt"
    "GL_EXT_transform_feedback" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_transform_feedback.txt"
    "GL_NV_texture_rectangle" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texture_rectangle.txt"
    "GL_ARB_shader_image_load_store" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_image_load_store.txt"
    "GL_EXT_shader_integer_mix" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_shader_integer_mix.txt"
    "GL_APPLE_fence" "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_fence.txt"
    "GL_AMD_vertex_shader_viewport_index" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_vertex_shader_viewport_index.txt"
    "GL_SGIX_resample" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_resample.txt"
    "GL_EXT_vertex_weighting" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_weighting.txt"
    "GL_ARB_debug_output" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_debug_output.txt"
    "GL_EXT_vertex_shader" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_shader.txt"
    "GL_SGIX_fog_offset" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_fog_offset.txt"
    "GL_IBM_multimode_draw_arrays" "https://www.khronos.org/registry/OpenGL/extensions/IBM/IBM_multimode_draw_arrays.txt"
    "GLU_EXT_object_space_tess" "https://www.khronos.org/registry/OpenGL/extensions/EXT/GLU_EXT_object_space_tess.txt"
    "GL_NV_tessellation_program5" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_tessellation_program5.txt"
    "GL_AMD_vertex_shader_tessellator" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_vertex_shader_tessellator.txt"
    "GL_NV_fragment_shader_interlock" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_fragment_shader_interlock.txt"
    "GL_SGIX_list_priority" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_list_priority.txt"
    "GL_EXT_misc_attribute" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_misc_attribute.txt"
    "GL_ARB_spirv_extensions" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_spirv_extensions.txt"
    "WGL_EXT_create_context_es2_profile" "https://www.khronos.org/registry/OpenGL/extensions/EXT/WGL_EXT_create_context_es2_profile.txt"
    "GL_MESA_ycbcr_texture" "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_ycbcr_texture.txt"
    "GLX_SGIX_fbconfig" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/GLX_SGIX_fbconfig.txt"
    "WGL_ARB_create_context_no_error" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_create_context_no_error.txt"
    "GL_ARB_shading_language_420pack" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shading_language_420pack.txt"
    "GL_ARB_shader_texture_image_samples" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_texture_image_samples.txt"
    "GL_PGI_vertex_hints" "https://www.khronos.org/registry/OpenGL/extensions/PGI/PGI_vertex_hints.txt"
    "GL_ARB_sparse_texture" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sparse_texture.txt"
    "GL_ATI_pn_triangles" "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_pn_triangles.txt"
    "WGL_ARB_render_texture" "https://www.khronos.org/registry/OpenGL/extensions/ARB/WGL_ARB_render_texture.txt"
    "GL_NV_clip_space_w_scaling" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_clip_space_w_scaling.txt"
    "GL_EXT_bgra" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_bgra.txt"
    "GL_NV_video_capture" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_video_capture.txt"
    "GL_EXT_packed_float" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_packed_float.txt"
    "GLX_EXT_buffer_age" "https://www.khronos.org/registry/OpenGL/extensions/EXT/GLX_EXT_buffer_age.txt"
    "GL_EXT_gpu_program_parameters" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_gpu_program_parameters.txt"
    "GLX_NV_copy_buffer" "https://www.khronos.org/registry/OpenGL/extensions/NV/GLX_NV_copy_buffer.txt"
    "WGL_ATI_render_texture_rectangle" "https://www.khronos.org/registry/OpenGL/extensions/ATI/WGL_ATI_render_texture_rectangle.txt"
    "GL_SGIX_reference_plane" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_reference_plane.txt"
    "GL_APPLE_element_array" "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_element_array.txt"
    "GL_KHR_shader_subgroup" "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_shader_subgroup.txt"
    "GL_ARB_map_buffer_alignment" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_map_buffer_alignment.txt"
    "GL_ARB_clear_buffer_object" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_clear_buffer_object.txt"
    "GLX_MESA_agp_offset" "https://www.khronos.org/registry/OpenGL/extensions/MESA/GLX_MESA_agp_offset.txt"
    "GLX_ARB_create_context_robustness" "https://www.khronos.org/registry/OpenGL/extensions/ARB/GLX_ARB_create_context_robustness.txt"
    "GLX_OML_sync_control" "https://www.khronos.org/registry/OpenGL/extensions/OML/GLX_OML_sync_control.txt"
    "GL_NV_timeline_semaphore" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_timeline_semaphore.txt"
    "GL_NV_alpha_to_coverage_dither_control" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_alpha_to_coverage_dither_control.txt"
    "GL_OML_interlace" "https://www.khronos.org/registry/OpenGL/extensions/OML/OML_interlace.txt"
    "GL_MESA_window_pos" "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_window_pos.txt"
    "GL_SGIX_texture_multi_buffer" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_texture_multi_buffer.txt"
    "GL_ARB_texture_compression" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_compression.txt"
    "GL_EXT_memory_object" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_external_objects.txt"
    "GL_EXT_texture_mirror_clamp" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_mirror_clamp.txt"
    "GL_EXT_rescale_normal" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_rescale_normal.txt"
    "GL_NV_query_resource" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_query_resource.txt"
    "GL_APPLE_flush_buffer_range" "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_flush_buffer_range.txt"
    "GL_EXT_compiled_vertex_array" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_compiled_vertex_array.txt"
    "GL_ARB_texture_cube_map" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_cube_map.txt"
    "GL_ARB_vertex_type_2_10_10_10_rev" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_type_2_10_10_10_rev.txt"
    "GL_EXT_texture_sRGB_decode" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_sRGB_decode.txt"
    "WGL_EXT_framebuffer_sRGB" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_framebuffer_sRGB.txt"
    "GL_NV_present_video" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_present_video.txt"
    "GL_EXT_texture_env_dot3" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_env_dot3.txt"
    "GL_MESA_program_binary_formats" "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_program_binary_formats.txt"
    "GLU_SGI_filter4_parameters" "https://www.khronos.org/registry/OpenGL/extensions/SGI/GLU_SGI_filter4_parameters.txt"
    "GL_EXT_semaphore_fd" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_external_objects_fd.txt"
    "GL_SGIX_pixel_texture_bits" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_pixel_texture_bits.txt"
    "GL_EXT_texture_sRGB_RG8" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_sRGB_RG8.txt"
    "GL_EXT_debug_label" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_debug_label.txt"
    "GL_AMD_depth_clamp_separate" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_depth_clamp_separate.txt"
    "GL_ARB_derivative_control" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_derivative_control.txt"
    "GL_PGI_misc_hints" "https://www.khronos.org/registry/OpenGL/extensions/PGI/PGI_misc_hints.txt"
    "WGL_I3D_genlock" "https://www.khronos.org/registry/OpenGL/extensions/I3D/WGL_I3D_genlock.txt"
    "GLX_SGIX_swap_group" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/GLX_SGIX_swap_group.txt"
    "GL_ARB_enhanced_layouts" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_enhanced_layouts.txt"
    "GL_EXT_multiview_timer_query" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_multiview_timer_query.txt"
    "WGL_OML_sync_control" "https://www.khronos.org/registry/OpenGL/extensions/OML/WGL_OML_sync_control.txt"
    "GL_NV_register_combiners2" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_register_combiners2.txt"
    "GL_ARB_texture_buffer_object_rgb32" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_buffer_object_rgb32.txt"
    "GL_OES_byte_coordinates" "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_byte_coordinates.txt"
    "GL_EXT_shader_framebuffer_fetch_non_coherent" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_shader_framebuffer_fetch.txt"
    "GL_NVX_blend_equation_advanced_multi_draw_buffers" "https://www.khronos.org/registry/OpenGL/extensions/NVX/NVX_blend_equation_advanced_multi_draw_buffers.txt"
    "WGL_EXT_extensions_string" "https://www.khronos.org/registry/OpenGL/extensions/EXT/WGL_EXT_extensions_string.txt"
    "GL_EXT_pixel_transform" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_pixel_transform.txt"
    "GL_AMD_texture_texture4" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_texture_texture4.txt"
    "GLX_SGIX_video_resize_float" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/GLX_SGIX_video_resize_float.txt"
    "GL_EXT_scene_marker" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_scene_marker.txt"
    "GL_ARB_framebuffer_object" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_object.txt"
    "GL_AMD_interleaved_elements" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_interleaved_elements.txt"
    "GLX_EXT_visual_rating" "https://www.khronos.org/registry/OpenGL/extensions/EXT/GLX_EXT_visual_rating.txt"
    "GL_ARB_texture_mirrored_repeat" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_mirrored_repeat.txt"
    "GLX_EXT_create_context_es2_profile" "https://www.khronos.org/registry/OpenGL/extensions/EXT/GLX_EXT_create_context_es2_profile.txt"
    "GLX_ARB_vertex_buffer_object" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_buffer_object.txt"
    "GL_ARB_ES3_2_compatibility" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_ES3_2_compatibility.txt"
    "GL_ARB_texture_buffer_range" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_buffer_range.txt"
    "GL_NV_compute_shader_derivatives" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_compute_shader_derivatives.txt"
    "GL_AMD_stencil_operation_extended" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_stencil_operation_extended.txt"
    "GL_NV_half_float" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_half_float.txt"
    "GL_GREMEDY_string_marker" "https://www.khronos.org/registry/OpenGL/extensions/GREMEDY/GREMEDY_string_marker.txt"
    "GL_NV_depth_clamp" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_depth_clamp.txt"
    "GL_SGIX_convolution_accuracy" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_convolution_accuracy.txt"
    "GL_KHR_robust_buffer_access_behavior" "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_robust_buffer_access_behavior.txt"
    "GL_EXT_texture_storage" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_storage.txt"
    "WGL_ARB_framebuffer_sRGB" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_framebuffer_sRGB.txt"
    "GL_ARB_texture_env_dot3" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_env_dot3.txt"
    "GL_EXT_texture_snorm" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_snorm.txt"
    "GL_ARB_depth_buffer_float" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_depth_buffer_float.txt"
    "GL_AMD_texture_gather_bias_lod" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_texture_gather_bias_lod.txt"
    "GL_NV_fragment_program2" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_fragment_program2.txt"
    "GL_SUN_mesh_array" "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_mesh_array.txt"
    "GL_NV_draw_texture" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_draw_texture.txt"
    "GL_3DFX_multisample" "https://www.khronos.org/registry/OpenGL/extensions/3DFX/3DFX_multisample.txt"
    "GL_NV_fog_distance" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_fog_distance.txt"
    "GL_EXT_clip_volume_hint" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_clip_volume_hint.txt"
    "GLX_SGIX_video_source" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/GLX_SGIX_video_source.txt"
    "GL_ARB_post_depth_coverage" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_post_depth_coverage.txt"
    "GL_ARB_internalformat_query2" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_internalformat_query2.txt"
    "GLX_EXT_get_drawable_type" "https://www.khronos.org/registry/OpenGL/extensions/EXT/GLX_EXT_get_drawable_type.txt"
    "GL_NV_shader_subgroup_partitioned" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_subgroup_partitioned.txt"
    "GLX_NV_swap_group" "https://www.khronos.org/registry/OpenGL/extensions/NV/GLX_NV_swap_group.txt"
    "GL_NV_fragment_program4" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_fragment_program4.txt"
    "GL_APPLE_ycbcr_422" "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_ycbcr_422.txt"
    "GL_ATI_text_fragment_shader" "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_text_fragment_shader.txt"
    "GL_NV_shader_atomic_fp16_vector" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_atomic_fp16_vector.txt"
    "GL_EXT_vertex_array" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_vertex_array.txt"
    "GL_AMD_shader_ballot" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_shader_ballot.txt"
    "GL_OES_compressed_paletted_texture" "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_compressed_paletted_texture.txt"
    "GL_ATI_texture_env_combine3" "https://www.khronos.org/registry/OpenGL/extensions/ATI/ATI_texture_env_combine3.txt"
    "WGL_ARB_pixel_format" "https://www.khronos.org/registry/OpenGL/extensions/ARB/WGL_ARB_pixel_format.txt"
    "GL_ARB_draw_elements_base_vertex" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_draw_elements_base_vertex.txt"
    "GL_ARB_compressed_texture_pixel_storage" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_compressed_texture_pixel_storage.txt"
    "GL_EXT_stencil_two_side" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_stencil_two_side.txt"
    "GLX_NV_copy_image" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_copy_image.txt"
    "GL_ARB_vertex_shader" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_shader.txt"
    "GL_EXT_texture_env_add" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_env_add.txt"
    "GL_EXT_texture_env" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_env.txt"
    "GL_SGIX_async_pixel" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_async_pixel.txt"
    "GL_NV_pixel_data_range" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_pixel_data_range.txt"
    "GL_SGIS_multisample" "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_multisample.txt"
    "GL_NV_conservative_raster_pre_snap_triangles" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_conservative_raster_pre_snap_triangles.txt"
    "GL_NV_vertex_attrib_integer_64bit" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_vertex_attrib_integer_64bit.txt"
    "GL_NV_texture_expand_normal" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texture_expand_normal.txt"
    "GL_NV_stereo_view_rendering" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_stereo_view_rendering.txt"
    "GL_EXT_multiview_tessellation_geometry_shader" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_multiview_tessellation_geometry_shader.txt"
    "GL_EXT_post_depth_coverage" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_post_depth_coverage.txt"
    "GL_NV_shader_thread_shuffle" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_thread_shuffle.txt"
    "WGL_NV_delay_before_swap" "https://www.khronos.org/registry/OpenGL/extensions/NV/WGL_NV_delay_before_swap.txt"
    "GL_NV_parameter_buffer_object2" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_parameter_buffer_object2.txt"
    "GLX_NV_video_capture" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_video_capture.txt"
    "GL_EXT_packed_depth_stencil" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_packed_depth_stencil.txt"
    "GL_NVX_gpu_multicast2" "https://www.khronos.org/registry/OpenGL/extensions/NVX/NVX_gpu_multicast2.txt"
    "WGL_NV_render_texture_rectangle" "https://www.khronos.org/registry/OpenGL/extensions/NV/WGL_NV_render_texture_rectangle.txt"
    "GL_NV_uniform_buffer_unified_memory" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_uniform_buffer_unified_memory.txt"
    "WGL_NV_gpu_affinity" "https://www.khronos.org/registry/OpenGL/extensions/NV/WGL_NV_gpu_affinity.txt"
    "GL_NV_occlusion_query" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_occlusion_query.txt"
    "GL_EXT_texture_swizzle" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_swizzle.txt"
    "GL_SGI_color_table" "https://www.khronos.org/registry/OpenGL/extensions/SGI/SGI_color_table.txt"
    "WGL_ARB_buffer_region" "https://www.khronos.org/registry/OpenGL/extensions/ARB/WGL_ARB_buffer_region.txt"
    "WGL_NV_video_output" "https://www.khronos.org/registry/OpenGL/extensions/NV/WGL_NV_video_output.txt"
    "GL_NV_geometry_shader4" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_geometry_shader4.txt"
    "GL_IBM_vertex_array_lists" "https://www.khronos.org/registry/OpenGL/extensions/IBM/IBM_vertex_array_lists.txt"
    "GL_INTEL_texture_scissor" "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_texture_scissor.txt"
    "GL_AMD_gpu_shader_half_float" "https://www.khronos.org/registry/OpenGL/extensions/AMD/AMD_gpu_shader_half_float.txt"
    "GL_OES_read_format" "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_read_format.txt"
    "GL_SGIX_texture_add_env" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_texture_add_env.txt"
    "WGL_ARB_create_context_profile" "https://www.khronos.org/registry/OpenGL/extensions/ARB/WGL_ARB_create_context.txt"
    "GL_KHR_blend_equation_advanced" "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_blend_equation_advanced.txt"
    "GL_EXT_semaphore" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_external_objects.txt"
    "GL_NV_compute_program5" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_compute_program5.txt"
    "GL_ARB_shader_viewport_layer_array" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shader_viewport_layer_array.txt"
    "GL_EXT_shared_texture_palette" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_shared_texture_palette.txt"
    "GL_NV_texgen_emboss" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texgen_emboss.txt"
    "GL_EXT_blend_minmax" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_blend_minmax.txt"
    "GL_ARB_sparse_texture_clamp" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_sparse_texture_clamp.txt"
    "GL_ARB_texture_cube_map_array" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_cube_map_array.txt"
    "GL_EXT_texture_filter_minmax" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_texture_filter_minmax.txt"
    "GL_OES_fixed_point" "https://www.khronos.org/registry/OpenGL/extensions/OES/OES_fixed_point.txt"
    "GL_ARB_shading_language_packing" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_shading_language_packing.txt"
    "GL_NV_texture_barrier" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texture_barrier.txt"
    "GLX_NV_present_video" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_present_video.txt"
    "WGL_EXT_display_color_table" "https://www.khronos.org/registry/OpenGL/extensions/EXT/WGL_EXT_display_color_table.txt"
    "GL_SUN_vertex" "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_vertex.txt"
    "GL_EXT_index_func" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_index_func.txt"
    "GL_EXT_subtexture" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_subtexture.txt"
    "GL_EXT_x11_sync_object" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_x11_sync_object.txt"
    "GL_EXT_separate_shader_objects" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_separate_shader_objects.gl.txt"
    "WGL_NV_video_capture" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_video_capture.txt"
    "GL_NV_texture_shader2" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texture_shader2.txt"
    "GL_EXT_abgr" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_abgr.txt"
    "GL_ARB_blend_func_extended" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_blend_func_extended.txt"
    "GL_ARB_transform_feedback_instanced" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_transform_feedback_instanced.txt"
    "GL_ARB_robust_buffer_access_behavior" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_robust_buffer_access_behavior.txt"
    "GL_KHR_robustness" "https://www.khronos.org/registry/OpenGL/extensions/KHR/KHR_robustness.txt"
    "GL_NV_texture_shader3" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_texture_shader3.txt"
    "GL_EXT_point_parameters" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_point_parameters.txt"
    "GL_NV_representative_fragment_test" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_representative_fragment_test.txt"
    "GL_MESA_tile_raster_order" "https://www.khronos.org/registry/OpenGL/extensions/MESA/MESA_tile_raster_order.txt"
    "GL_APPLE_texture_range" "https://www.khronos.org/registry/OpenGL/extensions/APPLE/APPLE_texture_range.txt"
    "WGL_NV_render_depth_texture" "https://www.khronos.org/registry/OpenGL/extensions/NV/WGL_NV_render_depth_texture.txt"
    "GL_SGIS_texture_edge_clamp" "https://www.khronos.org/registry/OpenGL/extensions/SGIS/SGIS_texture_edge_clamp.txt"
    "GL_ARB_vertex_array_object" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_vertex_array_object.txt"
    "GL_NV_memory_object_sparse" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_memory_object_sparse.txt"
    "GL_ARB_matrix_palette" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_matrix_palette.txt"
    "GL_ARB_conditional_render_inverted" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_conditional_render_inverted.txt"
    "GL_ARB_get_texture_sub_image" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_get_texture_sub_image.txt"
    "GLX_SGIX_hyperpipe" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/GLX_SGIX_hyperpipe.txt"
    "GL_ARB_direct_state_access" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_direct_state_access.txt"
    "GL_INTEL_shader_integer_functions2" "https://www.khronos.org/registry/OpenGL/extensions/INTEL/INTEL_shader_integer_functions2.txt"
    "GL_NV_blend_equation_advanced_coherent" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_blend_equation_advanced.txt"
    "GL_ARB_texture_mirror_clamp_to_edge" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_texture_mirror_clamp_to_edge.txt"
    "GLX_OML_swap_method" "https://www.khronos.org/registry/OpenGL/extensions/OML/GLX_OML_swap_method.txt"
    "GL_EXT_coordinate_frame" "https://www.khronos.org/registry/OpenGL/extensions/EXT/EXT_coordinate_frame.txt"
    "GL_NV_shader_atomic_float" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_shader_atomic_float.txt"
    "GL_HP_convolution_border_modes" "https://www.khronos.org/registry/OpenGL/extensions/HP/HP_convolution_border_modes.txt"
    "GL_NV_gpu_multicast" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_gpu_multicast.txt"
    "GL_NV_fence" "https://www.khronos.org/registry/OpenGL/extensions/NV/NV_fence.txt"
    "GL_SGIX_flush_raster" "https://www.khronos.org/registry/OpenGL/extensions/SGIX/SGIX_flush_raster.txt"
    "GL_ARB_seamless_cube_map" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_seamless_cube_map.txt"
    "GL_ARB_cl_event" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_cl_event.txt"
    "GL_SUN_slice_accum" "https://www.khronos.org/registry/OpenGL/extensions/SUN/SUN_slice_accum.txt"
    "GL_ARB_multi_draw_indirect" "https://www.khronos.org/registry/OpenGL/extensions/ARB/ARB_multi_draw_indirect.txt"
}

# Array of enums and corresponding OpenGL version or extension.
array set ::__tcl3dOglEnumVersion {
    "GL_VERSION_1_1" "GL_VERSION_1_1"
    "GL_ZERO" "GL_VERSION_1_1"
    "GL_FALSE" "GL_VERSION_1_1"
    "GL_LOGIC_OP" "GL_VERSION_1_1"
    "GL_NONE" "GL_VERSION_1_1"
    "GL_TEXTURE_COMPONENTS" "GL_VERSION_1_1"
    "GL_NO_ERROR" "GL_VERSION_1_1"
    "GL_POINTS" "GL_VERSION_1_1"
    "GL_CURRENT_BIT" "GL_VERSION_1_1"
    "GL_TRUE" "GL_VERSION_1_1"
    "GL_ONE" "GL_VERSION_1_1"
    "GL_CLIENT_PIXEL_STORE_BIT" "GL_VERSION_1_1"
    "GL_LINES" "GL_VERSION_1_1"
    "GL_LINE_LOOP" "GL_VERSION_1_1"
    "GL_POINT_BIT" "GL_VERSION_1_1"
    "GL_CLIENT_VERTEX_ARRAY_BIT" "GL_VERSION_1_1"
    "GL_LINE_STRIP" "GL_VERSION_1_1"
    "GL_LINE_BIT" "GL_VERSION_1_1"
    "GL_TRIANGLES" "GL_VERSION_1_1"
    "GL_TRIANGLE_STRIP" "GL_VERSION_1_1"
    "GL_TRIANGLE_FAN" "GL_VERSION_1_1"
    "GL_QUADS" "GL_VERSION_1_1"
    "GL_QUAD_STRIP" "GL_VERSION_1_1"
    "GL_POLYGON_BIT" "GL_VERSION_1_1"
    "GL_POLYGON" "GL_VERSION_1_1"
    "GL_POLYGON_STIPPLE_BIT" "GL_VERSION_1_1"
    "GL_PIXEL_MODE_BIT" "GL_VERSION_1_1"
    "GL_LIGHTING_BIT" "GL_VERSION_1_1"
    "GL_FOG_BIT" "GL_VERSION_1_1"
    "GL_DEPTH_BUFFER_BIT" "GL_VERSION_1_1"
    "GL_ACCUM" "GL_VERSION_1_1"
    "GL_LOAD" "GL_VERSION_1_1"
    "GL_RETURN" "GL_VERSION_1_1"
    "GL_MULT" "GL_VERSION_1_1"
    "GL_ADD" "GL_VERSION_1_1"
    "GL_NEVER" "GL_VERSION_1_1"
    "GL_ACCUM_BUFFER_BIT" "GL_VERSION_1_1"
    "GL_LESS" "GL_VERSION_1_1"
    "GL_EQUAL" "GL_VERSION_1_1"
    "GL_LEQUAL" "GL_VERSION_1_1"
    "GL_GREATER" "GL_VERSION_1_1"
    "GL_NOTEQUAL" "GL_VERSION_1_1"
    "GL_GEQUAL" "GL_VERSION_1_1"
    "GL_ALWAYS" "GL_VERSION_1_1"
    "GL_SRC_COLOR" "GL_VERSION_1_1"
    "GL_ONE_MINUS_SRC_COLOR" "GL_VERSION_1_1"
    "GL_SRC_ALPHA" "GL_VERSION_1_1"
    "GL_ONE_MINUS_SRC_ALPHA" "GL_VERSION_1_1"
    "GL_DST_ALPHA" "GL_VERSION_1_1"
    "GL_ONE_MINUS_DST_ALPHA" "GL_VERSION_1_1"
    "GL_DST_COLOR" "GL_VERSION_1_1"
    "GL_ONE_MINUS_DST_COLOR" "GL_VERSION_1_1"
    "GL_SRC_ALPHA_SATURATE" "GL_VERSION_1_1"
    "GL_STENCIL_BUFFER_BIT" "GL_VERSION_1_1"
    "GL_FRONT_LEFT" "GL_VERSION_1_1"
    "GL_FRONT_RIGHT" "GL_VERSION_1_1"
    "GL_BACK_LEFT" "GL_VERSION_1_1"
    "GL_BACK_RIGHT" "GL_VERSION_1_1"
    "GL_FRONT" "GL_VERSION_1_1"
    "GL_BACK" "GL_VERSION_1_1"
    "GL_LEFT" "GL_VERSION_1_1"
    "GL_RIGHT" "GL_VERSION_1_1"
    "GL_FRONT_AND_BACK" "GL_VERSION_1_1"
    "GL_AUX0" "GL_VERSION_1_1"
    "GL_AUX1" "GL_VERSION_1_1"
    "GL_AUX2" "GL_VERSION_1_1"
    "GL_AUX3" "GL_VERSION_1_1"
    "GL_INVALID_ENUM" "GL_VERSION_1_1"
    "GL_INVALID_VALUE" "GL_VERSION_1_1"
    "GL_INVALID_OPERATION" "GL_VERSION_1_1"
    "GL_STACK_OVERFLOW" "GL_VERSION_1_1"
    "GL_STACK_UNDERFLOW" "GL_VERSION_1_1"
    "GL_OUT_OF_MEMORY" "GL_VERSION_1_1"
    "GL_2D" "GL_VERSION_1_1"
    "GL_3D" "GL_VERSION_1_1"
    "GL_3D_COLOR" "GL_VERSION_1_1"
    "GL_3D_COLOR_TEXTURE" "GL_VERSION_1_1"
    "GL_4D_COLOR_TEXTURE" "GL_VERSION_1_1"
    "GL_PASS_THROUGH_TOKEN" "GL_VERSION_1_1"
    "GL_POINT_TOKEN" "GL_VERSION_1_1"
    "GL_LINE_TOKEN" "GL_VERSION_1_1"
    "GL_POLYGON_TOKEN" "GL_VERSION_1_1"
    "GL_BITMAP_TOKEN" "GL_VERSION_1_1"
    "GL_DRAW_PIXEL_TOKEN" "GL_VERSION_1_1"
    "GL_COPY_PIXEL_TOKEN" "GL_VERSION_1_1"
    "GL_LINE_RESET_TOKEN" "GL_VERSION_1_1"
    "GL_EXP" "GL_VERSION_1_1"
    "GL_VIEWPORT_BIT" "GL_VERSION_1_1"
    "GL_EXP2" "GL_VERSION_1_1"
    "GL_CW" "GL_VERSION_1_1"
    "GL_CCW" "GL_VERSION_1_1"
    "GL_COEFF" "GL_VERSION_1_1"
    "GL_ORDER" "GL_VERSION_1_1"
    "GL_DOMAIN" "GL_VERSION_1_1"
    "GL_CURRENT_COLOR" "GL_VERSION_1_1"
    "GL_CURRENT_INDEX" "GL_VERSION_1_1"
    "GL_CURRENT_NORMAL" "GL_VERSION_1_1"
    "GL_CURRENT_TEXTURE_COORDS" "GL_VERSION_1_1"
    "GL_CURRENT_RASTER_COLOR" "GL_VERSION_1_1"
    "GL_CURRENT_RASTER_INDEX" "GL_VERSION_1_1"
    "GL_CURRENT_RASTER_TEXTURE_COORDS" "GL_VERSION_1_1"
    "GL_CURRENT_RASTER_POSITION" "GL_VERSION_1_1"
    "GL_CURRENT_RASTER_POSITION_VALID" "GL_VERSION_1_1"
    "GL_CURRENT_RASTER_DISTANCE" "GL_VERSION_1_1"
    "GL_POINT_SMOOTH" "GL_VERSION_1_1"
    "GL_POINT_SIZE" "GL_VERSION_1_1"
    "GL_POINT_SIZE_RANGE" "GL_VERSION_1_1"
    "GL_POINT_SIZE_GRANULARITY" "GL_VERSION_1_1"
    "GL_LINE_SMOOTH" "GL_VERSION_1_1"
    "GL_LINE_WIDTH" "GL_VERSION_1_1"
    "GL_LINE_WIDTH_RANGE" "GL_VERSION_1_1"
    "GL_LINE_WIDTH_GRANULARITY" "GL_VERSION_1_1"
    "GL_LINE_STIPPLE" "GL_VERSION_1_1"
    "GL_LINE_STIPPLE_PATTERN" "GL_VERSION_1_1"
    "GL_LINE_STIPPLE_REPEAT" "GL_VERSION_1_1"
    "GL_LIST_MODE" "GL_VERSION_1_1"
    "GL_MAX_LIST_NESTING" "GL_VERSION_1_1"
    "GL_LIST_BASE" "GL_VERSION_1_1"
    "GL_LIST_INDEX" "GL_VERSION_1_1"
    "GL_POLYGON_MODE" "GL_VERSION_1_1"
    "GL_POLYGON_SMOOTH" "GL_VERSION_1_1"
    "GL_POLYGON_STIPPLE" "GL_VERSION_1_1"
    "GL_EDGE_FLAG" "GL_VERSION_1_1"
    "GL_CULL_FACE" "GL_VERSION_1_1"
    "GL_CULL_FACE_MODE" "GL_VERSION_1_1"
    "GL_FRONT_FACE" "GL_VERSION_1_1"
    "GL_LIGHTING" "GL_VERSION_1_1"
    "GL_LIGHT_MODEL_LOCAL_VIEWER" "GL_VERSION_1_1"
    "GL_LIGHT_MODEL_TWO_SIDE" "GL_VERSION_1_1"
    "GL_LIGHT_MODEL_AMBIENT" "GL_VERSION_1_1"
    "GL_SHADE_MODEL" "GL_VERSION_1_1"
    "GL_COLOR_MATERIAL_FACE" "GL_VERSION_1_1"
    "GL_COLOR_MATERIAL_PARAMETER" "GL_VERSION_1_1"
    "GL_COLOR_MATERIAL" "GL_VERSION_1_1"
    "GL_FOG" "GL_VERSION_1_1"
    "GL_FOG_INDEX" "GL_VERSION_1_1"
    "GL_FOG_DENSITY" "GL_VERSION_1_1"
    "GL_FOG_START" "GL_VERSION_1_1"
    "GL_FOG_END" "GL_VERSION_1_1"
    "GL_FOG_MODE" "GL_VERSION_1_1"
    "GL_FOG_COLOR" "GL_VERSION_1_1"
    "GL_DEPTH_RANGE" "GL_VERSION_1_1"
    "GL_DEPTH_TEST" "GL_VERSION_1_1"
    "GL_DEPTH_WRITEMASK" "GL_VERSION_1_1"
    "GL_DEPTH_CLEAR_VALUE" "GL_VERSION_1_1"
    "GL_DEPTH_FUNC" "GL_VERSION_1_1"
    "GL_ACCUM_CLEAR_VALUE" "GL_VERSION_1_1"
    "GL_STENCIL_TEST" "GL_VERSION_1_1"
    "GL_STENCIL_CLEAR_VALUE" "GL_VERSION_1_1"
    "GL_STENCIL_FUNC" "GL_VERSION_1_1"
    "GL_STENCIL_VALUE_MASK" "GL_VERSION_1_1"
    "GL_STENCIL_FAIL" "GL_VERSION_1_1"
    "GL_STENCIL_PASS_DEPTH_FAIL" "GL_VERSION_1_1"
    "GL_STENCIL_PASS_DEPTH_PASS" "GL_VERSION_1_1"
    "GL_STENCIL_REF" "GL_VERSION_1_1"
    "GL_STENCIL_WRITEMASK" "GL_VERSION_1_1"
    "GL_MATRIX_MODE" "GL_VERSION_1_1"
    "GL_NORMALIZE" "GL_VERSION_1_1"
    "GL_VIEWPORT" "GL_VERSION_1_1"
    "GL_MODELVIEW_STACK_DEPTH" "GL_VERSION_1_1"
    "GL_PROJECTION_STACK_DEPTH" "GL_VERSION_1_1"
    "GL_TEXTURE_STACK_DEPTH" "GL_VERSION_1_1"
    "GL_MODELVIEW_MATRIX" "GL_VERSION_1_1"
    "GL_PROJECTION_MATRIX" "GL_VERSION_1_1"
    "GL_TEXTURE_MATRIX" "GL_VERSION_1_1"
    "GL_ATTRIB_STACK_DEPTH" "GL_VERSION_1_1"
    "GL_CLIENT_ATTRIB_STACK_DEPTH" "GL_VERSION_1_1"
    "GL_ALPHA_TEST" "GL_VERSION_1_1"
    "GL_ALPHA_TEST_FUNC" "GL_VERSION_1_1"
    "GL_ALPHA_TEST_REF" "GL_VERSION_1_1"
    "GL_DITHER" "GL_VERSION_1_1"
    "GL_BLEND_DST" "GL_VERSION_1_1"
    "GL_BLEND_SRC" "GL_VERSION_1_1"
    "GL_BLEND" "GL_VERSION_1_1"
    "GL_LOGIC_OP_MODE" "GL_VERSION_1_1"
    "GL_INDEX_LOGIC_OP" "GL_VERSION_1_1"
    "GL_COLOR_LOGIC_OP" "GL_VERSION_1_1"
    "GL_AUX_BUFFERS" "GL_VERSION_1_1"
    "GL_DRAW_BUFFER" "GL_VERSION_1_1"
    "GL_READ_BUFFER" "GL_VERSION_1_1"
    "GL_SCISSOR_BOX" "GL_VERSION_1_1"
    "GL_SCISSOR_TEST" "GL_VERSION_1_1"
    "GL_INDEX_CLEAR_VALUE" "GL_VERSION_1_1"
    "GL_INDEX_WRITEMASK" "GL_VERSION_1_1"
    "GL_COLOR_CLEAR_VALUE" "GL_VERSION_1_1"
    "GL_COLOR_WRITEMASK" "GL_VERSION_1_1"
    "GL_INDEX_MODE" "GL_VERSION_1_1"
    "GL_RGBA_MODE" "GL_VERSION_1_1"
    "GL_DOUBLEBUFFER" "GL_VERSION_1_1"
    "GL_STEREO" "GL_VERSION_1_1"
    "GL_RENDER_MODE" "GL_VERSION_1_1"
    "GL_PERSPECTIVE_CORRECTION_HINT" "GL_VERSION_1_1"
    "GL_POINT_SMOOTH_HINT" "GL_VERSION_1_1"
    "GL_LINE_SMOOTH_HINT" "GL_VERSION_1_1"
    "GL_POLYGON_SMOOTH_HINT" "GL_VERSION_1_1"
    "GL_FOG_HINT" "GL_VERSION_1_1"
    "GL_TEXTURE_GEN_S" "GL_VERSION_1_1"
    "GL_TEXTURE_GEN_T" "GL_VERSION_1_1"
    "GL_TEXTURE_GEN_R" "GL_VERSION_1_1"
    "GL_TEXTURE_GEN_Q" "GL_VERSION_1_1"
    "GL_PIXEL_MAP_I_TO_I" "GL_VERSION_1_1"
    "GL_PIXEL_MAP_S_TO_S" "GL_VERSION_1_1"
    "GL_PIXEL_MAP_I_TO_R" "GL_VERSION_1_1"
    "GL_PIXEL_MAP_I_TO_G" "GL_VERSION_1_1"
    "GL_PIXEL_MAP_I_TO_B" "GL_VERSION_1_1"
    "GL_PIXEL_MAP_I_TO_A" "GL_VERSION_1_1"
    "GL_PIXEL_MAP_R_TO_R" "GL_VERSION_1_1"
    "GL_PIXEL_MAP_G_TO_G" "GL_VERSION_1_1"
    "GL_PIXEL_MAP_B_TO_B" "GL_VERSION_1_1"
    "GL_PIXEL_MAP_A_TO_A" "GL_VERSION_1_1"
    "GL_PIXEL_MAP_I_TO_I_SIZE" "GL_VERSION_1_1"
    "GL_PIXEL_MAP_S_TO_S_SIZE" "GL_VERSION_1_1"
    "GL_PIXEL_MAP_I_TO_R_SIZE" "GL_VERSION_1_1"
    "GL_PIXEL_MAP_I_TO_G_SIZE" "GL_VERSION_1_1"
    "GL_PIXEL_MAP_I_TO_B_SIZE" "GL_VERSION_1_1"
    "GL_PIXEL_MAP_I_TO_A_SIZE" "GL_VERSION_1_1"
    "GL_PIXEL_MAP_R_TO_R_SIZE" "GL_VERSION_1_1"
    "GL_PIXEL_MAP_G_TO_G_SIZE" "GL_VERSION_1_1"
    "GL_PIXEL_MAP_B_TO_B_SIZE" "GL_VERSION_1_1"
    "GL_PIXEL_MAP_A_TO_A_SIZE" "GL_VERSION_1_1"
    "GL_UNPACK_SWAP_BYTES" "GL_VERSION_1_1"
    "GL_UNPACK_LSB_FIRST" "GL_VERSION_1_1"
    "GL_UNPACK_ROW_LENGTH" "GL_VERSION_1_1"
    "GL_UNPACK_SKIP_ROWS" "GL_VERSION_1_1"
    "GL_UNPACK_SKIP_PIXELS" "GL_VERSION_1_1"
    "GL_UNPACK_ALIGNMENT" "GL_VERSION_1_1"
    "GL_PACK_SWAP_BYTES" "GL_VERSION_1_1"
    "GL_PACK_LSB_FIRST" "GL_VERSION_1_1"
    "GL_PACK_ROW_LENGTH" "GL_VERSION_1_1"
    "GL_PACK_SKIP_ROWS" "GL_VERSION_1_1"
    "GL_PACK_SKIP_PIXELS" "GL_VERSION_1_1"
    "GL_PACK_ALIGNMENT" "GL_VERSION_1_1"
    "GL_MAP_COLOR" "GL_VERSION_1_1"
    "GL_MAP_STENCIL" "GL_VERSION_1_1"
    "GL_INDEX_SHIFT" "GL_VERSION_1_1"
    "GL_INDEX_OFFSET" "GL_VERSION_1_1"
    "GL_RED_SCALE" "GL_VERSION_1_1"
    "GL_RED_BIAS" "GL_VERSION_1_1"
    "GL_ZOOM_X" "GL_VERSION_1_1"
    "GL_ZOOM_Y" "GL_VERSION_1_1"
    "GL_GREEN_SCALE" "GL_VERSION_1_1"
    "GL_GREEN_BIAS" "GL_VERSION_1_1"
    "GL_BLUE_SCALE" "GL_VERSION_1_1"
    "GL_BLUE_BIAS" "GL_VERSION_1_1"
    "GL_ALPHA_SCALE" "GL_VERSION_1_1"
    "GL_ALPHA_BIAS" "GL_VERSION_1_1"
    "GL_DEPTH_SCALE" "GL_VERSION_1_1"
    "GL_DEPTH_BIAS" "GL_VERSION_1_1"
    "GL_MAX_EVAL_ORDER" "GL_VERSION_1_1"
    "GL_MAX_LIGHTS" "GL_VERSION_1_1"
    "GL_MAX_CLIP_PLANES" "GL_VERSION_1_1"
    "GL_MAX_TEXTURE_SIZE" "GL_VERSION_1_1"
    "GL_MAX_PIXEL_MAP_TABLE" "GL_VERSION_1_1"
    "GL_MAX_ATTRIB_STACK_DEPTH" "GL_VERSION_1_1"
    "GL_MAX_MODELVIEW_STACK_DEPTH" "GL_VERSION_1_1"
    "GL_MAX_NAME_STACK_DEPTH" "GL_VERSION_1_1"
    "GL_MAX_PROJECTION_STACK_DEPTH" "GL_VERSION_1_1"
    "GL_MAX_TEXTURE_STACK_DEPTH" "GL_VERSION_1_1"
    "GL_MAX_VIEWPORT_DIMS" "GL_VERSION_1_1"
    "GL_MAX_CLIENT_ATTRIB_STACK_DEPTH" "GL_VERSION_1_1"
    "GL_SUBPIXEL_BITS" "GL_VERSION_1_1"
    "GL_INDEX_BITS" "GL_VERSION_1_1"
    "GL_RED_BITS" "GL_VERSION_1_1"
    "GL_GREEN_BITS" "GL_VERSION_1_1"
    "GL_BLUE_BITS" "GL_VERSION_1_1"
    "GL_ALPHA_BITS" "GL_VERSION_1_1"
    "GL_DEPTH_BITS" "GL_VERSION_1_1"
    "GL_STENCIL_BITS" "GL_VERSION_1_1"
    "GL_ACCUM_RED_BITS" "GL_VERSION_1_1"
    "GL_ACCUM_GREEN_BITS" "GL_VERSION_1_1"
    "GL_ACCUM_BLUE_BITS" "GL_VERSION_1_1"
    "GL_ACCUM_ALPHA_BITS" "GL_VERSION_1_1"
    "GL_NAME_STACK_DEPTH" "GL_VERSION_1_1"
    "GL_AUTO_NORMAL" "GL_VERSION_1_1"
    "GL_MAP1_COLOR_4" "GL_VERSION_1_1"
    "GL_MAP1_INDEX" "GL_VERSION_1_1"
    "GL_MAP1_NORMAL" "GL_VERSION_1_1"
    "GL_MAP1_TEXTURE_COORD_1" "GL_VERSION_1_1"
    "GL_MAP1_TEXTURE_COORD_2" "GL_VERSION_1_1"
    "GL_MAP1_TEXTURE_COORD_3" "GL_VERSION_1_1"
    "GL_MAP1_TEXTURE_COORD_4" "GL_VERSION_1_1"
    "GL_MAP1_VERTEX_3" "GL_VERSION_1_1"
    "GL_MAP1_VERTEX_4" "GL_VERSION_1_1"
    "GL_MAP2_COLOR_4" "GL_VERSION_1_1"
    "GL_MAP2_INDEX" "GL_VERSION_1_1"
    "GL_MAP2_NORMAL" "GL_VERSION_1_1"
    "GL_MAP2_TEXTURE_COORD_1" "GL_VERSION_1_1"
    "GL_MAP2_TEXTURE_COORD_2" "GL_VERSION_1_1"
    "GL_MAP2_TEXTURE_COORD_3" "GL_VERSION_1_1"
    "GL_MAP2_TEXTURE_COORD_4" "GL_VERSION_1_1"
    "GL_MAP2_VERTEX_3" "GL_VERSION_1_1"
    "GL_MAP2_VERTEX_4" "GL_VERSION_1_1"
    "GL_MAP1_GRID_DOMAIN" "GL_VERSION_1_1"
    "GL_MAP1_GRID_SEGMENTS" "GL_VERSION_1_1"
    "GL_MAP2_GRID_DOMAIN" "GL_VERSION_1_1"
    "GL_MAP2_GRID_SEGMENTS" "GL_VERSION_1_1"
    "GL_TEXTURE_1D" "GL_VERSION_1_1"
    "GL_TEXTURE_2D" "GL_VERSION_1_1"
    "GL_FEEDBACK_BUFFER_POINTER" "GL_VERSION_1_1"
    "GL_FEEDBACK_BUFFER_SIZE" "GL_VERSION_1_1"
    "GL_FEEDBACK_BUFFER_TYPE" "GL_VERSION_1_1"
    "GL_SELECTION_BUFFER_POINTER" "GL_VERSION_1_1"
    "GL_SELECTION_BUFFER_SIZE" "GL_VERSION_1_1"
    "GL_TEXTURE_WIDTH" "GL_VERSION_1_1"
    "GL_TRANSFORM_BIT" "GL_VERSION_1_1"
    "GL_TEXTURE_HEIGHT" "GL_VERSION_1_1"
    "GL_TEXTURE_INTERNAL_FORMAT" "GL_VERSION_1_1"
    "GL_TEXTURE_BORDER_COLOR" "GL_VERSION_1_1"
    "GL_TEXTURE_BORDER" "GL_VERSION_1_1"
    "GL_DONT_CARE" "GL_VERSION_1_1"
    "GL_FASTEST" "GL_VERSION_1_1"
    "GL_NICEST" "GL_VERSION_1_1"
    "GL_AMBIENT" "GL_VERSION_1_1"
    "GL_DIFFUSE" "GL_VERSION_1_1"
    "GL_SPECULAR" "GL_VERSION_1_1"
    "GL_POSITION" "GL_VERSION_1_1"
    "GL_SPOT_DIRECTION" "GL_VERSION_1_1"
    "GL_SPOT_EXPONENT" "GL_VERSION_1_1"
    "GL_SPOT_CUTOFF" "GL_VERSION_1_1"
    "GL_CONSTANT_ATTENUATION" "GL_VERSION_1_1"
    "GL_LINEAR_ATTENUATION" "GL_VERSION_1_1"
    "GL_QUADRATIC_ATTENUATION" "GL_VERSION_1_1"
    "GL_COMPILE" "GL_VERSION_1_1"
    "GL_COMPILE_AND_EXECUTE" "GL_VERSION_1_1"
    "GL_BYTE" "GL_VERSION_1_1"
    "GL_UNSIGNED_BYTE" "GL_VERSION_1_1"
    "GL_SHORT" "GL_VERSION_1_1"
    "GL_UNSIGNED_SHORT" "GL_VERSION_1_1"
    "GL_INT" "GL_VERSION_1_1"
    "GL_UNSIGNED_INT" "GL_VERSION_1_1"
    "GL_FLOAT" "GL_VERSION_1_1"
    "GL_2_BYTES" "GL_VERSION_1_1"
    "GL_3_BYTES" "GL_VERSION_1_1"
    "GL_4_BYTES" "GL_VERSION_1_1"
    "GL_DOUBLE" "GL_VERSION_1_1"
    "GL_CLEAR" "GL_VERSION_1_1"
    "GL_AND" "GL_VERSION_1_1"
    "GL_AND_REVERSE" "GL_VERSION_1_1"
    "GL_COPY" "GL_VERSION_1_1"
    "GL_AND_INVERTED" "GL_VERSION_1_1"
    "GL_NOOP" "GL_VERSION_1_1"
    "GL_XOR" "GL_VERSION_1_1"
    "GL_OR" "GL_VERSION_1_1"
    "GL_NOR" "GL_VERSION_1_1"
    "GL_EQUIV" "GL_VERSION_1_1"
    "GL_INVERT" "GL_VERSION_1_1"
    "GL_OR_REVERSE" "GL_VERSION_1_1"
    "GL_COPY_INVERTED" "GL_VERSION_1_1"
    "GL_OR_INVERTED" "GL_VERSION_1_1"
    "GL_NAND" "GL_VERSION_1_1"
    "GL_SET" "GL_VERSION_1_1"
    "GL_EMISSION" "GL_VERSION_1_1"
    "GL_SHININESS" "GL_VERSION_1_1"
    "GL_AMBIENT_AND_DIFFUSE" "GL_VERSION_1_1"
    "GL_COLOR_INDEXES" "GL_VERSION_1_1"
    "GL_MODELVIEW" "GL_VERSION_1_1"
    "GL_PROJECTION" "GL_VERSION_1_1"
    "GL_TEXTURE" "GL_VERSION_1_1"
    "GL_COLOR" "GL_VERSION_1_1"
    "GL_DEPTH" "GL_VERSION_1_1"
    "GL_STENCIL" "GL_VERSION_1_1"
    "GL_COLOR_INDEX" "GL_VERSION_1_1"
    "GL_STENCIL_INDEX" "GL_VERSION_1_1"
    "GL_DEPTH_COMPONENT" "GL_VERSION_1_1"
    "GL_RED" "GL_VERSION_1_1"
    "GL_GREEN" "GL_VERSION_1_1"
    "GL_BLUE" "GL_VERSION_1_1"
    "GL_ALPHA" "GL_VERSION_1_1"
    "GL_RGB" "GL_VERSION_1_1"
    "GL_RGBA" "GL_VERSION_1_1"
    "GL_LUMINANCE" "GL_VERSION_1_1"
    "GL_LUMINANCE_ALPHA" "GL_VERSION_1_1"
    "GL_BITMAP" "GL_VERSION_1_1"
    "GL_POINT" "GL_VERSION_1_1"
    "GL_LINE" "GL_VERSION_1_1"
    "GL_FILL" "GL_VERSION_1_1"
    "GL_RENDER" "GL_VERSION_1_1"
    "GL_FEEDBACK" "GL_VERSION_1_1"
    "GL_SELECT" "GL_VERSION_1_1"
    "GL_FLAT" "GL_VERSION_1_1"
    "GL_SMOOTH" "GL_VERSION_1_1"
    "GL_KEEP" "GL_VERSION_1_1"
    "GL_REPLACE" "GL_VERSION_1_1"
    "GL_INCR" "GL_VERSION_1_1"
    "GL_DECR" "GL_VERSION_1_1"
    "GL_VENDOR" "GL_VERSION_1_1"
    "GL_RENDERER" "GL_VERSION_1_1"
    "GL_VERSION" "GL_VERSION_1_1"
    "GL_EXTENSIONS" "GL_VERSION_1_1"
    "GL_S" "GL_VERSION_1_1"
    "GL_ENABLE_BIT" "GL_VERSION_1_1"
    "GL_T" "GL_VERSION_1_1"
    "GL_R" "GL_VERSION_1_1"
    "GL_Q" "GL_VERSION_1_1"
    "GL_MODULATE" "GL_VERSION_1_1"
    "GL_DECAL" "GL_VERSION_1_1"
    "GL_TEXTURE_ENV_MODE" "GL_VERSION_1_1"
    "GL_TEXTURE_ENV_COLOR" "GL_VERSION_1_1"
    "GL_TEXTURE_ENV" "GL_VERSION_1_1"
    "GL_EYE_LINEAR" "GL_VERSION_1_1"
    "GL_OBJECT_LINEAR" "GL_VERSION_1_1"
    "GL_SPHERE_MAP" "GL_VERSION_1_1"
    "GL_TEXTURE_GEN_MODE" "GL_VERSION_1_1"
    "GL_OBJECT_PLANE" "GL_VERSION_1_1"
    "GL_EYE_PLANE" "GL_VERSION_1_1"
    "GL_NEAREST" "GL_VERSION_1_1"
    "GL_LINEAR" "GL_VERSION_1_1"
    "GL_NEAREST_MIPMAP_NEAREST" "GL_VERSION_1_1"
    "GL_LINEAR_MIPMAP_NEAREST" "GL_VERSION_1_1"
    "GL_NEAREST_MIPMAP_LINEAR" "GL_VERSION_1_1"
    "GL_LINEAR_MIPMAP_LINEAR" "GL_VERSION_1_1"
    "GL_TEXTURE_MAG_FILTER" "GL_VERSION_1_1"
    "GL_TEXTURE_MIN_FILTER" "GL_VERSION_1_1"
    "GL_TEXTURE_WRAP_S" "GL_VERSION_1_1"
    "GL_TEXTURE_WRAP_T" "GL_VERSION_1_1"
    "GL_CLAMP" "GL_VERSION_1_1"
    "GL_REPEAT" "GL_VERSION_1_1"
    "GL_POLYGON_OFFSET_UNITS" "GL_VERSION_1_1"
    "GL_POLYGON_OFFSET_POINT" "GL_VERSION_1_1"
    "GL_POLYGON_OFFSET_LINE" "GL_VERSION_1_1"
    "GL_R3_G3_B2" "GL_VERSION_1_1"
    "GL_V2F" "GL_VERSION_1_1"
    "GL_V3F" "GL_VERSION_1_1"
    "GL_C4UB_V2F" "GL_VERSION_1_1"
    "GL_C4UB_V3F" "GL_VERSION_1_1"
    "GL_C3F_V3F" "GL_VERSION_1_1"
    "GL_N3F_V3F" "GL_VERSION_1_1"
    "GL_C4F_N3F_V3F" "GL_VERSION_1_1"
    "GL_T2F_V3F" "GL_VERSION_1_1"
    "GL_T4F_V4F" "GL_VERSION_1_1"
    "GL_T2F_C4UB_V3F" "GL_VERSION_1_1"
    "GL_T2F_C3F_V3F" "GL_VERSION_1_1"
    "GL_T2F_N3F_V3F" "GL_VERSION_1_1"
    "GL_T2F_C4F_N3F_V3F" "GL_VERSION_1_1"
    "GL_T4F_C4F_N3F_V4F" "GL_VERSION_1_1"
    "GL_CLIP_PLANE0" "GL_VERSION_1_1"
    "GL_CLIP_PLANE1" "GL_VERSION_1_1"
    "GL_CLIP_PLANE2" "GL_VERSION_1_1"
    "GL_CLIP_PLANE3" "GL_VERSION_1_1"
    "GL_CLIP_PLANE4" "GL_VERSION_1_1"
    "GL_CLIP_PLANE5" "GL_VERSION_1_1"
    "GL_LIGHT0" "GL_VERSION_1_1"
    "GL_COLOR_BUFFER_BIT" "GL_VERSION_1_1"
    "GL_LIGHT1" "GL_VERSION_1_1"
    "GL_LIGHT2" "GL_VERSION_1_1"
    "GL_LIGHT3" "GL_VERSION_1_1"
    "GL_LIGHT4" "GL_VERSION_1_1"
    "GL_LIGHT5" "GL_VERSION_1_1"
    "GL_LIGHT6" "GL_VERSION_1_1"
    "GL_LIGHT7" "GL_VERSION_1_1"
    "GL_HINT_BIT" "GL_VERSION_1_1"
    "GL_POLYGON_OFFSET_FILL" "GL_VERSION_1_1"
    "GL_POLYGON_OFFSET_FACTOR" "GL_VERSION_1_1"
    "GL_ALPHA4" "GL_VERSION_1_1"
    "GL_ALPHA8" "GL_VERSION_1_1"
    "GL_ALPHA12" "GL_VERSION_1_1"
    "GL_ALPHA16" "GL_VERSION_1_1"
    "GL_LUMINANCE4" "GL_VERSION_1_1"
    "GL_LUMINANCE8" "GL_VERSION_1_1"
    "GL_LUMINANCE12" "GL_VERSION_1_1"
    "GL_LUMINANCE16" "GL_VERSION_1_1"
    "GL_LUMINANCE4_ALPHA4" "GL_VERSION_1_1"
    "GL_LUMINANCE6_ALPHA2" "GL_VERSION_1_1"
    "GL_LUMINANCE8_ALPHA8" "GL_VERSION_1_1"
    "GL_LUMINANCE12_ALPHA4" "GL_VERSION_1_1"
    "GL_LUMINANCE12_ALPHA12" "GL_VERSION_1_1"
    "GL_LUMINANCE16_ALPHA16" "GL_VERSION_1_1"
    "GL_INTENSITY" "GL_VERSION_1_1"
    "GL_INTENSITY4" "GL_VERSION_1_1"
    "GL_INTENSITY8" "GL_VERSION_1_1"
    "GL_INTENSITY12" "GL_VERSION_1_1"
    "GL_INTENSITY16" "GL_VERSION_1_1"
    "GL_RGB4" "GL_VERSION_1_1"
    "GL_RGB5" "GL_VERSION_1_1"
    "GL_RGB8" "GL_VERSION_1_1"
    "GL_RGB10" "GL_VERSION_1_1"
    "GL_RGB12" "GL_VERSION_1_1"
    "GL_RGB16" "GL_VERSION_1_1"
    "GL_RGBA2" "GL_VERSION_1_1"
    "GL_RGBA4" "GL_VERSION_1_1"
    "GL_RGB5_A1" "GL_VERSION_1_1"
    "GL_RGBA8" "GL_VERSION_1_1"
    "GL_RGB10_A2" "GL_VERSION_1_1"
    "GL_RGBA12" "GL_VERSION_1_1"
    "GL_RGBA16" "GL_VERSION_1_1"
    "GL_TEXTURE_RED_SIZE" "GL_VERSION_1_1"
    "GL_TEXTURE_GREEN_SIZE" "GL_VERSION_1_1"
    "GL_TEXTURE_BLUE_SIZE" "GL_VERSION_1_1"
    "GL_TEXTURE_ALPHA_SIZE" "GL_VERSION_1_1"
    "GL_TEXTURE_LUMINANCE_SIZE" "GL_VERSION_1_1"
    "GL_TEXTURE_INTENSITY_SIZE" "GL_VERSION_1_1"
    "GL_PROXY_TEXTURE_1D" "GL_VERSION_1_1"
    "GL_PROXY_TEXTURE_2D" "GL_VERSION_1_1"
    "GL_TEXTURE_PRIORITY" "GL_VERSION_1_1"
    "GL_TEXTURE_RESIDENT" "GL_VERSION_1_1"
    "GL_TEXTURE_BINDING_1D" "GL_VERSION_1_1"
    "GL_TEXTURE_BINDING_2D" "GL_VERSION_1_1"
    "GL_VERTEX_ARRAY" "GL_VERSION_1_1"
    "GL_NORMAL_ARRAY" "GL_VERSION_1_1"
    "GL_COLOR_ARRAY" "GL_VERSION_1_1"
    "GL_INDEX_ARRAY" "GL_VERSION_1_1"
    "GL_TEXTURE_COORD_ARRAY" "GL_VERSION_1_1"
    "GL_EDGE_FLAG_ARRAY" "GL_VERSION_1_1"
    "GL_VERTEX_ARRAY_SIZE" "GL_VERSION_1_1"
    "GL_VERTEX_ARRAY_TYPE" "GL_VERSION_1_1"
    "GL_VERTEX_ARRAY_STRIDE" "GL_VERSION_1_1"
    "GL_NORMAL_ARRAY_TYPE" "GL_VERSION_1_1"
    "GL_NORMAL_ARRAY_STRIDE" "GL_VERSION_1_1"
    "GL_COLOR_ARRAY_SIZE" "GL_VERSION_1_1"
    "GL_COLOR_ARRAY_TYPE" "GL_VERSION_1_1"
    "GL_COLOR_ARRAY_STRIDE" "GL_VERSION_1_1"
    "GL_INDEX_ARRAY_TYPE" "GL_VERSION_1_1"
    "GL_INDEX_ARRAY_STRIDE" "GL_VERSION_1_1"
    "GL_TEXTURE_COORD_ARRAY_SIZE" "GL_VERSION_1_1"
    "GL_TEXTURE_COORD_ARRAY_TYPE" "GL_VERSION_1_1"
    "GL_TEXTURE_COORD_ARRAY_STRIDE" "GL_VERSION_1_1"
    "GL_EDGE_FLAG_ARRAY_STRIDE" "GL_VERSION_1_1"
    "GL_VERTEX_ARRAY_POINTER" "GL_VERSION_1_1"
    "GL_NORMAL_ARRAY_POINTER" "GL_VERSION_1_1"
    "GL_COLOR_ARRAY_POINTER" "GL_VERSION_1_1"
    "GL_INDEX_ARRAY_POINTER" "GL_VERSION_1_1"
    "GL_TEXTURE_COORD_ARRAY_POINTER" "GL_VERSION_1_1"
    "GL_EDGE_FLAG_ARRAY_POINTER" "GL_VERSION_1_1"
    "GL_COLOR_INDEX1_EXT" "GL_VERSION_1_1"
    "GL_COLOR_INDEX2_EXT" "GL_VERSION_1_1"
    "GL_COLOR_INDEX4_EXT" "GL_VERSION_1_1"
    "GL_COLOR_INDEX8_EXT" "GL_VERSION_1_1"
    "GL_COLOR_INDEX12_EXT" "GL_VERSION_1_1"
    "GL_COLOR_INDEX16_EXT" "GL_VERSION_1_1"
    "GL_EVAL_BIT" "GL_VERSION_1_1"
    "GL_LIST_BIT" "GL_VERSION_1_1"
    "GL_TEXTURE_BIT" "GL_VERSION_1_1"
    "GL_SCISSOR_BIT" "GL_VERSION_1_1"
    "GL_ALL_ATTRIB_BITS" "GL_VERSION_1_1"
    "GL_CLIENT_ALL_ATTRIB_BITS" "GL_VERSION_1_1"
    "GL_VERSION_1_2" "GL_VERSION_1_2"
    "GL_SMOOTH_POINT_SIZE_RANGE" "GL_VERSION_1_2"
    "GL_SMOOTH_POINT_SIZE_GRANULARITY" "GL_VERSION_1_2"
    "GL_SMOOTH_LINE_WIDTH_RANGE" "GL_VERSION_1_2"
    "GL_SMOOTH_LINE_WIDTH_GRANULARITY" "GL_VERSION_1_2"
    "GL_UNSIGNED_BYTE_3_3_2" "GL_VERSION_1_2"
    "GL_UNSIGNED_SHORT_4_4_4_4" "GL_VERSION_1_2"
    "GL_UNSIGNED_SHORT_5_5_5_1" "GL_VERSION_1_2"
    "GL_UNSIGNED_INT_8_8_8_8" "GL_VERSION_1_2"
    "GL_UNSIGNED_INT_10_10_10_2" "GL_VERSION_1_2"
    "GL_RESCALE_NORMAL" "GL_VERSION_1_2"
    "GL_TEXTURE_BINDING_3D" "GL_VERSION_1_2"
    "GL_PACK_SKIP_IMAGES" "GL_VERSION_1_2"
    "GL_PACK_IMAGE_HEIGHT" "GL_VERSION_1_2"
    "GL_UNPACK_SKIP_IMAGES" "GL_VERSION_1_2"
    "GL_UNPACK_IMAGE_HEIGHT" "GL_VERSION_1_2"
    "GL_TEXTURE_3D" "GL_VERSION_1_2"
    "GL_PROXY_TEXTURE_3D" "GL_VERSION_1_2"
    "GL_TEXTURE_DEPTH" "GL_VERSION_1_2"
    "GL_TEXTURE_WRAP_R" "GL_VERSION_1_2"
    "GL_MAX_3D_TEXTURE_SIZE" "GL_VERSION_1_2"
    "GL_BGR" "GL_VERSION_1_2"
    "GL_BGRA" "GL_VERSION_1_2"
    "GL_MAX_ELEMENTS_VERTICES" "GL_VERSION_1_2"
    "GL_MAX_ELEMENTS_INDICES" "GL_VERSION_1_2"
    "GL_CLAMP_TO_EDGE" "GL_VERSION_1_2"
    "GL_TEXTURE_MIN_LOD" "GL_VERSION_1_2"
    "GL_TEXTURE_MAX_LOD" "GL_VERSION_1_2"
    "GL_TEXTURE_BASE_LEVEL" "GL_VERSION_1_2"
    "GL_TEXTURE_MAX_LEVEL" "GL_VERSION_1_2"
    "GL_LIGHT_MODEL_COLOR_CONTROL" "GL_VERSION_1_2"
    "GL_SINGLE_COLOR" "GL_VERSION_1_2"
    "GL_SEPARATE_SPECULAR_COLOR" "GL_VERSION_1_2"
    "GL_UNSIGNED_BYTE_2_3_3_REV" "GL_VERSION_1_2"
    "GL_UNSIGNED_SHORT_5_6_5" "GL_VERSION_1_2"
    "GL_UNSIGNED_SHORT_5_6_5_REV" "GL_VERSION_1_2"
    "GL_UNSIGNED_SHORT_4_4_4_4_REV" "GL_VERSION_1_2"
    "GL_UNSIGNED_SHORT_1_5_5_5_REV" "GL_VERSION_1_2"
    "GL_UNSIGNED_INT_8_8_8_8_REV" "GL_VERSION_1_2"
    "GL_ALIASED_POINT_SIZE_RANGE" "GL_VERSION_1_2"
    "GL_ALIASED_LINE_WIDTH_RANGE" "GL_VERSION_1_2"
    "GL_VERSION_1_2_1" "GL_VERSION_1_2_1"
    "GL_VERSION_1_3" "GL_VERSION_1_3"
    "GL_MULTISAMPLE" "GL_VERSION_1_3"
    "GL_SAMPLE_ALPHA_TO_COVERAGE" "GL_VERSION_1_3"
    "GL_SAMPLE_ALPHA_TO_ONE" "GL_VERSION_1_3"
    "GL_SAMPLE_COVERAGE" "GL_VERSION_1_3"
    "GL_SAMPLE_BUFFERS" "GL_VERSION_1_3"
    "GL_SAMPLES" "GL_VERSION_1_3"
    "GL_SAMPLE_COVERAGE_VALUE" "GL_VERSION_1_3"
    "GL_SAMPLE_COVERAGE_INVERT" "GL_VERSION_1_3"
    "GL_CLAMP_TO_BORDER" "GL_VERSION_1_3"
    "GL_TEXTURE0" "GL_VERSION_1_3"
    "GL_TEXTURE1" "GL_VERSION_1_3"
    "GL_TEXTURE2" "GL_VERSION_1_3"
    "GL_TEXTURE3" "GL_VERSION_1_3"
    "GL_TEXTURE4" "GL_VERSION_1_3"
    "GL_TEXTURE5" "GL_VERSION_1_3"
    "GL_TEXTURE6" "GL_VERSION_1_3"
    "GL_TEXTURE7" "GL_VERSION_1_3"
    "GL_TEXTURE8" "GL_VERSION_1_3"
    "GL_TEXTURE9" "GL_VERSION_1_3"
    "GL_TEXTURE10" "GL_VERSION_1_3"
    "GL_TEXTURE11" "GL_VERSION_1_3"
    "GL_TEXTURE12" "GL_VERSION_1_3"
    "GL_TEXTURE13" "GL_VERSION_1_3"
    "GL_TEXTURE14" "GL_VERSION_1_3"
    "GL_TEXTURE15" "GL_VERSION_1_3"
    "GL_TEXTURE16" "GL_VERSION_1_3"
    "GL_TEXTURE17" "GL_VERSION_1_3"
    "GL_TEXTURE18" "GL_VERSION_1_3"
    "GL_TEXTURE19" "GL_VERSION_1_3"
    "GL_TEXTURE20" "GL_VERSION_1_3"
    "GL_TEXTURE21" "GL_VERSION_1_3"
    "GL_TEXTURE22" "GL_VERSION_1_3"
    "GL_TEXTURE23" "GL_VERSION_1_3"
    "GL_TEXTURE24" "GL_VERSION_1_3"
    "GL_TEXTURE25" "GL_VERSION_1_3"
    "GL_TEXTURE26" "GL_VERSION_1_3"
    "GL_TEXTURE27" "GL_VERSION_1_3"
    "GL_TEXTURE28" "GL_VERSION_1_3"
    "GL_TEXTURE29" "GL_VERSION_1_3"
    "GL_TEXTURE30" "GL_VERSION_1_3"
    "GL_TEXTURE31" "GL_VERSION_1_3"
    "GL_ACTIVE_TEXTURE" "GL_VERSION_1_3"
    "GL_CLIENT_ACTIVE_TEXTURE" "GL_VERSION_1_3"
    "GL_MAX_TEXTURE_UNITS" "GL_VERSION_1_3"
    "GL_TRANSPOSE_MODELVIEW_MATRIX" "GL_VERSION_1_3"
    "GL_TRANSPOSE_PROJECTION_MATRIX" "GL_VERSION_1_3"
    "GL_TRANSPOSE_TEXTURE_MATRIX" "GL_VERSION_1_3"
    "GL_TRANSPOSE_COLOR_MATRIX" "GL_VERSION_1_3"
    "GL_SUBTRACT" "GL_VERSION_1_3"
    "GL_COMPRESSED_ALPHA" "GL_VERSION_1_3"
    "GL_COMPRESSED_LUMINANCE" "GL_VERSION_1_3"
    "GL_COMPRESSED_LUMINANCE_ALPHA" "GL_VERSION_1_3"
    "GL_COMPRESSED_INTENSITY" "GL_VERSION_1_3"
    "GL_COMPRESSED_RGB" "GL_VERSION_1_3"
    "GL_COMPRESSED_RGBA" "GL_VERSION_1_3"
    "GL_TEXTURE_COMPRESSION_HINT" "GL_VERSION_1_3"
    "GL_NORMAL_MAP" "GL_VERSION_1_3"
    "GL_REFLECTION_MAP" "GL_VERSION_1_3"
    "GL_TEXTURE_CUBE_MAP" "GL_VERSION_1_3"
    "GL_TEXTURE_BINDING_CUBE_MAP" "GL_VERSION_1_3"
    "GL_TEXTURE_CUBE_MAP_POSITIVE_X" "GL_VERSION_1_3"
    "GL_TEXTURE_CUBE_MAP_NEGATIVE_X" "GL_VERSION_1_3"
    "GL_TEXTURE_CUBE_MAP_POSITIVE_Y" "GL_VERSION_1_3"
    "GL_TEXTURE_CUBE_MAP_NEGATIVE_Y" "GL_VERSION_1_3"
    "GL_TEXTURE_CUBE_MAP_POSITIVE_Z" "GL_VERSION_1_3"
    "GL_TEXTURE_CUBE_MAP_NEGATIVE_Z" "GL_VERSION_1_3"
    "GL_PROXY_TEXTURE_CUBE_MAP" "GL_VERSION_1_3"
    "GL_MAX_CUBE_MAP_TEXTURE_SIZE" "GL_VERSION_1_3"
    "GL_COMBINE" "GL_VERSION_1_3"
    "GL_COMBINE_RGB" "GL_VERSION_1_3"
    "GL_COMBINE_ALPHA" "GL_VERSION_1_3"
    "GL_RGB_SCALE" "GL_VERSION_1_3"
    "GL_ADD_SIGNED" "GL_VERSION_1_3"
    "GL_INTERPOLATE" "GL_VERSION_1_3"
    "GL_CONSTANT" "GL_VERSION_1_3"
    "GL_PRIMARY_COLOR" "GL_VERSION_1_3"
    "GL_PREVIOUS" "GL_VERSION_1_3"
    "GL_SOURCE0_RGB" "GL_VERSION_1_3"
    "GL_SOURCE1_RGB" "GL_VERSION_1_3"
    "GL_SOURCE2_RGB" "GL_VERSION_1_3"
    "GL_SOURCE0_ALPHA" "GL_VERSION_1_3"
    "GL_SOURCE1_ALPHA" "GL_VERSION_1_3"
    "GL_SOURCE2_ALPHA" "GL_VERSION_1_3"
    "GL_OPERAND0_RGB" "GL_VERSION_1_3"
    "GL_OPERAND1_RGB" "GL_VERSION_1_3"
    "GL_OPERAND2_RGB" "GL_VERSION_1_3"
    "GL_OPERAND0_ALPHA" "GL_VERSION_1_3"
    "GL_OPERAND1_ALPHA" "GL_VERSION_1_3"
    "GL_OPERAND2_ALPHA" "GL_VERSION_1_3"
    "GL_TEXTURE_COMPRESSED_IMAGE_SIZE" "GL_VERSION_1_3"
    "GL_TEXTURE_COMPRESSED" "GL_VERSION_1_3"
    "GL_NUM_COMPRESSED_TEXTURE_FORMATS" "GL_VERSION_1_3"
    "GL_COMPRESSED_TEXTURE_FORMATS" "GL_VERSION_1_3"
    "GL_DOT3_RGB" "GL_VERSION_1_3"
    "GL_DOT3_RGBA" "GL_VERSION_1_3"
    "GL_MULTISAMPLE_BIT" "GL_VERSION_1_3"
    "GL_VERSION_1_4" "GL_VERSION_1_4"
    "GL_BLEND_DST_RGB" "GL_VERSION_1_4"
    "GL_BLEND_SRC_RGB" "GL_VERSION_1_4"
    "GL_BLEND_DST_ALPHA" "GL_VERSION_1_4"
    "GL_BLEND_SRC_ALPHA" "GL_VERSION_1_4"
    "GL_POINT_SIZE_MIN" "GL_VERSION_1_4"
    "GL_POINT_SIZE_MAX" "GL_VERSION_1_4"
    "GL_POINT_FADE_THRESHOLD_SIZE" "GL_VERSION_1_4"
    "GL_POINT_DISTANCE_ATTENUATION" "GL_VERSION_1_4"
    "GL_GENERATE_MIPMAP" "GL_VERSION_1_4"
    "GL_GENERATE_MIPMAP_HINT" "GL_VERSION_1_4"
    "GL_DEPTH_COMPONENT16" "GL_VERSION_1_4"
    "GL_DEPTH_COMPONENT24" "GL_VERSION_1_4"
    "GL_DEPTH_COMPONENT32" "GL_VERSION_1_4"
    "GL_MIRRORED_REPEAT" "GL_VERSION_1_4"
    "GL_FOG_COORDINATE_SOURCE" "GL_VERSION_1_4"
    "GL_FOG_COORDINATE" "GL_VERSION_1_4"
    "GL_FRAGMENT_DEPTH" "GL_VERSION_1_4"
    "GL_CURRENT_FOG_COORDINATE" "GL_VERSION_1_4"
    "GL_FOG_COORDINATE_ARRAY_TYPE" "GL_VERSION_1_4"
    "GL_FOG_COORDINATE_ARRAY_STRIDE" "GL_VERSION_1_4"
    "GL_FOG_COORDINATE_ARRAY_POINTER" "GL_VERSION_1_4"
    "GL_FOG_COORDINATE_ARRAY" "GL_VERSION_1_4"
    "GL_COLOR_SUM" "GL_VERSION_1_4"
    "GL_CURRENT_SECONDARY_COLOR" "GL_VERSION_1_4"
    "GL_SECONDARY_COLOR_ARRAY_SIZE" "GL_VERSION_1_4"
    "GL_SECONDARY_COLOR_ARRAY_TYPE" "GL_VERSION_1_4"
    "GL_SECONDARY_COLOR_ARRAY_STRIDE" "GL_VERSION_1_4"
    "GL_SECONDARY_COLOR_ARRAY_POINTER" "GL_VERSION_1_4"
    "GL_SECONDARY_COLOR_ARRAY" "GL_VERSION_1_4"
    "GL_MAX_TEXTURE_LOD_BIAS" "GL_VERSION_1_4"
    "GL_TEXTURE_FILTER_CONTROL" "GL_VERSION_1_4"
    "GL_TEXTURE_LOD_BIAS" "GL_VERSION_1_4"
    "GL_INCR_WRAP" "GL_VERSION_1_4"
    "GL_DECR_WRAP" "GL_VERSION_1_4"
    "GL_TEXTURE_DEPTH_SIZE" "GL_VERSION_1_4"
    "GL_DEPTH_TEXTURE_MODE" "GL_VERSION_1_4"
    "GL_TEXTURE_COMPARE_MODE" "GL_VERSION_1_4"
    "GL_TEXTURE_COMPARE_FUNC" "GL_VERSION_1_4"
    "GL_COMPARE_R_TO_TEXTURE" "GL_VERSION_1_4"
    "GL_VERSION_1_5" "GL_VERSION_1_5"
    "GL_CURRENT_FOG_COORD" "GL_VERSION_1_5"
    "GL_FOG_COORD" "GL_VERSION_1_5"
    "GL_FOG_COORD_ARRAY" "GL_VERSION_1_5"
    "GL_FOG_COORD_ARRAY_BUFFER_BINDING" "GL_VERSION_1_5"
    "GL_FOG_COORD_ARRAY_POINTER" "GL_VERSION_1_5"
    "GL_FOG_COORD_ARRAY_STRIDE" "GL_VERSION_1_5"
    "GL_FOG_COORD_ARRAY_TYPE" "GL_VERSION_1_5"
    "GL_FOG_COORD_SRC" "GL_VERSION_1_5"
    "GL_SRC0_ALPHA" "GL_VERSION_1_5"
    "GL_SRC0_RGB" "GL_VERSION_1_5"
    "GL_SRC1_ALPHA" "GL_VERSION_1_5"
    "GL_SRC1_RGB" "GL_VERSION_1_5"
    "GL_SRC2_ALPHA" "GL_VERSION_1_5"
    "GL_SRC2_RGB" "GL_VERSION_1_5"
    "GL_BUFFER_SIZE" "GL_VERSION_1_5"
    "GL_BUFFER_USAGE" "GL_VERSION_1_5"
    "GL_QUERY_COUNTER_BITS" "GL_VERSION_1_5"
    "GL_CURRENT_QUERY" "GL_VERSION_1_5"
    "GL_QUERY_RESULT" "GL_VERSION_1_5"
    "GL_QUERY_RESULT_AVAILABLE" "GL_VERSION_1_5"
    "GL_ARRAY_BUFFER" "GL_VERSION_1_5"
    "GL_ELEMENT_ARRAY_BUFFER" "GL_VERSION_1_5"
    "GL_ARRAY_BUFFER_BINDING" "GL_VERSION_1_5"
    "GL_ELEMENT_ARRAY_BUFFER_BINDING" "GL_VERSION_1_5"
    "GL_VERTEX_ARRAY_BUFFER_BINDING" "GL_VERSION_1_5"
    "GL_NORMAL_ARRAY_BUFFER_BINDING" "GL_VERSION_1_5"
    "GL_COLOR_ARRAY_BUFFER_BINDING" "GL_VERSION_1_5"
    "GL_INDEX_ARRAY_BUFFER_BINDING" "GL_VERSION_1_5"
    "GL_TEXTURE_COORD_ARRAY_BUFFER_BINDING" "GL_VERSION_1_5"
    "GL_EDGE_FLAG_ARRAY_BUFFER_BINDING" "GL_VERSION_1_5"
    "GL_SECONDARY_COLOR_ARRAY_BUFFER_BINDING" "GL_VERSION_1_5"
    "GL_FOG_COORDINATE_ARRAY_BUFFER_BINDING" "GL_VERSION_1_5"
    "GL_WEIGHT_ARRAY_BUFFER_BINDING" "GL_VERSION_1_5"
    "GL_VERTEX_ATTRIB_ARRAY_BUFFER_BINDING" "GL_VERSION_1_5"
    "GL_READ_ONLY" "GL_VERSION_1_5"
    "GL_WRITE_ONLY" "GL_VERSION_1_5"
    "GL_READ_WRITE" "GL_VERSION_1_5"
    "GL_BUFFER_ACCESS" "GL_VERSION_1_5"
    "GL_BUFFER_MAPPED" "GL_VERSION_1_5"
    "GL_BUFFER_MAP_POINTER" "GL_VERSION_1_5"
    "GL_STREAM_DRAW" "GL_VERSION_1_5"
    "GL_STREAM_READ" "GL_VERSION_1_5"
    "GL_STREAM_COPY" "GL_VERSION_1_5"
    "GL_STATIC_DRAW" "GL_VERSION_1_5"
    "GL_STATIC_READ" "GL_VERSION_1_5"
    "GL_STATIC_COPY" "GL_VERSION_1_5"
    "GL_DYNAMIC_DRAW" "GL_VERSION_1_5"
    "GL_DYNAMIC_READ" "GL_VERSION_1_5"
    "GL_DYNAMIC_COPY" "GL_VERSION_1_5"
    "GL_SAMPLES_PASSED" "GL_VERSION_1_5"
    "GL_VERSION_2_0" "GL_VERSION_2_0"
    "GL_BLEND_EQUATION_RGB" "GL_VERSION_2_0"
    "GL_VERTEX_ATTRIB_ARRAY_ENABLED" "GL_VERSION_2_0"
    "GL_VERTEX_ATTRIB_ARRAY_SIZE" "GL_VERSION_2_0"
    "GL_VERTEX_ATTRIB_ARRAY_STRIDE" "GL_VERSION_2_0"
    "GL_VERTEX_ATTRIB_ARRAY_TYPE" "GL_VERSION_2_0"
    "GL_CURRENT_VERTEX_ATTRIB" "GL_VERSION_2_0"
    "GL_VERTEX_PROGRAM_POINT_SIZE" "GL_VERSION_2_0"
    "GL_VERTEX_PROGRAM_TWO_SIDE" "GL_VERSION_2_0"
    "GL_VERTEX_ATTRIB_ARRAY_POINTER" "GL_VERSION_2_0"
    "GL_STENCIL_BACK_FUNC" "GL_VERSION_2_0"
    "GL_STENCIL_BACK_FAIL" "GL_VERSION_2_0"
    "GL_STENCIL_BACK_PASS_DEPTH_FAIL" "GL_VERSION_2_0"
    "GL_STENCIL_BACK_PASS_DEPTH_PASS" "GL_VERSION_2_0"
    "GL_MAX_DRAW_BUFFERS" "GL_VERSION_2_0"
    "GL_DRAW_BUFFER0" "GL_VERSION_2_0"
    "GL_DRAW_BUFFER1" "GL_VERSION_2_0"
    "GL_DRAW_BUFFER2" "GL_VERSION_2_0"
    "GL_DRAW_BUFFER3" "GL_VERSION_2_0"
    "GL_DRAW_BUFFER4" "GL_VERSION_2_0"
    "GL_DRAW_BUFFER5" "GL_VERSION_2_0"
    "GL_DRAW_BUFFER6" "GL_VERSION_2_0"
    "GL_DRAW_BUFFER7" "GL_VERSION_2_0"
    "GL_DRAW_BUFFER8" "GL_VERSION_2_0"
    "GL_DRAW_BUFFER9" "GL_VERSION_2_0"
    "GL_DRAW_BUFFER10" "GL_VERSION_2_0"
    "GL_DRAW_BUFFER11" "GL_VERSION_2_0"
    "GL_DRAW_BUFFER12" "GL_VERSION_2_0"
    "GL_DRAW_BUFFER13" "GL_VERSION_2_0"
    "GL_DRAW_BUFFER14" "GL_VERSION_2_0"
    "GL_DRAW_BUFFER15" "GL_VERSION_2_0"
    "GL_BLEND_EQUATION_ALPHA" "GL_VERSION_2_0"
    "GL_POINT_SPRITE" "GL_VERSION_2_0"
    "GL_COORD_REPLACE" "GL_VERSION_2_0"
    "GL_MAX_VERTEX_ATTRIBS" "GL_VERSION_2_0"
    "GL_VERTEX_ATTRIB_ARRAY_NORMALIZED" "GL_VERSION_2_0"
    "GL_MAX_TEXTURE_COORDS" "GL_VERSION_2_0"
    "GL_MAX_TEXTURE_IMAGE_UNITS" "GL_VERSION_2_0"
    "GL_FRAGMENT_SHADER" "GL_VERSION_2_0"
    "GL_VERTEX_SHADER" "GL_VERSION_2_0"
    "GL_MAX_FRAGMENT_UNIFORM_COMPONENTS" "GL_VERSION_2_0"
    "GL_MAX_VERTEX_UNIFORM_COMPONENTS" "GL_VERSION_2_0"
    "GL_MAX_VARYING_FLOATS" "GL_VERSION_2_0"
    "GL_MAX_VERTEX_TEXTURE_IMAGE_UNITS" "GL_VERSION_2_0"
    "GL_MAX_COMBINED_TEXTURE_IMAGE_UNITS" "GL_VERSION_2_0"
    "GL_SHADER_TYPE" "GL_VERSION_2_0"
    "GL_FLOAT_VEC2" "GL_VERSION_2_0"
    "GL_FLOAT_VEC3" "GL_VERSION_2_0"
    "GL_FLOAT_VEC4" "GL_VERSION_2_0"
    "GL_INT_VEC2" "GL_VERSION_2_0"
    "GL_INT_VEC3" "GL_VERSION_2_0"
    "GL_INT_VEC4" "GL_VERSION_2_0"
    "GL_BOOL" "GL_VERSION_2_0"
    "GL_BOOL_VEC2" "GL_VERSION_2_0"
    "GL_BOOL_VEC3" "GL_VERSION_2_0"
    "GL_BOOL_VEC4" "GL_VERSION_2_0"
    "GL_FLOAT_MAT2" "GL_VERSION_2_0"
    "GL_FLOAT_MAT3" "GL_VERSION_2_0"
    "GL_FLOAT_MAT4" "GL_VERSION_2_0"
    "GL_SAMPLER_1D" "GL_VERSION_2_0"
    "GL_SAMPLER_2D" "GL_VERSION_2_0"
    "GL_SAMPLER_3D" "GL_VERSION_2_0"
    "GL_SAMPLER_CUBE" "GL_VERSION_2_0"
    "GL_SAMPLER_1D_SHADOW" "GL_VERSION_2_0"
    "GL_SAMPLER_2D_SHADOW" "GL_VERSION_2_0"
    "GL_DELETE_STATUS" "GL_VERSION_2_0"
    "GL_COMPILE_STATUS" "GL_VERSION_2_0"
    "GL_LINK_STATUS" "GL_VERSION_2_0"
    "GL_VALIDATE_STATUS" "GL_VERSION_2_0"
    "GL_INFO_LOG_LENGTH" "GL_VERSION_2_0"
    "GL_ATTACHED_SHADERS" "GL_VERSION_2_0"
    "GL_ACTIVE_UNIFORMS" "GL_VERSION_2_0"
    "GL_ACTIVE_UNIFORM_MAX_LENGTH" "GL_VERSION_2_0"
    "GL_SHADER_SOURCE_LENGTH" "GL_VERSION_2_0"
    "GL_ACTIVE_ATTRIBUTES" "GL_VERSION_2_0"
    "GL_ACTIVE_ATTRIBUTE_MAX_LENGTH" "GL_VERSION_2_0"
    "GL_FRAGMENT_SHADER_DERIVATIVE_HINT" "GL_VERSION_2_0"
    "GL_SHADING_LANGUAGE_VERSION" "GL_VERSION_2_0"
    "GL_CURRENT_PROGRAM" "GL_VERSION_2_0"
    "GL_POINT_SPRITE_COORD_ORIGIN" "GL_VERSION_2_0"
    "GL_LOWER_LEFT" "GL_VERSION_2_0"
    "GL_UPPER_LEFT" "GL_VERSION_2_0"
    "GL_STENCIL_BACK_REF" "GL_VERSION_2_0"
    "GL_STENCIL_BACK_VALUE_MASK" "GL_VERSION_2_0"
    "GL_STENCIL_BACK_WRITEMASK" "GL_VERSION_2_0"
    "GL_VERSION_2_1" "GL_VERSION_2_1"
    "GL_CURRENT_RASTER_SECONDARY_COLOR" "GL_VERSION_2_1"
    "GL_PIXEL_PACK_BUFFER" "GL_VERSION_2_1"
    "GL_PIXEL_UNPACK_BUFFER" "GL_VERSION_2_1"
    "GL_PIXEL_PACK_BUFFER_BINDING" "GL_VERSION_2_1"
    "GL_PIXEL_UNPACK_BUFFER_BINDING" "GL_VERSION_2_1"
    "GL_FLOAT_MAT2x3" "GL_VERSION_2_1"
    "GL_FLOAT_MAT2x4" "GL_VERSION_2_1"
    "GL_FLOAT_MAT3x2" "GL_VERSION_2_1"
    "GL_FLOAT_MAT3x4" "GL_VERSION_2_1"
    "GL_FLOAT_MAT4x2" "GL_VERSION_2_1"
    "GL_FLOAT_MAT4x3" "GL_VERSION_2_1"
    "GL_SRGB" "GL_VERSION_2_1"
    "GL_SRGB8" "GL_VERSION_2_1"
    "GL_SRGB_ALPHA" "GL_VERSION_2_1"
    "GL_SRGB8_ALPHA8" "GL_VERSION_2_1"
    "GL_SLUMINANCE_ALPHA" "GL_VERSION_2_1"
    "GL_SLUMINANCE8_ALPHA8" "GL_VERSION_2_1"
    "GL_SLUMINANCE" "GL_VERSION_2_1"
    "GL_SLUMINANCE8" "GL_VERSION_2_1"
    "GL_COMPRESSED_SRGB" "GL_VERSION_2_1"
    "GL_COMPRESSED_SRGB_ALPHA" "GL_VERSION_2_1"
    "GL_COMPRESSED_SLUMINANCE" "GL_VERSION_2_1"
    "GL_COMPRESSED_SLUMINANCE_ALPHA" "GL_VERSION_2_1"
    "GL_VERSION_3_0" "GL_VERSION_3_0"
    "GL_CLIP_DISTANCE0" "GL_VERSION_3_0"
    "GL_CLIP_DISTANCE1" "GL_VERSION_3_0"
    "GL_CLIP_DISTANCE2" "GL_VERSION_3_0"
    "GL_CLIP_DISTANCE3" "GL_VERSION_3_0"
    "GL_CLIP_DISTANCE4" "GL_VERSION_3_0"
    "GL_CLIP_DISTANCE5" "GL_VERSION_3_0"
    "GL_COMPARE_REF_TO_TEXTURE" "GL_VERSION_3_0"
    "GL_MAX_CLIP_DISTANCES" "GL_VERSION_3_0"
    "GL_MAX_VARYING_COMPONENTS" "GL_VERSION_3_0"
    "GL_CONTEXT_FLAG_FORWARD_COMPATIBLE_BIT" "GL_VERSION_3_0"
    "GL_MAJOR_VERSION" "GL_VERSION_3_0"
    "GL_MINOR_VERSION" "GL_VERSION_3_0"
    "GL_NUM_EXTENSIONS" "GL_VERSION_3_0"
    "GL_CONTEXT_FLAGS" "GL_VERSION_3_0"
    "GL_DEPTH_BUFFER" "GL_VERSION_3_0"
    "GL_STENCIL_BUFFER" "GL_VERSION_3_0"
    "GL_RGBA32F" "GL_VERSION_3_0"
    "GL_RGB32F" "GL_VERSION_3_0"
    "GL_RGBA16F" "GL_VERSION_3_0"
    "GL_RGB16F" "GL_VERSION_3_0"
    "GL_VERTEX_ATTRIB_ARRAY_INTEGER" "GL_VERSION_3_0"
    "GL_MAX_ARRAY_TEXTURE_LAYERS" "GL_VERSION_3_0"
    "GL_MIN_PROGRAM_TEXEL_OFFSET" "GL_VERSION_3_0"
    "GL_MAX_PROGRAM_TEXEL_OFFSET" "GL_VERSION_3_0"
    "GL_CLAMP_VERTEX_COLOR" "GL_VERSION_3_0"
    "GL_CLAMP_FRAGMENT_COLOR" "GL_VERSION_3_0"
    "GL_CLAMP_READ_COLOR" "GL_VERSION_3_0"
    "GL_FIXED_ONLY" "GL_VERSION_3_0"
    "GL_TEXTURE_RED_TYPE" "GL_VERSION_3_0"
    "GL_TEXTURE_GREEN_TYPE" "GL_VERSION_3_0"
    "GL_TEXTURE_BLUE_TYPE" "GL_VERSION_3_0"
    "GL_TEXTURE_ALPHA_TYPE" "GL_VERSION_3_0"
    "GL_TEXTURE_LUMINANCE_TYPE" "GL_VERSION_3_0"
    "GL_TEXTURE_INTENSITY_TYPE" "GL_VERSION_3_0"
    "GL_TEXTURE_DEPTH_TYPE" "GL_VERSION_3_0"
    "GL_TEXTURE_1D_ARRAY" "GL_VERSION_3_0"
    "GL_PROXY_TEXTURE_1D_ARRAY" "GL_VERSION_3_0"
    "GL_TEXTURE_2D_ARRAY" "GL_VERSION_3_0"
    "GL_PROXY_TEXTURE_2D_ARRAY" "GL_VERSION_3_0"
    "GL_TEXTURE_BINDING_1D_ARRAY" "GL_VERSION_3_0"
    "GL_TEXTURE_BINDING_2D_ARRAY" "GL_VERSION_3_0"
    "GL_R11F_G11F_B10F" "GL_VERSION_3_0"
    "GL_UNSIGNED_INT_10F_11F_11F_REV" "GL_VERSION_3_0"
    "GL_RGB9_E5" "GL_VERSION_3_0"
    "GL_UNSIGNED_INT_5_9_9_9_REV" "GL_VERSION_3_0"
    "GL_TEXTURE_SHARED_SIZE" "GL_VERSION_3_0"
    "GL_TRANSFORM_FEEDBACK_VARYING_MAX_LENGTH" "GL_VERSION_3_0"
    "GL_TRANSFORM_FEEDBACK_BUFFER_MODE" "GL_VERSION_3_0"
    "GL_MAX_TRANSFORM_FEEDBACK_SEPARATE_COMPONENTS" "GL_VERSION_3_0"
    "GL_TRANSFORM_FEEDBACK_VARYINGS" "GL_VERSION_3_0"
    "GL_TRANSFORM_FEEDBACK_BUFFER_START" "GL_VERSION_3_0"
    "GL_TRANSFORM_FEEDBACK_BUFFER_SIZE" "GL_VERSION_3_0"
    "GL_PRIMITIVES_GENERATED" "GL_VERSION_3_0"
    "GL_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN" "GL_VERSION_3_0"
    "GL_RASTERIZER_DISCARD" "GL_VERSION_3_0"
    "GL_MAX_TRANSFORM_FEEDBACK_INTERLEAVED_COMPONENTS" "GL_VERSION_3_0"
    "GL_MAX_TRANSFORM_FEEDBACK_SEPARATE_ATTRIBS" "GL_VERSION_3_0"
    "GL_INTERLEAVED_ATTRIBS" "GL_VERSION_3_0"
    "GL_SEPARATE_ATTRIBS" "GL_VERSION_3_0"
    "GL_TRANSFORM_FEEDBACK_BUFFER" "GL_VERSION_3_0"
    "GL_TRANSFORM_FEEDBACK_BUFFER_BINDING" "GL_VERSION_3_0"
    "GL_RGBA32UI" "GL_VERSION_3_0"
    "GL_RGB32UI" "GL_VERSION_3_0"
    "GL_RGBA16UI" "GL_VERSION_3_0"
    "GL_RGB16UI" "GL_VERSION_3_0"
    "GL_RGBA8UI" "GL_VERSION_3_0"
    "GL_RGB8UI" "GL_VERSION_3_0"
    "GL_RGBA32I" "GL_VERSION_3_0"
    "GL_RGB32I" "GL_VERSION_3_0"
    "GL_RGBA16I" "GL_VERSION_3_0"
    "GL_RGB16I" "GL_VERSION_3_0"
    "GL_RGBA8I" "GL_VERSION_3_0"
    "GL_RGB8I" "GL_VERSION_3_0"
    "GL_RED_INTEGER" "GL_VERSION_3_0"
    "GL_GREEN_INTEGER" "GL_VERSION_3_0"
    "GL_BLUE_INTEGER" "GL_VERSION_3_0"
    "GL_ALPHA_INTEGER" "GL_VERSION_3_0"
    "GL_RGB_INTEGER" "GL_VERSION_3_0"
    "GL_RGBA_INTEGER" "GL_VERSION_3_0"
    "GL_BGR_INTEGER" "GL_VERSION_3_0"
    "GL_BGRA_INTEGER" "GL_VERSION_3_0"
    "GL_SAMPLER_1D_ARRAY" "GL_VERSION_3_0"
    "GL_SAMPLER_2D_ARRAY" "GL_VERSION_3_0"
    "GL_SAMPLER_1D_ARRAY_SHADOW" "GL_VERSION_3_0"
    "GL_SAMPLER_2D_ARRAY_SHADOW" "GL_VERSION_3_0"
    "GL_SAMPLER_CUBE_SHADOW" "GL_VERSION_3_0"
    "GL_UNSIGNED_INT_VEC2" "GL_VERSION_3_0"
    "GL_UNSIGNED_INT_VEC3" "GL_VERSION_3_0"
    "GL_UNSIGNED_INT_VEC4" "GL_VERSION_3_0"
    "GL_INT_SAMPLER_1D" "GL_VERSION_3_0"
    "GL_INT_SAMPLER_2D" "GL_VERSION_3_0"
    "GL_INT_SAMPLER_3D" "GL_VERSION_3_0"
    "GL_INT_SAMPLER_CUBE" "GL_VERSION_3_0"
    "GL_INT_SAMPLER_1D_ARRAY" "GL_VERSION_3_0"
    "GL_INT_SAMPLER_2D_ARRAY" "GL_VERSION_3_0"
    "GL_UNSIGNED_INT_SAMPLER_1D" "GL_VERSION_3_0"
    "GL_UNSIGNED_INT_SAMPLER_2D" "GL_VERSION_3_0"
    "GL_UNSIGNED_INT_SAMPLER_3D" "GL_VERSION_3_0"
    "GL_UNSIGNED_INT_SAMPLER_CUBE" "GL_VERSION_3_0"
    "GL_UNSIGNED_INT_SAMPLER_1D_ARRAY" "GL_VERSION_3_0"
    "GL_UNSIGNED_INT_SAMPLER_2D_ARRAY" "GL_VERSION_3_0"
    "GL_QUERY_WAIT" "GL_VERSION_3_0"
    "GL_QUERY_NO_WAIT" "GL_VERSION_3_0"
    "GL_QUERY_BY_REGION_WAIT" "GL_VERSION_3_0"
    "GL_QUERY_BY_REGION_NO_WAIT" "GL_VERSION_3_0"
    "GL_VERSION_3_1" "GL_VERSION_3_1"
    "GL_TEXTURE_RECTANGLE" "GL_VERSION_3_1"
    "GL_TEXTURE_BINDING_RECTANGLE" "GL_VERSION_3_1"
    "GL_PROXY_TEXTURE_RECTANGLE" "GL_VERSION_3_1"
    "GL_MAX_RECTANGLE_TEXTURE_SIZE" "GL_VERSION_3_1"
    "GL_SAMPLER_2D_RECT" "GL_VERSION_3_1"
    "GL_SAMPLER_2D_RECT_SHADOW" "GL_VERSION_3_1"
    "GL_TEXTURE_BUFFER" "GL_VERSION_3_1"
    "GL_MAX_TEXTURE_BUFFER_SIZE" "GL_VERSION_3_1"
    "GL_TEXTURE_BINDING_BUFFER" "GL_VERSION_3_1"
    "GL_TEXTURE_BUFFER_DATA_STORE_BINDING" "GL_VERSION_3_1"
    "GL_TEXTURE_BUFFER_FORMAT" "GL_VERSION_3_1"
    "GL_SAMPLER_BUFFER" "GL_VERSION_3_1"
    "GL_INT_SAMPLER_2D_RECT" "GL_VERSION_3_1"
    "GL_INT_SAMPLER_BUFFER" "GL_VERSION_3_1"
    "GL_UNSIGNED_INT_SAMPLER_2D_RECT" "GL_VERSION_3_1"
    "GL_UNSIGNED_INT_SAMPLER_BUFFER" "GL_VERSION_3_1"
    "GL_RED_SNORM" "GL_VERSION_3_1"
    "GL_RG_SNORM" "GL_VERSION_3_1"
    "GL_RGB_SNORM" "GL_VERSION_3_1"
    "GL_RGBA_SNORM" "GL_VERSION_3_1"
    "GL_R8_SNORM" "GL_VERSION_3_1"
    "GL_RG8_SNORM" "GL_VERSION_3_1"
    "GL_RGB8_SNORM" "GL_VERSION_3_1"
    "GL_RGBA8_SNORM" "GL_VERSION_3_1"
    "GL_R16_SNORM" "GL_VERSION_3_1"
    "GL_RG16_SNORM" "GL_VERSION_3_1"
    "GL_RGB16_SNORM" "GL_VERSION_3_1"
    "GL_RGBA16_SNORM" "GL_VERSION_3_1"
    "GL_SIGNED_NORMALIZED" "GL_VERSION_3_1"
    "GL_PRIMITIVE_RESTART" "GL_VERSION_3_1"
    "GL_PRIMITIVE_RESTART_INDEX" "GL_VERSION_3_1"
    "GL_BUFFER_ACCESS_FLAGS" "GL_VERSION_3_1"
    "GL_BUFFER_MAP_LENGTH" "GL_VERSION_3_1"
    "GL_BUFFER_MAP_OFFSET" "GL_VERSION_3_1"
    "GL_VERSION_3_2" "GL_VERSION_3_2"
    "GL_CONTEXT_CORE_PROFILE_BIT" "GL_VERSION_3_2"
    "GL_CONTEXT_COMPATIBILITY_PROFILE_BIT" "GL_VERSION_3_2"
    "GL_LINES_ADJACENCY" "GL_VERSION_3_2"
    "GL_LINE_STRIP_ADJACENCY" "GL_VERSION_3_2"
    "GL_TRIANGLES_ADJACENCY" "GL_VERSION_3_2"
    "GL_TRIANGLE_STRIP_ADJACENCY" "GL_VERSION_3_2"
    "GL_PROGRAM_POINT_SIZE" "GL_VERSION_3_2"
    "GL_GEOMETRY_VERTICES_OUT" "GL_VERSION_3_2"
    "GL_GEOMETRY_INPUT_TYPE" "GL_VERSION_3_2"
    "GL_GEOMETRY_OUTPUT_TYPE" "GL_VERSION_3_2"
    "GL_MAX_GEOMETRY_TEXTURE_IMAGE_UNITS" "GL_VERSION_3_2"
    "GL_FRAMEBUFFER_ATTACHMENT_LAYERED" "GL_VERSION_3_2"
    "GL_FRAMEBUFFER_INCOMPLETE_LAYER_TARGETS" "GL_VERSION_3_2"
    "GL_GEOMETRY_SHADER" "GL_VERSION_3_2"
    "GL_MAX_GEOMETRY_UNIFORM_COMPONENTS" "GL_VERSION_3_2"
    "GL_MAX_GEOMETRY_OUTPUT_VERTICES" "GL_VERSION_3_2"
    "GL_MAX_GEOMETRY_TOTAL_OUTPUT_COMPONENTS" "GL_VERSION_3_2"
    "GL_MAX_VERTEX_OUTPUT_COMPONENTS" "GL_VERSION_3_2"
    "GL_MAX_GEOMETRY_INPUT_COMPONENTS" "GL_VERSION_3_2"
    "GL_MAX_GEOMETRY_OUTPUT_COMPONENTS" "GL_VERSION_3_2"
    "GL_MAX_FRAGMENT_INPUT_COMPONENTS" "GL_VERSION_3_2"
    "GL_CONTEXT_PROFILE_MASK" "GL_VERSION_3_2"
    "GL_VERSION_3_3" "GL_VERSION_3_3"
    "GL_VERTEX_ATTRIB_ARRAY_DIVISOR" "GL_VERSION_3_3"
    "GL_RGB10_A2UI" "GL_VERSION_3_3"
    "GL_VERSION_4_0" "GL_VERSION_4_0"
    "GL_SAMPLE_SHADING" "GL_VERSION_4_0"
    "GL_MIN_SAMPLE_SHADING_VALUE" "GL_VERSION_4_0"
    "GL_MIN_PROGRAM_TEXTURE_GATHER_OFFSET" "GL_VERSION_4_0"
    "GL_MAX_PROGRAM_TEXTURE_GATHER_OFFSET" "GL_VERSION_4_0"
    "GL_MAX_PROGRAM_TEXTURE_GATHER_COMPONENTS" "GL_VERSION_4_0"
    "GL_TEXTURE_CUBE_MAP_ARRAY" "GL_VERSION_4_0"
    "GL_TEXTURE_BINDING_CUBE_MAP_ARRAY" "GL_VERSION_4_0"
    "GL_PROXY_TEXTURE_CUBE_MAP_ARRAY" "GL_VERSION_4_0"
    "GL_SAMPLER_CUBE_MAP_ARRAY" "GL_VERSION_4_0"
    "GL_SAMPLER_CUBE_MAP_ARRAY_SHADOW" "GL_VERSION_4_0"
    "GL_INT_SAMPLER_CUBE_MAP_ARRAY" "GL_VERSION_4_0"
    "GL_UNSIGNED_INT_SAMPLER_CUBE_MAP_ARRAY" "GL_VERSION_4_0"
    "GL_VERSION_4_1" "GL_VERSION_4_1"
    "GL_VERSION_4_2" "GL_VERSION_4_2"
    "GL_TRANSFORM_FEEDBACK_PAUSED" "GL_VERSION_4_2"
    "GL_TRANSFORM_FEEDBACK_ACTIVE" "GL_VERSION_4_2"
    "GL_COMPRESSED_RGBA_BPTC_UNORM" "GL_VERSION_4_2"
    "GL_COMPRESSED_SRGB_ALPHA_BPTC_UNORM" "GL_VERSION_4_2"
    "GL_COMPRESSED_RGB_BPTC_SIGNED_FLOAT" "GL_VERSION_4_2"
    "GL_COMPRESSED_RGB_BPTC_UNSIGNED_FLOAT" "GL_VERSION_4_2"
    "GL_COPY_READ_BUFFER_BINDING" "GL_VERSION_4_2"
    "GL_COPY_WRITE_BUFFER_BINDING" "GL_VERSION_4_2"
    "GL_VERSION_4_3" "GL_VERSION_4_3"
    "GL_NUM_SHADING_LANGUAGE_VERSIONS" "GL_VERSION_4_3"
    "GL_VERTEX_ATTRIB_ARRAY_LONG" "GL_VERSION_4_3"
    "GL_VERSION_4_4" "GL_VERSION_4_4"
    "GL_PRIMITIVE_RESTART_FOR_PATCHES_SUPPORTED" "GL_VERSION_4_4"
    "GL_MAX_VERTEX_ATTRIB_STRIDE" "GL_VERSION_4_4"
    "GL_TEXTURE_BUFFER_BINDING" "GL_VERSION_4_4"
    "GL_VERSION_4_5" "GL_VERSION_4_5"
    "GL_CONTEXT_FLAG_ROBUST_ACCESS_BIT" "GL_VERSION_4_5"
    "GL_VERSION_4_6" "GL_VERSION_4_6"
    "GL_CONTEXT_FLAG_NO_ERROR_BIT" "GL_VERSION_4_6"
    "GL_PARAMETER_BUFFER" "GL_VERSION_4_6"
    "GL_PARAMETER_BUFFER_BINDING" "GL_VERSION_4_6"
    "GL_TRANSFORM_FEEDBACK_OVERFLOW" "GL_VERSION_4_6"
    "GL_TRANSFORM_FEEDBACK_STREAM_OVERFLOW" "GL_VERSION_4_6"
    "GL_VERTICES_SUBMITTED" "GL_VERSION_4_6"
    "GL_PRIMITIVES_SUBMITTED" "GL_VERSION_4_6"
    "GL_VERTEX_SHADER_INVOCATIONS" "GL_VERSION_4_6"
    "GL_TESS_CONTROL_SHADER_PATCHES" "GL_VERSION_4_6"
    "GL_TESS_EVALUATION_SHADER_INVOCATIONS" "GL_VERSION_4_6"
    "GL_GEOMETRY_SHADER_PRIMITIVES_EMITTED" "GL_VERSION_4_6"
    "GL_FRAGMENT_SHADER_INVOCATIONS" "GL_VERSION_4_6"
    "GL_COMPUTE_SHADER_INVOCATIONS" "GL_VERSION_4_6"
    "GL_CLIPPING_INPUT_PRIMITIVES" "GL_VERSION_4_6"
    "GL_CLIPPING_OUTPUT_PRIMITIVES" "GL_VERSION_4_6"
    "GL_TEXTURE_MAX_ANISOTROPY" "GL_VERSION_4_6"
    "GL_MAX_TEXTURE_MAX_ANISOTROPY" "GL_VERSION_4_6"
    "GL_POLYGON_OFFSET_CLAMP" "GL_VERSION_4_6"
    "GL_SHADER_BINARY_FORMAT_SPIR_V" "GL_VERSION_4_6"
    "GL_SPIR_V_BINARY" "GL_VERSION_4_6"
    "GL_SPIR_V_EXTENSIONS" "GL_VERSION_4_6"
    "GL_NUM_SPIR_V_EXTENSIONS" "GL_VERSION_4_6"
    "GL_3DFX_multisample" "GL_3DFX_multisample"
    "GL_MULTISAMPLE_3DFX" "GL_3DFX_multisample"
    "GL_SAMPLE_BUFFERS_3DFX" "GL_3DFX_multisample"
    "GL_SAMPLES_3DFX" "GL_3DFX_multisample"
    "GL_MULTISAMPLE_BIT_3DFX" "GL_3DFX_multisample"
    "GL_3DFX_tbuffer" "GL_3DFX_tbuffer"
    "GL_3DFX_texture_compression_FXT1" "GL_3DFX_texture_compression_FXT1"
    "GL_COMPRESSED_RGB_FXT1_3DFX" "GL_3DFX_texture_compression_FXT1"
    "GL_COMPRESSED_RGBA_FXT1_3DFX" "GL_3DFX_texture_compression_FXT1"
    "GL_AMD_blend_minmax_factor" "GL_AMD_blend_minmax_factor"
    "GL_FACTOR_MIN_AMD" "GL_AMD_blend_minmax_factor"
    "GL_FACTOR_MAX_AMD" "GL_AMD_blend_minmax_factor"
    "GL_AMD_compressed_3DC_texture" "GL_AMD_compressed_3DC_texture"
    "GL_3DC_X_AMD" "GL_AMD_compressed_3DC_texture"
    "GL_3DC_XY_AMD" "GL_AMD_compressed_3DC_texture"
    "GL_AMD_compressed_ATC_texture" "GL_AMD_compressed_ATC_texture"
    "GL_ATC_RGBA_INTERPOLATED_ALPHA_AMD" "GL_AMD_compressed_ATC_texture"
    "GL_ATC_RGB_AMD" "GL_AMD_compressed_ATC_texture"
    "GL_ATC_RGBA_EXPLICIT_ALPHA_AMD" "GL_AMD_compressed_ATC_texture"
    "GL_AMD_conservative_depth" "GL_AMD_conservative_depth"
    "GL_AMD_debug_output" "GL_AMD_debug_output"
    "GL_MAX_DEBUG_MESSAGE_LENGTH_AMD" "GL_AMD_debug_output"
    "GL_MAX_DEBUG_LOGGED_MESSAGES_AMD" "GL_AMD_debug_output"
    "GL_DEBUG_LOGGED_MESSAGES_AMD" "GL_AMD_debug_output"
    "GL_DEBUG_SEVERITY_HIGH_AMD" "GL_AMD_debug_output"
    "GL_DEBUG_SEVERITY_MEDIUM_AMD" "GL_AMD_debug_output"
    "GL_DEBUG_SEVERITY_LOW_AMD" "GL_AMD_debug_output"
    "GL_DEBUG_CATEGORY_API_ERROR_AMD" "GL_AMD_debug_output"
    "GL_DEBUG_CATEGORY_WINDOW_SYSTEM_AMD" "GL_AMD_debug_output"
    "GL_DEBUG_CATEGORY_DEPRECATION_AMD" "GL_AMD_debug_output"
    "GL_DEBUG_CATEGORY_UNDEFINED_BEHAVIOR_AMD" "GL_AMD_debug_output"
    "GL_DEBUG_CATEGORY_PERFORMANCE_AMD" "GL_AMD_debug_output"
    "GL_DEBUG_CATEGORY_SHADER_COMPILER_AMD" "GL_AMD_debug_output"
    "GL_DEBUG_CATEGORY_APPLICATION_AMD" "GL_AMD_debug_output"
    "GL_DEBUG_CATEGORY_OTHER_AMD" "GL_AMD_debug_output"
    "GL_AMD_depth_clamp_separate" "GL_AMD_depth_clamp_separate"
    "GL_DEPTH_CLAMP_NEAR_AMD" "GL_AMD_depth_clamp_separate"
    "GL_DEPTH_CLAMP_FAR_AMD" "GL_AMD_depth_clamp_separate"
    "GL_AMD_draw_buffers_blend" "GL_AMD_draw_buffers_blend"
    "GL_AMD_framebuffer_multisample_advanced" "GL_AMD_framebuffer_multisample_advanced"
    "GL_RENDERBUFFER_STORAGE_SAMPLES_AMD" "GL_AMD_framebuffer_multisample_advanced"
    "GL_MAX_COLOR_FRAMEBUFFER_SAMPLES_AMD" "GL_AMD_framebuffer_multisample_advanced"
    "GL_MAX_COLOR_FRAMEBUFFER_STORAGE_SAMPLES_AMD" "GL_AMD_framebuffer_multisample_advanced"
    "GL_MAX_DEPTH_STENCIL_FRAMEBUFFER_SAMPLES_AMD" "GL_AMD_framebuffer_multisample_advanced"
    "GL_NUM_SUPPORTED_MULTISAMPLE_MODES_AMD" "GL_AMD_framebuffer_multisample_advanced"
    "GL_SUPPORTED_MULTISAMPLE_MODES_AMD" "GL_AMD_framebuffer_multisample_advanced"
    "GL_AMD_framebuffer_sample_positions" "GL_AMD_framebuffer_sample_positions"
    "GL_SUBSAMPLE_DISTANCE_AMD" "GL_AMD_framebuffer_sample_positions"
    "GL_PIXELS_PER_SAMPLE_PATTERN_X_AMD" "GL_AMD_framebuffer_sample_positions"
    "GL_PIXELS_PER_SAMPLE_PATTERN_Y_AMD" "GL_AMD_framebuffer_sample_positions"
    "GL_ALL_PIXELS_AMD" "GL_AMD_framebuffer_sample_positions"
    "GL_AMD_gcn_shader" "GL_AMD_gcn_shader"
    "GL_AMD_gpu_shader_half_float" "GL_AMD_gpu_shader_half_float"
    "GL_FLOAT16_NV" "GL_AMD_gpu_shader_half_float"
    "GL_FLOAT16_VEC2_NV" "GL_AMD_gpu_shader_half_float"
    "GL_FLOAT16_VEC3_NV" "GL_AMD_gpu_shader_half_float"
    "GL_FLOAT16_VEC4_NV" "GL_AMD_gpu_shader_half_float"
    "GL_FLOAT16_MAT2_AMD" "GL_AMD_gpu_shader_half_float"
    "GL_FLOAT16_MAT3_AMD" "GL_AMD_gpu_shader_half_float"
    "GL_FLOAT16_MAT4_AMD" "GL_AMD_gpu_shader_half_float"
    "GL_FLOAT16_MAT2x3_AMD" "GL_AMD_gpu_shader_half_float"
    "GL_FLOAT16_MAT2x4_AMD" "GL_AMD_gpu_shader_half_float"
    "GL_FLOAT16_MAT3x2_AMD" "GL_AMD_gpu_shader_half_float"
    "GL_FLOAT16_MAT3x4_AMD" "GL_AMD_gpu_shader_half_float"
    "GL_FLOAT16_MAT4x2_AMD" "GL_AMD_gpu_shader_half_float"
    "GL_FLOAT16_MAT4x3_AMD" "GL_AMD_gpu_shader_half_float"
    "GL_AMD_gpu_shader_half_float_fetch" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_SAMPLER_1D_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_SAMPLER_2D_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_SAMPLER_3D_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_SAMPLER_CUBE_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_SAMPLER_2D_RECT_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_SAMPLER_1D_ARRAY_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_SAMPLER_2D_ARRAY_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_SAMPLER_CUBE_MAP_ARRAY_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_SAMPLER_BUFFER_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_SAMPLER_2D_MULTISAMPLE_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_SAMPLER_2D_MULTISAMPLE_ARRAY_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_SAMPLER_1D_SHADOW_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_SAMPLER_2D_SHADOW_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_SAMPLER_2D_RECT_SHADOW_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_SAMPLER_1D_ARRAY_SHADOW_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_SAMPLER_2D_ARRAY_SHADOW_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_SAMPLER_CUBE_SHADOW_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_SAMPLER_CUBE_MAP_ARRAY_SHADOW_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_IMAGE_1D_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_IMAGE_2D_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_IMAGE_3D_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_IMAGE_2D_RECT_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_IMAGE_CUBE_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_IMAGE_1D_ARRAY_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_IMAGE_2D_ARRAY_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_IMAGE_CUBE_MAP_ARRAY_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_IMAGE_BUFFER_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_IMAGE_2D_MULTISAMPLE_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_FLOAT16_IMAGE_2D_MULTISAMPLE_ARRAY_AMD" "GL_AMD_gpu_shader_half_float_fetch"
    "GL_AMD_gpu_shader_int16" "GL_AMD_gpu_shader_int16"
    "GL_AMD_gpu_shader_int64" "GL_AMD_gpu_shader_int64"
    "GL_AMD_interleaved_elements" "GL_AMD_interleaved_elements"
    "GL_RG8UI" "GL_AMD_interleaved_elements"
    "GL_RG16UI" "GL_AMD_interleaved_elements"
    "GL_VERTEX_ELEMENT_SWIZZLE_AMD" "GL_AMD_interleaved_elements"
    "GL_VERTEX_ID_SWIZZLE_AMD" "GL_AMD_interleaved_elements"
    "GL_AMD_multi_draw_indirect" "GL_AMD_multi_draw_indirect"
    "GL_AMD_name_gen_delete" "GL_AMD_name_gen_delete"
    "GL_DATA_BUFFER_AMD" "GL_AMD_name_gen_delete"
    "GL_PERFORMANCE_MONITOR_AMD" "GL_AMD_name_gen_delete"
    "GL_QUERY_OBJECT_AMD" "GL_AMD_name_gen_delete"
    "GL_VERTEX_ARRAY_OBJECT_AMD" "GL_AMD_name_gen_delete"
    "GL_SAMPLER_OBJECT_AMD" "GL_AMD_name_gen_delete"
    "GL_AMD_occlusion_query_event" "GL_AMD_occlusion_query_event"
    "GL_QUERY_DEPTH_PASS_EVENT_BIT_AMD" "GL_AMD_occlusion_query_event"
    "GL_QUERY_DEPTH_FAIL_EVENT_BIT_AMD" "GL_AMD_occlusion_query_event"
    "GL_QUERY_STENCIL_FAIL_EVENT_BIT_AMD" "GL_AMD_occlusion_query_event"
    "GL_QUERY_DEPTH_BOUNDS_FAIL_EVENT_BIT_AMD" "GL_AMD_occlusion_query_event"
    "GL_OCCLUSION_QUERY_EVENT_MASK_AMD" "GL_AMD_occlusion_query_event"
    "GL_QUERY_ALL_EVENT_BITS_AMD" "GL_AMD_occlusion_query_event"
    "GL_AMD_performance_monitor" "GL_AMD_performance_monitor"
    "GL_COUNTER_TYPE_AMD" "GL_AMD_performance_monitor"
    "GL_COUNTER_RANGE_AMD" "GL_AMD_performance_monitor"
    "GL_UNSIGNED_INT64_AMD" "GL_AMD_performance_monitor"
    "GL_PERCENTAGE_AMD" "GL_AMD_performance_monitor"
    "GL_PERFMON_RESULT_AVAILABLE_AMD" "GL_AMD_performance_monitor"
    "GL_PERFMON_RESULT_SIZE_AMD" "GL_AMD_performance_monitor"
    "GL_PERFMON_RESULT_AMD" "GL_AMD_performance_monitor"
    "GL_AMD_pinned_memory" "GL_AMD_pinned_memory"
    "GL_EXTERNAL_VIRTUAL_MEMORY_BUFFER_AMD" "GL_AMD_pinned_memory"
    "GL_AMD_program_binary_Z400" "GL_AMD_program_binary_Z400"
    "GL_Z400_BINARY_AMD" "GL_AMD_program_binary_Z400"
    "GL_AMD_query_buffer_object" "GL_AMD_query_buffer_object"
    "GL_QUERY_BUFFER_AMD" "GL_AMD_query_buffer_object"
    "GL_QUERY_BUFFER_BINDING_AMD" "GL_AMD_query_buffer_object"
    "GL_QUERY_RESULT_NO_WAIT_AMD" "GL_AMD_query_buffer_object"
    "GL_AMD_sample_positions" "GL_AMD_sample_positions"
    "GL_AMD_seamless_cubemap_per_texture" "GL_AMD_seamless_cubemap_per_texture"
    "GL_TEXTURE_CUBE_MAP_SEAMLESS" "GL_AMD_seamless_cubemap_per_texture"
    "GL_AMD_shader_atomic_counter_ops" "GL_AMD_shader_atomic_counter_ops"
    "GL_AMD_shader_ballot" "GL_AMD_shader_ballot"
    "GL_AMD_shader_explicit_vertex_parameter" "GL_AMD_shader_explicit_vertex_parameter"
    "GL_AMD_shader_image_load_store_lod" "GL_AMD_shader_image_load_store_lod"
    "GL_AMD_shader_stencil_export" "GL_AMD_shader_stencil_export"
    "GL_AMD_shader_stencil_value_export" "GL_AMD_shader_stencil_value_export"
    "GL_AMD_shader_trinary_minmax" "GL_AMD_shader_trinary_minmax"
    "GL_AMD_sparse_texture" "GL_AMD_sparse_texture"
    "GL_TEXTURE_STORAGE_SPARSE_BIT_AMD" "GL_AMD_sparse_texture"
    "GL_VIRTUAL_PAGE_SIZE_X_AMD" "GL_AMD_sparse_texture"
    "GL_VIRTUAL_PAGE_SIZE_Y_AMD" "GL_AMD_sparse_texture"
    "GL_VIRTUAL_PAGE_SIZE_Z_AMD" "GL_AMD_sparse_texture"
    "GL_MAX_SPARSE_TEXTURE_SIZE_AMD" "GL_AMD_sparse_texture"
    "GL_MAX_SPARSE_3D_TEXTURE_SIZE_AMD" "GL_AMD_sparse_texture"
    "GL_MAX_SPARSE_ARRAY_TEXTURE_LAYERS" "GL_AMD_sparse_texture"
    "GL_MIN_SPARSE_LEVEL_AMD" "GL_AMD_sparse_texture"
    "GL_MIN_LOD_WARNING_AMD" "GL_AMD_sparse_texture"
    "GL_AMD_stencil_operation_extended" "GL_AMD_stencil_operation_extended"
    "GL_SET_AMD" "GL_AMD_stencil_operation_extended"
    "GL_REPLACE_VALUE_AMD" "GL_AMD_stencil_operation_extended"
    "GL_STENCIL_OP_VALUE_AMD" "GL_AMD_stencil_operation_extended"
    "GL_STENCIL_BACK_OP_VALUE_AMD" "GL_AMD_stencil_operation_extended"
    "GL_AMD_texture_gather_bias_lod" "GL_AMD_texture_gather_bias_lod"
    "GL_AMD_texture_texture4" "GL_AMD_texture_texture4"
    "GL_AMD_transform_feedback3_lines_triangles" "GL_AMD_transform_feedback3_lines_triangles"
    "GL_AMD_transform_feedback4" "GL_AMD_transform_feedback4"
    "GL_STREAM_RASTERIZATION_AMD" "GL_AMD_transform_feedback4"
    "GL_AMD_vertex_shader_layer" "GL_AMD_vertex_shader_layer"
    "GL_AMD_vertex_shader_tessellator" "GL_AMD_vertex_shader_tessellator"
    "GL_SAMPLER_BUFFER_AMD" "GL_AMD_vertex_shader_tessellator"
    "GL_INT_SAMPLER_BUFFER_AMD" "GL_AMD_vertex_shader_tessellator"
    "GL_UNSIGNED_INT_SAMPLER_BUFFER_AMD" "GL_AMD_vertex_shader_tessellator"
    "GL_TESSELLATION_MODE_AMD" "GL_AMD_vertex_shader_tessellator"
    "GL_TESSELLATION_FACTOR_AMD" "GL_AMD_vertex_shader_tessellator"
    "GL_DISCRETE_AMD" "GL_AMD_vertex_shader_tessellator"
    "GL_CONTINUOUS_AMD" "GL_AMD_vertex_shader_tessellator"
    "GL_AMD_vertex_shader_viewport_index" "GL_AMD_vertex_shader_viewport_index"
    "GL_ANDROID_extension_pack_es31a" "GL_ANDROID_extension_pack_es31a"
    "GL_ANGLE_depth_texture" "GL_ANGLE_depth_texture"
    "GL_ANGLE_framebuffer_blit" "GL_ANGLE_framebuffer_blit"
    "GL_DRAW_FRAMEBUFFER_BINDING_ANGLE" "GL_ANGLE_framebuffer_blit"
    "GL_READ_FRAMEBUFFER_ANGLE" "GL_ANGLE_framebuffer_blit"
    "GL_DRAW_FRAMEBUFFER_ANGLE" "GL_ANGLE_framebuffer_blit"
    "GL_READ_FRAMEBUFFER_BINDING_ANGLE" "GL_ANGLE_framebuffer_blit"
    "GL_ANGLE_framebuffer_multisample" "GL_ANGLE_framebuffer_multisample"
    "GL_RENDERBUFFER_SAMPLES_ANGLE" "GL_ANGLE_framebuffer_multisample"
    "GL_FRAMEBUFFER_INCOMPLETE_MULTISAMPLE_ANGLE" "GL_ANGLE_framebuffer_multisample"
    "GL_MAX_SAMPLES_ANGLE" "GL_ANGLE_framebuffer_multisample"
    "GL_ANGLE_instanced_arrays" "GL_ANGLE_instanced_arrays"
    "GL_VERTEX_ATTRIB_ARRAY_DIVISOR_ANGLE" "GL_ANGLE_instanced_arrays"
    "GL_ANGLE_pack_reverse_row_order" "GL_ANGLE_pack_reverse_row_order"
    "GL_PACK_REVERSE_ROW_ORDER_ANGLE" "GL_ANGLE_pack_reverse_row_order"
    "GL_ANGLE_program_binary" "GL_ANGLE_program_binary"
    "GL_PROGRAM_BINARY_ANGLE" "GL_ANGLE_program_binary"
    "GL_ANGLE_texture_compression_dxt1" "GL_ANGLE_texture_compression_dxt1"
    "GL_COMPRESSED_RGB_S3TC_DXT1_ANGLE" "GL_ANGLE_texture_compression_dxt1"
    "GL_COMPRESSED_RGBA_S3TC_DXT1_ANGLE" "GL_ANGLE_texture_compression_dxt1"
    "GL_COMPRESSED_RGBA_S3TC_DXT3_ANGLE" "GL_ANGLE_texture_compression_dxt1"
    "GL_COMPRESSED_RGBA_S3TC_DXT5_ANGLE" "GL_ANGLE_texture_compression_dxt1"
    "GL_ANGLE_texture_compression_dxt3" "GL_ANGLE_texture_compression_dxt3"
    "GL_ANGLE_texture_compression_dxt5" "GL_ANGLE_texture_compression_dxt5"
    "GL_ANGLE_texture_usage" "GL_ANGLE_texture_usage"
    "GL_TEXTURE_USAGE_ANGLE" "GL_ANGLE_texture_usage"
    "GL_FRAMEBUFFER_ATTACHMENT_ANGLE" "GL_ANGLE_texture_usage"
    "GL_ANGLE_timer_query" "GL_ANGLE_timer_query"
    "GL_QUERY_COUNTER_BITS_ANGLE" "GL_ANGLE_timer_query"
    "GL_CURRENT_QUERY_ANGLE" "GL_ANGLE_timer_query"
    "GL_QUERY_RESULT_ANGLE" "GL_ANGLE_timer_query"
    "GL_QUERY_RESULT_AVAILABLE_ANGLE" "GL_ANGLE_timer_query"
    "GL_TIME_ELAPSED_ANGLE" "GL_ANGLE_timer_query"
    "GL_TIMESTAMP_ANGLE" "GL_ANGLE_timer_query"
    "GL_ANGLE_translated_shader_source" "GL_ANGLE_translated_shader_source"
    "GL_TRANSLATED_SHADER_SOURCE_LENGTH_ANGLE" "GL_ANGLE_translated_shader_source"
    "GL_APPLE_aux_depth_stencil" "GL_APPLE_aux_depth_stencil"
    "GL_AUX_DEPTH_STENCIL_APPLE" "GL_APPLE_aux_depth_stencil"
    "GL_APPLE_client_storage" "GL_APPLE_client_storage"
    "GL_UNPACK_CLIENT_STORAGE_APPLE" "GL_APPLE_client_storage"
    "GL_APPLE_clip_distance" "GL_APPLE_clip_distance"
    "GL_MAX_CLIP_DISTANCES_APPLE" "GL_APPLE_clip_distance"
    "GL_CLIP_DISTANCE0_APPLE" "GL_APPLE_clip_distance"
    "GL_CLIP_DISTANCE1_APPLE" "GL_APPLE_clip_distance"
    "GL_CLIP_DISTANCE2_APPLE" "GL_APPLE_clip_distance"
    "GL_CLIP_DISTANCE3_APPLE" "GL_APPLE_clip_distance"
    "GL_CLIP_DISTANCE4_APPLE" "GL_APPLE_clip_distance"
    "GL_CLIP_DISTANCE5_APPLE" "GL_APPLE_clip_distance"
    "GL_CLIP_DISTANCE6_APPLE" "GL_APPLE_clip_distance"
    "GL_CLIP_DISTANCE7_APPLE" "GL_APPLE_clip_distance"
    "GL_APPLE_color_buffer_packed_float" "GL_APPLE_color_buffer_packed_float"
    "GL_APPLE_copy_texture_levels" "GL_APPLE_copy_texture_levels"
    "GL_APPLE_element_array" "GL_APPLE_element_array"
    "GL_ELEMENT_ARRAY_APPLE" "GL_APPLE_element_array"
    "GL_ELEMENT_ARRAY_TYPE_APPLE" "GL_APPLE_element_array"
    "GL_ELEMENT_ARRAY_POINTER_APPLE" "GL_APPLE_element_array"
    "GL_APPLE_fence" "GL_APPLE_fence"
    "GL_DRAW_PIXELS_APPLE" "GL_APPLE_fence"
    "GL_FENCE_APPLE" "GL_APPLE_fence"
    "GL_APPLE_float_pixels" "GL_APPLE_float_pixels"
    "GL_HALF_APPLE" "GL_APPLE_float_pixels"
    "GL_RGBA_FLOAT32_APPLE" "GL_APPLE_float_pixels"
    "GL_RGB_FLOAT32_APPLE" "GL_APPLE_float_pixels"
    "GL_ALPHA_FLOAT32_APPLE" "GL_APPLE_float_pixels"
    "GL_INTENSITY_FLOAT32_APPLE" "GL_APPLE_float_pixels"
    "GL_LUMINANCE_FLOAT32_APPLE" "GL_APPLE_float_pixels"
    "GL_LUMINANCE_ALPHA_FLOAT32_APPLE" "GL_APPLE_float_pixels"
    "GL_RGBA_FLOAT16_APPLE" "GL_APPLE_float_pixels"
    "GL_RGB_FLOAT16_APPLE" "GL_APPLE_float_pixels"
    "GL_ALPHA_FLOAT16_APPLE" "GL_APPLE_float_pixels"
    "GL_INTENSITY_FLOAT16_APPLE" "GL_APPLE_float_pixels"
    "GL_LUMINANCE_FLOAT16_APPLE" "GL_APPLE_float_pixels"
    "GL_LUMINANCE_ALPHA_FLOAT16_APPLE" "GL_APPLE_float_pixels"
    "GL_COLOR_FLOAT_APPLE" "GL_APPLE_float_pixels"
    "GL_APPLE_flush_buffer_range" "GL_APPLE_flush_buffer_range"
    "GL_BUFFER_SERIALIZED_MODIFY_APPLE" "GL_APPLE_flush_buffer_range"
    "GL_BUFFER_FLUSHING_UNMAP_APPLE" "GL_APPLE_flush_buffer_range"
    "GL_APPLE_framebuffer_multisample" "GL_APPLE_framebuffer_multisample"
    "GL_DRAW_FRAMEBUFFER_BINDING_APPLE" "GL_APPLE_framebuffer_multisample"
    "GL_READ_FRAMEBUFFER_APPLE" "GL_APPLE_framebuffer_multisample"
    "GL_DRAW_FRAMEBUFFER_APPLE" "GL_APPLE_framebuffer_multisample"
    "GL_READ_FRAMEBUFFER_BINDING_APPLE" "GL_APPLE_framebuffer_multisample"
    "GL_RENDERBUFFER_SAMPLES_APPLE" "GL_APPLE_framebuffer_multisample"
    "GL_FRAMEBUFFER_INCOMPLETE_MULTISAMPLE_APPLE" "GL_APPLE_framebuffer_multisample"
    "GL_MAX_SAMPLES_APPLE" "GL_APPLE_framebuffer_multisample"
    "GL_APPLE_object_purgeable" "GL_APPLE_object_purgeable"
    "GL_BUFFER_OBJECT_APPLE" "GL_APPLE_object_purgeable"
    "GL_RELEASED_APPLE" "GL_APPLE_object_purgeable"
    "GL_VOLATILE_APPLE" "GL_APPLE_object_purgeable"
    "GL_RETAINED_APPLE" "GL_APPLE_object_purgeable"
    "GL_UNDEFINED_APPLE" "GL_APPLE_object_purgeable"
    "GL_PURGEABLE_APPLE" "GL_APPLE_object_purgeable"
    "GL_APPLE_pixel_buffer" "GL_APPLE_pixel_buffer"
    "GL_MIN_PBUFFER_VIEWPORT_DIMS_APPLE" "GL_APPLE_pixel_buffer"
    "GL_APPLE_rgb_422" "GL_APPLE_rgb_422"
    "GL_UNSIGNED_SHORT_8_8_APPLE" "GL_APPLE_rgb_422"
    "GL_UNSIGNED_SHORT_8_8_REV_APPLE" "GL_APPLE_rgb_422"
    "GL_RGB_422_APPLE" "GL_APPLE_rgb_422"
    "GL_RGB_RAW_422_APPLE" "GL_APPLE_rgb_422"
    "GL_APPLE_row_bytes" "GL_APPLE_row_bytes"
    "GL_PACK_ROW_BYTES_APPLE" "GL_APPLE_row_bytes"
    "GL_UNPACK_ROW_BYTES_APPLE" "GL_APPLE_row_bytes"
    "GL_APPLE_specular_vector" "GL_APPLE_specular_vector"
    "GL_LIGHT_MODEL_SPECULAR_VECTOR_APPLE" "GL_APPLE_specular_vector"
    "GL_APPLE_sync" "GL_APPLE_sync"
    "GL_SYNC_FLUSH_COMMANDS_BIT_APPLE" "GL_APPLE_sync"
    "GL_SYNC_OBJECT_APPLE" "GL_APPLE_sync"
    "GL_MAX_SERVER_WAIT_TIMEOUT_APPLE" "GL_APPLE_sync"
    "GL_OBJECT_TYPE_APPLE" "GL_APPLE_sync"
    "GL_SYNC_CONDITION_APPLE" "GL_APPLE_sync"
    "GL_SYNC_STATUS_APPLE" "GL_APPLE_sync"
    "GL_SYNC_FLAGS_APPLE" "GL_APPLE_sync"
    "GL_SYNC_FENCE_APPLE" "GL_APPLE_sync"
    "GL_SYNC_GPU_COMMANDS_COMPLETE_APPLE" "GL_APPLE_sync"
    "GL_UNSIGNALED_APPLE" "GL_APPLE_sync"
    "GL_SIGNALED_APPLE" "GL_APPLE_sync"
    "GL_ALREADY_SIGNALED_APPLE" "GL_APPLE_sync"
    "GL_TIMEOUT_EXPIRED_APPLE" "GL_APPLE_sync"
    "GL_CONDITION_SATISFIED_APPLE" "GL_APPLE_sync"
    "GL_WAIT_FAILED_APPLE" "GL_APPLE_sync"
    "GL_TIMEOUT_IGNORED_APPLE" "GL_APPLE_sync"
    "GL_APPLE_texture_2D_limited_npot" "GL_APPLE_texture_2D_limited_npot"
    "GL_APPLE_texture_format_BGRA8888" "GL_APPLE_texture_format_BGRA8888"
    "GL_BGRA_EXT" "GL_APPLE_texture_format_BGRA8888"
    "GL_BGRA8_EXT" "GL_APPLE_texture_format_BGRA8888"
    "GL_APPLE_texture_max_level" "GL_APPLE_texture_max_level"
    "GL_TEXTURE_MAX_LEVEL_APPLE" "GL_APPLE_texture_max_level"
    "GL_APPLE_texture_packed_float" "GL_APPLE_texture_packed_float"
    "GL_R11F_G11F_B10F_APPLE" "GL_APPLE_texture_packed_float"
    "GL_UNSIGNED_INT_10F_11F_11F_REV_APPLE" "GL_APPLE_texture_packed_float"
    "GL_RGB9_E5_APPLE" "GL_APPLE_texture_packed_float"
    "GL_UNSIGNED_INT_5_9_9_9_REV_APPLE" "GL_APPLE_texture_packed_float"
    "GL_APPLE_texture_range" "GL_APPLE_texture_range"
    "GL_TEXTURE_RANGE_LENGTH_APPLE" "GL_APPLE_texture_range"
    "GL_TEXTURE_RANGE_POINTER_APPLE" "GL_APPLE_texture_range"
    "GL_TEXTURE_STORAGE_HINT_APPLE" "GL_APPLE_texture_range"
    "GL_STORAGE_PRIVATE_APPLE" "GL_APPLE_texture_range"
    "GL_STORAGE_CACHED_APPLE" "GL_APPLE_texture_range"
    "GL_STORAGE_SHARED_APPLE" "GL_APPLE_texture_range"
    "GL_APPLE_transform_hint" "GL_APPLE_transform_hint"
    "GL_TRANSFORM_HINT_APPLE" "GL_APPLE_transform_hint"
    "GL_APPLE_vertex_array_object" "GL_APPLE_vertex_array_object"
    "GL_VERTEX_ARRAY_BINDING_APPLE" "GL_APPLE_vertex_array_object"
    "GL_APPLE_vertex_array_range" "GL_APPLE_vertex_array_range"
    "GL_VERTEX_ARRAY_RANGE_APPLE" "GL_APPLE_vertex_array_range"
    "GL_VERTEX_ARRAY_RANGE_LENGTH_APPLE" "GL_APPLE_vertex_array_range"
    "GL_VERTEX_ARRAY_STORAGE_HINT_APPLE" "GL_APPLE_vertex_array_range"
    "GL_MAX_VERTEX_ARRAY_RANGE_ELEMENT_APPLE" "GL_APPLE_vertex_array_range"
    "GL_VERTEX_ARRAY_RANGE_POINTER_APPLE" "GL_APPLE_vertex_array_range"
    "GL_STORAGE_CLIENT_APPLE" "GL_APPLE_vertex_array_range"
    "GL_APPLE_vertex_program_evaluators" "GL_APPLE_vertex_program_evaluators"
    "GL_VERTEX_ATTRIB_MAP1_APPLE" "GL_APPLE_vertex_program_evaluators"
    "GL_VERTEX_ATTRIB_MAP2_APPLE" "GL_APPLE_vertex_program_evaluators"
    "GL_VERTEX_ATTRIB_MAP1_SIZE_APPLE" "GL_APPLE_vertex_program_evaluators"
    "GL_VERTEX_ATTRIB_MAP1_COEFF_APPLE" "GL_APPLE_vertex_program_evaluators"
    "GL_VERTEX_ATTRIB_MAP1_ORDER_APPLE" "GL_APPLE_vertex_program_evaluators"
    "GL_VERTEX_ATTRIB_MAP1_DOMAIN_APPLE" "GL_APPLE_vertex_program_evaluators"
    "GL_VERTEX_ATTRIB_MAP2_SIZE_APPLE" "GL_APPLE_vertex_program_evaluators"
    "GL_VERTEX_ATTRIB_MAP2_COEFF_APPLE" "GL_APPLE_vertex_program_evaluators"
    "GL_VERTEX_ATTRIB_MAP2_ORDER_APPLE" "GL_APPLE_vertex_program_evaluators"
    "GL_VERTEX_ATTRIB_MAP2_DOMAIN_APPLE" "GL_APPLE_vertex_program_evaluators"
    "GL_APPLE_ycbcr_422" "GL_APPLE_ycbcr_422"
    "GL_YCBCR_422_APPLE" "GL_APPLE_ycbcr_422"
    "GL_ARB_ES2_compatibility" "GL_ARB_ES2_compatibility"
    "GL_FIXED" "GL_ARB_ES2_compatibility"
    "GL_IMPLEMENTATION_COLOR_READ_TYPE" "GL_ARB_ES2_compatibility"
    "GL_IMPLEMENTATION_COLOR_READ_FORMAT" "GL_ARB_ES2_compatibility"
    "GL_RGB565" "GL_ARB_ES2_compatibility"
    "GL_LOW_FLOAT" "GL_ARB_ES2_compatibility"
    "GL_MEDIUM_FLOAT" "GL_ARB_ES2_compatibility"
    "GL_HIGH_FLOAT" "GL_ARB_ES2_compatibility"
    "GL_LOW_INT" "GL_ARB_ES2_compatibility"
    "GL_MEDIUM_INT" "GL_ARB_ES2_compatibility"
    "GL_HIGH_INT" "GL_ARB_ES2_compatibility"
    "GL_SHADER_BINARY_FORMATS" "GL_ARB_ES2_compatibility"
    "GL_NUM_SHADER_BINARY_FORMATS" "GL_ARB_ES2_compatibility"
    "GL_SHADER_COMPILER" "GL_ARB_ES2_compatibility"
    "GL_MAX_VERTEX_UNIFORM_VECTORS" "GL_ARB_ES2_compatibility"
    "GL_MAX_VARYING_VECTORS" "GL_ARB_ES2_compatibility"
    "GL_MAX_FRAGMENT_UNIFORM_VECTORS" "GL_ARB_ES2_compatibility"
    "GL_ARB_ES3_1_compatibility" "GL_ARB_ES3_1_compatibility"
    "GL_ARB_ES3_2_compatibility" "GL_ARB_ES3_2_compatibility"
    "GL_PRIMITIVE_BOUNDING_BOX_ARB" "GL_ARB_ES3_2_compatibility"
    "GL_MULTISAMPLE_LINE_WIDTH_RANGE_ARB" "GL_ARB_ES3_2_compatibility"
    "GL_MULTISAMPLE_LINE_WIDTH_GRANULARITY_ARB" "GL_ARB_ES3_2_compatibility"
    "GL_ARB_ES3_compatibility" "GL_ARB_ES3_compatibility"
    "GL_TEXTURE_IMMUTABLE_LEVELS" "GL_ARB_ES3_compatibility"
    "GL_PRIMITIVE_RESTART_FIXED_INDEX" "GL_ARB_ES3_compatibility"
    "GL_ANY_SAMPLES_PASSED_CONSERVATIVE" "GL_ARB_ES3_compatibility"
    "GL_MAX_ELEMENT_INDEX" "GL_ARB_ES3_compatibility"
    "GL_COMPRESSED_R11_EAC" "GL_ARB_ES3_compatibility"
    "GL_COMPRESSED_SIGNED_R11_EAC" "GL_ARB_ES3_compatibility"
    "GL_COMPRESSED_RG11_EAC" "GL_ARB_ES3_compatibility"
    "GL_COMPRESSED_SIGNED_RG11_EAC" "GL_ARB_ES3_compatibility"
    "GL_COMPRESSED_RGB8_ETC2" "GL_ARB_ES3_compatibility"
    "GL_COMPRESSED_SRGB8_ETC2" "GL_ARB_ES3_compatibility"
    "GL_COMPRESSED_RGB8_PUNCHTHROUGH_ALPHA1_ETC2" "GL_ARB_ES3_compatibility"
    "GL_COMPRESSED_SRGB8_PUNCHTHROUGH_ALPHA1_ETC2" "GL_ARB_ES3_compatibility"
    "GL_COMPRESSED_RGBA8_ETC2_EAC" "GL_ARB_ES3_compatibility"
    "GL_COMPRESSED_SRGB8_ALPHA8_ETC2_EAC" "GL_ARB_ES3_compatibility"
    "GL_ARB_arrays_of_arrays" "GL_ARB_arrays_of_arrays"
    "GL_ARB_base_instance" "GL_ARB_base_instance"
    "GL_ARB_bindless_texture" "GL_ARB_bindless_texture"
    "GL_UNSIGNED_INT64_ARB" "GL_ARB_bindless_texture"
    "GL_ARB_blend_func_extended" "GL_ARB_blend_func_extended"
    "GL_SRC1_COLOR" "GL_ARB_blend_func_extended"
    "GL_ONE_MINUS_SRC1_COLOR" "GL_ARB_blend_func_extended"
    "GL_ONE_MINUS_SRC1_ALPHA" "GL_ARB_blend_func_extended"
    "GL_MAX_DUAL_SOURCE_DRAW_BUFFERS" "GL_ARB_blend_func_extended"
    "GL_ARB_buffer_storage" "GL_ARB_buffer_storage"
    "GL_MAP_READ_BIT" "GL_ARB_buffer_storage"
    "GL_MAP_WRITE_BIT" "GL_ARB_buffer_storage"
    "GL_MAP_PERSISTENT_BIT" "GL_ARB_buffer_storage"
    "GL_MAP_COHERENT_BIT" "GL_ARB_buffer_storage"
    "GL_DYNAMIC_STORAGE_BIT" "GL_ARB_buffer_storage"
    "GL_CLIENT_STORAGE_BIT" "GL_ARB_buffer_storage"
    "GL_CLIENT_MAPPED_BUFFER_BARRIER_BIT" "GL_ARB_buffer_storage"
    "GL_BUFFER_IMMUTABLE_STORAGE" "GL_ARB_buffer_storage"
    "GL_BUFFER_STORAGE_FLAGS" "GL_ARB_buffer_storage"
    "GL_ARB_cl_event" "GL_ARB_cl_event"
    "GL_SYNC_CL_EVENT_ARB" "GL_ARB_cl_event"
    "GL_SYNC_CL_EVENT_COMPLETE_ARB" "GL_ARB_cl_event"
    "GL_ARB_clear_buffer_object" "GL_ARB_clear_buffer_object"
    "GL_ARB_clear_texture" "GL_ARB_clear_texture"
    "GL_CLEAR_TEXTURE" "GL_ARB_clear_texture"
    "GL_ARB_clip_control" "GL_ARB_clip_control"
    "GL_CLIP_ORIGIN" "GL_ARB_clip_control"
    "GL_CLIP_DEPTH_MODE" "GL_ARB_clip_control"
    "GL_NEGATIVE_ONE_TO_ONE" "GL_ARB_clip_control"
    "GL_ZERO_TO_ONE" "GL_ARB_clip_control"
    "GL_ARB_color_buffer_float" "GL_ARB_color_buffer_float"
    "GL_RGBA_FLOAT_MODE_ARB" "GL_ARB_color_buffer_float"
    "GL_CLAMP_VERTEX_COLOR_ARB" "GL_ARB_color_buffer_float"
    "GL_CLAMP_FRAGMENT_COLOR_ARB" "GL_ARB_color_buffer_float"
    "GL_CLAMP_READ_COLOR_ARB" "GL_ARB_color_buffer_float"
    "GL_FIXED_ONLY_ARB" "GL_ARB_color_buffer_float"
    "GL_ARB_compatibility" "GL_ARB_compatibility"
    "GL_ARB_compressed_texture_pixel_storage" "GL_ARB_compressed_texture_pixel_storage"
    "GL_UNPACK_COMPRESSED_BLOCK_WIDTH" "GL_ARB_compressed_texture_pixel_storage"
    "GL_UNPACK_COMPRESSED_BLOCK_HEIGHT" "GL_ARB_compressed_texture_pixel_storage"
    "GL_UNPACK_COMPRESSED_BLOCK_DEPTH" "GL_ARB_compressed_texture_pixel_storage"
    "GL_UNPACK_COMPRESSED_BLOCK_SIZE" "GL_ARB_compressed_texture_pixel_storage"
    "GL_PACK_COMPRESSED_BLOCK_WIDTH" "GL_ARB_compressed_texture_pixel_storage"
    "GL_PACK_COMPRESSED_BLOCK_HEIGHT" "GL_ARB_compressed_texture_pixel_storage"
    "GL_PACK_COMPRESSED_BLOCK_DEPTH" "GL_ARB_compressed_texture_pixel_storage"
    "GL_PACK_COMPRESSED_BLOCK_SIZE" "GL_ARB_compressed_texture_pixel_storage"
    "GL_ARB_compute_shader" "GL_ARB_compute_shader"
    "GL_COMPUTE_SHADER_BIT" "GL_ARB_compute_shader"
    "GL_MAX_COMPUTE_SHARED_MEMORY_SIZE" "GL_ARB_compute_shader"
    "GL_MAX_COMPUTE_UNIFORM_COMPONENTS" "GL_ARB_compute_shader"
    "GL_MAX_COMPUTE_ATOMIC_COUNTER_BUFFERS" "GL_ARB_compute_shader"
    "GL_MAX_COMPUTE_ATOMIC_COUNTERS" "GL_ARB_compute_shader"
    "GL_MAX_COMBINED_COMPUTE_UNIFORM_COMPONENTS" "GL_ARB_compute_shader"
    "GL_COMPUTE_WORK_GROUP_SIZE" "GL_ARB_compute_shader"
    "GL_MAX_COMPUTE_WORK_GROUP_INVOCATIONS" "GL_ARB_compute_shader"
    "GL_UNIFORM_BLOCK_REFERENCED_BY_COMPUTE_SHADER" "GL_ARB_compute_shader"
    "GL_ATOMIC_COUNTER_BUFFER_REFERENCED_BY_COMPUTE_SHADER" "GL_ARB_compute_shader"
    "GL_DISPATCH_INDIRECT_BUFFER" "GL_ARB_compute_shader"
    "GL_DISPATCH_INDIRECT_BUFFER_BINDING" "GL_ARB_compute_shader"
    "GL_COMPUTE_SHADER" "GL_ARB_compute_shader"
    "GL_MAX_COMPUTE_UNIFORM_BLOCKS" "GL_ARB_compute_shader"
    "GL_MAX_COMPUTE_TEXTURE_IMAGE_UNITS" "GL_ARB_compute_shader"
    "GL_MAX_COMPUTE_IMAGE_UNIFORMS" "GL_ARB_compute_shader"
    "GL_MAX_COMPUTE_WORK_GROUP_COUNT" "GL_ARB_compute_shader"
    "GL_MAX_COMPUTE_WORK_GROUP_SIZE" "GL_ARB_compute_shader"
    "GL_ARB_compute_variable_group_size" "GL_ARB_compute_variable_group_size"
    "GL_MAX_COMPUTE_FIXED_GROUP_INVOCATIONS_ARB" "GL_ARB_compute_variable_group_size"
    "GL_MAX_COMPUTE_FIXED_GROUP_SIZE_ARB" "GL_ARB_compute_variable_group_size"
    "GL_MAX_COMPUTE_VARIABLE_GROUP_INVOCATIONS_ARB" "GL_ARB_compute_variable_group_size"
    "GL_MAX_COMPUTE_VARIABLE_GROUP_SIZE_ARB" "GL_ARB_compute_variable_group_size"
    "GL_ARB_conditional_render_inverted" "GL_ARB_conditional_render_inverted"
    "GL_QUERY_WAIT_INVERTED" "GL_ARB_conditional_render_inverted"
    "GL_QUERY_NO_WAIT_INVERTED" "GL_ARB_conditional_render_inverted"
    "GL_QUERY_BY_REGION_WAIT_INVERTED" "GL_ARB_conditional_render_inverted"
    "GL_QUERY_BY_REGION_NO_WAIT_INVERTED" "GL_ARB_conditional_render_inverted"
    "GL_ARB_conservative_depth" "GL_ARB_conservative_depth"
    "GL_ARB_copy_buffer" "GL_ARB_copy_buffer"
    "GL_COPY_READ_BUFFER" "GL_ARB_copy_buffer"
    "GL_COPY_WRITE_BUFFER" "GL_ARB_copy_buffer"
    "GL_ARB_copy_image" "GL_ARB_copy_image"
    "GL_ARB_cull_distance" "GL_ARB_cull_distance"
    "GL_MAX_CULL_DISTANCES" "GL_ARB_cull_distance"
    "GL_MAX_COMBINED_CLIP_AND_CULL_DISTANCES" "GL_ARB_cull_distance"
    "GL_ARB_debug_output" "GL_ARB_debug_output"
    "GL_DEBUG_OUTPUT_SYNCHRONOUS_ARB" "GL_ARB_debug_output"
    "GL_DEBUG_NEXT_LOGGED_MESSAGE_LENGTH_ARB" "GL_ARB_debug_output"
    "GL_DEBUG_CALLBACK_FUNCTION_ARB" "GL_ARB_debug_output"
    "GL_DEBUG_CALLBACK_USER_PARAM_ARB" "GL_ARB_debug_output"
    "GL_DEBUG_SOURCE_API_ARB" "GL_ARB_debug_output"
    "GL_DEBUG_SOURCE_WINDOW_SYSTEM_ARB" "GL_ARB_debug_output"
    "GL_DEBUG_SOURCE_SHADER_COMPILER_ARB" "GL_ARB_debug_output"
    "GL_DEBUG_SOURCE_THIRD_PARTY_ARB" "GL_ARB_debug_output"
    "GL_DEBUG_SOURCE_APPLICATION_ARB" "GL_ARB_debug_output"
    "GL_DEBUG_SOURCE_OTHER_ARB" "GL_ARB_debug_output"
    "GL_DEBUG_TYPE_ERROR_ARB" "GL_ARB_debug_output"
    "GL_DEBUG_TYPE_DEPRECATED_BEHAVIOR_ARB" "GL_ARB_debug_output"
    "GL_DEBUG_TYPE_UNDEFINED_BEHAVIOR_ARB" "GL_ARB_debug_output"
    "GL_DEBUG_TYPE_PORTABILITY_ARB" "GL_ARB_debug_output"
    "GL_DEBUG_TYPE_PERFORMANCE_ARB" "GL_ARB_debug_output"
    "GL_DEBUG_TYPE_OTHER_ARB" "GL_ARB_debug_output"
    "GL_MAX_DEBUG_MESSAGE_LENGTH_ARB" "GL_ARB_debug_output"
    "GL_MAX_DEBUG_LOGGED_MESSAGES_ARB" "GL_ARB_debug_output"
    "GL_DEBUG_LOGGED_MESSAGES_ARB" "GL_ARB_debug_output"
    "GL_DEBUG_SEVERITY_HIGH_ARB" "GL_ARB_debug_output"
    "GL_DEBUG_SEVERITY_MEDIUM_ARB" "GL_ARB_debug_output"
    "GL_DEBUG_SEVERITY_LOW_ARB" "GL_ARB_debug_output"
    "GL_ARB_depth_buffer_float" "GL_ARB_depth_buffer_float"
    "GL_DEPTH_COMPONENT32F" "GL_ARB_depth_buffer_float"
    "GL_DEPTH32F_STENCIL8" "GL_ARB_depth_buffer_float"
    "GL_FLOAT_32_UNSIGNED_INT_24_8_REV" "GL_ARB_depth_buffer_float"
    "GL_ARB_depth_clamp" "GL_ARB_depth_clamp"
    "GL_DEPTH_CLAMP" "GL_ARB_depth_clamp"
    "GL_ARB_depth_texture" "GL_ARB_depth_texture"
    "GL_DEPTH_COMPONENT16_ARB" "GL_ARB_depth_texture"
    "GL_DEPTH_COMPONENT24_ARB" "GL_ARB_depth_texture"
    "GL_DEPTH_COMPONENT32_ARB" "GL_ARB_depth_texture"
    "GL_TEXTURE_DEPTH_SIZE_ARB" "GL_ARB_depth_texture"
    "GL_DEPTH_TEXTURE_MODE_ARB" "GL_ARB_depth_texture"
    "GL_ARB_derivative_control" "GL_ARB_derivative_control"
    "GL_ARB_direct_state_access" "GL_ARB_direct_state_access"
    "GL_TEXTURE_TARGET" "GL_ARB_direct_state_access"
    "GL_QUERY_TARGET" "GL_ARB_direct_state_access"
    "GL_ARB_draw_buffers" "GL_ARB_draw_buffers"
    "GL_MAX_DRAW_BUFFERS_ARB" "GL_ARB_draw_buffers"
    "GL_DRAW_BUFFER0_ARB" "GL_ARB_draw_buffers"
    "GL_DRAW_BUFFER1_ARB" "GL_ARB_draw_buffers"
    "GL_DRAW_BUFFER2_ARB" "GL_ARB_draw_buffers"
    "GL_DRAW_BUFFER3_ARB" "GL_ARB_draw_buffers"
    "GL_DRAW_BUFFER4_ARB" "GL_ARB_draw_buffers"
    "GL_DRAW_BUFFER5_ARB" "GL_ARB_draw_buffers"
    "GL_DRAW_BUFFER6_ARB" "GL_ARB_draw_buffers"
    "GL_DRAW_BUFFER7_ARB" "GL_ARB_draw_buffers"
    "GL_DRAW_BUFFER8_ARB" "GL_ARB_draw_buffers"
    "GL_DRAW_BUFFER9_ARB" "GL_ARB_draw_buffers"
    "GL_DRAW_BUFFER10_ARB" "GL_ARB_draw_buffers"
    "GL_DRAW_BUFFER11_ARB" "GL_ARB_draw_buffers"
    "GL_DRAW_BUFFER12_ARB" "GL_ARB_draw_buffers"
    "GL_DRAW_BUFFER13_ARB" "GL_ARB_draw_buffers"
    "GL_DRAW_BUFFER14_ARB" "GL_ARB_draw_buffers"
    "GL_DRAW_BUFFER15_ARB" "GL_ARB_draw_buffers"
    "GL_ARB_draw_buffers_blend" "GL_ARB_draw_buffers_blend"
    "GL_ARB_draw_elements_base_vertex" "GL_ARB_draw_elements_base_vertex"
    "GL_ARB_draw_indirect" "GL_ARB_draw_indirect"
    "GL_DRAW_INDIRECT_BUFFER" "GL_ARB_draw_indirect"
    "GL_DRAW_INDIRECT_BUFFER_BINDING" "GL_ARB_draw_indirect"
    "GL_ARB_draw_instanced" "GL_ARB_draw_instanced"
    "GL_ARB_enhanced_layouts" "GL_ARB_enhanced_layouts"
    "GL_LOCATION_COMPONENT" "GL_ARB_enhanced_layouts"
    "GL_TRANSFORM_FEEDBACK_BUFFER_INDEX" "GL_ARB_enhanced_layouts"
    "GL_TRANSFORM_FEEDBACK_BUFFER_STRIDE" "GL_ARB_enhanced_layouts"
    "GL_ARB_explicit_attrib_location" "GL_ARB_explicit_attrib_location"
    "GL_ARB_explicit_uniform_location" "GL_ARB_explicit_uniform_location"
    "GL_MAX_UNIFORM_LOCATIONS" "GL_ARB_explicit_uniform_location"
    "GL_ARB_fragment_coord_conventions" "GL_ARB_fragment_coord_conventions"
    "GL_ARB_fragment_layer_viewport" "GL_ARB_fragment_layer_viewport"
    "GL_ARB_fragment_program" "GL_ARB_fragment_program"
    "GL_FRAGMENT_PROGRAM_ARB" "GL_ARB_fragment_program"
    "GL_PROGRAM_ALU_INSTRUCTIONS_ARB" "GL_ARB_fragment_program"
    "GL_PROGRAM_TEX_INSTRUCTIONS_ARB" "GL_ARB_fragment_program"
    "GL_PROGRAM_TEX_INDIRECTIONS_ARB" "GL_ARB_fragment_program"
    "GL_PROGRAM_NATIVE_ALU_INSTRUCTIONS_ARB" "GL_ARB_fragment_program"
    "GL_PROGRAM_NATIVE_TEX_INSTRUCTIONS_ARB" "GL_ARB_fragment_program"
    "GL_PROGRAM_NATIVE_TEX_INDIRECTIONS_ARB" "GL_ARB_fragment_program"
    "GL_MAX_PROGRAM_ALU_INSTRUCTIONS_ARB" "GL_ARB_fragment_program"
    "GL_MAX_PROGRAM_TEX_INSTRUCTIONS_ARB" "GL_ARB_fragment_program"
    "GL_MAX_PROGRAM_TEX_INDIRECTIONS_ARB" "GL_ARB_fragment_program"
    "GL_MAX_PROGRAM_NATIVE_ALU_INSTRUCTIONS_ARB" "GL_ARB_fragment_program"
    "GL_MAX_PROGRAM_NATIVE_TEX_INSTRUCTIONS_ARB" "GL_ARB_fragment_program"
    "GL_MAX_PROGRAM_NATIVE_TEX_INDIRECTIONS_ARB" "GL_ARB_fragment_program"
    "GL_MAX_TEXTURE_COORDS_ARB" "GL_ARB_fragment_program"
    "GL_MAX_TEXTURE_IMAGE_UNITS_ARB" "GL_ARB_fragment_program"
    "GL_ARB_fragment_program_shadow" "GL_ARB_fragment_program_shadow"
    "GL_ARB_fragment_shader" "GL_ARB_fragment_shader"
    "GL_FRAGMENT_SHADER_ARB" "GL_ARB_fragment_shader"
    "GL_MAX_FRAGMENT_UNIFORM_COMPONENTS_ARB" "GL_ARB_fragment_shader"
    "GL_FRAGMENT_SHADER_DERIVATIVE_HINT_ARB" "GL_ARB_fragment_shader"
    "GL_ARB_fragment_shader_interlock" "GL_ARB_fragment_shader_interlock"
    "GL_ARB_framebuffer_no_attachments" "GL_ARB_framebuffer_no_attachments"
    "GL_FRAMEBUFFER_DEFAULT_WIDTH" "GL_ARB_framebuffer_no_attachments"
    "GL_FRAMEBUFFER_DEFAULT_HEIGHT" "GL_ARB_framebuffer_no_attachments"
    "GL_FRAMEBUFFER_DEFAULT_LAYERS" "GL_ARB_framebuffer_no_attachments"
    "GL_FRAMEBUFFER_DEFAULT_SAMPLES" "GL_ARB_framebuffer_no_attachments"
    "GL_FRAMEBUFFER_DEFAULT_FIXED_SAMPLE_LOCATIONS" "GL_ARB_framebuffer_no_attachments"
    "GL_MAX_FRAMEBUFFER_WIDTH" "GL_ARB_framebuffer_no_attachments"
    "GL_MAX_FRAMEBUFFER_HEIGHT" "GL_ARB_framebuffer_no_attachments"
    "GL_MAX_FRAMEBUFFER_LAYERS" "GL_ARB_framebuffer_no_attachments"
    "GL_MAX_FRAMEBUFFER_SAMPLES" "GL_ARB_framebuffer_no_attachments"
    "GL_ARB_framebuffer_object" "GL_ARB_framebuffer_object"
    "GL_INVALID_FRAMEBUFFER_OPERATION" "GL_ARB_framebuffer_object"
    "GL_FRAMEBUFFER_ATTACHMENT_COLOR_ENCODING" "GL_ARB_framebuffer_object"
    "GL_FRAMEBUFFER_ATTACHMENT_COMPONENT_TYPE" "GL_ARB_framebuffer_object"
    "GL_FRAMEBUFFER_ATTACHMENT_RED_SIZE" "GL_ARB_framebuffer_object"
    "GL_FRAMEBUFFER_ATTACHMENT_GREEN_SIZE" "GL_ARB_framebuffer_object"
    "GL_FRAMEBUFFER_ATTACHMENT_BLUE_SIZE" "GL_ARB_framebuffer_object"
    "GL_FRAMEBUFFER_ATTACHMENT_ALPHA_SIZE" "GL_ARB_framebuffer_object"
    "GL_FRAMEBUFFER_ATTACHMENT_DEPTH_SIZE" "GL_ARB_framebuffer_object"
    "GL_FRAMEBUFFER_ATTACHMENT_STENCIL_SIZE" "GL_ARB_framebuffer_object"
    "GL_FRAMEBUFFER_DEFAULT" "GL_ARB_framebuffer_object"
    "GL_FRAMEBUFFER_UNDEFINED" "GL_ARB_framebuffer_object"
    "GL_DEPTH_STENCIL_ATTACHMENT" "GL_ARB_framebuffer_object"
    "GL_INDEX" "GL_ARB_framebuffer_object"
    "GL_MAX_RENDERBUFFER_SIZE" "GL_ARB_framebuffer_object"
    "GL_DEPTH_STENCIL" "GL_ARB_framebuffer_object"
    "GL_UNSIGNED_INT_24_8" "GL_ARB_framebuffer_object"
    "GL_DEPTH24_STENCIL8" "GL_ARB_framebuffer_object"
    "GL_TEXTURE_STENCIL_SIZE" "GL_ARB_framebuffer_object"
    "GL_UNSIGNED_NORMALIZED" "GL_ARB_framebuffer_object"
    "GL_DRAW_FRAMEBUFFER_BINDING" "GL_ARB_framebuffer_object"
    "GL_FRAMEBUFFER_BINDING" "GL_ARB_framebuffer_object"
    "GL_RENDERBUFFER_BINDING" "GL_ARB_framebuffer_object"
    "GL_READ_FRAMEBUFFER" "GL_ARB_framebuffer_object"
    "GL_DRAW_FRAMEBUFFER" "GL_ARB_framebuffer_object"
    "GL_READ_FRAMEBUFFER_BINDING" "GL_ARB_framebuffer_object"
    "GL_RENDERBUFFER_SAMPLES" "GL_ARB_framebuffer_object"
    "GL_FRAMEBUFFER_ATTACHMENT_OBJECT_TYPE" "GL_ARB_framebuffer_object"
    "GL_FRAMEBUFFER_ATTACHMENT_OBJECT_NAME" "GL_ARB_framebuffer_object"
    "GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_LEVEL" "GL_ARB_framebuffer_object"
    "GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_CUBE_MAP_FACE" "GL_ARB_framebuffer_object"
    "GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_LAYER" "GL_ARB_framebuffer_object"
    "GL_FRAMEBUFFER_COMPLETE" "GL_ARB_framebuffer_object"
    "GL_FRAMEBUFFER_INCOMPLETE_ATTACHMENT" "GL_ARB_framebuffer_object"
    "GL_FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT" "GL_ARB_framebuffer_object"
    "GL_FRAMEBUFFER_INCOMPLETE_DRAW_BUFFER" "GL_ARB_framebuffer_object"
    "GL_FRAMEBUFFER_INCOMPLETE_READ_BUFFER" "GL_ARB_framebuffer_object"
    "GL_FRAMEBUFFER_UNSUPPORTED" "GL_ARB_framebuffer_object"
    "GL_MAX_COLOR_ATTACHMENTS" "GL_ARB_framebuffer_object"
    "GL_COLOR_ATTACHMENT0" "GL_ARB_framebuffer_object"
    "GL_COLOR_ATTACHMENT1" "GL_ARB_framebuffer_object"
    "GL_COLOR_ATTACHMENT2" "GL_ARB_framebuffer_object"
    "GL_COLOR_ATTACHMENT3" "GL_ARB_framebuffer_object"
    "GL_COLOR_ATTACHMENT4" "GL_ARB_framebuffer_object"
    "GL_COLOR_ATTACHMENT5" "GL_ARB_framebuffer_object"
    "GL_COLOR_ATTACHMENT6" "GL_ARB_framebuffer_object"
    "GL_COLOR_ATTACHMENT7" "GL_ARB_framebuffer_object"
    "GL_COLOR_ATTACHMENT8" "GL_ARB_framebuffer_object"
    "GL_COLOR_ATTACHMENT9" "GL_ARB_framebuffer_object"
    "GL_COLOR_ATTACHMENT10" "GL_ARB_framebuffer_object"
    "GL_COLOR_ATTACHMENT11" "GL_ARB_framebuffer_object"
    "GL_COLOR_ATTACHMENT12" "GL_ARB_framebuffer_object"
    "GL_COLOR_ATTACHMENT13" "GL_ARB_framebuffer_object"
    "GL_COLOR_ATTACHMENT14" "GL_ARB_framebuffer_object"
    "GL_COLOR_ATTACHMENT15" "GL_ARB_framebuffer_object"
    "GL_DEPTH_ATTACHMENT" "GL_ARB_framebuffer_object"
    "GL_STENCIL_ATTACHMENT" "GL_ARB_framebuffer_object"
    "GL_FRAMEBUFFER" "GL_ARB_framebuffer_object"
    "GL_RENDERBUFFER" "GL_ARB_framebuffer_object"
    "GL_RENDERBUFFER_WIDTH" "GL_ARB_framebuffer_object"
    "GL_RENDERBUFFER_HEIGHT" "GL_ARB_framebuffer_object"
    "GL_RENDERBUFFER_INTERNAL_FORMAT" "GL_ARB_framebuffer_object"
    "GL_STENCIL_INDEX1" "GL_ARB_framebuffer_object"
    "GL_STENCIL_INDEX4" "GL_ARB_framebuffer_object"
    "GL_STENCIL_INDEX8" "GL_ARB_framebuffer_object"
    "GL_STENCIL_INDEX16" "GL_ARB_framebuffer_object"
    "GL_RENDERBUFFER_RED_SIZE" "GL_ARB_framebuffer_object"
    "GL_RENDERBUFFER_GREEN_SIZE" "GL_ARB_framebuffer_object"
    "GL_RENDERBUFFER_BLUE_SIZE" "GL_ARB_framebuffer_object"
    "GL_RENDERBUFFER_ALPHA_SIZE" "GL_ARB_framebuffer_object"
    "GL_RENDERBUFFER_DEPTH_SIZE" "GL_ARB_framebuffer_object"
    "GL_RENDERBUFFER_STENCIL_SIZE" "GL_ARB_framebuffer_object"
    "GL_FRAMEBUFFER_INCOMPLETE_MULTISAMPLE" "GL_ARB_framebuffer_object"
    "GL_MAX_SAMPLES" "GL_ARB_framebuffer_object"
    "GL_ARB_framebuffer_sRGB" "GL_ARB_framebuffer_sRGB"
    "GL_FRAMEBUFFER_SRGB" "GL_ARB_framebuffer_sRGB"
    "GL_ARB_geometry_shader4" "GL_ARB_geometry_shader4"
    "GL_LINES_ADJACENCY_ARB" "GL_ARB_geometry_shader4"
    "GL_LINE_STRIP_ADJACENCY_ARB" "GL_ARB_geometry_shader4"
    "GL_TRIANGLES_ADJACENCY_ARB" "GL_ARB_geometry_shader4"
    "GL_TRIANGLE_STRIP_ADJACENCY_ARB" "GL_ARB_geometry_shader4"
    "GL_PROGRAM_POINT_SIZE_ARB" "GL_ARB_geometry_shader4"
    "GL_MAX_GEOMETRY_TEXTURE_IMAGE_UNITS_ARB" "GL_ARB_geometry_shader4"
    "GL_FRAMEBUFFER_ATTACHMENT_LAYERED_ARB" "GL_ARB_geometry_shader4"
    "GL_FRAMEBUFFER_INCOMPLETE_LAYER_TARGETS_ARB" "GL_ARB_geometry_shader4"
    "GL_FRAMEBUFFER_INCOMPLETE_LAYER_COUNT_ARB" "GL_ARB_geometry_shader4"
    "GL_GEOMETRY_SHADER_ARB" "GL_ARB_geometry_shader4"
    "GL_GEOMETRY_VERTICES_OUT_ARB" "GL_ARB_geometry_shader4"
    "GL_GEOMETRY_INPUT_TYPE_ARB" "GL_ARB_geometry_shader4"
    "GL_GEOMETRY_OUTPUT_TYPE_ARB" "GL_ARB_geometry_shader4"
    "GL_MAX_GEOMETRY_VARYING_COMPONENTS_ARB" "GL_ARB_geometry_shader4"
    "GL_MAX_VERTEX_VARYING_COMPONENTS_ARB" "GL_ARB_geometry_shader4"
    "GL_MAX_GEOMETRY_UNIFORM_COMPONENTS_ARB" "GL_ARB_geometry_shader4"
    "GL_MAX_GEOMETRY_OUTPUT_VERTICES_ARB" "GL_ARB_geometry_shader4"
    "GL_MAX_GEOMETRY_TOTAL_OUTPUT_COMPONENTS_ARB" "GL_ARB_geometry_shader4"
    "GL_ARB_get_program_binary" "GL_ARB_get_program_binary"
    "GL_PROGRAM_BINARY_RETRIEVABLE_HINT" "GL_ARB_get_program_binary"
    "GL_PROGRAM_BINARY_LENGTH" "GL_ARB_get_program_binary"
    "GL_NUM_PROGRAM_BINARY_FORMATS" "GL_ARB_get_program_binary"
    "GL_PROGRAM_BINARY_FORMATS" "GL_ARB_get_program_binary"
    "GL_ARB_get_texture_sub_image" "GL_ARB_get_texture_sub_image"
    "GL_ARB_gl_spirv" "GL_ARB_gl_spirv"
    "GL_SHADER_BINARY_FORMAT_SPIR_V_ARB" "GL_ARB_gl_spirv"
    "GL_SPIR_V_BINARY_ARB" "GL_ARB_gl_spirv"
    "GL_ARB_gpu_shader5" "GL_ARB_gpu_shader5"
    "GL_GEOMETRY_SHADER_INVOCATIONS" "GL_ARB_gpu_shader5"
    "GL_MAX_GEOMETRY_SHADER_INVOCATIONS" "GL_ARB_gpu_shader5"
    "GL_MIN_FRAGMENT_INTERPOLATION_OFFSET" "GL_ARB_gpu_shader5"
    "GL_MAX_FRAGMENT_INTERPOLATION_OFFSET" "GL_ARB_gpu_shader5"
    "GL_FRAGMENT_INTERPOLATION_OFFSET_BITS" "GL_ARB_gpu_shader5"
    "GL_MAX_VERTEX_STREAMS" "GL_ARB_gpu_shader5"
    "GL_ARB_gpu_shader_fp64" "GL_ARB_gpu_shader_fp64"
    "GL_DOUBLE_MAT2" "GL_ARB_gpu_shader_fp64"
    "GL_DOUBLE_MAT3" "GL_ARB_gpu_shader_fp64"
    "GL_DOUBLE_MAT4" "GL_ARB_gpu_shader_fp64"
    "GL_DOUBLE_MAT2x3" "GL_ARB_gpu_shader_fp64"
    "GL_DOUBLE_MAT2x4" "GL_ARB_gpu_shader_fp64"
    "GL_DOUBLE_MAT3x2" "GL_ARB_gpu_shader_fp64"
    "GL_DOUBLE_MAT3x4" "GL_ARB_gpu_shader_fp64"
    "GL_DOUBLE_MAT4x2" "GL_ARB_gpu_shader_fp64"
    "GL_DOUBLE_MAT4x3" "GL_ARB_gpu_shader_fp64"
    "GL_DOUBLE_VEC2" "GL_ARB_gpu_shader_fp64"
    "GL_DOUBLE_VEC3" "GL_ARB_gpu_shader_fp64"
    "GL_DOUBLE_VEC4" "GL_ARB_gpu_shader_fp64"
    "GL_ARB_gpu_shader_int64" "GL_ARB_gpu_shader_int64"
    "GL_INT64_ARB" "GL_ARB_gpu_shader_int64"
    "GL_INT64_VEC2_ARB" "GL_ARB_gpu_shader_int64"
    "GL_INT64_VEC3_ARB" "GL_ARB_gpu_shader_int64"
    "GL_INT64_VEC4_ARB" "GL_ARB_gpu_shader_int64"
    "GL_UNSIGNED_INT64_VEC2_ARB" "GL_ARB_gpu_shader_int64"
    "GL_UNSIGNED_INT64_VEC3_ARB" "GL_ARB_gpu_shader_int64"
    "GL_UNSIGNED_INT64_VEC4_ARB" "GL_ARB_gpu_shader_int64"
    "GL_ARB_half_float_pixel" "GL_ARB_half_float_pixel"
    "GL_HALF_FLOAT_ARB" "GL_ARB_half_float_pixel"
    "GL_ARB_half_float_vertex" "GL_ARB_half_float_vertex"
    "GL_HALF_FLOAT" "GL_ARB_half_float_vertex"
    "GL_ARB_imaging" "GL_ARB_imaging"
    "GL_CONSTANT_COLOR" "GL_ARB_imaging"
    "GL_ONE_MINUS_CONSTANT_COLOR" "GL_ARB_imaging"
    "GL_CONSTANT_ALPHA" "GL_ARB_imaging"
    "GL_ONE_MINUS_CONSTANT_ALPHA" "GL_ARB_imaging"
    "GL_BLEND_COLOR" "GL_ARB_imaging"
    "GL_FUNC_ADD" "GL_ARB_imaging"
    "GL_MIN" "GL_ARB_imaging"
    "GL_MAX" "GL_ARB_imaging"
    "GL_BLEND_EQUATION" "GL_ARB_imaging"
    "GL_FUNC_SUBTRACT" "GL_ARB_imaging"
    "GL_FUNC_REVERSE_SUBTRACT" "GL_ARB_imaging"
    "GL_CONVOLUTION_1D" "GL_ARB_imaging"
    "GL_CONVOLUTION_2D" "GL_ARB_imaging"
    "GL_SEPARABLE_2D" "GL_ARB_imaging"
    "GL_CONVOLUTION_BORDER_MODE" "GL_ARB_imaging"
    "GL_CONVOLUTION_FILTER_SCALE" "GL_ARB_imaging"
    "GL_CONVOLUTION_FILTER_BIAS" "GL_ARB_imaging"
    "GL_REDUCE" "GL_ARB_imaging"
    "GL_CONVOLUTION_FORMAT" "GL_ARB_imaging"
    "GL_CONVOLUTION_WIDTH" "GL_ARB_imaging"
    "GL_CONVOLUTION_HEIGHT" "GL_ARB_imaging"
    "GL_MAX_CONVOLUTION_WIDTH" "GL_ARB_imaging"
    "GL_MAX_CONVOLUTION_HEIGHT" "GL_ARB_imaging"
    "GL_POST_CONVOLUTION_RED_SCALE" "GL_ARB_imaging"
    "GL_POST_CONVOLUTION_GREEN_SCALE" "GL_ARB_imaging"
    "GL_POST_CONVOLUTION_BLUE_SCALE" "GL_ARB_imaging"
    "GL_POST_CONVOLUTION_ALPHA_SCALE" "GL_ARB_imaging"
    "GL_POST_CONVOLUTION_RED_BIAS" "GL_ARB_imaging"
    "GL_POST_CONVOLUTION_GREEN_BIAS" "GL_ARB_imaging"
    "GL_POST_CONVOLUTION_BLUE_BIAS" "GL_ARB_imaging"
    "GL_POST_CONVOLUTION_ALPHA_BIAS" "GL_ARB_imaging"
    "GL_HISTOGRAM" "GL_ARB_imaging"
    "GL_PROXY_HISTOGRAM" "GL_ARB_imaging"
    "GL_HISTOGRAM_WIDTH" "GL_ARB_imaging"
    "GL_HISTOGRAM_FORMAT" "GL_ARB_imaging"
    "GL_HISTOGRAM_RED_SIZE" "GL_ARB_imaging"
    "GL_HISTOGRAM_GREEN_SIZE" "GL_ARB_imaging"
    "GL_HISTOGRAM_BLUE_SIZE" "GL_ARB_imaging"
    "GL_HISTOGRAM_ALPHA_SIZE" "GL_ARB_imaging"
    "GL_HISTOGRAM_LUMINANCE_SIZE" "GL_ARB_imaging"
    "GL_HISTOGRAM_SINK" "GL_ARB_imaging"
    "GL_MINMAX" "GL_ARB_imaging"
    "GL_MINMAX_FORMAT" "GL_ARB_imaging"
    "GL_MINMAX_SINK" "GL_ARB_imaging"
    "GL_TABLE_TOO_LARGE" "GL_ARB_imaging"
    "GL_COLOR_MATRIX" "GL_ARB_imaging"
    "GL_COLOR_MATRIX_STACK_DEPTH" "GL_ARB_imaging"
    "GL_MAX_COLOR_MATRIX_STACK_DEPTH" "GL_ARB_imaging"
    "GL_POST_COLOR_MATRIX_RED_SCALE" "GL_ARB_imaging"
    "GL_POST_COLOR_MATRIX_GREEN_SCALE" "GL_ARB_imaging"
    "GL_POST_COLOR_MATRIX_BLUE_SCALE" "GL_ARB_imaging"
    "GL_POST_COLOR_MATRIX_ALPHA_SCALE" "GL_ARB_imaging"
    "GL_POST_COLOR_MATRIX_RED_BIAS" "GL_ARB_imaging"
    "GL_POST_COLOR_MATRIX_GREEN_BIAS" "GL_ARB_imaging"
    "GL_POST_COLOR_MATRIX_BLUE_BIAS" "GL_ARB_imaging"
    "GL_POST_COLOR_MATRIX_ALPHA_BIAS" "GL_ARB_imaging"
    "GL_COLOR_TABLE" "GL_ARB_imaging"
    "GL_POST_CONVOLUTION_COLOR_TABLE" "GL_ARB_imaging"
    "GL_POST_COLOR_MATRIX_COLOR_TABLE" "GL_ARB_imaging"
    "GL_PROXY_COLOR_TABLE" "GL_ARB_imaging"
    "GL_PROXY_POST_CONVOLUTION_COLOR_TABLE" "GL_ARB_imaging"
    "GL_PROXY_POST_COLOR_MATRIX_COLOR_TABLE" "GL_ARB_imaging"
    "GL_COLOR_TABLE_SCALE" "GL_ARB_imaging"
    "GL_COLOR_TABLE_BIAS" "GL_ARB_imaging"
    "GL_COLOR_TABLE_FORMAT" "GL_ARB_imaging"
    "GL_COLOR_TABLE_WIDTH" "GL_ARB_imaging"
    "GL_COLOR_TABLE_RED_SIZE" "GL_ARB_imaging"
    "GL_COLOR_TABLE_GREEN_SIZE" "GL_ARB_imaging"
    "GL_COLOR_TABLE_BLUE_SIZE" "GL_ARB_imaging"
    "GL_COLOR_TABLE_ALPHA_SIZE" "GL_ARB_imaging"
    "GL_COLOR_TABLE_LUMINANCE_SIZE" "GL_ARB_imaging"
    "GL_COLOR_TABLE_INTENSITY_SIZE" "GL_ARB_imaging"
    "GL_IGNORE_BORDER" "GL_ARB_imaging"
    "GL_CONSTANT_BORDER" "GL_ARB_imaging"
    "GL_WRAP_BORDER" "GL_ARB_imaging"
    "GL_REPLICATE_BORDER" "GL_ARB_imaging"
    "GL_CONVOLUTION_BORDER_COLOR" "GL_ARB_imaging"
    "GL_ARB_indirect_parameters" "GL_ARB_indirect_parameters"
    "GL_PARAMETER_BUFFER_ARB" "GL_ARB_indirect_parameters"
    "GL_PARAMETER_BUFFER_BINDING_ARB" "GL_ARB_indirect_parameters"
    "GL_ARB_instanced_arrays" "GL_ARB_instanced_arrays"
    "GL_VERTEX_ATTRIB_ARRAY_DIVISOR_ARB" "GL_ARB_instanced_arrays"
    "GL_ARB_internalformat_query" "GL_ARB_internalformat_query"
    "GL_NUM_SAMPLE_COUNTS" "GL_ARB_internalformat_query"
    "GL_ARB_internalformat_query2" "GL_ARB_internalformat_query2"
    "GL_INTERNALFORMAT_SUPPORTED" "GL_ARB_internalformat_query2"
    "GL_INTERNALFORMAT_PREFERRED" "GL_ARB_internalformat_query2"
    "GL_INTERNALFORMAT_RED_SIZE" "GL_ARB_internalformat_query2"
    "GL_INTERNALFORMAT_GREEN_SIZE" "GL_ARB_internalformat_query2"
    "GL_INTERNALFORMAT_BLUE_SIZE" "GL_ARB_internalformat_query2"
    "GL_INTERNALFORMAT_ALPHA_SIZE" "GL_ARB_internalformat_query2"
    "GL_INTERNALFORMAT_DEPTH_SIZE" "GL_ARB_internalformat_query2"
    "GL_INTERNALFORMAT_STENCIL_SIZE" "GL_ARB_internalformat_query2"
    "GL_INTERNALFORMAT_SHARED_SIZE" "GL_ARB_internalformat_query2"
    "GL_INTERNALFORMAT_RED_TYPE" "GL_ARB_internalformat_query2"
    "GL_INTERNALFORMAT_GREEN_TYPE" "GL_ARB_internalformat_query2"
    "GL_INTERNALFORMAT_BLUE_TYPE" "GL_ARB_internalformat_query2"
    "GL_INTERNALFORMAT_ALPHA_TYPE" "GL_ARB_internalformat_query2"
    "GL_INTERNALFORMAT_DEPTH_TYPE" "GL_ARB_internalformat_query2"
    "GL_INTERNALFORMAT_STENCIL_TYPE" "GL_ARB_internalformat_query2"
    "GL_MAX_WIDTH" "GL_ARB_internalformat_query2"
    "GL_MAX_HEIGHT" "GL_ARB_internalformat_query2"
    "GL_MAX_DEPTH" "GL_ARB_internalformat_query2"
    "GL_MAX_LAYERS" "GL_ARB_internalformat_query2"
    "GL_MAX_COMBINED_DIMENSIONS" "GL_ARB_internalformat_query2"
    "GL_COLOR_COMPONENTS" "GL_ARB_internalformat_query2"
    "GL_DEPTH_COMPONENTS" "GL_ARB_internalformat_query2"
    "GL_STENCIL_COMPONENTS" "GL_ARB_internalformat_query2"
    "GL_COLOR_RENDERABLE" "GL_ARB_internalformat_query2"
    "GL_DEPTH_RENDERABLE" "GL_ARB_internalformat_query2"
    "GL_STENCIL_RENDERABLE" "GL_ARB_internalformat_query2"
    "GL_FRAMEBUFFER_RENDERABLE" "GL_ARB_internalformat_query2"
    "GL_FRAMEBUFFER_RENDERABLE_LAYERED" "GL_ARB_internalformat_query2"
    "GL_FRAMEBUFFER_BLEND" "GL_ARB_internalformat_query2"
    "GL_READ_PIXELS" "GL_ARB_internalformat_query2"
    "GL_READ_PIXELS_FORMAT" "GL_ARB_internalformat_query2"
    "GL_READ_PIXELS_TYPE" "GL_ARB_internalformat_query2"
    "GL_TEXTURE_IMAGE_FORMAT" "GL_ARB_internalformat_query2"
    "GL_TEXTURE_IMAGE_TYPE" "GL_ARB_internalformat_query2"
    "GL_GET_TEXTURE_IMAGE_FORMAT" "GL_ARB_internalformat_query2"
    "GL_GET_TEXTURE_IMAGE_TYPE" "GL_ARB_internalformat_query2"
    "GL_MIPMAP" "GL_ARB_internalformat_query2"
    "GL_MANUAL_GENERATE_MIPMAP" "GL_ARB_internalformat_query2"
    "GL_AUTO_GENERATE_MIPMAP" "GL_ARB_internalformat_query2"
    "GL_COLOR_ENCODING" "GL_ARB_internalformat_query2"
    "GL_SRGB_READ" "GL_ARB_internalformat_query2"
    "GL_SRGB_WRITE" "GL_ARB_internalformat_query2"
    "GL_SRGB_DECODE_ARB" "GL_ARB_internalformat_query2"
    "GL_FILTER" "GL_ARB_internalformat_query2"
    "GL_VERTEX_TEXTURE" "GL_ARB_internalformat_query2"
    "GL_TESS_CONTROL_TEXTURE" "GL_ARB_internalformat_query2"
    "GL_TESS_EVALUATION_TEXTURE" "GL_ARB_internalformat_query2"
    "GL_GEOMETRY_TEXTURE" "GL_ARB_internalformat_query2"
    "GL_FRAGMENT_TEXTURE" "GL_ARB_internalformat_query2"
    "GL_COMPUTE_TEXTURE" "GL_ARB_internalformat_query2"
    "GL_TEXTURE_SHADOW" "GL_ARB_internalformat_query2"
    "GL_TEXTURE_GATHER" "GL_ARB_internalformat_query2"
    "GL_TEXTURE_GATHER_SHADOW" "GL_ARB_internalformat_query2"
    "GL_SHADER_IMAGE_LOAD" "GL_ARB_internalformat_query2"
    "GL_SHADER_IMAGE_STORE" "GL_ARB_internalformat_query2"
    "GL_SHADER_IMAGE_ATOMIC" "GL_ARB_internalformat_query2"
    "GL_IMAGE_TEXEL_SIZE" "GL_ARB_internalformat_query2"
    "GL_IMAGE_COMPATIBILITY_CLASS" "GL_ARB_internalformat_query2"
    "GL_IMAGE_PIXEL_FORMAT" "GL_ARB_internalformat_query2"
    "GL_IMAGE_PIXEL_TYPE" "GL_ARB_internalformat_query2"
    "GL_SIMULTANEOUS_TEXTURE_AND_DEPTH_TEST" "GL_ARB_internalformat_query2"
    "GL_SIMULTANEOUS_TEXTURE_AND_STENCIL_TEST" "GL_ARB_internalformat_query2"
    "GL_SIMULTANEOUS_TEXTURE_AND_DEPTH_WRITE" "GL_ARB_internalformat_query2"
    "GL_SIMULTANEOUS_TEXTURE_AND_STENCIL_WRITE" "GL_ARB_internalformat_query2"
    "GL_TEXTURE_COMPRESSED_BLOCK_WIDTH" "GL_ARB_internalformat_query2"
    "GL_TEXTURE_COMPRESSED_BLOCK_HEIGHT" "GL_ARB_internalformat_query2"
    "GL_TEXTURE_COMPRESSED_BLOCK_SIZE" "GL_ARB_internalformat_query2"
    "GL_CLEAR_BUFFER" "GL_ARB_internalformat_query2"
    "GL_TEXTURE_VIEW" "GL_ARB_internalformat_query2"
    "GL_VIEW_COMPATIBILITY_CLASS" "GL_ARB_internalformat_query2"
    "GL_FULL_SUPPORT" "GL_ARB_internalformat_query2"
    "GL_CAVEAT_SUPPORT" "GL_ARB_internalformat_query2"
    "GL_IMAGE_CLASS_4_X_32" "GL_ARB_internalformat_query2"
    "GL_IMAGE_CLASS_2_X_32" "GL_ARB_internalformat_query2"
    "GL_IMAGE_CLASS_1_X_32" "GL_ARB_internalformat_query2"
    "GL_IMAGE_CLASS_4_X_16" "GL_ARB_internalformat_query2"
    "GL_IMAGE_CLASS_2_X_16" "GL_ARB_internalformat_query2"
    "GL_IMAGE_CLASS_1_X_16" "GL_ARB_internalformat_query2"
    "GL_IMAGE_CLASS_4_X_8" "GL_ARB_internalformat_query2"
    "GL_IMAGE_CLASS_2_X_8" "GL_ARB_internalformat_query2"
    "GL_IMAGE_CLASS_1_X_8" "GL_ARB_internalformat_query2"
    "GL_IMAGE_CLASS_11_11_10" "GL_ARB_internalformat_query2"
    "GL_IMAGE_CLASS_10_10_10_2" "GL_ARB_internalformat_query2"
    "GL_VIEW_CLASS_128_BITS" "GL_ARB_internalformat_query2"
    "GL_VIEW_CLASS_96_BITS" "GL_ARB_internalformat_query2"
    "GL_VIEW_CLASS_64_BITS" "GL_ARB_internalformat_query2"
    "GL_VIEW_CLASS_48_BITS" "GL_ARB_internalformat_query2"
    "GL_VIEW_CLASS_32_BITS" "GL_ARB_internalformat_query2"
    "GL_VIEW_CLASS_24_BITS" "GL_ARB_internalformat_query2"
    "GL_VIEW_CLASS_16_BITS" "GL_ARB_internalformat_query2"
    "GL_VIEW_CLASS_8_BITS" "GL_ARB_internalformat_query2"
    "GL_VIEW_CLASS_S3TC_DXT1_RGB" "GL_ARB_internalformat_query2"
    "GL_VIEW_CLASS_S3TC_DXT1_RGBA" "GL_ARB_internalformat_query2"
    "GL_VIEW_CLASS_S3TC_DXT3_RGBA" "GL_ARB_internalformat_query2"
    "GL_VIEW_CLASS_S3TC_DXT5_RGBA" "GL_ARB_internalformat_query2"
    "GL_VIEW_CLASS_RGTC1_RED" "GL_ARB_internalformat_query2"
    "GL_VIEW_CLASS_RGTC2_RG" "GL_ARB_internalformat_query2"
    "GL_VIEW_CLASS_BPTC_UNORM" "GL_ARB_internalformat_query2"
    "GL_VIEW_CLASS_BPTC_FLOAT" "GL_ARB_internalformat_query2"
    "GL_ARB_invalidate_subdata" "GL_ARB_invalidate_subdata"
    "GL_ARB_map_buffer_alignment" "GL_ARB_map_buffer_alignment"
    "GL_MIN_MAP_BUFFER_ALIGNMENT" "GL_ARB_map_buffer_alignment"
    "GL_ARB_map_buffer_range" "GL_ARB_map_buffer_range"
    "GL_MAP_INVALIDATE_RANGE_BIT" "GL_ARB_map_buffer_range"
    "GL_MAP_INVALIDATE_BUFFER_BIT" "GL_ARB_map_buffer_range"
    "GL_MAP_FLUSH_EXPLICIT_BIT" "GL_ARB_map_buffer_range"
    "GL_MAP_UNSYNCHRONIZED_BIT" "GL_ARB_map_buffer_range"
    "GL_ARB_matrix_palette" "GL_ARB_matrix_palette"
    "GL_MATRIX_PALETTE_ARB" "GL_ARB_matrix_palette"
    "GL_MAX_MATRIX_PALETTE_STACK_DEPTH_ARB" "GL_ARB_matrix_palette"
    "GL_MAX_PALETTE_MATRICES_ARB" "GL_ARB_matrix_palette"
    "GL_CURRENT_PALETTE_MATRIX_ARB" "GL_ARB_matrix_palette"
    "GL_MATRIX_INDEX_ARRAY_ARB" "GL_ARB_matrix_palette"
    "GL_CURRENT_MATRIX_INDEX_ARB" "GL_ARB_matrix_palette"
    "GL_MATRIX_INDEX_ARRAY_SIZE_ARB" "GL_ARB_matrix_palette"
    "GL_MATRIX_INDEX_ARRAY_TYPE_ARB" "GL_ARB_matrix_palette"
    "GL_MATRIX_INDEX_ARRAY_STRIDE_ARB" "GL_ARB_matrix_palette"
    "GL_MATRIX_INDEX_ARRAY_POINTER_ARB" "GL_ARB_matrix_palette"
    "GL_ARB_multi_bind" "GL_ARB_multi_bind"
    "GL_ARB_multi_draw_indirect" "GL_ARB_multi_draw_indirect"
    "GL_ARB_multisample" "GL_ARB_multisample"
    "GL_MULTISAMPLE_ARB" "GL_ARB_multisample"
    "GL_SAMPLE_ALPHA_TO_COVERAGE_ARB" "GL_ARB_multisample"
    "GL_SAMPLE_ALPHA_TO_ONE_ARB" "GL_ARB_multisample"
    "GL_SAMPLE_COVERAGE_ARB" "GL_ARB_multisample"
    "GL_SAMPLE_BUFFERS_ARB" "GL_ARB_multisample"
    "GL_SAMPLES_ARB" "GL_ARB_multisample"
    "GL_SAMPLE_COVERAGE_VALUE_ARB" "GL_ARB_multisample"
    "GL_SAMPLE_COVERAGE_INVERT_ARB" "GL_ARB_multisample"
    "GL_MULTISAMPLE_BIT_ARB" "GL_ARB_multisample"
    "GL_ARB_multitexture" "GL_ARB_multitexture"
    "GL_TEXTURE0_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE1_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE2_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE3_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE4_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE5_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE6_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE7_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE8_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE9_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE10_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE11_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE12_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE13_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE14_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE15_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE16_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE17_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE18_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE19_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE20_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE21_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE22_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE23_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE24_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE25_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE26_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE27_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE28_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE29_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE30_ARB" "GL_ARB_multitexture"
    "GL_TEXTURE31_ARB" "GL_ARB_multitexture"
    "GL_ACTIVE_TEXTURE_ARB" "GL_ARB_multitexture"
    "GL_CLIENT_ACTIVE_TEXTURE_ARB" "GL_ARB_multitexture"
    "GL_MAX_TEXTURE_UNITS_ARB" "GL_ARB_multitexture"
    "GL_ARB_occlusion_query" "GL_ARB_occlusion_query"
    "GL_QUERY_COUNTER_BITS_ARB" "GL_ARB_occlusion_query"
    "GL_CURRENT_QUERY_ARB" "GL_ARB_occlusion_query"
    "GL_QUERY_RESULT_ARB" "GL_ARB_occlusion_query"
    "GL_QUERY_RESULT_AVAILABLE_ARB" "GL_ARB_occlusion_query"
    "GL_SAMPLES_PASSED_ARB" "GL_ARB_occlusion_query"
    "GL_ARB_occlusion_query2" "GL_ARB_occlusion_query2"
    "GL_ANY_SAMPLES_PASSED" "GL_ARB_occlusion_query2"
    "GL_ARB_parallel_shader_compile" "GL_ARB_parallel_shader_compile"
    "GL_MAX_SHADER_COMPILER_THREADS_ARB" "GL_ARB_parallel_shader_compile"
    "GL_COMPLETION_STATUS_ARB" "GL_ARB_parallel_shader_compile"
    "GL_ARB_pipeline_statistics_query" "GL_ARB_pipeline_statistics_query"
    "GL_VERTICES_SUBMITTED_ARB" "GL_ARB_pipeline_statistics_query"
    "GL_PRIMITIVES_SUBMITTED_ARB" "GL_ARB_pipeline_statistics_query"
    "GL_VERTEX_SHADER_INVOCATIONS_ARB" "GL_ARB_pipeline_statistics_query"
    "GL_TESS_CONTROL_SHADER_PATCHES_ARB" "GL_ARB_pipeline_statistics_query"
    "GL_TESS_EVALUATION_SHADER_INVOCATIONS_ARB" "GL_ARB_pipeline_statistics_query"
    "GL_GEOMETRY_SHADER_PRIMITIVES_EMITTED_ARB" "GL_ARB_pipeline_statistics_query"
    "GL_FRAGMENT_SHADER_INVOCATIONS_ARB" "GL_ARB_pipeline_statistics_query"
    "GL_COMPUTE_SHADER_INVOCATIONS_ARB" "GL_ARB_pipeline_statistics_query"
    "GL_CLIPPING_INPUT_PRIMITIVES_ARB" "GL_ARB_pipeline_statistics_query"
    "GL_CLIPPING_OUTPUT_PRIMITIVES_ARB" "GL_ARB_pipeline_statistics_query"
    "GL_ARB_pixel_buffer_object" "GL_ARB_pixel_buffer_object"
    "GL_PIXEL_PACK_BUFFER_ARB" "GL_ARB_pixel_buffer_object"
    "GL_PIXEL_UNPACK_BUFFER_ARB" "GL_ARB_pixel_buffer_object"
    "GL_PIXEL_PACK_BUFFER_BINDING_ARB" "GL_ARB_pixel_buffer_object"
    "GL_PIXEL_UNPACK_BUFFER_BINDING_ARB" "GL_ARB_pixel_buffer_object"
    "GL_ARB_point_parameters" "GL_ARB_point_parameters"
    "GL_POINT_SIZE_MIN_ARB" "GL_ARB_point_parameters"
    "GL_POINT_SIZE_MAX_ARB" "GL_ARB_point_parameters"
    "GL_POINT_FADE_THRESHOLD_SIZE_ARB" "GL_ARB_point_parameters"
    "GL_POINT_DISTANCE_ATTENUATION_ARB" "GL_ARB_point_parameters"
    "GL_ARB_point_sprite" "GL_ARB_point_sprite"
    "GL_POINT_SPRITE_ARB" "GL_ARB_point_sprite"
    "GL_COORD_REPLACE_ARB" "GL_ARB_point_sprite"
    "GL_ARB_polygon_offset_clamp" "GL_ARB_polygon_offset_clamp"
    "GL_ARB_post_depth_coverage" "GL_ARB_post_depth_coverage"
    "GL_ARB_program_interface_query" "GL_ARB_program_interface_query"
    "GL_UNIFORM" "GL_ARB_program_interface_query"
    "GL_UNIFORM_BLOCK" "GL_ARB_program_interface_query"
    "GL_PROGRAM_INPUT" "GL_ARB_program_interface_query"
    "GL_PROGRAM_OUTPUT" "GL_ARB_program_interface_query"
    "GL_BUFFER_VARIABLE" "GL_ARB_program_interface_query"
    "GL_SHADER_STORAGE_BLOCK" "GL_ARB_program_interface_query"
    "GL_IS_PER_PATCH" "GL_ARB_program_interface_query"
    "GL_VERTEX_SUBROUTINE" "GL_ARB_program_interface_query"
    "GL_TESS_CONTROL_SUBROUTINE" "GL_ARB_program_interface_query"
    "GL_TESS_EVALUATION_SUBROUTINE" "GL_ARB_program_interface_query"
    "GL_GEOMETRY_SUBROUTINE" "GL_ARB_program_interface_query"
    "GL_FRAGMENT_SUBROUTINE" "GL_ARB_program_interface_query"
    "GL_COMPUTE_SUBROUTINE" "GL_ARB_program_interface_query"
    "GL_VERTEX_SUBROUTINE_UNIFORM" "GL_ARB_program_interface_query"
    "GL_TESS_CONTROL_SUBROUTINE_UNIFORM" "GL_ARB_program_interface_query"
    "GL_TESS_EVALUATION_SUBROUTINE_UNIFORM" "GL_ARB_program_interface_query"
    "GL_GEOMETRY_SUBROUTINE_UNIFORM" "GL_ARB_program_interface_query"
    "GL_FRAGMENT_SUBROUTINE_UNIFORM" "GL_ARB_program_interface_query"
    "GL_COMPUTE_SUBROUTINE_UNIFORM" "GL_ARB_program_interface_query"
    "GL_TRANSFORM_FEEDBACK_VARYING" "GL_ARB_program_interface_query"
    "GL_ACTIVE_RESOURCES" "GL_ARB_program_interface_query"
    "GL_MAX_NAME_LENGTH" "GL_ARB_program_interface_query"
    "GL_MAX_NUM_ACTIVE_VARIABLES" "GL_ARB_program_interface_query"
    "GL_MAX_NUM_COMPATIBLE_SUBROUTINES" "GL_ARB_program_interface_query"
    "GL_NAME_LENGTH" "GL_ARB_program_interface_query"
    "GL_TYPE" "GL_ARB_program_interface_query"
    "GL_ARRAY_SIZE" "GL_ARB_program_interface_query"
    "GL_OFFSET" "GL_ARB_program_interface_query"
    "GL_BLOCK_INDEX" "GL_ARB_program_interface_query"
    "GL_ARRAY_STRIDE" "GL_ARB_program_interface_query"
    "GL_MATRIX_STRIDE" "GL_ARB_program_interface_query"
    "GL_IS_ROW_MAJOR" "GL_ARB_program_interface_query"
    "GL_ATOMIC_COUNTER_BUFFER_INDEX" "GL_ARB_program_interface_query"
    "GL_BUFFER_BINDING" "GL_ARB_program_interface_query"
    "GL_BUFFER_DATA_SIZE" "GL_ARB_program_interface_query"
    "GL_NUM_ACTIVE_VARIABLES" "GL_ARB_program_interface_query"
    "GL_ACTIVE_VARIABLES" "GL_ARB_program_interface_query"
    "GL_REFERENCED_BY_VERTEX_SHADER" "GL_ARB_program_interface_query"
    "GL_REFERENCED_BY_TESS_CONTROL_SHADER" "GL_ARB_program_interface_query"
    "GL_REFERENCED_BY_TESS_EVALUATION_SHADER" "GL_ARB_program_interface_query"
    "GL_REFERENCED_BY_GEOMETRY_SHADER" "GL_ARB_program_interface_query"
    "GL_REFERENCED_BY_FRAGMENT_SHADER" "GL_ARB_program_interface_query"
    "GL_REFERENCED_BY_COMPUTE_SHADER" "GL_ARB_program_interface_query"
    "GL_TOP_LEVEL_ARRAY_SIZE" "GL_ARB_program_interface_query"
    "GL_TOP_LEVEL_ARRAY_STRIDE" "GL_ARB_program_interface_query"
    "GL_LOCATION" "GL_ARB_program_interface_query"
    "GL_LOCATION_INDEX" "GL_ARB_program_interface_query"
    "GL_ARB_provoking_vertex" "GL_ARB_provoking_vertex"
    "GL_QUADS_FOLLOW_PROVOKING_VERTEX_CONVENTION" "GL_ARB_provoking_vertex"
    "GL_FIRST_VERTEX_CONVENTION" "GL_ARB_provoking_vertex"
    "GL_LAST_VERTEX_CONVENTION" "GL_ARB_provoking_vertex"
    "GL_PROVOKING_VERTEX" "GL_ARB_provoking_vertex"
    "GL_ARB_query_buffer_object" "GL_ARB_query_buffer_object"
    "GL_QUERY_BUFFER_BARRIER_BIT" "GL_ARB_query_buffer_object"
    "GL_QUERY_BUFFER" "GL_ARB_query_buffer_object"
    "GL_QUERY_BUFFER_BINDING" "GL_ARB_query_buffer_object"
    "GL_QUERY_RESULT_NO_WAIT" "GL_ARB_query_buffer_object"
    "GL_ARB_robust_buffer_access_behavior" "GL_ARB_robust_buffer_access_behavior"
    "GL_ARB_robustness" "GL_ARB_robustness"
    "GL_CONTEXT_FLAG_ROBUST_ACCESS_BIT_ARB" "GL_ARB_robustness"
    "GL_LOSE_CONTEXT_ON_RESET_ARB" "GL_ARB_robustness"
    "GL_GUILTY_CONTEXT_RESET_ARB" "GL_ARB_robustness"
    "GL_INNOCENT_CONTEXT_RESET_ARB" "GL_ARB_robustness"
    "GL_UNKNOWN_CONTEXT_RESET_ARB" "GL_ARB_robustness"
    "GL_RESET_NOTIFICATION_STRATEGY_ARB" "GL_ARB_robustness"
    "GL_NO_RESET_NOTIFICATION_ARB" "GL_ARB_robustness"
    "GL_ARB_robustness_application_isolation" "GL_ARB_robustness_application_isolation"
    "GL_ARB_robustness_share_group_isolation" "GL_ARB_robustness_share_group_isolation"
    "GL_ARB_sample_locations" "GL_ARB_sample_locations"
    "GL_SAMPLE_LOCATION_ARB" "GL_ARB_sample_locations"
    "GL_SAMPLE_LOCATION_SUBPIXEL_BITS_ARB" "GL_ARB_sample_locations"
    "GL_SAMPLE_LOCATION_PIXEL_GRID_WIDTH_ARB" "GL_ARB_sample_locations"
    "GL_SAMPLE_LOCATION_PIXEL_GRID_HEIGHT_ARB" "GL_ARB_sample_locations"
    "GL_PROGRAMMABLE_SAMPLE_LOCATION_TABLE_SIZE_ARB" "GL_ARB_sample_locations"
    "GL_PROGRAMMABLE_SAMPLE_LOCATION_ARB" "GL_ARB_sample_locations"
    "GL_FRAMEBUFFER_PROGRAMMABLE_SAMPLE_LOCATIONS_ARB" "GL_ARB_sample_locations"
    "GL_FRAMEBUFFER_SAMPLE_LOCATION_PIXEL_GRID_ARB" "GL_ARB_sample_locations"
    "GL_ARB_sample_shading" "GL_ARB_sample_shading"
    "GL_SAMPLE_SHADING_ARB" "GL_ARB_sample_shading"
    "GL_MIN_SAMPLE_SHADING_VALUE_ARB" "GL_ARB_sample_shading"
    "GL_ARB_sampler_objects" "GL_ARB_sampler_objects"
    "GL_SAMPLER_BINDING" "GL_ARB_sampler_objects"
    "GL_ARB_seamless_cube_map" "GL_ARB_seamless_cube_map"
    "GL_ARB_seamless_cubemap_per_texture" "GL_ARB_seamless_cubemap_per_texture"
    "GL_ARB_separate_shader_objects" "GL_ARB_separate_shader_objects"
    "GL_VERTEX_SHADER_BIT" "GL_ARB_separate_shader_objects"
    "GL_FRAGMENT_SHADER_BIT" "GL_ARB_separate_shader_objects"
    "GL_GEOMETRY_SHADER_BIT" "GL_ARB_separate_shader_objects"
    "GL_TESS_CONTROL_SHADER_BIT" "GL_ARB_separate_shader_objects"
    "GL_TESS_EVALUATION_SHADER_BIT" "GL_ARB_separate_shader_objects"
    "GL_PROGRAM_SEPARABLE" "GL_ARB_separate_shader_objects"
    "GL_ACTIVE_PROGRAM" "GL_ARB_separate_shader_objects"
    "GL_PROGRAM_PIPELINE_BINDING" "GL_ARB_separate_shader_objects"
    "GL_ALL_SHADER_BITS" "GL_ARB_separate_shader_objects"
    "GL_ARB_shader_atomic_counter_ops" "GL_ARB_shader_atomic_counter_ops"
    "GL_ARB_shader_atomic_counters" "GL_ARB_shader_atomic_counters"
    "GL_ATOMIC_COUNTER_BUFFER" "GL_ARB_shader_atomic_counters"
    "GL_ATOMIC_COUNTER_BUFFER_BINDING" "GL_ARB_shader_atomic_counters"
    "GL_ATOMIC_COUNTER_BUFFER_START" "GL_ARB_shader_atomic_counters"
    "GL_ATOMIC_COUNTER_BUFFER_SIZE" "GL_ARB_shader_atomic_counters"
    "GL_ATOMIC_COUNTER_BUFFER_DATA_SIZE" "GL_ARB_shader_atomic_counters"
    "GL_ATOMIC_COUNTER_BUFFER_ACTIVE_ATOMIC_COUNTERS" "GL_ARB_shader_atomic_counters"
    "GL_ATOMIC_COUNTER_BUFFER_ACTIVE_ATOMIC_COUNTER_INDICES" "GL_ARB_shader_atomic_counters"
    "GL_ATOMIC_COUNTER_BUFFER_REFERENCED_BY_VERTEX_SHADER" "GL_ARB_shader_atomic_counters"
    "GL_ATOMIC_COUNTER_BUFFER_REFERENCED_BY_TESS_CONTROL_SHADER" "GL_ARB_shader_atomic_counters"
    "GL_ATOMIC_COUNTER_BUFFER_REFERENCED_BY_TESS_EVALUATION_SHADER" "GL_ARB_shader_atomic_counters"
    "GL_ATOMIC_COUNTER_BUFFER_REFERENCED_BY_GEOMETRY_SHADER" "GL_ARB_shader_atomic_counters"
    "GL_ATOMIC_COUNTER_BUFFER_REFERENCED_BY_FRAGMENT_SHADER" "GL_ARB_shader_atomic_counters"
    "GL_MAX_VERTEX_ATOMIC_COUNTER_BUFFERS" "GL_ARB_shader_atomic_counters"
    "GL_MAX_TESS_CONTROL_ATOMIC_COUNTER_BUFFERS" "GL_ARB_shader_atomic_counters"
    "GL_MAX_TESS_EVALUATION_ATOMIC_COUNTER_BUFFERS" "GL_ARB_shader_atomic_counters"
    "GL_MAX_GEOMETRY_ATOMIC_COUNTER_BUFFERS" "GL_ARB_shader_atomic_counters"
    "GL_MAX_FRAGMENT_ATOMIC_COUNTER_BUFFERS" "GL_ARB_shader_atomic_counters"
    "GL_MAX_COMBINED_ATOMIC_COUNTER_BUFFERS" "GL_ARB_shader_atomic_counters"
    "GL_MAX_VERTEX_ATOMIC_COUNTERS" "GL_ARB_shader_atomic_counters"
    "GL_MAX_TESS_CONTROL_ATOMIC_COUNTERS" "GL_ARB_shader_atomic_counters"
    "GL_MAX_TESS_EVALUATION_ATOMIC_COUNTERS" "GL_ARB_shader_atomic_counters"
    "GL_MAX_GEOMETRY_ATOMIC_COUNTERS" "GL_ARB_shader_atomic_counters"
    "GL_MAX_FRAGMENT_ATOMIC_COUNTERS" "GL_ARB_shader_atomic_counters"
    "GL_MAX_COMBINED_ATOMIC_COUNTERS" "GL_ARB_shader_atomic_counters"
    "GL_MAX_ATOMIC_COUNTER_BUFFER_SIZE" "GL_ARB_shader_atomic_counters"
    "GL_ACTIVE_ATOMIC_COUNTER_BUFFERS" "GL_ARB_shader_atomic_counters"
    "GL_UNIFORM_ATOMIC_COUNTER_BUFFER_INDEX" "GL_ARB_shader_atomic_counters"
    "GL_UNSIGNED_INT_ATOMIC_COUNTER" "GL_ARB_shader_atomic_counters"
    "GL_MAX_ATOMIC_COUNTER_BUFFER_BINDINGS" "GL_ARB_shader_atomic_counters"
    "GL_ARB_shader_ballot" "GL_ARB_shader_ballot"
    "GL_ARB_shader_bit_encoding" "GL_ARB_shader_bit_encoding"
    "GL_ARB_shader_clock" "GL_ARB_shader_clock"
    "GL_ARB_shader_draw_parameters" "GL_ARB_shader_draw_parameters"
    "GL_ARB_shader_group_vote" "GL_ARB_shader_group_vote"
    "GL_ARB_shader_image_load_store" "GL_ARB_shader_image_load_store"
    "GL_VERTEX_ATTRIB_ARRAY_BARRIER_BIT" "GL_ARB_shader_image_load_store"
    "GL_ELEMENT_ARRAY_BARRIER_BIT" "GL_ARB_shader_image_load_store"
    "GL_UNIFORM_BARRIER_BIT" "GL_ARB_shader_image_load_store"
    "GL_TEXTURE_FETCH_BARRIER_BIT" "GL_ARB_shader_image_load_store"
    "GL_SHADER_IMAGE_ACCESS_BARRIER_BIT" "GL_ARB_shader_image_load_store"
    "GL_COMMAND_BARRIER_BIT" "GL_ARB_shader_image_load_store"
    "GL_PIXEL_BUFFER_BARRIER_BIT" "GL_ARB_shader_image_load_store"
    "GL_TEXTURE_UPDATE_BARRIER_BIT" "GL_ARB_shader_image_load_store"
    "GL_BUFFER_UPDATE_BARRIER_BIT" "GL_ARB_shader_image_load_store"
    "GL_FRAMEBUFFER_BARRIER_BIT" "GL_ARB_shader_image_load_store"
    "GL_TRANSFORM_FEEDBACK_BARRIER_BIT" "GL_ARB_shader_image_load_store"
    "GL_ATOMIC_COUNTER_BARRIER_BIT" "GL_ARB_shader_image_load_store"
    "GL_MAX_IMAGE_UNITS" "GL_ARB_shader_image_load_store"
    "GL_MAX_COMBINED_IMAGE_UNITS_AND_FRAGMENT_OUTPUTS" "GL_ARB_shader_image_load_store"
    "GL_IMAGE_BINDING_NAME" "GL_ARB_shader_image_load_store"
    "GL_IMAGE_BINDING_LEVEL" "GL_ARB_shader_image_load_store"
    "GL_IMAGE_BINDING_LAYERED" "GL_ARB_shader_image_load_store"
    "GL_IMAGE_BINDING_LAYER" "GL_ARB_shader_image_load_store"
    "GL_IMAGE_BINDING_ACCESS" "GL_ARB_shader_image_load_store"
    "GL_IMAGE_1D" "GL_ARB_shader_image_load_store"
    "GL_IMAGE_2D" "GL_ARB_shader_image_load_store"
    "GL_IMAGE_3D" "GL_ARB_shader_image_load_store"
    "GL_IMAGE_2D_RECT" "GL_ARB_shader_image_load_store"
    "GL_IMAGE_CUBE" "GL_ARB_shader_image_load_store"
    "GL_IMAGE_BUFFER" "GL_ARB_shader_image_load_store"
    "GL_IMAGE_1D_ARRAY" "GL_ARB_shader_image_load_store"
    "GL_IMAGE_2D_ARRAY" "GL_ARB_shader_image_load_store"
    "GL_IMAGE_CUBE_MAP_ARRAY" "GL_ARB_shader_image_load_store"
    "GL_IMAGE_2D_MULTISAMPLE" "GL_ARB_shader_image_load_store"
    "GL_IMAGE_2D_MULTISAMPLE_ARRAY" "GL_ARB_shader_image_load_store"
    "GL_INT_IMAGE_1D" "GL_ARB_shader_image_load_store"
    "GL_INT_IMAGE_2D" "GL_ARB_shader_image_load_store"
    "GL_INT_IMAGE_3D" "GL_ARB_shader_image_load_store"
    "GL_INT_IMAGE_2D_RECT" "GL_ARB_shader_image_load_store"
    "GL_INT_IMAGE_CUBE" "GL_ARB_shader_image_load_store"
    "GL_INT_IMAGE_BUFFER" "GL_ARB_shader_image_load_store"
    "GL_INT_IMAGE_1D_ARRAY" "GL_ARB_shader_image_load_store"
    "GL_INT_IMAGE_2D_ARRAY" "GL_ARB_shader_image_load_store"
    "GL_INT_IMAGE_CUBE_MAP_ARRAY" "GL_ARB_shader_image_load_store"
    "GL_INT_IMAGE_2D_MULTISAMPLE" "GL_ARB_shader_image_load_store"
    "GL_INT_IMAGE_2D_MULTISAMPLE_ARRAY" "GL_ARB_shader_image_load_store"
    "GL_UNSIGNED_INT_IMAGE_1D" "GL_ARB_shader_image_load_store"
    "GL_UNSIGNED_INT_IMAGE_2D" "GL_ARB_shader_image_load_store"
    "GL_UNSIGNED_INT_IMAGE_3D" "GL_ARB_shader_image_load_store"
    "GL_UNSIGNED_INT_IMAGE_2D_RECT" "GL_ARB_shader_image_load_store"
    "GL_UNSIGNED_INT_IMAGE_CUBE" "GL_ARB_shader_image_load_store"
    "GL_UNSIGNED_INT_IMAGE_BUFFER" "GL_ARB_shader_image_load_store"
    "GL_UNSIGNED_INT_IMAGE_1D_ARRAY" "GL_ARB_shader_image_load_store"
    "GL_UNSIGNED_INT_IMAGE_2D_ARRAY" "GL_ARB_shader_image_load_store"
    "GL_UNSIGNED_INT_IMAGE_CUBE_MAP_ARRAY" "GL_ARB_shader_image_load_store"
    "GL_UNSIGNED_INT_IMAGE_2D_MULTISAMPLE" "GL_ARB_shader_image_load_store"
    "GL_UNSIGNED_INT_IMAGE_2D_MULTISAMPLE_ARRAY" "GL_ARB_shader_image_load_store"
    "GL_MAX_IMAGE_SAMPLES" "GL_ARB_shader_image_load_store"
    "GL_IMAGE_BINDING_FORMAT" "GL_ARB_shader_image_load_store"
    "GL_IMAGE_FORMAT_COMPATIBILITY_TYPE" "GL_ARB_shader_image_load_store"
    "GL_IMAGE_FORMAT_COMPATIBILITY_BY_SIZE" "GL_ARB_shader_image_load_store"
    "GL_IMAGE_FORMAT_COMPATIBILITY_BY_CLASS" "GL_ARB_shader_image_load_store"
    "GL_MAX_VERTEX_IMAGE_UNIFORMS" "GL_ARB_shader_image_load_store"
    "GL_MAX_TESS_CONTROL_IMAGE_UNIFORMS" "GL_ARB_shader_image_load_store"
    "GL_MAX_TESS_EVALUATION_IMAGE_UNIFORMS" "GL_ARB_shader_image_load_store"
    "GL_MAX_GEOMETRY_IMAGE_UNIFORMS" "GL_ARB_shader_image_load_store"
    "GL_MAX_FRAGMENT_IMAGE_UNIFORMS" "GL_ARB_shader_image_load_store"
    "GL_MAX_COMBINED_IMAGE_UNIFORMS" "GL_ARB_shader_image_load_store"
    "GL_ALL_BARRIER_BITS" "GL_ARB_shader_image_load_store"
    "GL_ARB_shader_image_size" "GL_ARB_shader_image_size"
    "GL_ARB_shader_objects" "GL_ARB_shader_objects"
    "GL_PROGRAM_OBJECT_ARB" "GL_ARB_shader_objects"
    "GL_SHADER_OBJECT_ARB" "GL_ARB_shader_objects"
    "GL_OBJECT_TYPE_ARB" "GL_ARB_shader_objects"
    "GL_OBJECT_SUBTYPE_ARB" "GL_ARB_shader_objects"
    "GL_FLOAT_VEC2_ARB" "GL_ARB_shader_objects"
    "GL_FLOAT_VEC3_ARB" "GL_ARB_shader_objects"
    "GL_FLOAT_VEC4_ARB" "GL_ARB_shader_objects"
    "GL_INT_VEC2_ARB" "GL_ARB_shader_objects"
    "GL_INT_VEC3_ARB" "GL_ARB_shader_objects"
    "GL_INT_VEC4_ARB" "GL_ARB_shader_objects"
    "GL_BOOL_ARB" "GL_ARB_shader_objects"
    "GL_BOOL_VEC2_ARB" "GL_ARB_shader_objects"
    "GL_BOOL_VEC3_ARB" "GL_ARB_shader_objects"
    "GL_BOOL_VEC4_ARB" "GL_ARB_shader_objects"
    "GL_FLOAT_MAT2_ARB" "GL_ARB_shader_objects"
    "GL_FLOAT_MAT3_ARB" "GL_ARB_shader_objects"
    "GL_FLOAT_MAT4_ARB" "GL_ARB_shader_objects"
    "GL_SAMPLER_1D_ARB" "GL_ARB_shader_objects"
    "GL_SAMPLER_2D_ARB" "GL_ARB_shader_objects"
    "GL_SAMPLER_3D_ARB" "GL_ARB_shader_objects"
    "GL_SAMPLER_CUBE_ARB" "GL_ARB_shader_objects"
    "GL_SAMPLER_1D_SHADOW_ARB" "GL_ARB_shader_objects"
    "GL_SAMPLER_2D_SHADOW_ARB" "GL_ARB_shader_objects"
    "GL_SAMPLER_2D_RECT_ARB" "GL_ARB_shader_objects"
    "GL_SAMPLER_2D_RECT_SHADOW_ARB" "GL_ARB_shader_objects"
    "GL_OBJECT_DELETE_STATUS_ARB" "GL_ARB_shader_objects"
    "GL_OBJECT_COMPILE_STATUS_ARB" "GL_ARB_shader_objects"
    "GL_OBJECT_LINK_STATUS_ARB" "GL_ARB_shader_objects"
    "GL_OBJECT_VALIDATE_STATUS_ARB" "GL_ARB_shader_objects"
    "GL_OBJECT_INFO_LOG_LENGTH_ARB" "GL_ARB_shader_objects"
    "GL_OBJECT_ATTACHED_OBJECTS_ARB" "GL_ARB_shader_objects"
    "GL_OBJECT_ACTIVE_UNIFORMS_ARB" "GL_ARB_shader_objects"
    "GL_OBJECT_ACTIVE_UNIFORM_MAX_LENGTH_ARB" "GL_ARB_shader_objects"
    "GL_OBJECT_SHADER_SOURCE_LENGTH_ARB" "GL_ARB_shader_objects"
    "GL_ARB_shader_precision" "GL_ARB_shader_precision"
    "GL_ARB_shader_stencil_export" "GL_ARB_shader_stencil_export"
    "GL_ARB_shader_storage_buffer_object" "GL_ARB_shader_storage_buffer_object"
    "GL_SHADER_STORAGE_BARRIER_BIT" "GL_ARB_shader_storage_buffer_object"
    "GL_MAX_COMBINED_SHADER_OUTPUT_RESOURCES" "GL_ARB_shader_storage_buffer_object"
    "GL_SHADER_STORAGE_BUFFER" "GL_ARB_shader_storage_buffer_object"
    "GL_SHADER_STORAGE_BUFFER_BINDING" "GL_ARB_shader_storage_buffer_object"
    "GL_SHADER_STORAGE_BUFFER_START" "GL_ARB_shader_storage_buffer_object"
    "GL_SHADER_STORAGE_BUFFER_SIZE" "GL_ARB_shader_storage_buffer_object"
    "GL_MAX_VERTEX_SHADER_STORAGE_BLOCKS" "GL_ARB_shader_storage_buffer_object"
    "GL_MAX_GEOMETRY_SHADER_STORAGE_BLOCKS" "GL_ARB_shader_storage_buffer_object"
    "GL_MAX_TESS_CONTROL_SHADER_STORAGE_BLOCKS" "GL_ARB_shader_storage_buffer_object"
    "GL_MAX_TESS_EVALUATION_SHADER_STORAGE_BLOCKS" "GL_ARB_shader_storage_buffer_object"
    "GL_MAX_FRAGMENT_SHADER_STORAGE_BLOCKS" "GL_ARB_shader_storage_buffer_object"
    "GL_MAX_COMPUTE_SHADER_STORAGE_BLOCKS" "GL_ARB_shader_storage_buffer_object"
    "GL_MAX_COMBINED_SHADER_STORAGE_BLOCKS" "GL_ARB_shader_storage_buffer_object"
    "GL_MAX_SHADER_STORAGE_BUFFER_BINDINGS" "GL_ARB_shader_storage_buffer_object"
    "GL_MAX_SHADER_STORAGE_BLOCK_SIZE" "GL_ARB_shader_storage_buffer_object"
    "GL_SHADER_STORAGE_BUFFER_OFFSET_ALIGNMENT" "GL_ARB_shader_storage_buffer_object"
    "GL_ARB_shader_subroutine" "GL_ARB_shader_subroutine"
    "GL_ACTIVE_SUBROUTINES" "GL_ARB_shader_subroutine"
    "GL_ACTIVE_SUBROUTINE_UNIFORMS" "GL_ARB_shader_subroutine"
    "GL_MAX_SUBROUTINES" "GL_ARB_shader_subroutine"
    "GL_MAX_SUBROUTINE_UNIFORM_LOCATIONS" "GL_ARB_shader_subroutine"
    "GL_ACTIVE_SUBROUTINE_UNIFORM_LOCATIONS" "GL_ARB_shader_subroutine"
    "GL_ACTIVE_SUBROUTINE_MAX_LENGTH" "GL_ARB_shader_subroutine"
    "GL_ACTIVE_SUBROUTINE_UNIFORM_MAX_LENGTH" "GL_ARB_shader_subroutine"
    "GL_NUM_COMPATIBLE_SUBROUTINES" "GL_ARB_shader_subroutine"
    "GL_COMPATIBLE_SUBROUTINES" "GL_ARB_shader_subroutine"
    "GL_ARB_shader_texture_image_samples" "GL_ARB_shader_texture_image_samples"
    "GL_ARB_shader_texture_lod" "GL_ARB_shader_texture_lod"
    "GL_ARB_shader_viewport_layer_array" "GL_ARB_shader_viewport_layer_array"
    "GL_ARB_shading_language_100" "GL_ARB_shading_language_100"
    "GL_SHADING_LANGUAGE_VERSION_ARB" "GL_ARB_shading_language_100"
    "GL_ARB_shading_language_420pack" "GL_ARB_shading_language_420pack"
    "GL_ARB_shading_language_include" "GL_ARB_shading_language_include"
    "GL_SHADER_INCLUDE_ARB" "GL_ARB_shading_language_include"
    "GL_NAMED_STRING_LENGTH_ARB" "GL_ARB_shading_language_include"
    "GL_NAMED_STRING_TYPE_ARB" "GL_ARB_shading_language_include"
    "GL_ARB_shading_language_packing" "GL_ARB_shading_language_packing"
    "GL_ARB_shadow" "GL_ARB_shadow"
    "GL_TEXTURE_COMPARE_MODE_ARB" "GL_ARB_shadow"
    "GL_TEXTURE_COMPARE_FUNC_ARB" "GL_ARB_shadow"
    "GL_COMPARE_R_TO_TEXTURE_ARB" "GL_ARB_shadow"
    "GL_ARB_shadow_ambient" "GL_ARB_shadow_ambient"
    "GL_TEXTURE_COMPARE_FAIL_VALUE_ARB" "GL_ARB_shadow_ambient"
    "GL_ARB_sparse_buffer" "GL_ARB_sparse_buffer"
    "GL_SPARSE_STORAGE_BIT_ARB" "GL_ARB_sparse_buffer"
    "GL_SPARSE_BUFFER_PAGE_SIZE_ARB" "GL_ARB_sparse_buffer"
    "GL_ARB_sparse_texture" "GL_ARB_sparse_texture"
    "GL_VIRTUAL_PAGE_SIZE_X_ARB" "GL_ARB_sparse_texture"
    "GL_VIRTUAL_PAGE_SIZE_Y_ARB" "GL_ARB_sparse_texture"
    "GL_VIRTUAL_PAGE_SIZE_Z_ARB" "GL_ARB_sparse_texture"
    "GL_MAX_SPARSE_TEXTURE_SIZE_ARB" "GL_ARB_sparse_texture"
    "GL_MAX_SPARSE_3D_TEXTURE_SIZE_ARB" "GL_ARB_sparse_texture"
    "GL_MAX_SPARSE_ARRAY_TEXTURE_LAYERS_ARB" "GL_ARB_sparse_texture"
    "GL_TEXTURE_SPARSE_ARB" "GL_ARB_sparse_texture"
    "GL_VIRTUAL_PAGE_SIZE_INDEX_ARB" "GL_ARB_sparse_texture"
    "GL_NUM_VIRTUAL_PAGE_SIZES_ARB" "GL_ARB_sparse_texture"
    "GL_SPARSE_TEXTURE_FULL_ARRAY_CUBE_MIPMAPS_ARB" "GL_ARB_sparse_texture"
    "GL_NUM_SPARSE_LEVELS_ARB" "GL_ARB_sparse_texture"
    "GL_ARB_sparse_texture2" "GL_ARB_sparse_texture2"
    "GL_ARB_sparse_texture_clamp" "GL_ARB_sparse_texture_clamp"
    "GL_ARB_spirv_extensions" "GL_ARB_spirv_extensions"
    "GL_ARB_stencil_texturing" "GL_ARB_stencil_texturing"
    "GL_DEPTH_STENCIL_TEXTURE_MODE" "GL_ARB_stencil_texturing"
    "GL_ARB_sync" "GL_ARB_sync"
    "GL_SYNC_FLUSH_COMMANDS_BIT" "GL_ARB_sync"
    "GL_MAX_SERVER_WAIT_TIMEOUT" "GL_ARB_sync"
    "GL_OBJECT_TYPE" "GL_ARB_sync"
    "GL_SYNC_CONDITION" "GL_ARB_sync"
    "GL_SYNC_STATUS" "GL_ARB_sync"
    "GL_SYNC_FLAGS" "GL_ARB_sync"
    "GL_SYNC_FENCE" "GL_ARB_sync"
    "GL_SYNC_GPU_COMMANDS_COMPLETE" "GL_ARB_sync"
    "GL_UNSIGNALED" "GL_ARB_sync"
    "GL_SIGNALED" "GL_ARB_sync"
    "GL_ALREADY_SIGNALED" "GL_ARB_sync"
    "GL_TIMEOUT_EXPIRED" "GL_ARB_sync"
    "GL_CONDITION_SATISFIED" "GL_ARB_sync"
    "GL_WAIT_FAILED" "GL_ARB_sync"
    "GL_TIMEOUT_IGNORED" "GL_ARB_sync"
    "GL_ARB_tessellation_shader" "GL_ARB_tessellation_shader"
    "GL_PATCHES" "GL_ARB_tessellation_shader"
    "GL_UNIFORM_BLOCK_REFERENCED_BY_TESS_CONTROL_SHADER" "GL_ARB_tessellation_shader"
    "GL_UNIFORM_BLOCK_REFERENCED_BY_TESS_EVALUATION_SHADER" "GL_ARB_tessellation_shader"
    "GL_MAX_TESS_CONTROL_INPUT_COMPONENTS" "GL_ARB_tessellation_shader"
    "GL_MAX_TESS_EVALUATION_INPUT_COMPONENTS" "GL_ARB_tessellation_shader"
    "GL_MAX_COMBINED_TESS_CONTROL_UNIFORM_COMPONENTS" "GL_ARB_tessellation_shader"
    "GL_MAX_COMBINED_TESS_EVALUATION_UNIFORM_COMPONENTS" "GL_ARB_tessellation_shader"
    "GL_PATCH_VERTICES" "GL_ARB_tessellation_shader"
    "GL_PATCH_DEFAULT_INNER_LEVEL" "GL_ARB_tessellation_shader"
    "GL_PATCH_DEFAULT_OUTER_LEVEL" "GL_ARB_tessellation_shader"
    "GL_TESS_CONTROL_OUTPUT_VERTICES" "GL_ARB_tessellation_shader"
    "GL_TESS_GEN_MODE" "GL_ARB_tessellation_shader"
    "GL_TESS_GEN_SPACING" "GL_ARB_tessellation_shader"
    "GL_TESS_GEN_VERTEX_ORDER" "GL_ARB_tessellation_shader"
    "GL_TESS_GEN_POINT_MODE" "GL_ARB_tessellation_shader"
    "GL_ISOLINES" "GL_ARB_tessellation_shader"
    "GL_FRACTIONAL_ODD" "GL_ARB_tessellation_shader"
    "GL_FRACTIONAL_EVEN" "GL_ARB_tessellation_shader"
    "GL_MAX_PATCH_VERTICES" "GL_ARB_tessellation_shader"
    "GL_MAX_TESS_GEN_LEVEL" "GL_ARB_tessellation_shader"
    "GL_MAX_TESS_CONTROL_UNIFORM_COMPONENTS" "GL_ARB_tessellation_shader"
    "GL_MAX_TESS_EVALUATION_UNIFORM_COMPONENTS" "GL_ARB_tessellation_shader"
    "GL_MAX_TESS_CONTROL_TEXTURE_IMAGE_UNITS" "GL_ARB_tessellation_shader"
    "GL_MAX_TESS_EVALUATION_TEXTURE_IMAGE_UNITS" "GL_ARB_tessellation_shader"
    "GL_MAX_TESS_CONTROL_OUTPUT_COMPONENTS" "GL_ARB_tessellation_shader"
    "GL_MAX_TESS_PATCH_COMPONENTS" "GL_ARB_tessellation_shader"
    "GL_MAX_TESS_CONTROL_TOTAL_OUTPUT_COMPONENTS" "GL_ARB_tessellation_shader"
    "GL_MAX_TESS_EVALUATION_OUTPUT_COMPONENTS" "GL_ARB_tessellation_shader"
    "GL_TESS_EVALUATION_SHADER" "GL_ARB_tessellation_shader"
    "GL_TESS_CONTROL_SHADER" "GL_ARB_tessellation_shader"
    "GL_MAX_TESS_CONTROL_UNIFORM_BLOCKS" "GL_ARB_tessellation_shader"
    "GL_MAX_TESS_EVALUATION_UNIFORM_BLOCKS" "GL_ARB_tessellation_shader"
    "GL_ARB_texture_barrier" "GL_ARB_texture_barrier"
    "GL_ARB_texture_border_clamp" "GL_ARB_texture_border_clamp"
    "GL_CLAMP_TO_BORDER_ARB" "GL_ARB_texture_border_clamp"
    "GL_ARB_texture_buffer_object" "GL_ARB_texture_buffer_object"
    "GL_TEXTURE_BUFFER_ARB" "GL_ARB_texture_buffer_object"
    "GL_MAX_TEXTURE_BUFFER_SIZE_ARB" "GL_ARB_texture_buffer_object"
    "GL_TEXTURE_BINDING_BUFFER_ARB" "GL_ARB_texture_buffer_object"
    "GL_TEXTURE_BUFFER_DATA_STORE_BINDING_ARB" "GL_ARB_texture_buffer_object"
    "GL_TEXTURE_BUFFER_FORMAT_ARB" "GL_ARB_texture_buffer_object"
    "GL_ARB_texture_buffer_object_rgb32" "GL_ARB_texture_buffer_object_rgb32"
    "GL_ARB_texture_buffer_range" "GL_ARB_texture_buffer_range"
    "GL_TEXTURE_BUFFER_OFFSET" "GL_ARB_texture_buffer_range"
    "GL_TEXTURE_BUFFER_SIZE" "GL_ARB_texture_buffer_range"
    "GL_TEXTURE_BUFFER_OFFSET_ALIGNMENT" "GL_ARB_texture_buffer_range"
    "GL_ARB_texture_compression" "GL_ARB_texture_compression"
    "GL_COMPRESSED_ALPHA_ARB" "GL_ARB_texture_compression"
    "GL_COMPRESSED_LUMINANCE_ARB" "GL_ARB_texture_compression"
    "GL_COMPRESSED_LUMINANCE_ALPHA_ARB" "GL_ARB_texture_compression"
    "GL_COMPRESSED_INTENSITY_ARB" "GL_ARB_texture_compression"
    "GL_COMPRESSED_RGB_ARB" "GL_ARB_texture_compression"
    "GL_COMPRESSED_RGBA_ARB" "GL_ARB_texture_compression"
    "GL_TEXTURE_COMPRESSION_HINT_ARB" "GL_ARB_texture_compression"
    "GL_TEXTURE_COMPRESSED_IMAGE_SIZE_ARB" "GL_ARB_texture_compression"
    "GL_TEXTURE_COMPRESSED_ARB" "GL_ARB_texture_compression"
    "GL_NUM_COMPRESSED_TEXTURE_FORMATS_ARB" "GL_ARB_texture_compression"
    "GL_COMPRESSED_TEXTURE_FORMATS_ARB" "GL_ARB_texture_compression"
    "GL_ARB_texture_compression_bptc" "GL_ARB_texture_compression_bptc"
    "GL_COMPRESSED_RGBA_BPTC_UNORM_ARB" "GL_ARB_texture_compression_bptc"
    "GL_COMPRESSED_SRGB_ALPHA_BPTC_UNORM_ARB" "GL_ARB_texture_compression_bptc"
    "GL_COMPRESSED_RGB_BPTC_SIGNED_FLOAT_ARB" "GL_ARB_texture_compression_bptc"
    "GL_COMPRESSED_RGB_BPTC_UNSIGNED_FLOAT_ARB" "GL_ARB_texture_compression_bptc"
    "GL_ARB_texture_compression_rgtc" "GL_ARB_texture_compression_rgtc"
    "GL_COMPRESSED_RED_RGTC1" "GL_ARB_texture_compression_rgtc"
    "GL_COMPRESSED_SIGNED_RED_RGTC1" "GL_ARB_texture_compression_rgtc"
    "GL_COMPRESSED_RG_RGTC2" "GL_ARB_texture_compression_rgtc"
    "GL_COMPRESSED_SIGNED_RG_RGTC2" "GL_ARB_texture_compression_rgtc"
    "GL_ARB_texture_cube_map" "GL_ARB_texture_cube_map"
    "GL_NORMAL_MAP_ARB" "GL_ARB_texture_cube_map"
    "GL_REFLECTION_MAP_ARB" "GL_ARB_texture_cube_map"
    "GL_TEXTURE_CUBE_MAP_ARB" "GL_ARB_texture_cube_map"
    "GL_TEXTURE_BINDING_CUBE_MAP_ARB" "GL_ARB_texture_cube_map"
    "GL_TEXTURE_CUBE_MAP_POSITIVE_X_ARB" "GL_ARB_texture_cube_map"
    "GL_TEXTURE_CUBE_MAP_NEGATIVE_X_ARB" "GL_ARB_texture_cube_map"
    "GL_TEXTURE_CUBE_MAP_POSITIVE_Y_ARB" "GL_ARB_texture_cube_map"
    "GL_TEXTURE_CUBE_MAP_NEGATIVE_Y_ARB" "GL_ARB_texture_cube_map"
    "GL_TEXTURE_CUBE_MAP_POSITIVE_Z_ARB" "GL_ARB_texture_cube_map"
    "GL_TEXTURE_CUBE_MAP_NEGATIVE_Z_ARB" "GL_ARB_texture_cube_map"
    "GL_PROXY_TEXTURE_CUBE_MAP_ARB" "GL_ARB_texture_cube_map"
    "GL_MAX_CUBE_MAP_TEXTURE_SIZE_ARB" "GL_ARB_texture_cube_map"
    "GL_ARB_texture_cube_map_array" "GL_ARB_texture_cube_map_array"
    "GL_TEXTURE_CUBE_MAP_ARRAY_ARB" "GL_ARB_texture_cube_map_array"
    "GL_TEXTURE_BINDING_CUBE_MAP_ARRAY_ARB" "GL_ARB_texture_cube_map_array"
    "GL_PROXY_TEXTURE_CUBE_MAP_ARRAY_ARB" "GL_ARB_texture_cube_map_array"
    "GL_SAMPLER_CUBE_MAP_ARRAY_ARB" "GL_ARB_texture_cube_map_array"
    "GL_SAMPLER_CUBE_MAP_ARRAY_SHADOW_ARB" "GL_ARB_texture_cube_map_array"
    "GL_INT_SAMPLER_CUBE_MAP_ARRAY_ARB" "GL_ARB_texture_cube_map_array"
    "GL_UNSIGNED_INT_SAMPLER_CUBE_MAP_ARRAY_ARB" "GL_ARB_texture_cube_map_array"
    "GL_ARB_texture_env_add" "GL_ARB_texture_env_add"
    "GL_ARB_texture_env_combine" "GL_ARB_texture_env_combine"
    "GL_SUBTRACT_ARB" "GL_ARB_texture_env_combine"
    "GL_COMBINE_ARB" "GL_ARB_texture_env_combine"
    "GL_COMBINE_RGB_ARB" "GL_ARB_texture_env_combine"
    "GL_COMBINE_ALPHA_ARB" "GL_ARB_texture_env_combine"
    "GL_RGB_SCALE_ARB" "GL_ARB_texture_env_combine"
    "GL_ADD_SIGNED_ARB" "GL_ARB_texture_env_combine"
    "GL_INTERPOLATE_ARB" "GL_ARB_texture_env_combine"
    "GL_CONSTANT_ARB" "GL_ARB_texture_env_combine"
    "GL_PRIMARY_COLOR_ARB" "GL_ARB_texture_env_combine"
    "GL_PREVIOUS_ARB" "GL_ARB_texture_env_combine"
    "GL_SOURCE0_RGB_ARB" "GL_ARB_texture_env_combine"
    "GL_SOURCE1_RGB_ARB" "GL_ARB_texture_env_combine"
    "GL_SOURCE2_RGB_ARB" "GL_ARB_texture_env_combine"
    "GL_SOURCE0_ALPHA_ARB" "GL_ARB_texture_env_combine"
    "GL_SOURCE1_ALPHA_ARB" "GL_ARB_texture_env_combine"
    "GL_SOURCE2_ALPHA_ARB" "GL_ARB_texture_env_combine"
    "GL_OPERAND0_RGB_ARB" "GL_ARB_texture_env_combine"
    "GL_OPERAND1_RGB_ARB" "GL_ARB_texture_env_combine"
    "GL_OPERAND2_RGB_ARB" "GL_ARB_texture_env_combine"
    "GL_OPERAND0_ALPHA_ARB" "GL_ARB_texture_env_combine"
    "GL_OPERAND1_ALPHA_ARB" "GL_ARB_texture_env_combine"
    "GL_OPERAND2_ALPHA_ARB" "GL_ARB_texture_env_combine"
    "GL_ARB_texture_env_crossbar" "GL_ARB_texture_env_crossbar"
    "GL_ARB_texture_env_dot3" "GL_ARB_texture_env_dot3"
    "GL_DOT3_RGB_ARB" "GL_ARB_texture_env_dot3"
    "GL_DOT3_RGBA_ARB" "GL_ARB_texture_env_dot3"
    "GL_ARB_texture_filter_anisotropic" "GL_ARB_texture_filter_anisotropic"
    "GL_ARB_texture_filter_minmax" "GL_ARB_texture_filter_minmax"
    "GL_TEXTURE_REDUCTION_MODE_ARB" "GL_ARB_texture_filter_minmax"
    "GL_WEIGHTED_AVERAGE_ARB" "GL_ARB_texture_filter_minmax"
    "GL_ARB_texture_float" "GL_ARB_texture_float"
    "GL_RGBA32F_ARB" "GL_ARB_texture_float"
    "GL_RGB32F_ARB" "GL_ARB_texture_float"
    "GL_ALPHA32F_ARB" "GL_ARB_texture_float"
    "GL_INTENSITY32F_ARB" "GL_ARB_texture_float"
    "GL_LUMINANCE32F_ARB" "GL_ARB_texture_float"
    "GL_LUMINANCE_ALPHA32F_ARB" "GL_ARB_texture_float"
    "GL_RGBA16F_ARB" "GL_ARB_texture_float"
    "GL_RGB16F_ARB" "GL_ARB_texture_float"
    "GL_ALPHA16F_ARB" "GL_ARB_texture_float"
    "GL_INTENSITY16F_ARB" "GL_ARB_texture_float"
    "GL_LUMINANCE16F_ARB" "GL_ARB_texture_float"
    "GL_LUMINANCE_ALPHA16F_ARB" "GL_ARB_texture_float"
    "GL_TEXTURE_RED_TYPE_ARB" "GL_ARB_texture_float"
    "GL_TEXTURE_GREEN_TYPE_ARB" "GL_ARB_texture_float"
    "GL_TEXTURE_BLUE_TYPE_ARB" "GL_ARB_texture_float"
    "GL_TEXTURE_ALPHA_TYPE_ARB" "GL_ARB_texture_float"
    "GL_TEXTURE_LUMINANCE_TYPE_ARB" "GL_ARB_texture_float"
    "GL_TEXTURE_INTENSITY_TYPE_ARB" "GL_ARB_texture_float"
    "GL_TEXTURE_DEPTH_TYPE_ARB" "GL_ARB_texture_float"
    "GL_UNSIGNED_NORMALIZED_ARB" "GL_ARB_texture_float"
    "GL_ARB_texture_gather" "GL_ARB_texture_gather"
    "GL_MIN_PROGRAM_TEXTURE_GATHER_OFFSET_ARB" "GL_ARB_texture_gather"
    "GL_MAX_PROGRAM_TEXTURE_GATHER_OFFSET_ARB" "GL_ARB_texture_gather"
    "GL_MAX_PROGRAM_TEXTURE_GATHER_COMPONENTS_ARB" "GL_ARB_texture_gather"
    "GL_ARB_texture_mirror_clamp_to_edge" "GL_ARB_texture_mirror_clamp_to_edge"
    "GL_MIRROR_CLAMP_TO_EDGE" "GL_ARB_texture_mirror_clamp_to_edge"
    "GL_ARB_texture_mirrored_repeat" "GL_ARB_texture_mirrored_repeat"
    "GL_MIRRORED_REPEAT_ARB" "GL_ARB_texture_mirrored_repeat"
    "GL_ARB_texture_multisample" "GL_ARB_texture_multisample"
    "GL_SAMPLE_POSITION" "GL_ARB_texture_multisample"
    "GL_SAMPLE_MASK" "GL_ARB_texture_multisample"
    "GL_SAMPLE_MASK_VALUE" "GL_ARB_texture_multisample"
    "GL_MAX_SAMPLE_MASK_WORDS" "GL_ARB_texture_multisample"
    "GL_TEXTURE_2D_MULTISAMPLE" "GL_ARB_texture_multisample"
    "GL_PROXY_TEXTURE_2D_MULTISAMPLE" "GL_ARB_texture_multisample"
    "GL_TEXTURE_2D_MULTISAMPLE_ARRAY" "GL_ARB_texture_multisample"
    "GL_PROXY_TEXTURE_2D_MULTISAMPLE_ARRAY" "GL_ARB_texture_multisample"
    "GL_TEXTURE_BINDING_2D_MULTISAMPLE" "GL_ARB_texture_multisample"
    "GL_TEXTURE_BINDING_2D_MULTISAMPLE_ARRAY" "GL_ARB_texture_multisample"
    "GL_TEXTURE_SAMPLES" "GL_ARB_texture_multisample"
    "GL_TEXTURE_FIXED_SAMPLE_LOCATIONS" "GL_ARB_texture_multisample"
    "GL_SAMPLER_2D_MULTISAMPLE" "GL_ARB_texture_multisample"
    "GL_INT_SAMPLER_2D_MULTISAMPLE" "GL_ARB_texture_multisample"
    "GL_UNSIGNED_INT_SAMPLER_2D_MULTISAMPLE" "GL_ARB_texture_multisample"
    "GL_SAMPLER_2D_MULTISAMPLE_ARRAY" "GL_ARB_texture_multisample"
    "GL_INT_SAMPLER_2D_MULTISAMPLE_ARRAY" "GL_ARB_texture_multisample"
    "GL_UNSIGNED_INT_SAMPLER_2D_MULTISAMPLE_ARRAY" "GL_ARB_texture_multisample"
    "GL_MAX_COLOR_TEXTURE_SAMPLES" "GL_ARB_texture_multisample"
    "GL_MAX_DEPTH_TEXTURE_SAMPLES" "GL_ARB_texture_multisample"
    "GL_MAX_INTEGER_SAMPLES" "GL_ARB_texture_multisample"
    "GL_ARB_texture_non_power_of_two" "GL_ARB_texture_non_power_of_two"
    "GL_ARB_texture_query_levels" "GL_ARB_texture_query_levels"
    "GL_ARB_texture_query_lod" "GL_ARB_texture_query_lod"
    "GL_ARB_texture_rectangle" "GL_ARB_texture_rectangle"
    "GL_TEXTURE_RECTANGLE_ARB" "GL_ARB_texture_rectangle"
    "GL_TEXTURE_BINDING_RECTANGLE_ARB" "GL_ARB_texture_rectangle"
    "GL_PROXY_TEXTURE_RECTANGLE_ARB" "GL_ARB_texture_rectangle"
    "GL_MAX_RECTANGLE_TEXTURE_SIZE_ARB" "GL_ARB_texture_rectangle"
    "GL_ARB_texture_rg" "GL_ARB_texture_rg"
    "GL_COMPRESSED_RED" "GL_ARB_texture_rg"
    "GL_COMPRESSED_RG" "GL_ARB_texture_rg"
    "GL_RG" "GL_ARB_texture_rg"
    "GL_RG_INTEGER" "GL_ARB_texture_rg"
    "GL_R8" "GL_ARB_texture_rg"
    "GL_R16" "GL_ARB_texture_rg"
    "GL_RG8" "GL_ARB_texture_rg"
    "GL_RG16" "GL_ARB_texture_rg"
    "GL_R16F" "GL_ARB_texture_rg"
    "GL_R32F" "GL_ARB_texture_rg"
    "GL_RG16F" "GL_ARB_texture_rg"
    "GL_RG32F" "GL_ARB_texture_rg"
    "GL_R8I" "GL_ARB_texture_rg"
    "GL_R8UI" "GL_ARB_texture_rg"
    "GL_R16I" "GL_ARB_texture_rg"
    "GL_R16UI" "GL_ARB_texture_rg"
    "GL_R32I" "GL_ARB_texture_rg"
    "GL_R32UI" "GL_ARB_texture_rg"
    "GL_RG8I" "GL_ARB_texture_rg"
    "GL_RG16I" "GL_ARB_texture_rg"
    "GL_RG32I" "GL_ARB_texture_rg"
    "GL_RG32UI" "GL_ARB_texture_rg"
    "GL_ARB_texture_rgb10_a2ui" "GL_ARB_texture_rgb10_a2ui"
    "GL_ARB_texture_stencil8" "GL_ARB_texture_stencil8"
    "GL_ARB_texture_storage" "GL_ARB_texture_storage"
    "GL_TEXTURE_IMMUTABLE_FORMAT" "GL_ARB_texture_storage"
    "GL_ARB_texture_storage_multisample" "GL_ARB_texture_storage_multisample"
    "GL_ARB_texture_swizzle" "GL_ARB_texture_swizzle"
    "GL_TEXTURE_SWIZZLE_R" "GL_ARB_texture_swizzle"
    "GL_TEXTURE_SWIZZLE_G" "GL_ARB_texture_swizzle"
    "GL_TEXTURE_SWIZZLE_B" "GL_ARB_texture_swizzle"
    "GL_TEXTURE_SWIZZLE_A" "GL_ARB_texture_swizzle"
    "GL_TEXTURE_SWIZZLE_RGBA" "GL_ARB_texture_swizzle"
    "GL_ARB_texture_view" "GL_ARB_texture_view"
    "GL_TEXTURE_VIEW_MIN_LEVEL" "GL_ARB_texture_view"
    "GL_TEXTURE_VIEW_NUM_LEVELS" "GL_ARB_texture_view"
    "GL_TEXTURE_VIEW_MIN_LAYER" "GL_ARB_texture_view"
    "GL_TEXTURE_VIEW_NUM_LAYERS" "GL_ARB_texture_view"
    "GL_ARB_timer_query" "GL_ARB_timer_query"
    "GL_TIME_ELAPSED" "GL_ARB_timer_query"
    "GL_TIMESTAMP" "GL_ARB_timer_query"
    "GL_ARB_transform_feedback2" "GL_ARB_transform_feedback2"
    "GL_TRANSFORM_FEEDBACK" "GL_ARB_transform_feedback2"
    "GL_TRANSFORM_FEEDBACK_BUFFER_PAUSED" "GL_ARB_transform_feedback2"
    "GL_TRANSFORM_FEEDBACK_BUFFER_ACTIVE" "GL_ARB_transform_feedback2"
    "GL_TRANSFORM_FEEDBACK_BINDING" "GL_ARB_transform_feedback2"
    "GL_ARB_transform_feedback3" "GL_ARB_transform_feedback3"
    "GL_MAX_TRANSFORM_FEEDBACK_BUFFERS" "GL_ARB_transform_feedback3"
    "GL_ARB_transform_feedback_instanced" "GL_ARB_transform_feedback_instanced"
    "GL_ARB_transform_feedback_overflow_query" "GL_ARB_transform_feedback_overflow_query"
    "GL_TRANSFORM_FEEDBACK_OVERFLOW_ARB" "GL_ARB_transform_feedback_overflow_query"
    "GL_TRANSFORM_FEEDBACK_STREAM_OVERFLOW_ARB" "GL_ARB_transform_feedback_overflow_query"
    "GL_ARB_transpose_matrix" "GL_ARB_transpose_matrix"
    "GL_TRANSPOSE_MODELVIEW_MATRIX_ARB" "GL_ARB_transpose_matrix"
    "GL_TRANSPOSE_PROJECTION_MATRIX_ARB" "GL_ARB_transpose_matrix"
    "GL_TRANSPOSE_TEXTURE_MATRIX_ARB" "GL_ARB_transpose_matrix"
    "GL_TRANSPOSE_COLOR_MATRIX_ARB" "GL_ARB_transpose_matrix"
    "GL_ARB_uniform_buffer_object" "GL_ARB_uniform_buffer_object"
    "GL_UNIFORM_BUFFER" "GL_ARB_uniform_buffer_object"
    "GL_UNIFORM_BUFFER_BINDING" "GL_ARB_uniform_buffer_object"
    "GL_UNIFORM_BUFFER_START" "GL_ARB_uniform_buffer_object"
    "GL_UNIFORM_BUFFER_SIZE" "GL_ARB_uniform_buffer_object"
    "GL_MAX_VERTEX_UNIFORM_BLOCKS" "GL_ARB_uniform_buffer_object"
    "GL_MAX_GEOMETRY_UNIFORM_BLOCKS" "GL_ARB_uniform_buffer_object"
    "GL_MAX_FRAGMENT_UNIFORM_BLOCKS" "GL_ARB_uniform_buffer_object"
    "GL_MAX_COMBINED_UNIFORM_BLOCKS" "GL_ARB_uniform_buffer_object"
    "GL_MAX_UNIFORM_BUFFER_BINDINGS" "GL_ARB_uniform_buffer_object"
    "GL_MAX_UNIFORM_BLOCK_SIZE" "GL_ARB_uniform_buffer_object"
    "GL_MAX_COMBINED_VERTEX_UNIFORM_COMPONENTS" "GL_ARB_uniform_buffer_object"
    "GL_MAX_COMBINED_GEOMETRY_UNIFORM_COMPONENTS" "GL_ARB_uniform_buffer_object"
    "GL_MAX_COMBINED_FRAGMENT_UNIFORM_COMPONENTS" "GL_ARB_uniform_buffer_object"
    "GL_UNIFORM_BUFFER_OFFSET_ALIGNMENT" "GL_ARB_uniform_buffer_object"
    "GL_ACTIVE_UNIFORM_BLOCK_MAX_NAME_LENGTH" "GL_ARB_uniform_buffer_object"
    "GL_ACTIVE_UNIFORM_BLOCKS" "GL_ARB_uniform_buffer_object"
    "GL_UNIFORM_TYPE" "GL_ARB_uniform_buffer_object"
    "GL_UNIFORM_SIZE" "GL_ARB_uniform_buffer_object"
    "GL_UNIFORM_NAME_LENGTH" "GL_ARB_uniform_buffer_object"
    "GL_UNIFORM_BLOCK_INDEX" "GL_ARB_uniform_buffer_object"
    "GL_UNIFORM_OFFSET" "GL_ARB_uniform_buffer_object"
    "GL_UNIFORM_ARRAY_STRIDE" "GL_ARB_uniform_buffer_object"
    "GL_UNIFORM_MATRIX_STRIDE" "GL_ARB_uniform_buffer_object"
    "GL_UNIFORM_IS_ROW_MAJOR" "GL_ARB_uniform_buffer_object"
    "GL_UNIFORM_BLOCK_BINDING" "GL_ARB_uniform_buffer_object"
    "GL_UNIFORM_BLOCK_DATA_SIZE" "GL_ARB_uniform_buffer_object"
    "GL_UNIFORM_BLOCK_NAME_LENGTH" "GL_ARB_uniform_buffer_object"
    "GL_UNIFORM_BLOCK_ACTIVE_UNIFORMS" "GL_ARB_uniform_buffer_object"
    "GL_UNIFORM_BLOCK_ACTIVE_UNIFORM_INDICES" "GL_ARB_uniform_buffer_object"
    "GL_UNIFORM_BLOCK_REFERENCED_BY_VERTEX_SHADER" "GL_ARB_uniform_buffer_object"
    "GL_UNIFORM_BLOCK_REFERENCED_BY_GEOMETRY_SHADER" "GL_ARB_uniform_buffer_object"
    "GL_UNIFORM_BLOCK_REFERENCED_BY_FRAGMENT_SHADER" "GL_ARB_uniform_buffer_object"
    "GL_INVALID_INDEX" "GL_ARB_uniform_buffer_object"
    "GL_ARB_vertex_array_bgra" "GL_ARB_vertex_array_bgra"
    "GL_ARB_vertex_array_object" "GL_ARB_vertex_array_object"
    "GL_VERTEX_ARRAY_BINDING" "GL_ARB_vertex_array_object"
    "GL_ARB_vertex_attrib_64bit" "GL_ARB_vertex_attrib_64bit"
    "GL_ARB_vertex_attrib_binding" "GL_ARB_vertex_attrib_binding"
    "GL_VERTEX_ATTRIB_BINDING" "GL_ARB_vertex_attrib_binding"
    "GL_VERTEX_ATTRIB_RELATIVE_OFFSET" "GL_ARB_vertex_attrib_binding"
    "GL_VERTEX_BINDING_DIVISOR" "GL_ARB_vertex_attrib_binding"
    "GL_VERTEX_BINDING_OFFSET" "GL_ARB_vertex_attrib_binding"
    "GL_VERTEX_BINDING_STRIDE" "GL_ARB_vertex_attrib_binding"
    "GL_MAX_VERTEX_ATTRIB_RELATIVE_OFFSET" "GL_ARB_vertex_attrib_binding"
    "GL_MAX_VERTEX_ATTRIB_BINDINGS" "GL_ARB_vertex_attrib_binding"
    "GL_VERTEX_BINDING_BUFFER" "GL_ARB_vertex_attrib_binding"
    "GL_ARB_vertex_blend" "GL_ARB_vertex_blend"
    "GL_MODELVIEW0_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW1_ARB" "GL_ARB_vertex_blend"
    "GL_MAX_VERTEX_UNITS_ARB" "GL_ARB_vertex_blend"
    "GL_ACTIVE_VERTEX_UNITS_ARB" "GL_ARB_vertex_blend"
    "GL_WEIGHT_SUM_UNITY_ARB" "GL_ARB_vertex_blend"
    "GL_VERTEX_BLEND_ARB" "GL_ARB_vertex_blend"
    "GL_CURRENT_WEIGHT_ARB" "GL_ARB_vertex_blend"
    "GL_WEIGHT_ARRAY_TYPE_ARB" "GL_ARB_vertex_blend"
    "GL_WEIGHT_ARRAY_STRIDE_ARB" "GL_ARB_vertex_blend"
    "GL_WEIGHT_ARRAY_SIZE_ARB" "GL_ARB_vertex_blend"
    "GL_WEIGHT_ARRAY_POINTER_ARB" "GL_ARB_vertex_blend"
    "GL_WEIGHT_ARRAY_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW2_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW3_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW4_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW5_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW6_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW7_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW8_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW9_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW10_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW11_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW12_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW13_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW14_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW15_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW16_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW17_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW18_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW19_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW20_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW21_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW22_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW23_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW24_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW25_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW26_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW27_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW28_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW29_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW30_ARB" "GL_ARB_vertex_blend"
    "GL_MODELVIEW31_ARB" "GL_ARB_vertex_blend"
    "GL_ARB_vertex_buffer_object" "GL_ARB_vertex_buffer_object"
    "GL_BUFFER_SIZE_ARB" "GL_ARB_vertex_buffer_object"
    "GL_BUFFER_USAGE_ARB" "GL_ARB_vertex_buffer_object"
    "GL_ARRAY_BUFFER_ARB" "GL_ARB_vertex_buffer_object"
    "GL_ELEMENT_ARRAY_BUFFER_ARB" "GL_ARB_vertex_buffer_object"
    "GL_ARRAY_BUFFER_BINDING_ARB" "GL_ARB_vertex_buffer_object"
    "GL_ELEMENT_ARRAY_BUFFER_BINDING_ARB" "GL_ARB_vertex_buffer_object"
    "GL_VERTEX_ARRAY_BUFFER_BINDING_ARB" "GL_ARB_vertex_buffer_object"
    "GL_NORMAL_ARRAY_BUFFER_BINDING_ARB" "GL_ARB_vertex_buffer_object"
    "GL_COLOR_ARRAY_BUFFER_BINDING_ARB" "GL_ARB_vertex_buffer_object"
    "GL_INDEX_ARRAY_BUFFER_BINDING_ARB" "GL_ARB_vertex_buffer_object"
    "GL_TEXTURE_COORD_ARRAY_BUFFER_BINDING_ARB" "GL_ARB_vertex_buffer_object"
    "GL_EDGE_FLAG_ARRAY_BUFFER_BINDING_ARB" "GL_ARB_vertex_buffer_object"
    "GL_SECONDARY_COLOR_ARRAY_BUFFER_BINDING_ARB" "GL_ARB_vertex_buffer_object"
    "GL_FOG_COORDINATE_ARRAY_BUFFER_BINDING_ARB" "GL_ARB_vertex_buffer_object"
    "GL_WEIGHT_ARRAY_BUFFER_BINDING_ARB" "GL_ARB_vertex_buffer_object"
    "GL_VERTEX_ATTRIB_ARRAY_BUFFER_BINDING_ARB" "GL_ARB_vertex_buffer_object"
    "GL_READ_ONLY_ARB" "GL_ARB_vertex_buffer_object"
    "GL_WRITE_ONLY_ARB" "GL_ARB_vertex_buffer_object"
    "GL_READ_WRITE_ARB" "GL_ARB_vertex_buffer_object"
    "GL_BUFFER_ACCESS_ARB" "GL_ARB_vertex_buffer_object"
    "GL_BUFFER_MAPPED_ARB" "GL_ARB_vertex_buffer_object"
    "GL_BUFFER_MAP_POINTER_ARB" "GL_ARB_vertex_buffer_object"
    "GL_STREAM_DRAW_ARB" "GL_ARB_vertex_buffer_object"
    "GL_STREAM_READ_ARB" "GL_ARB_vertex_buffer_object"
    "GL_STREAM_COPY_ARB" "GL_ARB_vertex_buffer_object"
    "GL_STATIC_DRAW_ARB" "GL_ARB_vertex_buffer_object"
    "GL_STATIC_READ_ARB" "GL_ARB_vertex_buffer_object"
    "GL_STATIC_COPY_ARB" "GL_ARB_vertex_buffer_object"
    "GL_DYNAMIC_DRAW_ARB" "GL_ARB_vertex_buffer_object"
    "GL_DYNAMIC_READ_ARB" "GL_ARB_vertex_buffer_object"
    "GL_DYNAMIC_COPY_ARB" "GL_ARB_vertex_buffer_object"
    "GL_ARB_vertex_program" "GL_ARB_vertex_program"
    "GL_COLOR_SUM_ARB" "GL_ARB_vertex_program"
    "GL_VERTEX_PROGRAM_ARB" "GL_ARB_vertex_program"
    "GL_VERTEX_ATTRIB_ARRAY_ENABLED_ARB" "GL_ARB_vertex_program"
    "GL_VERTEX_ATTRIB_ARRAY_SIZE_ARB" "GL_ARB_vertex_program"
    "GL_VERTEX_ATTRIB_ARRAY_STRIDE_ARB" "GL_ARB_vertex_program"
    "GL_VERTEX_ATTRIB_ARRAY_TYPE_ARB" "GL_ARB_vertex_program"
    "GL_CURRENT_VERTEX_ATTRIB_ARB" "GL_ARB_vertex_program"
    "GL_PROGRAM_LENGTH_ARB" "GL_ARB_vertex_program"
    "GL_PROGRAM_STRING_ARB" "GL_ARB_vertex_program"
    "GL_MAX_PROGRAM_MATRIX_STACK_DEPTH_ARB" "GL_ARB_vertex_program"
    "GL_MAX_PROGRAM_MATRICES_ARB" "GL_ARB_vertex_program"
    "GL_CURRENT_MATRIX_STACK_DEPTH_ARB" "GL_ARB_vertex_program"
    "GL_CURRENT_MATRIX_ARB" "GL_ARB_vertex_program"
    "GL_VERTEX_PROGRAM_POINT_SIZE_ARB" "GL_ARB_vertex_program"
    "GL_VERTEX_PROGRAM_TWO_SIDE_ARB" "GL_ARB_vertex_program"
    "GL_VERTEX_ATTRIB_ARRAY_POINTER_ARB" "GL_ARB_vertex_program"
    "GL_PROGRAM_ERROR_POSITION_ARB" "GL_ARB_vertex_program"
    "GL_PROGRAM_BINDING_ARB" "GL_ARB_vertex_program"
    "GL_MAX_VERTEX_ATTRIBS_ARB" "GL_ARB_vertex_program"
    "GL_VERTEX_ATTRIB_ARRAY_NORMALIZED_ARB" "GL_ARB_vertex_program"
    "GL_PROGRAM_ERROR_STRING_ARB" "GL_ARB_vertex_program"
    "GL_PROGRAM_FORMAT_ASCII_ARB" "GL_ARB_vertex_program"
    "GL_PROGRAM_FORMAT_ARB" "GL_ARB_vertex_program"
    "GL_PROGRAM_INSTRUCTIONS_ARB" "GL_ARB_vertex_program"
    "GL_MAX_PROGRAM_INSTRUCTIONS_ARB" "GL_ARB_vertex_program"
    "GL_PROGRAM_NATIVE_INSTRUCTIONS_ARB" "GL_ARB_vertex_program"
    "GL_MAX_PROGRAM_NATIVE_INSTRUCTIONS_ARB" "GL_ARB_vertex_program"
    "GL_PROGRAM_TEMPORARIES_ARB" "GL_ARB_vertex_program"
    "GL_MAX_PROGRAM_TEMPORARIES_ARB" "GL_ARB_vertex_program"
    "GL_PROGRAM_NATIVE_TEMPORARIES_ARB" "GL_ARB_vertex_program"
    "GL_MAX_PROGRAM_NATIVE_TEMPORARIES_ARB" "GL_ARB_vertex_program"
    "GL_PROGRAM_PARAMETERS_ARB" "GL_ARB_vertex_program"
    "GL_MAX_PROGRAM_PARAMETERS_ARB" "GL_ARB_vertex_program"
    "GL_PROGRAM_NATIVE_PARAMETERS_ARB" "GL_ARB_vertex_program"
    "GL_MAX_PROGRAM_NATIVE_PARAMETERS_ARB" "GL_ARB_vertex_program"
    "GL_PROGRAM_ATTRIBS_ARB" "GL_ARB_vertex_program"
    "GL_MAX_PROGRAM_ATTRIBS_ARB" "GL_ARB_vertex_program"
    "GL_PROGRAM_NATIVE_ATTRIBS_ARB" "GL_ARB_vertex_program"
    "GL_MAX_PROGRAM_NATIVE_ATTRIBS_ARB" "GL_ARB_vertex_program"
    "GL_PROGRAM_ADDRESS_REGISTERS_ARB" "GL_ARB_vertex_program"
    "GL_MAX_PROGRAM_ADDRESS_REGISTERS_ARB" "GL_ARB_vertex_program"
    "GL_PROGRAM_NATIVE_ADDRESS_REGISTERS_ARB" "GL_ARB_vertex_program"
    "GL_MAX_PROGRAM_NATIVE_ADDRESS_REGISTERS_ARB" "GL_ARB_vertex_program"
    "GL_MAX_PROGRAM_LOCAL_PARAMETERS_ARB" "GL_ARB_vertex_program"
    "GL_MAX_PROGRAM_ENV_PARAMETERS_ARB" "GL_ARB_vertex_program"
    "GL_PROGRAM_UNDER_NATIVE_LIMITS_ARB" "GL_ARB_vertex_program"
    "GL_TRANSPOSE_CURRENT_MATRIX_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX0_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX1_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX2_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX3_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX4_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX5_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX6_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX7_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX8_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX9_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX10_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX11_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX12_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX13_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX14_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX15_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX16_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX17_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX18_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX19_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX20_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX21_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX22_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX23_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX24_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX25_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX26_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX27_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX28_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX29_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX30_ARB" "GL_ARB_vertex_program"
    "GL_MATRIX31_ARB" "GL_ARB_vertex_program"
    "GL_ARB_vertex_shader" "GL_ARB_vertex_shader"
    "GL_VERTEX_SHADER_ARB" "GL_ARB_vertex_shader"
    "GL_MAX_VERTEX_UNIFORM_COMPONENTS_ARB" "GL_ARB_vertex_shader"
    "GL_MAX_VARYING_FLOATS_ARB" "GL_ARB_vertex_shader"
    "GL_MAX_VERTEX_TEXTURE_IMAGE_UNITS_ARB" "GL_ARB_vertex_shader"
    "GL_MAX_COMBINED_TEXTURE_IMAGE_UNITS_ARB" "GL_ARB_vertex_shader"
    "GL_OBJECT_ACTIVE_ATTRIBUTES_ARB" "GL_ARB_vertex_shader"
    "GL_OBJECT_ACTIVE_ATTRIBUTE_MAX_LENGTH_ARB" "GL_ARB_vertex_shader"
    "GL_ARB_vertex_type_10f_11f_11f_rev" "GL_ARB_vertex_type_10f_11f_11f_rev"
    "GL_ARB_vertex_type_2_10_10_10_rev" "GL_ARB_vertex_type_2_10_10_10_rev"
    "GL_UNSIGNED_INT_2_10_10_10_REV" "GL_ARB_vertex_type_2_10_10_10_rev"
    "GL_INT_2_10_10_10_REV" "GL_ARB_vertex_type_2_10_10_10_rev"
    "GL_ARB_viewport_array" "GL_ARB_viewport_array"
    "GL_MAX_VIEWPORTS" "GL_ARB_viewport_array"
    "GL_VIEWPORT_SUBPIXEL_BITS" "GL_ARB_viewport_array"
    "GL_VIEWPORT_BOUNDS_RANGE" "GL_ARB_viewport_array"
    "GL_LAYER_PROVOKING_VERTEX" "GL_ARB_viewport_array"
    "GL_VIEWPORT_INDEX_PROVOKING_VERTEX" "GL_ARB_viewport_array"
    "GL_UNDEFINED_VERTEX" "GL_ARB_viewport_array"
    "GL_ARB_window_pos" "GL_ARB_window_pos"
    "GL_ARM_mali_program_binary" "GL_ARM_mali_program_binary"
    "GL_MALI_PROGRAM_BINARY_ARM" "GL_ARM_mali_program_binary"
    "GL_ARM_mali_shader_binary" "GL_ARM_mali_shader_binary"
    "GL_MALI_SHADER_BINARY_ARM" "GL_ARM_mali_shader_binary"
    "GL_ARM_rgba8" "GL_ARM_rgba8"
    "GL_RGBA8_OES" "GL_ARM_rgba8"
    "GL_ARM_shader_framebuffer_fetch" "GL_ARM_shader_framebuffer_fetch"
    "GL_FETCH_PER_SAMPLE_ARM" "GL_ARM_shader_framebuffer_fetch"
    "GL_FRAGMENT_SHADER_FRAMEBUFFER_FETCH_MRT_ARM" "GL_ARM_shader_framebuffer_fetch"
    "GL_ARM_shader_framebuffer_fetch_depth_stencil" "GL_ARM_shader_framebuffer_fetch_depth_stencil"
    "GL_ARM_texture_unnormalized_coordinates" "GL_ARM_texture_unnormalized_coordinates"
    "GL_TEXTURE_UNNORMALIZED_COORDINATES_ARM" "GL_ARM_texture_unnormalized_coordinates"
    "GL_ATIX_point_sprites" "GL_ATIX_point_sprites"
    "GL_TEXTURE_POINT_MODE_ATIX" "GL_ATIX_point_sprites"
    "GL_TEXTURE_POINT_ONE_COORD_ATIX" "GL_ATIX_point_sprites"
    "GL_TEXTURE_POINT_SPRITE_ATIX" "GL_ATIX_point_sprites"
    "GL_POINT_SPRITE_CULL_MODE_ATIX" "GL_ATIX_point_sprites"
    "GL_POINT_SPRITE_CULL_CENTER_ATIX" "GL_ATIX_point_sprites"
    "GL_POINT_SPRITE_CULL_CLIP_ATIX" "GL_ATIX_point_sprites"
    "GL_ATIX_texture_env_combine3" "GL_ATIX_texture_env_combine3"
    "GL_MODULATE_ADD_ATIX" "GL_ATIX_texture_env_combine3"
    "GL_MODULATE_SIGNED_ADD_ATIX" "GL_ATIX_texture_env_combine3"
    "GL_MODULATE_SUBTRACT_ATIX" "GL_ATIX_texture_env_combine3"
    "GL_ATIX_texture_env_route" "GL_ATIX_texture_env_route"
    "GL_SECONDARY_COLOR_ATIX" "GL_ATIX_texture_env_route"
    "GL_TEXTURE_OUTPUT_RGB_ATIX" "GL_ATIX_texture_env_route"
    "GL_TEXTURE_OUTPUT_ALPHA_ATIX" "GL_ATIX_texture_env_route"
    "GL_ATIX_vertex_shader_output_point_size" "GL_ATIX_vertex_shader_output_point_size"
    "GL_OUTPUT_POINT_SIZE_ATIX" "GL_ATIX_vertex_shader_output_point_size"
    "GL_ATI_draw_buffers" "GL_ATI_draw_buffers"
    "GL_MAX_DRAW_BUFFERS_ATI" "GL_ATI_draw_buffers"
    "GL_DRAW_BUFFER0_ATI" "GL_ATI_draw_buffers"
    "GL_DRAW_BUFFER1_ATI" "GL_ATI_draw_buffers"
    "GL_DRAW_BUFFER2_ATI" "GL_ATI_draw_buffers"
    "GL_DRAW_BUFFER3_ATI" "GL_ATI_draw_buffers"
    "GL_DRAW_BUFFER4_ATI" "GL_ATI_draw_buffers"
    "GL_DRAW_BUFFER5_ATI" "GL_ATI_draw_buffers"
    "GL_DRAW_BUFFER6_ATI" "GL_ATI_draw_buffers"
    "GL_DRAW_BUFFER7_ATI" "GL_ATI_draw_buffers"
    "GL_DRAW_BUFFER8_ATI" "GL_ATI_draw_buffers"
    "GL_DRAW_BUFFER9_ATI" "GL_ATI_draw_buffers"
    "GL_DRAW_BUFFER10_ATI" "GL_ATI_draw_buffers"
    "GL_DRAW_BUFFER11_ATI" "GL_ATI_draw_buffers"
    "GL_DRAW_BUFFER12_ATI" "GL_ATI_draw_buffers"
    "GL_DRAW_BUFFER13_ATI" "GL_ATI_draw_buffers"
    "GL_DRAW_BUFFER14_ATI" "GL_ATI_draw_buffers"
    "GL_DRAW_BUFFER15_ATI" "GL_ATI_draw_buffers"
    "GL_ATI_element_array" "GL_ATI_element_array"
    "GL_ELEMENT_ARRAY_ATI" "GL_ATI_element_array"
    "GL_ELEMENT_ARRAY_TYPE_ATI" "GL_ATI_element_array"
    "GL_ELEMENT_ARRAY_POINTER_ATI" "GL_ATI_element_array"
    "GL_ATI_envmap_bumpmap" "GL_ATI_envmap_bumpmap"
    "GL_BUMP_ROT_MATRIX_ATI" "GL_ATI_envmap_bumpmap"
    "GL_BUMP_ROT_MATRIX_SIZE_ATI" "GL_ATI_envmap_bumpmap"
    "GL_BUMP_NUM_TEX_UNITS_ATI" "GL_ATI_envmap_bumpmap"
    "GL_BUMP_TEX_UNITS_ATI" "GL_ATI_envmap_bumpmap"
    "GL_DUDV_ATI" "GL_ATI_envmap_bumpmap"
    "GL_DU8DV8_ATI" "GL_ATI_envmap_bumpmap"
    "GL_BUMP_ENVMAP_ATI" "GL_ATI_envmap_bumpmap"
    "GL_BUMP_TARGET_ATI" "GL_ATI_envmap_bumpmap"
    "GL_ATI_fragment_shader" "GL_ATI_fragment_shader"
    "GL_2X_BIT_ATI" "GL_ATI_fragment_shader"
    "GL_RED_BIT_ATI" "GL_ATI_fragment_shader"
    "GL_4X_BIT_ATI" "GL_ATI_fragment_shader"
    "GL_COMP_BIT_ATI" "GL_ATI_fragment_shader"
    "GL_GREEN_BIT_ATI" "GL_ATI_fragment_shader"
    "GL_8X_BIT_ATI" "GL_ATI_fragment_shader"
    "GL_BLUE_BIT_ATI" "GL_ATI_fragment_shader"
    "GL_NEGATE_BIT_ATI" "GL_ATI_fragment_shader"
    "GL_BIAS_BIT_ATI" "GL_ATI_fragment_shader"
    "GL_HALF_BIT_ATI" "GL_ATI_fragment_shader"
    "GL_QUARTER_BIT_ATI" "GL_ATI_fragment_shader"
    "GL_EIGHTH_BIT_ATI" "GL_ATI_fragment_shader"
    "GL_SATURATE_BIT_ATI" "GL_ATI_fragment_shader"
    "GL_FRAGMENT_SHADER_ATI" "GL_ATI_fragment_shader"
    "GL_REG_0_ATI" "GL_ATI_fragment_shader"
    "GL_REG_1_ATI" "GL_ATI_fragment_shader"
    "GL_REG_2_ATI" "GL_ATI_fragment_shader"
    "GL_REG_3_ATI" "GL_ATI_fragment_shader"
    "GL_REG_4_ATI" "GL_ATI_fragment_shader"
    "GL_REG_5_ATI" "GL_ATI_fragment_shader"
    "GL_CON_0_ATI" "GL_ATI_fragment_shader"
    "GL_CON_1_ATI" "GL_ATI_fragment_shader"
    "GL_CON_2_ATI" "GL_ATI_fragment_shader"
    "GL_CON_3_ATI" "GL_ATI_fragment_shader"
    "GL_CON_4_ATI" "GL_ATI_fragment_shader"
    "GL_CON_5_ATI" "GL_ATI_fragment_shader"
    "GL_CON_6_ATI" "GL_ATI_fragment_shader"
    "GL_CON_7_ATI" "GL_ATI_fragment_shader"
    "GL_MOV_ATI" "GL_ATI_fragment_shader"
    "GL_ADD_ATI" "GL_ATI_fragment_shader"
    "GL_MUL_ATI" "GL_ATI_fragment_shader"
    "GL_SUB_ATI" "GL_ATI_fragment_shader"
    "GL_DOT3_ATI" "GL_ATI_fragment_shader"
    "GL_DOT4_ATI" "GL_ATI_fragment_shader"
    "GL_MAD_ATI" "GL_ATI_fragment_shader"
    "GL_LERP_ATI" "GL_ATI_fragment_shader"
    "GL_CND_ATI" "GL_ATI_fragment_shader"
    "GL_CND0_ATI" "GL_ATI_fragment_shader"
    "GL_DOT2_ADD_ATI" "GL_ATI_fragment_shader"
    "GL_SECONDARY_INTERPOLATOR_ATI" "GL_ATI_fragment_shader"
    "GL_NUM_FRAGMENT_REGISTERS_ATI" "GL_ATI_fragment_shader"
    "GL_NUM_FRAGMENT_CONSTANTS_ATI" "GL_ATI_fragment_shader"
    "GL_NUM_PASSES_ATI" "GL_ATI_fragment_shader"
    "GL_NUM_INSTRUCTIONS_PER_PASS_ATI" "GL_ATI_fragment_shader"
    "GL_NUM_INSTRUCTIONS_TOTAL_ATI" "GL_ATI_fragment_shader"
    "GL_NUM_INPUT_INTERPOLATOR_COMPONENTS_ATI" "GL_ATI_fragment_shader"
    "GL_NUM_LOOPBACK_COMPONENTS_ATI" "GL_ATI_fragment_shader"
    "GL_COLOR_ALPHA_PAIRING_ATI" "GL_ATI_fragment_shader"
    "GL_SWIZZLE_STR_ATI" "GL_ATI_fragment_shader"
    "GL_SWIZZLE_STQ_ATI" "GL_ATI_fragment_shader"
    "GL_SWIZZLE_STR_DR_ATI" "GL_ATI_fragment_shader"
    "GL_SWIZZLE_STQ_DQ_ATI" "GL_ATI_fragment_shader"
    "GL_SWIZZLE_STRQ_ATI" "GL_ATI_fragment_shader"
    "GL_SWIZZLE_STRQ_DQ_ATI" "GL_ATI_fragment_shader"
    "GL_ATI_map_object_buffer" "GL_ATI_map_object_buffer"
    "GL_ATI_meminfo" "GL_ATI_meminfo"
    "GL_VBO_FREE_MEMORY_ATI" "GL_ATI_meminfo"
    "GL_TEXTURE_FREE_MEMORY_ATI" "GL_ATI_meminfo"
    "GL_RENDERBUFFER_FREE_MEMORY_ATI" "GL_ATI_meminfo"
    "GL_ATI_pn_triangles" "GL_ATI_pn_triangles"
    "GL_PN_TRIANGLES_ATI" "GL_ATI_pn_triangles"
    "GL_MAX_PN_TRIANGLES_TESSELATION_LEVEL_ATI" "GL_ATI_pn_triangles"
    "GL_PN_TRIANGLES_POINT_MODE_ATI" "GL_ATI_pn_triangles"
    "GL_PN_TRIANGLES_NORMAL_MODE_ATI" "GL_ATI_pn_triangles"
    "GL_PN_TRIANGLES_TESSELATION_LEVEL_ATI" "GL_ATI_pn_triangles"
    "GL_PN_TRIANGLES_POINT_MODE_LINEAR_ATI" "GL_ATI_pn_triangles"
    "GL_PN_TRIANGLES_POINT_MODE_CUBIC_ATI" "GL_ATI_pn_triangles"
    "GL_PN_TRIANGLES_NORMAL_MODE_LINEAR_ATI" "GL_ATI_pn_triangles"
    "GL_PN_TRIANGLES_NORMAL_MODE_QUADRATIC_ATI" "GL_ATI_pn_triangles"
    "GL_ATI_separate_stencil" "GL_ATI_separate_stencil"
    "GL_STENCIL_BACK_FUNC_ATI" "GL_ATI_separate_stencil"
    "GL_STENCIL_BACK_FAIL_ATI" "GL_ATI_separate_stencil"
    "GL_STENCIL_BACK_PASS_DEPTH_FAIL_ATI" "GL_ATI_separate_stencil"
    "GL_STENCIL_BACK_PASS_DEPTH_PASS_ATI" "GL_ATI_separate_stencil"
    "GL_ATI_shader_texture_lod" "GL_ATI_shader_texture_lod"
    "GL_ATI_text_fragment_shader" "GL_ATI_text_fragment_shader"
    "GL_TEXT_FRAGMENT_SHADER_ATI" "GL_ATI_text_fragment_shader"
    "GL_ATI_texture_compression_3dc" "GL_ATI_texture_compression_3dc"
    "GL_COMPRESSED_LUMINANCE_ALPHA_3DC_ATI" "GL_ATI_texture_compression_3dc"
    "GL_ATI_texture_env_combine3" "GL_ATI_texture_env_combine3"
    "GL_MODULATE_ADD_ATI" "GL_ATI_texture_env_combine3"
    "GL_MODULATE_SIGNED_ADD_ATI" "GL_ATI_texture_env_combine3"
    "GL_MODULATE_SUBTRACT_ATI" "GL_ATI_texture_env_combine3"
    "GL_ATI_texture_float" "GL_ATI_texture_float"
    "GL_RGBA_FLOAT32_ATI" "GL_ATI_texture_float"
    "GL_RGB_FLOAT32_ATI" "GL_ATI_texture_float"
    "GL_ALPHA_FLOAT32_ATI" "GL_ATI_texture_float"
    "GL_INTENSITY_FLOAT32_ATI" "GL_ATI_texture_float"
    "GL_LUMINANCE_FLOAT32_ATI" "GL_ATI_texture_float"
    "GL_LUMINANCE_ALPHA_FLOAT32_ATI" "GL_ATI_texture_float"
    "GL_RGBA_FLOAT16_ATI" "GL_ATI_texture_float"
    "GL_RGB_FLOAT16_ATI" "GL_ATI_texture_float"
    "GL_ALPHA_FLOAT16_ATI" "GL_ATI_texture_float"
    "GL_INTENSITY_FLOAT16_ATI" "GL_ATI_texture_float"
    "GL_LUMINANCE_FLOAT16_ATI" "GL_ATI_texture_float"
    "GL_LUMINANCE_ALPHA_FLOAT16_ATI" "GL_ATI_texture_float"
    "GL_ATI_texture_mirror_once" "GL_ATI_texture_mirror_once"
    "GL_MIRROR_CLAMP_ATI" "GL_ATI_texture_mirror_once"
    "GL_MIRROR_CLAMP_TO_EDGE_ATI" "GL_ATI_texture_mirror_once"
    "GL_ATI_vertex_array_object" "GL_ATI_vertex_array_object"
    "GL_STATIC_ATI" "GL_ATI_vertex_array_object"
    "GL_DYNAMIC_ATI" "GL_ATI_vertex_array_object"
    "GL_PRESERVE_ATI" "GL_ATI_vertex_array_object"
    "GL_DISCARD_ATI" "GL_ATI_vertex_array_object"
    "GL_OBJECT_BUFFER_SIZE_ATI" "GL_ATI_vertex_array_object"
    "GL_OBJECT_BUFFER_USAGE_ATI" "GL_ATI_vertex_array_object"
    "GL_ARRAY_OBJECT_BUFFER_ATI" "GL_ATI_vertex_array_object"
    "GL_ARRAY_OBJECT_OFFSET_ATI" "GL_ATI_vertex_array_object"
    "GL_ATI_vertex_attrib_array_object" "GL_ATI_vertex_attrib_array_object"
    "GL_ATI_vertex_streams" "GL_ATI_vertex_streams"
    "GL_MAX_VERTEX_STREAMS_ATI" "GL_ATI_vertex_streams"
    "GL_VERTEX_SOURCE_ATI" "GL_ATI_vertex_streams"
    "GL_VERTEX_STREAM0_ATI" "GL_ATI_vertex_streams"
    "GL_VERTEX_STREAM1_ATI" "GL_ATI_vertex_streams"
    "GL_VERTEX_STREAM2_ATI" "GL_ATI_vertex_streams"
    "GL_VERTEX_STREAM3_ATI" "GL_ATI_vertex_streams"
    "GL_VERTEX_STREAM4_ATI" "GL_ATI_vertex_streams"
    "GL_VERTEX_STREAM5_ATI" "GL_ATI_vertex_streams"
    "GL_VERTEX_STREAM6_ATI" "GL_ATI_vertex_streams"
    "GL_VERTEX_STREAM7_ATI" "GL_ATI_vertex_streams"
    "GL_DMP_program_binary" "GL_DMP_program_binary"
    "GL_SMAPHS30_PROGRAM_BINARY_DMP" "GL_DMP_program_binary"
    "GL_SMAPHS_PROGRAM_BINARY_DMP" "GL_DMP_program_binary"
    "GL_DMP_PROGRAM_BINARY_DMP" "GL_DMP_program_binary"
    "GL_DMP_shader_binary" "GL_DMP_shader_binary"
    "GL_SHADER_BINARY_DMP" "GL_DMP_shader_binary"
    "GL_EXT_422_pixels" "GL_EXT_422_pixels"
    "GL_422_EXT" "GL_EXT_422_pixels"
    "GL_422_REV_EXT" "GL_EXT_422_pixels"
    "GL_422_AVERAGE_EXT" "GL_EXT_422_pixels"
    "GL_422_REV_AVERAGE_EXT" "GL_EXT_422_pixels"
    "GL_EXT_Cg_shader" "GL_EXT_Cg_shader"
    "GL_CG_VERTEX_SHADER_EXT" "GL_EXT_Cg_shader"
    "GL_CG_FRAGMENT_SHADER_EXT" "GL_EXT_Cg_shader"
    "GL_EXT_EGL_image_array" "GL_EXT_EGL_image_array"
    "GL_EXT_EGL_image_external_wrap_modes" "GL_EXT_EGL_image_external_wrap_modes"
    "GL_EXT_EGL_image_storage" "GL_EXT_EGL_image_storage"
    "GL_EXT_EGL_sync" "GL_EXT_EGL_sync"
    "GL_EXT_YUV_target" "GL_EXT_YUV_target"
    "GL_SAMPLER_EXTERNAL_2D_Y2Y_EXT" "GL_EXT_YUV_target"
    "GL_EXT_abgr" "GL_EXT_abgr"
    "GL_ABGR_EXT" "GL_EXT_abgr"
    "GL_EXT_base_instance" "GL_EXT_base_instance"
    "GL_EXT_bgra" "GL_EXT_bgra"
    "GL_BGR_EXT" "GL_EXT_bgra"
    "GL_EXT_bindable_uniform" "GL_EXT_bindable_uniform"
    "GL_MAX_VERTEX_BINDABLE_UNIFORMS_EXT" "GL_EXT_bindable_uniform"
    "GL_MAX_FRAGMENT_BINDABLE_UNIFORMS_EXT" "GL_EXT_bindable_uniform"
    "GL_MAX_GEOMETRY_BINDABLE_UNIFORMS_EXT" "GL_EXT_bindable_uniform"
    "GL_MAX_BINDABLE_UNIFORM_SIZE_EXT" "GL_EXT_bindable_uniform"
    "GL_UNIFORM_BUFFER_EXT" "GL_EXT_bindable_uniform"
    "GL_UNIFORM_BUFFER_BINDING_EXT" "GL_EXT_bindable_uniform"
    "GL_EXT_blend_color" "GL_EXT_blend_color"
    "GL_CONSTANT_COLOR_EXT" "GL_EXT_blend_color"
    "GL_ONE_MINUS_CONSTANT_COLOR_EXT" "GL_EXT_blend_color"
    "GL_CONSTANT_ALPHA_EXT" "GL_EXT_blend_color"
    "GL_ONE_MINUS_CONSTANT_ALPHA_EXT" "GL_EXT_blend_color"
    "GL_BLEND_COLOR_EXT" "GL_EXT_blend_color"
    "GL_EXT_blend_equation_separate" "GL_EXT_blend_equation_separate"
    "GL_BLEND_EQUATION_RGB_EXT" "GL_EXT_blend_equation_separate"
    "GL_BLEND_EQUATION_ALPHA_EXT" "GL_EXT_blend_equation_separate"
    "GL_EXT_blend_func_extended" "GL_EXT_blend_func_extended"
    "GL_SRC_ALPHA_SATURATE_EXT" "GL_EXT_blend_func_extended"
    "GL_SRC1_ALPHA_EXT" "GL_EXT_blend_func_extended"
    "GL_SRC1_COLOR_EXT" "GL_EXT_blend_func_extended"
    "GL_ONE_MINUS_SRC1_COLOR_EXT" "GL_EXT_blend_func_extended"
    "GL_ONE_MINUS_SRC1_ALPHA_EXT" "GL_EXT_blend_func_extended"
    "GL_MAX_DUAL_SOURCE_DRAW_BUFFERS_EXT" "GL_EXT_blend_func_extended"
    "GL_LOCATION_INDEX_EXT" "GL_EXT_blend_func_extended"
    "GL_EXT_blend_func_separate" "GL_EXT_blend_func_separate"
    "GL_BLEND_DST_RGB_EXT" "GL_EXT_blend_func_separate"
    "GL_BLEND_SRC_RGB_EXT" "GL_EXT_blend_func_separate"
    "GL_BLEND_DST_ALPHA_EXT" "GL_EXT_blend_func_separate"
    "GL_BLEND_SRC_ALPHA_EXT" "GL_EXT_blend_func_separate"
    "GL_EXT_blend_logic_op" "GL_EXT_blend_logic_op"
    "GL_EXT_blend_minmax" "GL_EXT_blend_minmax"
    "GL_FUNC_ADD_EXT" "GL_EXT_blend_minmax"
    "GL_MIN_EXT" "GL_EXT_blend_minmax"
    "GL_MAX_EXT" "GL_EXT_blend_minmax"
    "GL_BLEND_EQUATION_EXT" "GL_EXT_blend_minmax"
    "GL_EXT_blend_subtract" "GL_EXT_blend_subtract"
    "GL_FUNC_SUBTRACT_EXT" "GL_EXT_blend_subtract"
    "GL_FUNC_REVERSE_SUBTRACT_EXT" "GL_EXT_blend_subtract"
    "GL_EXT_buffer_storage" "GL_EXT_buffer_storage"
    "GL_MAP_PERSISTENT_BIT_EXT" "GL_EXT_buffer_storage"
    "GL_MAP_COHERENT_BIT_EXT" "GL_EXT_buffer_storage"
    "GL_DYNAMIC_STORAGE_BIT_EXT" "GL_EXT_buffer_storage"
    "GL_CLIENT_STORAGE_BIT_EXT" "GL_EXT_buffer_storage"
    "GL_CLIENT_MAPPED_BUFFER_BARRIER_BIT_EXT" "GL_EXT_buffer_storage"
    "GL_BUFFER_IMMUTABLE_STORAGE_EXT" "GL_EXT_buffer_storage"
    "GL_BUFFER_STORAGE_FLAGS_EXT" "GL_EXT_buffer_storage"
    "GL_EXT_clear_texture" "GL_EXT_clear_texture"
    "GL_EXT_clip_control" "GL_EXT_clip_control"
    "GL_LOWER_LEFT_EXT" "GL_EXT_clip_control"
    "GL_UPPER_LEFT_EXT" "GL_EXT_clip_control"
    "GL_CLIP_ORIGIN_EXT" "GL_EXT_clip_control"
    "GL_CLIP_DEPTH_MODE_EXT" "GL_EXT_clip_control"
    "GL_NEGATIVE_ONE_TO_ONE_EXT" "GL_EXT_clip_control"
    "GL_ZERO_TO_ONE_EXT" "GL_EXT_clip_control"
    "GL_EXT_clip_cull_distance" "GL_EXT_clip_cull_distance"
    "GL_MAX_CLIP_DISTANCES_EXT" "GL_EXT_clip_cull_distance"
    "GL_CLIP_DISTANCE0_EXT" "GL_EXT_clip_cull_distance"
    "GL_CLIP_DISTANCE1_EXT" "GL_EXT_clip_cull_distance"
    "GL_CLIP_DISTANCE2_EXT" "GL_EXT_clip_cull_distance"
    "GL_CLIP_DISTANCE3_EXT" "GL_EXT_clip_cull_distance"
    "GL_CLIP_DISTANCE4_EXT" "GL_EXT_clip_cull_distance"
    "GL_CLIP_DISTANCE5_EXT" "GL_EXT_clip_cull_distance"
    "GL_CLIP_DISTANCE6_EXT" "GL_EXT_clip_cull_distance"
    "GL_CLIP_DISTANCE7_EXT" "GL_EXT_clip_cull_distance"
    "GL_MAX_CULL_DISTANCES_EXT" "GL_EXT_clip_cull_distance"
    "GL_MAX_COMBINED_CLIP_AND_CULL_DISTANCES_EXT" "GL_EXT_clip_cull_distance"
    "GL_EXT_clip_volume_hint" "GL_EXT_clip_volume_hint"
    "GL_CLIP_VOLUME_CLIPPING_HINT_EXT" "GL_EXT_clip_volume_hint"
    "GL_EXT_cmyka" "GL_EXT_cmyka"
    "GL_CMYK_EXT" "GL_EXT_cmyka"
    "GL_CMYKA_EXT" "GL_EXT_cmyka"
    "GL_PACK_CMYK_HINT_EXT" "GL_EXT_cmyka"
    "GL_UNPACK_CMYK_HINT_EXT" "GL_EXT_cmyka"
    "GL_EXT_color_buffer_float" "GL_EXT_color_buffer_float"
    "GL_EXT_color_buffer_half_float" "GL_EXT_color_buffer_half_float"
    "GL_FRAMEBUFFER_ATTACHMENT_COMPONENT_TYPE_EXT" "GL_EXT_color_buffer_half_float"
    "GL_R16F_EXT" "GL_EXT_color_buffer_half_float"
    "GL_RG16F_EXT" "GL_EXT_color_buffer_half_float"
    "GL_RGBA16F_EXT" "GL_EXT_color_buffer_half_float"
    "GL_RGB16F_EXT" "GL_EXT_color_buffer_half_float"
    "GL_UNSIGNED_NORMALIZED_EXT" "GL_EXT_color_buffer_half_float"
    "GL_EXT_color_subtable" "GL_EXT_color_subtable"
    "GL_EXT_compiled_vertex_array" "GL_EXT_compiled_vertex_array"
    "GL_ARRAY_ELEMENT_LOCK_FIRST_EXT" "GL_EXT_compiled_vertex_array"
    "GL_ARRAY_ELEMENT_LOCK_COUNT_EXT" "GL_EXT_compiled_vertex_array"
    "GL_EXT_compressed_ETC1_RGB8_sub_texture" "GL_EXT_compressed_ETC1_RGB8_sub_texture"
    "GL_EXT_conservative_depth" "GL_EXT_conservative_depth"
    "GL_EXT_convolution" "GL_EXT_convolution"
    "GL_CONVOLUTION_1D_EXT" "GL_EXT_convolution"
    "GL_CONVOLUTION_2D_EXT" "GL_EXT_convolution"
    "GL_SEPARABLE_2D_EXT" "GL_EXT_convolution"
    "GL_CONVOLUTION_BORDER_MODE_EXT" "GL_EXT_convolution"
    "GL_CONVOLUTION_FILTER_SCALE_EXT" "GL_EXT_convolution"
    "GL_CONVOLUTION_FILTER_BIAS_EXT" "GL_EXT_convolution"
    "GL_REDUCE_EXT" "GL_EXT_convolution"
    "GL_CONVOLUTION_FORMAT_EXT" "GL_EXT_convolution"
    "GL_CONVOLUTION_WIDTH_EXT" "GL_EXT_convolution"
    "GL_CONVOLUTION_HEIGHT_EXT" "GL_EXT_convolution"
    "GL_MAX_CONVOLUTION_WIDTH_EXT" "GL_EXT_convolution"
    "GL_MAX_CONVOLUTION_HEIGHT_EXT" "GL_EXT_convolution"
    "GL_POST_CONVOLUTION_RED_SCALE_EXT" "GL_EXT_convolution"
    "GL_POST_CONVOLUTION_GREEN_SCALE_EXT" "GL_EXT_convolution"
    "GL_POST_CONVOLUTION_BLUE_SCALE_EXT" "GL_EXT_convolution"
    "GL_POST_CONVOLUTION_ALPHA_SCALE_EXT" "GL_EXT_convolution"
    "GL_POST_CONVOLUTION_RED_BIAS_EXT" "GL_EXT_convolution"
    "GL_POST_CONVOLUTION_GREEN_BIAS_EXT" "GL_EXT_convolution"
    "GL_POST_CONVOLUTION_BLUE_BIAS_EXT" "GL_EXT_convolution"
    "GL_POST_CONVOLUTION_ALPHA_BIAS_EXT" "GL_EXT_convolution"
    "GL_EXT_coordinate_frame" "GL_EXT_coordinate_frame"
    "GL_TANGENT_ARRAY_EXT" "GL_EXT_coordinate_frame"
    "GL_BINORMAL_ARRAY_EXT" "GL_EXT_coordinate_frame"
    "GL_CURRENT_TANGENT_EXT" "GL_EXT_coordinate_frame"
    "GL_CURRENT_BINORMAL_EXT" "GL_EXT_coordinate_frame"
    "GL_TANGENT_ARRAY_TYPE_EXT" "GL_EXT_coordinate_frame"
    "GL_TANGENT_ARRAY_STRIDE_EXT" "GL_EXT_coordinate_frame"
    "GL_BINORMAL_ARRAY_TYPE_EXT" "GL_EXT_coordinate_frame"
    "GL_BINORMAL_ARRAY_STRIDE_EXT" "GL_EXT_coordinate_frame"
    "GL_TANGENT_ARRAY_POINTER_EXT" "GL_EXT_coordinate_frame"
    "GL_BINORMAL_ARRAY_POINTER_EXT" "GL_EXT_coordinate_frame"
    "GL_MAP1_TANGENT_EXT" "GL_EXT_coordinate_frame"
    "GL_MAP2_TANGENT_EXT" "GL_EXT_coordinate_frame"
    "GL_MAP1_BINORMAL_EXT" "GL_EXT_coordinate_frame"
    "GL_MAP2_BINORMAL_EXT" "GL_EXT_coordinate_frame"
    "GL_EXT_copy_image" "GL_EXT_copy_image"
    "GL_EXT_copy_texture" "GL_EXT_copy_texture"
    "GL_EXT_cull_vertex" "GL_EXT_cull_vertex"
    "GL_CULL_VERTEX_EXT" "GL_EXT_cull_vertex"
    "GL_CULL_VERTEX_EYE_POSITION_EXT" "GL_EXT_cull_vertex"
    "GL_CULL_VERTEX_OBJECT_POSITION_EXT" "GL_EXT_cull_vertex"
    "GL_EXT_debug_label" "GL_EXT_debug_label"
    "GL_PROGRAM_PIPELINE_OBJECT_EXT" "GL_EXT_debug_label"
    "GL_PROGRAM_OBJECT_EXT" "GL_EXT_debug_label"
    "GL_SHADER_OBJECT_EXT" "GL_EXT_debug_label"
    "GL_BUFFER_OBJECT_EXT" "GL_EXT_debug_label"
    "GL_QUERY_OBJECT_EXT" "GL_EXT_debug_label"
    "GL_VERTEX_ARRAY_OBJECT_EXT" "GL_EXT_debug_label"
    "GL_EXT_debug_marker" "GL_EXT_debug_marker"
    "GL_EXT_depth_bounds_test" "GL_EXT_depth_bounds_test"
    "GL_DEPTH_BOUNDS_TEST_EXT" "GL_EXT_depth_bounds_test"
    "GL_DEPTH_BOUNDS_EXT" "GL_EXT_depth_bounds_test"
    "GL_EXT_depth_clamp" "GL_EXT_depth_clamp"
    "GL_DEPTH_CLAMP_EXT" "GL_EXT_depth_clamp"
    "GL_EXT_direct_state_access" "GL_EXT_direct_state_access"
    "GL_PROGRAM_MATRIX_EXT" "GL_EXT_direct_state_access"
    "GL_TRANSPOSE_PROGRAM_MATRIX_EXT" "GL_EXT_direct_state_access"
    "GL_PROGRAM_MATRIX_STACK_DEPTH_EXT" "GL_EXT_direct_state_access"
    "GL_EXT_discard_framebuffer" "GL_EXT_discard_framebuffer"
    "GL_COLOR_EXT" "GL_EXT_discard_framebuffer"
    "GL_DEPTH_EXT" "GL_EXT_discard_framebuffer"
    "GL_STENCIL_EXT" "GL_EXT_discard_framebuffer"
    "GL_EXT_disjoint_timer_query" "GL_EXT_disjoint_timer_query"
    "GL_QUERY_COUNTER_BITS_EXT" "GL_EXT_disjoint_timer_query"
    "GL_CURRENT_QUERY_EXT" "GL_EXT_disjoint_timer_query"
    "GL_QUERY_RESULT_EXT" "GL_EXT_disjoint_timer_query"
    "GL_QUERY_RESULT_AVAILABLE_EXT" "GL_EXT_disjoint_timer_query"
    "GL_TIME_ELAPSED_EXT" "GL_EXT_disjoint_timer_query"
    "GL_TIMESTAMP_EXT" "GL_EXT_disjoint_timer_query"
    "GL_GPU_DISJOINT_EXT" "GL_EXT_disjoint_timer_query"
    "GL_EXT_draw_buffers" "GL_EXT_draw_buffers"
    "GL_MAX_DRAW_BUFFERS_EXT" "GL_EXT_draw_buffers"
    "GL_DRAW_BUFFER0_EXT" "GL_EXT_draw_buffers"
    "GL_DRAW_BUFFER1_EXT" "GL_EXT_draw_buffers"
    "GL_DRAW_BUFFER2_EXT" "GL_EXT_draw_buffers"
    "GL_DRAW_BUFFER3_EXT" "GL_EXT_draw_buffers"
    "GL_DRAW_BUFFER4_EXT" "GL_EXT_draw_buffers"
    "GL_DRAW_BUFFER5_EXT" "GL_EXT_draw_buffers"
    "GL_DRAW_BUFFER6_EXT" "GL_EXT_draw_buffers"
    "GL_DRAW_BUFFER7_EXT" "GL_EXT_draw_buffers"
    "GL_DRAW_BUFFER8_EXT" "GL_EXT_draw_buffers"
    "GL_DRAW_BUFFER9_EXT" "GL_EXT_draw_buffers"
    "GL_DRAW_BUFFER10_EXT" "GL_EXT_draw_buffers"
    "GL_DRAW_BUFFER11_EXT" "GL_EXT_draw_buffers"
    "GL_DRAW_BUFFER12_EXT" "GL_EXT_draw_buffers"
    "GL_DRAW_BUFFER13_EXT" "GL_EXT_draw_buffers"
    "GL_DRAW_BUFFER14_EXT" "GL_EXT_draw_buffers"
    "GL_DRAW_BUFFER15_EXT" "GL_EXT_draw_buffers"
    "GL_MAX_COLOR_ATTACHMENTS_EXT" "GL_EXT_draw_buffers"
    "GL_COLOR_ATTACHMENT0_EXT" "GL_EXT_draw_buffers"
    "GL_COLOR_ATTACHMENT1_EXT" "GL_EXT_draw_buffers"
    "GL_COLOR_ATTACHMENT2_EXT" "GL_EXT_draw_buffers"
    "GL_COLOR_ATTACHMENT3_EXT" "GL_EXT_draw_buffers"
    "GL_COLOR_ATTACHMENT4_EXT" "GL_EXT_draw_buffers"
    "GL_COLOR_ATTACHMENT5_EXT" "GL_EXT_draw_buffers"
    "GL_COLOR_ATTACHMENT6_EXT" "GL_EXT_draw_buffers"
    "GL_COLOR_ATTACHMENT7_EXT" "GL_EXT_draw_buffers"
    "GL_COLOR_ATTACHMENT8_EXT" "GL_EXT_draw_buffers"
    "GL_COLOR_ATTACHMENT9_EXT" "GL_EXT_draw_buffers"
    "GL_COLOR_ATTACHMENT10_EXT" "GL_EXT_draw_buffers"
    "GL_COLOR_ATTACHMENT11_EXT" "GL_EXT_draw_buffers"
    "GL_COLOR_ATTACHMENT12_EXT" "GL_EXT_draw_buffers"
    "GL_COLOR_ATTACHMENT13_EXT" "GL_EXT_draw_buffers"
    "GL_COLOR_ATTACHMENT14_EXT" "GL_EXT_draw_buffers"
    "GL_COLOR_ATTACHMENT15_EXT" "GL_EXT_draw_buffers"
    "GL_EXT_draw_buffers2" "GL_EXT_draw_buffers2"
    "GL_EXT_draw_buffers_indexed" "GL_EXT_draw_buffers_indexed"
    "GL_EXT_draw_elements_base_vertex" "GL_EXT_draw_elements_base_vertex"
    "GL_EXT_draw_instanced" "GL_EXT_draw_instanced"
    "GL_EXT_draw_range_elements" "GL_EXT_draw_range_elements"
    "GL_MAX_ELEMENTS_VERTICES_EXT" "GL_EXT_draw_range_elements"
    "GL_MAX_ELEMENTS_INDICES_EXT" "GL_EXT_draw_range_elements"
    "GL_EXT_draw_transform_feedback" "GL_EXT_draw_transform_feedback"
    "GL_EXT_external_buffer" "GL_EXT_external_buffer"
    "GL_EXT_float_blend" "GL_EXT_float_blend"
    "GL_EXT_fog_coord" "GL_EXT_fog_coord"
    "GL_FOG_COORDINATE_SOURCE_EXT" "GL_EXT_fog_coord"
    "GL_FOG_COORDINATE_EXT" "GL_EXT_fog_coord"
    "GL_FRAGMENT_DEPTH_EXT" "GL_EXT_fog_coord"
    "GL_CURRENT_FOG_COORDINATE_EXT" "GL_EXT_fog_coord"
    "GL_FOG_COORDINATE_ARRAY_TYPE_EXT" "GL_EXT_fog_coord"
    "GL_FOG_COORDINATE_ARRAY_STRIDE_EXT" "GL_EXT_fog_coord"
    "GL_FOG_COORDINATE_ARRAY_POINTER_EXT" "GL_EXT_fog_coord"
    "GL_FOG_COORDINATE_ARRAY_EXT" "GL_EXT_fog_coord"
    "GL_EXT_frag_depth" "GL_EXT_frag_depth"
    "GL_EXT_fragment_lighting" "GL_EXT_fragment_lighting"
    "GL_FRAGMENT_LIGHTING_EXT" "GL_EXT_fragment_lighting"
    "GL_FRAGMENT_COLOR_MATERIAL_EXT" "GL_EXT_fragment_lighting"
    "GL_FRAGMENT_COLOR_MATERIAL_FACE_EXT" "GL_EXT_fragment_lighting"
    "GL_FRAGMENT_COLOR_MATERIAL_PARAMETER_EXT" "GL_EXT_fragment_lighting"
    "GL_MAX_FRAGMENT_LIGHTS_EXT" "GL_EXT_fragment_lighting"
    "GL_MAX_ACTIVE_LIGHTS_EXT" "GL_EXT_fragment_lighting"
    "GL_CURRENT_RASTER_NORMAL_EXT" "GL_EXT_fragment_lighting"
    "GL_LIGHT_ENV_MODE_EXT" "GL_EXT_fragment_lighting"
    "GL_FRAGMENT_LIGHT_MODEL_LOCAL_VIEWER_EXT" "GL_EXT_fragment_lighting"
    "GL_FRAGMENT_LIGHT_MODEL_TWO_SIDE_EXT" "GL_EXT_fragment_lighting"
    "GL_FRAGMENT_LIGHT_MODEL_AMBIENT_EXT" "GL_EXT_fragment_lighting"
    "GL_FRAGMENT_LIGHT_MODEL_NORMAL_INTERPOLATION_EXT" "GL_EXT_fragment_lighting"
    "GL_FRAGMENT_LIGHT0_EXT" "GL_EXT_fragment_lighting"
    "GL_FRAGMENT_LIGHT7_EXT" "GL_EXT_fragment_lighting"
    "GL_EXT_framebuffer_blit" "GL_EXT_framebuffer_blit"
    "GL_DRAW_FRAMEBUFFER_BINDING_EXT" "GL_EXT_framebuffer_blit"
    "GL_READ_FRAMEBUFFER_EXT" "GL_EXT_framebuffer_blit"
    "GL_DRAW_FRAMEBUFFER_EXT" "GL_EXT_framebuffer_blit"
    "GL_READ_FRAMEBUFFER_BINDING_EXT" "GL_EXT_framebuffer_blit"
    "GL_EXT_framebuffer_multisample" "GL_EXT_framebuffer_multisample"
    "GL_RENDERBUFFER_SAMPLES_EXT" "GL_EXT_framebuffer_multisample"
    "GL_FRAMEBUFFER_INCOMPLETE_MULTISAMPLE_EXT" "GL_EXT_framebuffer_multisample"
    "GL_MAX_SAMPLES_EXT" "GL_EXT_framebuffer_multisample"
    "GL_EXT_framebuffer_multisample_blit_scaled" "GL_EXT_framebuffer_multisample_blit_scaled"
    "GL_SCALED_RESOLVE_FASTEST_EXT" "GL_EXT_framebuffer_multisample_blit_scaled"
    "GL_SCALED_RESOLVE_NICEST_EXT" "GL_EXT_framebuffer_multisample_blit_scaled"
    "GL_EXT_framebuffer_object" "GL_EXT_framebuffer_object"
    "GL_INVALID_FRAMEBUFFER_OPERATION_EXT" "GL_EXT_framebuffer_object"
    "GL_MAX_RENDERBUFFER_SIZE_EXT" "GL_EXT_framebuffer_object"
    "GL_FRAMEBUFFER_BINDING_EXT" "GL_EXT_framebuffer_object"
    "GL_RENDERBUFFER_BINDING_EXT" "GL_EXT_framebuffer_object"
    "GL_FRAMEBUFFER_ATTACHMENT_OBJECT_TYPE_EXT" "GL_EXT_framebuffer_object"
    "GL_FRAMEBUFFER_ATTACHMENT_OBJECT_NAME_EXT" "GL_EXT_framebuffer_object"
    "GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_LEVEL_EXT" "GL_EXT_framebuffer_object"
    "GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_CUBE_MAP_FACE_EXT" "GL_EXT_framebuffer_object"
    "GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_3D_ZOFFSET_EXT" "GL_EXT_framebuffer_object"
    "GL_FRAMEBUFFER_COMPLETE_EXT" "GL_EXT_framebuffer_object"
    "GL_FRAMEBUFFER_INCOMPLETE_ATTACHMENT_EXT" "GL_EXT_framebuffer_object"
    "GL_FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT_EXT" "GL_EXT_framebuffer_object"
    "GL_FRAMEBUFFER_INCOMPLETE_DIMENSIONS_EXT" "GL_EXT_framebuffer_object"
    "GL_FRAMEBUFFER_INCOMPLETE_FORMATS_EXT" "GL_EXT_framebuffer_object"
    "GL_FRAMEBUFFER_INCOMPLETE_DRAW_BUFFER_EXT" "GL_EXT_framebuffer_object"
    "GL_FRAMEBUFFER_INCOMPLETE_READ_BUFFER_EXT" "GL_EXT_framebuffer_object"
    "GL_FRAMEBUFFER_UNSUPPORTED_EXT" "GL_EXT_framebuffer_object"
    "GL_DEPTH_ATTACHMENT_EXT" "GL_EXT_framebuffer_object"
    "GL_STENCIL_ATTACHMENT_EXT" "GL_EXT_framebuffer_object"
    "GL_FRAMEBUFFER_EXT" "GL_EXT_framebuffer_object"
    "GL_RENDERBUFFER_EXT" "GL_EXT_framebuffer_object"
    "GL_RENDERBUFFER_WIDTH_EXT" "GL_EXT_framebuffer_object"
    "GL_RENDERBUFFER_HEIGHT_EXT" "GL_EXT_framebuffer_object"
    "GL_RENDERBUFFER_INTERNAL_FORMAT_EXT" "GL_EXT_framebuffer_object"
    "GL_STENCIL_INDEX1_EXT" "GL_EXT_framebuffer_object"
    "GL_STENCIL_INDEX4_EXT" "GL_EXT_framebuffer_object"
    "GL_STENCIL_INDEX8_EXT" "GL_EXT_framebuffer_object"
    "GL_STENCIL_INDEX16_EXT" "GL_EXT_framebuffer_object"
    "GL_RENDERBUFFER_RED_SIZE_EXT" "GL_EXT_framebuffer_object"
    "GL_RENDERBUFFER_GREEN_SIZE_EXT" "GL_EXT_framebuffer_object"
    "GL_RENDERBUFFER_BLUE_SIZE_EXT" "GL_EXT_framebuffer_object"
    "GL_RENDERBUFFER_ALPHA_SIZE_EXT" "GL_EXT_framebuffer_object"
    "GL_RENDERBUFFER_DEPTH_SIZE_EXT" "GL_EXT_framebuffer_object"
    "GL_RENDERBUFFER_STENCIL_SIZE_EXT" "GL_EXT_framebuffer_object"
    "GL_EXT_framebuffer_sRGB" "GL_EXT_framebuffer_sRGB"
    "GL_FRAMEBUFFER_SRGB_EXT" "GL_EXT_framebuffer_sRGB"
    "GL_FRAMEBUFFER_SRGB_CAPABLE_EXT" "GL_EXT_framebuffer_sRGB"
    "GL_EXT_geometry_point_size" "GL_EXT_geometry_point_size"
    "GL_GEOMETRY_SHADER_BIT_EXT" "GL_EXT_geometry_point_size"
    "GL_LINES_ADJACENCY_EXT" "GL_EXT_geometry_point_size"
    "GL_LINE_STRIP_ADJACENCY_EXT" "GL_EXT_geometry_point_size"
    "GL_TRIANGLES_ADJACENCY_EXT" "GL_EXT_geometry_point_size"
    "GL_TRIANGLE_STRIP_ADJACENCY_EXT" "GL_EXT_geometry_point_size"
    "GL_LAYER_PROVOKING_VERTEX_EXT" "GL_EXT_geometry_point_size"
    "GL_UNDEFINED_VERTEX_EXT" "GL_EXT_geometry_point_size"
    "GL_GEOMETRY_SHADER_INVOCATIONS_EXT" "GL_EXT_geometry_point_size"
    "GL_GEOMETRY_LINKED_VERTICES_OUT_EXT" "GL_EXT_geometry_point_size"
    "GL_GEOMETRY_LINKED_INPUT_TYPE_EXT" "GL_EXT_geometry_point_size"
    "GL_GEOMETRY_LINKED_OUTPUT_TYPE_EXT" "GL_EXT_geometry_point_size"
    "GL_MAX_GEOMETRY_UNIFORM_BLOCKS_EXT" "GL_EXT_geometry_point_size"
    "GL_MAX_COMBINED_GEOMETRY_UNIFORM_COMPONENTS_EXT" "GL_EXT_geometry_point_size"
    "GL_MAX_GEOMETRY_TEXTURE_IMAGE_UNITS_EXT" "GL_EXT_geometry_point_size"
    "GL_PRIMITIVES_GENERATED_EXT" "GL_EXT_geometry_point_size"
    "GL_FRAMEBUFFER_ATTACHMENT_LAYERED_EXT" "GL_EXT_geometry_point_size"
    "GL_FRAMEBUFFER_INCOMPLETE_LAYER_TARGETS_EXT" "GL_EXT_geometry_point_size"
    "GL_GEOMETRY_SHADER_EXT" "GL_EXT_geometry_point_size"
    "GL_MAX_GEOMETRY_UNIFORM_COMPONENTS_EXT" "GL_EXT_geometry_point_size"
    "GL_MAX_GEOMETRY_OUTPUT_VERTICES_EXT" "GL_EXT_geometry_point_size"
    "GL_MAX_GEOMETRY_TOTAL_OUTPUT_COMPONENTS_EXT" "GL_EXT_geometry_point_size"
    "GL_FIRST_VERTEX_CONVENTION_EXT" "GL_EXT_geometry_point_size"
    "GL_LAST_VERTEX_CONVENTION_EXT" "GL_EXT_geometry_point_size"
    "GL_MAX_GEOMETRY_SHADER_INVOCATIONS_EXT" "GL_EXT_geometry_point_size"
    "GL_MAX_GEOMETRY_IMAGE_UNIFORMS_EXT" "GL_EXT_geometry_point_size"
    "GL_MAX_GEOMETRY_SHADER_STORAGE_BLOCKS_EXT" "GL_EXT_geometry_point_size"
    "GL_MAX_GEOMETRY_INPUT_COMPONENTS_EXT" "GL_EXT_geometry_point_size"
    "GL_MAX_GEOMETRY_OUTPUT_COMPONENTS_EXT" "GL_EXT_geometry_point_size"
    "GL_MAX_GEOMETRY_ATOMIC_COUNTER_BUFFERS_EXT" "GL_EXT_geometry_point_size"
    "GL_MAX_GEOMETRY_ATOMIC_COUNTERS_EXT" "GL_EXT_geometry_point_size"
    "GL_REFERENCED_BY_GEOMETRY_SHADER_EXT" "GL_EXT_geometry_point_size"
    "GL_FRAMEBUFFER_DEFAULT_LAYERS_EXT" "GL_EXT_geometry_point_size"
    "GL_MAX_FRAMEBUFFER_LAYERS_EXT" "GL_EXT_geometry_point_size"
    "GL_EXT_geometry_shader" "GL_EXT_geometry_shader"
    "GL_EXT_geometry_shader4" "GL_EXT_geometry_shader4"
    "GL_PROGRAM_POINT_SIZE_EXT" "GL_EXT_geometry_shader4"
    "GL_MAX_VARYING_COMPONENTS_EXT" "GL_EXT_geometry_shader4"
    "GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_LAYER_EXT" "GL_EXT_geometry_shader4"
    "GL_FRAMEBUFFER_INCOMPLETE_LAYER_COUNT_EXT" "GL_EXT_geometry_shader4"
    "GL_GEOMETRY_VERTICES_OUT_EXT" "GL_EXT_geometry_shader4"
    "GL_GEOMETRY_INPUT_TYPE_EXT" "GL_EXT_geometry_shader4"
    "GL_GEOMETRY_OUTPUT_TYPE_EXT" "GL_EXT_geometry_shader4"
    "GL_MAX_GEOMETRY_VARYING_COMPONENTS_EXT" "GL_EXT_geometry_shader4"
    "GL_MAX_VERTEX_VARYING_COMPONENTS_EXT" "GL_EXT_geometry_shader4"
    "GL_EXT_gpu_program_parameters" "GL_EXT_gpu_program_parameters"
    "GL_EXT_gpu_shader4" "GL_EXT_gpu_shader4"
    "GL_VERTEX_ATTRIB_ARRAY_INTEGER_EXT" "GL_EXT_gpu_shader4"
    "GL_SAMPLER_1D_ARRAY_EXT" "GL_EXT_gpu_shader4"
    "GL_SAMPLER_2D_ARRAY_EXT" "GL_EXT_gpu_shader4"
    "GL_SAMPLER_BUFFER_EXT" "GL_EXT_gpu_shader4"
    "GL_SAMPLER_1D_ARRAY_SHADOW_EXT" "GL_EXT_gpu_shader4"
    "GL_SAMPLER_2D_ARRAY_SHADOW_EXT" "GL_EXT_gpu_shader4"
    "GL_SAMPLER_CUBE_SHADOW_EXT" "GL_EXT_gpu_shader4"
    "GL_UNSIGNED_INT_VEC2_EXT" "GL_EXT_gpu_shader4"
    "GL_UNSIGNED_INT_VEC3_EXT" "GL_EXT_gpu_shader4"
    "GL_UNSIGNED_INT_VEC4_EXT" "GL_EXT_gpu_shader4"
    "GL_INT_SAMPLER_1D_EXT" "GL_EXT_gpu_shader4"
    "GL_INT_SAMPLER_2D_EXT" "GL_EXT_gpu_shader4"
    "GL_INT_SAMPLER_3D_EXT" "GL_EXT_gpu_shader4"
    "GL_INT_SAMPLER_CUBE_EXT" "GL_EXT_gpu_shader4"
    "GL_INT_SAMPLER_2D_RECT_EXT" "GL_EXT_gpu_shader4"
    "GL_INT_SAMPLER_1D_ARRAY_EXT" "GL_EXT_gpu_shader4"
    "GL_INT_SAMPLER_2D_ARRAY_EXT" "GL_EXT_gpu_shader4"
    "GL_INT_SAMPLER_BUFFER_EXT" "GL_EXT_gpu_shader4"
    "GL_UNSIGNED_INT_SAMPLER_1D_EXT" "GL_EXT_gpu_shader4"
    "GL_UNSIGNED_INT_SAMPLER_2D_EXT" "GL_EXT_gpu_shader4"
    "GL_UNSIGNED_INT_SAMPLER_3D_EXT" "GL_EXT_gpu_shader4"
    "GL_UNSIGNED_INT_SAMPLER_CUBE_EXT" "GL_EXT_gpu_shader4"
    "GL_UNSIGNED_INT_SAMPLER_2D_RECT_EXT" "GL_EXT_gpu_shader4"
    "GL_UNSIGNED_INT_SAMPLER_1D_ARRAY_EXT" "GL_EXT_gpu_shader4"
    "GL_UNSIGNED_INT_SAMPLER_2D_ARRAY_EXT" "GL_EXT_gpu_shader4"
    "GL_UNSIGNED_INT_SAMPLER_BUFFER_EXT" "GL_EXT_gpu_shader4"
    "GL_EXT_gpu_shader5" "GL_EXT_gpu_shader5"
    "GL_EXT_histogram" "GL_EXT_histogram"
    "GL_HISTOGRAM_EXT" "GL_EXT_histogram"
    "GL_PROXY_HISTOGRAM_EXT" "GL_EXT_histogram"
    "GL_HISTOGRAM_WIDTH_EXT" "GL_EXT_histogram"
    "GL_HISTOGRAM_FORMAT_EXT" "GL_EXT_histogram"
    "GL_HISTOGRAM_RED_SIZE_EXT" "GL_EXT_histogram"
    "GL_HISTOGRAM_GREEN_SIZE_EXT" "GL_EXT_histogram"
    "GL_HISTOGRAM_BLUE_SIZE_EXT" "GL_EXT_histogram"
    "GL_HISTOGRAM_ALPHA_SIZE_EXT" "GL_EXT_histogram"
    "GL_HISTOGRAM_LUMINANCE_SIZE_EXT" "GL_EXT_histogram"
    "GL_HISTOGRAM_SINK_EXT" "GL_EXT_histogram"
    "GL_MINMAX_EXT" "GL_EXT_histogram"
    "GL_MINMAX_FORMAT_EXT" "GL_EXT_histogram"
    "GL_MINMAX_SINK_EXT" "GL_EXT_histogram"
    "GL_EXT_index_array_formats" "GL_EXT_index_array_formats"
    "GL_EXT_index_func" "GL_EXT_index_func"
    "GL_EXT_index_material" "GL_EXT_index_material"
    "GL_EXT_index_texture" "GL_EXT_index_texture"
    "GL_EXT_instanced_arrays" "GL_EXT_instanced_arrays"
    "GL_VERTEX_ATTRIB_ARRAY_DIVISOR_EXT" "GL_EXT_instanced_arrays"
    "GL_EXT_light_texture" "GL_EXT_light_texture"
    "GL_FRAGMENT_MATERIAL_EXT" "GL_EXT_light_texture"
    "GL_FRAGMENT_NORMAL_EXT" "GL_EXT_light_texture"
    "GL_FRAGMENT_COLOR_EXT" "GL_EXT_light_texture"
    "GL_ATTENUATION_EXT" "GL_EXT_light_texture"
    "GL_SHADOW_ATTENUATION_EXT" "GL_EXT_light_texture"
    "GL_TEXTURE_APPLICATION_MODE_EXT" "GL_EXT_light_texture"
    "GL_TEXTURE_LIGHT_EXT" "GL_EXT_light_texture"
    "GL_TEXTURE_MATERIAL_FACE_EXT" "GL_EXT_light_texture"
    "GL_TEXTURE_MATERIAL_PARAMETER_EXT" "GL_EXT_light_texture"
    "GL_EXT_map_buffer_range" "GL_EXT_map_buffer_range"
    "GL_MAP_READ_BIT_EXT" "GL_EXT_map_buffer_range"
    "GL_MAP_WRITE_BIT_EXT" "GL_EXT_map_buffer_range"
    "GL_MAP_INVALIDATE_RANGE_BIT_EXT" "GL_EXT_map_buffer_range"
    "GL_MAP_INVALIDATE_BUFFER_BIT_EXT" "GL_EXT_map_buffer_range"
    "GL_MAP_FLUSH_EXPLICIT_BIT_EXT" "GL_EXT_map_buffer_range"
    "GL_MAP_UNSYNCHRONIZED_BIT_EXT" "GL_EXT_map_buffer_range"
    "GL_EXT_memory_object" "GL_EXT_memory_object"
    "GL_UUID_SIZE_EXT" "GL_EXT_memory_object"
    "GL_TEXTURE_TILING_EXT" "GL_EXT_memory_object"
    "GL_DEDICATED_MEMORY_OBJECT_EXT" "GL_EXT_memory_object"
    "GL_NUM_TILING_TYPES_EXT" "GL_EXT_memory_object"
    "GL_TILING_TYPES_EXT" "GL_EXT_memory_object"
    "GL_OPTIMAL_TILING_EXT" "GL_EXT_memory_o