/* SPDX-FileCopyrightText: 2020-2023 Blender Authors
 *
 * SPDX-License-Identifier: GPL-2.0-or-later */

#include "infos/workbench_prepass_info.hh"

FRAGMENT_SHADER_CREATE_INFO(workbench_prepass)
FRAGMENT_SHADER_CREATE_INFO(workbench_transparent_accum)
FRAGMENT_SHADER_CREATE_INFO(workbench_lighting_matcap)

#include "draw_view_lib.glsl"
#include "workbench_common_lib.glsl"
#include "workbench_image_lib.glsl"
#include "workbench_matcap_lib.glsl"
#include "workbench_world_light_lib.glsl"

/* Special function only to be used with calculate_transparent_weight(). */
float linear_zdepth(float depth, mat4 proj_mat)
{
  if (proj_mat[3][3] == 0.0) {
    float d = 2.0 * depth - 1.0;
    return -proj_mat[3][2] / (d + proj_mat[2][2]);
  }
  else {
    /* Return depth from near plane. */
    float z_delta = -2.0 / proj_mat[2][2];
    return depth * z_delta;
  }
}

/* Based on :
 * McGuire and Bavoil, Weighted Blended Order-Independent Transparency, Journal of
 * Computer Graphics Techniques (JCGT), vol. 2, no. 2, 122–141, 2013
 */
float calculate_transparent_weight()
{
  float z = linear_zdepth(gl_FragCoord.z, drw_view.winmat);
#if 0
  /* Eq 10 : Good for surfaces with varying opacity (like particles) */
  float a = min(1.0, alpha * 10.0) + 0.01;
  float b = -gl_FragCoord.z * 0.95 + 1.0;
  float w = a * a * a * 3e2 * b * b * b;
#else
  /* Eq 7 put more emphasis on surfaces closer to the view. */
  // float w = 10.0 / (1e-5 + pow(abs(z) / 5.0, 2.0) + pow(abs(z) / 200.0, 6.0)); /* Eq 7 */
  // float w = 10.0 / (1e-5 + pow(abs(z) / 10.0, 3.0) + pow(abs(z) / 200.0, 6.0)); /* Eq 8 */
  // float w = 10.0 / (1e-5 + pow(abs(z) / 200.0, 4.0)); /* Eq 9 */
  /* Same as eq 7, but optimized. */
  float a = abs(z) / 5.0;
  float b = abs(z) / 200.0;
  b *= b;
  float w = 10.0 / ((1e-5 + a * a) + b * (b * b)); /* Eq 7 */
#endif
  return clamp(w, 1e-2, 3e2);
}

void main()
{
  /* Normal and Incident vector are in view-space. Lighting is evaluated in view-space. */
  vec2 uv_viewport = gl_FragCoord.xy * world_data.viewport_size_inv;
  vec3 vP = drw_point_screen_to_view(vec3(uv_viewport, 0.5));
  vec3 I = drw_view_incident_vector(vP);
  vec3 N = normalize(normal_interp);

  vec3 color = color_interp;

#ifdef WORKBENCH_COLOR_TEXTURE
  color = workbench_image_color(uv_interp);
#endif

#ifdef WORKBENCH_LIGHTING_MATCAP
  vec3 shaded_color = get_matcap_lighting(matcap_tx, color, N, I);
#endif

#ifdef WORKBENCH_LIGHTING_STUDIO
  vec3 shaded_color = get_world_lighting(color, _roughness, metallic, N, I);
#endif

#ifdef WORKBENCH_LIGHTING_FLAT
  vec3 shaded_color = color;
#endif

  shaded_color *= get_shadow(N, forceShadowing);

  /* Listing 4 */
  float alpha = alpha_interp * world_data.xray_alpha;
  float weight = calculate_transparent_weight() * alpha;
  out_transparent_accum = vec4(shaded_color * weight, alpha);
  out_revealage_accum = vec4(weight);

  out_object_id = uint(object_id);
}
