/** @file
  Dynamic Table Factory Dxe

  Copyright (c) 2017 - 2019, ARM Limited. All rights reserved.

  SPDX-License-Identifier: BSD-2-Clause-Patent

**/

#include <Library/DebugLib.h>
#include <Library/PcdLib.h>
#include <Library/UefiBootServicesTableLib.h>
#include <Protocol/AcpiTable.h>

// Module specific include files.
#include <AcpiTableGenerator.h>
#include <ConfigurationManagerObject.h>
#include <ConfigurationManagerHelper.h>
#include <DeviceTreeTableGenerator.h>
#include <Library/TableHelperLib.h>
#include <Protocol/ConfigurationManagerProtocol.h>
#include <Protocol/DynamicTableFactoryProtocol.h>
#include <SmbiosTableGenerator.h>
#include <Library/MetadataHandlerLib.h>

#include "DynamicTableFactory.h"

/** The Dynamic Table Factory protocol structure that holds the
    list of registered ACPI and SMBIOS table generators.
*/
EDKII_DYNAMIC_TABLE_FACTORY_INFO  TableFactoryInfo;

STATIC METADATA_ROOT_HANDLE  mMetadataRoot;

/** Get the Root handle of the MetadataObjLib.

  During the firmware table generation, some Metadata information might be
  generated by different generators. This Metadata might be subject to
  additional validation.

  @return The Metadata Root handle.
**/
METADATA_ROOT_HANDLE
EFIAPI
GetMetadataRoot (
  VOID
  )
{
  return mMetadataRoot;
}

/** A structure describing the Dynamic Table Factory protocol.
*/
STATIC
CONST
EDKII_DYNAMIC_TABLE_FACTORY_PROTOCOL  DynamicTableFactoryProtocol = {
  CREATE_REVISION (1,             0),
  GetAcpiTableGenerator,
  RegisterAcpiTableGenerator,
  DeregisterAcpiTableGenerator,
  GetSmbiosTableGenerator,
  RegisterSmbiosTableGenerator,
  DeregisterSmbiosTableGenerator,
  GetDtTableGenerator,
  RegisterDtTableGenerator,
  DeregisterDtTableGenerator,
  GetMetadataRoot,
  AddSmbiosHandle,
  FindSmbiosHandle,
  FindSmbiosHandleEx,
  &TableFactoryInfo
};

/** Entrypoint for Dynamic Table Factory Dxe.

  @param  ImageHandle
  @param  SystemTable

  @retval EFI_SUCCESS           Success.
  @retval EFI_OUT_OF_RESOURCES  Memory allocation failed.
  @retval EFI_NOT_FOUND         Required interface/object was not found.
  @retval EFI_INVALID_PARAMETER Some parameter is incorrect/invalid.
**/
EFI_STATUS
EFIAPI
DynamicTableFactoryDxeInitialize (
  IN  EFI_HANDLE        ImageHandle,
  IN  EFI_SYSTEM_TABLE  *SystemTable
  )
{
  EFI_STATUS  Status;
  UINTN       Idx;

  for (Idx = 0; Idx < FixedPcdGet16 (PcdMaxSmbiosHandleMapEntries); Idx++) {
    TableFactoryInfo.SmbiosHandleMap[Idx].SmbiosTblHandle = SMBIOS_HANDLE_PI_RESERVED;
    TableFactoryInfo.SmbiosHandleMap[Idx].SmbiosCmToken   = 0;
  }

  Status = MetadataInitializeHandle (&mMetadataRoot);
  if (EFI_ERROR (Status)) {
    ASSERT_EFI_ERROR (Status);
    return Status;
  }

  Status = gBS->InstallProtocolInterface (
                  &ImageHandle,
                  &gEdkiiDynamicTableFactoryProtocolGuid,
                  EFI_NATIVE_INTERFACE,
                  (VOID *)&DynamicTableFactoryProtocol
                  );
  if (EFI_ERROR (Status)) {
    DEBUG ((
      DEBUG_ERROR,
      "ERROR: Failed to install the Dynamic Table Factory Protocol." \
      " Status = %r\n",
      Status
      ));
  }

  return Status;
}
