/*
  SDL_mixer:  An audio mixer library based on the SDL library
  Copyright (C) 1997-2025 Sam Lantinga <slouken@libsdl.org>

  This software is provided 'as-is', without any express or implied
  warranty.  In no event will the authors be held liable for any damages
  arising from the use of this software.

  Permission is granted to anyone to use this software for any purpose,
  including commercial applications, and to alter it and redistribute it
  freely, subject to the following restrictions:

  1. The origin of this software must not be misrepresented; you must not
     claim that you wrote the original software. If you use this software
     in a product, an acknowledgment in the product documentation would be
     appreciated but is not required.
  2. Altered source versions must be plainly marked as such, and must not be
     misrepresented as being the original software.
  3. This notice may not be removed or altered from any source distribution.

  This is the source needed to decode an AIFF file into a waveform.
  It's pretty straightforward once you get going. The only
  externally-callable function is Mix_LoadAIFF_IO(), which is meant to
  act as identically to SDL_LoadWAV_IO() as possible.

  This file by Torbjörn Andersson (torbjorn.andersson@eurotime.se)
  8SVX file support added by Marc Le Douarain (mavati@club-internet.fr)
  in december 2002.
*/

#include <SDL3/SDL_endian.h>
#include <SDL3_mixer/SDL_mixer.h>
#include "load_aiff.h"

/*********************************************/
/* Define values for AIFF (IFF audio) format */
/*********************************************/
#define FORM        0x4d524f46      /* "FORM" */

#define AIFF        0x46464941      /* "AIFF" */
#define SSND        0x444e5353      /* "SSND" */
#define COMM        0x4d4d4f43      /* "COMM" */

#define _8SVX       0x58565338      /* "8SVX" */
#define VHDR        0x52444856      /* "VHDR" */
#define BODY        0x59444F42      /* "BODY" */

/* This function was taken from libsndfile. I don't pretend to fully
 * understand it.
 */

static Uint32 SANE_to_Uint32 (Uint8 *sanebuf)
{
    /* Is the frequency outside of what we can represent with Uint32? */
    if ((sanebuf[0] & 0x80) || (sanebuf[0] <= 0x3F) || (sanebuf[0] > 0x40)
        || (sanebuf[0] == 0x40 && sanebuf[1] > 0x1C))
        return 0;

    return ((sanebuf[2] << 23) | (sanebuf[3] << 15) | (sanebuf[4] << 7)
        | (sanebuf[5] >> 1)) >> (29 - sanebuf[1]);
}

/* This function is based on SDL_LoadWAV_IO(). */

SDL_AudioSpec *Mix_LoadAIFF_IO (SDL_IOStream *src, bool closeio,
                                SDL_AudioSpec *spec, Uint8 **audio_buf, Uint32 *audio_len)
{
    int found_SSND;
    int found_COMM;
    int found_VHDR;
    int found_BODY;
    bool was_error = true;
    Sint64 start = 0;

    Uint32 chunk_type;
    Uint32 chunk_length;
    Sint64 next_chunk;

    /* AIFF magic header */
    Uint32 FORMchunk;
    Uint32 AIFFmagic;

    /* SSND chunk */
    Uint32 offset;
    Uint32 blocksize;

    /* COMM format chunk */
    Uint16 channels = 0;
    Uint32 numsamples = 0;
    Uint16 samplesize = 0;
    Uint8 sane_freq[10];
    Uint32 frequency = 0;

    /* VHDR chunk */
    Uint16 frequency16 = 0;

    /* Sanity checks */
    if (audio_buf) {
        *audio_buf = NULL;
    }
    if (!src) {
        SDL_InvalidParamError("src");
        goto done;
    }
    if (!spec) {
        SDL_InvalidParamError("spec");
        goto done;
    }
    if (!audio_buf) {
        SDL_InvalidParamError("audio_buf");
        goto done;
    }
    if (!audio_len) {
        SDL_InvalidParamError("audio_len");
        goto done;
    }

    if (!SDL_ReadU32LE(src, &FORMchunk) ||
        !SDL_ReadU32BE(src, &chunk_length)) {
        goto done;
    }
    if (chunk_length == AIFF) { /* The FORMchunk has already been read */
        AIFFmagic    = chunk_length;
        chunk_length = FORMchunk;
        FORMchunk    = FORM;
    } else {
        if (!SDL_ReadU32LE(src, &AIFFmagic)) {
            goto done;
        }
    }
    if ((FORMchunk != FORM) || ((AIFFmagic != AIFF) && (AIFFmagic != _8SVX))) {
        SDL_SetError("Unrecognized file type (not AIFF nor 8SVX)");
        goto done;
    }

    /* TODO: Better santity-checking. */

    found_SSND = 0;
    found_COMM = 0;
    found_VHDR = 0;
    found_BODY = 0;

    do {
        if (!SDL_ReadU32LE(src, &chunk_type) ||
            !SDL_ReadU32BE(src, &chunk_length)) {
            goto done;
        }
        next_chunk  = SDL_TellIO(src) + chunk_length;

        /* Paranoia to avoid infinite loops */
        if (chunk_length == 0) {
            break;
        }

        switch (chunk_type) {
            case SSND:
                found_SSND  = 1;
                if (!SDL_ReadU32BE(src, &offset) ||
                    !SDL_ReadU32BE(src, &blocksize)) {
                    goto done;
                }
                start = SDL_TellIO(src) + offset;
                (void)blocksize; /* unused. */
                break;

            case COMM:
                found_COMM  = 1;
                if (!SDL_ReadU16BE(src, &channels) ||
                    !SDL_ReadU32BE(src, &numsamples) ||
                    !SDL_ReadU16BE(src, &samplesize)) {
                    goto done;
                }
                if (SDL_ReadIO(src, sane_freq, sizeof(sane_freq)) != sizeof(sane_freq)) {
                    SDL_SetError("Bad AIFF sample frequency");
                    goto done;
                }
                frequency = SANE_to_Uint32(sane_freq);
                if (frequency == 0) {
                    SDL_SetError("Bad AIFF sample frequency");
                    goto done;
                }
                break;

            case VHDR:
                found_VHDR  = 1;
                if (!SDL_ReadU32BE(src, NULL) ||
                    !SDL_ReadU32BE(src, NULL) ||
                    !SDL_ReadU32BE(src, NULL) ||
                    !SDL_ReadU16BE(src, &frequency16)) {
                    goto done;
                }
                channels = 1;
                samplesize = 8;
                frequency = frequency16;
                break;

            case BODY:
                found_BODY  = 1;
                numsamples  = chunk_length;
                start       = SDL_TellIO(src);
                break;

            default:
                break;
        }
        /* a 0 pad byte can be stored for any odd-length chunk */
        if (chunk_length&1) {
            next_chunk++;
        }
    } while ((((AIFFmagic == AIFF) && (!found_SSND || !found_COMM))
          || ((AIFFmagic == _8SVX) && (!found_VHDR || !found_BODY)))
          && SDL_SeekIO(src, next_chunk, SDL_IO_SEEK_SET) != 1);

    if ((AIFFmagic == AIFF) && !found_SSND) {
        SDL_SetError("Bad AIFF (no SSND chunk)");
        goto done;
    }

    if ((AIFFmagic == AIFF) && !found_COMM) {
        SDL_SetError("Bad AIFF (no COMM chunk)");
        goto done;
    }

    if ((AIFFmagic == _8SVX) && !found_VHDR) {
        SDL_SetError("Bad 8SVX (no VHDR chunk)");
        goto done;
    }

    if ((AIFFmagic == _8SVX) && !found_BODY) {
        SDL_SetError("Bad 8SVX (no BODY chunk)");
        goto done;
    }

    /* Decode the audio data format */
    SDL_zerop(spec);
    spec->freq = frequency;
    switch (samplesize) {
        case 8:
            spec->format = SDL_AUDIO_S8;
            break;
        case 16:
            spec->format = SDL_AUDIO_S16BE;
            break;
        default:
            SDL_SetError("Unsupported AIFF samplesize");
            goto done;
    }
    spec->channels = (Uint8) channels;

    *audio_len = channels * numsamples * (samplesize / 8);
    *audio_buf = (Uint8 *)SDL_malloc(*audio_len);
    if (*audio_buf == NULL) {
        goto done;
    }
    SDL_SeekIO(src, start, SDL_IO_SEEK_SET);
    if (SDL_ReadIO(src, *audio_buf, *audio_len) != *audio_len) {
        SDL_SetError("Unable to read audio data");
        goto done;
    }

    /* Don't return a buffer that isn't a multiple of samplesize */
    *audio_len &= ~((samplesize / 8) - 1);

    was_error = false;

done:
    if (closeio && src) {
        SDL_CloseIO(src);
    }
    if (was_error) {
        if (audio_buf && *audio_buf) {
            SDL_free(*audio_buf);
            *audio_buf = NULL;
        }
        if (audio_len) {
            *audio_len = 0;
        }
        spec = NULL;
    }
    return spec;
}

/* vi: set ts=4 sw=4 expandtab: */
