package XML::Malware;

use strict;
use warnings;

our $VERSION = '0.00_04';
$VERSION = eval $VERSION;  # see L<perlmodstyle>

use XML::Compile;
use XML::Compile::Schema;
use XML::Compile::Util qw/pack_type/;
use XML::LibXML;

my $type = pack_type 'http://xml/metadataSharing.xsd', 'malwareMetaData';
XML::Compile->addSchemaDirs(__FILE__);
use constant MYNS => 'http://xml/metadataSharing.xsd';
XML::Compile->knownNamespace(&MYNS => 'metadataSharing.xsd');

# Preloaded methods go here.

sub new {
    my $class = shift;
    my $args = shift; 
    my $self = {};
    bless($self,$class);
    $self->_hash($args);
    return($self);
}

sub in {
    my $self = shift;
    my $xml = shift;
    return(undef) unless($xml);

    my $s = XML::Compile::Schema->new();
    $s->importDefinitions(MYNS);

    my $read = $s->compile(READER => $type);
    my $h = $read->($xml);
    return(undef) unless($h && ref($h) eq 'HASH');
    $self->_hash($h);
    return($self->_hash());
}

sub out {
    my $self = shift;
    my $s       = XML::Compile::Schema->new();
    $s->importDefinitions(MYNS);
    my $doc    = XML::LibXML::Document->new('1.0', 'UTF-8');
    my $write  = $s->compile(WRITER => $type);
    my $xml    = $write->($doc, $self->_hash());
    return $xml->toString();
}

# accessors

sub _hash { 
    my ($self,$v) = @_;
    $self->{'_hash'} = $v if(defined($v));
    return($self->{'_hash'});
}

1;
__END__

=head1 NAME

XML::Malware - Perl extension for representing malware samples in XML 

=head1 SYNOPSIS

  use XML::Malware;
  my $h;
  $h->{'company'} = 'wes co';
  $h->{'author'} = 'wes';
  $h->{'comment'} = 'test';
  $h->{'timestamp'} = '2009-02-23T17:34:56';
  $h->{'id'} = '';
  push(@{$h->{'objects'}->{'file'}}, { id => '',  md5 => "aa31da6402d850ce94e7c19bc97effe1" });
  push(@{$h->{'objects'}->{'classification'}}, { id => '', companyName => 'some company', type => 'dirty', classificationName => 'className'});

  my $m = XML::Malware->new($h);
  warn $m->out();

=head1 DESCRIPTION

This module works with ICSG schema v1.1

=head1 SEE ALSO

XML::IODEF, XML::IDMEF

http://code.google.com/p/perl-xml-malware/

http://standards.ieee.org/prod-serv/indconn/icsg/index.html

http://grouper.ieee.org/groups/malware/malwg/Schema1.1/

=head1 AUTHOR

Wes Young, E<lt>wes@barely3am.com<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2010 by Wes Young

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.10.0 or,
at your option, any later version of Perl 5 you may have available.


=cut
