﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/resiliencehub/ResilienceHubRequest.h>
#include <aws/resiliencehub/ResilienceHub_EXPORTS.h>
#include <aws/resiliencehub/model/AppAssessmentScheduleType.h>
#include <aws/resiliencehub/model/EventSubscription.h>
#include <aws/resiliencehub/model/PermissionModel.h>

#include <utility>

namespace Aws {
namespace ResilienceHub {
namespace Model {

/**
 */
class CreateAppRequest : public ResilienceHubRequest {
 public:
  AWS_RESILIENCEHUB_API CreateAppRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateApp"; }

  AWS_RESILIENCEHUB_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p> Assessment execution schedule with 'Daily' or 'Disabled' values. </p>
   */
  inline AppAssessmentScheduleType GetAssessmentSchedule() const { return m_assessmentSchedule; }
  inline bool AssessmentScheduleHasBeenSet() const { return m_assessmentScheduleHasBeenSet; }
  inline void SetAssessmentSchedule(AppAssessmentScheduleType value) {
    m_assessmentScheduleHasBeenSet = true;
    m_assessmentSchedule = value;
  }
  inline CreateAppRequest& WithAssessmentSchedule(AppAssessmentScheduleType value) {
    SetAssessmentSchedule(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Amazon Resource Name (ARN) of Resource Groups group that is integrated with
   * an AppRegistry application. For more information about ARNs, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">
   * Amazon Resource Names (ARNs)</a> in the <i>Amazon Web Services General
   * Reference</i> guide.</p>
   */
  inline const Aws::String& GetAwsApplicationArn() const { return m_awsApplicationArn; }
  inline bool AwsApplicationArnHasBeenSet() const { return m_awsApplicationArnHasBeenSet; }
  template <typename AwsApplicationArnT = Aws::String>
  void SetAwsApplicationArn(AwsApplicationArnT&& value) {
    m_awsApplicationArnHasBeenSet = true;
    m_awsApplicationArn = std::forward<AwsApplicationArnT>(value);
  }
  template <typename AwsApplicationArnT = Aws::String>
  CreateAppRequest& WithAwsApplicationArn(AwsApplicationArnT&& value) {
    SetAwsApplicationArn(std::forward<AwsApplicationArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Used for an idempotency token. A client token is a unique, case-sensitive
   * string of up to 64 ASCII characters. You should not reuse the same client token
   * for other API requests.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateAppRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The optional description for an app.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateAppRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The list of events you would like to subscribe and get notification for.
   * Currently, Resilience Hub supports only <b>Drift detected</b> and <b>Scheduled
   * assessment failure</b> events notification.</p>
   */
  inline const Aws::Vector<EventSubscription>& GetEventSubscriptions() const { return m_eventSubscriptions; }
  inline bool EventSubscriptionsHasBeenSet() const { return m_eventSubscriptionsHasBeenSet; }
  template <typename EventSubscriptionsT = Aws::Vector<EventSubscription>>
  void SetEventSubscriptions(EventSubscriptionsT&& value) {
    m_eventSubscriptionsHasBeenSet = true;
    m_eventSubscriptions = std::forward<EventSubscriptionsT>(value);
  }
  template <typename EventSubscriptionsT = Aws::Vector<EventSubscription>>
  CreateAppRequest& WithEventSubscriptions(EventSubscriptionsT&& value) {
    SetEventSubscriptions(std::forward<EventSubscriptionsT>(value));
    return *this;
  }
  template <typename EventSubscriptionsT = EventSubscription>
  CreateAppRequest& AddEventSubscriptions(EventSubscriptionsT&& value) {
    m_eventSubscriptionsHasBeenSet = true;
    m_eventSubscriptions.emplace_back(std::forward<EventSubscriptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Name of the application.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateAppRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Defines the roles and credentials that Resilience Hub would use while
   * creating the application, importing its resources, and running an
   * assessment.</p>
   */
  inline const PermissionModel& GetPermissionModel() const { return m_permissionModel; }
  inline bool PermissionModelHasBeenSet() const { return m_permissionModelHasBeenSet; }
  template <typename PermissionModelT = PermissionModel>
  void SetPermissionModel(PermissionModelT&& value) {
    m_permissionModelHasBeenSet = true;
    m_permissionModel = std::forward<PermissionModelT>(value);
  }
  template <typename PermissionModelT = PermissionModel>
  CreateAppRequest& WithPermissionModel(PermissionModelT&& value) {
    SetPermissionModel(std::forward<PermissionModelT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Amazon Resource Name (ARN) of the resiliency policy. The format for this ARN
   * is:
   * arn:<code>partition</code>:resiliencehub:<code>region</code>:<code>account</code>:resiliency-policy/<code>policy-id</code>.
   * For more information about ARNs, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">
   * Amazon Resource Names (ARNs)</a> in the <i>Amazon Web Services General
   * Reference</i> guide.</p>
   */
  inline const Aws::String& GetPolicyArn() const { return m_policyArn; }
  inline bool PolicyArnHasBeenSet() const { return m_policyArnHasBeenSet; }
  template <typename PolicyArnT = Aws::String>
  void SetPolicyArn(PolicyArnT&& value) {
    m_policyArnHasBeenSet = true;
    m_policyArn = std::forward<PolicyArnT>(value);
  }
  template <typename PolicyArnT = Aws::String>
  CreateAppRequest& WithPolicyArn(PolicyArnT&& value) {
    SetPolicyArn(std::forward<PolicyArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags assigned to the resource. A tag is a label that you assign to an Amazon
   * Web Services resource. Each tag consists of a key/value pair.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateAppRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateAppRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  AppAssessmentScheduleType m_assessmentSchedule{AppAssessmentScheduleType::NOT_SET};
  bool m_assessmentScheduleHasBeenSet = false;

  Aws::String m_awsApplicationArn;
  bool m_awsApplicationArnHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::Vector<EventSubscription> m_eventSubscriptions;
  bool m_eventSubscriptionsHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  PermissionModel m_permissionModel;
  bool m_permissionModelHasBeenSet = false;

  Aws::String m_policyArn;
  bool m_policyArnHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace ResilienceHub
}  // namespace Aws
