﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/lakeformation/LakeFormationRequest.h>
#include <aws/lakeformation/LakeFormation_EXPORTS.h>
#include <aws/lakeformation/model/WriteOperation.h>

#include <utility>

namespace Aws {
namespace LakeFormation {
namespace Model {

/**
 */
class UpdateTableObjectsRequest : public LakeFormationRequest {
 public:
  AWS_LAKEFORMATION_API UpdateTableObjectsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateTableObjects"; }

  AWS_LAKEFORMATION_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The catalog containing the governed table to update. Defaults to the caller’s
   * account ID.</p>
   */
  inline const Aws::String& GetCatalogId() const { return m_catalogId; }
  inline bool CatalogIdHasBeenSet() const { return m_catalogIdHasBeenSet; }
  template <typename CatalogIdT = Aws::String>
  void SetCatalogId(CatalogIdT&& value) {
    m_catalogIdHasBeenSet = true;
    m_catalogId = std::forward<CatalogIdT>(value);
  }
  template <typename CatalogIdT = Aws::String>
  UpdateTableObjectsRequest& WithCatalogId(CatalogIdT&& value) {
    SetCatalogId(std::forward<CatalogIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The database containing the governed table to update.</p>
   */
  inline const Aws::String& GetDatabaseName() const { return m_databaseName; }
  inline bool DatabaseNameHasBeenSet() const { return m_databaseNameHasBeenSet; }
  template <typename DatabaseNameT = Aws::String>
  void SetDatabaseName(DatabaseNameT&& value) {
    m_databaseNameHasBeenSet = true;
    m_databaseName = std::forward<DatabaseNameT>(value);
  }
  template <typename DatabaseNameT = Aws::String>
  UpdateTableObjectsRequest& WithDatabaseName(DatabaseNameT&& value) {
    SetDatabaseName(std::forward<DatabaseNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The governed table to update.</p>
   */
  inline const Aws::String& GetTableName() const { return m_tableName; }
  inline bool TableNameHasBeenSet() const { return m_tableNameHasBeenSet; }
  template <typename TableNameT = Aws::String>
  void SetTableName(TableNameT&& value) {
    m_tableNameHasBeenSet = true;
    m_tableName = std::forward<TableNameT>(value);
  }
  template <typename TableNameT = Aws::String>
  UpdateTableObjectsRequest& WithTableName(TableNameT&& value) {
    SetTableName(std::forward<TableNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The transaction at which to do the write.</p>
   */
  inline const Aws::String& GetTransactionId() const { return m_transactionId; }
  inline bool TransactionIdHasBeenSet() const { return m_transactionIdHasBeenSet; }
  template <typename TransactionIdT = Aws::String>
  void SetTransactionId(TransactionIdT&& value) {
    m_transactionIdHasBeenSet = true;
    m_transactionId = std::forward<TransactionIdT>(value);
  }
  template <typename TransactionIdT = Aws::String>
  UpdateTableObjectsRequest& WithTransactionId(TransactionIdT&& value) {
    SetTransactionId(std::forward<TransactionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of <code>WriteOperation</code> objects that define an object to add to
   * or delete from the manifest for a governed table.</p>
   */
  inline const Aws::Vector<WriteOperation>& GetWriteOperations() const { return m_writeOperations; }
  inline bool WriteOperationsHasBeenSet() const { return m_writeOperationsHasBeenSet; }
  template <typename WriteOperationsT = Aws::Vector<WriteOperation>>
  void SetWriteOperations(WriteOperationsT&& value) {
    m_writeOperationsHasBeenSet = true;
    m_writeOperations = std::forward<WriteOperationsT>(value);
  }
  template <typename WriteOperationsT = Aws::Vector<WriteOperation>>
  UpdateTableObjectsRequest& WithWriteOperations(WriteOperationsT&& value) {
    SetWriteOperations(std::forward<WriteOperationsT>(value));
    return *this;
  }
  template <typename WriteOperationsT = WriteOperation>
  UpdateTableObjectsRequest& AddWriteOperations(WriteOperationsT&& value) {
    m_writeOperationsHasBeenSet = true;
    m_writeOperations.emplace_back(std::forward<WriteOperationsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_catalogId;
  bool m_catalogIdHasBeenSet = false;

  Aws::String m_databaseName;
  bool m_databaseNameHasBeenSet = false;

  Aws::String m_tableName;
  bool m_tableNameHasBeenSet = false;

  Aws::String m_transactionId;
  bool m_transactionIdHasBeenSet = false;

  Aws::Vector<WriteOperation> m_writeOperations;
  bool m_writeOperationsHasBeenSet = false;
};

}  // namespace Model
}  // namespace LakeFormation
}  // namespace Aws
