﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/fis/FISRequest.h>
#include <aws/fis/FIS_EXPORTS.h>
#include <aws/fis/model/StartExperimentExperimentOptionsInput.h>

#include <utility>

namespace Aws {
namespace FIS {
namespace Model {

/**
 */
class StartExperimentRequest : public FISRequest {
 public:
  AWS_FIS_API StartExperimentRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartExperiment"; }

  AWS_FIS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Unique, case-sensitive identifier that you provide to ensure the idempotency
   * of the request.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  StartExperimentRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the experiment template.</p>
   */
  inline const Aws::String& GetExperimentTemplateId() const { return m_experimentTemplateId; }
  inline bool ExperimentTemplateIdHasBeenSet() const { return m_experimentTemplateIdHasBeenSet; }
  template <typename ExperimentTemplateIdT = Aws::String>
  void SetExperimentTemplateId(ExperimentTemplateIdT&& value) {
    m_experimentTemplateIdHasBeenSet = true;
    m_experimentTemplateId = std::forward<ExperimentTemplateIdT>(value);
  }
  template <typename ExperimentTemplateIdT = Aws::String>
  StartExperimentRequest& WithExperimentTemplateId(ExperimentTemplateIdT&& value) {
    SetExperimentTemplateId(std::forward<ExperimentTemplateIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The experiment options for running the experiment.</p>
   */
  inline const StartExperimentExperimentOptionsInput& GetExperimentOptions() const { return m_experimentOptions; }
  inline bool ExperimentOptionsHasBeenSet() const { return m_experimentOptionsHasBeenSet; }
  template <typename ExperimentOptionsT = StartExperimentExperimentOptionsInput>
  void SetExperimentOptions(ExperimentOptionsT&& value) {
    m_experimentOptionsHasBeenSet = true;
    m_experimentOptions = std::forward<ExperimentOptionsT>(value);
  }
  template <typename ExperimentOptionsT = StartExperimentExperimentOptionsInput>
  StartExperimentRequest& WithExperimentOptions(ExperimentOptionsT&& value) {
    SetExperimentOptions(std::forward<ExperimentOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags to apply to the experiment.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  StartExperimentRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  StartExperimentRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  Aws::String m_experimentTemplateId;
  bool m_experimentTemplateIdHasBeenSet = false;

  StartExperimentExperimentOptionsInput m_experimentOptions;
  bool m_experimentOptionsHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace FIS
}  // namespace Aws
