﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/directconnect/DirectConnectRequest.h>
#include <aws/directconnect/DirectConnect_EXPORTS.h>

#include <utility>

namespace Aws {
namespace DirectConnect {
namespace Model {

/**
 */
class UpdateConnectionRequest : public DirectConnectRequest {
 public:
  AWS_DIRECTCONNECT_API UpdateConnectionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateConnection"; }

  AWS_DIRECTCONNECT_API Aws::String SerializePayload() const override;

  AWS_DIRECTCONNECT_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the connection.</p> <p>You can use <a>DescribeConnections</a> to
   * retrieve the connection ID.</p>
   */
  inline const Aws::String& GetConnectionId() const { return m_connectionId; }
  inline bool ConnectionIdHasBeenSet() const { return m_connectionIdHasBeenSet; }
  template <typename ConnectionIdT = Aws::String>
  void SetConnectionId(ConnectionIdT&& value) {
    m_connectionIdHasBeenSet = true;
    m_connectionId = std::forward<ConnectionIdT>(value);
  }
  template <typename ConnectionIdT = Aws::String>
  UpdateConnectionRequest& WithConnectionId(ConnectionIdT&& value) {
    SetConnectionId(std::forward<ConnectionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the connection.</p>
   */
  inline const Aws::String& GetConnectionName() const { return m_connectionName; }
  inline bool ConnectionNameHasBeenSet() const { return m_connectionNameHasBeenSet; }
  template <typename ConnectionNameT = Aws::String>
  void SetConnectionName(ConnectionNameT&& value) {
    m_connectionNameHasBeenSet = true;
    m_connectionName = std::forward<ConnectionNameT>(value);
  }
  template <typename ConnectionNameT = Aws::String>
  UpdateConnectionRequest& WithConnectionName(ConnectionNameT&& value) {
    SetConnectionName(std::forward<ConnectionNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The connection MAC Security (MACsec) encryption mode.</p> <p>The valid values
   * are <code>no_encrypt</code>, <code>should_encrypt</code>, and
   * <code>must_encrypt</code>.</p>
   */
  inline const Aws::String& GetEncryptionMode() const { return m_encryptionMode; }
  inline bool EncryptionModeHasBeenSet() const { return m_encryptionModeHasBeenSet; }
  template <typename EncryptionModeT = Aws::String>
  void SetEncryptionMode(EncryptionModeT&& value) {
    m_encryptionModeHasBeenSet = true;
    m_encryptionMode = std::forward<EncryptionModeT>(value);
  }
  template <typename EncryptionModeT = Aws::String>
  UpdateConnectionRequest& WithEncryptionMode(EncryptionModeT&& value) {
    SetEncryptionMode(std::forward<EncryptionModeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_connectionId;
  bool m_connectionIdHasBeenSet = false;

  Aws::String m_connectionName;
  bool m_connectionNameHasBeenSet = false;

  Aws::String m_encryptionMode;
  bool m_encryptionModeHasBeenSet = false;
};

}  // namespace Model
}  // namespace DirectConnect
}  // namespace Aws
