#pragma once

#include "xmpp/jid.hpp"
#include <config/config.hpp>

namespace utils
{
/**
 * Check config if a given requester is allowed to persist target
 *
 * The following values are checked
 *
 * - A user's jid (foo@example.com)
 * - A user's domain (example.com)
 *
 * The following config options are checked in order
 *
 * 1. `persist_user_denylist` contains the requester's jid
 * 2. `persist_user_allowlist` contains the requester's jid
 * 3. `persist_user_denylist` contains the requester's domain
 * 4. `persist_user_allowlist` contains the contains the requester's domain
 * 5. `default_persistence_allowed` is true (default true)
 */
inline bool is_requester_allowed_to_persist(const Jid &requester) {
  // requester.bare checks
  // deny
  // 1.
  bool jid_in_user_denylist = Config::is_in_list("persist_user_denylist", requester.bare());
  if (jid_in_user_denylist)
    return false;

  // allow
  // 2.
  bool jid_in_user_allowlist = Config::is_in_list("persist_user_allowlist", requester.bare());
  if (jid_in_user_allowlist)
    return true;

  // requester.domain checks
  // deny
  // 3.
  bool domain_in_user_denylist = Config::is_in_list("persist_user_denylist", requester.domain);
  if (domain_in_user_denylist)
    return false;

  // allow
  // 4.
  bool domain_in_user_allowlist = Config::is_in_list("persist_user_allowlist", requester.domain);
  if (domain_in_user_allowlist)
    return true;

  // default allow
  // 5.
  return Config::get_bool("default_persistence_allowed", true);
}
}
